@** HTML Generation.

This sink translates parsed body copy (emitted by
|etextBodyParser|) into HTML.  An index document is written,
which contains links to the individual chapters which, in turn,
link to one another and an additional document containing any
footnotes. When the sink is created, it must be passed the
|basename| used to generate the names for these files; the
|basename| may not be ``\.{-}'' denoting standard output.

HTML generation is a sink (or, more precisely, a variable-tentacled
squid with built-in sinks for each) because, unless instantiated
with the |singleFile| option, it generates an index file, a file
for each chapter, and a footnote file if needed.  Since there's
no way to know which files will be needed prior to processing
the text, higher level code can't be expected to create a
forked pipeline to write all these files.  So, |HTMLGenerationSink|
simply consumes the parsed document body it receives and creates
whatever files are required to build an HTML document from the text
on the fly.

@<Class definitions@>=
class HTMLGenerationSink : public textSink {
private:@/
    static const int defaultFootnotePad = 60;	// Default blank lines between footnotes

    bool italics, inmath, infoot, firstchap, quoth, hastitle, hasauthor;
    string basename;	    	   	    // Document base name
    string stime;   	    	    	    // Processing date and time for comment
    string indexFileName;   	    	    // Index (or main document if |singleFile| name
    ofstream *index;  	    	    	    // Index output stream
    ofstream *chap;   	    	    	    // Chapter output stream
    ofstream *foot;   	    	    	    // Footnote output stream
    string htitle;  	    	    	    // HTML \.{<title>} of document
    string hauthor; 	    	    	    // Author of document
    int chapno;     	    	    	    // Chapter index ($1\cdots n$)
    string chapnumber;	    	    	    // Chapter number from text
    string chapname;	    	    	    // Chapter name
    string qchapnumber, qchapname;  	    // Chapter number and name in queue
    string chtitle; 	    	    	    // Generated HTML title for chapter document
    bool singleFile;   	    	    	    // Make one HTML file for entire document
    string footdocname;     	    	    // Footnote document name
    int footnum;	    	    	    // Footnote number
    int footnest;   	    	    	    // Footnote nesting level
    bool fitalics, fquoth;	    	    // Text processing modes saved during footnote
    bool breakPending;	    	    	    // Float-clearing break pending ?
    int footnotePad;	    	    	    // Footnote pad lines
    queue <string> chapterCache;    	    // Pending chapter cache
    int indexline, chapline, footline;	    // Index, chapter, and footnote line counters
    deque <string> declarationsQueue;	    // HTML special declarations
    string headingAlign;                    // Heading alignment declaration
   
    string quoteHTMLString(string s);
     
    string translateHTMLString(string t);
    
    void emitq(string s) {
    	emit(translateHTMLString(s));
    }
    
    void generateAlignedParagraph(string alignment, char bracket, string text,
    	    string terminator = "<br />");
	    
    void writeHTMLDocumentPreamble(ostream &os, string title, int *lineCounter);
	    
    void writeHTMLDocumentBodyStart(ostream &os, string title, int *lineCounter);
    
    void writeHTMLDocumentPostamble(ostream &os, int *lineCounter);
    
    static void createNavButton(string button, const unsigned char source[], int length);
    
    static void createNavButtons(void);
    
    void createNavigationPanel(int prev, int next, bool inParagraph = true);
    
    virtual void emit(string s, textComponent *destination = NULL) {
    	if (singleFile) {
	    *index << s << "\n";
	    indexline++;
	} else {
    	    if (infoot || (chap == NULL)) {
	    	*(infoot ? foot : index) << s << "\n";
		if (infoot) {
		    footline++;
		} else {
		    indexline++;
		}
	    } else {
	    	chapterCache.push(s + "\n");
	    }
	}
	lineNumber++;
    }
    
    static string pruneIndent(string s) {
    	assert(s != "");
    	return s.substr(s.find_first_not_of(' '));
    }
    
    static string elideNewLines(string s) {
    	string o = s;
	unsigned int i;
	
	while ((i = o.find('\n')) != string::npos) {
	    o.replace(i, 1, " ");
	}
	while ((o.length() > 0) && (o[o.length() - 1] == ' ')) {
	    o = o.substr(0, o.length() - 1);
	}
	return o;
    }
    
    static unsigned int linesIn(string s) { 	// Count lines in string
    	return count(s.begin(), s.end(), '\n');
    }
    
    void flushBreak(void) {
    	if (breakPending) {
            if (strictCompliance) {
	        emit("<br class=\"cb\" />");
             } else {
	        emit("<br clear=\"all\" />");
            }
	    breakPending = false;
	}
    }
    
public:@/
    void setFootnotePad(int fp = defaultFootnotePad) {
    	footnotePad = fp;
    }
    
    int getFootnotePad(void) {
    	return footnotePad;
    }

    HTMLGenerationSink(string bname, bool make_one_file = false) {
    	time_t t = time(NULL);
    	stime = ctime(&t);
    	stime = stime.substr(0, stime.length() - 1);
	
    	italics = inmath = infoot = quoth = false;
	hastitle = hasauthor = false;
	firstchap = false;
	if (bname == "-") {
	    cerr << "Cannot write HTML document set to standard output.\n";
	    exit(1);
	}
	basename = bname;
	singleFile = make_one_file;
	index = chap = foot = NULL;
	footnum = footnest = 0;
	footdocname = "";
	setFootnotePad();
	breakPending = false;
	chapno = 0;
	indexline = chapline = footline = 0;
    }
    
    virtual ~HTMLGenerationSink() {
    }

    string componentName(void) {
        return "HTMLGenerationSink";
    }
    
    string getBaseName(void) {
    	return basename;
    }
    
    void put(string s);
};

@
The |put| method of the HTMLGenerationSink wraps HTML tags
around the line-level structure of the text to achieve the
desired formatting.  Since almost all of the real work is
done upstream (by |etextBodyParserFilter|) and downstream
(by |translateHTMLString|) there is actually little that needs
doing here.

@<Class definitions@>=
void HTMLGenerationSink::put(string s) {
    bodyState state = DecodeBodyState(s[0]);
    char bracket = s[1];
    string text = s.substr(2);
    ostringstream efn;

    switch (state) {
	case BeginText:
	    indexFileName = basename + ".html";
	    index = new ofstream(indexFileName.c_str(), ios::out);
	    break;
	    
	case Declarations:
	    @<Process declarations in HTML@>;

	case DocumentTitle:
	    @<Process document title in HTML@>;

	case Author:
	    @<Process author in HTML@>;

	case ChapterNumber:
	    @<Process chapter number in HTML@>;

	case ChapterName:
	    @<Process chapter name in HTML@>;

	case InTextParagraph:
	    generateAlignedParagraph("justify", bracket, text, "");
	    break;

	case InBlockQuote:
	    generateAlignedParagraph("quote", bracket, text, "");
	    break;

	case InRaggedRight:
	    generateAlignedParagraph("left", bracket, text);
	    break;

	case InRaggedLeft:
	    generateAlignedParagraph("right", bracket, text);
	    break;
	    
	case InPreformattedTable:
	    generateAlignedParagraph("table", bracket, text, "");
	    break;

	case InCentred:
	    generateAlignedParagraph("center", bracket, text);
	    break;

	case EndOfText:
	    if (!singleFile) {
    	    	*index << "</table>\n";
		indexline++;
	    }
	    *index << "</div>\n";
	    indexline++;
	    writeHTMLDocumentPostamble(*index, &indexline);
	    index->close();
	    @<Complete chapter file generation in HTML@>;
	    if (foot != NULL) {
	    	*foot << "</div>\n";
		footline++;
	    	writeHTMLDocumentPostamble(*foot, &footline);
	    	foot->close();
		if (verbose) {
		    cerr << footdocname << ": " << footline << " lines.\n";
		}
	    }
	    if (verbose) {
		cerr << indexFileName << ": " << indexline << " lines.\n";
	    }
	    break;

	default:
	    cerr << "*** State " << stateNames[state] << " " << bracket <<
		" not handled in HTMLGenerationSink ***\n";
	    exit(1);
    }
}

@
HTML-specific declarations are saved in |declarationsQueue|
whence they are emitted in the \.{<head>} section of each
HTML file generated.

@<Process declarations in HTML@>=
    if (bracket == Body) {
    	assert(etextBodyParserFilter::isLineSpecial(text));
	declarationsQueue.push_back(etextBodyParserFilter::specialCommand(text));
    }
    break;

@
When we see the title, we save it in |htitle| for use in the
\.{<title>} tag of each of the HTML documents we generate.
Titles may span multiple lines; we concatenate them into a
single line in |htitle|.

@<Process document title in HTML@>=
    switch (bracket) {
    	case Begin:
            quoth = false;
	    htitle = "";
	    break;
	    
	case Body:
	    if (htitle != "") {
	    	htitle += " ";
	    }
	    htitle += quoteHTMLString(pruneIndent(text));
	    hastitle = true;
	    break;

	case Void:
	    hastitle = false;
	    htitle = "";
	    break;
    }
    break;

@
Once we've seen the author or received the |Void| notification
that no author was given, we're ready to generate the header for
the |index| document.  The author specification may also span
multiple lines, which are concatenated into |hauthor|.

@<Process author in HTML@>=
    switch (bracket) {
    	case Begin:
            quoth = false;
	    hauthor = "";
	    break;
	    
	case Body:
	    if (hauthor != "") {
	    	hauthor += " ";
	    }
	    hauthor += quoteHTMLString(pruneIndent(text));
	    break;

	case End:
	    hasauthor = true;@/@,
	    // Note fall-through

	case Void:
	    @<Generate index document header in HTML@>;
	    break;
    }
    break;

@
The HTML index document is a list of links to the individual chapter
documents.  Generate the canned HTML header, the title and author
information (if any), and begin the list of chapters.

@<Generate index document header in HTML@>=
    writeHTMLDocumentPreamble(*index, htitle, &indexline);
    writeHTMLDocumentBodyStart(*index, htitle, &indexline);
    *index << "<div class=\"bodycopy\">\n";
    indexline++;
    if (hastitle) {
    	*index << "<h1 " << headingAlign << ">" << translateHTMLString(htitle) << "</h1>\n";
	indexline++;
    }
    if (hasauthor) {
    	*index << "<h2 " << headingAlign << ">" << translateHTMLString(hauthor) << "</h2>\n";
	indexline++;
    }

@
We save the chapter number for later use in the index document or
chapter titles written in |singleFile| mode.

@<Process chapter number in HTML@>=
    switch (bracket) {
	case Begin:
	    chapnumber = "";
	    break;

	case Body:
	    chapnumber += quoteHTMLString(pruneIndent(text)) + "\n";
	    break;

	case Void:
	    chapnumber = "";@/@,
	    // Note fall-through
	    
	case End:
	    break;
    }
    break;

@
Unless |singleFile| is set, each chapter is written into its
own HTML document named |basename|\.{\_chap}{\it n}\.{.html},
linked to the |index| document.  In |singleFile| we simply generate
a chapter break within the unified document file.

@<Process chapter name in HTML@>=
    switch (bracket) {
	case Begin:
	    chapname = "";
	    break;

	case Body:
	    chapname += quoteHTMLString(pruneIndent(text)) + "\n";
	    break;

	case Void:
	    chapname = "";@/@,
	    // Note fall-through
	    
	case End:
	    flushBreak();
	    if (singleFile) {
	    	@<Generate chapter title for single file output in HTML@>;
	    } else {
	    	@<Generate chapter title for document tree output in HTML@>;
	    }
	    break;
    }
    break;
    
@
When a single HTML document is being generated containing all chapters,
the chapter break simply causes the generation of a title within
the output document.  The title contains the chapter number and
chapter name given in the chapter break sequence with, if both are
given, a horizontal rule separating them.  If a chapter break specifies
neither a chapter number nor title, only the horizontal rule is
generated, while if only a number or title appear, no rule is
output.

@<Generate chapter title for single file output in HTML@>=
    *index << "<h2 " << headingAlign << " style=\"margin-left: 5%; margin-right: 5%;\">\n";
    indexline++;
    if (chapnumber != "") {
	*index << translateHTMLString(chapnumber);
	indexline += linesIn(chapnumber);
    }
    if (((chapname != "") || (chapname != "")) ||
	((chapname == "") && (chapname == ""))) {
	*index << "</h2>\n";
        if (strictCompliance) {
	    *index << "<hr class=\"s\" />\n";
        } else {
	    *index << "<hr width=\"25%\" size=\"2\" noshade=\"noshade\" />\n";
        }
	*index << "<h2 " << headingAlign << " style=\"margin-left: 5%; margin-right: 5%;\">\n";
	indexline += 3;
    }
    if (chapname != "") {
	*index << translateHTMLString(chapname);
	indexline += linesIn(chapname);
    }
    *index << "</h2>\n";
    indexline++;

@
When generating a multiple-file HTML document tree from an Etext,
one chapter per file, a chapter break is understandably more of
an event.  We need to close out the last chapter (if any), open a
new HTML document for the new chapter, add an index entry pointing
to it in the |index| document, and crank out the HTML preamble
boilerplate and chapter title for the next chapter.  If this is
the first chapter mark, we create the PNG files used as
the navigation buttons in the chapter documents.

@<Generate chapter title for document tree output in HTML@>=
    @<Complete chapter file generation in HTML@>;
    if (chapno == 0) {
    	createNavButtons(); 	    // Create PNG navigation buttons
        if (strictCompliance) {     // Begin chapter table
     	    *index << "<table class=\"ctable\">\n";
        } else {
     	    *index << "<table width=\"80%\" align=\"center\">\n";
        }
	indexline++;
    }
    chapno++;
    chapline = 0;
    efn << basename << "_chap" << chapno << ".html";
    chap = new ofstream(efn.str().c_str(), ios::out);
    chtitle = htitle;
    if (chapnumber != "") {
    	chtitle += ": ";
	chtitle += chapnumber;
    }
    qchapnumber = chapnumber;
    qchapname = chapname;

    //	Create link to chapter in index document

    if (strictCompliance) {
        *index << "<tr><th><a href=\"" << efn.str() << "\">";
    } else {
        *index << "<tr><th align=\"right\" width=\"15%\"><a href=\"" << efn.str() << "\">";
    }
    if (chapnumber == "") {
    	*index << chapno << ".";
    } else {
    	*index << elideNewLines(chapnumber);
	indexline += linesIn(elideNewLines(chapnumber));
    }
    if (strictCompliance) {
        *index << "</a></th> <td class=\"space\">&nbsp;</td> <td class=\"name\"><a href=\""
    	       << efn.str() << "\">" <<
    	          translateHTMLString(elideNewLines(chapname)) << "</a></td></tr>\n";
    } else {
        *index << "</a></th> <td width=\"5%\">&nbsp;</td> <td width=\"80%\"><a href=\""
    	       << efn.str() << "\">" <<
    	          translateHTMLString(elideNewLines(chapname)) << "</a></td></tr>\n";
    }
    indexline += linesIn(elideNewLines(chapname)) + 1;
    
@
Complete generation of the current chapter document and close the
file.  This may be called even if no chapter document is open,
for example, when a single file document is being written.
\.{<link>} tags are included in the header to indicate the
order of the chapters and their relationship to the parent
index document.

@<Complete chapter file generation in HTML@>=
    if (chap != NULL) {
    	string s;
	
	writeHTMLDocumentPreamble(*chap, chtitle, &chapline);
	*chap << "<link href=\"" << indexFileName << "\" rel=\"parent\" rev=\"child\" />\n";
	chapline++;
	if (chapno > 1) {
	    *chap << "<link href=\"" << basename << "_chap" << (chapno - 1) << ".html" <<
	    	     "\" rel=\"prev\" rev=\"next\" />\n";
	    chapline++;
	}
	if (state != EndOfText) {
	    *chap << "<link href=\"" << basename << "_chap" << (chapno + 1) << ".html" <<
	    	     "\" rel=\"next\" rev=\"prev\" />\n";
	    chapline++;
	}
	writeHTMLDocumentBodyStart(*chap, chtitle, &chapline);
    	if (hastitle) {
            if (strictCompliance) {
    	        *chap << "<table class=\"chfile\">\n";
	        *chap << "<tr><td class=\"l\">&nbsp;</td>\n";
	        *chap << "<td class=\"m\"><h1>" << htitle << "</h1></td>\n";
	        *chap << "<td class=\"r\">\n";
            } else {
    	        *chap << "<table width=\"100%\">\n";
	        *chap << "<tr><td width=\"25%\" valign=\"top\">&nbsp;</td>\n";
	        *chap << "<td width=\"50%\" align=\"center\"><h1>" << htitle << "</h1></td>\n";
	        *chap << "<td width=\"25%\" align=\"right\">\n";
            }
	    createNavigationPanel(chapno - 1, (state == EndOfText) ? 0 : (chapno + 1), false);
	    *chap << "</td></tr></table>\n";
	    chapline += 5;
	} else {
	    createNavigationPanel(chapno - 1, (state == EndOfText) ? 0 : (chapno + 1));
	}
	*chap << "<div class=\"bodycopy\">\n";
	*chap << "<h1 " << headingAlign << " style=\"margin-left: 5%; margin-right: 5%;\">\n";
	chapline += 2;
	if (qchapnumber != "") {
	    *chap << translateHTMLString(qchapnumber);
	    chapline += linesIn(qchapnumber);
	}
	if (((qchapname != "") || (qchapname != "")) ||
	    ((qchapname == "") && (qchapname == ""))) {
	    *chap << "</h1>\n";
            if (strictCompliance) {
	        *chap << "<hr class=\"s\" />\n";
            } else {
	        *chap << "<hr width=\"25%\" size=\"2\" noshade=\"noshade\" />\n";
            }
	    *chap << "<h1 " << headingAlign << " style=\"margin-left: 5%; margin-right: 5%;\">\n";
	    chapline += 3;
	}
	if (qchapname != "") {
	    *chap << translateHTMLString(qchapname);
	    chapline += linesIn(qchapname);
	}
	*chap << "</h1>\n";
	*chap << "\n";
	chapline += 2;
	while (!chapterCache.empty()) {
	    *chap << chapterCache.front();
	    chapline++;
	    chapterCache.pop();
	}
	*chap << "</div>\n";
	chapline += 2;
	createNavigationPanel(chapno - 1, (state == EndOfText) ? 0 : (chapno + 1));
	writeHTMLDocumentPostamble(*chap, &chapline);
	chap->close();
	if (verbose) {
	    cerr << basename << "_chap" << chapno << ".html: " << chapline << " lines.\n";
	}
    }

@
This function handles the various kinds of aligned paragraphs we encounter
in a document.  It precedes the body of the paragraph with a
\.{<p>} tag with the specified alignment.  Block quotes are also
handled here using the pseudo-alignment of ``\.{quote}''---they are
wrapped by a \.{<blockquote>} tag instead.

@<Class definitions@>=
void HTMLGenerationSink::generateAlignedParagraph(string alignment, char bracket, string text,
    	    string terminator) {
    string s;
    
    switch (bracket) {
	case Begin:
	    emit("");	    	    // Purely for readability when hand-editing HTML
	    if (alignment == "quote") {
	    	emit("<blockquote>");
                if (strictCompliance) {
                    emit("<div class=\"quote\">");
                }
	    } else if (alignment == "table") {
	    	emit("<pre>");
	    } else {
                if (strictCompliance) {
	    	    emit("<p class=\"" + alignment + "\">");
                } else {
	    	    emit("<p align=\"" + alignment + "\">");
                }
	    }
	    break;

	case Body:
	    if (etextBodyParserFilter::isLineSpecial(text)) {
	    	emit(etextBodyParserFilter::specialCommand(text));
	    } else {
	    	s = translateHTMLString(text);
	    	if (!infoot) {
	    	    s += terminator;
		}
		emit(s);
	    }
	    break;

	case End:
	    if (alignment == "quote") {
                if (strictCompliance) {
                    emit("</div>");
                }
	    	emit("</blockquote>");
	    } else if (alignment == "table") {
	    	emit("</pre>");
	    } else {
	    	emit("</p>");
	    }
	    break;
	    
	case Void:
	    break;
    }
}

@
Quote string |s| for output to HTML, replacing HTML metacharacters
with their corresponding ``\.{\&}'' entities.  With typical text it
would probably be faster to first test for the presence of any
of the metacharacters in a line using |find_first_of|, returning
the line unmodified if none were found.  This would only require
character-by-character examination in the normally rare circumstance
where the line contains a character we need to quote.  Given that
this program is extremely unlikely to be run frequently, we'll
forego such refinements in the interest of clarity.

@<Class definitions@>=
string HTMLGenerationSink::quoteHTMLString(string s)
{
    string::iterator cp;
    string o = "";
    
    for (cp = s.begin(); cp < s.end(); cp++) {
    	int c = (*cp) & 0xFF;
	
    	switch (c) {
	    case '&':
	    	o += "&amp;";
		break;
		
	    case '<':
	    	o += "&lt;";
		break;
		
	    case '>':
	    	o += "&gt";
		break;
		
	    default:
		if (c < ' ') {
		    @<Quote control character in HTML@>;
		} else {
                    if (unicodeChars) {
                        @<Check for Unicode text entity translation@>
                    } else {
	    	        o += c;
                    }
		}
		break;
	}
    }
    return o;
}

@
This is a control character.  Emit as \.{\caret}{\it letter} unless it is
considered as white space (for example, carriage return and
line feed), in which case it's sent directly to the output.
	       
@<Quote control character in HTML@>=
   if (isspace(c)) {
        o += c;
    } else {
    	o += "^" + ('@@' + c);
    }
    
@
If the \.{--unicode} option is specified, check for character sequences
which should be translated into Unicode text entities.

@<Check for Unicode text entity translation@>=
    if (!inmath && (c == '-') && ((cp + 1) < s.end()) && (cp[1] == '-')) {
        @<Translate em-dash to Unicode entity@>;
    } else if (!inmath && (c == '.') && ((cp + 2) < s.end()) &&
                (cp[1] == '.') && (cp[2] == '.')) {
        @<Translate ellipsis Unicode entity@>;
    } else if (!inmath && c == '"') {
        @<Convert ASCII quotes to open and close quote Unicode entities@>;
    } else {
        o += c;
    }

@
Two adjacent hyphens, ``\.{-}\.{-}'' denote an {\it em} dash in
an ASCII Etext.  Translate this sequence into the XHTML
``\.{\&mdash;}'' entity.

@<Translate em-dash to Unicode entity@>=
    o += "&mdash;";
    cp++;

@
Three consecutive periods are translated into a
``\.{\&hellip;}'' XHTML text entity.
    
@<Translate ellipsis Unicode entity@>=
   o += "&hellip;";
   cp += 2;
@
ASCII quote characters are translated into open and close quote symbols.
Note that the flag |quoth| is unconditionally reset at the end of
a paragraph so that mismatched quotes won't propagate beyond one paragraph.
This allows continued quotes in multiple paragraphs to work properly.
We also save and restore |quoth| around footnotes so quote
matching works when a footnote appears within quotes.

@<Convert ASCII quotes to open and close quote Unicode entities@>=
    o += quoth ? "&rdquo;" : "&ldquo;";
    quoth = !quoth;
  
@
Translate the text string argument into HTML, processing
control sequences as required.  Note that metacharacters are
expanded by the call on |quoteHTMLString| right at the top of
the function, so the balance of the code needn't worry about
quoting them.  That leaves italic and math mode toggles,
and footnotes to be handled here.

@<Class definitions@>=
string HTMLGenerationSink::translateHTMLString(string t)
{
    string::iterator cp;
    string o = "";
    char c;
    static const string punctuation = PUNCTUATION;
    string s = quoteHTMLString(t);
    
    for (cp = s.begin(); cp < s.end(); cp++) {
    	c = *cp;
	
        if (!inmath && c == '_') {
	    @<Toggle italic text mode in HTML@>;
	} else if (c == '\\' && ((cp + 1) < s.end()) && ((cp[1] == '(') || (cp[1] == ')'))) {
	    @<Toggle math mode in HTML@>;
	} else if (!inmath && ((cp + 2) < s.end()) &&
	    	   ((c == '[') || ((c == ' ') && (cp[1] == '[')) ||
                               ((c == ' ') && (cp[1] == ' ') && (cp[2] == '[')))) {
	    @<Begin footnote in HTML@>;
	} else if (!inmath && c == ']') {
	    @<End footnote in HTML@>;
	} else {
            if (c == '&') {
                @<Transcribe HTML text entity literally to output@>
            } else {
	        @<Output text character in HTML@>;
            }
	}
    }
    return o;
}

@
The underscore character, ``\.{\_}'', toggles text between the normal
roman and italic fonts.
  
@<Toggle italic text mode in HTML@>=
    italics = !italics;
    if (italics) {
        o += "<i> ";
    } else {
        o +="</i>";
    }

@
HTML doesn't support mathematics, at least not without plug-ins
or XML horrors few users at this writing are likely to have
installed. When we encounter mathematics in the text, we simply
enclose it in a \.{<table>} box, tinted pink to indicate it
requires attention, and proceed.  The editor of the document
can then use \TeX to{\tt GIF} or an equivalent tool to render
the equation for Web publication.
    
@<Toggle math mode in HTML@>=
    inmath = cp[1] == '(';
    cp++;
    if (inmath) {
        if (strictCompliance) {
    	    o += "<table class=\"math\"><tr><td>";
        } else {
    	    o += "<table bgcolor=\"#FFA0A0\"><tr><td>";
        }
    } else {
    	o += "</td></tr></table>";
    }

@
Footnotes appear in-line, within [square brackets].  If we're writing
a single file, we simply output them in-line, in square backets, using
a small sans-serif font with a yellow background (assuming the
browser comprehends such things, naturally).  When generating a
multiple file document tree, footnotes are placed in a
dedicated file, with a link to that file and the fragment
ID of the footnote in the body copy where the note appeared.
The footnote link is the footnote number as a superscript.
We don't support nested footnotes.  If the input document contains
them, we render nested footnotes in-line in the same manner
as outer level footnotes in |singleFile| mode.

Browsers which support targeted windows will open the footnote
document in a new window which will be scrolled as
subsequent footnote links are clicked.

@<Begin footnote in HTML@>=
#define NETSCRAPE_SUCKS     	    // Work around moronic style/table interaction in Netscrap
    if (footnest > 0) {
    	issueMessage("Cannot nest footnotes in HTML document output.");
        if (strictCompliance) {
    	    o += "<span class=\"footnote\">[";
	    o += "<span class=\"foottext\">";
        } else {
    	    o += "<span style=\"background-color: #FFFFA0\">[<small>";
	    o += "<font face=\"Helvetica, Arial\">";
        }
    } else {
    	if (singleFile) {
#ifdef NETSCRAPE_SUCKS
            if (strictCompliance) {
    	        o += "<span class=\"footnote\">[";
	        o += "<span class=\"foottext\">";
            } else {
    	        o += "<span style=\"background-color: #FFFFA0\">[<small>";
	        o += "<font face=\"Helvetica, Arial\">";
            }
#else
	    flushBreak();
            if (strictCompliance) {
	        o += "<sup>*</sup>\n<table class=\"footnote\">\n";
	        o += "<tr><td><b>*</b> <small>\n";
            } else {
	        o += "<sup>*</sup>\n<table width=\"25%\" "
                     "align=\"right\" hspace=\"6\" bgcolor=\"#FFFFD0\">\n";
	        o += "<tr><td><b>*</b> <small>\n";
            }
#endif
	} else {
    	    ostringstream eflink;
	    
    	    @<Create footnote file for first footnote in HTML@>;
	    footnum++;
            if (strictCompliance) {
	        eflink << "<a href=\"" << footdocname << "#" << footnum <<
	    	          "\" rel=\"Target:" << basename << "_foot\">" <<
	    	          "<sup>" << footnum << "</sup></a>";
            } else {
	        eflink << "<a href=\"" << footdocname << "#" << footnum <<
	    	          "\" target=\"" << basename << "_foot\">" <<
	    	          "<sup>" << footnum << "</sup></a>";
            }
	    o += eflink.str();
	    emit(o);
	    o = "";
	    infoot = true;
	    eflink.str("");
	    eflink << "<a name=\"" << footnum << "\">" << footnum << ".</a> ";
	    o += eflink.str();
	}
	fitalics = italics;
	fquoth = quoth;
	italics = quoth = false;
    }
    if ((c == ' ') && ((cp + 1) < s.end())) {
        if (cp[1] == ' ') {
            cp++;
        }
        cp++;
    }
    footnest++;

@
Close a footnote when the right bracket is encountered.  When
generating a separate footnote document, we use a \.{<pre>}
element with |footnotePad| blank lines following the footnote
so the next one won't appear in the window for typical browser
window sizes.
    
@<End footnote in HTML@>=
    if (footnest == 1) {
    	if (singleFile) {
#ifdef NETSCRAPE_SUCKS
            if (strictCompliance) {
    	        o += "</span>]</span>";
            } else {
    	        o += "</font></small>]</span>";
            }
#else
	    o += "\n</small>\n</table>\n";
#endif
	    breakPending = true;
	} else {
    	    int l;
	    
    	    emit(o);
	    emit("<p />");
	    emit("<pre>");
	    for (l = 0; l < footnotePad; l++) {
	    	emit("");
	    }
	    emit("</pre>");
	    o = "";
	    infoot = false;
	}
	italics = fitalics;
	quoth = fquoth;
    } else if (footnest > 1) {
        if (strictCompliance) {
    	    o += "</span>]</span>";
        } else {
    	    o += "</font></small>]</span>";
        }
    }
    if (footnest == 0) {
    	issueMessage("Mismatched end of footnote (\"]\") bracket.");
    } else {
    	footnest--;
    }

@
Upon encountering a footnote while creating a multiple file HTML
document, we check whether a footnote document has already been
created.  If not, this is the first footnote; a footnote
document is created to receive it and any subsequent footnotes.

@<Create footnote file for first footnote in HTML@>= 
    if (foot == NULL) {
	footdocname = basename + "_foot" + ".html";
	foot = new ofstream(footdocname.c_str(), ios::out);
	writeHTMLDocumentPreamble(*foot, htitle + ": Notes", &footline);
	writeHTMLDocumentBodyStart(*foot, htitle + ": Notes", &footline);
	*foot << "<div class=\"bodycopy\">\n";
	footline++;
    }

@
HTML text entities (for example, ``\.{\&amp;}'' are transcribed
directly to the output stream, using the final semicolon to denote
the end of the entity.  We cannot allow the entity to pass through
|Output text character in HTML|, as it would insert an erroneous
space before the final semicolon if |frenchPunct| is set.

@<Transcribe HTML text entity literally to output@>=
    o += c;
    while (++cp < s.end()) {
        c = *cp;
        o += c;
        if (c == ';') {
            break;
        }
    }

@
Output a text character.  Since we've already handled quoting of
metacharacters, the only thing we need to worry about here is adding
a nonbreaking space around eligible punctuation if |frenchPunct| is set.

@<Output text character in HTML@>=
    if (!inmath && frenchPunct && ((c & 0xFF) == C_LEFT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK)
    	&& (cp != s.end()) && (!isspace(cp[1]))) {
    	o += c;
	o += "&nbsp;";
    } else if (!inmath && frenchPunct && (punctuation.find_first_of(c) != string::npos) &&
    	(((cp + 1) == s.end()) || isspace(cp[1]) ||
	 ((cp[1] & 0xFF) == C_RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK) || (cp[1] == ','))) {
	o += "&nbsp;";
	o += c;
    } else {
        o += c;
    }

@
HTML documents have a stereotyped preamble.  This function writes the
preamble at the start of a document we're writing to a stream.

@<Class definitions@>=
void HTMLGenerationSink::writeHTMLDocumentPreamble(ostream &os, string title, int *lineCounter) {
    int preambleLines = 13;
    deque <string>::iterator decl;
    
    if (strictCompliance) {
        os << "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\"\n";
        os << "    \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n";
    } else {
        os << "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n";
        os << "    \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n";
    }
    os << "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n";
    os << "<!--  Translated by " @, PRODUCT @, " " @, VERSION @,
    	  " (" @, REVDATE @, ") on " + stime + "-->\n";
    os << "<head>\n";
    os << "<title>" << title << "</title>\n";
    os << "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\" />\n";
    if (hastitle) {
    	os << "<meta name=\"description\" content=\"" << elideNewLines(title) << "\" />\n";
	preambleLines += 1;
    }
    if (hasauthor) {
    	os << "<meta name=\"author\" content=\"" << elideNewLines(hauthor) << "\" />\n";
	preambleLines += 1;
    }
    os << "<style type=\"text/css\">\n";
    os << "div.bodycopy {\n";
    os << "    margin-left: 15%;\n";
    os << "    margin-right: 10%\n";
    os << "}\n";
    
    headingAlign = "align=\"center\"";
    if (strictCompliance) {
        headingAlign = "class=\"c\"";
        os << "\n";
    
        os << "body {\n";
        os << "    background-color: #FFFFFF;\n";
        os << "    color: #000000;\n";
        os << "}\n\n";
    
        os << "br.cb {\n";
        os << "    clear: both;\n";
        os << "}\n\n";
    
        os << "h1.c, h2.c {\n";
        os << "    text-align: center;\n";
        os << "}\n\n";
        
        os << "hr.s {\n";
        os << "    width: 25%;\n";
        os << "    height: 2px;\n";
        os << "    background-color: #000000;\n";
        os << "    color: #010101;\n";
        os << "}\n\n";
        
        os << "img.m0 {\n";
        os << "    vertical-align: middle;\n";
        os << "    border: 0px;\n";
        os << "}\n\n";
        
        os << "p.justify {\n";
        os << "    text-align: justify;\n";
        os << "}\n\n";
        
        os << "p.right {\n";
        os << "    text-align: right;\n";
        os << "}\n\n";
        
        os << "p.center {\n";
        os << "    text-align: center;\n";
        os << "}\n\n";
            
        os << "span.footnote {\n";
        os << "    background-color: #FFFFA0;\n";
        os << "    color: #000000;\n";
        os << "}\n\n";
            
        os << "span.foottext {\n";
        os << "    font-family: Helvetica, Arial, sans-serif;\n";
        os << "    font-size: smaller;\n";
        os << "}\n\n";
         
        os << "table.chfile {\n";
        os << "    width: 100%;\n";
        os << "}\n\n";
         
        os << "table.chfile td.l {\n";
        os << "    width: 25%;\n";
        os << "    vertical-align: top;\n";
        os << "}\n\n";
       
        os << "table.chfile td.m {\n";
        os << "    width: 50%;\n";
        os << "    text-align: center;\n";
        os << "}\n\n";
         
        os << "table.chfile td.r {\n";
        os << "    width: 25%;\n";
        os << "    text-align: right;\n";
        os << "}\n\n";
         
        os << "table.ctable {\n";
        os << "    width: 80%;\n";
        os << "    margin-left: auto;\n";
        os << "    margin-right: auto;\n";
        os << "}\n\n";

        os << "table.ctable th {\n";
        os << "    width: 15%;\n";
        os << "    text-align: right;\n";
        os << "}\n\n";

        os << "table.ctable td.space {\n";
        os << "    width: 5%;\n";
        os << "}\n\n";

        os << "table.ctable td.name {\n";
        os << "    width: 80%;\n";
        os << "}\n\n";
        
        os << "table.footnote {\n";
        os << "    color: #000000;\n";
        os << "    background-color: #FFFFD0;\n";
        os << "    width: 25%;\n";
        os << "    float: right;\n";
        os << "    margin-left: 6px;\n";
        os << "}\n\n";
        
        os << "table.math {\n";
        os << "    color: #000000;\n";
        os << "    background-color: #FFA0A0;\n";
        os << "}\n";
	
	preambleLines += 98;
    }
    os << "</style>\n";
    
    @<Generate JavaScript for external links in Strict mode@>
    
    *lineCounter += preambleLines;
    
    for (decl = declarationsQueue.begin(); decl != declarationsQueue.end(); decl++) {
    	os << *decl << "\n";
	*lineCounter += 1;
    }
}

@
If we're generating Strict mode XHTML, we can't use the ``\.{target=}'' attribute
to open footnote links in another window.  (I shall decline comment upon what
I think of this decision in the interest of this document's suitability
for a family audience, and yes, I do know about the target module in XHTML 1.1.)
The only way to work around this is to use a ``\.{rel=}'' link, which is
permitted, which is flagged as specifying a target, then walking through the
document after it has been loaded with the following JavaScript kludge which
sets the target property for the link.  This is ugly, and regretfully requires
JavaScript in a document which would otherwise be able to dispense with it, but
this problem is not of my making.  If the user doesn't have JavaScript, the only
downside is that the footnote opens in the main document window and the user
has to use the ``Back'' button to return to the original document.

@<Generate JavaScript for external links in Strict mode@>=
    if (strictCompliance && (!singleFile)) {
        os << "<script type=\"text/javascript\">\n";
        os << "/* <![CDATA[ */\n";
        os << "    function externalLinks() {\n";
        os << "        if (!document.getElementsByTagName) {\n";
        os << "            return;\n";
        os << "        }\n";
        os << "        var anchors = document.getElementsByTagName(\"a\");\n";
        os << "        for (var i = 0; i < anchors.length; i++) {\n";
        os << "            var anchor = anchors[i], target;\n";
        os << "            if (anchor.getAttribute(\"href\") &&\n";
        os << "                anchor.getAttribute(\"rel\") &&\n";
        os << "                anchor.getAttribute(\"rel\").match(/^Target:/)) {\n";
        os << "                target = anchor.getAttribute(\"rel\")."
              "match(/(^Target:)(\\w+$)/);\n";
        os << "                anchor.target = target[2];\n";
        os << "            }\n";
        os << "        }\n";
        os << "    }\n";
        os << "/* ]]> */\n";
        os << "</script>\n";
	preambleLines += 19;
    }

@
After the HTML preamble comes the sequence which ends the header and begins
the body of the document.  We write this in a separate function to allow
declarations and links to be added before the end of the header.

@<Class definitions@>=
void HTMLGenerationSink::writeHTMLDocumentBodyStart(ostream &os, string title, int *lineCounter) {
    const int bodyStartLines = 3;
    
    os << "</head>\n";
    os << "\n";
    if (strictCompliance) {
    	if  (singleFile) {
            os << "<body>\n";
	} else {
            os << "<body onload=\"externalLinks();\">\n";
	}
    } else {
        os << "<body bgcolor=\"#FFFFFF\">\n";
    }
    *lineCounter += bodyStartLines;
}

@
HTML documents similarly close with a stereotyped postamble.  This
function appends the boilerplate to the end of a stream.

@<Class definitions@>=
void HTMLGenerationSink::writeHTMLDocumentPostamble(ostream &os, int *lineCounter) {
    const int postambleLines = 2;
    
    os << "</body>\n";
    os << "</html>\n";
    *lineCounter += postambleLines;
}

@
Each HTML chapter document contains a navigation panel
consisting of ``next'', ``previous'', and ``up'' buttons,
the first two of which are replaced by greyed-out versions for
the last and first chapters respectively.  These reference
the PNG image buttons created by |createNavButtons|.

|createNavigationPanel| writes a navigation panel to the
current chapter document, assumed open as output stream
|chap|.  The numbers of the previous and next chapters are
given by the |prev| and |next| arguments, which are zero if
no such chapter exists---in that case a disabled (greyed-out)
button with no link appears in the panel.

@<Class definitions@>=
void HTMLGenerationSink::createNavigationPanel(int prev, int next, bool inParagraph)
{
    // Previous chapter button.

    if (inParagraph) {
        if (strictCompliance) {
    	    *chap << "<p class=\"right\">\n";
        } else {
    	    *chap << "<p align=\"right\">\n";
        }
	chapline++;
    }
    if (prev != 0) {
        if (strictCompliance) {
    	    *chap << "<a href=\"" << basename << "_chap" << prev << ".html\">" <<
    	    	     "<img class=\"m0\" src=\"prev.png\" "
    	    	     "height=\"32\" width=\"32\" alt=\"Previous\" /></a> &nbsp; &nbsp;\n";
        } else {
    	    *chap << "<a href=\"" << basename << "_chap" << prev << ".html\">" <<
    	    	     "<img align=\"middle\" src=\"prev.png\" "
    	    	     "height=\"32\" width=\"32\" border=\"0\" alt=\"Previous\" /></a> &nbsp; &nbsp;\n";
        }
    } else {
        if (strictCompliance) {
            *chap << "<img class=\"m0\" src=\"prev_gr.png\" "
    	    	     "height=\"32\" width=\"32\" alt=\"\" /> &nbsp; &nbsp;\n";
        } else {
            *chap << "<img align=\"middle\" src=\"prev_gr.png\" "
    	    	     "height=\"32\" width=\"32\" border=\"0\" alt=\"\" /> &nbsp; &nbsp;\n";
        }
    }
    chapline++;

    // Up to table of contents button.

    if (strictCompliance) {
        *chap << "<a href=\"" << basename << ".html\"> <img class=\"m0\" src=\"up.png\" "
    	         "height=\"32\" width=\"32\" alt=\"Contents\" /></a> &nbsp; &nbsp;\n";
    } else {
        *chap << "<a href=\"" << basename << ".html\"> <img align=\"middle\" src=\"up.png\" "
    	         "height=\"32\" width=\"32\" border=\"0\" alt=\"Contents\" /></a> &nbsp; &nbsp;\n";
    }
    chapline++;

    // Next chapter button.

    if (next != 0) {
        if (strictCompliance) {
    	    *chap << "<a href=\"" << basename << "_chap" << next << ".html\">" <<
    	    	     "<img class=\"m0\" src=\"next.png\" "
    	    	     "height=\"32\" width=\"32\" alt=\"Next\" /></a>\n";
        } else {
    	    *chap << "<a href=\"" << basename << "_chap" << next << ".html\">" <<
    	    	     "<img align=\"middle\" src=\"next.png\" "
    	    	     "height=\"32\" width=\"32\" border=\"0\" alt=\"Next\" /></a>\n";
        }
    } else {
        if (strictCompliance) {
            *chap << "<img class=\"m0\" src=\"next_gr.png\" "
    	    	     "height=\"32\" width=\"32\" alt=\"\" />\n";
        } else {
            *chap << "<img align=\"middle\" src=\"next_gr.png\" "
    	    	     "height=\"32\" width=\"32\" border=\"0\" alt=\"\" />\n";
        }
    }
    chapline++;
    
    if (inParagraph) {
    	*chap << "</p>\n";
	chapline++;
    }
}


@
This |static| function creates the PNG files for the buttons in the
navigation panel.  It is called when we are creating a multi-file
document tree and encounter the first chapter title.

@<Class definitions@>=
void HTMLGenerationSink::createNavButtons(void) {
    @<Definition of navigation buttons in HTML@>;

#ifdef FOOTNOTE_BUTTON_NEEDED
    createNavButton("foot", d_foot, sizeof d_foot); 	    	// Footnote
#endif
    createNavButton("next", d_next, sizeof d_next); 	    	// Next
    createNavButton("prev", d_prev, sizeof d_prev); 	    	// Previous
    createNavButton("up", d_up, sizeof d_up);	    	    	// Up (to Table of Contents)
    createNavButton("next_gr", d_next_gr, sizeof d_next_gr);	// Greyed out Next (for last chapter)
    createNavButton("prev_gr", d_prev_gr, sizeof d_prev_gr);	// Greyed out Prev (for first chapter)
}

@
The PNG navigation buttons in the HTML chapter documents are
created by calling |createNavButton| for each button definition
embedded by the \.{\#include} in the following section. 
|createNavButtons| calls this function for each button file. 
There's actually nothing at all PNG-specific about this
function---it just writes out arbitrary binary data from memory
to an |ofstream|.

@<Class definitions@>=
void HTMLGenerationSink::createNavButton(string button, const unsigned char source[], int length)
{
    string buttonFile;
    ofstream *bf;

    buttonFile = button + ".png";
    bf = new ofstream(buttonFile.c_str(), ios::out | ios::binary);
    bf->write(reinterpret_cast<const char *>(source), length);
    bf->close();
}

@
When generating HTML output, we want to include
language-neutral navigation buttons.  These are small PNG
images, which present a problem if we wish to distribute this
program in text form.  To avoid the need for system-specific
text to binary and/or archive extraction utilities, we simply
embed the PNG images in this file as binary data definitions
and write them into the HTML document directory.

@<Definition of navigation buttons in HTML@>=
#include "buttons.h"
