@** LaTeX Generation.

This filter translates parsed body copy (emitted by |etextBodyParser|)
into \LaTeX\ source code, which it passes down the pipeline.

@<Class definitions@>=
class LaTeXGenerationFilter : public textFilter {
private:@/
    bool italics, inmath, quoth, hastitle, hasauthor,
    	 intable, firstchap;
    int footnest;
    textSubstituter transformer;
    
    string quoteLaTeXString(string s);
    
    void emitq(string s) {
    	emit(quoteLaTeXString(s));
    }
    
    void generateAlignedParagraph(string envtype, char bracket, string text,
    	    string terminator = "\\\\");
	    
    bool isSubstitution(string s);
    
public:@/
    LaTeXGenerationFilter() {
    	italics = inmath = quoth = false;
	hastitle = hasauthor = false;
	intable = firstchap = false;
	footnest = 0;
    }

    virtual ~LaTeXGenerationFilter() {
    }

    string componentName(void) {
        return "LaTeXGenerationFilter";
    }
    
    void put(string s);
};

@
The |put| method of the LaTeXGenerationFilter wraps \LaTeX\ commands
around the line-level structure of the text to achieve the
desired formatting.  Since almost all of the real work is
done upstream (by |etextBodyParserFilter|) and downstream
(by |quoteLaTeXString|) there is actually little that needs
doing here.

@<Class definitions@>=
void LaTeXGenerationFilter::put(string s) {
    bodyState state = DecodeBodyState(s[0]);
    char bracket = s[1];
    string text = s.substr(2);

    switch (state) {
	case BeginText:
	    @<Generate start of document in LaTeX@>;
	    
	case Declarations:
	    @<Process declarations in LaTeX@>;

	case DocumentTitle:
	    @<Process document title in LaTeX@>;

	case Author:
	    @<Process author in LaTeX@>;

	case ChapterNumber:
	    break;	    	    	// Chapter numbers are totally ignored

	case ChapterName:
	    @<Process chapter name in LaTeX@>;

	case InTextParagraph:
	    @<Generate justified text paragraph in LaTeX@>;

	case InBlockQuote:
	    generateAlignedParagraph("quote", bracket, text, "");
	    break;

	case InRaggedRight:
	    generateAlignedParagraph("flushleft", bracket, text);
	    break;

	case InRaggedLeft:
	    generateAlignedParagraph("flushright", bracket, text);
	    break;

    	case InPreformattedTable:
	    if (bracket == Begin) {
	    	intable = true;
	    }
	    generateAlignedParagraph("verbatim", bracket, text, "");
	    if (bracket == End) {
	    	intable = false;
	    }
	    break;

	case InCentred:
	    generateAlignedParagraph("center", bracket, text);
	    break;

	case EndOfText:
	    emit("\\end{document}");
	    if (verbose) {
	    	cerr << "LaTeX: " << getLineNumber() << " lines output.\n";
	    }
	    break;

	default:
	    cerr << "*** State " << stateNames[state] << " " << bracket <<
		" not handled in LaTeXGenerationFilter ***\n";
	    exit(1);
    }
}

@
Generate the boilerplate at the start of a \LaTeX\ document and
declarations appropriate for the type of document we're
about to write.

@<Generate start of document in LaTeX@>=
    emit("\\documentclass{book}");
    {
	time_t t = time(NULL);
	string stime = ctime(&t);
	stime = stime.substr(0, stime.length() - 1);

	emit("% Translated by " PRODUCT " " VERSION " (" REVDATE ") on " + stime);
    }
    if (babelon) {
	emit("\\usepackage[" + babelang + "]{babel}");
	emit("\\usepackage[latin1]{inputenc}");
	emit("\\usepackage[T1]{fontenc}");
    } else {
	if (frenchPunct) {
	    emit("\\frenchspacing");
	}
    }
    break;

@
Any \LaTeX-specific special commands appearing in
declaration blocks before the title are output in
the preamble of the document where they may load
packages and/or make definitions in global scope.

@<Process declarations in LaTeX@>=
    if (bracket == Body) {
    	assert(etextBodyParserFilter::isLineSpecial(text));
	if (!isSubstitution(text)) {
	    emit(etextBodyParserFilter::specialCommand(text));
	}
    }
    break;

@
For the document title we need only wrap it in a \LaTeX\ \.{title}
declaration in the preamble and set |hastitle| to remind us to emit
a \.{maketitle} command at the start of the document body.

@<Process document title in LaTeX@>=
    switch (bracket) {
	case Begin:
	    emit("\\title{");
	    break;

	case Body:
	    emitq(text);
	    break;

	case End:
	    emit("}");
	    hastitle = true;
	    break;

	case Void:
	    hastitle = false;
	    break;
    }
    break;

@
Similarly, the author is wrapped in a \LaTeX\ \.{author}
declaration in the preamble.  When we see the |End| bracket
for the author specification (or the |Void| bracket if no
author is given), it's time to close the preamble and
begin the body of the document.  If either a title or
author were specified, we then need to emit a \.{maketitle}
command to create the title.  Since \.{maketitle} requires
both a title and author declaration, if either is missing we
supply a blank one before closing the preamble.

@<Process author in LaTeX@>=
    switch (bracket) {
	case Begin:
	    emit("\\author{");
	    break;

	case Body:
	    emitq(text);
	    break;

	case End:
	    emit("}");
	    hasauthor = true;@/@,
	    // Note fall-through

	case Void:
	    if (hastitle || hasauthor) {
		if (!hastitle) {
		    emit("\\title{}");
		}
		if (!hasauthor) {
		    emit("\\author{}");
		}
	    }
	    emit("\\begin{document}");
	    if (hastitle || hasauthor) {
		emit("\\maketitle");
	    }
	    break;
    }
    break;

@
Chapter names cause a \.{chapter} command to be generated with the
chapter title as its argument.  |Void| chapter names generate
chapters with blank titles.

@<Process chapter name in LaTeX@>=
    switch (bracket) {
	case Begin:
	    if (!firstchap) {
	    	firstchap = true;
		emit("\\tableofcontents");
	    }
	    emit("\\chapter{");
	    break;

	case Body:
	    emitq(text);
	    break;

	case End:
	    emit("}");
	    break;

	case Void:
	    emit("\\chapter{}");
	    break;
    }
    break;

@
Regular justified text paragraphs don't need any special wrapper, as
they're the default in \LaTeX.  We output a blank line before the
first line of the body to break the previous paragraph and reset
the quote parity so the first ASCII quote in the paragraph will be
turned into an open quote.
    
@<Generate justified text paragraph in LaTeX@>=
    switch (bracket) {
	case Begin:
	    emit("");
	    quoth = false;
	    break;

	case Body:
	    if (etextBodyParserFilter::isLineSpecial(text)) {
	    	if (!isSubstitution(text)) {
	    	    emit(etextBodyParserFilter::specialCommand(text));
		}
	    } else {
	    	emitq(text);
	    }
	    break;

	case End:
	case Void:
	    break;
    }
    break;


@
This function handles the various kinds of aligned paragraphs we encounter
in a document.  It wraps the contents of the paragraph in a
\LaTeX\ environment of the type specified by |envtype|.

@<Class definitions@>=
void LaTeXGenerationFilter::generateAlignedParagraph(string envtype, char bracket, string text,
    	    string terminator) {
    switch (bracket) {
	case Begin:
	    emit("");
	    emit("\\begin{" + envtype + "}");
	    quoth = false;
	    break;

	case Body:
	    if (etextBodyParserFilter::isLineSpecial(text)) {
	    	if (!isSubstitution(text)) {
	    	    emit(etextBodyParserFilter::specialCommand(text));
		}
	    } else {
	    	emit(quoteLaTeXString(text) + terminator);
	    }
	    break;

	case End:
	    emit("\\end{" + envtype + "}");
	    break;
	    
	case Void:
	    break;
    }
}

@
Translate text string |s| into \LaTeX, quoting metacharacters
and expanding Latin-1 characters as required.  This is where
we handle switching to and from math mode, toggling italic,
expanding ellipses and em-dashes, and recognising footnotes.

@<Class definitions@>=
string LaTeXGenerationFilter::quoteLaTeXString(string s)
{
    string::iterator cp;
    string o = "";
    int c;
    static const string mathModeQuoted = "|<>",
    	    	    	punctuation = "?!:;",	    	// Punctuation set after space for |frenchPunct|
			quotedCharacters = "$&%#",
			quotedTextCharacters = "{}";
    
    for (cp = s.begin(); cp < s.end(); cp++) {
    	c = (*cp) & 0xFF;
	
	if (c < ' ') {
	    @<Quote control character in LaTeX@>;
	    
	} else if (c > 160) {
	    @<Translate ISO graphic character in LaTeX@>;
	
	} else if (c >= ' ' && c <= '~') {
            if (!inmath && !intable && c == '_') {
	    	@<Toggle italic text mode in LaTeX@>;
	    } else if (!intable && c == '\\' &&
	    	       ((cp + 1) < s.end()) && ((cp[1] == '(') || (cp[1] == ')'))) {
	    	@<Toggle math mode in LaTeX@>;
	    } else if (!inmath && !intable && ((cp + 2) < s.end()) &&
	    	       ((c == '[') || ((c == ' ') && (cp[1] == '[')) ||
                                   ((c == ' ') && (cp[1] == ' ') && (cp[2] == '[')))) {
		@<Begin footnote in LaTeX@>;
	    } else if (!inmath && !intable && c == ']') {
	    	@<End footnote in LaTeX@>;
	    } else if (!inmath && !intable && (c == '-') && ((cp + 1) < s.end()) && (cp[1] == '-')) {
	    	@<Translate em-dash in LaTeX@>;
    	    } else if (!inmath && !intable && (c == '.') && ((cp + 2) < s.end()) &&
	    	    	(cp[1] == '.') && (cp[2] == '.')) {
	    	@<Translate ellipsis in LaTeX@>;
	    } else if (!intable && ((c == '~') || ((!inmath) && (c == '^')))) {
	    	@<Quote ASCII character as verbatim in LaTeX@>;
	    } else if ((!inmath && !intable) && (mathModeQuoted.find_first_of(c) != string::npos)) {
	    	@<Quote character as math mode in LaTeX@>;
	    } else if (!inmath && !intable && c == '"') {
	    	@<Convert ASCII quotes to open and close quotes in LaTeX@>;
	    } else {
	    	@<Output ASCII text character in LaTeX@>;
	    }
	}
        /* Note that other characters, specifically those in the
           range from 127 through 160, get dropped. */
    }
    o = transformer.substitute(o);  	    // Apply substitutions, if any
    return o;
}

@
This is a control character.  Emit as \.{\caret}{\it letter} unless it is
considered as white space (for example, carriage return and
line feed), in which case it's sent directly to the output.
	       
@<Quote control character in LaTeX@>=
   if (isspace(c)) {
        o += c;
    } else {
        o += "\\verb+^";
	o += ('@@' + c);
	o += '+';
    }

@
This is a graphic character belonging to the ISO 8859 set with
character codes between |0xA0| and |0xFF|.  Translate it into
the best \LaTeX\ equivalent or pass it on to be handled by the
\.{babel} package if |babelon| is set.
    
@<Translate ISO graphic character in LaTeX@>=
    if (babelon) {
    	o += c;
    } else {
    	o += texform[c - 161];
    }

@
The underscore character, ``\.{\_}'', toggles text between the normal
roman and italic fonts.
  
@<Toggle italic text mode in LaTeX@>=
    italics = !italics;
    if (italics) {
        o += "{\\it ";
    } else {
        o +="}";
    }

@
We use the same sequences as \LaTeX, ``\.{\bslash(}'' and
``\.{\bslash)}'', to toggle between text and math mode, so we
can simply emit the sequence unmodifed.  We need to take note
of it, however, to keep track of whether we're in math mode as
that affects handling of some other sequences.
    
@<Toggle math mode in LaTeX@>=
    o += c;
    inmath = cp[1] == '(';

@
Footnotes appear in-line, within [square brackets].  Translate them to
a \LaTeX\ footnote environment.  Footnotes aren't supposed to be
nested in Etexts, but \LaTeX\ handles them just fine.

@<Begin footnote in LaTeX@>=
    footnest++;
    o += "\\footnote{";
    if ((c == ' ') && ((cp + 1) < s.end())) {
        if (cp[1] == ' ') {
            cp++;
        }
        cp++;
    }

@
Close a footnote when the right bracket is encountered.
    
@<End footnote in LaTeX@>=
    o += '}';
    if (footnest == 0) {
    	issueMessage("Mismatched end of footnote (\"]\") bracket.");
    } else {
    	footnest--;
    }

@
Two adjacent hyphens, ``\.{-}\.{-}'' denote an {\it em} dash in
an ASCII Etext.  Translate this sequence into three hyphens as
used by \LaTeX.

@<Translate em-dash in LaTeX@>=
    o += "---";
    cp++;

@
Three consecutive periods are translated into a
\.{\bslash ldots} ellipsis.
    
@<Translate ellipsis in LaTeX@>=
   o += "\\ldots";
   if (cp[2] == ' ') {
       o += " ";
   } else {
      o += "\\ ";
   }
   cp += 2;

@
This code handles tilde and circumflex characters (the latter only
when not in math mode), which must be quoted in a
\.{\bslash verb} sequence to appear in text.

@<Quote ASCII character as verbatim in LaTeX@>=
    o += "\\verb+";
    o += c;
    o += '+';
    
@
The greater, less, and vertical bar symbols cannot appear in regular
text in \LaTeX; output them in math mode.

@<Quote character as math mode in LaTeX@>=
    o += '$';
    o += c;
    o += '$';

@
ASCII quote characters are translated into open and close quote symbols.
Note that the flag |quoth| is unconditionally reset at the end of
a paragraph so mismatched quotes won't propagate beyond one paragraph.
This allows continued quotes in multiple paragraphs to work properly.

@<Convert ASCII quotes to open and close quotes in LaTeX@>=
    o += quoth ? "''" : "``";
    quoth = !quoth;
    
@
Output a text character.  Some \LaTeX\ metacharacters require
backslash quoting in any mode, others only outside math mode.

@<Output ASCII text character in LaTeX@>=
   if (!inmath && frenchPunct && (punctuation.find_first_of(c) != string::npos) &&
    	(((cp + 1) == s.end()) || isspace(cp[1]) ||
	 ((cp[1] & 0xFF) == C_RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK) || (cp[1] == ','))) {
	o += "{\\small~}";
	o += c;
    } else {
    
        /* Characters requiring protection against interpretation
           as control sequences. */

        if (quotedCharacters.find_first_of(c) != string::npos) {
            o += '\\';
        }
        if (!inmath && (quotedTextCharacters.find_first_of(c) != string::npos)) {
            o += '\\';
        }
        o += c;
    }

@
Special commands which define text substitutions are recognised by
this method, parsed, and added to the substitution list.

@<Class definitions@>=
bool LaTeXGenerationFilter::isSubstitution(string cmd)
{
    string s = etextBodyParserFilter::specialCommand(cmd);
    bool isSub = false, bogus = true;
    unsigned int n, m, l;
    char delim;
    
    if (s.find("Substitute ") == 0) {
    	isSub = true;
    	s = s.substr(11);
	n = s.find_first_not_of(' ');
	if (n != string::npos) {
	    delim = s[n];
	    m = s.find(delim, n + 1);
	    if (m != string::npos) {
	    	l = s.find(delim, m + 1);
		if (l != string::npos) {
		    bogus = false;
		    transformer.addSubstitution(s.substr(n + 1, (m - n) - 1),
		    	s.substr(m + 1, (l - m) - 1));
		}
	    }
	}
	if (bogus) {
	    issueMessage("Invalid LaTeX Substitute special", cerr);
	    issueMessage(auditFilter::quoteArbitraryString(cmd), cerr);
	}
    }
    return isSub;
}
