#! /usr/bin/perl -CA
#                ^^^   This allows Unicode command-line arguments to be
#                      accepted if the underlying system supports it.
#                      If it causes an error, your version of Perl does
#                      not support this feature.  You can remove the option
#                      and continue to use the program with all other forms
#                      of arguments.

        use utf8;

=head1 NAME

unum - Interconvert numbers, Unicode, and HTML/XHTML characters

=head1 SYNOPSIS

B<unum> I<argument...>

=head1 DESCRIPTION

The B<unum> program is a command line utility which allows you to
convert decimal, octal, hexadecimal, and binary numbers; Unicode
character and block names; and HTML/XHTML character reference names and
numbers into one another.  It can be used as an on-line special
character reference for Web authors.

=head2 Arguments

The command line may contain any number of the following
forms of I<argument>.

=over 10

=item S<>123

Decimal number.

=item S<>0371

Octal number preceded by a zero.

=item S<>0x1D351

Hexadecimal number preceded by C<0x>.  Letters may be upper or
lower case, but the C<x> must be lower case.

=item S<>0b110101

Binary number.

=item b=I<block>

Unicode character blocks matching I<block> are listed.
The I<block> specification may be a regular expression.
For example, C<b=greek> lists all Greek character blocks
in Unicode.

=item c=I<char>...

The Unicode character codes for the characters I<char>... are listed.
If the first character is not a decimal digit and the second not an
equal sign, the C<c=> may be omitted.

=item h=I<entity>

List all HTML/XHTML character references matching I<entity>, which may
be a regular expression.  Matching is case-insensitive, so C<h=alpha>
finds both C<&Alpha;> and C<&alpha;>.  If the reference is composed of
multiple Unicode code points, the components are printed after the name
of the composed character reference.

=item '&#I<number>;&#xI<hexnum>;...'

List the characters corresponding to the specified HTML/XHTML
character entities, which may be given in either decimal or
hexadecimal.  Note that the "x" in XHTML entities must be lower case.
On most Unix-like operating systems, you'll need to quote the argument
so the ampersand, octothorpe, and semicolon aren't interpreted by the
shell.

=item l=I<block>

List all Unicode blocks matching I<block> and all characters
within each block; C<l=goth> lists the C<Gothic> block
and the 32 characters it contains.

=item n=I<name>

List all Unicode character whose names match I<name>, which may be
a regular expression.  For example, C<n=telephone> finds the twelve
Unicode characters for telephone symbols.

=item utf8=I<number>

Treating the number (which may be specified as either decimal,
octal, hexadecimal, or binary, as for numeric arguments) as a
stream of from one to four bytes, decode the bytes as the
UTF-8 representation of a character.  For example, the
specification "utf8=0xE298A2" decodes to Unicode code
point 0x2622, the radioactive sign.

=back

=head2 Options

=over 10

=item --nent

When showing an HTML character reference, always show its numerical 
form (for example, &#8212;), even if it has a named character 
reference.

=item --utf8

Show UTF-8 encoding of characters as a byte sequence in a
hexadecimal number.  This is the same format as is accepted
by the utf8= argument.  The option applies to the display of
all arguments which follow on the command line.

=back

=head2 Output

For number or character arguments, the value(s) are listed in
all of the input formats, save binary.

   Octal  Decimal      Hex        HTML    Character   Unicode
     056       46     0x2E    &period;    "."         FULL STOP

If the terminal font cannot display the character being listed,
the "Character" field will contain whatever default is shown in
such circumstances.  Control characters are shown as a Perl
hexadecimal escape.  If multiple HTML named character references
map to the same Unicode code point, all are shown separated by
commas.

Unicode blocks are listed as follows:

    Start        End  Unicode Block
   U+2460 -   U+24FF  Enclosed Alphanumerics
  U+1D400 -  U+1D7FF  Mathematical Alphanumeric Symbols


=head1 VERSION

This is B<unum> version 3.4-14.0.0, released on September 20th, 2021.
The current version of this program is always posted at
http://www.fourmilab.ch/webtools/unum/.

=head1 AUTHOR

John Walker

http://www.fourmilab.ch/

=head1 BUGS

Specification of Unicode characters on the command line requires
an operating system and shell which support that feature and a
version of Perl with the B<-CA> command line option
(v5.8.5 has it, but v5.8.0 does not; I don't know in which
intermediate release it was introduced).  If your version of
Perl does not implement this switch, you'll have to remove it
from the C<#!> statement at the top of the program, and Unicode
characters on the command line will not be interpreted correctly.

If you specify a regular expression, be sure to quote the argument
if it contains any characters the shell would otherwise interpret.

If you run B<perldoc> on the compressed version of the program,
a large amount of gibberish will be displayed after the end of the
embedded documentation.  B<perldoc> gets confused by sequences in
the compressed data table and tries to interpret it as documentation.
This doesn't happen with the uncompressed version.

Please report any bugs to bugs@fourmilab.ch.

=head1 COPYRIGHT

This is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

=cut

    use strict;
    use warnings;

    sub usage {
        print << "EOD";
usage: unum arg...
    Arguments:
        147               Decimal number
        0371              Octal number
        0xfa75            Hexadecimal number (letters may be A-F or a-f)
        0b11010011        Binary number
        '&#8747;&#x3c0;'  One or more XHTML numeric entities (hex or decimal)
        utf8=0xc397       Character encoded as UTF-8 byte stream (any number format)
        xyz               The characters xyz (non-digit)
        c=7Y              The characters 7Y (any Unicode characters)
        b=cherokee        List Unicode blocks containing "CHEROKEE" (b=. to list all)
        h=alpha           List XHTML entities containing "alpha" (h=. to list all)
        n=aggravation     Unicode characters with "AGGRAVATION" in the name
        n=^greek.*rho     Unicode characters beginning with "GREEK" and containing "RHO"
        l=gothic          List all characters in matching Unicode blocks

    Options:
        --nent            Always show HTML character entities as numeric
        --utf8            Show UTF-8 encoding of characters

        All name queries are case-insensitive and accept regular
        expressions.  Be sure to quote regular expressions if they
        contain characters with meaning to the shell.

        Run perldoc on this program or visit:
            http://www.fourmilab.ch/webtools/unum/
        for additional information.
    Version 3.4-14.0.0, September 2021
EOD
    }

    my $debug_decompress = 0;           # Debug code point table decompression?

    my (@HTML_CHARACTER_REFERENCES, @HTML_COMPOSED_CHARACTER_REFERENCES,
        %UNICODE_NAMES, @UNICODE_BLOCKS);

    binmode(STDOUT, ":utf8");

    if ($#ARGV < 0) {
        usage();
        exit(0);
    }

    init_names();

    my $utf8o = 0;                      # Show UTF-8 encoding ?
    my $numHTMLent = 0;                 # Always generate numeric HTML entities ?

    my ($chartitle, $blocktitle) = (0, 0);
    my $arg = 0;
    while ($#ARGV >= 0) {
        my $n = shift();

        $arg++;

        #   Process options
        if ($n eq "--utf8") {           # --utf8  Show UTF-8 encoding
            $utf8o = 1;
            next;
        } elsif ($n eq "--nent") {      # --nent  Always generate numeric HTML entities
            $numHTMLent = 1;
            next;
        }
=begin test_UTF8
        elsif ($n eq "--test8") {
            test_UTF8();
            next;
        }
=end test_UTF8
=cut

        if ($n =~ m/^\d/) {

            #   Number                  List numeric and character representations

            #   Argument is a number: use oct() to convert to binary
            $n = oct($n) if ($n =~ m/^0/);

        } elsif ($n =~ m/^(b|l)=(.+)/) {

            #   b=<block name>          List Unicode blocks matching name

            my $bl = $1;
            my $cpat = qr/$2/i;
            my $listall = $bl =~ m/l/i;

            my $k;
            for $k (@UNICODE_BLOCKS) {
                if ($k->[2] =~ m/$cpat/) {
                    if (!$blocktitle) {
                        $chartitle = 0;
                        $blocktitle = 1;
                        print("   Start        End  Unicode Block\n");
                    }
                    printf("%8s - %8s  %s\n",
                        sprintf("U+%04X", $k->[0]),
                        sprintf("U+%04X", $k->[1]),
                        $k->[2]);

                    if ($listall) {
                        for (my $i = $k->[0]; $i <= $k->[1]; $i++) {
                            showchar($i);
                        }
                    }
                }
            }
            next;

        } elsif ($n =~ m/^h=(.+)/) {

            #   h=<character name>      List HTML character entities matching name

            my $cpat = qr/$1/i;

            #   Scan through the table of names and build a hash of all
            #   the code points that matches map to.  Then sort those
            #   code points in ascending order and display them,
            #   counting on showchar() to show all of the character
            #   reference names which mapped from the code points
            #   displayed.

            my %htmlCodePoints;
            for (my $i = 0; $i < scalar(@HTML_CHARACTER_REFERENCES); $i += 2) {
                if ($HTML_CHARACTER_REFERENCES[$i] =~ m/$cpat/) {
                    $htmlCodePoints{$HTML_CHARACTER_REFERENCES[$i + 1]} = 1;
                }
            }

            my $k;
            for $k (sort {$a <=> $b} keys(%htmlCodePoints)) {
                showchar($k);
            }

            #   Now we must scan through the table of composed character
            #   references.  These are logical characters which are made
            #   up by combining multiple code points.

            for (my $i = 0; $i < scalar(@HTML_COMPOSED_CHARACTER_REFERENCES); $i += 2) {
                if ($HTML_COMPOSED_CHARACTER_REFERENCES[$i] =~ m/$cpat/) {
                    my $hcp = $HTML_COMPOSED_CHARACTER_REFERENCES[$i + 1];
                    print("                                &$HTML_COMPOSED_CHARACTER_REFERENCES[$i]; =\n");
                    $chartitle = 0;
                    while ($hcp =~ s/\s*(\d+)(?:,\s*)?//) {
                        $k = $1;
                        showchar($k);
                    }
                    $chartitle = 0;
                }
            }
            next;

        } elsif ($n =~ m/^n=(.+)/) {

            #   n=<character name>      List Unicode characters matching name

            my $cpat = qr/$1/i;

            #   The following would be faster if we selected matching
            #   characters into an auxiliary array and then sorted
            #   the selected ones before printing.  In fact, the time it
            #   takes to sort the entire list is less than that consumed
            #   in init_names() loading it, so there's little point bothering
            #   with this refinement.
            my $k;
            for $k (sort {oct("0x$a") <=> oct("0x$b")} keys(%UNICODE_NAMES)) {
                if ($UNICODE_NAMES{$k} =~ m/$cpat/) {
                    showchar(oct("0x$k"));
                }
            }
            next;

        } elsif ($n =~ m/^utf8=(.+)/) {

            #   utf8=<number>           UTF-8 character encoded as number

            my $u = $1;
            #   Argument is a number: use oct() to convert to binary if leading 0
            $u = oct($u) if ($u =~ m/^0/);

            $n = decode_UTF8($u);

        } elsif ($n =~ m/^&#/) {

            #   '&#NNN;&#xNNNN;...'     One or more XHTML numeric entities

            my @htmlent;
            while ($n =~ s/&#(x?[0-9A-Fa-f]+);//) {
                my $hch = $1;
                $hch =~ s/^x/0x/;
                push(@htmlent, $hch);
            }
            unshift(@ARGV, @htmlent);
            next;

        } else {

            #   =<char>... or c=<char>...   List code for one or more characters

            #   If argument is an equal sign followed by a single
            #   character, take the second character as the argument.
            #   This allows treating digits as characters to be looked
            #   up.
            $n =~ s/^c?=(.+)$/$1/i;

            while ($n =~ s/^(.)//) {
                showchar(ord($1));
            }
            next;
        }

        showchar($n);
    }

    #   Show a numeric code in all its manifestations

    sub showchar {
        my ($n) = @_;


        my $ch = ((($n >= 32) && ($n < 128)) || ($n > 160)) ?
            chr($n) :
            sprintf("\\x{%X}", $n);

        #   Determine the Unicode character code as best we can

        my $u = uname($n);
        if (!defined($u)) {
            $u = ublock($n);
            if (defined($u)) {
                $u = sprintf("%s U+%05X", $u, $n);
            } else {
                $u = sprintf("Undefined U+%05X", $n);
            }
        }

        my $ut8l = "";
        if ($utf8o) {
            $ut8l = "       UTF-8  ";
        }

        if (!$chartitle) {
            $blocktitle = 0;
            $chartitle = 1;
            print("   Octal  Decimal      Hex        HTML$ut8l    Character   Unicode\n");
        }

        #   With the advent of HTML5, (aka, W3C meets crap sandwich), the mapping
        #   of named character references to Unicode code points is many-to-many.
        #   If there is more than one character reference name for the given
        #   code point, list all of them separated by commas.  They are listed
        #   in the vaguely alphabetical order given in the W3C table.  We only
        #   display direct mappings of code points to named character references,
        #   not composed character references of which the code point is a
        #   part.

        my $htmlcr;
        if (!$numHTMLent) {
            for (my $i = 0; $i < scalar(@HTML_CHARACTER_REFERENCES); $i += 2) {
                if ($HTML_CHARACTER_REFERENCES[$i + 1] == $n) {
                    if ($htmlcr) {
                        $htmlcr .= ",";
                    }
                    $htmlcr .= "&" . $HTML_CHARACTER_REFERENCES[$i] . ";";
                }
            }
        }
        if (!$htmlcr) {
            $htmlcr = sprintf("&#%d;", $n);
        }

        my $u8e = "";
        if ($utf8o) {
            $u8e = sprintf("  %10s  ", sprintf("0x%X", encode_UTF8($n)));
        }

        printf("%8s %8d %8s %11s%s    %-8s    %s\n",
            sprintf("0%lo", $n),
            $n,
            sprintf("0x%X", $n),
            $htmlcr,
            $u8e,
            sprintf("\"%s\"", $ch),
            $u);
    }

    #   Decode a stream of bytes, stored in an integer, into a
    #   single UTF-8 character.  Leading zero bytes are
    #   ignored.  The following validations are performed
    #   and warning messages issued in case of violations of
    #   the UTF-8 standard.
    #
    #       1.  No extraneous bytes following UTF-8 character
    #       2.  No continuation code in first byte
    #       3.  All continuation bytes have 0b10 as the two
    #           highest bits
    #       4.  No bytes forbidden or undefined in UTF-8
    #           (0xC0, 0xC1, 0xF5-0xFF)
    #       5.  No "overlong" encoding of code points into
    #           more bytes than necessary.
    #
    #   The code point of the UTF-8 character is returned as
    #   an integer.
    #
    #   Test cases:
    #           0x0                 NULL
    #           0x4B                LATIN CAPITAL LETTER K
    #           0xC397              MULTIPLICATION SIGN
    #           0xE298A2            RADIOACTIVE SIGN
    #           0xF09F918C          OK HAND SIGN

    sub decode_UTF8 {

        my ($u) = @_;

        #   Now we run the gauntlet of our very exacting UTF-8
        #   decoder.  Many UTF-8 decoders are tolerant of
        #   sloppiness, but we are not.  That would compromise
        #   our mission of accepting only well-formed input and
        #   diagnosing errors.

        my $err = 0;
        my $n;
        my @bytes;
        my $m = 0xFF000000;
        for (my $i = 0; $i < 4; $i++) {
            my $b = ($u & $m) >> (8 * (3 - $i));
            if (($b != 0) || ($i == 3)) {
                push(@bytes, ($u & $m) >> (8 * (3 - $i)));
            }
            $m = $m >> 8;
        }

        #   Verify that the first byte is not a continuation
        #   code.
        if (($bytes[0] & 0b1100_0000) == 0b1000_0000) {
            printf("First byte is a continuation code " .
                   "in UTF-8 code 0x%X\n", $u);
            $err++;
        }

        #   If there is more than a single byte of UTF-8
        #   code, validate that all continuation bytes
        #   have the correct 0b10xx_xxxx high bits.
        if (scalar(@bytes) > 1) {
            for (my $i = 1; $i < scalar(@bytes); $i++) {
                if (($bytes[$i] & 0b1100_0000) != 0b1000_0000) {
                    printf("Incorrect continuation code in byte $i " .
                           "of UTF-8 code 0x%X\n", $u);
                    $err++;
                }
            }
        }

        #   Verify that no byte contains a value forbidden in
        #   a valid UTF-8 stream.
        for (my $i = 0; $i < scalar(@bytes); $i++) {
            my $b = $bytes[$i];
            if (($b == 0xC0) || ($b == 0xC1) ||
                ($b >= 0xF5)) {
                printf("Byte $i contains invalid UTF-8 code 0x%X\n", $b);
                $err++;
            }
        }

        #   Bail out on gross encoding errors.  This avoids blundering
        #   into undefined variable references and other horrors in
        #   the following decoder.
        if ($err > 0) {
            printf("Erroneous UTF-8 encoding: returning code point 0\n");
            @bytes = ( 0 );
        }

        #   Decode the bytes according to the length specified
        #   by the high-order bits in the first byte.

        if (($bytes[0] & 0b1000_0000) == 0) {                   # Code points 0000 - 007F
            $n = $bytes[0];
            if (scalar(@bytes) > 1) {
                printf("Excess byte(s) in UTF-8 code 0x%X: 1 byte expected, %d specified\n",
                       $u, scalar(@bytes));
            }

        } elsif (($bytes[0] & 0b1110_0000) == 0b1100_0000) {    # Code points 0080 - 07FF
            $n = (($bytes[0] & 0b1_1111) << 6) | ($bytes[1] & 0b11_1111);
            if (($bytes[0] & 0b1_1111) == 0) {
                printf("Overlong 2 byte UTF-8 code 0x%X for code point 0x%X\n", $u, $n);
            }
            if (scalar(@bytes) > 2) {
                printf("Excess byte(s) in UTF-8 code 0x%X: 2 bytes expected, %d specified\n",
                       $u, scalar(@bytes));
            }

        } elsif (($bytes[0] & 0b1111_0000) == 0b1110_0000) {    # Code points 0800 - 0FFF
            $n = (($bytes[0] & 0b1111) << 12) |
                 (($bytes[1] & 0b11_1111) << 6) |
                  ($bytes[2] & 0b11_1111);
            if ((($bytes[0] & 0b1111) == 0) &&
                (($bytes[1] & 0b1000_0000) == 0)) {
                printf("Overlong 3 byte UTF-8 code 0x%X for code point 0x%X\n", $u, $n);
            }
            if (scalar(@bytes) > 3) {
                printf("Excess byte(s) in UTF-8 code 0x%X: 3 bytes expected, %d specified\n",
                       $u, scalar(@bytes));
            }

        } elsif (($bytes[0] & 0b1111_1000) == 0b1111_0000) {    # Code points 10000 - 10FFFF
            $n = (($bytes[0] & 0b0111) << 18) |
                 (($bytes[1] & 0b11_1111) << 12) |
                 (($bytes[2] & 0b11_1111) << 6) |
                  ($bytes[3] & 0b11_1111);
            if ((($bytes[0] & 0b0111) == 0) &&
                (($bytes[1] & 0b0011_0000) == 0)) {
                printf("Overlong 4 byte UTF-8 code 0x%X for code point 0x%X\n", $u, $n);
            }
        }

        return $n;
    }

    #   Encode a single UTF-8 character code point as a byte
    #   stream in an integer.

    sub encode_UTF8 {
        my ($n) = @_;

        my $u;

        if ($n < 0x80) {
            $u = $n;
        } elsif ($n < 0x800) {
            $u = ((0b1100_0000 | ($n >> 6)) << 8) |
                  (0b1000_0000 | ($n & 0b0011_1111));
        } elsif ($n < 0x10000) {
            $u = ((0b1110_0000 | ($n >> 12)) << 16) |
                 ((0b1000_0000 | (($n >> 6) & 0b0011_1111)) << 8) |
                  (0b1000_0000 | ($n & 0b0011_1111));
        } else {
            $u = ((0b1111_0000 | ($n >> 18)) << 24) |
                 ((0b1000_0000 | (($n >> 12) & 0b0011_1111)) << 16) |
                 ((0b1000_0000 | (($n >> 6) & 0b0011_1111)) << 8) |
                  (0b1000_0000 | ($n & 0b0011_1111));
        }

        return $u;
    }

=begin test_UTF8
    #   Test UTF-8 encoding and decoding

    sub test_UTF8 {
        for (my $c = 0; $c <= 0x10FFFF; $c++) {
            my $n = encode_UTF8($c);
            my $u = decode_UTF8($n);
            if ($c != $u) {
                printf("UTF-8 encode/decode failure for code point 0x%X: encoded 0x%X  decoded 0x%X\n",
                    $c, $n, $u);
            }
        }
    }
=end test-UTF8
=cut

=pod

The Unicode character tables are based upon the Unicode 14.0.0
(2021) standard.

The control characters in this B<unum> version have been annotated
with their Unicode abbreviations, names, and for U+0000 to U+001F,
the Ctrl-key code which generates them.

The HTML named character references are from the World Wide Web
Consortium HTML standard.  Some browsers may not support all of
these references.

=cut

    sub uname {
        my $code = shift;
        if ($code >= 0x4E00) {
            if ($code >= 0xD800 && $code <= 0xF8FF) {
                # Surrogate and private
                if ($code <= 0xDFFF) {
                    return "<surrogate>";
                } else {
                    return "<private>";
                }
            }

        }
        $UNICODE_NAMES{sprintf("%04X", $code)}
    }

    sub ublock {
        my $code = shift;
        # XXX: could use a binary search, but I am too lazy today...
        my $block;
        for $block (@UNICODE_BLOCKS) {
            return $block->[2] if $block->[0] <= $code && $block->[1] >= $code;
        }
        undef;
    }

    sub init_names {
        #   Pre-dimension array and hash bucket sizes to reduce overhead
        #   in dynamic allocation as they are built below.
        $#UNICODE_BLOCKS = 320;
        $#HTML_CHARACTER_REFERENCES = 2032;
        $#HTML_COMPOSED_CHARACTER_REFERENCES = 93;
        keys %UNICODE_NAMES = 144762;

        #   The following code allows us to build two versions of the program
        #   from the same template file.  The table of Unicode code points
        #   is enormous (7.9 Mb as of Unicode 14.0.0), and we'd prefer not
        #   to carry it around within this program.  We read the table from
        #   a __DATA__ block appended to the program.  Following this can
        #   either be the table itself, appended from a separate file when
        #   the program is built, or the table compressed with bzip2,
        #   preceded by a single text line that says "COMPRESSED".  If
        #   that sentinel is read, we switch the data stream to binary and
        #   feed it through bunzip, creating a temporary file.  Then we
        #   read the temporary file to load the data.  Otherwise, we
        #   continue to read and process the embedded uncompressed table.

        my $l;
        while ($l = <DATA>) {
            if ($l =~ m/^COMPRESSED/) {
                #       The code point table is compressed.  There are two ways
                #       we can approach uncompressing and loading it.  The first
                #       is to use the system's bunzip2 utility, decompressing to
                #       a temporary file which we then read.  The second is to use
                #       Perl's IO::Uncompress::Bunzip2, which is a core module in
                #       recent releases of Perl.  The first approach will only work
                #       on Unix-like systems, while the second should work on any
                #       implementation of Perl which supports all of the core
                #       modules.  The choice should be simple: use the module if
                #       it's present and otherwise fall back to the utility if
                #       we're on a system which provides it.
                #
                #       As with most things, what should be simple is actually more
                #       complicated.  The Perl module is very slow compared to the
                #       utility: almost four times slower.  This results in a
                #       noticeable pause on each invocation of unum, due solely to
                #       the decompression of the table.  There is no clean solution
                #       to this, so here's what I'm going to do.  If the file is
                #       compressed, we test for the existence of an executable of
                #       bunzip2 in the library locations where it's likely to be
                #       found on Unix-like systems.  If it's not found (which will
                #       be the case on legacy systems) and the IO::Uncompress::Bunzip2
                #       module exists, we use it, slow as it may be.  Otherwise,
                #       we try using bunzip2, whether or not we found it.  This
                #       will fail only if the system doesn't support the module
                #       and doesn't have an executable bunzip2.  In all other cases,
                #       the most efficient available alternative will be used.

                my $decomp_start = times() if $debug_decompress;
                                my $cmd_bunzip2 = (-x "/bin/bunzip2") || (-x "/usr/bin/bunzip2") ||
                                        (-x "/usr/local/bin/bunzip2");

                if ((!$cmd_bunzip2) && eval { require IO::Uncompress::Bunzip2; }) {
                    print(STDERR "Using IO::Uncompress::Bunzip2 module\n") if $debug_decompress;
                    my $bz = IO::Uncompress::Bunzip2->new(\*DATA);
                    while ($l = <$bz>) {
                        chop($l);
                        my ($code, $name) = split(' ', $l, 2);
                        $UNICODE_NAMES{$code} = $name;
                    }
                } else {
                    print(STDERR "Using system bunzip2\n") if $debug_decompress;
                    use File::Temp qw(tempfile);
                    my ($fh, $filename) = tempfile("unumXXXXXX", DIR => "/tmp",
                                                   SUFFIX => ".tmp", UNLINK => 1);
                    {
                        local $/ = undef;       # Set to read entire file at once
                        binmode(DATA);
                        open(UCHARS, "| bunzip2 -c >$filename") ||
                            die("Unable to open pipe to bunzip2 code point database");
                        print(UCHARS <DATA>);
                        close(UCHARS);
                    }

                    while ($l = <$fh>) {
                        chop($l);
                        my ($code, $name) = split(' ', $l, 2);
                        $UNICODE_NAMES{$code} = $name;
                    }
                    close($fh);
                }
                printf(STDERR "Decompression time: %.4f seconds.\n",
                    times() - $decomp_start) if $debug_decompress;
                last;
            } else {
                #   Code point table is uncompressed: read text directly
                do {
                    chop($l);
                    my ($code, $name) = split(' ', $l, 2);
                    $UNICODE_NAMES{$code} = $name;
                } while ($l = <DATA>);
                last;
            }
        }
        close(DATA);

        #   Remember to update $#UNICODE_BLOCKS pre-dimension above!
        @UNICODE_BLOCKS = (
        #  start   end        block name
          [0x0000, 0x007F => 'Basic Latin'],
          [0x0080, 0x00FF => 'Latin-1 Supplement'],
          [0x0100, 0x017F => 'Latin Extended-A'],
          [0x0180, 0x024F => 'Latin Extended-B'],
          [0x0250, 0x02AF => 'IPA Extensions'],
          [0x02B0, 0x02FF => 'Spacing Modifier Letters'],
          [0x0300, 0x036F => 'Combining Diacritical Marks'],
          [0x0370, 0x03FF => 'Greek and Coptic'],
          [0x0400, 0x04FF => 'Cyrillic'],
          [0x0500, 0x052F => 'Cyrillic Supplement'],
          [0x0530, 0x058F => 'Armenian'],
          [0x0590, 0x05FF => 'Hebrew'],
          [0x0600, 0x06FF => 'Arabic'],
          [0x0700, 0x074F => 'Syriac'],
          [0x0750, 0x077F => 'Arabic Supplement'],
          [0x0780, 0x07BF => 'Thaana'],
          [0x07C0, 0x07FF => 'NKo'],
          [0x0800, 0x083F => 'Samaritan'],
          [0x0840, 0x085F => 'Mandaic'],
          [0x0860, 0x086F => 'Syriac Supplement'],
          [0x0870, 0x089F => 'Arabic Extended-B'],
          [0x08A0, 0x08FF => 'Arabic Extended-A'],
          [0x0900, 0x097F => 'Devanagari'],
          [0x0980, 0x09FF => 'Bengali'],
          [0x0A00, 0x0A7F => 'Gurmukhi'],
          [0x0A80, 0x0AFF => 'Gujarati'],
          [0x0B00, 0x0B7F => 'Oriya'],
          [0x0B80, 0x0BFF => 'Tamil'],
          [0x0C00, 0x0C7F => 'Telugu'],
          [0x0C80, 0x0CFF => 'Kannada'],
          [0x0D00, 0x0D7F => 'Malayalam'],
          [0x0D80, 0x0DFF => 'Sinhala'],
          [0x0E00, 0x0E7F => 'Thai'],
          [0x0E80, 0x0EFF => 'Lao'],
          [0x0F00, 0x0FFF => 'Tibetan'],
          [0x1000, 0x109F => 'Myanmar'],
          [0x10A0, 0x10FF => 'Georgian'],
          [0x1100, 0x11FF => 'Hangul Jamo'],
          [0x1200, 0x137F => 'Ethiopic'],
          [0x1380, 0x139F => 'Ethiopic Supplement'],
          [0x13A0, 0x13FF => 'Cherokee'],
          [0x1400, 0x167F => 'Unified Canadian Aboriginal Syllabics'],
          [0x1680, 0x169F => 'Ogham'],
          [0x16A0, 0x16FF => 'Runic'],
          [0x1700, 0x171F => 'Tagalog'],
          [0x1720, 0x173F => 'Hanunoo'],
          [0x1740, 0x175F => 'Buhid'],
          [0x1760, 0x177F => 'Tagbanwa'],
          [0x1780, 0x17FF => 'Khmer'],
          [0x1800, 0x18AF => 'Mongolian'],
          [0x18B0, 0x18FF => 'Unified Canadian Aboriginal Syllabics Extended'],
          [0x1900, 0x194F => 'Limbu'],
          [0x1950, 0x197F => 'Tai Le'],
          [0x1980, 0x19DF => 'New Tai Lue'],
          [0x19E0, 0x19FF => 'Khmer Symbols'],
          [0x1A00, 0x1A1F => 'Buginese'],
          [0x1A20, 0x1AAF => 'Tai Tham'],
          [0x1AB0, 0x1AFF => 'Combining Diacritical Marks Extended'],
          [0x1B00, 0x1B7F => 'Balinese'],
          [0x1B80, 0x1BBF => 'Sundanese'],
          [0x1BC0, 0x1BFF => 'Batak'],
          [0x1C00, 0x1C4F => 'Lepcha'],
          [0x1C50, 0x1C7F => 'Ol Chiki'],
          [0x1C80, 0x1C8F => 'Cyrillic Extended-C'],
          [0x1C90, 0x1CBF => 'Georgian Extended'],
          [0x1CC0, 0x1CCF => 'Sundanese Supplement'],
          [0x1CD0, 0x1CFF => 'Vedic Extensions'],
          [0x1D00, 0x1D7F => 'Phonetic Extensions'],
          [0x1D80, 0x1DBF => 'Phonetic Extensions Supplement'],
          [0x1DC0, 0x1DFF => 'Combining Diacritical Marks Supplement'],
          [0x1E00, 0x1EFF => 'Latin Extended Additional'],
          [0x1F00, 0x1FFF => 'Greek Extended'],
          [0x2000, 0x206F => 'General Punctuation'],
          [0x2070, 0x209F => 'Superscripts and Subscripts'],
          [0x20A0, 0x20CF => 'Currency Symbols'],
          [0x20D0, 0x20FF => 'Combining Diacritical Marks for Symbols'],
          [0x2100, 0x214F => 'Letterlike Symbols'],
          [0x2150, 0x218F => 'Number Forms'],
          [0x2190, 0x21FF => 'Arrows'],
          [0x2200, 0x22FF => 'Mathematical Operators'],
          [0x2300, 0x23FF => 'Miscellaneous Technical'],
          [0x2400, 0x243F => 'Control Pictures'],
          [0x2440, 0x245F => 'Optical Character Recognition'],
          [0x2460, 0x24FF => 'Enclosed Alphanumerics'],
          [0x2500, 0x257F => 'Box Drawing'],
          [0x2580, 0x259F => 'Block Elements'],
          [0x25A0, 0x25FF => 'Geometric Shapes'],
          [0x2600, 0x26FF => 'Miscellaneous Symbols'],
          [0x2700, 0x27BF => 'Dingbats'],
          [0x27C0, 0x27EF => 'Miscellaneous Mathematical Symbols-A'],
          [0x27F0, 0x27FF => 'Supplemental Arrows-A'],
          [0x2800, 0x28FF => 'Braille Patterns'],
          [0x2900, 0x297F => 'Supplemental Arrows-B'],
          [0x2980, 0x29FF => 'Miscellaneous Mathematical Symbols-B'],
          [0x2A00, 0x2AFF => 'Supplemental Mathematical Operators'],
          [0x2B00, 0x2BFF => 'Miscellaneous Symbols and Arrows'],
          [0x2C00, 0x2C5F => 'Glagolitic'],
          [0x2C60, 0x2C7F => 'Latin Extended-C'],
          [0x2C80, 0x2CFF => 'Coptic'],
          [0x2D00, 0x2D2F => 'Georgian Supplement'],
          [0x2D30, 0x2D7F => 'Tifinagh'],
          [0x2D80, 0x2DDF => 'Ethiopic Extended'],
          [0x2DE0, 0x2DFF => 'Cyrillic Extended-A'],
          [0x2E00, 0x2E7F => 'Supplemental Punctuation'],
          [0x2E80, 0x2EFF => 'CJK Radicals Supplement'],
          [0x2F00, 0x2FDF => 'Kangxi Radicals'],
          [0x2FF0, 0x2FFF => 'Ideographic Description Characters'],
          [0x3000, 0x303F => 'CJK Symbols and Punctuation'],
          [0x3040, 0x309F => 'Hiragana'],
          [0x30A0, 0x30FF => 'Katakana'],
          [0x3100, 0x312F => 'Bopomofo'],
          [0x3130, 0x318F => 'Hangul Compatibility Jamo'],
          [0x3190, 0x319F => 'Kanbun'],
          [0x31A0, 0x31BF => 'Bopomofo Extended'],
          [0x31C0, 0x31EF => 'CJK Strokes'],
          [0x31F0, 0x31FF => 'Katakana Phonetic Extensions'],
          [0x3200, 0x32FF => 'Enclosed CJK Letters and Months'],
          [0x3300, 0x33FF => 'CJK Compatibility'],
          [0x3400, 0x4DBF => 'CJK Unified Ideographs Extension A'],
          [0x4DC0, 0x4DFF => 'Yijing Hexagram Symbols'],
          [0x4E00, 0x9FFF => 'CJK Unified Ideographs'],
          [0xA000, 0xA48F => 'Yi Syllables'],
          [0xA490, 0xA4CF => 'Yi Radicals'],
          [0xA4D0, 0xA4FF => 'Lisu'],
          [0xA500, 0xA63F => 'Vai'],
          [0xA640, 0xA69F => 'Cyrillic Extended-B'],
          [0xA6A0, 0xA6FF => 'Bamum'],
          [0xA700, 0xA71F => 'Modifier Tone Letters'],
          [0xA720, 0xA7FF => 'Latin Extended-D'],
          [0xA800, 0xA82F => 'Syloti Nagri'],
          [0xA830, 0xA83F => 'Common Indic Number Forms'],
          [0xA840, 0xA87F => 'Phags-pa'],
          [0xA880, 0xA8DF => 'Saurashtra'],
          [0xA8E0, 0xA8FF => 'Devanagari Extended'],
          [0xA900, 0xA92F => 'Kayah Li'],
          [0xA930, 0xA95F => 'Rejang'],
          [0xA960, 0xA97F => 'Hangul Jamo Extended-A'],
          [0xA980, 0xA9DF => 'Javanese'],
          [0xA9E0, 0xA9FF => 'Myanmar Extended-B'],
          [0xAA00, 0xAA5F => 'Cham'],
          [0xAA60, 0xAA7F => 'Myanmar Extended-A'],
          [0xAA80, 0xAADF => 'Tai Viet'],
          [0xAAE0, 0xAAFF => 'Meetei Mayek Extensions'],
          [0xAB00, 0xAB2F => 'Ethiopic Extended-A'],
          [0xAB30, 0xAB6F => 'Latin Extended-E'],
          [0xAB70, 0xABBF => 'Cherokee Supplement'],
          [0xABC0, 0xABFF => 'Meetei Mayek'],
          [0xAC00, 0xD7AF => 'Hangul Syllables'],
          [0xD7B0, 0xD7FF => 'Hangul Jamo Extended-B'],
          [0xD800, 0xDB7F => 'High Surrogates'],
          [0xDB80, 0xDBFF => 'High Private Use Surrogates'],
          [0xDC00, 0xDFFF => 'Low Surrogates'],
          [0xE000, 0xF8FF => 'Private Use Area'],
          [0xF900, 0xFAFF => 'CJK Compatibility Ideographs'],
          [0xFB00, 0xFB4F => 'Alphabetic Presentation Forms'],
          [0xFB50, 0xFDFF => 'Arabic Presentation Forms-A'],
          [0xFE00, 0xFE0F => 'Variation Selectors'],
          [0xFE10, 0xFE1F => 'Vertical Forms'],
          [0xFE20, 0xFE2F => 'Combining Half Marks'],
          [0xFE30, 0xFE4F => 'CJK Compatibility Forms'],
          [0xFE50, 0xFE6F => 'Small Form Variants'],
          [0xFE70, 0xFEFF => 'Arabic Presentation Forms-B'],
          [0xFF00, 0xFFEF => 'Halfwidth and Fullwidth Forms'],
          [0xFFF0, 0xFFFF => 'Specials'],
          [0x10000, 0x1007F => 'Linear B Syllabary'],
          [0x10080, 0x100FF => 'Linear B Ideograms'],
          [0x10100, 0x1013F => 'Aegean Numbers'],
          [0x10140, 0x1018F => 'Ancient Greek Numbers'],
          [0x10190, 0x101CF => 'Ancient Symbols'],
          [0x101D0, 0x101FF => 'Phaistos Disc'],
          [0x10280, 0x1029F => 'Lycian'],
          [0x102A0, 0x102DF => 'Carian'],
          [0x102E0, 0x102FF => 'Coptic Epact Numbers'],
          [0x10300, 0x1032F => 'Old Italic'],
          [0x10330, 0x1034F => 'Gothic'],
          [0x10350, 0x1037F => 'Old Permic'],
          [0x10380, 0x1039F => 'Ugaritic'],
          [0x103A0, 0x103DF => 'Old Persian'],
          [0x10400, 0x1044F => 'Deseret'],
          [0x10450, 0x1047F => 'Shavian'],
          [0x10480, 0x104AF => 'Osmanya'],
          [0x104B0, 0x104FF => 'Osage'],
          [0x10500, 0x1052F => 'Elbasan'],
          [0x10530, 0x1056F => 'Caucasian Albanian'],
          [0x10570, 0x105BF => 'Vithkuqi'],
          [0x10600, 0x1077F => 'Linear A'],
          [0x10780, 0x107BF => 'Latin Extended-F'],
          [0x10800, 0x1083F => 'Cypriot Syllabary'],
          [0x10840, 0x1085F => 'Imperial Aramaic'],
          [0x10860, 0x1087F => 'Palmyrene'],
          [0x10880, 0x108AF => 'Nabataean'],
          [0x108E0, 0x108FF => 'Hatran'],
          [0x10900, 0x1091F => 'Phoenician'],
          [0x10920, 0x1093F => 'Lydian'],
          [0x10980, 0x1099F => 'Meroitic Hieroglyphs'],
          [0x109A0, 0x109FF => 'Meroitic Cursive'],
          [0x10A00, 0x10A5F => 'Kharoshthi'],
          [0x10A60, 0x10A7F => 'Old South Arabian'],
          [0x10A80, 0x10A9F => 'Old North Arabian'],
          [0x10AC0, 0x10AFF => 'Manichaean'],
          [0x10B00, 0x10B3F => 'Avestan'],
          [0x10B40, 0x10B5F => 'Inscriptional Parthian'],
          [0x10B60, 0x10B7F => 'Inscriptional Pahlavi'],
          [0x10B80, 0x10BAF => 'Psalter Pahlavi'],
          [0x10C00, 0x10C4F => 'Old Turkic'],
          [0x10C80, 0x10CFF => 'Old Hungarian'],
          [0x10D00, 0x10D3F => 'Hanifi Rohingya'],
          [0x10E60, 0x10E7F => 'Rumi Numeral Symbols'],
          [0x10E80, 0x10EBF => 'Yezidi'],
          [0x10F00, 0x10F2F => 'Old Sogdian'],
          [0x10F30, 0x10F6F => 'Sogdian'],
          [0x10F70, 0x10FAF => 'Old Uyghur'],
          [0x10FB0, 0x10FDF => 'Chorasmian'],
          [0x10FE0, 0x10FFF => 'Elymaic'],
          [0x11000, 0x1107F => 'Brahmi'],
          [0x11080, 0x110CF => 'Kaithi'],
          [0x110D0, 0x110FF => 'Sora Sompeng'],
          [0x11100, 0x1114F => 'Chakma'],
          [0x11150, 0x1117F => 'Mahajani'],
          [0x11180, 0x111DF => 'Sharada'],
          [0x111E0, 0x111FF => 'Sinhala Archaic Numbers'],
          [0x11200, 0x1124F => 'Khojki'],
          [0x11280, 0x112AF => 'Multani'],
          [0x112B0, 0x112FF => 'Khudawadi'],
          [0x11300, 0x1137F => 'Grantha'],
          [0x11400, 0x1147F => 'Newa'],
          [0x11480, 0x114DF => 'Tirhuta'],
          [0x11580, 0x115FF => 'Siddham'],
          [0x11600, 0x1165F => 'Modi'],
          [0x11660, 0x1167F => 'Mongolian Supplement'],
          [0x11680, 0x116CF => 'Takri'],
          [0x11700, 0x1174F => 'Ahom'],
          [0x11800, 0x1184F => 'Dogra'],
          [0x118A0, 0x118FF => 'Warang Citi'],
          [0x11900, 0x1195F => 'Dives Akuru'],
          [0x119A0, 0x119FF => 'Nandinagari'],
          [0x11A00, 0x11A4F => 'Zanabazar Square'],
          [0x11A50, 0x11AAF => 'Soyombo'],
          [0x11AB0, 0x11ABF => 'Unified Canadian Aboriginal Syllabics Extended-A'],
          [0x11AC0, 0x11AFF => 'Pau Cin Hau'],
          [0x11C00, 0x11C6F => 'Bhaiksuki'],
          [0x11C70, 0x11CBF => 'Marchen'],
          [0x11D00, 0x11D5F => 'Masaram Gondi'],
          [0x11D60, 0x11DAF => 'Gunjala Gondi'],
          [0x11EE0, 0x11EFF => 'Makasar'],
          [0x11FB0, 0x11FBF => 'Lisu Supplement'],
          [0x11FC0, 0x11FFF => 'Tamil Supplement'],
          [0x12000, 0x123FF => 'Cuneiform'],
          [0x12400, 0x1247F => 'Cuneiform Numbers and Punctuation'],
          [0x12480, 0x1254F => 'Early Dynastic Cuneiform'],
          [0x12F90, 0x12FFF => 'Cypro-Minoan'],
          [0x13000, 0x1342F => 'Egyptian Hieroglyphs'],
          [0x13430, 0x1343F => 'Egyptian Hieroglyph Format Controls'],
          [0x14400, 0x1467F => 'Anatolian Hieroglyphs'],
          [0x16800, 0x16A3F => 'Bamum Supplement'],
          [0x16A40, 0x16A6F => 'Mro'],
          [0x16A70, 0x16ACF => 'Tangsa'],
          [0x16AD0, 0x16AFF => 'Bassa Vah'],
          [0x16B00, 0x16B8F => 'Pahawh Hmong'],
          [0x16E40, 0x16E9F => 'Medefaidrin'],
          [0x16F00, 0x16F9F => 'Miao'],
          [0x16FE0, 0x16FFF => 'Ideographic Symbols and Punctuation'],
          [0x17000, 0x187FF => 'Tangut'],
          [0x18800, 0x18AFF => 'Tangut Components'],
          [0x18B00, 0x18CFF => 'Khitan Small Script'],
          [0x18D00, 0x18D7F => 'Tangut Supplement'],
          [0x1AFF0, 0x1AFFF => 'Kana Extended-B'],
          [0x1B000, 0x1B0FF => 'Kana Supplement'],
          [0x1B100, 0x1B12F => 'Kana Extended-A'],
          [0x1B130, 0x1B16F => 'Small Kana Extension'],
          [0x1B170, 0x1B2FF => 'Nushu'],
          [0x1BC00, 0x1BC9F => 'Duployan'],
          [0x1BCA0, 0x1BCAF => 'Shorthand Format Controls'],
          [0x1CF00, 0x1CFCF => 'Znamenny Musical Notation'],
          [0x1D000, 0x1D0FF => 'Byzantine Musical Symbols'],
          [0x1D100, 0x1D1FF => 'Musical Symbols'],
          [0x1D200, 0x1D24F => 'Ancient Greek Musical Notation'],
          [0x1D2E0, 0x1D2FF => 'Mayan Numerals'],
          [0x1D300, 0x1D35F => 'Tai Xuan Jing Symbols'],
          [0x1D360, 0x1D37F => 'Counting Rod Numerals'],
          [0x1D400, 0x1D7FF => 'Mathematical Alphanumeric Symbols'],
          [0x1D800, 0x1DAAF => 'Sutton SignWriting'],
          [0x1DF00, 0x1DFFF => 'Latin Extended-G'],
          [0x1E000, 0x1E02F => 'Glagolitic Supplement'],
          [0x1E100, 0x1E14F => 'Nyiakeng Puachue Hmong'],
          [0x1E290, 0x1E2BF => 'Toto'],
          [0x1E2C0, 0x1E2FF => 'Wancho'],
          [0x1E7E0, 0x1E7FF => 'Ethiopic Extended-B'],
          [0x1E800, 0x1E8DF => 'Mende Kikakui'],
          [0x1E900, 0x1E95F => 'Adlam'],
          [0x1EC70, 0x1ECBF => 'Indic Siyaq Numbers'],
          [0x1ED00, 0x1ED4F => 'Ottoman Siyaq Numbers'],
          [0x1EE00, 0x1EEFF => 'Arabic Mathematical Alphabetic Symbols'],
          [0x1F000, 0x1F02F => 'Mahjong Tiles'],
          [0x1F030, 0x1F09F => 'Domino Tiles'],
          [0x1F0A0, 0x1F0FF => 'Playing Cards'],
          [0x1F100, 0x1F1FF => 'Enclosed Alphanumeric Supplement'],
          [0x1F200, 0x1F2FF => 'Enclosed Ideographic Supplement'],
          [0x1F300, 0x1F5FF => 'Miscellaneous Symbols and Pictographs'],
          [0x1F600, 0x1F64F => 'Emoticons'],
          [0x1F650, 0x1F67F => 'Ornamental Dingbats'],
          [0x1F680, 0x1F6FF => 'Transport and Map Symbols'],
          [0x1F700, 0x1F77F => 'Alchemical Symbols'],
          [0x1F780, 0x1F7FF => 'Geometric Shapes Extended'],
          [0x1F800, 0x1F8FF => 'Supplemental Arrows-C'],
          [0x1F900, 0x1F9FF => 'Supplemental Symbols and Pictographs'],
          [0x1FA00, 0x1FA6F => 'Chess Symbols'],
          [0x1FA70, 0x1FAFF => 'Symbols and Pictographs Extended-A'],
          [0x1FB00, 0x1FBFF => 'Symbols for Legacy Computing'],
          [0x20000, 0x2A6DF => 'CJK Unified Ideographs Extension B'],
          [0x2A700, 0x2B73F => 'CJK Unified Ideographs Extension C'],
          [0x2B740, 0x2B81F => 'CJK Unified Ideographs Extension D'],
          [0x2B820, 0x2CEAF => 'CJK Unified Ideographs Extension E'],
          [0x2CEB0, 0x2EBEF => 'CJK Unified Ideographs Extension F'],
          [0x2F800, 0x2FA1F => 'CJK Compatibility Ideographs Supplement'],
          [0x30000, 0x3134F => 'CJK Unified Ideographs Extension G'],
          [0xE0000, 0xE007F => 'Tags'],
          [0xE0100, 0xE01EF => 'Variation Selectors Supplement'],
          [0xF0000, 0xFFFFF => 'Supplementary Private Use Area-A'],
          [0x100000, 0x10FFFF => 'Supplementary Private Use Area-B'],
        );

        #   HTML5 Named Character References

        #   Remember to update $#HTML_CHARACTER_REFERENCES pre-dimension above!
        @HTML_CHARACTER_REFERENCES = (
            #   From https://www.w3.org/TR/html5/entities.json
            'Aacute', 193,
            'aacute', 225,
            'Abreve', 258,
            'abreve', 259,
            'ac', 8766,
            'acd', 8767,
            'Acirc', 194,
            'acirc', 226,
            'acute', 180,
            'Acy', 1040,
            'acy', 1072,
            'AElig', 198,
            'aelig', 230,
            'af', 8289,
            'Afr', 120068,
            'afr', 120094,
            'Agrave', 192,
            'agrave', 224,
            'alefsym', 8501,
            'aleph', 8501,
            'Alpha', 913,
            'alpha', 945,
            'Amacr', 256,
            'amacr', 257,
            'amalg', 10815,
            'AMP', 38,
            'amp', 38,
            'And', 10835,
            'and', 8743,
            'andand', 10837,
            'andd', 10844,
            'andslope', 10840,
            'andv', 10842,
            'ang', 8736,
            'ange', 10660,
            'angle', 8736,
            'angmsd', 8737,
            'angmsdaa', 10664,
            'angmsdab', 10665,
            'angmsdac', 10666,
            'angmsdad', 10667,
            'angmsdae', 10668,
            'angmsdaf', 10669,
            'angmsdag', 10670,
            'angmsdah', 10671,
            'angrt', 8735,
            'angrtvb', 8894,
            'angrtvbd', 10653,
            'angsph', 8738,
            'angst', 197,
            'angzarr', 9084,
            'Aogon', 260,
            'aogon', 261,
            'Aopf', 120120,
            'aopf', 120146,
            'ap', 8776,
            'apacir', 10863,
            'apE', 10864,
            'ape', 8778,
            'apid', 8779,
            'apos', 39,
            'ApplyFunction', 8289,
            'approx', 8776,
            'approxeq', 8778,
            'Aring', 197,
            'aring', 229,
            'Ascr', 119964,
            'ascr', 119990,
            'Assign', 8788,
            'ast', 42,
            'asymp', 8776,
            'asympeq', 8781,
            'Atilde', 195,
            'atilde', 227,
            'Auml', 196,
            'auml', 228,
            'awconint', 8755,
            'awint', 10769,
            'backcong', 8780,
            'backepsilon', 1014,
            'backprime', 8245,
            'backsim', 8765,
            'backsimeq', 8909,
            'Backslash', 8726,
            'Barv', 10983,
            'barvee', 8893,
            'Barwed', 8966,
            'barwed', 8965,
            'barwedge', 8965,
            'bbrk', 9141,
            'bbrktbrk', 9142,
            'bcong', 8780,
            'Bcy', 1041,
            'bcy', 1073,
            'bdquo', 8222,
            'becaus', 8757,
            'Because', 8757,
            'because', 8757,
            'bemptyv', 10672,
            'bepsi', 1014,
            'bernou', 8492,
            'Bernoullis', 8492,
            'Beta', 914,
            'beta', 946,
            'beth', 8502,
            'between', 8812,
            'Bfr', 120069,
            'bfr', 120095,
            'bigcap', 8898,
            'bigcirc', 9711,
            'bigcup', 8899,
            'bigodot', 10752,
            'bigoplus', 10753,
            'bigotimes', 10754,
            'bigsqcup', 10758,
            'bigstar', 9733,
            'bigtriangledown', 9661,
            'bigtriangleup', 9651,
            'biguplus', 10756,
            'bigvee', 8897,
            'bigwedge', 8896,
            'bkarow', 10509,
            'blacklozenge', 10731,
            'blacksquare', 9642,
            'blacktriangle', 9652,
            'blacktriangledown', 9662,
            'blacktriangleleft', 9666,
            'blacktriangleright', 9656,
            'blank', 9251,
            'blk12', 9618,
            'blk14', 9617,
            'blk34', 9619,
            'block', 9608,
            'bNot', 10989,
            'bnot', 8976,
            'Bopf', 120121,
            'bopf', 120147,
            'bot', 8869,
            'bottom', 8869,
            'bowtie', 8904,
            'boxbox', 10697,
            'boxDL', 9559,
            'boxDl', 9558,
            'boxdL', 9557,
            'boxdl', 9488,
            'boxDR', 9556,
            'boxDr', 9555,
            'boxdR', 9554,
            'boxdr', 9484,
            'boxH', 9552,
            'boxh', 9472,
            'boxHD', 9574,
            'boxHd', 9572,
            'boxhD', 9573,
            'boxhd', 9516,
            'boxHU', 9577,
            'boxHu', 9575,
            'boxhU', 9576,
            'boxhu', 9524,
            'boxminus', 8863,
            'boxplus', 8862,
            'boxtimes', 8864,
            'boxUL', 9565,
            'boxUl', 9564,
            'boxuL', 9563,
            'boxul', 9496,
            'boxUR', 9562,
            'boxUr', 9561,
            'boxuR', 9560,
            'boxur', 9492,
            'boxV', 9553,
            'boxv', 9474,
            'boxVH', 9580,
            'boxVh', 9579,
            'boxvH', 9578,
            'boxvh', 9532,
            'boxVL', 9571,
            'boxVl', 9570,
            'boxvL', 9569,
            'boxvl', 9508,
            'boxVR', 9568,
            'boxVr', 9567,
            'boxvR', 9566,
            'boxvr', 9500,
            'bprime', 8245,
            'Breve', 728,
            'breve', 728,
            'brvbar', 166,
            'Bscr', 8492,
            'bscr', 119991,
            'bsemi', 8271,
            'bsim', 8765,
            'bsime', 8909,
            'bsol', 92,
            'bsolb', 10693,
            'bsolhsub', 10184,
            'bull', 8226,
            'bullet', 8226,
            'bump', 8782,
            'bumpE', 10926,
            'bumpe', 8783,
            'Bumpeq', 8782,
            'bumpeq', 8783,
            'Cacute', 262,
            'cacute', 263,
            'Cap', 8914,
            'cap', 8745,
            'capand', 10820,
            'capbrcup', 10825,
            'capcap', 10827,
            'capcup', 10823,
            'capdot', 10816,
            'CapitalDifferentialD', 8517,
            'caret', 8257,
            'caron', 711,
            'Cayleys', 8493,
            'ccaps', 10829,
            'Ccaron', 268,
            'ccaron', 269,
            'Ccedil', 199,
            'ccedil', 231,
            'Ccirc', 264,
            'ccirc', 265,
            'Cconint', 8752,
            'ccups', 10828,
            'ccupssm', 10832,
            'Cdot', 266,
            'cdot', 267,
            'cedil', 184,
            'Cedilla', 184,
            'cemptyv', 10674,
            'cent', 162,
            'CenterDot', 183,
            'centerdot', 183,
            'Cfr', 8493,
            'cfr', 120096,
            'CHcy', 1063,
            'chcy', 1095,
            'check', 10003,
            'checkmark', 10003,
            'Chi', 935,
            'chi', 967,
            'cir', 9675,
            'circ', 710,
            'circeq', 8791,
            'circlearrowleft', 8634,
            'circlearrowright', 8635,
            'circledast', 8859,
            'circledcirc', 8858,
            'circleddash', 8861,
            'CircleDot', 8857,
            'circledR', 174,
            'circledS', 9416,
            'CircleMinus', 8854,
            'CirclePlus', 8853,
            'CircleTimes', 8855,
            'cirE', 10691,
            'cire', 8791,
            'cirfnint', 10768,
            'cirmid', 10991,
            'cirscir', 10690,
            'ClockwiseContourIntegral', 8754,
            'CloseCurlyDoubleQuote', 8221,
            'CloseCurlyQuote', 8217,
            'clubs', 9827,
            'clubsuit', 9827,
            'Colon', 8759,
            'colon', 58,
            'Colone', 10868,
            'colone', 8788,
            'coloneq', 8788,
            'comma', 44,
            'commat', 64,
            'comp', 8705,
            'compfn', 8728,
            'complement', 8705,
            'complexes', 8450,
            'cong', 8773,
            'congdot', 10861,
            'Congruent', 8801,
            'Conint', 8751,
            'conint', 8750,
            'ContourIntegral', 8750,
            'Copf', 8450,
            'copf', 120148,
            'coprod', 8720,
            'Coproduct', 8720,
            'COPY', 169,
            'copy', 169,
            'copysr', 8471,
            'CounterClockwiseContourIntegral', 8755,
            'crarr', 8629,
            'Cross', 10799,
            'cross', 10007,
            'Cscr', 119966,
            'cscr', 119992,
            'csub', 10959,
            'csube', 10961,
            'csup', 10960,
            'csupe', 10962,
            'ctdot', 8943,
            'cudarrl', 10552,
            'cudarrr', 10549,
            'cuepr', 8926,
            'cuesc', 8927,
            'cularr', 8630,
            'cularrp', 10557,
            'Cup', 8915,
            'cup', 8746,
            'cupbrcap', 10824,
            'CupCap', 8781,
            'cupcap', 10822,
            'cupcup', 10826,
            'cupdot', 8845,
            'cupor', 10821,
            'curarr', 8631,
            'curarrm', 10556,
            'curlyeqprec', 8926,
            'curlyeqsucc', 8927,
            'curlyvee', 8910,
            'curlywedge', 8911,
            'curren', 164,
            'curvearrowleft', 8630,
            'curvearrowright', 8631,
            'cuvee', 8910,
            'cuwed', 8911,
            'cwconint', 8754,
            'cwint', 8753,
            'cylcty', 9005,
            'Dagger', 8225,
            'dagger', 8224,
            'daleth', 8504,
            'Darr', 8609,
            'dArr', 8659,
            'darr', 8595,
            'dash', 8208,
            'Dashv', 10980,
            'dashv', 8867,
            'dbkarow', 10511,
            'dblac', 733,
            'Dcaron', 270,
            'dcaron', 271,
            'Dcy', 1044,
            'dcy', 1076,
            'DD', 8517,
            'dd', 8518,
            'ddagger', 8225,
            'ddarr', 8650,
            'DDotrahd', 10513,
            'ddotseq', 10871,
            'deg', 176,
            'Del', 8711,
            'Delta', 916,
            'delta', 948,
            'demptyv', 10673,
            'dfisht', 10623,
            'Dfr', 120071,
            'dfr', 120097,
            'dHar', 10597,
            'dharl', 8643,
            'dharr', 8642,
            'DiacriticalAcute', 180,
            'DiacriticalDot', 729,
            'DiacriticalDoubleAcute', 733,
            'DiacriticalGrave', 96,
            'DiacriticalTilde', 732,
            'diam', 8900,
            'Diamond', 8900,
            'diamond', 8900,
            'diamondsuit', 9830,
            'diams', 9830,
            'die', 168,
            'DifferentialD', 8518,
            'digamma', 989,
            'disin', 8946,
            'div', 247,
            'divide', 247,
            'divideontimes', 8903,
            'divonx', 8903,
            'DJcy', 1026,
            'djcy', 1106,
            'dlcorn', 8990,
            'dlcrop', 8973,
            'dollar', 36,
            'Dopf', 120123,
            'dopf', 120149,
            'Dot', 168,
            'dot', 729,
            'DotDot', 8412,
            'doteq', 8784,
            'doteqdot', 8785,
            'DotEqual', 8784,
            'dotminus', 8760,
            'dotplus', 8724,
            'dotsquare', 8865,
            'doublebarwedge', 8966,
            'DoubleContourIntegral', 8751,
            'DoubleDot', 168,
            'DoubleDownArrow', 8659,
            'DoubleLeftArrow', 8656,
            'DoubleLeftRightArrow', 8660,
            'DoubleLeftTee', 10980,
            'DoubleLongLeftArrow', 10232,
            'DoubleLongLeftRightArrow', 10234,
            'DoubleLongRightArrow', 10233,
            'DoubleRightArrow', 8658,
            'DoubleRightTee', 8872,
            'DoubleUpArrow', 8657,
            'DoubleUpDownArrow', 8661,
            'DoubleVerticalBar', 8741,
            'DownArrow', 8595,
            'Downarrow', 8659,
            'downarrow', 8595,
            'DownArrowBar', 10515,
            'DownArrowUpArrow', 8693,
            'DownBreve', 785,
            'downdownarrows', 8650,
            'downharpoonleft', 8643,
            'downharpoonright', 8642,
            'DownLeftRightVector', 10576,
            'DownLeftTeeVector', 10590,
            'DownLeftVector', 8637,
            'DownLeftVectorBar', 10582,
            'DownRightTeeVector', 10591,
            'DownRightVector', 8641,
            'DownRightVectorBar', 10583,
            'DownTee', 8868,
            'DownTeeArrow', 8615,
            'drbkarow', 10512,
            'drcorn', 8991,
            'drcrop', 8972,
            'Dscr', 119967,
            'dscr', 119993,
            'DScy', 1029,
            'dscy', 1109,
            'dsol', 10742,
            'Dstrok', 272,
            'dstrok', 273,
            'dtdot', 8945,
            'dtri', 9663,
            'dtrif', 9662,
            'duarr', 8693,
            'duhar', 10607,
            'dwangle', 10662,
            'DZcy', 1039,
            'dzcy', 1119,
            'dzigrarr', 10239,
            'Eacute', 201,
            'eacute', 233,
            'easter', 10862,
            'Ecaron', 282,
            'ecaron', 283,
            'ecir', 8790,
            'Ecirc', 202,
            'ecirc', 234,
            'ecolon', 8789,
            'Ecy', 1069,
            'ecy', 1101,
            'eDDot', 10871,
            'Edot', 278,
            'eDot', 8785,
            'edot', 279,
            'ee', 8519,
            'efDot', 8786,
            'Efr', 120072,
            'efr', 120098,
            'eg', 10906,
            'Egrave', 200,
            'egrave', 232,
            'egs', 10902,
            'egsdot', 10904,
            'el', 10905,
            'Element', 8712,
            'elinters', 9191,
            'ell', 8467,
            'els', 10901,
            'elsdot', 10903,
            'Emacr', 274,
            'emacr', 275,
            'empty', 8709,
            'emptyset', 8709,
            'EmptySmallSquare', 9723,
            'emptyv', 8709,
            'EmptyVerySmallSquare', 9643,
            'emsp', 8195,
            'emsp13', 8196,
            'emsp14', 8197,
            'ENG', 330,
            'eng', 331,
            'ensp', 8194,
            'Eogon', 280,
            'eogon', 281,
            'Eopf', 120124,
            'eopf', 120150,
            'epar', 8917,
            'eparsl', 10723,
            'eplus', 10865,
            'epsi', 949,
            'Epsilon', 917,
            'epsilon', 949,
            'epsiv', 1013,
            'eqcirc', 8790,
            'eqcolon', 8789,
            'eqsim', 8770,
            'eqslantgtr', 10902,
            'eqslantless', 10901,
            'Equal', 10869,
            'equals', 61,
            'EqualTilde', 8770,
            'equest', 8799,
            'Equilibrium', 8652,
            'equiv', 8801,
            'equivDD', 10872,
            'eqvparsl', 10725,
            'erarr', 10609,
            'erDot', 8787,
            'Escr', 8496,
            'escr', 8495,
            'esdot', 8784,
            'Esim', 10867,
            'esim', 8770,
            'Eta', 919,
            'eta', 951,
            'ETH', 208,
            'eth', 240,
            'Euml', 203,
            'euml', 235,
            'euro', 8364,
            'excl', 33,
            'exist', 8707,
            'Exists', 8707,
            'expectation', 8496,
            'ExponentialE', 8519,
            'exponentiale', 8519,
            'fallingdotseq', 8786,
            'Fcy', 1060,
            'fcy', 1092,
            'female', 9792,
            'ffilig', 64259,
            'fflig', 64256,
            'ffllig', 64260,
            'Ffr', 120073,
            'ffr', 120099,
            'filig', 64257,
            'FilledSmallSquare', 9724,
            'FilledVerySmallSquare', 9642,
            'flat', 9837,
            'fllig', 64258,
            'fltns', 9649,
            'fnof', 402,
            'Fopf', 120125,
            'fopf', 120151,
            'ForAll', 8704,
            'forall', 8704,
            'fork', 8916,
            'forkv', 10969,
            'Fouriertrf', 8497,
            'fpartint', 10765,
            'frac12', 189,
            'frac13', 8531,
            'frac14', 188,
            'frac15', 8533,
            'frac16', 8537,
            'frac18', 8539,
            'frac23', 8532,
            'frac25', 8534,
            'frac34', 190,
            'frac35', 8535,
            'frac38', 8540,
            'frac45', 8536,
            'frac56', 8538,
            'frac58', 8541,
            'frac78', 8542,
            'frasl', 8260,
            'frown', 8994,
            'Fscr', 8497,
            'fscr', 119995,
            'gacute', 501,
            'Gamma', 915,
            'gamma', 947,
            'Gammad', 988,
            'gammad', 989,
            'gap', 10886,
            'Gbreve', 286,
            'gbreve', 287,
            'Gcedil', 290,
            'Gcirc', 284,
            'gcirc', 285,
            'Gcy', 1043,
            'gcy', 1075,
            'Gdot', 288,
            'gdot', 289,
            'gE', 8807,
            'ge', 8805,
            'gEl', 10892,
            'gel', 8923,
            'geq', 8805,
            'geqq', 8807,
            'geqslant', 10878,
            'ges', 10878,
            'gescc', 10921,
            'gesdot', 10880,
            'gesdoto', 10882,
            'gesdotol', 10884,
            'gesles', 10900,
            'Gfr', 120074,
            'gfr', 120100,
            'Gg', 8921,
            'gg', 8811,
            'ggg', 8921,
            'gimel', 8503,
            'GJcy', 1027,
            'gjcy', 1107,
            'gl', 8823,
            'gla', 10917,
            'glE', 10898,
            'glj', 10916,
            'gnap', 10890,
            'gnapprox', 10890,
            'gnE', 8809,
            'gne', 10888,
            'gneq', 10888,
            'gneqq', 8809,
            'gnsim', 8935,
            'Gopf', 120126,
            'gopf', 120152,
            'grave', 96,
            'GreaterEqual', 8805,
            'GreaterEqualLess', 8923,
            'GreaterFullEqual', 8807,
            'GreaterGreater', 10914,
            'GreaterLess', 8823,
            'GreaterSlantEqual', 10878,
            'GreaterTilde', 8819,
            'Gscr', 119970,
            'gscr', 8458,
            'gsim', 8819,
            'gsime', 10894,
            'gsiml', 10896,
            'GT', 62,
            'Gt', 8811,
            'gt', 62,
            'gtcc', 10919,
            'gtcir', 10874,
            'gtdot', 8919,
            'gtlPar', 10645,
            'gtquest', 10876,
            'gtrapprox', 10886,
            'gtrarr', 10616,
            'gtrdot', 8919,
            'gtreqless', 8923,
            'gtreqqless', 10892,
            'gtrless', 8823,
            'gtrsim', 8819,
            'Hacek', 711,
            'hairsp', 8202,
            'half', 189,
            'hamilt', 8459,
            'HARDcy', 1066,
            'hardcy', 1098,
            'hArr', 8660,
            'harr', 8596,
            'harrcir', 10568,
            'harrw', 8621,
            'Hat', 94,
            'hbar', 8463,
            'Hcirc', 292,
            'hcirc', 293,
            'hearts', 9829,
            'heartsuit', 9829,
            'hellip', 8230,
            'hercon', 8889,
            'Hfr', 8460,
            'hfr', 120101,
            'HilbertSpace', 8459,
            'hksearow', 10533,
            'hkswarow', 10534,
            'hoarr', 8703,
            'homtht', 8763,
            'hookleftarrow', 8617,
            'hookrightarrow', 8618,
            'Hopf', 8461,
            'hopf', 120153,
            'horbar', 8213,
            'HorizontalLine', 9472,
            'Hscr', 8459,
            'hscr', 119997,
            'hslash', 8463,
            'Hstrok', 294,
            'hstrok', 295,
            'HumpDownHump', 8782,
            'HumpEqual', 8783,
            'hybull', 8259,
            'hyphen', 8208,
            'Iacute', 205,
            'iacute', 237,
            'ic', 8291,
            'Icirc', 206,
            'icirc', 238,
            'Icy', 1048,
            'icy', 1080,
            'Idot', 304,
            'IEcy', 1045,
            'iecy', 1077,
            'iexcl', 161,
            'iff', 8660,
            'Ifr', 8465,
            'ifr', 120102,
            'Igrave', 204,
            'igrave', 236,
            'ii', 8520,
            'iiiint', 10764,
            'iiint', 8749,
            'iinfin', 10716,
            'iiota', 8489,
            'IJlig', 306,
            'ijlig', 307,
            'Im', 8465,
            'Imacr', 298,
            'imacr', 299,
            'image', 8465,
            'ImaginaryI', 8520,
            'imagline', 8464,
            'imagpart', 8465,
            'imath', 305,
            'imof', 8887,
            'imped', 437,
            'Implies', 8658,
            'in', 8712,
            'incare', 8453,
            'infin', 8734,
            'infintie', 10717,
            'inodot', 305,
            'Int', 8748,
            'int', 8747,
            'intcal', 8890,
            'integers', 8484,
            'Integral', 8747,
            'intercal', 8890,
            'Intersection', 8898,
            'intlarhk', 10775,
            'intprod', 10812,
            'InvisibleComma', 8291,
            'InvisibleTimes', 8290,
            'IOcy', 1025,
            'iocy', 1105,
            'Iogon', 302,
            'iogon', 303,
            'Iopf', 120128,
            'iopf', 120154,
            'Iota', 921,
            'iota', 953,
            'iprod', 10812,
            'iquest', 191,
            'Iscr', 8464,
            'iscr', 119998,
            'isin', 8712,
            'isindot', 8949,
            'isinE', 8953,
            'isins', 8948,
            'isinsv', 8947,
            'isinv', 8712,
            'it', 8290,
            'Itilde', 296,
            'itilde', 297,
            'Iukcy', 1030,
            'iukcy', 1110,
            'Iuml', 207,
            'iuml', 239,
            'Jcirc', 308,
            'jcirc', 309,
            'Jcy', 1049,
            'jcy', 1081,
            'Jfr', 120077,
            'jfr', 120103,
            'jmath', 567,
            'Jopf', 120129,
            'jopf', 120155,
            'Jscr', 119973,
            'jscr', 119999,
            'Jsercy', 1032,
            'jsercy', 1112,
            'Jukcy', 1028,
            'jukcy', 1108,
            'Kappa', 922,
            'kappa', 954,
            'kappav', 1008,
            'Kcedil', 310,
            'kcedil', 311,
            'Kcy', 1050,
            'kcy', 1082,
            'Kfr', 120078,
            'kfr', 120104,
            'kgreen', 312,
            'KHcy', 1061,
            'khcy', 1093,
            'KJcy', 1036,
            'kjcy', 1116,
            'Kopf', 120130,
            'kopf', 120156,
            'Kscr', 119974,
            'kscr', 120000,
            'lAarr', 8666,
            'Lacute', 313,
            'lacute', 314,
            'laemptyv', 10676,
            'lagran', 8466,
            'Lambda', 923,
            'lambda', 955,
            'Lang', 10218,
            'lang', 10216,
            'langd', 10641,
            'langle', 10216,
            'lap', 10885,
            'Laplacetrf', 8466,
            'laquo', 171,
            'Larr', 8606,
            'lArr', 8656,
            'larr', 8592,
            'larrb', 8676,
            'larrbfs', 10527,
            'larrfs', 10525,
            'larrhk', 8617,
            'larrlp', 8619,
            'larrpl', 10553,
            'larrsim', 10611,
            'larrtl', 8610,
            'lat', 10923,
            'lAtail', 10523,
            'latail', 10521,
            'late', 10925,
            'lBarr', 10510,
            'lbarr', 10508,
            'lbbrk', 10098,
            'lbrace', 123,
            'lbrack', 91,
            'lbrke', 10635,
            'lbrksld', 10639,
            'lbrkslu', 10637,
            'Lcaron', 317,
            'lcaron', 318,
            'Lcedil', 315,
            'lcedil', 316,
            'lceil', 8968,
            'lcub', 123,
            'Lcy', 1051,
            'lcy', 1083,
            'ldca', 10550,
            'ldquo', 8220,
            'ldquor', 8222,
            'ldrdhar', 10599,
            'ldrushar', 10571,
            'ldsh', 8626,
            'lE', 8806,
            'le', 8804,
            'LeftAngleBracket', 10216,
            'LeftArrow', 8592,
            'Leftarrow', 8656,
            'leftarrow', 8592,
            'LeftArrowBar', 8676,
            'LeftArrowRightArrow', 8646,
            'leftarrowtail', 8610,
            'LeftCeiling', 8968,
            'LeftDoubleBracket', 10214,
            'LeftDownTeeVector', 10593,
            'LeftDownVector', 8643,
            'LeftDownVectorBar', 10585,
            'LeftFloor', 8970,
            'leftharpoondown', 8637,
            'leftharpoonup', 8636,
            'leftleftarrows', 8647,
            'LeftRightArrow', 8596,
            'Leftrightarrow', 8660,
            'leftrightarrow', 8596,
            'leftrightarrows', 8646,
            'leftrightharpoons', 8651,
            'leftrightsquigarrow', 8621,
            'LeftRightVector', 10574,
            'LeftTee', 8867,
            'LeftTeeArrow', 8612,
            'LeftTeeVector', 10586,
            'leftthreetimes', 8907,
            'LeftTriangle', 8882,
            'LeftTriangleBar', 10703,
            'LeftTriangleEqual', 8884,
            'LeftUpDownVector', 10577,
            'LeftUpTeeVector', 10592,
            'LeftUpVector', 8639,
            'LeftUpVectorBar', 10584,
            'LeftVector', 8636,
            'LeftVectorBar', 10578,
            'lEg', 10891,
            'leg', 8922,
            'leq', 8804,
            'leqq', 8806,
            'leqslant', 10877,
            'les', 10877,
            'lescc', 10920,
            'lesdot', 10879,
            'lesdoto', 10881,
            'lesdotor', 10883,
            'lesges', 10899,
            'lessapprox', 10885,
            'lessdot', 8918,
            'lesseqgtr', 8922,
            'lesseqqgtr', 10891,
            'LessEqualGreater', 8922,
            'LessFullEqual', 8806,
            'LessGreater', 8822,
            'lessgtr', 8822,
            'LessLess', 10913,
            'lesssim', 8818,
            'LessSlantEqual', 10877,
            'LessTilde', 8818,
            'lfisht', 10620,
            'lfloor', 8970,
            'Lfr', 120079,
            'lfr', 120105,
            'lg', 8822,
            'lgE', 10897,
            'lHar', 10594,
            'lhard', 8637,
            'lharu', 8636,
            'lharul', 10602,
            'lhblk', 9604,
            'LJcy', 1033,
            'ljcy', 1113,
            'Ll', 8920,
            'll', 8810,
            'llarr', 8647,
            'llcorner', 8990,
            'Lleftarrow', 8666,
            'llhard', 10603,
            'lltri', 9722,
            'Lmidot', 319,
            'lmidot', 320,
            'lmoust', 9136,
            'lmoustache', 9136,
            'lnap', 10889,
            'lnapprox', 10889,
            'lnE', 8808,
            'lne', 10887,
            'lneq', 10887,
            'lneqq', 8808,
            'lnsim', 8934,
            'loang', 10220,
            'loarr', 8701,
            'lobrk', 10214,
            'LongLeftArrow', 10229,
            'Longleftarrow', 10232,
            'longleftarrow', 10229,
            'LongLeftRightArrow', 10231,
            'Longleftrightarrow', 10234,
            'longleftrightarrow', 10231,
            'longmapsto', 10236,
            'LongRightArrow', 10230,
            'Longrightarrow', 10233,
            'longrightarrow', 10230,
            'looparrowleft', 8619,
            'looparrowright', 8620,
            'lopar', 10629,
            'Lopf', 120131,
            'lopf', 120157,
            'loplus', 10797,
            'lotimes', 10804,
            'lowast', 8727,
            'lowbar', 95,
            'LowerLeftArrow', 8601,
            'LowerRightArrow', 8600,
            'loz', 9674,
            'lozenge', 9674,
            'lozf', 10731,
            'lpar', 40,
            'lparlt', 10643,
            'lrarr', 8646,
            'lrcorner', 8991,
            'lrhar', 8651,
            'lrhard', 10605,
            'lrm', 8206,
            'lrtri', 8895,
            'lsaquo', 8249,
            'Lscr', 8466,
            'lscr', 120001,
            'Lsh', 8624,
            'lsh', 8624,
            'lsim', 8818,
            'lsime', 10893,
            'lsimg', 10895,
            'lsqb', 91,
            'lsquo', 8216,
            'lsquor', 8218,
            'Lstrok', 321,
            'lstrok', 322,
            'LT', 60,
            'Lt', 8810,
            'lt', 60,
            'ltcc', 10918,
            'ltcir', 10873,
            'ltdot', 8918,
            'lthree', 8907,
            'ltimes', 8905,
            'ltlarr', 10614,
            'ltquest', 10875,
            'ltri', 9667,
            'ltrie', 8884,
            'ltrif', 9666,
            'ltrPar', 10646,
            'lurdshar', 10570,
            'luruhar', 10598,
            'macr', 175,
            'male', 9794,
            'malt', 10016,
            'maltese', 10016,
            'Map', 10501,
            'map', 8614,
            'mapsto', 8614,
            'mapstodown', 8615,
            'mapstoleft', 8612,
            'mapstoup', 8613,
            'marker', 9646,
            'mcomma', 10793,
            'Mcy', 1052,
            'mcy', 1084,
            'mdash', 8212,
            'mDDot', 8762,
            'measuredangle', 8737,
            'MediumSpace', 8287,
            'Mellintrf', 8499,
            'Mfr', 120080,
            'mfr', 120106,
            'mho', 8487,
            'micro', 181,
            'mid', 8739,
            'midast', 42,
            'midcir', 10992,
            'middot', 183,
            'minus', 8722,
            'minusb', 8863,
            'minusd', 8760,
            'minusdu', 10794,
            'MinusPlus', 8723,
            'mlcp', 10971,
            'mldr', 8230,
            'mnplus', 8723,
            'models', 8871,
            'Mopf', 120132,
            'mopf', 120158,
            'mp', 8723,
            'Mscr', 8499,
            'mscr', 120002,
            'mstpos', 8766,
            'Mu', 924,
            'mu', 956,
            'multimap', 8888,
            'mumap', 8888,
            'nabla', 8711,
            'Nacute', 323,
            'nacute', 324,
            'nap', 8777,
            'napos', 329,
            'napprox', 8777,
            'natur', 9838,
            'natural', 9838,
            'naturals', 8469,
            'nbsp', 160,
            'ncap', 10819,
            'Ncaron', 327,
            'ncaron', 328,
            'Ncedil', 325,
            'ncedil', 326,
            'ncong', 8775,
            'ncup', 10818,
            'Ncy', 1053,
            'ncy', 1085,
            'ndash', 8211,
            'ne', 8800,
            'nearhk', 10532,
            'neArr', 8663,
            'nearr', 8599,
            'nearrow', 8599,
            'NegativeMediumSpace', 8203,
            'NegativeThickSpace', 8203,
            'NegativeThinSpace', 8203,
            'NegativeVeryThinSpace', 8203,
            'nequiv', 8802,
            'nesear', 10536,
            'NestedGreaterGreater', 8811,
            'NestedLessLess', 8810,
            'NewLine', 10,
            'nexist', 8708,
            'nexists', 8708,
            'Nfr', 120081,
            'nfr', 120107,
            'nge', 8817,
            'ngeq', 8817,
            'ngsim', 8821,
            'ngt', 8815,
            'ngtr', 8815,
            'nhArr', 8654,
            'nharr', 8622,
            'nhpar', 10994,
            'ni', 8715,
            'nis', 8956,
            'nisd', 8954,
            'niv', 8715,
            'NJcy', 1034,
            'njcy', 1114,
            'nlArr', 8653,
            'nlarr', 8602,
            'nldr', 8229,
            'nle', 8816,
            'nLeftarrow', 8653,
            'nleftarrow', 8602,
            'nLeftrightarrow', 8654,
            'nleftrightarrow', 8622,
            'nleq', 8816,
            'nless', 8814,
            'nlsim', 8820,
            'nlt', 8814,
            'nltri', 8938,
            'nltrie', 8940,
            'nmid', 8740,
            'NoBreak', 8288,
            'NonBreakingSpace', 160,
            'Nopf', 8469,
            'nopf', 120159,
            'Not', 10988,
            'not', 172,
            'NotCongruent', 8802,
            'NotCupCap', 8813,
            'NotDoubleVerticalBar', 8742,
            'NotElement', 8713,
            'NotEqual', 8800,
            'NotExists', 8708,
            'NotGreater', 8815,
            'NotGreaterEqual', 8817,
            'NotGreaterLess', 8825,
            'NotGreaterTilde', 8821,
            'notin', 8713,
            'notinva', 8713,
            'notinvb', 8951,
            'notinvc', 8950,
            'NotLeftTriangle', 8938,
            'NotLeftTriangleEqual', 8940,
            'NotLess', 8814,
            'NotLessEqual', 8816,
            'NotLessGreater', 8824,
            'NotLessTilde', 8820,
            'notni', 8716,
            'notniva', 8716,
            'notnivb', 8958,
            'notnivc', 8957,
            'NotPrecedes', 8832,
            'NotPrecedesSlantEqual', 8928,
            'NotReverseElement', 8716,
            'NotRightTriangle', 8939,
            'NotRightTriangleEqual', 8941,
            'NotSquareSubsetEqual', 8930,
            'NotSquareSupersetEqual', 8931,
            'NotSubsetEqual', 8840,
            'NotSucceeds', 8833,
            'NotSucceedsSlantEqual', 8929,
            'NotSupersetEqual', 8841,
            'NotTilde', 8769,
            'NotTildeEqual', 8772,
            'NotTildeFullEqual', 8775,
            'NotTildeTilde', 8777,
            'NotVerticalBar', 8740,
            'npar', 8742,
            'nparallel', 8742,
            'npolint', 10772,
            'npr', 8832,
            'nprcue', 8928,
            'nprec', 8832,
            'nrArr', 8655,
            'nrarr', 8603,
            'nRightarrow', 8655,
            'nrightarrow', 8603,
            'nrtri', 8939,
            'nrtrie', 8941,
            'nsc', 8833,
            'nsccue', 8929,
            'Nscr', 119977,
            'nscr', 120003,
            'nshortmid', 8740,
            'nshortparallel', 8742,
            'nsim', 8769,
            'nsime', 8772,
            'nsimeq', 8772,
            'nsmid', 8740,
            'nspar', 8742,
            'nsqsube', 8930,
            'nsqsupe', 8931,
            'nsub', 8836,
            'nsube', 8840,
            'nsubseteq', 8840,
            'nsucc', 8833,
            'nsup', 8837,
            'nsupe', 8841,
            'nsupseteq', 8841,
            'ntgl', 8825,
            'Ntilde', 209,
            'ntilde', 241,
            'ntlg', 8824,
            'ntriangleleft', 8938,
            'ntrianglelefteq', 8940,
            'ntriangleright', 8939,
            'ntrianglerighteq', 8941,
            'Nu', 925,
            'nu', 957,
            'num', 35,
            'numero', 8470,
            'numsp', 8199,
            'nVDash', 8879,
            'nVdash', 8878,
            'nvDash', 8877,
            'nvdash', 8876,
            'nvHarr', 10500,
            'nvinfin', 10718,
            'nvlArr', 10498,
            'nvrArr', 10499,
            'nwarhk', 10531,
            'nwArr', 8662,
            'nwarr', 8598,
            'nwarrow', 8598,
            'nwnear', 10535,
            'Oacute', 211,
            'oacute', 243,
            'oast', 8859,
            'ocir', 8858,
            'Ocirc', 212,
            'ocirc', 244,
            'Ocy', 1054,
            'ocy', 1086,
            'odash', 8861,
            'Odblac', 336,
            'odblac', 337,
            'odiv', 10808,
            'odot', 8857,
            'odsold', 10684,
            'OElig', 338,
            'oelig', 339,
            'ofcir', 10687,
            'Ofr', 120082,
            'ofr', 120108,
            'ogon', 731,
            'Ograve', 210,
            'ograve', 242,
            'ogt', 10689,
            'ohbar', 10677,
            'ohm', 937,
            'oint', 8750,
            'olarr', 8634,
            'olcir', 10686,
            'olcross', 10683,
            'oline', 8254,
            'olt', 10688,
            'Omacr', 332,
            'omacr', 333,
            'Omega', 937,
            'omega', 969,
            'Omicron', 927,
            'omicron', 959,
            'omid', 10678,
            'ominus', 8854,
            'Oopf', 120134,
            'oopf', 120160,
            'opar', 10679,
            'OpenCurlyDoubleQuote', 8220,
            'OpenCurlyQuote', 8216,
            'operp', 10681,
            'oplus', 8853,
            'Or', 10836,
            'or', 8744,
            'orarr', 8635,
            'ord', 10845,
            'order', 8500,
            'orderof', 8500,
            'ordf', 170,
            'ordm', 186,
            'origof', 8886,
            'oror', 10838,
            'orslope', 10839,
            'orv', 10843,
            'oS', 9416,
            'Oscr', 119978,
            'oscr', 8500,
            'Oslash', 216,
            'oslash', 248,
            'osol', 8856,
            'Otilde', 213,
            'otilde', 245,
            'Otimes', 10807,
            'otimes', 8855,
            'otimesas', 10806,
            'Ouml', 214,
            'ouml', 246,
            'ovbar', 9021,
            'OverBar', 8254,
            'OverBrace', 9182,
            'OverBracket', 9140,
            'OverParenthesis', 9180,
            'par', 8741,
            'para', 182,
            'parallel', 8741,
            'parsim', 10995,
            'parsl', 11005,
            'part', 8706,
            'PartialD', 8706,
            'Pcy', 1055,
            'pcy', 1087,
            'percnt', 37,
            'period', 46,
            'permil', 8240,
            'perp', 8869,
            'pertenk', 8241,
            'Pfr', 120083,
            'pfr', 120109,
            'Phi', 934,
            'phi', 966,
            'phiv', 981,
            'phmmat', 8499,
            'phone', 9742,
            'Pi', 928,
            'pi', 960,
            'pitchfork', 8916,
            'piv', 982,
            'planck', 8463,
            'planckh', 8462,
            'plankv', 8463,
            'plus', 43,
            'plusacir', 10787,
            'plusb', 8862,
            'pluscir', 10786,
            'plusdo', 8724,
            'plusdu', 10789,
            'pluse', 10866,
            'PlusMinus', 177,
            'plusmn', 177,
            'plussim', 10790,
            'plustwo', 10791,
            'pm', 177,
            'Poincareplane', 8460,
            'pointint', 10773,
            'Popf', 8473,
            'popf', 120161,
            'pound', 163,
            'Pr', 10939,
            'pr', 8826,
            'prap', 10935,
            'prcue', 8828,
            'prE', 10931,
            'pre', 10927,
            'prec', 8826,
            'precapprox', 10935,
            'preccurlyeq', 8828,
            'Precedes', 8826,
            'PrecedesEqual', 10927,
            'PrecedesSlantEqual', 8828,
            'PrecedesTilde', 8830,
            'preceq', 10927,
            'precnapprox', 10937,
            'precneqq', 10933,
            'precnsim', 8936,
            'precsim', 8830,
            'Prime', 8243,
            'prime', 8242,
            'primes', 8473,
            'prnap', 10937,
            'prnE', 10933,
            'prnsim', 8936,
            'prod', 8719,
            'Product', 8719,
            'profalar', 9006,
            'profline', 8978,
            'profsurf', 8979,
            'prop', 8733,
            'Proportion', 8759,
            'Proportional', 8733,
            'propto', 8733,
            'prsim', 8830,
            'prurel', 8880,
            'Pscr', 119979,
            'pscr', 120005,
            'Psi', 936,
            'psi', 968,
            'puncsp', 8200,
            'Qfr', 120084,
            'qfr', 120110,
            'qint', 10764,
            'Qopf', 8474,
            'qopf', 120162,
            'qprime', 8279,
            'Qscr', 119980,
            'qscr', 120006,
            'quaternions', 8461,
            'quatint', 10774,
            'quest', 63,
            'questeq', 8799,
            'QUOT', 34,
            'quot', 34,
            'rAarr', 8667,
            'Racute', 340,
            'racute', 341,
            'radic', 8730,
            'raemptyv', 10675,
            'Rang', 10219,
            'rang', 10217,
            'rangd', 10642,
            'range', 10661,
            'rangle', 10217,
            'raquo', 187,
            'Rarr', 8608,
            'rArr', 8658,
            'rarr', 8594,
            'rarrap', 10613,
            'rarrb', 8677,
            'rarrbfs', 10528,
            'rarrc', 10547,
            'rarrfs', 10526,
            'rarrhk', 8618,
            'rarrlp', 8620,
            'rarrpl', 10565,
            'rarrsim', 10612,
            'Rarrtl', 10518,
            'rarrtl', 8611,
            'rarrw', 8605,
            'rAtail', 10524,
            'ratail', 10522,
            'ratio', 8758,
            'rationals', 8474,
            'RBarr', 10512,
            'rBarr', 10511,
            'rbarr', 10509,
            'rbbrk', 10099,
            'rbrace', 125,
            'rbrack', 93,
            'rbrke', 10636,
            'rbrksld', 10638,
            'rbrkslu', 10640,
            'Rcaron', 344,
            'rcaron', 345,
            'Rcedil', 342,
            'rcedil', 343,
            'rceil', 8969,
            'rcub', 125,
            'Rcy', 1056,
            'rcy', 1088,
            'rdca', 10551,
            'rdldhar', 10601,
            'rdquo', 8221,
            'rdquor', 8221,
            'rdsh', 8627,
            'Re', 8476,
            'real', 8476,
            'realine', 8475,
            'realpart', 8476,
            'reals', 8477,
            'rect', 9645,
            'REG', 174,
            'reg', 174,
            'ReverseElement', 8715,
            'ReverseEquilibrium', 8651,
            'ReverseUpEquilibrium', 10607,
            'rfisht', 10621,
            'rfloor', 8971,
            'Rfr', 8476,
            'rfr', 120111,
            'rHar', 10596,
            'rhard', 8641,
            'rharu', 8640,
            'rharul', 10604,
            'Rho', 929,
            'rho', 961,
            'rhov', 1009,
            'RightAngleBracket', 10217,
            'RightArrow', 8594,
            'Rightarrow', 8658,
            'rightarrow', 8594,
            'RightArrowBar', 8677,
            'RightArrowLeftArrow', 8644,
            'rightarrowtail', 8611,
            'RightCeiling', 8969,
            'RightDoubleBracket', 10215,
            'RightDownTeeVector', 10589,
            'RightDownVector', 8642,
            'RightDownVectorBar', 10581,
            'RightFloor', 8971,
            'rightharpoondown', 8641,
            'rightharpoonup', 8640,
            'rightleftarrows', 8644,
            'rightleftharpoons', 8652,
            'rightrightarrows', 8649,
            'rightsquigarrow', 8605,
            'RightTee', 8866,
            'RightTeeArrow', 8614,
            'RightTeeVector', 10587,
            'rightthreetimes', 8908,
            'RightTriangle', 8883,
            'RightTriangleBar', 10704,
            'RightTriangleEqual', 8885,
            'RightUpDownVector', 10575,
            'RightUpTeeVector', 10588,
            'RightUpVector', 8638,
            'RightUpVectorBar', 10580,
            'RightVector', 8640,
            'RightVectorBar', 10579,
            'ring', 730,
            'risingdotseq', 8787,
            'rlarr', 8644,
            'rlhar', 8652,
            'rlm', 8207,
            'rmoust', 9137,
            'rmoustache', 9137,
            'rnmid', 10990,
            'roang', 10221,
            'roarr', 8702,
            'robrk', 10215,
            'ropar', 10630,
            'Ropf', 8477,
            'ropf', 120163,
            'roplus', 10798,
            'rotimes', 10805,
            'RoundImplies', 10608,
            'rpar', 41,
            'rpargt', 10644,
            'rppolint', 10770,
            'rrarr', 8649,
            'Rrightarrow', 8667,
            'rsaquo', 8250,
            'Rscr', 8475,
            'rscr', 120007,
            'Rsh', 8625,
            'rsh', 8625,
            'rsqb', 93,
            'rsquo', 8217,
            'rsquor', 8217,
            'rthree', 8908,
            'rtimes', 8906,
            'rtri', 9657,
            'rtrie', 8885,
            'rtrif', 9656,
            'rtriltri', 10702,
            'RuleDelayed', 10740,
            'ruluhar', 10600,
            'rx', 8478,
            'Sacute', 346,
            'sacute', 347,
            'sbquo', 8218,
            'Sc', 10940,
            'sc', 8827,
            'scap', 10936,
            'Scaron', 352,
            'scaron', 353,
            'sccue', 8829,
            'scE', 10932,
            'sce', 10928,
            'Scedil', 350,
            'scedil', 351,
            'Scirc', 348,
            'scirc', 349,
            'scnap', 10938,
            'scnE', 10934,
            'scnsim', 8937,
            'scpolint', 10771,
            'scsim', 8831,
            'Scy', 1057,
            'scy', 1089,
            'sdot', 8901,
            'sdotb', 8865,
            'sdote', 10854,
            'searhk', 10533,
            'seArr', 8664,
            'searr', 8600,
            'searrow', 8600,
            'sect', 167,
            'semi', 59,
            'seswar', 10537,
            'setminus', 8726,
            'setmn', 8726,
            'sext', 10038,
            'Sfr', 120086,
            'sfr', 120112,
            'sfrown', 8994,
            'sharp', 9839,
            'SHCHcy', 1065,
            'shchcy', 1097,
            'SHcy', 1064,
            'shcy', 1096,
            'ShortDownArrow', 8595,
            'ShortLeftArrow', 8592,
            'shortmid', 8739,
            'shortparallel', 8741,
            'ShortRightArrow', 8594,
            'ShortUpArrow', 8593,
            'shy', 173,
            'Sigma', 931,
            'sigma', 963,
            'sigmaf', 962,
            'sigmav', 962,
            'sim', 8764,
            'simdot', 10858,
            'sime', 8771,
            'simeq', 8771,
            'simg', 10910,
            'simgE', 10912,
            'siml', 10909,
            'simlE', 10911,
            'simne', 8774,
            'simplus', 10788,
            'simrarr', 10610,
            'slarr', 8592,
            'SmallCircle', 8728,
            'smallsetminus', 8726,
            'smashp', 10803,
            'smeparsl', 10724,
            'smid', 8739,
            'smile', 8995,
            'smt', 10922,
            'smte', 10924,
            'SOFTcy', 1068,
            'softcy', 1100,
            'sol', 47,
            'solb', 10692,
            'solbar', 9023,
            'Sopf', 120138,
            'sopf', 120164,
            'spades', 9824,
            'spadesuit', 9824,
            'spar', 8741,
            'sqcap', 8851,
            'sqcup', 8852,
            'Sqrt', 8730,
            'sqsub', 8847,
            'sqsube', 8849,
            'sqsubset', 8847,
            'sqsubseteq', 8849,
            'sqsup', 8848,
            'sqsupe', 8850,
            'sqsupset', 8848,
            'sqsupseteq', 8850,
            'squ', 9633,
            'Square', 9633,
            'square', 9633,
            'SquareIntersection', 8851,
            'SquareSubset', 8847,
            'SquareSubsetEqual', 8849,
            'SquareSuperset', 8848,
            'SquareSupersetEqual', 8850,
            'SquareUnion', 8852,
            'squarf', 9642,
            'squf', 9642,
            'srarr', 8594,
            'Sscr', 119982,
            'sscr', 120008,
            'ssetmn', 8726,
            'ssmile', 8995,
            'sstarf', 8902,
            'Star', 8902,
            'star', 9734,
            'starf', 9733,
            'straightepsilon', 1013,
            'straightphi', 981,
            'strns', 175,
            'Sub', 8912,
            'sub', 8834,
            'subdot', 10941,
            'subE', 10949,
            'sube', 8838,
            'subedot', 10947,
            'submult', 10945,
            'subnE', 10955,
            'subne', 8842,
            'subplus', 10943,
            'subrarr', 10617,
            'Subset', 8912,
            'subset', 8834,
            'subseteq', 8838,
            'subseteqq', 10949,
            'SubsetEqual', 8838,
            'subsetneq', 8842,
            'subsetneqq', 10955,
            'subsim', 10951,
            'subsub', 10965,
            'subsup', 10963,
            'succ', 8827,
            'succapprox', 10936,
            'succcurlyeq', 8829,
            'Succeeds', 8827,
            'SucceedsEqual', 10928,
            'SucceedsSlantEqual', 8829,
            'SucceedsTilde', 8831,
            'succeq', 10928,
            'succnapprox', 10938,
            'succneqq', 10934,
            'succnsim', 8937,
            'succsim', 8831,
            'SuchThat', 8715,
            'Sum', 8721,
            'sum', 8721,
            'sung', 9834,
            'Sup', 8913,
            'sup', 8835,
            'sup1', 185,
            'sup2', 178,
            'sup3', 179,
            'supdot', 10942,
            'supdsub', 10968,
            'supE', 10950,
            'supe', 8839,
            'supedot', 10948,
            'Superset', 8835,
            'SupersetEqual', 8839,
            'suphsol', 10185,
            'suphsub', 10967,
            'suplarr', 10619,
            'supmult', 10946,
            'supnE', 10956,
            'supne', 8843,
            'supplus', 10944,
            'Supset', 8913,
            'supset', 8835,
            'supseteq', 8839,
            'supseteqq', 10950,
            'supsetneq', 8843,
            'supsetneqq', 10956,
            'supsim', 10952,
            'supsub', 10964,
            'supsup', 10966,
            'swarhk', 10534,
            'swArr', 8665,
            'swarr', 8601,
            'swarrow', 8601,
            'swnwar', 10538,
            'szlig', 223,
            'Tab', 9,
            'target', 8982,
            'Tau', 932,
            'tau', 964,
            'tbrk', 9140,
            'Tcaron', 356,
            'tcaron', 357,
            'Tcedil', 354,
            'tcedil', 355,
            'Tcy', 1058,
            'tcy', 1090,
            'tdot', 8411,
            'telrec', 8981,
            'Tfr', 120087,
            'tfr', 120113,
            'there4', 8756,
            'Therefore', 8756,
            'therefore', 8756,
            'Theta', 920,
            'theta', 952,
            'thetasym', 977,
            'thetav', 977,
            'thickapprox', 8776,
            'thicksim', 8764,
            'thinsp', 8201,
            'ThinSpace', 8201,
            'thkap', 8776,
            'thksim', 8764,
            'THORN', 222,
            'thorn', 254,
            'Tilde', 8764,
            'tilde', 732,
            'TildeEqual', 8771,
            'TildeFullEqual', 8773,
            'TildeTilde', 8776,
            'times', 215,
            'timesb', 8864,
            'timesbar', 10801,
            'timesd', 10800,
            'tint', 8749,
            'toea', 10536,
            'top', 8868,
            'topbot', 9014,
            'topcir', 10993,
            'Topf', 120139,
            'topf', 120165,
            'topfork', 10970,
            'tosa', 10537,
            'tprime', 8244,
            'TRADE', 8482,
            'trade', 8482,
            'triangle', 9653,
            'triangledown', 9663,
            'triangleleft', 9667,
            'trianglelefteq', 8884,
            'triangleq', 8796,
            'triangleright', 9657,
            'trianglerighteq', 8885,
            'tridot', 9708,
            'trie', 8796,
            'triminus', 10810,
            'TripleDot', 8411,
            'triplus', 10809,
            'trisb', 10701,
            'tritime', 10811,
            'trpezium', 9186,
            'Tscr', 119983,
            'tscr', 120009,
            'TScy', 1062,
            'tscy', 1094,
            'TSHcy', 1035,
            'tshcy', 1115,
            'Tstrok', 358,
            'tstrok', 359,
            'twixt', 8812,
            'twoheadleftarrow', 8606,
            'twoheadrightarrow', 8608,
            'Uacute', 218,
            'uacute', 250,
            'Uarr', 8607,
            'uArr', 8657,
            'uarr', 8593,
            'Uarrocir', 10569,
            'Ubrcy', 1038,
            'ubrcy', 1118,
            'Ubreve', 364,
            'ubreve', 365,
            'Ucirc', 219,
            'ucirc', 251,
            'Ucy', 1059,
            'ucy', 1091,
            'udarr', 8645,
            'Udblac', 368,
            'udblac', 369,
            'udhar', 10606,
            'ufisht', 10622,
            'Ufr', 120088,
            'ufr', 120114,
            'Ugrave', 217,
            'ugrave', 249,
            'uHar', 10595,
            'uharl', 8639,
            'uharr', 8638,
            'uhblk', 9600,
            'ulcorn', 8988,
            'ulcorner', 8988,
            'ulcrop', 8975,
            'ultri', 9720,
            'Umacr', 362,
            'umacr', 363,
            'uml', 168,
            'UnderBar', 95,
            'UnderBrace', 9183,
            'UnderBracket', 9141,
            'UnderParenthesis', 9181,
            'Union', 8899,
            'UnionPlus', 8846,
            'Uogon', 370,
            'uogon', 371,
            'Uopf', 120140,
            'uopf', 120166,
            'UpArrow', 8593,
            'Uparrow', 8657,
            'uparrow', 8593,
            'UpArrowBar', 10514,
            'UpArrowDownArrow', 8645,
            'UpDownArrow', 8597,
            'Updownarrow', 8661,
            'updownarrow', 8597,
            'UpEquilibrium', 10606,
            'upharpoonleft', 8639,
            'upharpoonright', 8638,
            'uplus', 8846,
            'UpperLeftArrow', 8598,
            'UpperRightArrow', 8599,
            'Upsi', 978,
            'upsi', 965,
            'upsih', 978,
            'Upsilon', 933,
            'upsilon', 965,
            'UpTee', 8869,
            'UpTeeArrow', 8613,
            'upuparrows', 8648,
            'urcorn', 8989,
            'urcorner', 8989,
            'urcrop', 8974,
            'Uring', 366,
            'uring', 367,
            'urtri', 9721,
            'Uscr', 119984,
            'uscr', 120010,
            'utdot', 8944,
            'Utilde', 360,
            'utilde', 361,
            'utri', 9653,
            'utrif', 9652,
            'uuarr', 8648,
            'Uuml', 220,
            'uuml', 252,
            'uwangle', 10663,
            'vangrt', 10652,
            'varepsilon', 1013,
            'varkappa', 1008,
            'varnothing', 8709,
            'varphi', 981,
            'varpi', 982,
            'varpropto', 8733,
            'vArr', 8661,
            'varr', 8597,
            'varrho', 1009,
            'varsigma', 962,
            'vartheta', 977,
            'vartriangleleft', 8882,
            'vartriangleright', 8883,
            'Vbar', 10987,
            'vBar', 10984,
            'vBarv', 10985,
            'Vcy', 1042,
            'vcy', 1074,
            'VDash', 8875,
            'Vdash', 8873,
            'vDash', 8872,
            'vdash', 8866,
            'Vdashl', 10982,
            'Vee', 8897,
            'vee', 8744,
            'veebar', 8891,
            'veeeq', 8794,
            'vellip', 8942,
            'Verbar', 8214,
            'verbar', 124,
            'Vert', 8214,
            'vert', 124,
            'VerticalBar', 8739,
            'VerticalLine', 124,
            'VerticalSeparator', 10072,
            'VerticalTilde', 8768,
            'VeryThinSpace', 8202,
            'Vfr', 120089,
            'vfr', 120115,
            'vltri', 8882,
            'Vopf', 120141,
            'vopf', 120167,
            'vprop', 8733,
            'vrtri', 8883,
            'Vscr', 119985,
            'vscr', 120011,
            'Vvdash', 8874,
            'vzigzag', 10650,
            'Wcirc', 372,
            'wcirc', 373,
            'wedbar', 10847,
            'Wedge', 8896,
            'wedge', 8743,
            'wedgeq', 8793,
            'weierp', 8472,
            'Wfr', 120090,
            'wfr', 120116,
            'Wopf', 120142,
            'wopf', 120168,
            'wp', 8472,
            'wr', 8768,
            'wreath', 8768,
            'Wscr', 119986,
            'wscr', 120012,
            'xcap', 8898,
            'xcirc', 9711,
            'xcup', 8899,
            'xdtri', 9661,
            'Xfr', 120091,
            'xfr', 120117,
            'xhArr', 10234,
            'xharr', 10231,
            'Xi', 926,
            'xi', 958,
            'xlArr', 10232,
            'xlarr', 10229,
            'xmap', 10236,
            'xnis', 8955,
            'xodot', 10752,
            'Xopf', 120143,
            'xopf', 120169,
            'xoplus', 10753,
            'xotime', 10754,
            'xrArr', 10233,
            'xrarr', 10230,
            'Xscr', 119987,
            'xscr', 120013,
            'xsqcup', 10758,
            'xuplus', 10756,
            'xutri', 9651,
            'xvee', 8897,
            'xwedge', 8896,
            'Yacute', 221,
            'yacute', 253,
            'YAcy', 1071,
            'yacy', 1103,
            'Ycirc', 374,
            'ycirc', 375,
            'Ycy', 1067,
            'ycy', 1099,
            'yen', 165,
            'Yfr', 120092,
            'yfr', 120118,
            'YIcy', 1031,
            'yicy', 1111,
            'Yopf', 120144,
            'yopf', 120170,
            'Yscr', 119988,
            'yscr', 120014,
            'YUcy', 1070,
            'yucy', 1102,
            'Yuml', 376,
            'yuml', 255,
            'Zacute', 377,
            'zacute', 378,
            'Zcaron', 381,
            'zcaron', 382,
            'Zcy', 1047,
            'zcy', 1079,
            'Zdot', 379,
            'zdot', 380,
            'zeetrf', 8488,
            'ZeroWidthSpace', 8203,
            'Zeta', 918,
            'zeta', 950,
            'Zfr', 8488,
            'zfr', 120119,
            'ZHcy', 1046,
            'zhcy', 1078,
            'zigrarr', 8669,
            'Zopf', 8484,
            'zopf', 120171,
            'Zscr', 119989,
            'zscr', 120015,
            'zwj', 8205,
            'zwnj', 8204,
        );

        #   HTML5 Composed Character References

        #   These are logical characters formed by combining multiple
        #   Unicode code points.  The code points which make up the
        #   character are given in a comma-separated string.

        #   Remember to update $#HTML_COMPOSED_CHARACTER_REFERENCES pre-dimension above!
        @HTML_COMPOSED_CHARACTER_REFERENCES = (
            #   From https://www.w3.org/TR/html5/entities.json
            'acE', '8766, 819',
            'bne', '61, 8421',
            'bnequiv', '8801, 8421',
            'caps', '8745, 65024',
            'cups', '8746, 65024',
            'fjlig', '102, 106',
            'gesl', '8923, 65024',
            'gvertneqq', '8809, 65024',
            'gvnE', '8809, 65024',
            'lates', '10925, 65024',
            'lesg', '8922, 65024',
            'lvertneqq', '8808, 65024',
            'lvnE', '8808, 65024',
            'nang', '8736, 8402',
            'napE', '10864, 824',
            'napid', '8779, 824',
            'nbump', '8782, 824',
            'nbumpe', '8783, 824',
            'ncongdot', '10861, 824',
            'nedot', '8784, 824',
            'nesim', '8770, 824',
            'ngE', '8807, 824',
            'ngeqq', '8807, 824',
            'ngeqslant', '10878, 824',
            'nges', '10878, 824',
            'nGg', '8921, 824',
            'nGt', '8811, 8402',
            'nGtv', '8811, 824',
            'nlE', '8806, 824',
            'nleqq', '8806, 824',
            'nleqslant', '10877, 824',
            'nles', '10877, 824',
            'nLl', '8920, 824',
            'nLt', '8810, 8402',
            'nLtv', '8810, 824',
            'NotEqualTilde', '8770, 824',
            'NotGreaterFullEqual', '8807, 824',
            'NotGreaterGreater', '8811, 824',
            'NotGreaterSlantEqual', '10878, 824',
            'NotHumpDownHump', '8782, 824',
            'NotHumpEqual', '8783, 824',
            'notindot', '8949, 824',
            'notinE', '8953, 824',
            'NotLeftTriangleBar', '10703, 824',
            'NotLessLess', '8810, 824',
            'NotLessSlantEqual', '10877, 824',
            'NotNestedGreaterGreater', '10914, 824',
            'NotNestedLessLess', '10913, 824',
            'NotPrecedesEqual', '10927, 824',
            'NotRightTriangleBar', '10704, 824',
            'NotSquareSubset', '8847, 824',
            'NotSquareSuperset', '8848, 824',
            'NotSubset', '8834, 8402',
            'NotSucceedsEqual', '10928, 824',
            'NotSucceedsTilde', '8831, 824',
            'NotSuperset', '8835, 8402',
            'nparsl', '11005, 8421',
            'npart', '8706, 824',
            'npre', '10927, 824',
            'npreceq', '10927, 824',
            'nrarrc', '10547, 824',
            'nrarrw', '8605, 824',
            'nsce', '10928, 824',
            'nsubE', '10949, 824',
            'nsubset', '8834, 8402',
            'nsubseteqq', '10949, 824',
            'nsucceq', '10928, 824',
            'nsupE', '10950, 824',
            'nsupset', '8835, 8402',
            'nsupseteqq', '10950, 824',
            'nvap', '8781, 8402',
            'nvge', '8805, 8402',
            'nvgt', '62, 8402',
            'nvle', '8804, 8402',
            'nvlt', '60, 8402',
            'nvltrie', '8884, 8402',
            'nvrtrie', '8885, 8402',
            'nvsim', '8764, 8402',
            'race', '8765, 817',
            'smtes', '10924, 65024',
            'sqcaps', '8851, 65024',
            'sqcups', '8852, 65024',
            'ThickSpace', '8287, 8202',
            'varsubsetneq', '8842, 65024',
            'varsubsetneqq', '10955, 65024',
            'varsupsetneq', '8843, 65024',
            'varsupsetneqq', '10956, 65024',
            'vnsub', '8834, 8402',
            'vnsup', '8835, 8402',
            'vsubnE', '10955, 65024',
            'vsubne', '8842, 65024',
            'vsupnE', '10956, 65024',
            'vsupne', '8843, 65024',
        );
    }

__DATA__
0000 <control> NULL, NUL, Ctrl-@
0001 <control> START OF HEADING, SOH, Ctrl-A
0002 <control> START OF TEXT, STX, Ctrl-B
0003 <control> END OF TEXT, ETX, Ctrl-C
0004 <control> END OF TRANSMISSION, EOT, Ctrl-D
0005 <control> ENQUIRY, ENQ, Ctrl-E
0006 <control> ACKNOWLEDGE, ACK, Ctrl-F
0007 <control> BELL, ALERT, BEL, Ctrl-G
0008 <control> BACKSPACE, BS, Ctrl-H
0009 <control> CHARACTER TABULATION, HORIZONTAL TABULATION, HT TAB, Ctrl-I
000A <control> LINE FEED (LF), END OF LINE LINE FEED NEW LINE, EOL LF NL, Ctrl-J
000B <control> LINE TABULATION, VERTICAL TABULATION, VT, Ctrl-K
000C <control> FORM FEED (FF), FORM FEED, FF, Ctrl-L
000D <control> CARRIAGE RETURN (CR), CARRIAGE RETURN, CR, Ctrl-M
000E <control> SHIFT OUT, LOCKING-SHIFT ONE, SO, Ctrl-N
000F <control> SHIFT IN, LOCKING-SHIFT ZERO, SI, Ctrl-O
0010 <control> DATA LINK ESCAPE, DLE, Ctrl-P
0011 <control> DEVICE CONTROL ONE, DC1, Ctrl-Q
0012 <control> DEVICE CONTROL TWO, DC2, Ctrl-R
0013 <control> DEVICE CONTROL THREE, DC3, Ctrl-S
0014 <control> DEVICE CONTROL FOUR, DC4, Ctrl-T
0015 <control> NEGATIVE ACKNOWLEDGE, NAK, Ctrl-U
0016 <control> SYNCHRONOUS IDLE, SYN, Ctrl-V
0017 <control> END OF TRANSMISSION BLOCK, ETB, Ctrl-W
0018 <control> CANCEL, CAN, Ctrl-X
0019 <control> END OF MEDIUM, EOM, Ctrl-Y
001A <control> SUBSTITUTE, SUB, Ctrl-Z
001B <control> ESCAPE, ESC, Ctrl-[
001C <control> INFORMATION SEPARATOR FOUR, FILE SEPARATOR, FS, Ctrl-\
001D <control> INFORMATION SEPARATOR THREE, GROUP SEPARATOR, GS, Ctrl-]
001E <control> INFORMATION SEPARATOR TWO, RECORD SEPARATOR, RS, Ctrl-^
001F <control> INFORMATION SEPARATOR ONE, UNIT SEPARATOR, US, Ctrl-_
0020 SPACE, SP
0021 EXCLAMATION MARK
0022 QUOTATION MARK
0023 NUMBER SIGN
0024 DOLLAR SIGN
0025 PERCENT SIGN
0026 AMPERSAND
0027 APOSTROPHE
0028 LEFT PARENTHESIS
0029 RIGHT PARENTHESIS
002A ASTERISK
002B PLUS SIGN
002C COMMA
002D HYPHEN-MINUS
002E FULL STOP
002F SOLIDUS
0030 DIGIT ZERO
0031 DIGIT ONE
0032 DIGIT TWO
0033 DIGIT THREE
0034 DIGIT FOUR
0035 DIGIT FIVE
0036 DIGIT SIX
0037 DIGIT SEVEN
0038 DIGIT EIGHT
0039 DIGIT NINE
003A COLON
003B SEMICOLON
003C LESS-THAN SIGN
003D EQUALS SIGN
003E GREATER-THAN SIGN
003F QUESTION MARK
0040 COMMERCIAL AT
0041 LATIN CAPITAL LETTER A
0042 LATIN CAPITAL LETTER B
0043 LATIN CAPITAL LETTER C
0044 LATIN CAPITAL LETTER D
0045 LATIN CAPITAL LETTER E
0046 LATIN CAPITAL LETTER F
0047 LATIN CAPITAL LETTER G
0048 LATIN CAPITAL LETTER H
0049 LATIN CAPITAL LETTER I
004A LATIN CAPITAL LETTER J
004B LATIN CAPITAL LETTER K
004C LATIN CAPITAL LETTER L
004D LATIN CAPITAL LETTER M
004E LATIN CAPITAL LETTER N
004F LATIN CAPITAL LETTER O
0050 LATIN CAPITAL LETTER P
0051 LATIN CAPITAL LETTER Q
0052 LATIN CAPITAL LETTER R
0053 LATIN CAPITAL LETTER S
0054 LATIN CAPITAL LETTER T
0055 LATIN CAPITAL LETTER U
0056 LATIN CAPITAL LETTER V
0057 LATIN CAPITAL LETTER W
0058 LATIN CAPITAL LETTER X
0059 LATIN CAPITAL LETTER Y
005A LATIN CAPITAL LETTER Z
005B LEFT SQUARE BRACKET
005C REVERSE SOLIDUS
005D RIGHT SQUARE BRACKET
005E CIRCUMFLEX ACCENT
005F LOW LINE
0060 GRAVE ACCENT
0061 LATIN SMALL LETTER A
0062 LATIN SMALL LETTER B
0063 LATIN SMALL LETTER C
0064 LATIN SMALL LETTER D
0065 LATIN SMALL LETTER E
0066 LATIN SMALL LETTER F
0067 LATIN SMALL LETTER G
0068 LATIN SMALL LETTER H
0069 LATIN SMALL LETTER I
006A LATIN SMALL LETTER J
006B LATIN SMALL LETTER K
006C LATIN SMALL LETTER L
006D LATIN SMALL LETTER M
006E LATIN SMALL LETTER N
006F LATIN SMALL LETTER O
0070 LATIN SMALL LETTER P
0071 LATIN SMALL LETTER Q
0072 LATIN SMALL LETTER R
0073 LATIN SMALL LETTER S
0074 LATIN SMALL LETTER T
0075 LATIN SMALL LETTER U
0076 LATIN SMALL LETTER V
0077 LATIN SMALL LETTER W
0078 LATIN SMALL LETTER X
0079 LATIN SMALL LETTER Y
007A LATIN SMALL LETTER Z
007B LEFT CURLY BRACKET
007C VERTICAL LINE
007D RIGHT CURLY BRACKET
007E TILDE
007F <control> DELETE, DEL
0080 XXX, PAD, PADDING CHARACTER
0081 XXX, HOP, HIGH OCTET PRESET
0082 <control> BREAK PERMITTED HERE, BPH
0083 <control> NO BREAK HERE, NBH
0084 <control> XXX, INDEX, IND
0085 <control> NEXT LINE (NEL), NEXT LINE, NEL
0086 <control> START OF SELECTED AREA, SSA
0087 <control> END OF SELECTED AREA, ESA
0088 <control> CHARACTER TABULATION SET, HORIZONTAL TABULATION SET, HTS
0089 <control> CHARACTER TABULATION WITH JUSTIFICATION, HORIZONTAL TABULATION WITH JUSTIFICATION, HTJ
008A <control> LINE TABULATION SET, VERTICAL TABULATION SET, VTS
008B <control> PARTIAL LINE FORWARD, PARTIAL LINE DOWN, PLD
008C <control> PARTIAL LINE BACKWARD, PARTIAL LINE UP, PLU
008D <control> REVERSE LINE FEED, REVERSE INDEX, RI
008E <control> SINGLE SHIFT TWO, SINGLE-SHIFT-2, SS2
008F <control> SINGLE SHIFT THREE, SINGLE-SHIFT-3, SS3
0090 <control> DEVICE CONTROL STRING, DCS
0091 <control> PRIVATE USE ONE, PRIVATE USE-1, PU1
0092 <control> PRIVATE USE TWO, PRIVATE USE-2, PU2
0093 <control> SET TRANSMIT STATE, STS
0094 <control> CANCEL CHARACTER, CCH
0095 <control> MESSAGE WAITING, MW
0096 <control> START OF GUARDED AREA, START OF PROTECTED AREA, SPA
0097 <control> END OF GUARDED AREA, END OF PROTECTED AREA, EPA
0098 <control> START OF STRING, SOS
0099 XXX, SGC, SINGLE GRAPHIC CHARACTER INTRODUCER
009A <control> SINGLE CHARACTER INTRODUCER, SCI
009B <control> CONTROL SEQUENCE INTRODUCER, CSI
009C <control> STRING TERMINATOR, ST
009D <control> OPERATING SYSTEM COMMAND, OSC
009E <control> PRIVACY MESSAGE, PM
009F <control> APPLICATION PROGRAM COMMAND, APC
00A0 NO-BREAK SPACE, NBSP
00A1 INVERTED EXCLAMATION MARK
00A2 CENT SIGN
00A3 POUND SIGN
00A4 CURRENCY SIGN
00A5 YEN SIGN
00A6 BROKEN BAR
00A7 SECTION SIGN
00A8 DIAERESIS
00A9 COPYRIGHT SIGN
00AA FEMININE ORDINAL INDICATOR
00AB LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
00AC NOT SIGN
00AD SOFT HYPHEN, SHY
00AE REGISTERED SIGN
00AF MACRON
00B0 DEGREE SIGN
00B1 PLUS-MINUS SIGN
00B2 SUPERSCRIPT TWO
00B3 SUPERSCRIPT THREE
00B4 ACUTE ACCENT
00B5 MICRO SIGN
00B6 PILCROW SIGN
00B7 MIDDLE DOT
00B8 CEDILLA
00B9 SUPERSCRIPT ONE
00BA MASCULINE ORDINAL INDICATOR
00BB RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
00BC VULGAR FRACTION ONE QUARTER
00BD VULGAR FRACTION ONE HALF
00BE VULGAR FRACTION THREE QUARTERS
00BF INVERTED QUESTION MARK
00C0 LATIN CAPITAL LETTER A WITH GRAVE
00C1 LATIN CAPITAL LETTER A WITH ACUTE
00C2 LATIN CAPITAL LETTER A WITH CIRCUMFLEX
00C3 LATIN CAPITAL LETTER A WITH TILDE
00C4 LATIN CAPITAL LETTER A WITH DIAERESIS
00C5 LATIN CAPITAL LETTER A WITH RING ABOVE
00C6 LATIN CAPITAL LETTER AE
00C7 LATIN CAPITAL LETTER C WITH CEDILLA
00C8 LATIN CAPITAL LETTER E WITH GRAVE
00C9 LATIN CAPITAL LETTER E WITH ACUTE
00CA LATIN CAPITAL LETTER E WITH CIRCUMFLEX
00CB LATIN CAPITAL LETTER E WITH DIAERESIS
00CC LATIN CAPITAL LETTER I WITH GRAVE
00CD LATIN CAPITAL LETTER I WITH ACUTE
00CE LATIN CAPITAL LETTER I WITH CIRCUMFLEX
00CF LATIN CAPITAL LETTER I WITH DIAERESIS
00D0 LATIN CAPITAL LETTER ETH
00D1 LATIN CAPITAL LETTER N WITH TILDE
00D2 LATIN CAPITAL LETTER O WITH GRAVE
00D3 LATIN CAPITAL LETTER O WITH ACUTE
00D4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX
00D5 LATIN CAPITAL LETTER O WITH TILDE
00D6 LATIN CAPITAL LETTER O WITH DIAERESIS
00D7 MULTIPLICATION SIGN
00D8 LATIN CAPITAL LETTER O WITH STROKE
00D9 LATIN CAPITAL LETTER U WITH GRAVE
00DA LATIN CAPITAL LETTER U WITH ACUTE
00DB LATIN CAPITAL LETTER U WITH CIRCUMFLEX
00DC LATIN CAPITAL LETTER U WITH DIAERESIS
00DD LATIN CAPITAL LETTER Y WITH ACUTE
00DE LATIN CAPITAL LETTER THORN
00DF LATIN SMALL LETTER SHARP S
00E0 LATIN SMALL LETTER A WITH GRAVE
00E1 LATIN SMALL LETTER A WITH ACUTE
00E2 LATIN SMALL LETTER A WITH CIRCUMFLEX
00E3 LATIN SMALL LETTER A WITH TILDE
00E4 LATIN SMALL LETTER A WITH DIAERESIS
00E5 LATIN SMALL LETTER A WITH RING ABOVE
00E6 LATIN SMALL LETTER AE
00E7 LATIN SMALL LETTER C WITH CEDILLA
00E8 LATIN SMALL LETTER E WITH GRAVE
00E9 LATIN SMALL LETTER E WITH ACUTE
00EA LATIN SMALL LETTER E WITH CIRCUMFLEX
00EB LATIN SMALL LETTER E WITH DIAERESIS
00EC LATIN SMALL LETTER I WITH GRAVE
00ED LATIN SMALL LETTER I WITH ACUTE
00EE LATIN SMALL LETTER I WITH CIRCUMFLEX
00EF LATIN SMALL LETTER I WITH DIAERESIS
00F0 LATIN SMALL LETTER ETH
00F1 LATIN SMALL LETTER N WITH TILDE
00F2 LATIN SMALL LETTER O WITH GRAVE
00F3 LATIN SMALL LETTER O WITH ACUTE
00F4 LATIN SMALL LETTER O WITH CIRCUMFLEX
00F5 LATIN SMALL LETTER O WITH TILDE
00F6 LATIN SMALL LETTER O WITH DIAERESIS
00F7 DIVISION SIGN
00F8 LATIN SMALL LETTER O WITH STROKE
00F9 LATIN SMALL LETTER U WITH GRAVE
00FA LATIN SMALL LETTER U WITH ACUTE
00FB LATIN SMALL LETTER U WITH CIRCUMFLEX
00FC LATIN SMALL LETTER U WITH DIAERESIS
00FD LATIN SMALL LETTER Y WITH ACUTE
00FE LATIN SMALL LETTER THORN
00FF LATIN SMALL LETTER Y WITH DIAERESIS
0100 LATIN CAPITAL LETTER A WITH MACRON
0101 LATIN SMALL LETTER A WITH MACRON
0102 LATIN CAPITAL LETTER A WITH BREVE
0103 LATIN SMALL LETTER A WITH BREVE
0104 LATIN CAPITAL LETTER A WITH OGONEK
0105 LATIN SMALL LETTER A WITH OGONEK
0106 LATIN CAPITAL LETTER C WITH ACUTE
0107 LATIN SMALL LETTER C WITH ACUTE
0108 LATIN CAPITAL LETTER C WITH CIRCUMFLEX
0109 LATIN SMALL LETTER C WITH CIRCUMFLEX
010A LATIN CAPITAL LETTER C WITH DOT ABOVE
010B LATIN SMALL LETTER C WITH DOT ABOVE
010C LATIN CAPITAL LETTER C WITH CARON
010D LATIN SMALL LETTER C WITH CARON
010E LATIN CAPITAL LETTER D WITH CARON
010F LATIN SMALL LETTER D WITH CARON
0110 LATIN CAPITAL LETTER D WITH STROKE
0111 LATIN SMALL LETTER D WITH STROKE
0112 LATIN CAPITAL LETTER E WITH MACRON
0113 LATIN SMALL LETTER E WITH MACRON
0114 LATIN CAPITAL LETTER E WITH BREVE
0115 LATIN SMALL LETTER E WITH BREVE
0116 LATIN CAPITAL LETTER E WITH DOT ABOVE
0117 LATIN SMALL LETTER E WITH DOT ABOVE
0118 LATIN CAPITAL LETTER E WITH OGONEK
0119 LATIN SMALL LETTER E WITH OGONEK
011A LATIN CAPITAL LETTER E WITH CARON
011B LATIN SMALL LETTER E WITH CARON
011C LATIN CAPITAL LETTER G WITH CIRCUMFLEX
011D LATIN SMALL LETTER G WITH CIRCUMFLEX
011E LATIN CAPITAL LETTER G WITH BREVE
011F LATIN SMALL LETTER G WITH BREVE
0120 LATIN CAPITAL LETTER G WITH DOT ABOVE
0121 LATIN SMALL LETTER G WITH DOT ABOVE
0122 LATIN CAPITAL LETTER G WITH CEDILLA
0123 LATIN SMALL LETTER G WITH CEDILLA
0124 LATIN CAPITAL LETTER H WITH CIRCUMFLEX
0125 LATIN SMALL LETTER H WITH CIRCUMFLEX
0126 LATIN CAPITAL LETTER H WITH STROKE
0127 LATIN SMALL LETTER H WITH STROKE
0128 LATIN CAPITAL LETTER I WITH TILDE
0129 LATIN SMALL LETTER I WITH TILDE
012A LATIN CAPITAL LETTER I WITH MACRON
012B LATIN SMALL LETTER I WITH MACRON
012C LATIN CAPITAL LETTER I WITH BREVE
012D LATIN SMALL LETTER I WITH BREVE
012E LATIN CAPITAL LETTER I WITH OGONEK
012F LATIN SMALL LETTER I WITH OGONEK
0130 LATIN CAPITAL LETTER I WITH DOT ABOVE
0131 LATIN SMALL LETTER DOTLESS I
0132 LATIN CAPITAL LIGATURE IJ
0133 LATIN SMALL LIGATURE IJ
0134 LATIN CAPITAL LETTER J WITH CIRCUMFLEX
0135 LATIN SMALL LETTER J WITH CIRCUMFLEX
0136 LATIN CAPITAL LETTER K WITH CEDILLA
0137 LATIN SMALL LETTER K WITH CEDILLA
0138 LATIN SMALL LETTER KRA
0139 LATIN CAPITAL LETTER L WITH ACUTE
013A LATIN SMALL LETTER L WITH ACUTE
013B LATIN CAPITAL LETTER L WITH CEDILLA
013C LATIN SMALL LETTER L WITH CEDILLA
013D LATIN CAPITAL LETTER L WITH CARON
013E LATIN SMALL LETTER L WITH CARON
013F LATIN CAPITAL LETTER L WITH MIDDLE DOT
0140 LATIN SMALL LETTER L WITH MIDDLE DOT
0141 LATIN CAPITAL LETTER L WITH STROKE
0142 LATIN SMALL LETTER L WITH STROKE
0143 LATIN CAPITAL LETTER N WITH ACUTE
0144 LATIN SMALL LETTER N WITH ACUTE
0145 LATIN CAPITAL LETTER N WITH CEDILLA
0146 LATIN SMALL LETTER N WITH CEDILLA
0147 LATIN CAPITAL LETTER N WITH CARON
0148 LATIN SMALL LETTER N WITH CARON
0149 LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
014A LATIN CAPITAL LETTER ENG
014B LATIN SMALL LETTER ENG
014C LATIN CAPITAL LETTER O WITH MACRON
014D LATIN SMALL LETTER O WITH MACRON
014E LATIN CAPITAL LETTER O WITH BREVE
014F LATIN SMALL LETTER O WITH BREVE
0150 LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
0151 LATIN SMALL LETTER O WITH DOUBLE ACUTE
0152 LATIN CAPITAL LIGATURE OE
0153 LATIN SMALL LIGATURE OE
0154 LATIN CAPITAL LETTER R WITH ACUTE
0155 LATIN SMALL LETTER R WITH ACUTE
0156 LATIN CAPITAL LETTER R WITH CEDILLA
0157 LATIN SMALL LETTER R WITH CEDILLA
0158 LATIN CAPITAL LETTER R WITH CARON
0159 LATIN SMALL LETTER R WITH CARON
015A LATIN CAPITAL LETTER S WITH ACUTE
015B LATIN SMALL LETTER S WITH ACUTE
015C LATIN CAPITAL LETTER S WITH CIRCUMFLEX
015D LATIN SMALL LETTER S WITH CIRCUMFLEX
015E LATIN CAPITAL LETTER S WITH CEDILLA
015F LATIN SMALL LETTER S WITH CEDILLA
0160 LATIN CAPITAL LETTER S WITH CARON
0161 LATIN SMALL LETTER S WITH CARON
0162 LATIN CAPITAL LETTER T WITH CEDILLA
0163 LATIN SMALL LETTER T WITH CEDILLA
0164 LATIN CAPITAL LETTER T WITH CARON
0165 LATIN SMALL LETTER T WITH CARON
0166 LATIN CAPITAL LETTER T WITH STROKE
0167 LATIN SMALL LETTER T WITH STROKE
0168 LATIN CAPITAL LETTER U WITH TILDE
0169 LATIN SMALL LETTER U WITH TILDE
016A LATIN CAPITAL LETTER U WITH MACRON
016B LATIN SMALL LETTER U WITH MACRON
016C LATIN CAPITAL LETTER U WITH BREVE
016D LATIN SMALL LETTER U WITH BREVE
016E LATIN CAPITAL LETTER U WITH RING ABOVE
016F LATIN SMALL LETTER U WITH RING ABOVE
0170 LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
0171 LATIN SMALL LETTER U WITH DOUBLE ACUTE
0172 LATIN CAPITAL LETTER U WITH OGONEK
0173 LATIN SMALL LETTER U WITH OGONEK
0174 LATIN CAPITAL LETTER W WITH CIRCUMFLEX
0175 LATIN SMALL LETTER W WITH CIRCUMFLEX
0176 LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
0177 LATIN SMALL LETTER Y WITH CIRCUMFLEX
0178 LATIN CAPITAL LETTER Y WITH DIAERESIS
0179 LATIN CAPITAL LETTER Z WITH ACUTE
017A LATIN SMALL LETTER Z WITH ACUTE
017B LATIN CAPITAL LETTER Z WITH DOT ABOVE
017C LATIN SMALL LETTER Z WITH DOT ABOVE
017D LATIN CAPITAL LETTER Z WITH CARON
017E LATIN SMALL LETTER Z WITH CARON
017F LATIN SMALL LETTER LONG S
0180 LATIN SMALL LETTER B WITH STROKE
0181 LATIN CAPITAL LETTER B WITH HOOK
0182 LATIN CAPITAL LETTER B WITH TOPBAR
0183 LATIN SMALL LETTER B WITH TOPBAR
0184 LATIN CAPITAL LETTER TONE SIX
0185 LATIN SMALL LETTER TONE SIX
0186 LATIN CAPITAL LETTER OPEN O
0187 LATIN CAPITAL LETTER C WITH HOOK
0188 LATIN SMALL LETTER C WITH HOOK
0189 LATIN CAPITAL LETTER AFRICAN D
018A LATIN CAPITAL LETTER D WITH HOOK
018B LATIN CAPITAL LETTER D WITH TOPBAR
018C LATIN SMALL LETTER D WITH TOPBAR
018D LATIN SMALL LETTER TURNED DELTA
018E LATIN CAPITAL LETTER REVERSED E
018F LATIN CAPITAL LETTER SCHWA
0190 LATIN CAPITAL LETTER OPEN E
0191 LATIN CAPITAL LETTER F WITH HOOK
0192 LATIN SMALL LETTER F WITH HOOK
0193 LATIN CAPITAL LETTER G WITH HOOK
0194 LATIN CAPITAL LETTER GAMMA
0195 LATIN SMALL LETTER HV
0196 LATIN CAPITAL LETTER IOTA
0197 LATIN CAPITAL LETTER I WITH STROKE
0198 LATIN CAPITAL LETTER K WITH HOOK
0199 LATIN SMALL LETTER K WITH HOOK
019A LATIN SMALL LETTER L WITH BAR
019B LATIN SMALL LETTER LAMBDA WITH STROKE
019C LATIN CAPITAL LETTER TURNED M
019D LATIN CAPITAL LETTER N WITH LEFT HOOK
019E LATIN SMALL LETTER N WITH LONG RIGHT LEG
019F LATIN CAPITAL LETTER O WITH MIDDLE TILDE
01A0 LATIN CAPITAL LETTER O WITH HORN
01A1 LATIN SMALL LETTER O WITH HORN
01A2 LATIN CAPITAL LETTER OI
01A3 LATIN SMALL LETTER OI
01A4 LATIN CAPITAL LETTER P WITH HOOK
01A5 LATIN SMALL LETTER P WITH HOOK
01A6 LATIN LETTER YR
01A7 LATIN CAPITAL LETTER TONE TWO
01A8 LATIN SMALL LETTER TONE TWO
01A9 LATIN CAPITAL LETTER ESH
01AA LATIN LETTER REVERSED ESH LOOP
01AB LATIN SMALL LETTER T WITH PALATAL HOOK
01AC LATIN CAPITAL LETTER T WITH HOOK
01AD LATIN SMALL LETTER T WITH HOOK
01AE LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
01AF LATIN CAPITAL LETTER U WITH HORN
01B0 LATIN SMALL LETTER U WITH HORN
01B1 LATIN CAPITAL LETTER UPSILON
01B2 LATIN CAPITAL LETTER V WITH HOOK
01B3 LATIN CAPITAL LETTER Y WITH HOOK
01B4 LATIN SMALL LETTER Y WITH HOOK
01B5 LATIN CAPITAL LETTER Z WITH STROKE
01B6 LATIN SMALL LETTER Z WITH STROKE
01B7 LATIN CAPITAL LETTER EZH
01B8 LATIN CAPITAL LETTER EZH REVERSED
01B9 LATIN SMALL LETTER EZH REVERSED
01BA LATIN SMALL LETTER EZH WITH TAIL
01BB LATIN LETTER TWO WITH STROKE
01BC LATIN CAPITAL LETTER TONE FIVE
01BD LATIN SMALL LETTER TONE FIVE
01BE LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE
01BF LATIN LETTER WYNN
01C0 LATIN LETTER DENTAL CLICK
01C1 LATIN LETTER LATERAL CLICK
01C2 LATIN LETTER ALVEOLAR CLICK
01C3 LATIN LETTER RETROFLEX CLICK
01C4 LATIN CAPITAL LETTER DZ WITH CARON
01C5 LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
01C6 LATIN SMALL LETTER DZ WITH CARON
01C7 LATIN CAPITAL LETTER LJ
01C8 LATIN CAPITAL LETTER L WITH SMALL LETTER J
01C9 LATIN SMALL LETTER LJ
01CA LATIN CAPITAL LETTER NJ
01CB LATIN CAPITAL LETTER N WITH SMALL LETTER J
01CC LATIN SMALL LETTER NJ
01CD LATIN CAPITAL LETTER A WITH CARON
01CE LATIN SMALL LETTER A WITH CARON
01CF LATIN CAPITAL LETTER I WITH CARON
01D0 LATIN SMALL LETTER I WITH CARON
01D1 LATIN CAPITAL LETTER O WITH CARON
01D2 LATIN SMALL LETTER O WITH CARON
01D3 LATIN CAPITAL LETTER U WITH CARON
01D4 LATIN SMALL LETTER U WITH CARON
01D5 LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
01D6 LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
01D7 LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
01D8 LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
01D9 LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
01DA LATIN SMALL LETTER U WITH DIAERESIS AND CARON
01DB LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
01DC LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
01DD LATIN SMALL LETTER TURNED E
01DE LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
01DF LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
01E0 LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
01E1 LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
01E2 LATIN CAPITAL LETTER AE WITH MACRON
01E3 LATIN SMALL LETTER AE WITH MACRON
01E4 LATIN CAPITAL LETTER G WITH STROKE
01E5 LATIN SMALL LETTER G WITH STROKE
01E6 LATIN CAPITAL LETTER G WITH CARON
01E7 LATIN SMALL LETTER G WITH CARON
01E8 LATIN CAPITAL LETTER K WITH CARON
01E9 LATIN SMALL LETTER K WITH CARON
01EA LATIN CAPITAL LETTER O WITH OGONEK
01EB LATIN SMALL LETTER O WITH OGONEK
01EC LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
01ED LATIN SMALL LETTER O WITH OGONEK AND MACRON
01EE LATIN CAPITAL LETTER EZH WITH CARON
01EF LATIN SMALL LETTER EZH WITH CARON
01F0 LATIN SMALL LETTER J WITH CARON
01F1 LATIN CAPITAL LETTER DZ
01F2 LATIN CAPITAL LETTER D WITH SMALL LETTER Z
01F3 LATIN SMALL LETTER DZ
01F4 LATIN CAPITAL LETTER G WITH ACUTE
01F5 LATIN SMALL LETTER G WITH ACUTE
01F6 LATIN CAPITAL LETTER HWAIR
01F7 LATIN CAPITAL LETTER WYNN
01F8 LATIN CAPITAL LETTER N WITH GRAVE
01F9 LATIN SMALL LETTER N WITH GRAVE
01FA LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
01FB LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
01FC LATIN CAPITAL LETTER AE WITH ACUTE
01FD LATIN SMALL LETTER AE WITH ACUTE
01FE LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
01FF LATIN SMALL LETTER O WITH STROKE AND ACUTE
0200 LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
0201 LATIN SMALL LETTER A WITH DOUBLE GRAVE
0202 LATIN CAPITAL LETTER A WITH INVERTED BREVE
0203 LATIN SMALL LETTER A WITH INVERTED BREVE
0204 LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
0205 LATIN SMALL LETTER E WITH DOUBLE GRAVE
0206 LATIN CAPITAL LETTER E WITH INVERTED BREVE
0207 LATIN SMALL LETTER E WITH INVERTED BREVE
0208 LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
0209 LATIN SMALL LETTER I WITH DOUBLE GRAVE
020A LATIN CAPITAL LETTER I WITH INVERTED BREVE
020B LATIN SMALL LETTER I WITH INVERTED BREVE
020C LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
020D LATIN SMALL LETTER O WITH DOUBLE GRAVE
020E LATIN CAPITAL LETTER O WITH INVERTED BREVE
020F LATIN SMALL LETTER O WITH INVERTED BREVE
0210 LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
0211 LATIN SMALL LETTER R WITH DOUBLE GRAVE
0212 LATIN CAPITAL LETTER R WITH INVERTED BREVE
0213 LATIN SMALL LETTER R WITH INVERTED BREVE
0214 LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
0215 LATIN SMALL LETTER U WITH DOUBLE GRAVE
0216 LATIN CAPITAL LETTER U WITH INVERTED BREVE
0217 LATIN SMALL LETTER U WITH INVERTED BREVE
0218 LATIN CAPITAL LETTER S WITH COMMA BELOW
0219 LATIN SMALL LETTER S WITH COMMA BELOW
021A LATIN CAPITAL LETTER T WITH COMMA BELOW
021B LATIN SMALL LETTER T WITH COMMA BELOW
021C LATIN CAPITAL LETTER YOGH
021D LATIN SMALL LETTER YOGH
021E LATIN CAPITAL LETTER H WITH CARON
021F LATIN SMALL LETTER H WITH CARON
0220 LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
0221 LATIN SMALL LETTER D WITH CURL
0222 LATIN CAPITAL LETTER OU
0223 LATIN SMALL LETTER OU
0224 LATIN CAPITAL LETTER Z WITH HOOK
0225 LATIN SMALL LETTER Z WITH HOOK
0226 LATIN CAPITAL LETTER A WITH DOT ABOVE
0227 LATIN SMALL LETTER A WITH DOT ABOVE
0228 LATIN CAPITAL LETTER E WITH CEDILLA
0229 LATIN SMALL LETTER E WITH CEDILLA
022A LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
022B LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
022C LATIN CAPITAL LETTER O WITH TILDE AND MACRON
022D LATIN SMALL LETTER O WITH TILDE AND MACRON
022E LATIN CAPITAL LETTER O WITH DOT ABOVE
022F LATIN SMALL LETTER O WITH DOT ABOVE
0230 LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
0231 LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
0232 LATIN CAPITAL LETTER Y WITH MACRON
0233 LATIN SMALL LETTER Y WITH MACRON
0234 LATIN SMALL LETTER L WITH CURL
0235 LATIN SMALL LETTER N WITH CURL
0236 LATIN SMALL LETTER T WITH CURL
0237 LATIN SMALL LETTER DOTLESS J
0238 LATIN SMALL LETTER DB DIGRAPH
0239 LATIN SMALL LETTER QP DIGRAPH
023A LATIN CAPITAL LETTER A WITH STROKE
023B LATIN CAPITAL LETTER C WITH STROKE
023C LATIN SMALL LETTER C WITH STROKE
023D LATIN CAPITAL LETTER L WITH BAR
023E LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
023F LATIN SMALL LETTER S WITH SWASH TAIL
0240 LATIN SMALL LETTER Z WITH SWASH TAIL
0241 LATIN CAPITAL LETTER GLOTTAL STOP
0242 LATIN SMALL LETTER GLOTTAL STOP
0243 LATIN CAPITAL LETTER B WITH STROKE
0244 LATIN CAPITAL LETTER U BAR
0245 LATIN CAPITAL LETTER TURNED V
0246 LATIN CAPITAL LETTER E WITH STROKE
0247 LATIN SMALL LETTER E WITH STROKE
0248 LATIN CAPITAL LETTER J WITH STROKE
0249 LATIN SMALL LETTER J WITH STROKE
024A LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
024B LATIN SMALL LETTER Q WITH HOOK TAIL
024C LATIN CAPITAL LETTER R WITH STROKE
024D LATIN SMALL LETTER R WITH STROKE
024E LATIN CAPITAL LETTER Y WITH STROKE
024F LATIN SMALL LETTER Y WITH STROKE
0250 LATIN SMALL LETTER TURNED A
0251 LATIN SMALL LETTER ALPHA
0252 LATIN SMALL LETTER TURNED ALPHA
0253 LATIN SMALL LETTER B WITH HOOK
0254 LATIN SMALL LETTER OPEN O
0255 LATIN SMALL LETTER C WITH CURL
0256 LATIN SMALL LETTER D WITH TAIL
0257 LATIN SMALL LETTER D WITH HOOK
0258 LATIN SMALL LETTER REVERSED E
0259 LATIN SMALL LETTER SCHWA
025A LATIN SMALL LETTER SCHWA WITH HOOK
025B LATIN SMALL LETTER OPEN E
025C LATIN SMALL LETTER REVERSED OPEN E
025D LATIN SMALL LETTER REVERSED OPEN E WITH HOOK
025E LATIN SMALL LETTER CLOSED REVERSED OPEN E
025F LATIN SMALL LETTER DOTLESS J WITH STROKE
0260 LATIN SMALL LETTER G WITH HOOK
0261 LATIN SMALL LETTER SCRIPT G
0262 LATIN LETTER SMALL CAPITAL G
0263 LATIN SMALL LETTER GAMMA
0264 LATIN SMALL LETTER RAMS HORN
0265 LATIN SMALL LETTER TURNED H
0266 LATIN SMALL LETTER H WITH HOOK
0267 LATIN SMALL LETTER HENG WITH HOOK
0268 LATIN SMALL LETTER I WITH STROKE
0269 LATIN SMALL LETTER IOTA
026A LATIN LETTER SMALL CAPITAL I
026B LATIN SMALL LETTER L WITH MIDDLE TILDE
026C LATIN SMALL LETTER L WITH BELT
026D LATIN SMALL LETTER L WITH RETROFLEX HOOK
026E LATIN SMALL LETTER LEZH
026F LATIN SMALL LETTER TURNED M
0270 LATIN SMALL LETTER TURNED M WITH LONG LEG
0271 LATIN SMALL LETTER M WITH HOOK
0272 LATIN SMALL LETTER N WITH LEFT HOOK
0273 LATIN SMALL LETTER N WITH RETROFLEX HOOK
0274 LATIN LETTER SMALL CAPITAL N
0275 LATIN SMALL LETTER BARRED O
0276 LATIN LETTER SMALL CAPITAL OE
0277 LATIN SMALL LETTER CLOSED OMEGA
0278 LATIN SMALL LETTER PHI
0279 LATIN SMALL LETTER TURNED R
027A LATIN SMALL LETTER TURNED R WITH LONG LEG
027B LATIN SMALL LETTER TURNED R WITH HOOK
027C LATIN SMALL LETTER R WITH LONG LEG
027D LATIN SMALL LETTER R WITH TAIL
027E LATIN SMALL LETTER R WITH FISHHOOK
027F LATIN SMALL LETTER REVERSED R WITH FISHHOOK
0280 LATIN LETTER SMALL CAPITAL R
0281 LATIN LETTER SMALL CAPITAL INVERTED R
0282 LATIN SMALL LETTER S WITH HOOK
0283 LATIN SMALL LETTER ESH
0284 LATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOK
0285 LATIN SMALL LETTER SQUAT REVERSED ESH
0286 LATIN SMALL LETTER ESH WITH CURL
0287 LATIN SMALL LETTER TURNED T
0288 LATIN SMALL LETTER T WITH RETROFLEX HOOK
0289 LATIN SMALL LETTER U BAR
028A LATIN SMALL LETTER UPSILON
028B LATIN SMALL LETTER V WITH HOOK
028C LATIN SMALL LETTER TURNED V
028D LATIN SMALL LETTER TURNED W
028E LATIN SMALL LETTER TURNED Y
028F LATIN LETTER SMALL CAPITAL Y
0290 LATIN SMALL LETTER Z WITH RETROFLEX HOOK
0291 LATIN SMALL LETTER Z WITH CURL
0292 LATIN SMALL LETTER EZH
0293 LATIN SMALL LETTER EZH WITH CURL
0294 LATIN LETTER GLOTTAL STOP
0295 LATIN LETTER PHARYNGEAL VOICED FRICATIVE
0296 LATIN LETTER INVERTED GLOTTAL STOP
0297 LATIN LETTER STRETCHED C
0298 LATIN LETTER BILABIAL CLICK
0299 LATIN LETTER SMALL CAPITAL B
029A LATIN SMALL LETTER CLOSED OPEN E
029B LATIN LETTER SMALL CAPITAL G WITH HOOK
029C LATIN LETTER SMALL CAPITAL H
029D LATIN SMALL LETTER J WITH CROSSED-TAIL
029E LATIN SMALL LETTER TURNED K
029F LATIN LETTER SMALL CAPITAL L
02A0 LATIN SMALL LETTER Q WITH HOOK
02A1 LATIN LETTER GLOTTAL STOP WITH STROKE
02A2 LATIN LETTER REVERSED GLOTTAL STOP WITH STROKE
02A3 LATIN SMALL LETTER DZ DIGRAPH
02A4 LATIN SMALL LETTER DEZH DIGRAPH
02A5 LATIN SMALL LETTER DZ DIGRAPH WITH CURL
02A6 LATIN SMALL LETTER TS DIGRAPH
02A7 LATIN SMALL LETTER TESH DIGRAPH
02A8 LATIN SMALL LETTER TC DIGRAPH WITH CURL
02A9 LATIN SMALL LETTER FENG DIGRAPH
02AA LATIN SMALL LETTER LS DIGRAPH
02AB LATIN SMALL LETTER LZ DIGRAPH
02AC LATIN LETTER BILABIAL PERCUSSIVE
02AD LATIN LETTER BIDENTAL PERCUSSIVE
02AE LATIN SMALL LETTER TURNED H WITH FISHHOOK
02AF LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
02B0 MODIFIER LETTER SMALL H
02B1 MODIFIER LETTER SMALL H WITH HOOK
02B2 MODIFIER LETTER SMALL J
02B3 MODIFIER LETTER SMALL R
02B4 MODIFIER LETTER SMALL TURNED R
02B5 MODIFIER LETTER SMALL TURNED R WITH HOOK
02B6 MODIFIER LETTER SMALL CAPITAL INVERTED R
02B7 MODIFIER LETTER SMALL W
02B8 MODIFIER LETTER SMALL Y
02B9 MODIFIER LETTER PRIME
02BA MODIFIER LETTER DOUBLE PRIME
02BB MODIFIER LETTER TURNED COMMA
02BC MODIFIER LETTER APOSTROPHE
02BD MODIFIER LETTER REVERSED COMMA
02BE MODIFIER LETTER RIGHT HALF RING
02BF MODIFIER LETTER LEFT HALF RING
02C0 MODIFIER LETTER GLOTTAL STOP
02C1 MODIFIER LETTER REVERSED GLOTTAL STOP
02C2 MODIFIER LETTER LEFT ARROWHEAD
02C3 MODIFIER LETTER RIGHT ARROWHEAD
02C4 MODIFIER LETTER UP ARROWHEAD
02C5 MODIFIER LETTER DOWN ARROWHEAD
02C6 MODIFIER LETTER CIRCUMFLEX ACCENT
02C7 CARON
02C8 MODIFIER LETTER VERTICAL LINE
02C9 MODIFIER LETTER MACRON
02CA MODIFIER LETTER ACUTE ACCENT
02CB MODIFIER LETTER GRAVE ACCENT
02CC MODIFIER LETTER LOW VERTICAL LINE
02CD MODIFIER LETTER LOW MACRON
02CE MODIFIER LETTER LOW GRAVE ACCENT
02CF MODIFIER LETTER LOW ACUTE ACCENT
02D0 MODIFIER LETTER TRIANGULAR COLON
02D1 MODIFIER LETTER HALF TRIANGULAR COLON
02D2 MODIFIER LETTER CENTRED RIGHT HALF RING
02D3 MODIFIER LETTER CENTRED LEFT HALF RING
02D4 MODIFIER LETTER UP TACK
02D5 MODIFIER LETTER DOWN TACK
02D6 MODIFIER LETTER PLUS SIGN
02D7 MODIFIER LETTER MINUS SIGN
02D8 BREVE
02D9 DOT ABOVE
02DA RING ABOVE
02DB OGONEK
02DC SMALL TILDE
02DD DOUBLE ACUTE ACCENT
02DE MODIFIER LETTER RHOTIC HOOK
02DF MODIFIER LETTER CROSS ACCENT
02E0 MODIFIER LETTER SMALL GAMMA
02E1 MODIFIER LETTER SMALL L
02E2 MODIFIER LETTER SMALL S
02E3 MODIFIER LETTER SMALL X
02E4 MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
02E5 MODIFIER LETTER EXTRA-HIGH TONE BAR
02E6 MODIFIER LETTER HIGH TONE BAR
02E7 MODIFIER LETTER MID TONE BAR
02E8 MODIFIER LETTER LOW TONE BAR
02E9 MODIFIER LETTER EXTRA-LOW TONE BAR
02EA MODIFIER LETTER YIN DEPARTING TONE MARK
02EB MODIFIER LETTER YANG DEPARTING TONE MARK
02EC MODIFIER LETTER VOICING
02ED MODIFIER LETTER UNASPIRATED
02EE MODIFIER LETTER DOUBLE APOSTROPHE
02EF MODIFIER LETTER LOW DOWN ARROWHEAD
02F0 MODIFIER LETTER LOW UP ARROWHEAD
02F1 MODIFIER LETTER LOW LEFT ARROWHEAD
02F2 MODIFIER LETTER LOW RIGHT ARROWHEAD
02F3 MODIFIER LETTER LOW RING
02F4 MODIFIER LETTER MIDDLE GRAVE ACCENT
02F5 MODIFIER LETTER MIDDLE DOUBLE GRAVE ACCENT
02F6 MODIFIER LETTER MIDDLE DOUBLE ACUTE ACCENT
02F7 MODIFIER LETTER LOW TILDE
02F8 MODIFIER LETTER RAISED COLON
02F9 MODIFIER LETTER BEGIN HIGH TONE
02FA MODIFIER LETTER END HIGH TONE
02FB MODIFIER LETTER BEGIN LOW TONE
02FC MODIFIER LETTER END LOW TONE
02FD MODIFIER LETTER SHELF
02FE MODIFIER LETTER OPEN SHELF
02FF MODIFIER LETTER LOW LEFT ARROW
0300 COMBINING GRAVE ACCENT
0301 COMBINING ACUTE ACCENT
0302 COMBINING CIRCUMFLEX ACCENT
0303 COMBINING TILDE
0304 COMBINING MACRON
0305 COMBINING OVERLINE
0306 COMBINING BREVE
0307 COMBINING DOT ABOVE
0308 COMBINING DIAERESIS
0309 COMBINING HOOK ABOVE
030A COMBINING RING ABOVE
030B COMBINING DOUBLE ACUTE ACCENT
030C COMBINING CARON
030D COMBINING VERTICAL LINE ABOVE
030E COMBINING DOUBLE VERTICAL LINE ABOVE
030F COMBINING DOUBLE GRAVE ACCENT
0310 COMBINING CANDRABINDU
0311 COMBINING INVERTED BREVE
0312 COMBINING TURNED COMMA ABOVE
0313 COMBINING COMMA ABOVE
0314 COMBINING REVERSED COMMA ABOVE
0315 COMBINING COMMA ABOVE RIGHT
0316 COMBINING GRAVE ACCENT BELOW
0317 COMBINING ACUTE ACCENT BELOW
0318 COMBINING LEFT TACK BELOW
0319 COMBINING RIGHT TACK BELOW
031A COMBINING LEFT ANGLE ABOVE
031B COMBINING HORN
031C COMBINING LEFT HALF RING BELOW
031D COMBINING UP TACK BELOW
031E COMBINING DOWN TACK BELOW
031F COMBINING PLUS SIGN BELOW
0320 COMBINING MINUS SIGN BELOW
0321 COMBINING PALATALIZED HOOK BELOW
0322 COMBINING RETROFLEX HOOK BELOW
0323 COMBINING DOT BELOW
0324 COMBINING DIAERESIS BELOW
0325 COMBINING RING BELOW
0326 COMBINING COMMA BELOW
0327 COMBINING CEDILLA
0328 COMBINING OGONEK
0329 COMBINING VERTICAL LINE BELOW
032A COMBINING BRIDGE BELOW
032B COMBINING INVERTED DOUBLE ARCH BELOW
032C COMBINING CARON BELOW
032D COMBINING CIRCUMFLEX ACCENT BELOW
032E COMBINING BREVE BELOW
032F COMBINING INVERTED BREVE BELOW
0330 COMBINING TILDE BELOW
0331 COMBINING MACRON BELOW
0332 COMBINING LOW LINE
0333 COMBINING DOUBLE LOW LINE
0334 COMBINING TILDE OVERLAY
0335 COMBINING SHORT STROKE OVERLAY
0336 COMBINING LONG STROKE OVERLAY
0337 COMBINING SHORT SOLIDUS OVERLAY
0338 COMBINING LONG SOLIDUS OVERLAY
0339 COMBINING RIGHT HALF RING BELOW
033A COMBINING INVERTED BRIDGE BELOW
033B COMBINING SQUARE BELOW
033C COMBINING SEAGULL BELOW
033D COMBINING X ABOVE
033E COMBINING VERTICAL TILDE
033F COMBINING DOUBLE OVERLINE
0340 COMBINING GRAVE TONE MARK
0341 COMBINING ACUTE TONE MARK
0342 COMBINING GREEK PERISPOMENI
0343 COMBINING GREEK KORONIS
0344 COMBINING GREEK DIALYTIKA TONOS
0345 COMBINING GREEK YPOGEGRAMMENI
0346 COMBINING BRIDGE ABOVE
0347 COMBINING EQUALS SIGN BELOW
0348 COMBINING DOUBLE VERTICAL LINE BELOW
0349 COMBINING LEFT ANGLE BELOW
034A COMBINING NOT TILDE ABOVE
034B COMBINING HOMOTHETIC ABOVE
034C COMBINING ALMOST EQUAL TO ABOVE
034D COMBINING LEFT RIGHT ARROW BELOW
034E COMBINING UPWARDS ARROW BELOW
034F COMBINING GRAPHEME JOINER, CGJ
0350 COMBINING RIGHT ARROWHEAD ABOVE
0351 COMBINING LEFT HALF RING ABOVE
0352 COMBINING FERMATA
0353 COMBINING X BELOW
0354 COMBINING LEFT ARROWHEAD BELOW
0355 COMBINING RIGHT ARROWHEAD BELOW
0356 COMBINING RIGHT ARROWHEAD AND UP ARROWHEAD BELOW
0357 COMBINING RIGHT HALF RING ABOVE
0358 COMBINING DOT ABOVE RIGHT
0359 COMBINING ASTERISK BELOW
035A COMBINING DOUBLE RING BELOW
035B COMBINING ZIGZAG ABOVE
035C COMBINING DOUBLE BREVE BELOW
035D COMBINING DOUBLE BREVE
035E COMBINING DOUBLE MACRON
035F COMBINING DOUBLE MACRON BELOW
0360 COMBINING DOUBLE TILDE
0361 COMBINING DOUBLE INVERTED BREVE
0362 COMBINING DOUBLE RIGHTWARDS ARROW BELOW
0363 COMBINING LATIN SMALL LETTER A
0364 COMBINING LATIN SMALL LETTER E
0365 COMBINING LATIN SMALL LETTER I
0366 COMBINING LATIN SMALL LETTER O
0367 COMBINING LATIN SMALL LETTER U
0368 COMBINING LATIN SMALL LETTER C
0369 COMBINING LATIN SMALL LETTER D
036A COMBINING LATIN SMALL LETTER H
036B COMBINING LATIN SMALL LETTER M
036C COMBINING LATIN SMALL LETTER R
036D COMBINING LATIN SMALL LETTER T
036E COMBINING LATIN SMALL LETTER V
036F COMBINING LATIN SMALL LETTER X
0370 GREEK CAPITAL LETTER HETA
0371 GREEK SMALL LETTER HETA
0372 GREEK CAPITAL LETTER ARCHAIC SAMPI
0373 GREEK SMALL LETTER ARCHAIC SAMPI
0374 GREEK NUMERAL SIGN
0375 GREEK LOWER NUMERAL SIGN
0376 GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
0377 GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
037A GREEK YPOGEGRAMMENI
037B GREEK SMALL REVERSED LUNATE SIGMA SYMBOL
037C GREEK SMALL DOTTED LUNATE SIGMA SYMBOL
037D GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
037E GREEK QUESTION MARK
037F GREEK CAPITAL LETTER YOT
0384 GREEK TONOS
0385 GREEK DIALYTIKA TONOS
0386 GREEK CAPITAL LETTER ALPHA WITH TONOS
0387 GREEK ANO TELEIA
0388 GREEK CAPITAL LETTER EPSILON WITH TONOS
0389 GREEK CAPITAL LETTER ETA WITH TONOS
038A GREEK CAPITAL LETTER IOTA WITH TONOS
038C GREEK CAPITAL LETTER OMICRON WITH TONOS
038E GREEK CAPITAL LETTER UPSILON WITH TONOS
038F GREEK CAPITAL LETTER OMEGA WITH TONOS
0390 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
0391 GREEK CAPITAL LETTER ALPHA
0392 GREEK CAPITAL LETTER BETA
0393 GREEK CAPITAL LETTER GAMMA
0394 GREEK CAPITAL LETTER DELTA
0395 GREEK CAPITAL LETTER EPSILON
0396 GREEK CAPITAL LETTER ZETA
0397 GREEK CAPITAL LETTER ETA
0398 GREEK CAPITAL LETTER THETA
0399 GREEK CAPITAL LETTER IOTA
039A GREEK CAPITAL LETTER KAPPA
039B GREEK CAPITAL LETTER LAMDA
039C GREEK CAPITAL LETTER MU
039D GREEK CAPITAL LETTER NU
039E GREEK CAPITAL LETTER XI
039F GREEK CAPITAL LETTER OMICRON
03A0 GREEK CAPITAL LETTER PI
03A1 GREEK CAPITAL LETTER RHO
03A3 GREEK CAPITAL LETTER SIGMA
03A4 GREEK CAPITAL LETTER TAU
03A5 GREEK CAPITAL LETTER UPSILON
03A6 GREEK CAPITAL LETTER PHI
03A7 GREEK CAPITAL LETTER CHI
03A8 GREEK CAPITAL LETTER PSI
03A9 GREEK CAPITAL LETTER OMEGA
03AA GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
03AB GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
03AC GREEK SMALL LETTER ALPHA WITH TONOS
03AD GREEK SMALL LETTER EPSILON WITH TONOS
03AE GREEK SMALL LETTER ETA WITH TONOS
03AF GREEK SMALL LETTER IOTA WITH TONOS
03B0 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
03B1 GREEK SMALL LETTER ALPHA
03B2 GREEK SMALL LETTER BETA
03B3 GREEK SMALL LETTER GAMMA
03B4 GREEK SMALL LETTER DELTA
03B5 GREEK SMALL LETTER EPSILON
03B6 GREEK SMALL LETTER ZETA
03B7 GREEK SMALL LETTER ETA
03B8 GREEK SMALL LETTER THETA
03B9 GREEK SMALL LETTER IOTA
03BA GREEK SMALL LETTER KAPPA
03BB GREEK SMALL LETTER LAMDA
03BC GREEK SMALL LETTER MU
03BD GREEK SMALL LETTER NU
03BE GREEK SMALL LETTER XI
03BF GREEK SMALL LETTER OMICRON
03C0 GREEK SMALL LETTER PI
03C1 GREEK SMALL LETTER RHO
03C2 GREEK SMALL LETTER FINAL SIGMA
03C3 GREEK SMALL LETTER SIGMA
03C4 GREEK SMALL LETTER TAU
03C5 GREEK SMALL LETTER UPSILON
03C6 GREEK SMALL LETTER PHI
03C7 GREEK SMALL LETTER CHI
03C8 GREEK SMALL LETTER PSI
03C9 GREEK SMALL LETTER OMEGA
03CA GREEK SMALL LETTER IOTA WITH DIALYTIKA
03CB GREEK SMALL LETTER UPSILON WITH DIALYTIKA
03CC GREEK SMALL LETTER OMICRON WITH TONOS
03CD GREEK SMALL LETTER UPSILON WITH TONOS
03CE GREEK SMALL LETTER OMEGA WITH TONOS
03CF GREEK CAPITAL KAI SYMBOL
03D0 GREEK BETA SYMBOL
03D1 GREEK THETA SYMBOL
03D2 GREEK UPSILON WITH HOOK SYMBOL
03D3 GREEK UPSILON WITH ACUTE AND HOOK SYMBOL
03D4 GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL
03D5 GREEK PHI SYMBOL
03D6 GREEK PI SYMBOL
03D7 GREEK KAI SYMBOL
03D8 GREEK LETTER ARCHAIC KOPPA
03D9 GREEK SMALL LETTER ARCHAIC KOPPA
03DA GREEK LETTER STIGMA
03DB GREEK SMALL LETTER STIGMA
03DC GREEK LETTER DIGAMMA
03DD GREEK SMALL LETTER DIGAMMA
03DE GREEK LETTER KOPPA
03DF GREEK SMALL LETTER KOPPA
03E0 GREEK LETTER SAMPI
03E1 GREEK SMALL LETTER SAMPI
03E2 COPTIC CAPITAL LETTER SHEI
03E3 COPTIC SMALL LETTER SHEI
03E4 COPTIC CAPITAL LETTER FEI
03E5 COPTIC SMALL LETTER FEI
03E6 COPTIC CAPITAL LETTER KHEI
03E7 COPTIC SMALL LETTER KHEI
03E8 COPTIC CAPITAL LETTER HORI
03E9 COPTIC SMALL LETTER HORI
03EA COPTIC CAPITAL LETTER GANGIA
03EB COPTIC SMALL LETTER GANGIA
03EC COPTIC CAPITAL LETTER SHIMA
03ED COPTIC SMALL LETTER SHIMA
03EE COPTIC CAPITAL LETTER DEI
03EF COPTIC SMALL LETTER DEI
03F0 GREEK KAPPA SYMBOL
03F1 GREEK RHO SYMBOL
03F2 GREEK LUNATE SIGMA SYMBOL
03F3 GREEK LETTER YOT
03F4 GREEK CAPITAL THETA SYMBOL
03F5 GREEK LUNATE EPSILON SYMBOL
03F6 GREEK REVERSED LUNATE EPSILON SYMBOL
03F7 GREEK CAPITAL LETTER SHO
03F8 GREEK SMALL LETTER SHO
03F9 GREEK CAPITAL LUNATE SIGMA SYMBOL
03FA GREEK CAPITAL LETTER SAN
03FB GREEK SMALL LETTER SAN
03FC GREEK RHO WITH STROKE SYMBOL
03FD GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL
03FE GREEK CAPITAL DOTTED LUNATE SIGMA SYMBOL
03FF GREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOL
0400 CYRILLIC CAPITAL LETTER IE WITH GRAVE
0401 CYRILLIC CAPITAL LETTER IO
0402 CYRILLIC CAPITAL LETTER DJE
0403 CYRILLIC CAPITAL LETTER GJE
0404 CYRILLIC CAPITAL LETTER UKRAINIAN IE
0405 CYRILLIC CAPITAL LETTER DZE
0406 CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
0407 CYRILLIC CAPITAL LETTER YI
0408 CYRILLIC CAPITAL LETTER JE
0409 CYRILLIC CAPITAL LETTER LJE
040A CYRILLIC CAPITAL LETTER NJE
040B CYRILLIC CAPITAL LETTER TSHE
040C CYRILLIC CAPITAL LETTER KJE
040D CYRILLIC CAPITAL LETTER I WITH GRAVE
040E CYRILLIC CAPITAL LETTER SHORT U
040F CYRILLIC CAPITAL LETTER DZHE
0410 CYRILLIC CAPITAL LETTER A
0411 CYRILLIC CAPITAL LETTER BE
0412 CYRILLIC CAPITAL LETTER VE
0413 CYRILLIC CAPITAL LETTER GHE
0414 CYRILLIC CAPITAL LETTER DE
0415 CYRILLIC CAPITAL LETTER IE
0416 CYRILLIC CAPITAL LETTER ZHE
0417 CYRILLIC CAPITAL LETTER ZE
0418 CYRILLIC CAPITAL LETTER I
0419 CYRILLIC CAPITAL LETTER SHORT I
041A CYRILLIC CAPITAL LETTER KA
041B CYRILLIC CAPITAL LETTER EL
041C CYRILLIC CAPITAL LETTER EM
041D CYRILLIC CAPITAL LETTER EN
041E CYRILLIC CAPITAL LETTER O
041F CYRILLIC CAPITAL LETTER PE
0420 CYRILLIC CAPITAL LETTER ER
0421 CYRILLIC CAPITAL LETTER ES
0422 CYRILLIC CAPITAL LETTER TE
0423 CYRILLIC CAPITAL LETTER U
0424 CYRILLIC CAPITAL LETTER EF
0425 CYRILLIC CAPITAL LETTER HA
0426 CYRILLIC CAPITAL LETTER TSE
0427 CYRILLIC CAPITAL LETTER CHE
0428 CYRILLIC CAPITAL LETTER SHA
0429 CYRILLIC CAPITAL LETTER SHCHA
042A CYRILLIC CAPITAL LETTER HARD SIGN
042B CYRILLIC CAPITAL LETTER YERU
042C CYRILLIC CAPITAL LETTER SOFT SIGN
042D CYRILLIC CAPITAL LETTER E
042E CYRILLIC CAPITAL LETTER YU
042F CYRILLIC CAPITAL LETTER YA
0430 CYRILLIC SMALL LETTER A
0431 CYRILLIC SMALL LETTER BE
0432 CYRILLIC SMALL LETTER VE
0433 CYRILLIC SMALL LETTER GHE
0434 CYRILLIC SMALL LETTER DE
0435 CYRILLIC SMALL LETTER IE
0436 CYRILLIC SMALL LETTER ZHE
0437 CYRILLIC SMALL LETTER ZE
0438 CYRILLIC SMALL LETTER I
0439 CYRILLIC SMALL LETTER SHORT I
043A CYRILLIC SMALL LETTER KA
043B CYRILLIC SMALL LETTER EL
043C CYRILLIC SMALL LETTER EM
043D CYRILLIC SMALL LETTER EN
043E CYRILLIC SMALL LETTER O
043F CYRILLIC SMALL LETTER PE
0440 CYRILLIC SMALL LETTER ER
0441 CYRILLIC SMALL LETTER ES
0442 CYRILLIC SMALL LETTER TE
0443 CYRILLIC SMALL LETTER U
0444 CYRILLIC SMALL LETTER EF
0445 CYRILLIC SMALL LETTER HA
0446 CYRILLIC SMALL LETTER TSE
0447 CYRILLIC SMALL LETTER CHE
0448 CYRILLIC SMALL LETTER SHA
0449 CYRILLIC SMALL LETTER SHCHA
044A CYRILLIC SMALL LETTER HARD SIGN
044B CYRILLIC SMALL LETTER YERU
044C CYRILLIC SMALL LETTER SOFT SIGN
044D CYRILLIC SMALL LETTER E
044E CYRILLIC SMALL LETTER YU
044F CYRILLIC SMALL LETTER YA
0450 CYRILLIC SMALL LETTER IE WITH GRAVE
0451 CYRILLIC SMALL LETTER IO
0452 CYRILLIC SMALL LETTER DJE
0453 CYRILLIC SMALL LETTER GJE
0454 CYRILLIC SMALL LETTER UKRAINIAN IE
0455 CYRILLIC SMALL LETTER DZE
0456 CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
0457 CYRILLIC SMALL LETTER YI
0458 CYRILLIC SMALL LETTER JE
0459 CYRILLIC SMALL LETTER LJE
045A CYRILLIC SMALL LETTER NJE
045B CYRILLIC SMALL LETTER TSHE
045C CYRILLIC SMALL LETTER KJE
045D CYRILLIC SMALL LETTER I WITH GRAVE
045E CYRILLIC SMALL LETTER SHORT U
045F CYRILLIC SMALL LETTER DZHE
0460 CYRILLIC CAPITAL LETTER OMEGA
0461 CYRILLIC SMALL LETTER OMEGA
0462 CYRILLIC CAPITAL LETTER YAT
0463 CYRILLIC SMALL LETTER YAT
0464 CYRILLIC CAPITAL LETTER IOTIFIED E
0465 CYRILLIC SMALL LETTER IOTIFIED E
0466 CYRILLIC CAPITAL LETTER LITTLE YUS
0467 CYRILLIC SMALL LETTER LITTLE YUS
0468 CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
0469 CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
046A CYRILLIC CAPITAL LETTER BIG YUS
046B CYRILLIC SMALL LETTER BIG YUS
046C CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
046D CYRILLIC SMALL LETTER IOTIFIED BIG YUS
046E CYRILLIC CAPITAL LETTER KSI
046F CYRILLIC SMALL LETTER KSI
0470 CYRILLIC CAPITAL LETTER PSI
0471 CYRILLIC SMALL LETTER PSI
0472 CYRILLIC CAPITAL LETTER FITA
0473 CYRILLIC SMALL LETTER FITA
0474 CYRILLIC CAPITAL LETTER IZHITSA
0475 CYRILLIC SMALL LETTER IZHITSA
0476 CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
0477 CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
0478 CYRILLIC CAPITAL LETTER UK
0479 CYRILLIC SMALL LETTER UK
047A CYRILLIC CAPITAL LETTER ROUND OMEGA
047B CYRILLIC SMALL LETTER ROUND OMEGA
047C CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
047D CYRILLIC SMALL LETTER OMEGA WITH TITLO
047E CYRILLIC CAPITAL LETTER OT
047F CYRILLIC SMALL LETTER OT
0480 CYRILLIC CAPITAL LETTER KOPPA
0481 CYRILLIC SMALL LETTER KOPPA
0482 CYRILLIC THOUSANDS SIGN
0483 COMBINING CYRILLIC TITLO
0484 COMBINING CYRILLIC PALATALIZATION
0485 COMBINING CYRILLIC DASIA PNEUMATA
0486 COMBINING CYRILLIC PSILI PNEUMATA
0487 COMBINING CYRILLIC POKRYTIE
0488 COMBINING CYRILLIC HUNDRED THOUSANDS SIGN
0489 COMBINING CYRILLIC MILLIONS SIGN
048A CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
048B CYRILLIC SMALL LETTER SHORT I WITH TAIL
048C CYRILLIC CAPITAL LETTER SEMISOFT SIGN
048D CYRILLIC SMALL LETTER SEMISOFT SIGN
048E CYRILLIC CAPITAL LETTER ER WITH TICK
048F CYRILLIC SMALL LETTER ER WITH TICK
0490 CYRILLIC CAPITAL LETTER GHE WITH UPTURN
0491 CYRILLIC SMALL LETTER GHE WITH UPTURN
0492 CYRILLIC CAPITAL LETTER GHE WITH STROKE
0493 CYRILLIC SMALL LETTER GHE WITH STROKE
0494 CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
0495 CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
0496 CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
0497 CYRILLIC SMALL LETTER ZHE WITH DESCENDER
0498 CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
0499 CYRILLIC SMALL LETTER ZE WITH DESCENDER
049A CYRILLIC CAPITAL LETTER KA WITH DESCENDER
049B CYRILLIC SMALL LETTER KA WITH DESCENDER
049C CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
049D CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
049E CYRILLIC CAPITAL LETTER KA WITH STROKE
049F CYRILLIC SMALL LETTER KA WITH STROKE
04A0 CYRILLIC CAPITAL LETTER BASHKIR KA
04A1 CYRILLIC SMALL LETTER BASHKIR KA
04A2 CYRILLIC CAPITAL LETTER EN WITH DESCENDER
04A3 CYRILLIC SMALL LETTER EN WITH DESCENDER
04A4 CYRILLIC CAPITAL LIGATURE EN GHE
04A5 CYRILLIC SMALL LIGATURE EN GHE
04A6 CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
04A7 CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
04A8 CYRILLIC CAPITAL LETTER ABKHASIAN HA
04A9 CYRILLIC SMALL LETTER ABKHASIAN HA
04AA CYRILLIC CAPITAL LETTER ES WITH DESCENDER
04AB CYRILLIC SMALL LETTER ES WITH DESCENDER
04AC CYRILLIC CAPITAL LETTER TE WITH DESCENDER
04AD CYRILLIC SMALL LETTER TE WITH DESCENDER
04AE CYRILLIC CAPITAL LETTER STRAIGHT U
04AF CYRILLIC SMALL LETTER STRAIGHT U
04B0 CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
04B1 CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
04B2 CYRILLIC CAPITAL LETTER HA WITH DESCENDER
04B3 CYRILLIC SMALL LETTER HA WITH DESCENDER
04B4 CYRILLIC CAPITAL LIGATURE TE TSE
04B5 CYRILLIC SMALL LIGATURE TE TSE
04B6 CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
04B7 CYRILLIC SMALL LETTER CHE WITH DESCENDER
04B8 CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
04B9 CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
04BA CYRILLIC CAPITAL LETTER SHHA
04BB CYRILLIC SMALL LETTER SHHA
04BC CYRILLIC CAPITAL LETTER ABKHASIAN CHE
04BD CYRILLIC SMALL LETTER ABKHASIAN CHE
04BE CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
04BF CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
04C0 CYRILLIC LETTER PALOCHKA
04C1 CYRILLIC CAPITAL LETTER ZHE WITH BREVE
04C2 CYRILLIC SMALL LETTER ZHE WITH BREVE
04C3 CYRILLIC CAPITAL LETTER KA WITH HOOK
04C4 CYRILLIC SMALL LETTER KA WITH HOOK
04C5 CYRILLIC CAPITAL LETTER EL WITH TAIL
04C6 CYRILLIC SMALL LETTER EL WITH TAIL
04C7 CYRILLIC CAPITAL LETTER EN WITH HOOK
04C8 CYRILLIC SMALL LETTER EN WITH HOOK
04C9 CYRILLIC CAPITAL LETTER EN WITH TAIL
04CA CYRILLIC SMALL LETTER EN WITH TAIL
04CB CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
04CC CYRILLIC SMALL LETTER KHAKASSIAN CHE
04CD CYRILLIC CAPITAL LETTER EM WITH TAIL
04CE CYRILLIC SMALL LETTER EM WITH TAIL
04CF CYRILLIC SMALL LETTER PALOCHKA
04D0 CYRILLIC CAPITAL LETTER A WITH BREVE
04D1 CYRILLIC SMALL LETTER A WITH BREVE
04D2 CYRILLIC CAPITAL LETTER A WITH DIAERESIS
04D3 CYRILLIC SMALL LETTER A WITH DIAERESIS
04D4 CYRILLIC CAPITAL LIGATURE A IE
04D5 CYRILLIC SMALL LIGATURE A IE
04D6 CYRILLIC CAPITAL LETTER IE WITH BREVE
04D7 CYRILLIC SMALL LETTER IE WITH BREVE
04D8 CYRILLIC CAPITAL LETTER SCHWA
04D9 CYRILLIC SMALL LETTER SCHWA
04DA CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
04DB CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
04DC CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
04DD CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
04DE CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
04DF CYRILLIC SMALL LETTER ZE WITH DIAERESIS
04E0 CYRILLIC CAPITAL LETTER ABKHASIAN DZE
04E1 CYRILLIC SMALL LETTER ABKHASIAN DZE
04E2 CYRILLIC CAPITAL LETTER I WITH MACRON
04E3 CYRILLIC SMALL LETTER I WITH MACRON
04E4 CYRILLIC CAPITAL LETTER I WITH DIAERESIS
04E5 CYRILLIC SMALL LETTER I WITH DIAERESIS
04E6 CYRILLIC CAPITAL LETTER O WITH DIAERESIS
04E7 CYRILLIC SMALL LETTER O WITH DIAERESIS
04E8 CYRILLIC CAPITAL LETTER BARRED O
04E9 CYRILLIC SMALL LETTER BARRED O
04EA CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
04EB CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
04EC CYRILLIC CAPITAL LETTER E WITH DIAERESIS
04ED CYRILLIC SMALL LETTER E WITH DIAERESIS
04EE CYRILLIC CAPITAL LETTER U WITH MACRON
04EF CYRILLIC SMALL LETTER U WITH MACRON
04F0 CYRILLIC CAPITAL LETTER U WITH DIAERESIS
04F1 CYRILLIC SMALL LETTER U WITH DIAERESIS
04F2 CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
04F3 CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
04F4 CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
04F5 CYRILLIC SMALL LETTER CHE WITH DIAERESIS
04F6 CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
04F7 CYRILLIC SMALL LETTER GHE WITH DESCENDER
04F8 CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
04F9 CYRILLIC SMALL LETTER YERU WITH DIAERESIS
04FA CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
04FB CYRILLIC SMALL LETTER GHE WITH STROKE AND HOOK
04FC CYRILLIC CAPITAL LETTER HA WITH HOOK
04FD CYRILLIC SMALL LETTER HA WITH HOOK
04FE CYRILLIC CAPITAL LETTER HA WITH STROKE
04FF CYRILLIC SMALL LETTER HA WITH STROKE
0500 CYRILLIC CAPITAL LETTER KOMI DE
0501 CYRILLIC SMALL LETTER KOMI DE
0502 CYRILLIC CAPITAL LETTER KOMI DJE
0503 CYRILLIC SMALL LETTER KOMI DJE
0504 CYRILLIC CAPITAL LETTER KOMI ZJE
0505 CYRILLIC SMALL LETTER KOMI ZJE
0506 CYRILLIC CAPITAL LETTER KOMI DZJE
0507 CYRILLIC SMALL LETTER KOMI DZJE
0508 CYRILLIC CAPITAL LETTER KOMI LJE
0509 CYRILLIC SMALL LETTER KOMI LJE
050A CYRILLIC CAPITAL LETTER KOMI NJE
050B CYRILLIC SMALL LETTER KOMI NJE
050C CYRILLIC CAPITAL LETTER KOMI SJE
050D CYRILLIC SMALL LETTER KOMI SJE
050E CYRILLIC CAPITAL LETTER KOMI TJE
050F CYRILLIC SMALL LETTER KOMI TJE
0510 CYRILLIC CAPITAL LETTER REVERSED ZE
0511 CYRILLIC SMALL LETTER REVERSED ZE
0512 CYRILLIC CAPITAL LETTER EL WITH HOOK
0513 CYRILLIC SMALL LETTER EL WITH HOOK
0514 CYRILLIC CAPITAL LETTER LHA
0515 CYRILLIC SMALL LETTER LHA
0516 CYRILLIC CAPITAL LETTER RHA
0517 CYRILLIC SMALL LETTER RHA
0518 CYRILLIC CAPITAL LETTER YAE
0519 CYRILLIC SMALL LETTER YAE
051A CYRILLIC CAPITAL LETTER QA
051B CYRILLIC SMALL LETTER QA
051C CYRILLIC CAPITAL LETTER WE
051D CYRILLIC SMALL LETTER WE
051E CYRILLIC CAPITAL LETTER ALEUT KA
051F CYRILLIC SMALL LETTER ALEUT KA
0520 CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
0521 CYRILLIC SMALL LETTER EL WITH MIDDLE HOOK
0522 CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
0523 CYRILLIC SMALL LETTER EN WITH MIDDLE HOOK
0524 CYRILLIC CAPITAL LETTER PE WITH DESCENDER
0525 CYRILLIC SMALL LETTER PE WITH DESCENDER
0526 CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER
0527 CYRILLIC SMALL LETTER SHHA WITH DESCENDER
0528 CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK
0529 CYRILLIC SMALL LETTER EN WITH LEFT HOOK
052A CYRILLIC CAPITAL LETTER DZZHE
052B CYRILLIC SMALL LETTER DZZHE
052C CYRILLIC CAPITAL LETTER DCHE
052D CYRILLIC SMALL LETTER DCHE
052E CYRILLIC CAPITAL LETTER EL WITH DESCENDER
052F CYRILLIC SMALL LETTER EL WITH DESCENDER
0531 ARMENIAN CAPITAL LETTER AYB
0532 ARMENIAN CAPITAL LETTER BEN
0533 ARMENIAN CAPITAL LETTER GIM
0534 ARMENIAN CAPITAL LETTER DA
0535 ARMENIAN CAPITAL LETTER ECH
0536 ARMENIAN CAPITAL LETTER ZA
0537 ARMENIAN CAPITAL LETTER EH
0538 ARMENIAN CAPITAL LETTER ET
0539 ARMENIAN CAPITAL LETTER TO
053A ARMENIAN CAPITAL LETTER ZHE
053B ARMENIAN CAPITAL LETTER INI
053C ARMENIAN CAPITAL LETTER LIWN
053D ARMENIAN CAPITAL LETTER XEH
053E ARMENIAN CAPITAL LETTER CA
053F ARMENIAN CAPITAL LETTER KEN
0540 ARMENIAN CAPITAL LETTER HO
0541 ARMENIAN CAPITAL LETTER JA
0542 ARMENIAN CAPITAL LETTER GHAD
0543 ARMENIAN CAPITAL LETTER CHEH
0544 ARMENIAN CAPITAL LETTER MEN
0545 ARMENIAN CAPITAL LETTER YI
0546 ARMENIAN CAPITAL LETTER NOW
0547 ARMENIAN CAPITAL LETTER SHA
0548 ARMENIAN CAPITAL LETTER VO
0549 ARMENIAN CAPITAL LETTER CHA
054A ARMENIAN CAPITAL LETTER PEH
054B ARMENIAN CAPITAL LETTER JHEH
054C ARMENIAN CAPITAL LETTER RA
054D ARMENIAN CAPITAL LETTER SEH
054E ARMENIAN CAPITAL LETTER VEW
054F ARMENIAN CAPITAL LETTER TIWN
0550 ARMENIAN CAPITAL LETTER REH
0551 ARMENIAN CAPITAL LETTER CO
0552 ARMENIAN CAPITAL LETTER YIWN
0553 ARMENIAN CAPITAL LETTER PIWR
0554 ARMENIAN CAPITAL LETTER KEH
0555 ARMENIAN CAPITAL LETTER OH
0556 ARMENIAN CAPITAL LETTER FEH
0559 ARMENIAN MODIFIER LETTER LEFT HALF RING
055A ARMENIAN APOSTROPHE
055B ARMENIAN EMPHASIS MARK
055C ARMENIAN EXCLAMATION MARK
055D ARMENIAN COMMA
055E ARMENIAN QUESTION MARK
055F ARMENIAN ABBREVIATION MARK
0560 ARMENIAN SMALL LETTER TURNED AYB
0561 ARMENIAN SMALL LETTER AYB
0562 ARMENIAN SMALL LETTER BEN
0563 ARMENIAN SMALL LETTER GIM
0564 ARMENIAN SMALL LETTER DA
0565 ARMENIAN SMALL LETTER ECH
0566 ARMENIAN SMALL LETTER ZA
0567 ARMENIAN SMALL LETTER EH
0568 ARMENIAN SMALL LETTER ET
0569 ARMENIAN SMALL LETTER TO
056A ARMENIAN SMALL LETTER ZHE
056B ARMENIAN SMALL LETTER INI
056C ARMENIAN SMALL LETTER LIWN
056D ARMENIAN SMALL LETTER XEH
056E ARMENIAN SMALL LETTER CA
056F ARMENIAN SMALL LETTER KEN
0570 ARMENIAN SMALL LETTER HO
0571 ARMENIAN SMALL LETTER JA
0572 ARMENIAN SMALL LETTER GHAD
0573 ARMENIAN SMALL LETTER CHEH
0574 ARMENIAN SMALL LETTER MEN
0575 ARMENIAN SMALL LETTER YI
0576 ARMENIAN SMALL LETTER NOW
0577 ARMENIAN SMALL LETTER SHA
0578 ARMENIAN SMALL LETTER VO
0579 ARMENIAN SMALL LETTER CHA
057A ARMENIAN SMALL LETTER PEH
057B ARMENIAN SMALL LETTER JHEH
057C ARMENIAN SMALL LETTER RA
057D ARMENIAN SMALL LETTER SEH
057E ARMENIAN SMALL LETTER VEW
057F ARMENIAN SMALL LETTER TIWN
0580 ARMENIAN SMALL LETTER REH
0581 ARMENIAN SMALL LETTER CO
0582 ARMENIAN SMALL LETTER YIWN
0583 ARMENIAN SMALL LETTER PIWR
0584 ARMENIAN SMALL LETTER KEH
0585 ARMENIAN SMALL LETTER OH
0586 ARMENIAN SMALL LETTER FEH
0587 ARMENIAN SMALL LIGATURE ECH YIWN
0588 ARMENIAN SMALL LETTER YI WITH STROKE
0589 ARMENIAN FULL STOP
058A ARMENIAN HYPHEN
058D RIGHT-FACING ARMENIAN ETERNITY SIGN
058E LEFT-FACING ARMENIAN ETERNITY SIGN
058F ARMENIAN DRAM SIGN
0591 HEBREW ACCENT ETNAHTA
0592 HEBREW ACCENT SEGOL
0593 HEBREW ACCENT SHALSHELET
0594 HEBREW ACCENT ZAQEF QATAN
0595 HEBREW ACCENT ZAQEF GADOL
0596 HEBREW ACCENT TIPEHA
0597 HEBREW ACCENT REVIA
0598 HEBREW ACCENT ZARQA
0599 HEBREW ACCENT PASHTA
059A HEBREW ACCENT YETIV
059B HEBREW ACCENT TEVIR
059C HEBREW ACCENT GERESH
059D HEBREW ACCENT GERESH MUQDAM
059E HEBREW ACCENT GERSHAYIM
059F HEBREW ACCENT QARNEY PARA
05A0 HEBREW ACCENT TELISHA GEDOLA
05A1 HEBREW ACCENT PAZER
05A2 HEBREW ACCENT ATNAH HAFUKH
05A3 HEBREW ACCENT MUNAH
05A4 HEBREW ACCENT MAHAPAKH
05A5 HEBREW ACCENT MERKHA
05A6 HEBREW ACCENT MERKHA KEFULA
05A7 HEBREW ACCENT DARGA
05A8 HEBREW ACCENT QADMA
05A9 HEBREW ACCENT TELISHA QETANA
05AA HEBREW ACCENT YERAH BEN YOMO
05AB HEBREW ACCENT OLE
05AC HEBREW ACCENT ILUY
05AD HEBREW ACCENT DEHI
05AE HEBREW ACCENT ZINOR
05AF HEBREW MARK MASORA CIRCLE
05B0 HEBREW POINT SHEVA
05B1 HEBREW POINT HATAF SEGOL
05B2 HEBREW POINT HATAF PATAH
05B3 HEBREW POINT HATAF QAMATS
05B4 HEBREW POINT HIRIQ
05B5 HEBREW POINT TSERE
05B6 HEBREW POINT SEGOL
05B7 HEBREW POINT PATAH
05B8 HEBREW POINT QAMATS
05B9 HEBREW POINT HOLAM
05BA HEBREW POINT HOLAM HASER FOR VAV
05BB HEBREW POINT QUBUTS
05BC HEBREW POINT DAGESH OR MAPIQ
05BD HEBREW POINT METEG
05BE HEBREW PUNCTUATION MAQAF
05BF HEBREW POINT RAFE
05C0 HEBREW PUNCTUATION PASEQ
05C1 HEBREW POINT SHIN DOT
05C2 HEBREW POINT SIN DOT
05C3 HEBREW PUNCTUATION SOF PASUQ
05C4 HEBREW MARK UPPER DOT
05C5 HEBREW MARK LOWER DOT
05C6 HEBREW PUNCTUATION NUN HAFUKHA
05C7 HEBREW POINT QAMATS QATAN
05D0 HEBREW LETTER ALEF
05D1 HEBREW LETTER BET
05D2 HEBREW LETTER GIMEL
05D3 HEBREW LETTER DALET
05D4 HEBREW LETTER HE
05D5 HEBREW LETTER VAV
05D6 HEBREW LETTER ZAYIN
05D7 HEBREW LETTER HET
05D8 HEBREW LETTER TET
05D9 HEBREW LETTER YOD
05DA HEBREW LETTER FINAL KAF
05DB HEBREW LETTER KAF
05DC HEBREW LETTER LAMED
05DD HEBREW LETTER FINAL MEM
05DE HEBREW LETTER MEM
05DF HEBREW LETTER FINAL NUN
05E0 HEBREW LETTER NUN
05E1 HEBREW LETTER SAMEKH
05E2 HEBREW LETTER AYIN
05E3 HEBREW LETTER FINAL PE
05E4 HEBREW LETTER PE
05E5 HEBREW LETTER FINAL TSADI
05E6 HEBREW LETTER TSADI
05E7 HEBREW LETTER QOF
05E8 HEBREW LETTER RESH
05E9 HEBREW LETTER SHIN
05EA HEBREW LETTER TAV
05EF HEBREW YOD TRIANGLE
05F0 HEBREW LIGATURE YIDDISH DOUBLE VAV
05F1 HEBREW LIGATURE YIDDISH VAV YOD
05F2 HEBREW LIGATURE YIDDISH DOUBLE YOD
05F3 HEBREW PUNCTUATION GERESH
05F4 HEBREW PUNCTUATION GERSHAYIM
0600 ARABIC NUMBER SIGN
0601 ARABIC SIGN SANAH
0602 ARABIC FOOTNOTE MARKER
0603 ARABIC SIGN SAFHA
0604 ARABIC SIGN SAMVAT
0605 ARABIC NUMBER MARK ABOVE
0606 ARABIC-INDIC CUBE ROOT
0607 ARABIC-INDIC FOURTH ROOT
0608 ARABIC RAY
0609 ARABIC-INDIC PER MILLE SIGN
060A ARABIC-INDIC PER TEN THOUSAND SIGN
060B AFGHANI SIGN
060C ARABIC COMMA
060D ARABIC DATE SEPARATOR
060E ARABIC POETIC VERSE SIGN
060F ARABIC SIGN MISRA
0610 ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM
0611 ARABIC SIGN ALAYHE ASSALLAM
0612 ARABIC SIGN RAHMATULLAH ALAYHE
0613 ARABIC SIGN RADI ALLAHOU ANHU
0614 ARABIC SIGN TAKHALLUS
0615 ARABIC SMALL HIGH TAH
0616 ARABIC SMALL HIGH LIGATURE ALEF WITH LAM WITH YEH
0617 ARABIC SMALL HIGH ZAIN
0618 ARABIC SMALL FATHA
0619 ARABIC SMALL DAMMA
061A ARABIC SMALL KASRA
061B ARABIC SEMICOLON
061C ARABIC LETTER MARK, ALM
061D ARABIC END OF TEXT MARK
061E ARABIC TRIPLE DOT PUNCTUATION MARK
061F ARABIC QUESTION MARK
0620 ARABIC LETTER KASHMIRI YEH
0621 ARABIC LETTER HAMZA
0622 ARABIC LETTER ALEF WITH MADDA ABOVE
0623 ARABIC LETTER ALEF WITH HAMZA ABOVE
0624 ARABIC LETTER WAW WITH HAMZA ABOVE
0625 ARABIC LETTER ALEF WITH HAMZA BELOW
0626 ARABIC LETTER YEH WITH HAMZA ABOVE
0627 ARABIC LETTER ALEF
0628 ARABIC LETTER BEH
0629 ARABIC LETTER TEH MARBUTA
062A ARABIC LETTER TEH
062B ARABIC LETTER THEH
062C ARABIC LETTER JEEM
062D ARABIC LETTER HAH
062E ARABIC LETTER KHAH
062F ARABIC LETTER DAL
0630 ARABIC LETTER THAL
0631 ARABIC LETTER REH
0632 ARABIC LETTER ZAIN
0633 ARABIC LETTER SEEN
0634 ARABIC LETTER SHEEN
0635 ARABIC LETTER SAD
0636 ARABIC LETTER DAD
0637 ARABIC LETTER TAH
0638 ARABIC LETTER ZAH
0639 ARABIC LETTER AIN
063A ARABIC LETTER GHAIN
063B ARABIC LETTER KEHEH WITH TWO DOTS ABOVE
063C ARABIC LETTER KEHEH WITH THREE DOTS BELOW
063D ARABIC LETTER FARSI YEH WITH INVERTED V
063E ARABIC LETTER FARSI YEH WITH TWO DOTS ABOVE
063F ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE
0640 ARABIC TATWEEL
0641 ARABIC LETTER FEH
0642 ARABIC LETTER QAF
0643 ARABIC LETTER KAF
0644 ARABIC LETTER LAM
0645 ARABIC LETTER MEEM
0646 ARABIC LETTER NOON
0647 ARABIC LETTER HEH
0648 ARABIC LETTER WAW
0649 ARABIC LETTER ALEF MAKSURA
064A ARABIC LETTER YEH
064B ARABIC FATHATAN
064C ARABIC DAMMATAN
064D ARABIC KASRATAN
064E ARABIC FATHA
064F ARABIC DAMMA
0650 ARABIC KASRA
0651 ARABIC SHADDA
0652 ARABIC SUKUN
0653 ARABIC MADDAH ABOVE
0654 ARABIC HAMZA ABOVE
0655 ARABIC HAMZA BELOW
0656 ARABIC SUBSCRIPT ALEF
0657 ARABIC INVERTED DAMMA
0658 ARABIC MARK NOON GHUNNA
0659 ARABIC ZWARAKAY
065A ARABIC VOWEL SIGN SMALL V ABOVE
065B ARABIC VOWEL SIGN INVERTED SMALL V ABOVE
065C ARABIC VOWEL SIGN DOT BELOW
065D ARABIC REVERSED DAMMA
065E ARABIC FATHA WITH TWO DOTS
065F ARABIC WAVY HAMZA BELOW
0660 ARABIC-INDIC DIGIT ZERO
0661 ARABIC-INDIC DIGIT ONE
0662 ARABIC-INDIC DIGIT TWO
0663 ARABIC-INDIC DIGIT THREE
0664 ARABIC-INDIC DIGIT FOUR
0665 ARABIC-INDIC DIGIT FIVE
0666 ARABIC-INDIC DIGIT SIX
0667 ARABIC-INDIC DIGIT SEVEN
0668 ARABIC-INDIC DIGIT EIGHT
0669 ARABIC-INDIC DIGIT NINE
066A ARABIC PERCENT SIGN
066B ARABIC DECIMAL SEPARATOR
066C ARABIC THOUSANDS SEPARATOR
066D ARABIC FIVE POINTED STAR
066E ARABIC LETTER DOTLESS BEH
066F ARABIC LETTER DOTLESS QAF
0670 ARABIC LETTER SUPERSCRIPT ALEF
0671 ARABIC LETTER ALEF WASLA
0672 ARABIC LETTER ALEF WITH WAVY HAMZA ABOVE
0673 ARABIC LETTER ALEF WITH WAVY HAMZA BELOW
0674 ARABIC LETTER HIGH HAMZA
0675 ARABIC LETTER HIGH HAMZA ALEF
0676 ARABIC LETTER HIGH HAMZA WAW
0677 ARABIC LETTER U WITH HAMZA ABOVE
0678 ARABIC LETTER HIGH HAMZA YEH
0679 ARABIC LETTER TTEH
067A ARABIC LETTER TTEHEH
067B ARABIC LETTER BEEH
067C ARABIC LETTER TEH WITH RING
067D ARABIC LETTER TEH WITH THREE DOTS ABOVE DOWNWARDS
067E ARABIC LETTER PEH
067F ARABIC LETTER TEHEH
0680 ARABIC LETTER BEHEH
0681 ARABIC LETTER HAH WITH HAMZA ABOVE
0682 ARABIC LETTER HAH WITH TWO DOTS VERTICAL ABOVE
0683 ARABIC LETTER NYEH
0684 ARABIC LETTER DYEH
0685 ARABIC LETTER HAH WITH THREE DOTS ABOVE
0686 ARABIC LETTER TCHEH
0687 ARABIC LETTER TCHEHEH
0688 ARABIC LETTER DDAL
0689 ARABIC LETTER DAL WITH RING
068A ARABIC LETTER DAL WITH DOT BELOW
068B ARABIC LETTER DAL WITH DOT BELOW AND SMALL TAH
068C ARABIC LETTER DAHAL
068D ARABIC LETTER DDAHAL
068E ARABIC LETTER DUL
068F ARABIC LETTER DAL WITH THREE DOTS ABOVE DOWNWARDS
0690 ARABIC LETTER DAL WITH FOUR DOTS ABOVE
0691 ARABIC LETTER RREH
0692 ARABIC LETTER REH WITH SMALL V
0693 ARABIC LETTER REH WITH RING
0694 ARABIC LETTER REH WITH DOT BELOW
0695 ARABIC LETTER REH WITH SMALL V BELOW
0696 ARABIC LETTER REH WITH DOT BELOW AND DOT ABOVE
0697 ARABIC LETTER REH WITH TWO DOTS ABOVE
0698 ARABIC LETTER JEH
0699 ARABIC LETTER REH WITH FOUR DOTS ABOVE
069A ARABIC LETTER SEEN WITH DOT BELOW AND DOT ABOVE
069B ARABIC LETTER SEEN WITH THREE DOTS BELOW
069C ARABIC LETTER SEEN WITH THREE DOTS BELOW AND THREE DOTS ABOVE
069D ARABIC LETTER SAD WITH TWO DOTS BELOW
069E ARABIC LETTER SAD WITH THREE DOTS ABOVE
069F ARABIC LETTER TAH WITH THREE DOTS ABOVE
06A0 ARABIC LETTER AIN WITH THREE DOTS ABOVE
06A1 ARABIC LETTER DOTLESS FEH
06A2 ARABIC LETTER FEH WITH DOT MOVED BELOW
06A3 ARABIC LETTER FEH WITH DOT BELOW
06A4 ARABIC LETTER VEH
06A5 ARABIC LETTER FEH WITH THREE DOTS BELOW
06A6 ARABIC LETTER PEHEH
06A7 ARABIC LETTER QAF WITH DOT ABOVE
06A8 ARABIC LETTER QAF WITH THREE DOTS ABOVE
06A9 ARABIC LETTER KEHEH
06AA ARABIC LETTER SWASH KAF
06AB ARABIC LETTER KAF WITH RING
06AC ARABIC LETTER KAF WITH DOT ABOVE
06AD ARABIC LETTER NG
06AE ARABIC LETTER KAF WITH THREE DOTS BELOW
06AF ARABIC LETTER GAF
06B0 ARABIC LETTER GAF WITH RING
06B1 ARABIC LETTER NGOEH
06B2 ARABIC LETTER GAF WITH TWO DOTS BELOW
06B3 ARABIC LETTER GUEH
06B4 ARABIC LETTER GAF WITH THREE DOTS ABOVE
06B5 ARABIC LETTER LAM WITH SMALL V
06B6 ARABIC LETTER LAM WITH DOT ABOVE
06B7 ARABIC LETTER LAM WITH THREE DOTS ABOVE
06B8 ARABIC LETTER LAM WITH THREE DOTS BELOW
06B9 ARABIC LETTER NOON WITH DOT BELOW
06BA ARABIC LETTER NOON GHUNNA
06BB ARABIC LETTER RNOON
06BC ARABIC LETTER NOON WITH RING
06BD ARABIC LETTER NOON WITH THREE DOTS ABOVE
06BE ARABIC LETTER HEH DOACHASHMEE
06BF ARABIC LETTER TCHEH WITH DOT ABOVE
06C0 ARABIC LETTER HEH WITH YEH ABOVE
06C1 ARABIC LETTER HEH GOAL
06C2 ARABIC LETTER HEH GOAL WITH HAMZA ABOVE
06C3 ARABIC LETTER TEH MARBUTA GOAL
06C4 ARABIC LETTER WAW WITH RING
06C5 ARABIC LETTER KIRGHIZ OE
06C6 ARABIC LETTER OE
06C7 ARABIC LETTER U
06C8 ARABIC LETTER YU
06C9 ARABIC LETTER KIRGHIZ YU
06CA ARABIC LETTER WAW WITH TWO DOTS ABOVE
06CB ARABIC LETTER VE
06CC ARABIC LETTER FARSI YEH
06CD ARABIC LETTER YEH WITH TAIL
06CE ARABIC LETTER YEH WITH SMALL V
06CF ARABIC LETTER WAW WITH DOT ABOVE
06D0 ARABIC LETTER E
06D1 ARABIC LETTER YEH WITH THREE DOTS BELOW
06D2 ARABIC LETTER YEH BARREE
06D3 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
06D4 ARABIC FULL STOP
06D5 ARABIC LETTER AE
06D6 ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA
06D7 ARABIC SMALL HIGH LIGATURE QAF WITH LAM WITH ALEF MAKSURA
06D8 ARABIC SMALL HIGH MEEM INITIAL FORM
06D9 ARABIC SMALL HIGH LAM ALEF
06DA ARABIC SMALL HIGH JEEM
06DB ARABIC SMALL HIGH THREE DOTS
06DC ARABIC SMALL HIGH SEEN
06DD ARABIC END OF AYAH
06DE ARABIC START OF RUB EL HIZB
06DF ARABIC SMALL HIGH ROUNDED ZERO
06E0 ARABIC SMALL HIGH UPRIGHT RECTANGULAR ZERO
06E1 ARABIC SMALL HIGH DOTLESS HEAD OF KHAH
06E2 ARABIC SMALL HIGH MEEM ISOLATED FORM
06E3 ARABIC SMALL LOW SEEN
06E4 ARABIC SMALL HIGH MADDA
06E5 ARABIC SMALL WAW
06E6 ARABIC SMALL YEH
06E7 ARABIC SMALL HIGH YEH
06E8 ARABIC SMALL HIGH NOON
06E9 ARABIC PLACE OF SAJDAH
06EA ARABIC EMPTY CENTRE LOW STOP
06EB ARABIC EMPTY CENTRE HIGH STOP
06EC ARABIC ROUNDED HIGH STOP WITH FILLED CENTRE
06ED ARABIC SMALL LOW MEEM
06EE ARABIC LETTER DAL WITH INVERTED V
06EF ARABIC LETTER REH WITH INVERTED V
06F0 EXTENDED ARABIC-INDIC DIGIT ZERO
06F1 EXTENDED ARABIC-INDIC DIGIT ONE
06F2 EXTENDED ARABIC-INDIC DIGIT TWO
06F3 EXTENDED ARABIC-INDIC DIGIT THREE
06F4 EXTENDED ARABIC-INDIC DIGIT FOUR
06F5 EXTENDED ARABIC-INDIC DIGIT FIVE
06F6 EXTENDED ARABIC-INDIC DIGIT SIX
06F7 EXTENDED ARABIC-INDIC DIGIT SEVEN
06F8 EXTENDED ARABIC-INDIC DIGIT EIGHT
06F9 EXTENDED ARABIC-INDIC DIGIT NINE
06FA ARABIC LETTER SHEEN WITH DOT BELOW
06FB ARABIC LETTER DAD WITH DOT BELOW
06FC ARABIC LETTER GHAIN WITH DOT BELOW
06FD ARABIC SIGN SINDHI AMPERSAND
06FE ARABIC SIGN SINDHI POSTPOSITION MEN
06FF ARABIC LETTER HEH WITH INVERTED V
0700 SYRIAC END OF PARAGRAPH
0701 SYRIAC SUPRALINEAR FULL STOP
0702 SYRIAC SUBLINEAR FULL STOP
0703 SYRIAC SUPRALINEAR COLON
0704 SYRIAC SUBLINEAR COLON
0705 SYRIAC HORIZONTAL COLON
0706 SYRIAC COLON SKEWED LEFT
0707 SYRIAC COLON SKEWED RIGHT
0708 SYRIAC SUPRALINEAR COLON SKEWED LEFT
0709 SYRIAC SUBLINEAR COLON SKEWED RIGHT
070A SYRIAC CONTRACTION
070B SYRIAC HARKLEAN OBELUS
070C SYRIAC HARKLEAN METOBELUS
070D SYRIAC HARKLEAN ASTERISCUS
070F SYRIAC ABBREVIATION MARK
0710 SYRIAC LETTER ALAPH
0711 SYRIAC LETTER SUPERSCRIPT ALAPH
0712 SYRIAC LETTER BETH
0713 SYRIAC LETTER GAMAL
0714 SYRIAC LETTER GAMAL GARSHUNI
0715 SYRIAC LETTER DALATH
0716 SYRIAC LETTER DOTLESS DALATH RISH
0717 SYRIAC LETTER HE
0718 SYRIAC LETTER WAW
0719 SYRIAC LETTER ZAIN
071A SYRIAC LETTER HETH
071B SYRIAC LETTER TETH
071C SYRIAC LETTER TETH GARSHUNI
071D SYRIAC LETTER YUDH
071E SYRIAC LETTER YUDH HE
071F SYRIAC LETTER KAPH
0720 SYRIAC LETTER LAMADH
0721 SYRIAC LETTER MIM
0722 SYRIAC LETTER NUN
0723 SYRIAC LETTER SEMKATH
0724 SYRIAC LETTER FINAL SEMKATH
0725 SYRIAC LETTER E
0726 SYRIAC LETTER PE
0727 SYRIAC LETTER REVERSED PE
0728 SYRIAC LETTER SADHE
0729 SYRIAC LETTER QAPH
072A SYRIAC LETTER RISH
072B SYRIAC LETTER SHIN
072C SYRIAC LETTER TAW
072D SYRIAC LETTER PERSIAN BHETH
072E SYRIAC LETTER PERSIAN GHAMAL
072F SYRIAC LETTER PERSIAN DHALATH
0730 SYRIAC PTHAHA ABOVE
0731 SYRIAC PTHAHA BELOW
0732 SYRIAC PTHAHA DOTTED
0733 SYRIAC ZQAPHA ABOVE
0734 SYRIAC ZQAPHA BELOW
0735 SYRIAC ZQAPHA DOTTED
0736 SYRIAC RBASA ABOVE
0737 SYRIAC RBASA BELOW
0738 SYRIAC DOTTED ZLAMA HORIZONTAL
0739 SYRIAC DOTTED ZLAMA ANGULAR
073A SYRIAC HBASA ABOVE
073B SYRIAC HBASA BELOW
073C SYRIAC HBASA-ESASA DOTTED
073D SYRIAC ESASA ABOVE
073E SYRIAC ESASA BELOW
073F SYRIAC RWAHA
0740 SYRIAC FEMININE DOT
0741 SYRIAC QUSHSHAYA
0742 SYRIAC RUKKAKHA
0743 SYRIAC TWO VERTICAL DOTS ABOVE
0744 SYRIAC TWO VERTICAL DOTS BELOW
0745 SYRIAC THREE DOTS ABOVE
0746 SYRIAC THREE DOTS BELOW
0747 SYRIAC OBLIQUE LINE ABOVE
0748 SYRIAC OBLIQUE LINE BELOW
0749 SYRIAC MUSIC
074A SYRIAC BARREKH
074D SYRIAC LETTER SOGDIAN ZHAIN
074E SYRIAC LETTER SOGDIAN KHAPH
074F SYRIAC LETTER SOGDIAN FE
0750 ARABIC LETTER BEH WITH THREE DOTS HORIZONTALLY BELOW
0751 ARABIC LETTER BEH WITH DOT BELOW AND THREE DOTS ABOVE
0752 ARABIC LETTER BEH WITH THREE DOTS POINTING UPWARDS BELOW
0753 ARABIC LETTER BEH WITH THREE DOTS POINTING UPWARDS BELOW AND TWO DOTS ABOVE
0754 ARABIC LETTER BEH WITH TWO DOTS BELOW AND DOT ABOVE
0755 ARABIC LETTER BEH WITH INVERTED SMALL V BELOW
0756 ARABIC LETTER BEH WITH SMALL V
0757 ARABIC LETTER HAH WITH TWO DOTS ABOVE
0758 ARABIC LETTER HAH WITH THREE DOTS POINTING UPWARDS BELOW
0759 ARABIC LETTER DAL WITH TWO DOTS VERTICALLY BELOW AND SMALL TAH
075A ARABIC LETTER DAL WITH INVERTED SMALL V BELOW
075B ARABIC LETTER REH WITH STROKE
075C ARABIC LETTER SEEN WITH FOUR DOTS ABOVE
075D ARABIC LETTER AIN WITH TWO DOTS ABOVE
075E ARABIC LETTER AIN WITH THREE DOTS POINTING DOWNWARDS ABOVE
075F ARABIC LETTER AIN WITH TWO DOTS VERTICALLY ABOVE
0760 ARABIC LETTER FEH WITH TWO DOTS BELOW
0761 ARABIC LETTER FEH WITH THREE DOTS POINTING UPWARDS BELOW
0762 ARABIC LETTER KEHEH WITH DOT ABOVE
0763 ARABIC LETTER KEHEH WITH THREE DOTS ABOVE
0764 ARABIC LETTER KEHEH WITH THREE DOTS POINTING UPWARDS BELOW
0765 ARABIC LETTER MEEM WITH DOT ABOVE
0766 ARABIC LETTER MEEM WITH DOT BELOW
0767 ARABIC LETTER NOON WITH TWO DOTS BELOW
0768 ARABIC LETTER NOON WITH SMALL TAH
0769 ARABIC LETTER NOON WITH SMALL V
076A ARABIC LETTER LAM WITH BAR
076B ARABIC LETTER REH WITH TWO DOTS VERTICALLY ABOVE
076C ARABIC LETTER REH WITH HAMZA ABOVE
076D ARABIC LETTER SEEN WITH TWO DOTS VERTICALLY ABOVE
076E ARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH BELOW
076F ARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH AND TWO DOTS
0770 ARABIC LETTER SEEN WITH SMALL ARABIC LETTER TAH AND TWO DOTS
0771 ARABIC LETTER REH WITH SMALL ARABIC LETTER TAH AND TWO DOTS
0772 ARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH ABOVE
0773 ARABIC LETTER ALEF WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE
0774 ARABIC LETTER ALEF WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE
0775 ARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE
0776 ARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE
0777 ARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT FOUR BELOW
0778 ARABIC LETTER WAW WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE
0779 ARABIC LETTER WAW WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE
077A ARABIC LETTER YEH BARREE WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE
077B ARABIC LETTER YEH BARREE WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE
077C ARABIC LETTER HAH WITH EXTENDED ARABIC-INDIC DIGIT FOUR BELOW
077D ARABIC LETTER SEEN WITH EXTENDED ARABIC-INDIC DIGIT FOUR ABOVE
077E ARABIC LETTER SEEN WITH INVERTED V
077F ARABIC LETTER KAF WITH TWO DOTS ABOVE
0780 THAANA LETTER HAA
0781 THAANA LETTER SHAVIYANI
0782 THAANA LETTER NOONU
0783 THAANA LETTER RAA
0784 THAANA LETTER BAA
0785 THAANA LETTER LHAVIYANI
0786 THAANA LETTER KAAFU
0787 THAANA LETTER ALIFU
0788 THAANA LETTER VAAVU
0789 THAANA LETTER MEEMU
078A THAANA LETTER FAAFU
078B THAANA LETTER DHAALU
078C THAANA LETTER THAA
078D THAANA LETTER LAAMU
078E THAANA LETTER GAAFU
078F THAANA LETTER GNAVIYANI
0790 THAANA LETTER SEENU
0791 THAANA LETTER DAVIYANI
0792 THAANA LETTER ZAVIYANI
0793 THAANA LETTER TAVIYANI
0794 THAANA LETTER YAA
0795 THAANA LETTER PAVIYANI
0796 THAANA LETTER JAVIYANI
0797 THAANA LETTER CHAVIYANI
0798 THAANA LETTER TTAA
0799 THAANA LETTER HHAA
079A THAANA LETTER KHAA
079B THAANA LETTER THAALU
079C THAANA LETTER ZAA
079D THAANA LETTER SHEENU
079E THAANA LETTER SAADHU
079F THAANA LETTER DAADHU
07A0 THAANA LETTER TO
07A1 THAANA LETTER ZO
07A2 THAANA LETTER AINU
07A3 THAANA LETTER GHAINU
07A4 THAANA LETTER QAAFU
07A5 THAANA LETTER WAAVU
07A6 THAANA ABAFILI
07A7 THAANA AABAAFILI
07A8 THAANA IBIFILI
07A9 THAANA EEBEEFILI
07AA THAANA UBUFILI
07AB THAANA OOBOOFILI
07AC THAANA EBEFILI
07AD THAANA EYBEYFILI
07AE THAANA OBOFILI
07AF THAANA OABOAFILI
07B0 THAANA SUKUN
07B1 THAANA LETTER NAA
07C0 NKO DIGIT ZERO
07C1 NKO DIGIT ONE
07C2 NKO DIGIT TWO
07C3 NKO DIGIT THREE
07C4 NKO DIGIT FOUR
07C5 NKO DIGIT FIVE
07C6 NKO DIGIT SIX
07C7 NKO DIGIT SEVEN
07C8 NKO DIGIT EIGHT
07C9 NKO DIGIT NINE
07CA NKO LETTER A
07CB NKO LETTER EE
07CC NKO LETTER I
07CD NKO LETTER E
07CE NKO LETTER U
07CF NKO LETTER OO
07D0 NKO LETTER O
07D1 NKO LETTER DAGBASINNA
07D2 NKO LETTER N
07D3 NKO LETTER BA
07D4 NKO LETTER PA
07D5 NKO LETTER TA
07D6 NKO LETTER JA
07D7 NKO LETTER CHA
07D8 NKO LETTER DA
07D9 NKO LETTER RA
07DA NKO LETTER RRA
07DB NKO LETTER SA
07DC NKO LETTER GBA
07DD NKO LETTER FA
07DE NKO LETTER KA
07DF NKO LETTER LA
07E0 NKO LETTER NA WOLOSO
07E1 NKO LETTER MA
07E2 NKO LETTER NYA
07E3 NKO LETTER NA
07E4 NKO LETTER HA
07E5 NKO LETTER WA
07E6 NKO LETTER YA
07E7 NKO LETTER NYA WOLOSO
07E8 NKO LETTER JONA JA
07E9 NKO LETTER JONA CHA
07EA NKO LETTER JONA RA
07EB NKO COMBINING SHORT HIGH TONE
07EC NKO COMBINING SHORT LOW TONE
07ED NKO COMBINING SHORT RISING TONE
07EE NKO COMBINING LONG DESCENDING TONE
07EF NKO COMBINING LONG HIGH TONE
07F0 NKO COMBINING LONG LOW TONE
07F1 NKO COMBINING LONG RISING TONE
07F2 NKO COMBINING NASALIZATION MARK
07F3 NKO COMBINING DOUBLE DOT ABOVE
07F4 NKO HIGH TONE APOSTROPHE
07F5 NKO LOW TONE APOSTROPHE
07F6 NKO SYMBOL OO DENNEN
07F7 NKO SYMBOL GBAKURUNEN
07F8 NKO COMMA
07F9 NKO EXCLAMATION MARK
07FA NKO LAJANYALAN
07FD NKO DANTAYALAN
07FE NKO DOROME SIGN
07FF NKO TAMAN SIGN
0800 SAMARITAN LETTER ALAF
0801 SAMARITAN LETTER BIT
0802 SAMARITAN LETTER GAMAN
0803 SAMARITAN LETTER DALAT
0804 SAMARITAN LETTER IY
0805 SAMARITAN LETTER BAA
0806 SAMARITAN LETTER ZEN
0807 SAMARITAN LETTER IT
0808 SAMARITAN LETTER TIT
0809 SAMARITAN LETTER YUT
080A SAMARITAN LETTER KAAF
080B SAMARITAN LETTER LABAT
080C SAMARITAN LETTER MIM
080D SAMARITAN LETTER NUN
080E SAMARITAN LETTER SINGAAT
080F SAMARITAN LETTER IN
0810 SAMARITAN LETTER FI
0811 SAMARITAN LETTER TSAADIY
0812 SAMARITAN LETTER QUF
0813 SAMARITAN LETTER RISH
0814 SAMARITAN LETTER SHAN
0815 SAMARITAN LETTER TAAF
0816 SAMARITAN MARK IN
0817 SAMARITAN MARK IN-ALAF
0818 SAMARITAN MARK OCCLUSION
0819 SAMARITAN MARK DAGESH
081A SAMARITAN MODIFIER LETTER EPENTHETIC YUT
081B SAMARITAN MARK EPENTHETIC YUT
081C SAMARITAN VOWEL SIGN LONG E
081D SAMARITAN VOWEL SIGN E
081E SAMARITAN VOWEL SIGN OVERLONG AA
081F SAMARITAN VOWEL SIGN LONG AA
0820 SAMARITAN VOWEL SIGN AA
0821 SAMARITAN VOWEL SIGN OVERLONG A
0822 SAMARITAN VOWEL SIGN LONG A
0823 SAMARITAN VOWEL SIGN A
0824 SAMARITAN MODIFIER LETTER SHORT A
0825 SAMARITAN VOWEL SIGN SHORT A
0826 SAMARITAN VOWEL SIGN LONG U
0827 SAMARITAN VOWEL SIGN U
0828 SAMARITAN MODIFIER LETTER I
0829 SAMARITAN VOWEL SIGN LONG I
082A SAMARITAN VOWEL SIGN I
082B SAMARITAN VOWEL SIGN O
082C SAMARITAN VOWEL SIGN SUKUN
082D SAMARITAN MARK NEQUDAA
0830 SAMARITAN PUNCTUATION NEQUDAA
0831 SAMARITAN PUNCTUATION AFSAAQ
0832 SAMARITAN PUNCTUATION ANGED
0833 SAMARITAN PUNCTUATION BAU
0834 SAMARITAN PUNCTUATION ATMAAU
0835 SAMARITAN PUNCTUATION SHIYYAALAA
0836 SAMARITAN ABBREVIATION MARK
0837 SAMARITAN PUNCTUATION MELODIC QITSA
0838 SAMARITAN PUNCTUATION ZIQAA
0839 SAMARITAN PUNCTUATION QITSA
083A SAMARITAN PUNCTUATION ZAEF
083B SAMARITAN PUNCTUATION TURU
083C SAMARITAN PUNCTUATION ARKAANU
083D SAMARITAN PUNCTUATION SOF MASHFAAT
083E SAMARITAN PUNCTUATION ANNAAU
0840 MANDAIC LETTER HALQA
0841 MANDAIC LETTER AB
0842 MANDAIC LETTER AG
0843 MANDAIC LETTER AD
0844 MANDAIC LETTER AH
0845 MANDAIC LETTER USHENNA
0846 MANDAIC LETTER AZ
0847 MANDAIC LETTER IT
0848 MANDAIC LETTER ATT
0849 MANDAIC LETTER AKSA
084A MANDAIC LETTER AK
084B MANDAIC LETTER AL
084C MANDAIC LETTER AM
084D MANDAIC LETTER AN
084E MANDAIC LETTER AS
084F MANDAIC LETTER IN
0850 MANDAIC LETTER AP
0851 MANDAIC LETTER ASZ
0852 MANDAIC LETTER AQ
0853 MANDAIC LETTER AR
0854 MANDAIC LETTER ASH
0855 MANDAIC LETTER AT
0856 MANDAIC LETTER DUSHENNA
0857 MANDAIC LETTER KAD
0858 MANDAIC LETTER AIN
0859 MANDAIC AFFRICATION MARK
085A MANDAIC VOCALIZATION MARK
085B MANDAIC GEMINATION MARK
085E MANDAIC PUNCTUATION
0860 SYRIAC LETTER MALAYALAM NGA
0861 SYRIAC LETTER MALAYALAM JA
0862 SYRIAC LETTER MALAYALAM NYA
0863 SYRIAC LETTER MALAYALAM TTA
0864 SYRIAC LETTER MALAYALAM NNA
0865 SYRIAC LETTER MALAYALAM NNNA
0866 SYRIAC LETTER MALAYALAM BHA
0867 SYRIAC LETTER MALAYALAM RA
0868 SYRIAC LETTER MALAYALAM LLA
0869 SYRIAC LETTER MALAYALAM LLLA
086A SYRIAC LETTER MALAYALAM SSA
0870 ARABIC LETTER ALEF WITH ATTACHED FATHA
0871 ARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHA
0872 ARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE
0873 ARABIC LETTER ALEF WITH LEFT MIDDLE STROKE
0874 ARABIC LETTER ALEF WITH ATTACHED KASRA
0875 ARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA
0876 ARABIC LETTER ALEF WITH ATTACHED ROUND DOT ABOVE
0877 ARABIC LETTER ALEF WITH ATTACHED RIGHT ROUND DOT
0878 ARABIC LETTER ALEF WITH ATTACHED LEFT ROUND DOT
0879 ARABIC LETTER ALEF WITH ATTACHED ROUND DOT BELOW
087A ARABIC LETTER ALEF WITH DOT ABOVE
087B ARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHA AND DOT ABOVE
087C ARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE AND DOT ABOVE
087D ARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA AND DOT ABOVE
087E ARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHA AND LEFT RING
087F ARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE AND LEFT RING
0880 ARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA AND LEFT RING
0881 ARABIC LETTER ALEF WITH ATTACHED RIGHT HAMZA
0882 ARABIC LETTER ALEF WITH ATTACHED LEFT HAMZA
0883 ARABIC TATWEEL WITH OVERSTRUCK HAMZA
0884 ARABIC TATWEEL WITH OVERSTRUCK WAW
0885 ARABIC TATWEEL WITH TWO DOTS BELOW
0886 ARABIC LETTER THIN YEH
0887 ARABIC BASELINE ROUND DOT
0888 ARABIC RAISED ROUND DOT
0889 ARABIC LETTER NOON WITH INVERTED SMALL V
088A ARABIC LETTER HAH WITH INVERTED SMALL V BELOW
088B ARABIC LETTER TAH WITH DOT BELOW
088C ARABIC LETTER TAH WITH THREE DOTS BELOW
088D ARABIC LETTER KEHEH WITH TWO DOTS VERTICALLY BELOW
088E ARABIC VERTICAL TAIL
0890 ARABIC POUND MARK ABOVE
0891 ARABIC PIASTRE MARK ABOVE
0898 ARABIC SMALL HIGH WORD AL-JUZ
0899 ARABIC SMALL LOW WORD ISHMAAM
089A ARABIC SMALL LOW WORD IMAALA
089B ARABIC SMALL LOW WORD TASHEEL
089C ARABIC MADDA WAAJIB
089D ARABIC SUPERSCRIPT ALEF MOKHASSAS
089E ARABIC DOUBLED MADDA
089F ARABIC HALF MADDA OVER MADDA
08A0 ARABIC LETTER BEH WITH SMALL V BELOW
08A1 ARABIC LETTER BEH WITH HAMZA ABOVE
08A2 ARABIC LETTER JEEM WITH TWO DOTS ABOVE
08A3 ARABIC LETTER TAH WITH TWO DOTS ABOVE
08A4 ARABIC LETTER FEH WITH DOT BELOW AND THREE DOTS ABOVE
08A5 ARABIC LETTER QAF WITH DOT BELOW
08A6 ARABIC LETTER LAM WITH DOUBLE BAR
08A7 ARABIC LETTER MEEM WITH THREE DOTS ABOVE
08A8 ARABIC LETTER YEH WITH TWO DOTS BELOW AND HAMZA ABOVE
08A9 ARABIC LETTER YEH WITH TWO DOTS BELOW AND DOT ABOVE
08AA ARABIC LETTER REH WITH LOOP
08AB ARABIC LETTER WAW WITH DOT WITHIN
08AC ARABIC LETTER ROHINGYA YEH
08AD ARABIC LETTER LOW ALEF
08AE ARABIC LETTER DAL WITH THREE DOTS BELOW
08AF ARABIC LETTER SAD WITH THREE DOTS BELOW
08B0 ARABIC LETTER GAF WITH INVERTED STROKE
08B1 ARABIC LETTER STRAIGHT WAW
08B2 ARABIC LETTER ZAIN WITH INVERTED V ABOVE
08B3 ARABIC LETTER AIN WITH THREE DOTS BELOW
08B4 ARABIC LETTER KAF WITH DOT BELOW
08B5 ARABIC LETTER QAF WITH DOT BELOW AND NO DOTS ABOVE
08B6 ARABIC LETTER BEH WITH SMALL MEEM ABOVE
08B7 ARABIC LETTER PEH WITH SMALL MEEM ABOVE
08B8 ARABIC LETTER TEH WITH SMALL TEH ABOVE
08B9 ARABIC LETTER REH WITH SMALL NOON ABOVE
08BA ARABIC LETTER YEH WITH TWO DOTS BELOW AND SMALL NOON ABOVE
08BB ARABIC LETTER AFRICAN FEH
08BC ARABIC LETTER AFRICAN QAF
08BD ARABIC LETTER AFRICAN NOON
08BE ARABIC LETTER PEH WITH SMALL V
08BF ARABIC LETTER TEH WITH SMALL V
08C0 ARABIC LETTER TTEH WITH SMALL V
08C1 ARABIC LETTER TCHEH WITH SMALL V
08C2 ARABIC LETTER KEHEH WITH SMALL V
08C3 ARABIC LETTER GHAIN WITH THREE DOTS ABOVE
08C4 ARABIC LETTER AFRICAN QAF WITH THREE DOTS ABOVE
08C5 ARABIC LETTER JEEM WITH THREE DOTS ABOVE
08C6 ARABIC LETTER JEEM WITH THREE DOTS BELOW
08C7 ARABIC LETTER LAM WITH SMALL ARABIC LETTER TAH ABOVE
08C8 ARABIC LETTER GRAF
08C9 ARABIC SMALL FARSI YEH
08CA ARABIC SMALL HIGH FARSI YEH
08CB ARABIC SMALL HIGH YEH BARREE WITH TWO DOTS BELOW
08CC ARABIC SMALL HIGH WORD SAH
08CD ARABIC SMALL HIGH ZAH
08CE ARABIC LARGE ROUND DOT ABOVE
08CF ARABIC LARGE ROUND DOT BELOW
08D0 ARABIC SUKUN BELOW
08D1 ARABIC LARGE CIRCLE BELOW
08D2 ARABIC LARGE ROUND DOT INSIDE CIRCLE BELOW
08D3 ARABIC SMALL LOW WAW
08D4 ARABIC SMALL HIGH WORD AR-RUB
08D5 ARABIC SMALL HIGH SAD
08D6 ARABIC SMALL HIGH AIN
08D7 ARABIC SMALL HIGH QAF
08D8 ARABIC SMALL HIGH NOON WITH KASRA
08D9 ARABIC SMALL LOW NOON WITH KASRA
08DA ARABIC SMALL HIGH WORD ATH-THALATHA
08DB ARABIC SMALL HIGH WORD AS-SAJDA
08DC ARABIC SMALL HIGH WORD AN-NISF
08DD ARABIC SMALL HIGH WORD SAKTA
08DE ARABIC SMALL HIGH WORD QIF
08DF ARABIC SMALL HIGH WORD WAQFA
08E0 ARABIC SMALL HIGH FOOTNOTE MARKER
08E1 ARABIC SMALL HIGH SIGN SAFHA
08E2 ARABIC DISPUTED END OF AYAH
08E3 ARABIC TURNED DAMMA BELOW
08E4 ARABIC CURLY FATHA
08E5 ARABIC CURLY DAMMA
08E6 ARABIC CURLY KASRA
08E7 ARABIC CURLY FATHATAN
08E8 ARABIC CURLY DAMMATAN
08E9 ARABIC CURLY KASRATAN
08EA ARABIC TONE ONE DOT ABOVE
08EB ARABIC TONE TWO DOTS ABOVE
08EC ARABIC TONE LOOP ABOVE
08ED ARABIC TONE ONE DOT BELOW
08EE ARABIC TONE TWO DOTS BELOW
08EF ARABIC TONE LOOP BELOW
08F0 ARABIC OPEN FATHATAN
08F1 ARABIC OPEN DAMMATAN
08F2 ARABIC OPEN KASRATAN
08F3 ARABIC SMALL HIGH WAW
08F4 ARABIC FATHA WITH RING
08F5 ARABIC FATHA WITH DOT ABOVE
08F6 ARABIC KASRA WITH DOT BELOW
08F7 ARABIC LEFT ARROWHEAD ABOVE
08F8 ARABIC RIGHT ARROWHEAD ABOVE
08F9 ARABIC LEFT ARROWHEAD BELOW
08FA ARABIC RIGHT ARROWHEAD BELOW
08FB ARABIC DOUBLE RIGHT ARROWHEAD ABOVE
08FC ARABIC DOUBLE RIGHT ARROWHEAD ABOVE WITH DOT
08FD ARABIC RIGHT ARROWHEAD ABOVE WITH DOT
08FE ARABIC DAMMA WITH DOT
08FF ARABIC MARK SIDEWAYS NOON GHUNNA
0900 DEVANAGARI SIGN INVERTED CANDRABINDU
0901 DEVANAGARI SIGN CANDRABINDU
0902 DEVANAGARI SIGN ANUSVARA
0903 DEVANAGARI SIGN VISARGA
0904 DEVANAGARI LETTER SHORT A
0905 DEVANAGARI LETTER A
0906 DEVANAGARI LETTER AA
0907 DEVANAGARI LETTER I
0908 DEVANAGARI LETTER II
0909 DEVANAGARI LETTER U
090A DEVANAGARI LETTER UU
090B DEVANAGARI LETTER VOCALIC R
090C DEVANAGARI LETTER VOCALIC L
090D DEVANAGARI LETTER CANDRA E
090E DEVANAGARI LETTER SHORT E
090F DEVANAGARI LETTER E
0910 DEVANAGARI LETTER AI
0911 DEVANAGARI LETTER CANDRA O
0912 DEVANAGARI LETTER SHORT O
0913 DEVANAGARI LETTER O
0914 DEVANAGARI LETTER AU
0915 DEVANAGARI LETTER KA
0916 DEVANAGARI LETTER KHA
0917 DEVANAGARI LETTER GA
0918 DEVANAGARI LETTER GHA
0919 DEVANAGARI LETTER NGA
091A DEVANAGARI LETTER CA
091B DEVANAGARI LETTER CHA
091C DEVANAGARI LETTER JA
091D DEVANAGARI LETTER JHA
091E DEVANAGARI LETTER NYA
091F DEVANAGARI LETTER TTA
0920 DEVANAGARI LETTER TTHA
0921 DEVANAGARI LETTER DDA
0922 DEVANAGARI LETTER DDHA
0923 DEVANAGARI LETTER NNA
0924 DEVANAGARI LETTER TA
0925 DEVANAGARI LETTER THA
0926 DEVANAGARI LETTER DA
0927 DEVANAGARI LETTER DHA
0928 DEVANAGARI LETTER NA
0929 DEVANAGARI LETTER NNNA
092A DEVANAGARI LETTER PA
092B DEVANAGARI LETTER PHA
092C DEVANAGARI LETTER BA
092D DEVANAGARI LETTER BHA
092E DEVANAGARI LETTER MA
092F DEVANAGARI LETTER YA
0930 DEVANAGARI LETTER RA
0931 DEVANAGARI LETTER RRA
0932 DEVANAGARI LETTER LA
0933 DEVANAGARI LETTER LLA
0934 DEVANAGARI LETTER LLLA
0935 DEVANAGARI LETTER VA
0936 DEVANAGARI LETTER SHA
0937 DEVANAGARI LETTER SSA
0938 DEVANAGARI LETTER SA
0939 DEVANAGARI LETTER HA
093A DEVANAGARI VOWEL SIGN OE
093B DEVANAGARI VOWEL SIGN OOE
093C DEVANAGARI SIGN NUKTA
093D DEVANAGARI SIGN AVAGRAHA
093E DEVANAGARI VOWEL SIGN AA
093F DEVANAGARI VOWEL SIGN I
0940 DEVANAGARI VOWEL SIGN II
0941 DEVANAGARI VOWEL SIGN U
0942 DEVANAGARI VOWEL SIGN UU
0943 DEVANAGARI VOWEL SIGN VOCALIC R
0944 DEVANAGARI VOWEL SIGN VOCALIC RR
0945 DEVANAGARI VOWEL SIGN CANDRA E
0946 DEVANAGARI VOWEL SIGN SHORT E
0947 DEVANAGARI VOWEL SIGN E
0948 DEVANAGARI VOWEL SIGN AI
0949 DEVANAGARI VOWEL SIGN CANDRA O
094A DEVANAGARI VOWEL SIGN SHORT O
094B DEVANAGARI VOWEL SIGN O
094C DEVANAGARI VOWEL SIGN AU
094D DEVANAGARI SIGN VIRAMA
094E DEVANAGARI VOWEL SIGN PRISHTHAMATRA E
094F DEVANAGARI VOWEL SIGN AW
0950 DEVANAGARI OM
0951 DEVANAGARI STRESS SIGN UDATTA
0952 DEVANAGARI STRESS SIGN ANUDATTA
0953 DEVANAGARI GRAVE ACCENT
0954 DEVANAGARI ACUTE ACCENT
0955 DEVANAGARI VOWEL SIGN CANDRA LONG E
0956 DEVANAGARI VOWEL SIGN UE
0957 DEVANAGARI VOWEL SIGN UUE
0958 DEVANAGARI LETTER QA
0959 DEVANAGARI LETTER KHHA
095A DEVANAGARI LETTER GHHA
095B DEVANAGARI LETTER ZA
095C DEVANAGARI LETTER DDDHA
095D DEVANAGARI LETTER RHA
095E DEVANAGARI LETTER FA
095F DEVANAGARI LETTER YYA
0960 DEVANAGARI LETTER VOCALIC RR
0961 DEVANAGARI LETTER VOCALIC LL
0962 DEVANAGARI VOWEL SIGN VOCALIC L
0963 DEVANAGARI VOWEL SIGN VOCALIC LL
0964 DEVANAGARI DANDA
0965 DEVANAGARI DOUBLE DANDA
0966 DEVANAGARI DIGIT ZERO
0967 DEVANAGARI DIGIT ONE
0968 DEVANAGARI DIGIT TWO
0969 DEVANAGARI DIGIT THREE
096A DEVANAGARI DIGIT FOUR
096B DEVANAGARI DIGIT FIVE
096C DEVANAGARI DIGIT SIX
096D DEVANAGARI DIGIT SEVEN
096E DEVANAGARI DIGIT EIGHT
096F DEVANAGARI DIGIT NINE
0970 DEVANAGARI ABBREVIATION SIGN
0971 DEVANAGARI SIGN HIGH SPACING DOT
0972 DEVANAGARI LETTER CANDRA A
0973 DEVANAGARI LETTER OE
0974 DEVANAGARI LETTER OOE
0975 DEVANAGARI LETTER AW
0976 DEVANAGARI LETTER UE
0977 DEVANAGARI LETTER UUE
0978 DEVANAGARI LETTER MARWARI DDA
0979 DEVANAGARI LETTER ZHA
097A DEVANAGARI LETTER HEAVY YA
097B DEVANAGARI LETTER GGA
097C DEVANAGARI LETTER JJA
097D DEVANAGARI LETTER GLOTTAL STOP
097E DEVANAGARI LETTER DDDA
097F DEVANAGARI LETTER BBA
0980 BENGALI ANJI
0981 BENGALI SIGN CANDRABINDU
0982 BENGALI SIGN ANUSVARA
0983 BENGALI SIGN VISARGA
0985 BENGALI LETTER A
0986 BENGALI LETTER AA
0987 BENGALI LETTER I
0988 BENGALI LETTER II
0989 BENGALI LETTER U
098A BENGALI LETTER UU
098B BENGALI LETTER VOCALIC R
098C BENGALI LETTER VOCALIC L
098F BENGALI LETTER E
0990 BENGALI LETTER AI
0993 BENGALI LETTER O
0994 BENGALI LETTER AU
0995 BENGALI LETTER KA
0996 BENGALI LETTER KHA
0997 BENGALI LETTER GA
0998 BENGALI LETTER GHA
0999 BENGALI LETTER NGA
099A BENGALI LETTER CA
099B BENGALI LETTER CHA
099C BENGALI LETTER JA
099D BENGALI LETTER JHA
099E BENGALI LETTER NYA
099F BENGALI LETTER TTA
09A0 BENGALI LETTER TTHA
09A1 BENGALI LETTER DDA
09A2 BENGALI LETTER DDHA
09A3 BENGALI LETTER NNA
09A4 BENGALI LETTER TA
09A5 BENGALI LETTER THA
09A6 BENGALI LETTER DA
09A7 BENGALI LETTER DHA
09A8 BENGALI LETTER NA
09AA BENGALI LETTER PA
09AB BENGALI LETTER PHA
09AC BENGALI LETTER BA
09AD BENGALI LETTER BHA
09AE BENGALI LETTER MA
09AF BENGALI LETTER YA
09B0 BENGALI LETTER RA
09B2 BENGALI LETTER LA
09B6 BENGALI LETTER SHA
09B7 BENGALI LETTER SSA
09B8 BENGALI LETTER SA
09B9 BENGALI LETTER HA
09BC BENGALI SIGN NUKTA
09BD BENGALI SIGN AVAGRAHA
09BE BENGALI VOWEL SIGN AA
09BF BENGALI VOWEL SIGN I
09C0 BENGALI VOWEL SIGN II
09C1 BENGALI VOWEL SIGN U
09C2 BENGALI VOWEL SIGN UU
09C3 BENGALI VOWEL SIGN VOCALIC R
09C4 BENGALI VOWEL SIGN VOCALIC RR
09C7 BENGALI VOWEL SIGN E
09C8 BENGALI VOWEL SIGN AI
09CB BENGALI VOWEL SIGN O
09CC BENGALI VOWEL SIGN AU
09CD BENGALI SIGN VIRAMA
09CE BENGALI LETTER KHANDA TA
09D7 BENGALI AU LENGTH MARK
09DC BENGALI LETTER RRA
09DD BENGALI LETTER RHA
09DF BENGALI LETTER YYA
09E0 BENGALI LETTER VOCALIC RR
09E1 BENGALI LETTER VOCALIC LL
09E2 BENGALI VOWEL SIGN VOCALIC L
09E3 BENGALI VOWEL SIGN VOCALIC LL
09E6 BENGALI DIGIT ZERO
09E7 BENGALI DIGIT ONE
09E8 BENGALI DIGIT TWO
09E9 BENGALI DIGIT THREE
09EA BENGALI DIGIT FOUR
09EB BENGALI DIGIT FIVE
09EC BENGALI DIGIT SIX
09ED BENGALI DIGIT SEVEN
09EE BENGALI DIGIT EIGHT
09EF BENGALI DIGIT NINE
09F0 BENGALI LETTER RA WITH MIDDLE DIAGONAL
09F1 BENGALI LETTER RA WITH LOWER DIAGONAL
09F2 BENGALI RUPEE MARK
09F3 BENGALI RUPEE SIGN
09F4 BENGALI CURRENCY NUMERATOR ONE
09F5 BENGALI CURRENCY NUMERATOR TWO
09F6 BENGALI CURRENCY NUMERATOR THREE
09F7 BENGALI CURRENCY NUMERATOR FOUR
09F8 BENGALI CURRENCY NUMERATOR ONE LESS THAN THE DENOMINATOR
09F9 BENGALI CURRENCY DENOMINATOR SIXTEEN
09FA BENGALI ISSHAR
09FB BENGALI GANDA MARK
09FC BENGALI LETTER VEDIC ANUSVARA
09FD BENGALI ABBREVIATION SIGN
09FE BENGALI SANDHI MARK
0A01 GURMUKHI SIGN ADAK BINDI
0A02 GURMUKHI SIGN BINDI
0A03 GURMUKHI SIGN VISARGA
0A05 GURMUKHI LETTER A
0A06 GURMUKHI LETTER AA
0A07 GURMUKHI LETTER I
0A08 GURMUKHI LETTER II
0A09 GURMUKHI LETTER U
0A0A GURMUKHI LETTER UU
0A0F GURMUKHI LETTER EE
0A10 GURMUKHI LETTER AI
0A13 GURMUKHI LETTER OO
0A14 GURMUKHI LETTER AU
0A15 GURMUKHI LETTER KA
0A16 GURMUKHI LETTER KHA
0A17 GURMUKHI LETTER GA
0A18 GURMUKHI LETTER GHA
0A19 GURMUKHI LETTER NGA
0A1A GURMUKHI LETTER CA
0A1B GURMUKHI LETTER CHA
0A1C GURMUKHI LETTER JA
0A1D GURMUKHI LETTER JHA
0A1E GURMUKHI LETTER NYA
0A1F GURMUKHI LETTER TTA
0A20 GURMUKHI LETTER TTHA
0A21 GURMUKHI LETTER DDA
0A22 GURMUKHI LETTER DDHA
0A23 GURMUKHI LETTER NNA
0A24 GURMUKHI LETTER TA
0A25 GURMUKHI LETTER THA
0A26 GURMUKHI LETTER DA
0A27 GURMUKHI LETTER DHA
0A28 GURMUKHI LETTER NA
0A2A GURMUKHI LETTER PA
0A2B GURMUKHI LETTER PHA
0A2C GURMUKHI LETTER BA
0A2D GURMUKHI LETTER BHA
0A2E GURMUKHI LETTER MA
0A2F GURMUKHI LETTER YA
0A30 GURMUKHI LETTER RA
0A32 GURMUKHI LETTER LA
0A33 GURMUKHI LETTER LLA
0A35 GURMUKHI LETTER VA
0A36 GURMUKHI LETTER SHA
0A38 GURMUKHI LETTER SA
0A39 GURMUKHI LETTER HA
0A3C GURMUKHI SIGN NUKTA
0A3E GURMUKHI VOWEL SIGN AA
0A3F GURMUKHI VOWEL SIGN I
0A40 GURMUKHI VOWEL SIGN II
0A41 GURMUKHI VOWEL SIGN U
0A42 GURMUKHI VOWEL SIGN UU
0A47 GURMUKHI VOWEL SIGN EE
0A48 GURMUKHI VOWEL SIGN AI
0A4B GURMUKHI VOWEL SIGN OO
0A4C GURMUKHI VOWEL SIGN AU
0A4D GURMUKHI SIGN VIRAMA
0A51 GURMUKHI SIGN UDAAT
0A59 GURMUKHI LETTER KHHA
0A5A GURMUKHI LETTER GHHA
0A5B GURMUKHI LETTER ZA
0A5C GURMUKHI LETTER RRA
0A5E GURMUKHI LETTER FA
0A66 GURMUKHI DIGIT ZERO
0A67 GURMUKHI DIGIT ONE
0A68 GURMUKHI DIGIT TWO
0A69 GURMUKHI DIGIT THREE
0A6A GURMUKHI DIGIT FOUR
0A6B GURMUKHI DIGIT FIVE
0A6C GURMUKHI DIGIT SIX
0A6D GURMUKHI DIGIT SEVEN
0A6E GURMUKHI DIGIT EIGHT
0A6F GURMUKHI DIGIT NINE
0A70 GURMUKHI TIPPI
0A71 GURMUKHI ADDAK
0A72 GURMUKHI IRI
0A73 GURMUKHI URA
0A74 GURMUKHI EK ONKAR
0A75 GURMUKHI SIGN YAKASH
0A76 GURMUKHI ABBREVIATION SIGN
0A81 GUJARATI SIGN CANDRABINDU
0A82 GUJARATI SIGN ANUSVARA
0A83 GUJARATI SIGN VISARGA
0A85 GUJARATI LETTER A
0A86 GUJARATI LETTER AA
0A87 GUJARATI LETTER I
0A88 GUJARATI LETTER II
0A89 GUJARATI LETTER U
0A8A GUJARATI LETTER UU
0A8B GUJARATI LETTER VOCALIC R
0A8C GUJARATI LETTER VOCALIC L
0A8D GUJARATI VOWEL CANDRA E
0A8F GUJARATI LETTER E
0A90 GUJARATI LETTER AI
0A91 GUJARATI VOWEL CANDRA O
0A93 GUJARATI LETTER O
0A94 GUJARATI LETTER AU
0A95 GUJARATI LETTER KA
0A96 GUJARATI LETTER KHA
0A97 GUJARATI LETTER GA
0A98 GUJARATI LETTER GHA
0A99 GUJARATI LETTER NGA
0A9A GUJARATI LETTER CA
0A9B GUJARATI LETTER CHA
0A9C GUJARATI LETTER JA
0A9D GUJARATI LETTER JHA
0A9E GUJARATI LETTER NYA
0A9F GUJARATI LETTER TTA
0AA0 GUJARATI LETTER TTHA
0AA1 GUJARATI LETTER DDA
0AA2 GUJARATI LETTER DDHA
0AA3 GUJARATI LETTER NNA
0AA4 GUJARATI LETTER TA
0AA5 GUJARATI LETTER THA
0AA6 GUJARATI LETTER DA
0AA7 GUJARATI LETTER DHA
0AA8 GUJARATI LETTER NA
0AAA GUJARATI LETTER PA
0AAB GUJARATI LETTER PHA
0AAC GUJARATI LETTER BA
0AAD GUJARATI LETTER BHA
0AAE GUJARATI LETTER MA
0AAF GUJARATI LETTER YA
0AB0 GUJARATI LETTER RA
0AB2 GUJARATI LETTER LA
0AB3 GUJARATI LETTER LLA
0AB5 GUJARATI LETTER VA
0AB6 GUJARATI LETTER SHA
0AB7 GUJARATI LETTER SSA
0AB8 GUJARATI LETTER SA
0AB9 GUJARATI LETTER HA
0ABC GUJARATI SIGN NUKTA
0ABD GUJARATI SIGN AVAGRAHA
0ABE GUJARATI VOWEL SIGN AA
0ABF GUJARATI VOWEL SIGN I
0AC0 GUJARATI VOWEL SIGN II
0AC1 GUJARATI VOWEL SIGN U
0AC2 GUJARATI VOWEL SIGN UU
0AC3 GUJARATI VOWEL SIGN VOCALIC R
0AC4 GUJARATI VOWEL SIGN VOCALIC RR
0AC5 GUJARATI VOWEL SIGN CANDRA E
0AC7 GUJARATI VOWEL SIGN E
0AC8 GUJARATI VOWEL SIGN AI
0AC9 GUJARATI VOWEL SIGN CANDRA O
0ACB GUJARATI VOWEL SIGN O
0ACC GUJARATI VOWEL SIGN AU
0ACD GUJARATI SIGN VIRAMA
0AD0 GUJARATI OM
0AE0 GUJARATI LETTER VOCALIC RR
0AE1 GUJARATI LETTER VOCALIC LL
0AE2 GUJARATI VOWEL SIGN VOCALIC L
0AE3 GUJARATI VOWEL SIGN VOCALIC LL
0AE6 GUJARATI DIGIT ZERO
0AE7 GUJARATI DIGIT ONE
0AE8 GUJARATI DIGIT TWO
0AE9 GUJARATI DIGIT THREE
0AEA GUJARATI DIGIT FOUR
0AEB GUJARATI DIGIT FIVE
0AEC GUJARATI DIGIT SIX
0AED GUJARATI DIGIT SEVEN
0AEE GUJARATI DIGIT EIGHT
0AEF GUJARATI DIGIT NINE
0AF0 GUJARATI ABBREVIATION SIGN
0AF1 GUJARATI RUPEE SIGN
0AF9 GUJARATI LETTER ZHA
0AFA GUJARATI SIGN SUKUN
0AFB GUJARATI SIGN SHADDA
0AFC GUJARATI SIGN MADDAH
0AFD GUJARATI SIGN THREE-DOT NUKTA ABOVE
0AFE GUJARATI SIGN CIRCLE NUKTA ABOVE
0AFF GUJARATI SIGN TWO-CIRCLE NUKTA ABOVE
0B01 ORIYA SIGN CANDRABINDU
0B02 ORIYA SIGN ANUSVARA
0B03 ORIYA SIGN VISARGA
0B05 ORIYA LETTER A
0B06 ORIYA LETTER AA
0B07 ORIYA LETTER I
0B08 ORIYA LETTER II
0B09 ORIYA LETTER U
0B0A ORIYA LETTER UU
0B0B ORIYA LETTER VOCALIC R
0B0C ORIYA LETTER VOCALIC L
0B0F ORIYA LETTER E
0B10 ORIYA LETTER AI
0B13 ORIYA LETTER O
0B14 ORIYA LETTER AU
0B15 ORIYA LETTER KA
0B16 ORIYA LETTER KHA
0B17 ORIYA LETTER GA
0B18 ORIYA LETTER GHA
0B19 ORIYA LETTER NGA
0B1A ORIYA LETTER CA
0B1B ORIYA LETTER CHA
0B1C ORIYA LETTER JA
0B1D ORIYA LETTER JHA
0B1E ORIYA LETTER NYA
0B1F ORIYA LETTER TTA
0B20 ORIYA LETTER TTHA
0B21 ORIYA LETTER DDA
0B22 ORIYA LETTER DDHA
0B23 ORIYA LETTER NNA
0B24 ORIYA LETTER TA
0B25 ORIYA LETTER THA
0B26 ORIYA LETTER DA
0B27 ORIYA LETTER DHA
0B28 ORIYA LETTER NA
0B2A ORIYA LETTER PA
0B2B ORIYA LETTER PHA
0B2C ORIYA LETTER BA
0B2D ORIYA LETTER BHA
0B2E ORIYA LETTER MA
0B2F ORIYA LETTER YA
0B30 ORIYA LETTER RA
0B32 ORIYA LETTER LA
0B33 ORIYA LETTER LLA
0B35 ORIYA LETTER VA
0B36 ORIYA LETTER SHA
0B37 ORIYA LETTER SSA
0B38 ORIYA LETTER SA
0B39 ORIYA LETTER HA
0B3C ORIYA SIGN NUKTA
0B3D ORIYA SIGN AVAGRAHA
0B3E ORIYA VOWEL SIGN AA
0B3F ORIYA VOWEL SIGN I
0B40 ORIYA VOWEL SIGN II
0B41 ORIYA VOWEL SIGN U
0B42 ORIYA VOWEL SIGN UU
0B43 ORIYA VOWEL SIGN VOCALIC R
0B44 ORIYA VOWEL SIGN VOCALIC RR
0B47 ORIYA VOWEL SIGN E
0B48 ORIYA VOWEL SIGN AI
0B4B ORIYA VOWEL SIGN O
0B4C ORIYA VOWEL SIGN AU
0B4D ORIYA SIGN VIRAMA
0B55 ORIYA SIGN OVERLINE
0B56 ORIYA AI LENGTH MARK
0B57 ORIYA AU LENGTH MARK
0B5C ORIYA LETTER RRA
0B5D ORIYA LETTER RHA
0B5F ORIYA LETTER YYA
0B60 ORIYA LETTER VOCALIC RR
0B61 ORIYA LETTER VOCALIC LL
0B62 ORIYA VOWEL SIGN VOCALIC L
0B63 ORIYA VOWEL SIGN VOCALIC LL
0B66 ORIYA DIGIT ZERO
0B67 ORIYA DIGIT ONE
0B68 ORIYA DIGIT TWO
0B69 ORIYA DIGIT THREE
0B6A ORIYA DIGIT FOUR
0B6B ORIYA DIGIT FIVE
0B6C ORIYA DIGIT SIX
0B6D ORIYA DIGIT SEVEN
0B6E ORIYA DIGIT EIGHT
0B6F ORIYA DIGIT NINE
0B70 ORIYA ISSHAR
0B71 ORIYA LETTER WA
0B72 ORIYA FRACTION ONE QUARTER
0B73 ORIYA FRACTION ONE HALF
0B74 ORIYA FRACTION THREE QUARTERS
0B75 ORIYA FRACTION ONE SIXTEENTH
0B76 ORIYA FRACTION ONE EIGHTH
0B77 ORIYA FRACTION THREE SIXTEENTHS
0B82 TAMIL SIGN ANUSVARA
0B83 TAMIL SIGN VISARGA
0B85 TAMIL LETTER A
0B86 TAMIL LETTER AA
0B87 TAMIL LETTER I
0B88 TAMIL LETTER II
0B89 TAMIL LETTER U
0B8A TAMIL LETTER UU
0B8E TAMIL LETTER E
0B8F TAMIL LETTER EE
0B90 TAMIL LETTER AI
0B92 TAMIL LETTER O
0B93 TAMIL LETTER OO
0B94 TAMIL LETTER AU
0B95 TAMIL LETTER KA
0B99 TAMIL LETTER NGA
0B9A TAMIL LETTER CA
0B9C TAMIL LETTER JA
0B9E TAMIL LETTER NYA
0B9F TAMIL LETTER TTA
0BA3 TAMIL LETTER NNA
0BA4 TAMIL LETTER TA
0BA8 TAMIL LETTER NA
0BA9 TAMIL LETTER NNNA
0BAA TAMIL LETTER PA
0BAE TAMIL LETTER MA
0BAF TAMIL LETTER YA
0BB0 TAMIL LETTER RA
0BB1 TAMIL LETTER RRA
0BB2 TAMIL LETTER LA
0BB3 TAMIL LETTER LLA
0BB4 TAMIL LETTER LLLA
0BB5 TAMIL LETTER VA
0BB6 TAMIL LETTER SHA
0BB7 TAMIL LETTER SSA
0BB8 TAMIL LETTER SA
0BB9 TAMIL LETTER HA
0BBE TAMIL VOWEL SIGN AA
0BBF TAMIL VOWEL SIGN I
0BC0 TAMIL VOWEL SIGN II
0BC1 TAMIL VOWEL SIGN U
0BC2 TAMIL VOWEL SIGN UU
0BC6 TAMIL VOWEL SIGN E
0BC7 TAMIL VOWEL SIGN EE
0BC8 TAMIL VOWEL SIGN AI
0BCA TAMIL VOWEL SIGN O
0BCB TAMIL VOWEL SIGN OO
0BCC TAMIL VOWEL SIGN AU
0BCD TAMIL SIGN VIRAMA
0BD0 TAMIL OM
0BD7 TAMIL AU LENGTH MARK
0BE6 TAMIL DIGIT ZERO
0BE7 TAMIL DIGIT ONE
0BE8 TAMIL DIGIT TWO
0BE9 TAMIL DIGIT THREE
0BEA TAMIL DIGIT FOUR
0BEB TAMIL DIGIT FIVE
0BEC TAMIL DIGIT SIX
0BED TAMIL DIGIT SEVEN
0BEE TAMIL DIGIT EIGHT
0BEF TAMIL DIGIT NINE
0BF0 TAMIL NUMBER TEN
0BF1 TAMIL NUMBER ONE HUNDRED
0BF2 TAMIL NUMBER ONE THOUSAND
0BF3 TAMIL DAY SIGN
0BF4 TAMIL MONTH SIGN
0BF5 TAMIL YEAR SIGN
0BF6 TAMIL DEBIT SIGN
0BF7 TAMIL CREDIT SIGN
0BF8 TAMIL AS ABOVE SIGN
0BF9 TAMIL RUPEE SIGN
0BFA TAMIL NUMBER SIGN
0C00 TELUGU SIGN COMBINING CANDRABINDU ABOVE
0C01 TELUGU SIGN CANDRABINDU
0C02 TELUGU SIGN ANUSVARA
0C03 TELUGU SIGN VISARGA
0C04 TELUGU SIGN COMBINING ANUSVARA ABOVE
0C05 TELUGU LETTER A
0C06 TELUGU LETTER AA
0C07 TELUGU LETTER I
0C08 TELUGU LETTER II
0C09 TELUGU LETTER U
0C0A TELUGU LETTER UU
0C0B TELUGU LETTER VOCALIC R
0C0C TELUGU LETTER VOCALIC L
0C0E TELUGU LETTER E
0C0F TELUGU LETTER EE
0C10 TELUGU LETTER AI
0C12 TELUGU LETTER O
0C13 TELUGU LETTER OO
0C14 TELUGU LETTER AU
0C15 TELUGU LETTER KA
0C16 TELUGU LETTER KHA
0C17 TELUGU LETTER GA
0C18 TELUGU LETTER GHA
0C19 TELUGU LETTER NGA
0C1A TELUGU LETTER CA
0C1B TELUGU LETTER CHA
0C1C TELUGU LETTER JA
0C1D TELUGU LETTER JHA
0C1E TELUGU LETTER NYA
0C1F TELUGU LETTER TTA
0C20 TELUGU LETTER TTHA
0C21 TELUGU LETTER DDA
0C22 TELUGU LETTER DDHA
0C23 TELUGU LETTER NNA
0C24 TELUGU LETTER TA
0C25 TELUGU LETTER THA
0C26 TELUGU LETTER DA
0C27 TELUGU LETTER DHA
0C28 TELUGU LETTER NA
0C2A TELUGU LETTER PA
0C2B TELUGU LETTER PHA
0C2C TELUGU LETTER BA
0C2D TELUGU LETTER BHA
0C2E TELUGU LETTER MA
0C2F TELUGU LETTER YA
0C30 TELUGU LETTER RA
0C31 TELUGU LETTER RRA
0C32 TELUGU LETTER LA
0C33 TELUGU LETTER LLA
0C34 TELUGU LETTER LLLA
0C35 TELUGU LETTER VA
0C36 TELUGU LETTER SHA
0C37 TELUGU LETTER SSA
0C38 TELUGU LETTER SA
0C39 TELUGU LETTER HA
0C3C TELUGU SIGN NUKTA
0C3D TELUGU SIGN AVAGRAHA
0C3E TELUGU VOWEL SIGN AA
0C3F TELUGU VOWEL SIGN I
0C40 TELUGU VOWEL SIGN II
0C41 TELUGU VOWEL SIGN U
0C42 TELUGU VOWEL SIGN UU
0C43 TELUGU VOWEL SIGN VOCALIC R
0C44 TELUGU VOWEL SIGN VOCALIC RR
0C46 TELUGU VOWEL SIGN E
0C47 TELUGU VOWEL SIGN EE
0C48 TELUGU VOWEL SIGN AI
0C4A TELUGU VOWEL SIGN O
0C4B TELUGU VOWEL SIGN OO
0C4C TELUGU VOWEL SIGN AU
0C4D TELUGU SIGN VIRAMA
0C55 TELUGU LENGTH MARK
0C56 TELUGU AI LENGTH MARK
0C58 TELUGU LETTER TSA
0C59 TELUGU LETTER DZA
0C5A TELUGU LETTER RRRA
0C5D TELUGU LETTER NAKAARA POLLU
0C60 TELUGU LETTER VOCALIC RR
0C61 TELUGU LETTER VOCALIC LL
0C62 TELUGU VOWEL SIGN VOCALIC L
0C63 TELUGU VOWEL SIGN VOCALIC LL
0C66 TELUGU DIGIT ZERO
0C67 TELUGU DIGIT ONE
0C68 TELUGU DIGIT TWO
0C69 TELUGU DIGIT THREE
0C6A TELUGU DIGIT FOUR
0C6B TELUGU DIGIT FIVE
0C6C TELUGU DIGIT SIX
0C6D TELUGU DIGIT SEVEN
0C6E TELUGU DIGIT EIGHT
0C6F TELUGU DIGIT NINE
0C77 TELUGU SIGN SIDDHAM
0C78 TELUGU FRACTION DIGIT ZERO FOR ODD POWERS OF FOUR
0C79 TELUGU FRACTION DIGIT ONE FOR ODD POWERS OF FOUR
0C7A TELUGU FRACTION DIGIT TWO FOR ODD POWERS OF FOUR
0C7B TELUGU FRACTION DIGIT THREE FOR ODD POWERS OF FOUR
0C7C TELUGU FRACTION DIGIT ONE FOR EVEN POWERS OF FOUR
0C7D TELUGU FRACTION DIGIT TWO FOR EVEN POWERS OF FOUR
0C7E TELUGU FRACTION DIGIT THREE FOR EVEN POWERS OF FOUR
0C7F TELUGU SIGN TUUMU
0C80 KANNADA SIGN SPACING CANDRABINDU
0C81 KANNADA SIGN CANDRABINDU
0C82 KANNADA SIGN ANUSVARA
0C83 KANNADA SIGN VISARGA
0C84 KANNADA SIGN SIDDHAM
0C85 KANNADA LETTER A
0C86 KANNADA LETTER AA
0C87 KANNADA LETTER I
0C88 KANNADA LETTER II
0C89 KANNADA LETTER U
0C8A KANNADA LETTER UU
0C8B KANNADA LETTER VOCALIC R
0C8C KANNADA LETTER VOCALIC L
0C8E KANNADA LETTER E
0C8F KANNADA LETTER EE
0C90 KANNADA LETTER AI
0C92 KANNADA LETTER O
0C93 KANNADA LETTER OO
0C94 KANNADA LETTER AU
0C95 KANNADA LETTER KA
0C96 KANNADA LETTER KHA
0C97 KANNADA LETTER GA
0C98 KANNADA LETTER GHA
0C99 KANNADA LETTER NGA
0C9A KANNADA LETTER CA
0C9B KANNADA LETTER CHA
0C9C KANNADA LETTER JA
0C9D KANNADA LETTER JHA
0C9E KANNADA LETTER NYA
0C9F KANNADA LETTER TTA
0CA0 KANNADA LETTER TTHA
0CA1 KANNADA LETTER DDA
0CA2 KANNADA LETTER DDHA
0CA3 KANNADA LETTER NNA
0CA4 KANNADA LETTER TA
0CA5 KANNADA LETTER THA
0CA6 KANNADA LETTER DA
0CA7 KANNADA LETTER DHA
0CA8 KANNADA LETTER NA
0CAA KANNADA LETTER PA
0CAB KANNADA LETTER PHA
0CAC KANNADA LETTER BA
0CAD KANNADA LETTER BHA
0CAE KANNADA LETTER MA
0CAF KANNADA LETTER YA
0CB0 KANNADA LETTER RA
0CB1 KANNADA LETTER RRA
0CB2 KANNADA LETTER LA
0CB3 KANNADA LETTER LLA
0CB5 KANNADA LETTER VA
0CB6 KANNADA LETTER SHA
0CB7 KANNADA LETTER SSA
0CB8 KANNADA LETTER SA
0CB9 KANNADA LETTER HA
0CBC KANNADA SIGN NUKTA
0CBD KANNADA SIGN AVAGRAHA
0CBE KANNADA VOWEL SIGN AA
0CBF KANNADA VOWEL SIGN I
0CC0 KANNADA VOWEL SIGN II
0CC1 KANNADA VOWEL SIGN U
0CC2 KANNADA VOWEL SIGN UU
0CC3 KANNADA VOWEL SIGN VOCALIC R
0CC4 KANNADA VOWEL SIGN VOCALIC RR
0CC6 KANNADA VOWEL SIGN E
0CC7 KANNADA VOWEL SIGN EE
0CC8 KANNADA VOWEL SIGN AI
0CCA KANNADA VOWEL SIGN O
0CCB KANNADA VOWEL SIGN OO
0CCC KANNADA VOWEL SIGN AU
0CCD KANNADA SIGN VIRAMA
0CD5 KANNADA LENGTH MARK
0CD6 KANNADA AI LENGTH MARK
0CDD KANNADA LETTER NAKAARA POLLU
0CDE KANNADA LETTER FA
0CE0 KANNADA LETTER VOCALIC RR
0CE1 KANNADA LETTER VOCALIC LL
0CE2 KANNADA VOWEL SIGN VOCALIC L
0CE3 KANNADA VOWEL SIGN VOCALIC LL
0CE6 KANNADA DIGIT ZERO
0CE7 KANNADA DIGIT ONE
0CE8 KANNADA DIGIT TWO
0CE9 KANNADA DIGIT THREE
0CEA KANNADA DIGIT FOUR
0CEB KANNADA DIGIT FIVE
0CEC KANNADA DIGIT SIX
0CED KANNADA DIGIT SEVEN
0CEE KANNADA DIGIT EIGHT
0CEF KANNADA DIGIT NINE
0CF1 KANNADA SIGN JIHVAMULIYA
0CF2 KANNADA SIGN UPADHMANIYA
0D00 MALAYALAM SIGN COMBINING ANUSVARA ABOVE
0D01 MALAYALAM SIGN CANDRABINDU
0D02 MALAYALAM SIGN ANUSVARA
0D03 MALAYALAM SIGN VISARGA
0D04 MALAYALAM LETTER VEDIC ANUSVARA
0D05 MALAYALAM LETTER A
0D06 MALAYALAM LETTER AA
0D07 MALAYALAM LETTER I
0D08 MALAYALAM LETTER II
0D09 MALAYALAM LETTER U
0D0A MALAYALAM LETTER UU
0D0B MALAYALAM LETTER VOCALIC R
0D0C MALAYALAM LETTER VOCALIC L
0D0E MALAYALAM LETTER E
0D0F MALAYALAM LETTER EE
0D10 MALAYALAM LETTER AI
0D12 MALAYALAM LETTER O
0D13 MALAYALAM LETTER OO
0D14 MALAYALAM LETTER AU
0D15 MALAYALAM LETTER KA
0D16 MALAYALAM LETTER KHA
0D17 MALAYALAM LETTER GA
0D18 MALAYALAM LETTER GHA
0D19 MALAYALAM LETTER NGA
0D1A MALAYALAM LETTER CA
0D1B MALAYALAM LETTER CHA
0D1C MALAYALAM LETTER JA
0D1D MALAYALAM LETTER JHA
0D1E MALAYALAM LETTER NYA
0D1F MALAYALAM LETTER TTA
0D20 MALAYALAM LETTER TTHA
0D21 MALAYALAM LETTER DDA
0D22 MALAYALAM LETTER DDHA
0D23 MALAYALAM LETTER NNA
0D24 MALAYALAM LETTER TA
0D25 MALAYALAM LETTER THA
0D26 MALAYALAM LETTER DA
0D27 MALAYALAM LETTER DHA
0D28 MALAYALAM LETTER NA
0D29 MALAYALAM LETTER NNNA
0D2A MALAYALAM LETTER PA
0D2B MALAYALAM LETTER PHA
0D2C MALAYALAM LETTER BA
0D2D MALAYALAM LETTER BHA
0D2E MALAYALAM LETTER MA
0D2F MALAYALAM LETTER YA
0D30 MALAYALAM LETTER RA
0D31 MALAYALAM LETTER RRA
0D32 MALAYALAM LETTER LA
0D33 MALAYALAM LETTER LLA
0D34 MALAYALAM LETTER LLLA
0D35 MALAYALAM LETTER VA
0D36 MALAYALAM LETTER SHA
0D37 MALAYALAM LETTER SSA
0D38 MALAYALAM LETTER SA
0D39 MALAYALAM LETTER HA
0D3A MALAYALAM LETTER TTTA
0D3B MALAYALAM SIGN VERTICAL BAR VIRAMA
0D3C MALAYALAM SIGN CIRCULAR VIRAMA
0D3D MALAYALAM SIGN AVAGRAHA
0D3E MALAYALAM VOWEL SIGN AA
0D3F MALAYALAM VOWEL SIGN I
0D40 MALAYALAM VOWEL SIGN II
0D41 MALAYALAM VOWEL SIGN U
0D42 MALAYALAM VOWEL SIGN UU
0D43 MALAYALAM VOWEL SIGN VOCALIC R
0D44 MALAYALAM VOWEL SIGN VOCALIC RR
0D46 MALAYALAM VOWEL SIGN E
0D47 MALAYALAM VOWEL SIGN EE
0D48 MALAYALAM VOWEL SIGN AI
0D4A MALAYALAM VOWEL SIGN O
0D4B MALAYALAM VOWEL SIGN OO
0D4C MALAYALAM VOWEL SIGN AU
0D4D MALAYALAM SIGN VIRAMA
0D4E MALAYALAM LETTER DOT REPH
0D4F MALAYALAM SIGN PARA
0D54 MALAYALAM LETTER CHILLU M
0D55 MALAYALAM LETTER CHILLU Y
0D56 MALAYALAM LETTER CHILLU LLL
0D57 MALAYALAM AU LENGTH MARK
0D58 MALAYALAM FRACTION ONE ONE-HUNDRED-AND-SIXTIETH
0D59 MALAYALAM FRACTION ONE FORTIETH
0D5A MALAYALAM FRACTION THREE EIGHTIETHS
0D5B MALAYALAM FRACTION ONE TWENTIETH
0D5C MALAYALAM FRACTION ONE TENTH
0D5D MALAYALAM FRACTION THREE TWENTIETHS
0D5E MALAYALAM FRACTION ONE FIFTH
0D5F MALAYALAM LETTER ARCHAIC II
0D60 MALAYALAM LETTER VOCALIC RR
0D61 MALAYALAM LETTER VOCALIC LL
0D62 MALAYALAM VOWEL SIGN VOCALIC L
0D63 MALAYALAM VOWEL SIGN VOCALIC LL
0D66 MALAYALAM DIGIT ZERO
0D67 MALAYALAM DIGIT ONE
0D68 MALAYALAM DIGIT TWO
0D69 MALAYALAM DIGIT THREE
0D6A MALAYALAM DIGIT FOUR
0D6B MALAYALAM DIGIT FIVE
0D6C MALAYALAM DIGIT SIX
0D6D MALAYALAM DIGIT SEVEN
0D6E MALAYALAM DIGIT EIGHT
0D6F MALAYALAM DIGIT NINE
0D70 MALAYALAM NUMBER TEN
0D71 MALAYALAM NUMBER ONE HUNDRED
0D72 MALAYALAM NUMBER ONE THOUSAND
0D73 MALAYALAM FRACTION ONE QUARTER
0D74 MALAYALAM FRACTION ONE HALF
0D75 MALAYALAM FRACTION THREE QUARTERS
0D76 MALAYALAM FRACTION ONE SIXTEENTH
0D77 MALAYALAM FRACTION ONE EIGHTH
0D78 MALAYALAM FRACTION THREE SIXTEENTHS
0D79 MALAYALAM DATE MARK
0D7A MALAYALAM LETTER CHILLU NN
0D7B MALAYALAM LETTER CHILLU N
0D7C MALAYALAM LETTER CHILLU RR
0D7D MALAYALAM LETTER CHILLU L
0D7E MALAYALAM LETTER CHILLU LL
0D7F MALAYALAM LETTER CHILLU K
0D81 SINHALA SIGN CANDRABINDU
0D82 SINHALA SIGN ANUSVARAYA
0D83 SINHALA SIGN VISARGAYA
0D85 SINHALA LETTER AYANNA
0D86 SINHALA LETTER AAYANNA
0D87 SINHALA LETTER AEYANNA
0D88 SINHALA LETTER AEEYANNA
0D89 SINHALA LETTER IYANNA
0D8A SINHALA LETTER IIYANNA
0D8B SINHALA LETTER UYANNA
0D8C SINHALA LETTER UUYANNA
0D8D SINHALA LETTER IRUYANNA
0D8E SINHALA LETTER IRUUYANNA
0D8F SINHALA LETTER ILUYANNA
0D90 SINHALA LETTER ILUUYANNA
0D91 SINHALA LETTER EYANNA
0D92 SINHALA LETTER EEYANNA
0D93 SINHALA LETTER AIYANNA
0D94 SINHALA LETTER OYANNA
0D95 SINHALA LETTER OOYANNA
0D96 SINHALA LETTER AUYANNA
0D9A SINHALA LETTER ALPAPRAANA KAYANNA
0D9B SINHALA LETTER MAHAAPRAANA KAYANNA
0D9C SINHALA LETTER ALPAPRAANA GAYANNA
0D9D SINHALA LETTER MAHAAPRAANA GAYANNA
0D9E SINHALA LETTER KANTAJA NAASIKYAYA
0D9F SINHALA LETTER SANYAKA GAYANNA
0DA0 SINHALA LETTER ALPAPRAANA CAYANNA
0DA1 SINHALA LETTER MAHAAPRAANA CAYANNA
0DA2 SINHALA LETTER ALPAPRAANA JAYANNA
0DA3 SINHALA LETTER MAHAAPRAANA JAYANNA
0DA4 SINHALA LETTER TAALUJA NAASIKYAYA
0DA5 SINHALA LETTER TAALUJA SANYOOGA NAAKSIKYAYA
0DA6 SINHALA LETTER SANYAKA JAYANNA
0DA7 SINHALA LETTER ALPAPRAANA TTAYANNA
0DA8 SINHALA LETTER MAHAAPRAANA TTAYANNA
0DA9 SINHALA LETTER ALPAPRAANA DDAYANNA
0DAA SINHALA LETTER MAHAAPRAANA DDAYANNA
0DAB SINHALA LETTER MUURDHAJA NAYANNA
0DAC SINHALA LETTER SANYAKA DDAYANNA
0DAD SINHALA LETTER ALPAPRAANA TAYANNA
0DAE SINHALA LETTER MAHAAPRAANA TAYANNA
0DAF SINHALA LETTER ALPAPRAANA DAYANNA
0DB0 SINHALA LETTER MAHAAPRAANA DAYANNA
0DB1 SINHALA LETTER DANTAJA NAYANNA
0DB3 SINHALA LETTER SANYAKA DAYANNA
0DB4 SINHALA LETTER ALPAPRAANA PAYANNA
0DB5 SINHALA LETTER MAHAAPRAANA PAYANNA
0DB6 SINHALA LETTER ALPAPRAANA BAYANNA
0DB7 SINHALA LETTER MAHAAPRAANA BAYANNA
0DB8 SINHALA LETTER MAYANNA
0DB9 SINHALA LETTER AMBA BAYANNA
0DBA SINHALA LETTER YAYANNA
0DBB SINHALA LETTER RAYANNA
0DBD SINHALA LETTER DANTAJA LAYANNA
0DC0 SINHALA LETTER VAYANNA
0DC1 SINHALA LETTER TAALUJA SAYANNA
0DC2 SINHALA LETTER MUURDHAJA SAYANNA
0DC3 SINHALA LETTER DANTAJA SAYANNA
0DC4 SINHALA LETTER HAYANNA
0DC5 SINHALA LETTER MUURDHAJA LAYANNA
0DC6 SINHALA LETTER FAYANNA
0DCA SINHALA SIGN AL-LAKUNA
0DCF SINHALA VOWEL SIGN AELA-PILLA
0DD0 SINHALA VOWEL SIGN KETTI AEDA-PILLA
0DD1 SINHALA VOWEL SIGN DIGA AEDA-PILLA
0DD2 SINHALA VOWEL SIGN KETTI IS-PILLA
0DD3 SINHALA VOWEL SIGN DIGA IS-PILLA
0DD4 SINHALA VOWEL SIGN KETTI PAA-PILLA
0DD6 SINHALA VOWEL SIGN DIGA PAA-PILLA
0DD8 SINHALA VOWEL SIGN GAETTA-PILLA
0DD9 SINHALA VOWEL SIGN KOMBUVA
0DDA SINHALA VOWEL SIGN DIGA KOMBUVA
0DDB SINHALA VOWEL SIGN KOMBU DEKA
0DDC SINHALA VOWEL SIGN KOMBUVA HAA AELA-PILLA
0DDD SINHALA VOWEL SIGN KOMBUVA HAA DIGA AELA-PILLA
0DDE SINHALA VOWEL SIGN KOMBUVA HAA GAYANUKITTA
0DDF SINHALA VOWEL SIGN GAYANUKITTA
0DE6 SINHALA LITH DIGIT ZERO
0DE7 SINHALA LITH DIGIT ONE
0DE8 SINHALA LITH DIGIT TWO
0DE9 SINHALA LITH DIGIT THREE
0DEA SINHALA LITH DIGIT FOUR
0DEB SINHALA LITH DIGIT FIVE
0DEC SINHALA LITH DIGIT SIX
0DED SINHALA LITH DIGIT SEVEN
0DEE SINHALA LITH DIGIT EIGHT
0DEF SINHALA LITH DIGIT NINE
0DF2 SINHALA VOWEL SIGN DIGA GAETTA-PILLA
0DF3 SINHALA VOWEL SIGN DIGA GAYANUKITTA
0DF4 SINHALA PUNCTUATION KUNDDALIYA
0E01 THAI CHARACTER KO KAI
0E02 THAI CHARACTER KHO KHAI
0E03 THAI CHARACTER KHO KHUAT
0E04 THAI CHARACTER KHO KHWAI
0E05 THAI CHARACTER KHO KHON
0E06 THAI CHARACTER KHO RAKHANG
0E07 THAI CHARACTER NGO NGU
0E08 THAI CHARACTER CHO CHAN
0E09 THAI CHARACTER CHO CHING
0E0A THAI CHARACTER CHO CHANG
0E0B THAI CHARACTER SO SO
0E0C THAI CHARACTER CHO CHOE
0E0D THAI CHARACTER YO YING
0E0E THAI CHARACTER DO CHADA
0E0F THAI CHARACTER TO PATAK
0E10 THAI CHARACTER THO THAN
0E11 THAI CHARACTER THO NANGMONTHO
0E12 THAI CHARACTER THO PHUTHAO
0E13 THAI CHARACTER NO NEN
0E14 THAI CHARACTER DO DEK
0E15 THAI CHARACTER TO TAO
0E16 THAI CHARACTER THO THUNG
0E17 THAI CHARACTER THO THAHAN
0E18 THAI CHARACTER THO THONG
0E19 THAI CHARACTER NO NU
0E1A THAI CHARACTER BO BAIMAI
0E1B THAI CHARACTER PO PLA
0E1C THAI CHARACTER PHO PHUNG
0E1D THAI CHARACTER FO FA
0E1E THAI CHARACTER PHO PHAN
0E1F THAI CHARACTER FO FAN
0E20 THAI CHARACTER PHO SAMPHAO
0E21 THAI CHARACTER MO MA
0E22 THAI CHARACTER YO YAK
0E23 THAI CHARACTER RO RUA
0E24 THAI CHARACTER RU
0E25 THAI CHARACTER LO LING
0E26 THAI CHARACTER LU
0E27 THAI CHARACTER WO WAEN
0E28 THAI CHARACTER SO SALA
0E29 THAI CHARACTER SO RUSI
0E2A THAI CHARACTER SO SUA
0E2B THAI CHARACTER HO HIP
0E2C THAI CHARACTER LO CHULA
0E2D THAI CHARACTER O ANG
0E2E THAI CHARACTER HO NOKHUK
0E2F THAI CHARACTER PAIYANNOI
0E30 THAI CHARACTER SARA A
0E31 THAI CHARACTER MAI HAN-AKAT
0E32 THAI CHARACTER SARA AA
0E33 THAI CHARACTER SARA AM
0E34 THAI CHARACTER SARA I
0E35 THAI CHARACTER SARA II
0E36 THAI CHARACTER SARA UE
0E37 THAI CHARACTER SARA UEE
0E38 THAI CHARACTER SARA U
0E39 THAI CHARACTER SARA UU
0E3A THAI CHARACTER PHINTHU
0E3F THAI CURRENCY SYMBOL BAHT
0E40 THAI CHARACTER SARA E
0E41 THAI CHARACTER SARA AE
0E42 THAI CHARACTER SARA O
0E43 THAI CHARACTER SARA AI MAIMUAN
0E44 THAI CHARACTER SARA AI MAIMALAI
0E45 THAI CHARACTER LAKKHANGYAO
0E46 THAI CHARACTER MAIYAMOK
0E47 THAI CHARACTER MAITAIKHU
0E48 THAI CHARACTER MAI EK
0E49 THAI CHARACTER MAI THO
0E4A THAI CHARACTER MAI TRI
0E4B THAI CHARACTER MAI CHATTAWA
0E4C THAI CHARACTER THANTHAKHAT
0E4D THAI CHARACTER NIKHAHIT
0E4E THAI CHARACTER YAMAKKAN
0E4F THAI CHARACTER FONGMAN
0E50 THAI DIGIT ZERO
0E51 THAI DIGIT ONE
0E52 THAI DIGIT TWO
0E53 THAI DIGIT THREE
0E54 THAI DIGIT FOUR
0E55 THAI DIGIT FIVE
0E56 THAI DIGIT SIX
0E57 THAI DIGIT SEVEN
0E58 THAI DIGIT EIGHT
0E59 THAI DIGIT NINE
0E5A THAI CHARACTER ANGKHANKHU
0E5B THAI CHARACTER KHOMUT
0E81 LAO LETTER KO
0E82 LAO LETTER KHO SUNG
0E84 LAO LETTER KHO TAM
0E86 LAO LETTER PALI GHA
0E87 LAO LETTER NGO
0E88 LAO LETTER CO
0E89 LAO LETTER PALI CHA
0E8A LAO LETTER SO TAM
0E8C LAO LETTER PALI JHA
0E8D LAO LETTER NYO
0E8E LAO LETTER PALI NYA
0E8F LAO LETTER PALI TTA
0E90 LAO LETTER PALI TTHA
0E91 LAO LETTER PALI DDA
0E92 LAO LETTER PALI DDHA
0E93 LAO LETTER PALI NNA
0E94 LAO LETTER DO
0E95 LAO LETTER TO
0E96 LAO LETTER THO SUNG
0E97 LAO LETTER THO TAM
0E98 LAO LETTER PALI DHA
0E99 LAO LETTER NO
0E9A LAO LETTER BO
0E9B LAO LETTER PO
0E9C LAO LETTER PHO SUNG
0E9D LAO LETTER FO TAM
0E9E LAO LETTER PHO TAM
0E9F LAO LETTER FO SUNG
0EA0 LAO LETTER PALI BHA
0EA1 LAO LETTER MO
0EA2 LAO LETTER YO
0EA3 LAO LETTER LO LING
0EA5 LAO LETTER LO LOOT
0EA7 LAO LETTER WO
0EA8 LAO LETTER SANSKRIT SHA
0EA9 LAO LETTER SANSKRIT SSA
0EAA LAO LETTER SO SUNG
0EAB LAO LETTER HO SUNG
0EAC LAO LETTER PALI LLA
0EAD LAO LETTER O
0EAE LAO LETTER HO TAM
0EAF LAO ELLIPSIS
0EB0 LAO VOWEL SIGN A
0EB1 LAO VOWEL SIGN MAI KAN
0EB2 LAO VOWEL SIGN AA
0EB3 LAO VOWEL SIGN AM
0EB4 LAO VOWEL SIGN I
0EB5 LAO VOWEL SIGN II
0EB6 LAO VOWEL SIGN Y
0EB7 LAO VOWEL SIGN YY
0EB8 LAO VOWEL SIGN U
0EB9 LAO VOWEL SIGN UU
0EBA LAO SIGN PALI VIRAMA
0EBB LAO VOWEL SIGN MAI KON
0EBC LAO SEMIVOWEL SIGN LO
0EBD LAO SEMIVOWEL SIGN NYO
0EC0 LAO VOWEL SIGN E
0EC1 LAO VOWEL SIGN EI
0EC2 LAO VOWEL SIGN O
0EC3 LAO VOWEL SIGN AY
0EC4 LAO VOWEL SIGN AI
0EC6 LAO KO LA
0EC8 LAO TONE MAI EK
0EC9 LAO TONE MAI THO
0ECA LAO TONE MAI TI
0ECB LAO TONE MAI CATAWA
0ECC LAO CANCELLATION MARK
0ECD LAO NIGGAHITA
0ED0 LAO DIGIT ZERO
0ED1 LAO DIGIT ONE
0ED2 LAO DIGIT TWO
0ED3 LAO DIGIT THREE
0ED4 LAO DIGIT FOUR
0ED5 LAO DIGIT FIVE
0ED6 LAO DIGIT SIX
0ED7 LAO DIGIT SEVEN
0ED8 LAO DIGIT EIGHT
0ED9 LAO DIGIT NINE
0EDC LAO HO NO
0EDD LAO HO MO
0EDE LAO LETTER KHMU GO
0EDF LAO LETTER KHMU NYO
0F00 TIBETAN SYLLABLE OM
0F01 TIBETAN MARK GTER YIG MGO TRUNCATED A
0F02 TIBETAN MARK GTER YIG MGO -UM RNAM BCAD MA
0F03 TIBETAN MARK GTER YIG MGO -UM GTER TSHEG MA
0F04 TIBETAN MARK INITIAL YIG MGO MDUN MA
0F05 TIBETAN MARK CLOSING YIG MGO SGAB MA
0F06 TIBETAN MARK CARET YIG MGO PHUR SHAD MA
0F07 TIBETAN MARK YIG MGO TSHEG SHAD MA
0F08 TIBETAN MARK SBRUL SHAD
0F09 TIBETAN MARK BSKUR YIG MGO
0F0A TIBETAN MARK BKA- SHOG YIG MGO
0F0B TIBETAN MARK INTERSYLLABIC TSHEG
0F0C TIBETAN MARK DELIMITER TSHEG BSTAR
0F0D TIBETAN MARK SHAD
0F0E TIBETAN MARK NYIS SHAD
0F0F TIBETAN MARK TSHEG SHAD
0F10 TIBETAN MARK NYIS TSHEG SHAD
0F11 TIBETAN MARK RIN CHEN SPUNGS SHAD
0F12 TIBETAN MARK RGYA GRAM SHAD
0F13 TIBETAN MARK CARET -DZUD RTAGS ME LONG CAN
0F14 TIBETAN MARK GTER TSHEG
0F15 TIBETAN LOGOTYPE SIGN CHAD RTAGS
0F16 TIBETAN LOGOTYPE SIGN LHAG RTAGS
0F17 TIBETAN ASTROLOGICAL SIGN SGRA GCAN -CHAR RTAGS
0F18 TIBETAN ASTROLOGICAL SIGN -KHYUD PA
0F19 TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS
0F1A TIBETAN SIGN RDEL DKAR GCIG
0F1B TIBETAN SIGN RDEL DKAR GNYIS
0F1C TIBETAN SIGN RDEL DKAR GSUM
0F1D TIBETAN SIGN RDEL NAG GCIG
0F1E TIBETAN SIGN RDEL NAG GNYIS
0F1F TIBETAN SIGN RDEL DKAR RDEL NAG
0F20 TIBETAN DIGIT ZERO
0F21 TIBETAN DIGIT ONE
0F22 TIBETAN DIGIT TWO
0F23 TIBETAN DIGIT THREE
0F24 TIBETAN DIGIT FOUR
0F25 TIBETAN DIGIT FIVE
0F26 TIBETAN DIGIT SIX
0F27 TIBETAN DIGIT SEVEN
0F28 TIBETAN DIGIT EIGHT
0F29 TIBETAN DIGIT NINE
0F2A TIBETAN DIGIT HALF ONE
0F2B TIBETAN DIGIT HALF TWO
0F2C TIBETAN DIGIT HALF THREE
0F2D TIBETAN DIGIT HALF FOUR
0F2E TIBETAN DIGIT HALF FIVE
0F2F TIBETAN DIGIT HALF SIX
0F30 TIBETAN DIGIT HALF SEVEN
0F31 TIBETAN DIGIT HALF EIGHT
0F32 TIBETAN DIGIT HALF NINE
0F33 TIBETAN DIGIT HALF ZERO
0F34 TIBETAN MARK BSDUS RTAGS
0F35 TIBETAN MARK NGAS BZUNG NYI ZLA
0F36 TIBETAN MARK CARET -DZUD RTAGS BZHI MIG CAN
0F37 TIBETAN MARK NGAS BZUNG SGOR RTAGS
0F38 TIBETAN MARK CHE MGO
0F39 TIBETAN MARK TSA -PHRU
0F3A TIBETAN MARK GUG RTAGS GYON
0F3B TIBETAN MARK GUG RTAGS GYAS
0F3C TIBETAN MARK ANG KHANG GYON
0F3D TIBETAN MARK ANG KHANG GYAS
0F3E TIBETAN SIGN YAR TSHES
0F3F TIBETAN SIGN MAR TSHES
0F40 TIBETAN LETTER KA
0F41 TIBETAN LETTER KHA
0F42 TIBETAN LETTER GA
0F43 TIBETAN LETTER GHA
0F44 TIBETAN LETTER NGA
0F45 TIBETAN LETTER CA
0F46 TIBETAN LETTER CHA
0F47 TIBETAN LETTER JA
0F49 TIBETAN LETTER NYA
0F4A TIBETAN LETTER TTA
0F4B TIBETAN LETTER TTHA
0F4C TIBETAN LETTER DDA
0F4D TIBETAN LETTER DDHA
0F4E TIBETAN LETTER NNA
0F4F TIBETAN LETTER TA
0F50 TIBETAN LETTER THA
0F51 TIBETAN LETTER DA
0F52 TIBETAN LETTER DHA
0F53 TIBETAN LETTER NA
0F54 TIBETAN LETTER PA
0F55 TIBETAN LETTER PHA
0F56 TIBETAN LETTER BA
0F57 TIBETAN LETTER BHA
0F58 TIBETAN LETTER MA
0F59 TIBETAN LETTER TSA
0F5A TIBETAN LETTER TSHA
0F5B TIBETAN LETTER DZA
0F5C TIBETAN LETTER DZHA
0F5D TIBETAN LETTER WA
0F5E TIBETAN LETTER ZHA
0F5F TIBETAN LETTER ZA
0F60 TIBETAN LETTER -A
0F61 TIBETAN LETTER YA
0F62 TIBETAN LETTER RA
0F63 TIBETAN LETTER LA
0F64 TIBETAN LETTER SHA
0F65 TIBETAN LETTER SSA
0F66 TIBETAN LETTER SA
0F67 TIBETAN LETTER HA
0F68 TIBETAN LETTER A
0F69 TIBETAN LETTER KSSA
0F6A TIBETAN LETTER FIXED-FORM RA
0F6B TIBETAN LETTER KKA
0F6C TIBETAN LETTER RRA
0F71 TIBETAN VOWEL SIGN AA
0F72 TIBETAN VOWEL SIGN I
0F73 TIBETAN VOWEL SIGN II
0F74 TIBETAN VOWEL SIGN U
0F75 TIBETAN VOWEL SIGN UU
0F76 TIBETAN VOWEL SIGN VOCALIC R
0F77 TIBETAN VOWEL SIGN VOCALIC RR
0F78 TIBETAN VOWEL SIGN VOCALIC L
0F79 TIBETAN VOWEL SIGN VOCALIC LL
0F7A TIBETAN VOWEL SIGN E
0F7B TIBETAN VOWEL SIGN EE
0F7C TIBETAN VOWEL SIGN O
0F7D TIBETAN VOWEL SIGN OO
0F7E TIBETAN SIGN RJES SU NGA RO
0F7F TIBETAN SIGN RNAM BCAD
0F80 TIBETAN VOWEL SIGN REVERSED I
0F81 TIBETAN VOWEL SIGN REVERSED II
0F82 TIBETAN SIGN NYI ZLA NAA DA
0F83 TIBETAN SIGN SNA LDAN
0F84 TIBETAN MARK HALANTA
0F85 TIBETAN MARK PALUTA
0F86 TIBETAN SIGN LCI RTAGS
0F87 TIBETAN SIGN YANG RTAGS
0F88 TIBETAN SIGN LCE TSA CAN
0F89 TIBETAN SIGN MCHU CAN
0F8A TIBETAN SIGN GRU CAN RGYINGS
0F8B TIBETAN SIGN GRU MED RGYINGS
0F8C TIBETAN SIGN INVERTED MCHU CAN
0F8D TIBETAN SUBJOINED SIGN LCE TSA CAN
0F8E TIBETAN SUBJOINED SIGN MCHU CAN
0F8F TIBETAN SUBJOINED SIGN INVERTED MCHU CAN
0F90 TIBETAN SUBJOINED LETTER KA
0F91 TIBETAN SUBJOINED LETTER KHA
0F92 TIBETAN SUBJOINED LETTER GA
0F93 TIBETAN SUBJOINED LETTER GHA
0F94 TIBETAN SUBJOINED LETTER NGA
0F95 TIBETAN SUBJOINED LETTER CA
0F96 TIBETAN SUBJOINED LETTER CHA
0F97 TIBETAN SUBJOINED LETTER JA
0F99 TIBETAN SUBJOINED LETTER NYA
0F9A TIBETAN SUBJOINED LETTER TTA
0F9B TIBETAN SUBJOINED LETTER TTHA
0F9C TIBETAN SUBJOINED LETTER DDA
0F9D TIBETAN SUBJOINED LETTER DDHA
0F9E TIBETAN SUBJOINED LETTER NNA
0F9F TIBETAN SUBJOINED LETTER TA
0FA0 TIBETAN SUBJOINED LETTER THA
0FA1 TIBETAN SUBJOINED LETTER DA
0FA2 TIBETAN SUBJOINED LETTER DHA
0FA3 TIBETAN SUBJOINED LETTER NA
0FA4 TIBETAN SUBJOINED LETTER PA
0FA5 TIBETAN SUBJOINED LETTER PHA
0FA6 TIBETAN SUBJOINED LETTER BA
0FA7 TIBETAN SUBJOINED LETTER BHA
0FA8 TIBETAN SUBJOINED LETTER MA
0FA9 TIBETAN SUBJOINED LETTER TSA
0FAA TIBETAN SUBJOINED LETTER TSHA
0FAB TIBETAN SUBJOINED LETTER DZA
0FAC TIBETAN SUBJOINED LETTER DZHA
0FAD TIBETAN SUBJOINED LETTER WA
0FAE TIBETAN SUBJOINED LETTER ZHA
0FAF TIBETAN SUBJOINED LETTER ZA
0FB0 TIBETAN SUBJOINED LETTER -A
0FB1 TIBETAN SUBJOINED LETTER YA
0FB2 TIBETAN SUBJOINED LETTER RA
0FB3 TIBETAN SUBJOINED LETTER LA
0FB4 TIBETAN SUBJOINED LETTER SHA
0FB5 TIBETAN SUBJOINED LETTER SSA
0FB6 TIBETAN SUBJOINED LETTER SA
0FB7 TIBETAN SUBJOINED LETTER HA
0FB8 TIBETAN SUBJOINED LETTER A
0FB9 TIBETAN SUBJOINED LETTER KSSA
0FBA TIBETAN SUBJOINED LETTER FIXED-FORM WA
0FBB TIBETAN SUBJOINED LETTER FIXED-FORM YA
0FBC TIBETAN SUBJOINED LETTER FIXED-FORM RA
0FBE TIBETAN KU RU KHA
0FBF TIBETAN KU RU KHA BZHI MIG CAN
0FC0 TIBETAN CANTILLATION SIGN HEAVY BEAT
0FC1 TIBETAN CANTILLATION SIGN LIGHT BEAT
0FC2 TIBETAN CANTILLATION SIGN CANG TE-U
0FC3 TIBETAN CANTILLATION SIGN SBUB -CHAL
0FC4 TIBETAN SYMBOL DRIL BU
0FC5 TIBETAN SYMBOL RDO RJE
0FC6 TIBETAN SYMBOL PADMA GDAN
0FC7 TIBETAN SYMBOL RDO RJE RGYA GRAM
0FC8 TIBETAN SYMBOL PHUR PA
0FC9 TIBETAN SYMBOL NOR BU
0FCA TIBETAN SYMBOL NOR BU NYIS -KHYIL
0FCB TIBETAN SYMBOL NOR BU GSUM -KHYIL
0FCC TIBETAN SYMBOL NOR BU BZHI -KHYIL
0FCE TIBETAN SIGN RDEL NAG RDEL DKAR
0FCF TIBETAN SIGN RDEL NAG GSUM
0FD0 TIBETAN MARK BSKA- SHOG GI MGO RGYAN
0FD1 TIBETAN MARK MNYAM YIG GI MGO RGYAN
0FD2 TIBETAN MARK NYIS TSHEG
0FD3 TIBETAN MARK INITIAL BRDA RNYING YIG MGO MDUN MA
0FD4 TIBETAN MARK CLOSING BRDA RNYING YIG MGO SGAB MA
0FD5 RIGHT-FACING SVASTI SIGN
0FD6 LEFT-FACING SVASTI SIGN
0FD7 RIGHT-FACING SVASTI SIGN WITH DOTS
0FD8 LEFT-FACING SVASTI SIGN WITH DOTS
0FD9 TIBETAN MARK LEADING MCHAN RTAGS
0FDA TIBETAN MARK TRAILING MCHAN RTAGS
1000 MYANMAR LETTER KA
1001 MYANMAR LETTER KHA
1002 MYANMAR LETTER GA
1003 MYANMAR LETTER GHA
1004 MYANMAR LETTER NGA
1005 MYANMAR LETTER CA
1006 MYANMAR LETTER CHA
1007 MYANMAR LETTER JA
1008 MYANMAR LETTER JHA
1009 MYANMAR LETTER NYA
100A MYANMAR LETTER NNYA
100B MYANMAR LETTER TTA
100C MYANMAR LETTER TTHA
100D MYANMAR LETTER DDA
100E MYANMAR LETTER DDHA
100F MYANMAR LETTER NNA
1010 MYANMAR LETTER TA
1011 MYANMAR LETTER THA
1012 MYANMAR LETTER DA
1013 MYANMAR LETTER DHA
1014 MYANMAR LETTER NA
1015 MYANMAR LETTER PA
1016 MYANMAR LETTER PHA
1017 MYANMAR LETTER BA
1018 MYANMAR LETTER BHA
1019 MYANMAR LETTER MA
101A MYANMAR LETTER YA
101B MYANMAR LETTER RA
101C MYANMAR LETTER LA
101D MYANMAR LETTER WA
101E MYANMAR LETTER SA
101F MYANMAR LETTER HA
1020 MYANMAR LETTER LLA
1021 MYANMAR LETTER A
1022 MYANMAR LETTER SHAN A
1023 MYANMAR LETTER I
1024 MYANMAR LETTER II
1025 MYANMAR LETTER U
1026 MYANMAR LETTER UU
1027 MYANMAR LETTER E
1028 MYANMAR LETTER MON E
1029 MYANMAR LETTER O
102A MYANMAR LETTER AU
102B MYANMAR VOWEL SIGN TALL AA
102C MYANMAR VOWEL SIGN AA
102D MYANMAR VOWEL SIGN I
102E MYANMAR VOWEL SIGN II
102F MYANMAR VOWEL SIGN U
1030 MYANMAR VOWEL SIGN UU
1031 MYANMAR VOWEL SIGN E
1032 MYANMAR VOWEL SIGN AI
1033 MYANMAR VOWEL SIGN MON II
1034 MYANMAR VOWEL SIGN MON O
1035 MYANMAR VOWEL SIGN E ABOVE
1036 MYANMAR SIGN ANUSVARA
1037 MYANMAR SIGN DOT BELOW
1038 MYANMAR SIGN VISARGA
1039 MYANMAR SIGN VIRAMA
103A MYANMAR SIGN ASAT
103B MYANMAR CONSONANT SIGN MEDIAL YA
103C MYANMAR CONSONANT SIGN MEDIAL RA
103D MYANMAR CONSONANT SIGN MEDIAL WA
103E MYANMAR CONSONANT SIGN MEDIAL HA
103F MYANMAR LETTER GREAT SA
1040 MYANMAR DIGIT ZERO
1041 MYANMAR DIGIT ONE
1042 MYANMAR DIGIT TWO
1043 MYANMAR DIGIT THREE
1044 MYANMAR DIGIT FOUR
1045 MYANMAR DIGIT FIVE
1046 MYANMAR DIGIT SIX
1047 MYANMAR DIGIT SEVEN
1048 MYANMAR DIGIT EIGHT
1049 MYANMAR DIGIT NINE
104A MYANMAR SIGN LITTLE SECTION
104B MYANMAR SIGN SECTION
104C MYANMAR SYMBOL LOCATIVE
104D MYANMAR SYMBOL COMPLETED
104E MYANMAR SYMBOL AFOREMENTIONED
104F MYANMAR SYMBOL GENITIVE
1050 MYANMAR LETTER SHA
1051 MYANMAR LETTER SSA
1052 MYANMAR LETTER VOCALIC R
1053 MYANMAR LETTER VOCALIC RR
1054 MYANMAR LETTER VOCALIC L
1055 MYANMAR LETTER VOCALIC LL
1056 MYANMAR VOWEL SIGN VOCALIC R
1057 MYANMAR VOWEL SIGN VOCALIC RR
1058 MYANMAR VOWEL SIGN VOCALIC L
1059 MYANMAR VOWEL SIGN VOCALIC LL
105A MYANMAR LETTER MON NGA
105B MYANMAR LETTER MON JHA
105C MYANMAR LETTER MON BBA
105D MYANMAR LETTER MON BBE
105E MYANMAR CONSONANT SIGN MON MEDIAL NA
105F MYANMAR CONSONANT SIGN MON MEDIAL MA
1060 MYANMAR CONSONANT SIGN MON MEDIAL LA
1061 MYANMAR LETTER SGAW KAREN SHA
1062 MYANMAR VOWEL SIGN SGAW KAREN EU
1063 MYANMAR TONE MARK SGAW KAREN HATHI
1064 MYANMAR TONE MARK SGAW KAREN KE PHO
1065 MYANMAR LETTER WESTERN PWO KAREN THA
1066 MYANMAR LETTER WESTERN PWO KAREN PWA
1067 MYANMAR VOWEL SIGN WESTERN PWO KAREN EU
1068 MYANMAR VOWEL SIGN WESTERN PWO KAREN UE
1069 MYANMAR SIGN WESTERN PWO KAREN TONE-1
106A MYANMAR SIGN WESTERN PWO KAREN TONE-2
106B MYANMAR SIGN WESTERN PWO KAREN TONE-3
106C MYANMAR SIGN WESTERN PWO KAREN TONE-4
106D MYANMAR SIGN WESTERN PWO KAREN TONE-5
106E MYANMAR LETTER EASTERN PWO KAREN NNA
106F MYANMAR LETTER EASTERN PWO KAREN YWA
1070 MYANMAR LETTER EASTERN PWO KAREN GHWA
1071 MYANMAR VOWEL SIGN GEBA KAREN I
1072 MYANMAR VOWEL SIGN KAYAH OE
1073 MYANMAR VOWEL SIGN KAYAH U
1074 MYANMAR VOWEL SIGN KAYAH EE
1075 MYANMAR LETTER SHAN KA
1076 MYANMAR LETTER SHAN KHA
1077 MYANMAR LETTER SHAN GA
1078 MYANMAR LETTER SHAN CA
1079 MYANMAR LETTER SHAN ZA
107A MYANMAR LETTER SHAN NYA
107B MYANMAR LETTER SHAN DA
107C MYANMAR LETTER SHAN NA
107D MYANMAR LETTER SHAN PHA
107E MYANMAR LETTER SHAN FA
107F MYANMAR LETTER SHAN BA
1080 MYANMAR LETTER SHAN THA
1081 MYANMAR LETTER SHAN HA
1082 MYANMAR CONSONANT SIGN SHAN MEDIAL WA
1083 MYANMAR VOWEL SIGN SHAN AA
1084 MYANMAR VOWEL SIGN SHAN E
1085 MYANMAR VOWEL SIGN SHAN E ABOVE
1086 MYANMAR VOWEL SIGN SHAN FINAL Y
1087 MYANMAR SIGN SHAN TONE-2
1088 MYANMAR SIGN SHAN TONE-3
1089 MYANMAR SIGN SHAN TONE-5
108A MYANMAR SIGN SHAN TONE-6
108B MYANMAR SIGN SHAN COUNCIL TONE-2
108C MYANMAR SIGN SHAN COUNCIL TONE-3
108D MYANMAR SIGN SHAN COUNCIL EMPHATIC TONE
108E MYANMAR LETTER RUMAI PALAUNG FA
108F MYANMAR SIGN RUMAI PALAUNG TONE-5
1090 MYANMAR SHAN DIGIT ZERO
1091 MYANMAR SHAN DIGIT ONE
1092 MYANMAR SHAN DIGIT TWO
1093 MYANMAR SHAN DIGIT THREE
1094 MYANMAR SHAN DIGIT FOUR
1095 MYANMAR SHAN DIGIT FIVE
1096 MYANMAR SHAN DIGIT SIX
1097 MYANMAR SHAN DIGIT SEVEN
1098 MYANMAR SHAN DIGIT EIGHT
1099 MYANMAR SHAN DIGIT NINE
109A MYANMAR SIGN KHAMTI TONE-1
109B MYANMAR SIGN KHAMTI TONE-3
109C MYANMAR VOWEL SIGN AITON A
109D MYANMAR VOWEL SIGN AITON AI
109E MYANMAR SYMBOL SHAN ONE
109F MYANMAR SYMBOL SHAN EXCLAMATION
10A0 GEORGIAN CAPITAL LETTER AN
10A1 GEORGIAN CAPITAL LETTER BAN
10A2 GEORGIAN CAPITAL LETTER GAN
10A3 GEORGIAN CAPITAL LETTER DON
10A4 GEORGIAN CAPITAL LETTER EN
10A5 GEORGIAN CAPITAL LETTER VIN
10A6 GEORGIAN CAPITAL LETTER ZEN
10A7 GEORGIAN CAPITAL LETTER TAN
10A8 GEORGIAN CAPITAL LETTER IN
10A9 GEORGIAN CAPITAL LETTER KAN
10AA GEORGIAN CAPITAL LETTER LAS
10AB GEORGIAN CAPITAL LETTER MAN
10AC GEORGIAN CAPITAL LETTER NAR
10AD GEORGIAN CAPITAL LETTER ON
10AE GEORGIAN CAPITAL LETTER PAR
10AF GEORGIAN CAPITAL LETTER ZHAR
10B0 GEORGIAN CAPITAL LETTER RAE
10B1 GEORGIAN CAPITAL LETTER SAN
10B2 GEORGIAN CAPITAL LETTER TAR
10B3 GEORGIAN CAPITAL LETTER UN
10B4 GEORGIAN CAPITAL LETTER PHAR
10B5 GEORGIAN CAPITAL LETTER KHAR
10B6 GEORGIAN CAPITAL LETTER GHAN
10B7 GEORGIAN CAPITAL LETTER QAR
10B8 GEORGIAN CAPITAL LETTER SHIN
10B9 GEORGIAN CAPITAL LETTER CHIN
10BA GEORGIAN CAPITAL LETTER CAN
10BB GEORGIAN CAPITAL LETTER JIL
10BC GEORGIAN CAPITAL LETTER CIL
10BD GEORGIAN CAPITAL LETTER CHAR
10BE GEORGIAN CAPITAL LETTER XAN
10BF GEORGIAN CAPITAL LETTER JHAN
10C0 GEORGIAN CAPITAL LETTER HAE
10C1 GEORGIAN CAPITAL LETTER HE
10C2 GEORGIAN CAPITAL LETTER HIE
10C3 GEORGIAN CAPITAL LETTER WE
10C4 GEORGIAN CAPITAL LETTER HAR
10C5 GEORGIAN CAPITAL LETTER HOE
10C7 GEORGIAN CAPITAL LETTER YN
10CD GEORGIAN CAPITAL LETTER AEN
10D0 GEORGIAN LETTER AN
10D1 GEORGIAN LETTER BAN
10D2 GEORGIAN LETTER GAN
10D3 GEORGIAN LETTER DON
10D4 GEORGIAN LETTER EN
10D5 GEORGIAN LETTER VIN
10D6 GEORGIAN LETTER ZEN
10D7 GEORGIAN LETTER TAN
10D8 GEORGIAN LETTER IN
10D9 GEORGIAN LETTER KAN
10DA GEORGIAN LETTER LAS
10DB GEORGIAN LETTER MAN
10DC GEORGIAN LETTER NAR
10DD GEORGIAN LETTER ON
10DE GEORGIAN LETTER PAR
10DF GEORGIAN LETTER ZHAR
10E0 GEORGIAN LETTER RAE
10E1 GEORGIAN LETTER SAN
10E2 GEORGIAN LETTER TAR
10E3 GEORGIAN LETTER UN
10E4 GEORGIAN LETTER PHAR
10E5 GEORGIAN LETTER KHAR
10E6 GEORGIAN LETTER GHAN
10E7 GEORGIAN LETTER QAR
10E8 GEORGIAN LETTER SHIN
10E9 GEORGIAN LETTER CHIN
10EA GEORGIAN LETTER CAN
10EB GEORGIAN LETTER JIL
10EC GEORGIAN LETTER CIL
10ED GEORGIAN LETTER CHAR
10EE GEORGIAN LETTER XAN
10EF GEORGIAN LETTER JHAN
10F0 GEORGIAN LETTER HAE
10F1 GEORGIAN LETTER HE
10F2 GEORGIAN LETTER HIE
10F3 GEORGIAN LETTER WE
10F4 GEORGIAN LETTER HAR
10F5 GEORGIAN LETTER HOE
10F6 GEORGIAN LETTER FI
10F7 GEORGIAN LETTER YN
10F8 GEORGIAN LETTER ELIFI
10F9 GEORGIAN LETTER TURNED GAN
10FA GEORGIAN LETTER AIN
10FB GEORGIAN PARAGRAPH SEPARATOR
10FC MODIFIER LETTER GEORGIAN NAR
10FD GEORGIAN LETTER AEN
10FE GEORGIAN LETTER HARD SIGN
10FF GEORGIAN LETTER LABIAL SIGN
1100 HANGUL CHOSEONG KIYEOK
1101 HANGUL CHOSEONG SSANGKIYEOK
1102 HANGUL CHOSEONG NIEUN
1103 HANGUL CHOSEONG TIKEUT
1104 HANGUL CHOSEONG SSANGTIKEUT
1105 HANGUL CHOSEONG RIEUL
1106 HANGUL CHOSEONG MIEUM
1107 HANGUL CHOSEONG PIEUP
1108 HANGUL CHOSEONG SSANGPIEUP
1109 HANGUL CHOSEONG SIOS
110A HANGUL CHOSEONG SSANGSIOS
110B HANGUL CHOSEONG IEUNG
110C HANGUL CHOSEONG CIEUC
110D HANGUL CHOSEONG SSANGCIEUC
110E HANGUL CHOSEONG CHIEUCH
110F HANGUL CHOSEONG KHIEUKH
1110 HANGUL CHOSEONG THIEUTH
1111 HANGUL CHOSEONG PHIEUPH
1112 HANGUL CHOSEONG HIEUH
1113 HANGUL CHOSEONG NIEUN-KIYEOK
1114 HANGUL CHOSEONG SSANGNIEUN
1115 HANGUL CHOSEONG NIEUN-TIKEUT
1116 HANGUL CHOSEONG NIEUN-PIEUP
1117 HANGUL CHOSEONG TIKEUT-KIYEOK
1118 HANGUL CHOSEONG RIEUL-NIEUN
1119 HANGUL CHOSEONG SSANGRIEUL
111A HANGUL CHOSEONG RIEUL-HIEUH
111B HANGUL CHOSEONG KAPYEOUNRIEUL
111C HANGUL CHOSEONG MIEUM-PIEUP
111D HANGUL CHOSEONG KAPYEOUNMIEUM
111E HANGUL CHOSEONG PIEUP-KIYEOK
111F HANGUL CHOSEONG PIEUP-NIEUN
1120 HANGUL CHOSEONG PIEUP-TIKEUT
1121 HANGUL CHOSEONG PIEUP-SIOS
1122 HANGUL CHOSEONG PIEUP-SIOS-KIYEOK
1123 HANGUL CHOSEONG PIEUP-SIOS-TIKEUT
1124 HANGUL CHOSEONG PIEUP-SIOS-PIEUP
1125 HANGUL CHOSEONG PIEUP-SSANGSIOS
1126 HANGUL CHOSEONG PIEUP-SIOS-CIEUC
1127 HANGUL CHOSEONG PIEUP-CIEUC
1128 HANGUL CHOSEONG PIEUP-CHIEUCH
1129 HANGUL CHOSEONG PIEUP-THIEUTH
112A HANGUL CHOSEONG PIEUP-PHIEUPH
112B HANGUL CHOSEONG KAPYEOUNPIEUP
112C HANGUL CHOSEONG KAPYEOUNSSANGPIEUP
112D HANGUL CHOSEONG SIOS-KIYEOK
112E HANGUL CHOSEONG SIOS-NIEUN
112F HANGUL CHOSEONG SIOS-TIKEUT
1130 HANGUL CHOSEONG SIOS-RIEUL
1131 HANGUL CHOSEONG SIOS-MIEUM
1132 HANGUL CHOSEONG SIOS-PIEUP
1133 HANGUL CHOSEONG SIOS-PIEUP-KIYEOK
1134 HANGUL CHOSEONG SIOS-SSANGSIOS
1135 HANGUL CHOSEONG SIOS-IEUNG
1136 HANGUL CHOSEONG SIOS-CIEUC
1137 HANGUL CHOSEONG SIOS-CHIEUCH
1138 HANGUL CHOSEONG SIOS-KHIEUKH
1139 HANGUL CHOSEONG SIOS-THIEUTH
113A HANGUL CHOSEONG SIOS-PHIEUPH
113B HANGUL CHOSEONG SIOS-HIEUH
113C HANGUL CHOSEONG CHITUEUMSIOS
113D HANGUL CHOSEONG CHITUEUMSSANGSIOS
113E HANGUL CHOSEONG CEONGCHIEUMSIOS
113F HANGUL CHOSEONG CEONGCHIEUMSSANGSIOS
1140 HANGUL CHOSEONG PANSIOS
1141 HANGUL CHOSEONG IEUNG-KIYEOK
1142 HANGUL CHOSEONG IEUNG-TIKEUT
1143 HANGUL CHOSEONG IEUNG-MIEUM
1144 HANGUL CHOSEONG IEUNG-PIEUP
1145 HANGUL CHOSEONG IEUNG-SIOS
1146 HANGUL CHOSEONG IEUNG-PANSIOS
1147 HANGUL CHOSEONG SSANGIEUNG
1148 HANGUL CHOSEONG IEUNG-CIEUC
1149 HANGUL CHOSEONG IEUNG-CHIEUCH
114A HANGUL CHOSEONG IEUNG-THIEUTH
114B HANGUL CHOSEONG IEUNG-PHIEUPH
114C HANGUL CHOSEONG YESIEUNG
114D HANGUL CHOSEONG CIEUC-IEUNG
114E HANGUL CHOSEONG CHITUEUMCIEUC
114F HANGUL CHOSEONG CHITUEUMSSANGCIEUC
1150 HANGUL CHOSEONG CEONGCHIEUMCIEUC
1151 HANGUL CHOSEONG CEONGCHIEUMSSANGCIEUC
1152 HANGUL CHOSEONG CHIEUCH-KHIEUKH
1153 HANGUL CHOSEONG CHIEUCH-HIEUH
1154 HANGUL CHOSEONG CHITUEUMCHIEUCH
1155 HANGUL CHOSEONG CEONGCHIEUMCHIEUCH
1156 HANGUL CHOSEONG PHIEUPH-PIEUP
1157 HANGUL CHOSEONG KAPYEOUNPHIEUPH
1158 HANGUL CHOSEONG SSANGHIEUH
1159 HANGUL CHOSEONG YEORINHIEUH
115A HANGUL CHOSEONG KIYEOK-TIKEUT
115B HANGUL CHOSEONG NIEUN-SIOS
115C HANGUL CHOSEONG NIEUN-CIEUC
115D HANGUL CHOSEONG NIEUN-HIEUH
115E HANGUL CHOSEONG TIKEUT-RIEUL
115F HANGUL CHOSEONG FILLER
1160 HANGUL JUNGSEONG FILLER
1161 HANGUL JUNGSEONG A
1162 HANGUL JUNGSEONG AE
1163 HANGUL JUNGSEONG YA
1164 HANGUL JUNGSEONG YAE
1165 HANGUL JUNGSEONG EO
1166 HANGUL JUNGSEONG E
1167 HANGUL JUNGSEONG YEO
1168 HANGUL JUNGSEONG YE
1169 HANGUL JUNGSEONG O
116A HANGUL JUNGSEONG WA
116B HANGUL JUNGSEONG WAE
116C HANGUL JUNGSEONG OE
116D HANGUL JUNGSEONG YO
116E HANGUL JUNGSEONG U
116F HANGUL JUNGSEONG WEO
1170 HANGUL JUNGSEONG WE
1171 HANGUL JUNGSEONG WI
1172 HANGUL JUNGSEONG YU
1173 HANGUL JUNGSEONG EU
1174 HANGUL JUNGSEONG YI
1175 HANGUL JUNGSEONG I
1176 HANGUL JUNGSEONG A-O
1177 HANGUL JUNGSEONG A-U
1178 HANGUL JUNGSEONG YA-O
1179 HANGUL JUNGSEONG YA-YO
117A HANGUL JUNGSEONG EO-O
117B HANGUL JUNGSEONG EO-U
117C HANGUL JUNGSEONG EO-EU
117D HANGUL JUNGSEONG YEO-O
117E HANGUL JUNGSEONG YEO-U
117F HANGUL JUNGSEONG O-EO
1180 HANGUL JUNGSEONG O-E
1181 HANGUL JUNGSEONG O-YE
1182 HANGUL JUNGSEONG O-O
1183 HANGUL JUNGSEONG O-U
1184 HANGUL JUNGSEONG YO-YA
1185 HANGUL JUNGSEONG YO-YAE
1186 HANGUL JUNGSEONG YO-YEO
1187 HANGUL JUNGSEONG YO-O
1188 HANGUL JUNGSEONG YO-I
1189 HANGUL JUNGSEONG U-A
118A HANGUL JUNGSEONG U-AE
118B HANGUL JUNGSEONG U-EO-EU
118C HANGUL JUNGSEONG U-YE
118D HANGUL JUNGSEONG U-U
118E HANGUL JUNGSEONG YU-A
118F HANGUL JUNGSEONG YU-EO
1190 HANGUL JUNGSEONG YU-E
1191 HANGUL JUNGSEONG YU-YEO
1192 HANGUL JUNGSEONG YU-YE
1193 HANGUL JUNGSEONG YU-U
1194 HANGUL JUNGSEONG YU-I
1195 HANGUL JUNGSEONG EU-U
1196 HANGUL JUNGSEONG EU-EU
1197 HANGUL JUNGSEONG YI-U
1198 HANGUL JUNGSEONG I-A
1199 HANGUL JUNGSEONG I-YA
119A HANGUL JUNGSEONG I-O
119B HANGUL JUNGSEONG I-U
119C HANGUL JUNGSEONG I-EU
119D HANGUL JUNGSEONG I-ARAEA
119E HANGUL JUNGSEONG ARAEA
119F HANGUL JUNGSEONG ARAEA-EO
11A0 HANGUL JUNGSEONG ARAEA-U
11A1 HANGUL JUNGSEONG ARAEA-I
11A2 HANGUL JUNGSEONG SSANGARAEA
11A3 HANGUL JUNGSEONG A-EU
11A4 HANGUL JUNGSEONG YA-U
11A5 HANGUL JUNGSEONG YEO-YA
11A6 HANGUL JUNGSEONG O-YA
11A7 HANGUL JUNGSEONG O-YAE
11A8 HANGUL JONGSEONG KIYEOK
11A9 HANGUL JONGSEONG SSANGKIYEOK
11AA HANGUL JONGSEONG KIYEOK-SIOS
11AB HANGUL JONGSEONG NIEUN
11AC HANGUL JONGSEONG NIEUN-CIEUC
11AD HANGUL JONGSEONG NIEUN-HIEUH
11AE HANGUL JONGSEONG TIKEUT
11AF HANGUL JONGSEONG RIEUL
11B0 HANGUL JONGSEONG RIEUL-KIYEOK
11B1 HANGUL JONGSEONG RIEUL-MIEUM
11B2 HANGUL JONGSEONG RIEUL-PIEUP
11B3 HANGUL JONGSEONG RIEUL-SIOS
11B4 HANGUL JONGSEONG RIEUL-THIEUTH
11B5 HANGUL JONGSEONG RIEUL-PHIEUPH
11B6 HANGUL JONGSEONG RIEUL-HIEUH
11B7 HANGUL JONGSEONG MIEUM
11B8 HANGUL JONGSEONG PIEUP
11B9 HANGUL JONGSEONG PIEUP-SIOS
11BA HANGUL JONGSEONG SIOS
11BB HANGUL JONGSEONG SSANGSIOS
11BC HANGUL JONGSEONG IEUNG
11BD HANGUL JONGSEONG CIEUC
11BE HANGUL JONGSEONG CHIEUCH
11BF HANGUL JONGSEONG KHIEUKH
11C0 HANGUL JONGSEONG THIEUTH
11C1 HANGUL JONGSEONG PHIEUPH
11C2 HANGUL JONGSEONG HIEUH
11C3 HANGUL JONGSEONG KIYEOK-RIEUL
11C4 HANGUL JONGSEONG KIYEOK-SIOS-KIYEOK
11C5 HANGUL JONGSEONG NIEUN-KIYEOK
11C6 HANGUL JONGSEONG NIEUN-TIKEUT
11C7 HANGUL JONGSEONG NIEUN-SIOS
11C8 HANGUL JONGSEONG NIEUN-PANSIOS
11C9 HANGUL JONGSEONG NIEUN-THIEUTH
11CA HANGUL JONGSEONG TIKEUT-KIYEOK
11CB HANGUL JONGSEONG TIKEUT-RIEUL
11CC HANGUL JONGSEONG RIEUL-KIYEOK-SIOS
11CD HANGUL JONGSEONG RIEUL-NIEUN
11CE HANGUL JONGSEONG RIEUL-TIKEUT
11CF HANGUL JONGSEONG RIEUL-TIKEUT-HIEUH
11D0 HANGUL JONGSEONG SSANGRIEUL
11D1 HANGUL JONGSEONG RIEUL-MIEUM-KIYEOK
11D2 HANGUL JONGSEONG RIEUL-MIEUM-SIOS
11D3 HANGUL JONGSEONG RIEUL-PIEUP-SIOS
11D4 HANGUL JONGSEONG RIEUL-PIEUP-HIEUH
11D5 HANGUL JONGSEONG RIEUL-KAPYEOUNPIEUP
11D6 HANGUL JONGSEONG RIEUL-SSANGSIOS
11D7 HANGUL JONGSEONG RIEUL-PANSIOS
11D8 HANGUL JONGSEONG RIEUL-KHIEUKH
11D9 HANGUL JONGSEONG RIEUL-YEORINHIEUH
11DA HANGUL JONGSEONG MIEUM-KIYEOK
11DB HANGUL JONGSEONG MIEUM-RIEUL
11DC HANGUL JONGSEONG MIEUM-PIEUP
11DD HANGUL JONGSEONG MIEUM-SIOS
11DE HANGUL JONGSEONG MIEUM-SSANGSIOS
11DF HANGUL JONGSEONG MIEUM-PANSIOS
11E0 HANGUL JONGSEONG MIEUM-CHIEUCH
11E1 HANGUL JONGSEONG MIEUM-HIEUH
11E2 HANGUL JONGSEONG KAPYEOUNMIEUM
11E3 HANGUL JONGSEONG PIEUP-RIEUL
11E4 HANGUL JONGSEONG PIEUP-PHIEUPH
11E5 HANGUL JONGSEONG PIEUP-HIEUH
11E6 HANGUL JONGSEONG KAPYEOUNPIEUP
11E7 HANGUL JONGSEONG SIOS-KIYEOK
11E8 HANGUL JONGSEONG SIOS-TIKEUT
11E9 HANGUL JONGSEONG SIOS-RIEUL
11EA HANGUL JONGSEONG SIOS-PIEUP
11EB HANGUL JONGSEONG PANSIOS
11EC HANGUL JONGSEONG IEUNG-KIYEOK
11ED HANGUL JONGSEONG IEUNG-SSANGKIYEOK
11EE HANGUL JONGSEONG SSANGIEUNG
11EF HANGUL JONGSEONG IEUNG-KHIEUKH
11F0 HANGUL JONGSEONG YESIEUNG
11F1 HANGUL JONGSEONG YESIEUNG-SIOS
11F2 HANGUL JONGSEONG YESIEUNG-PANSIOS
11F3 HANGUL JONGSEONG PHIEUPH-PIEUP
11F4 HANGUL JONGSEONG KAPYEOUNPHIEUPH
11F5 HANGUL JONGSEONG HIEUH-NIEUN
11F6 HANGUL JONGSEONG HIEUH-RIEUL
11F7 HANGUL JONGSEONG HIEUH-MIEUM
11F8 HANGUL JONGSEONG HIEUH-PIEUP
11F9 HANGUL JONGSEONG YEORINHIEUH
11FA HANGUL JONGSEONG KIYEOK-NIEUN
11FB HANGUL JONGSEONG KIYEOK-PIEUP
11FC HANGUL JONGSEONG KIYEOK-CHIEUCH
11FD HANGUL JONGSEONG KIYEOK-KHIEUKH
11FE HANGUL JONGSEONG KIYEOK-HIEUH
11FF HANGUL JONGSEONG SSANGNIEUN
1200 ETHIOPIC SYLLABLE HA
1201 ETHIOPIC SYLLABLE HU
1202 ETHIOPIC SYLLABLE HI
1203 ETHIOPIC SYLLABLE HAA
1204 ETHIOPIC SYLLABLE HEE
1205 ETHIOPIC SYLLABLE HE
1206 ETHIOPIC SYLLABLE HO
1207 ETHIOPIC SYLLABLE HOA
1208 ETHIOPIC SYLLABLE LA
1209 ETHIOPIC SYLLABLE LU
120A ETHIOPIC SYLLABLE LI
120B ETHIOPIC SYLLABLE LAA
120C ETHIOPIC SYLLABLE LEE
120D ETHIOPIC SYLLABLE LE
120E ETHIOPIC SYLLABLE LO
120F ETHIOPIC SYLLABLE LWA
1210 ETHIOPIC SYLLABLE HHA
1211 ETHIOPIC SYLLABLE HHU
1212 ETHIOPIC SYLLABLE HHI
1213 ETHIOPIC SYLLABLE HHAA
1214 ETHIOPIC SYLLABLE HHEE
1215 ETHIOPIC SYLLABLE HHE
1216 ETHIOPIC SYLLABLE HHO
1217 ETHIOPIC SYLLABLE HHWA
1218 ETHIOPIC SYLLABLE MA
1219 ETHIOPIC SYLLABLE MU
121A ETHIOPIC SYLLABLE MI
121B ETHIOPIC SYLLABLE MAA
121C ETHIOPIC SYLLABLE MEE
121D ETHIOPIC SYLLABLE ME
121E ETHIOPIC SYLLABLE MO
121F ETHIOPIC SYLLABLE MWA
1220 ETHIOPIC SYLLABLE SZA
1221 ETHIOPIC SYLLABLE SZU
1222 ETHIOPIC SYLLABLE SZI
1223 ETHIOPIC SYLLABLE SZAA
1224 ETHIOPIC SYLLABLE SZEE
1225 ETHIOPIC SYLLABLE SZE
1226 ETHIOPIC SYLLABLE SZO
1227 ETHIOPIC SYLLABLE SZWA
1228 ETHIOPIC SYLLABLE RA
1229 ETHIOPIC SYLLABLE RU
122A ETHIOPIC SYLLABLE RI
122B ETHIOPIC SYLLABLE RAA
122C ETHIOPIC SYLLABLE REE
122D ETHIOPIC SYLLABLE RE
122E ETHIOPIC SYLLABLE RO
122F ETHIOPIC SYLLABLE RWA
1230 ETHIOPIC SYLLABLE SA
1231 ETHIOPIC SYLLABLE SU
1232 ETHIOPIC SYLLABLE SI
1233 ETHIOPIC SYLLABLE SAA
1234 ETHIOPIC SYLLABLE SEE
1235 ETHIOPIC SYLLABLE SE
1236 ETHIOPIC SYLLABLE SO
1237 ETHIOPIC SYLLABLE SWA
1238 ETHIOPIC SYLLABLE SHA
1239 ETHIOPIC SYLLABLE SHU
123A ETHIOPIC SYLLABLE SHI
123B ETHIOPIC SYLLABLE SHAA
123C ETHIOPIC SYLLABLE SHEE
123D ETHIOPIC SYLLABLE SHE
123E ETHIOPIC SYLLABLE SHO
123F ETHIOPIC SYLLABLE SHWA
1240 ETHIOPIC SYLLABLE QA
1241 ETHIOPIC SYLLABLE QU
1242 ETHIOPIC SYLLABLE QI
1243 ETHIOPIC SYLLABLE QAA
1244 ETHIOPIC SYLLABLE QEE
1245 ETHIOPIC SYLLABLE QE
1246 ETHIOPIC SYLLABLE QO
1247 ETHIOPIC SYLLABLE QOA
1248 ETHIOPIC SYLLABLE QWA
124A ETHIOPIC SYLLABLE QWI
124B ETHIOPIC SYLLABLE QWAA
124C ETHIOPIC SYLLABLE QWEE
124D ETHIOPIC SYLLABLE QWE
1250 ETHIOPIC SYLLABLE QHA
1251 ETHIOPIC SYLLABLE QHU
1252 ETHIOPIC SYLLABLE QHI
1253 ETHIOPIC SYLLABLE QHAA
1254 ETHIOPIC SYLLABLE QHEE
1255 ETHIOPIC SYLLABLE QHE
1256 ETHIOPIC SYLLABLE QHO
1258 ETHIOPIC SYLLABLE QHWA
125A ETHIOPIC SYLLABLE QHWI
125B ETHIOPIC SYLLABLE QHWAA
125C ETHIOPIC SYLLABLE QHWEE
125D ETHIOPIC SYLLABLE QHWE
1260 ETHIOPIC SYLLABLE BA
1261 ETHIOPIC SYLLABLE BU
1262 ETHIOPIC SYLLABLE BI
1263 ETHIOPIC SYLLABLE BAA
1264 ETHIOPIC SYLLABLE BEE
1265 ETHIOPIC SYLLABLE BE
1266 ETHIOPIC SYLLABLE BO
1267 ETHIOPIC SYLLABLE BWA
1268 ETHIOPIC SYLLABLE VA
1269 ETHIOPIC SYLLABLE VU
126A ETHIOPIC SYLLABLE VI
126B ETHIOPIC SYLLABLE VAA
126C ETHIOPIC SYLLABLE VEE
126D ETHIOPIC SYLLABLE VE
126E ETHIOPIC SYLLABLE VO
126F ETHIOPIC SYLLABLE VWA
1270 ETHIOPIC SYLLABLE TA
1271 ETHIOPIC SYLLABLE TU
1272 ETHIOPIC SYLLABLE TI
1273 ETHIOPIC SYLLABLE TAA
1274 ETHIOPIC SYLLABLE TEE
1275 ETHIOPIC SYLLABLE TE
1276 ETHIOPIC SYLLABLE TO
1277 ETHIOPIC SYLLABLE TWA
1278 ETHIOPIC SYLLABLE CA
1279 ETHIOPIC SYLLABLE CU
127A ETHIOPIC SYLLABLE CI
127B ETHIOPIC SYLLABLE CAA
127C ETHIOPIC SYLLABLE CEE
127D ETHIOPIC SYLLABLE CE
127E ETHIOPIC SYLLABLE CO
127F ETHIOPIC SYLLABLE CWA
1280 ETHIOPIC SYLLABLE XA
1281 ETHIOPIC SYLLABLE XU
1282 ETHIOPIC SYLLABLE XI
1283 ETHIOPIC SYLLABLE XAA
1284 ETHIOPIC SYLLABLE XEE
1285 ETHIOPIC SYLLABLE XE
1286 ETHIOPIC SYLLABLE XO
1287 ETHIOPIC SYLLABLE XOA
1288 ETHIOPIC SYLLABLE XWA
128A ETHIOPIC SYLLABLE XWI
128B ETHIOPIC SYLLABLE XWAA
128C ETHIOPIC SYLLABLE XWEE
128D ETHIOPIC SYLLABLE XWE
1290 ETHIOPIC SYLLABLE NA
1291 ETHIOPIC SYLLABLE NU
1292 ETHIOPIC SYLLABLE NI
1293 ETHIOPIC SYLLABLE NAA
1294 ETHIOPIC SYLLABLE NEE
1295 ETHIOPIC SYLLABLE NE
1296 ETHIOPIC SYLLABLE NO
1297 ETHIOPIC SYLLABLE NWA
1298 ETHIOPIC SYLLABLE NYA
1299 ETHIOPIC SYLLABLE NYU
129A ETHIOPIC SYLLABLE NYI
129B ETHIOPIC SYLLABLE NYAA
129C ETHIOPIC SYLLABLE NYEE
129D ETHIOPIC SYLLABLE NYE
129E ETHIOPIC SYLLABLE NYO
129F ETHIOPIC SYLLABLE NYWA
12A0 ETHIOPIC SYLLABLE GLOTTAL A
12A1 ETHIOPIC SYLLABLE GLOTTAL U
12A2 ETHIOPIC SYLLABLE GLOTTAL I
12A3 ETHIOPIC SYLLABLE GLOTTAL AA
12A4 ETHIOPIC SYLLABLE GLOTTAL EE
12A5 ETHIOPIC SYLLABLE GLOTTAL E
12A6 ETHIOPIC SYLLABLE GLOTTAL O
12A7 ETHIOPIC SYLLABLE GLOTTAL WA
12A8 ETHIOPIC SYLLABLE KA
12A9 ETHIOPIC SYLLABLE KU
12AA ETHIOPIC SYLLABLE KI
12AB ETHIOPIC SYLLABLE KAA
12AC ETHIOPIC SYLLABLE KEE
12AD ETHIOPIC SYLLABLE KE
12AE ETHIOPIC SYLLABLE KO
12AF ETHIOPIC SYLLABLE KOA
12B0 ETHIOPIC SYLLABLE KWA
12B2 ETHIOPIC SYLLABLE KWI
12B3 ETHIOPIC SYLLABLE KWAA
12B4 ETHIOPIC SYLLABLE KWEE
12B5 ETHIOPIC SYLLABLE KWE
12B8 ETHIOPIC SYLLABLE KXA
12B9 ETHIOPIC SYLLABLE KXU
12BA ETHIOPIC SYLLABLE KXI
12BB ETHIOPIC SYLLABLE KXAA
12BC ETHIOPIC SYLLABLE KXEE
12BD ETHIOPIC SYLLABLE KXE
12BE ETHIOPIC SYLLABLE KXO
12C0 ETHIOPIC SYLLABLE KXWA
12C2 ETHIOPIC SYLLABLE KXWI
12C3 ETHIOPIC SYLLABLE KXWAA
12C4 ETHIOPIC SYLLABLE KXWEE
12C5 ETHIOPIC SYLLABLE KXWE
12C8 ETHIOPIC SYLLABLE WA
12C9 ETHIOPIC SYLLABLE WU
12CA ETHIOPIC SYLLABLE WI
12CB ETHIOPIC SYLLABLE WAA
12CC ETHIOPIC SYLLABLE WEE
12CD ETHIOPIC SYLLABLE WE
12CE ETHIOPIC SYLLABLE WO
12CF ETHIOPIC SYLLABLE WOA
12D0 ETHIOPIC SYLLABLE PHARYNGEAL A
12D1 ETHIOPIC SYLLABLE PHARYNGEAL U
12D2 ETHIOPIC SYLLABLE PHARYNGEAL I
12D3 ETHIOPIC SYLLABLE PHARYNGEAL AA
12D4 ETHIOPIC SYLLABLE PHARYNGEAL EE
12D5 ETHIOPIC SYLLABLE PHARYNGEAL E
12D6 ETHIOPIC SYLLABLE PHARYNGEAL O
12D8 ETHIOPIC SYLLABLE ZA
12D9 ETHIOPIC SYLLABLE ZU
12DA ETHIOPIC SYLLABLE ZI
12DB ETHIOPIC SYLLABLE ZAA
12DC ETHIOPIC SYLLABLE ZEE
12DD ETHIOPIC SYLLABLE ZE
12DE ETHIOPIC SYLLABLE ZO
12DF ETHIOPIC SYLLABLE ZWA
12E0 ETHIOPIC SYLLABLE ZHA
12E1 ETHIOPIC SYLLABLE ZHU
12E2 ETHIOPIC SYLLABLE ZHI
12E3 ETHIOPIC SYLLABLE ZHAA
12E4 ETHIOPIC SYLLABLE ZHEE
12E5 ETHIOPIC SYLLABLE ZHE
12E6 ETHIOPIC SYLLABLE ZHO
12E7 ETHIOPIC SYLLABLE ZHWA
12E8 ETHIOPIC SYLLABLE YA
12E9 ETHIOPIC SYLLABLE YU
12EA ETHIOPIC SYLLABLE YI
12EB ETHIOPIC SYLLABLE YAA
12EC ETHIOPIC SYLLABLE YEE
12ED ETHIOPIC SYLLABLE YE
12EE ETHIOPIC SYLLABLE YO
12EF ETHIOPIC SYLLABLE YOA
12F0 ETHIOPIC SYLLABLE DA
12F1 ETHIOPIC SYLLABLE DU
12F2 ETHIOPIC SYLLABLE DI
12F3 ETHIOPIC SYLLABLE DAA
12F4 ETHIOPIC SYLLABLE DEE
12F5 ETHIOPIC SYLLABLE DE
12F6 ETHIOPIC SYLLABLE DO
12F7 ETHIOPIC SYLLABLE DWA
12F8 ETHIOPIC SYLLABLE DDA
12F9 ETHIOPIC SYLLABLE DDU
12FA ETHIOPIC SYLLABLE DDI
12FB ETHIOPIC SYLLABLE DDAA
12FC ETHIOPIC SYLLABLE DDEE
12FD ETHIOPIC SYLLABLE DDE
12FE ETHIOPIC SYLLABLE DDO
12FF ETHIOPIC SYLLABLE DDWA
1300 ETHIOPIC SYLLABLE JA
1301 ETHIOPIC SYLLABLE JU
1302 ETHIOPIC SYLLABLE JI
1303 ETHIOPIC SYLLABLE JAA
1304 ETHIOPIC SYLLABLE JEE
1305 ETHIOPIC SYLLABLE JE
1306 ETHIOPIC SYLLABLE JO
1307 ETHIOPIC SYLLABLE JWA
1308 ETHIOPIC SYLLABLE GA
1309 ETHIOPIC SYLLABLE GU
130A ETHIOPIC SYLLABLE GI
130B ETHIOPIC SYLLABLE GAA
130C ETHIOPIC SYLLABLE GEE
130D ETHIOPIC SYLLABLE GE
130E ETHIOPIC SYLLABLE GO
130F ETHIOPIC SYLLABLE GOA
1310 ETHIOPIC SYLLABLE GWA
1312 ETHIOPIC SYLLABLE GWI
1313 ETHIOPIC SYLLABLE GWAA
1314 ETHIOPIC SYLLABLE GWEE
1315 ETHIOPIC SYLLABLE GWE
1318 ETHIOPIC SYLLABLE GGA
1319 ETHIOPIC SYLLABLE GGU
131A ETHIOPIC SYLLABLE GGI
131B ETHIOPIC SYLLABLE GGAA
131C ETHIOPIC SYLLABLE GGEE
131D ETHIOPIC SYLLABLE GGE
131E ETHIOPIC SYLLABLE GGO
131F ETHIOPIC SYLLABLE GGWAA
1320 ETHIOPIC SYLLABLE THA
1321 ETHIOPIC SYLLABLE THU
1322 ETHIOPIC SYLLABLE THI
1323 ETHIOPIC SYLLABLE THAA
1324 ETHIOPIC SYLLABLE THEE
1325 ETHIOPIC SYLLABLE THE
1326 ETHIOPIC SYLLABLE THO
1327 ETHIOPIC SYLLABLE THWA
1328 ETHIOPIC SYLLABLE CHA
1329 ETHIOPIC SYLLABLE CHU
132A ETHIOPIC SYLLABLE CHI
132B ETHIOPIC SYLLABLE CHAA
132C ETHIOPIC SYLLABLE CHEE
132D ETHIOPIC SYLLABLE CHE
132E ETHIOPIC SYLLABLE CHO
132F ETHIOPIC SYLLABLE CHWA
1330 ETHIOPIC SYLLABLE PHA
1331 ETHIOPIC SYLLABLE PHU
1332 ETHIOPIC SYLLABLE PHI
1333 ETHIOPIC SYLLABLE PHAA
1334 ETHIOPIC SYLLABLE PHEE
1335 ETHIOPIC SYLLABLE PHE
1336 ETHIOPIC SYLLABLE PHO
1337 ETHIOPIC SYLLABLE PHWA
1338 ETHIOPIC SYLLABLE TSA
1339 ETHIOPIC SYLLABLE TSU
133A ETHIOPIC SYLLABLE TSI
133B ETHIOPIC SYLLABLE TSAA
133C ETHIOPIC SYLLABLE TSEE
133D ETHIOPIC SYLLABLE TSE
133E ETHIOPIC SYLLABLE TSO
133F ETHIOPIC SYLLABLE TSWA
1340 ETHIOPIC SYLLABLE TZA
1341 ETHIOPIC SYLLABLE TZU
1342 ETHIOPIC SYLLABLE TZI
1343 ETHIOPIC SYLLABLE TZAA
1344 ETHIOPIC SYLLABLE TZEE
1345 ETHIOPIC SYLLABLE TZE
1346 ETHIOPIC SYLLABLE TZO
1347 ETHIOPIC SYLLABLE TZOA
1348 ETHIOPIC SYLLABLE FA
1349 ETHIOPIC SYLLABLE FU
134A ETHIOPIC SYLLABLE FI
134B ETHIOPIC SYLLABLE FAA
134C ETHIOPIC SYLLABLE FEE
134D ETHIOPIC SYLLABLE FE
134E ETHIOPIC SYLLABLE FO
134F ETHIOPIC SYLLABLE FWA
1350 ETHIOPIC SYLLABLE PA
1351 ETHIOPIC SYLLABLE PU
1352 ETHIOPIC SYLLABLE PI
1353 ETHIOPIC SYLLABLE PAA
1354 ETHIOPIC SYLLABLE PEE
1355 ETHIOPIC SYLLABLE PE
1356 ETHIOPIC SYLLABLE PO
1357 ETHIOPIC SYLLABLE PWA
1358 ETHIOPIC SYLLABLE RYA
1359 ETHIOPIC SYLLABLE MYA
135A ETHIOPIC SYLLABLE FYA
135D ETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARK
135E ETHIOPIC COMBINING VOWEL LENGTH MARK
135F ETHIOPIC COMBINING GEMINATION MARK
1360 ETHIOPIC SECTION MARK
1361 ETHIOPIC WORDSPACE
1362 ETHIOPIC FULL STOP
1363 ETHIOPIC COMMA
1364 ETHIOPIC SEMICOLON
1365 ETHIOPIC COLON
1366 ETHIOPIC PREFACE COLON
1367 ETHIOPIC QUESTION MARK
1368 ETHIOPIC PARAGRAPH SEPARATOR
1369 ETHIOPIC DIGIT ONE
136A ETHIOPIC DIGIT TWO
136B ETHIOPIC DIGIT THREE
136C ETHIOPIC DIGIT FOUR
136D ETHIOPIC DIGIT FIVE
136E ETHIOPIC DIGIT SIX
136F ETHIOPIC DIGIT SEVEN
1370 ETHIOPIC DIGIT EIGHT
1371 ETHIOPIC DIGIT NINE
1372 ETHIOPIC NUMBER TEN
1373 ETHIOPIC NUMBER TWENTY
1374 ETHIOPIC NUMBER THIRTY
1375 ETHIOPIC NUMBER FORTY
1376 ETHIOPIC NUMBER FIFTY
1377 ETHIOPIC NUMBER SIXTY
1378 ETHIOPIC NUMBER SEVENTY
1379 ETHIOPIC NUMBER EIGHTY
137A ETHIOPIC NUMBER NINETY
137B ETHIOPIC NUMBER HUNDRED
137C ETHIOPIC NUMBER TEN THOUSAND
1380 ETHIOPIC SYLLABLE SEBATBEIT MWA
1381 ETHIOPIC SYLLABLE MWI
1382 ETHIOPIC SYLLABLE MWEE
1383 ETHIOPIC SYLLABLE MWE
1384 ETHIOPIC SYLLABLE SEBATBEIT BWA
1385 ETHIOPIC SYLLABLE BWI
1386 ETHIOPIC SYLLABLE BWEE
1387 ETHIOPIC SYLLABLE BWE
1388 ETHIOPIC SYLLABLE SEBATBEIT FWA
1389 ETHIOPIC SYLLABLE FWI
138A ETHIOPIC SYLLABLE FWEE
138B ETHIOPIC SYLLABLE FWE
138C ETHIOPIC SYLLABLE SEBATBEIT PWA
138D ETHIOPIC SYLLABLE PWI
138E ETHIOPIC SYLLABLE PWEE
138F ETHIOPIC SYLLABLE PWE
1390 ETHIOPIC TONAL MARK YIZET
1391 ETHIOPIC TONAL MARK DERET
1392 ETHIOPIC TONAL MARK RIKRIK
1393 ETHIOPIC TONAL MARK SHORT RIKRIK
1394 ETHIOPIC TONAL MARK DIFAT
1395 ETHIOPIC TONAL MARK KENAT
1396 ETHIOPIC TONAL MARK CHIRET
1397 ETHIOPIC TONAL MARK HIDET
1398 ETHIOPIC TONAL MARK DERET-HIDET
1399 ETHIOPIC TONAL MARK KURT
13A0 CHEROKEE LETTER A
13A1 CHEROKEE LETTER E
13A2 CHEROKEE LETTER I
13A3 CHEROKEE LETTER O
13A4 CHEROKEE LETTER U
13A5 CHEROKEE LETTER V
13A6 CHEROKEE LETTER GA
13A7 CHEROKEE LETTER KA
13A8 CHEROKEE LETTER GE
13A9 CHEROKEE LETTER GI
13AA CHEROKEE LETTER GO
13AB CHEROKEE LETTER GU
13AC CHEROKEE LETTER GV
13AD CHEROKEE LETTER HA
13AE CHEROKEE LETTER HE
13AF CHEROKEE LETTER HI
13B0 CHEROKEE LETTER HO
13B1 CHEROKEE LETTER HU
13B2 CHEROKEE LETTER HV
13B3 CHEROKEE LETTER LA
13B4 CHEROKEE LETTER LE
13B5 CHEROKEE LETTER LI
13B6 CHEROKEE LETTER LO
13B7 CHEROKEE LETTER LU
13B8 CHEROKEE LETTER LV
13B9 CHEROKEE LETTER MA
13BA CHEROKEE LETTER ME
13BB CHEROKEE LETTER MI
13BC CHEROKEE LETTER MO
13BD CHEROKEE LETTER MU
13BE CHEROKEE LETTER NA
13BF CHEROKEE LETTER HNA
13C0 CHEROKEE LETTER NAH
13C1 CHEROKEE LETTER NE
13C2 CHEROKEE LETTER NI
13C3 CHEROKEE LETTER NO
13C4 CHEROKEE LETTER NU
13C5 CHEROKEE LETTER NV
13C6 CHEROKEE LETTER QUA
13C7 CHEROKEE LETTER QUE
13C8 CHEROKEE LETTER QUI
13C9 CHEROKEE LETTER QUO
13CA CHEROKEE LETTER QUU
13CB CHEROKEE LETTER QUV
13CC CHEROKEE LETTER SA
13CD CHEROKEE LETTER S
13CE CHEROKEE LETTER SE
13CF CHEROKEE LETTER SI
13D0 CHEROKEE LETTER SO
13D1 CHEROKEE LETTER SU
13D2 CHEROKEE LETTER SV
13D3 CHEROKEE LETTER DA
13D4 CHEROKEE LETTER TA
13D5 CHEROKEE LETTER DE
13D6 CHEROKEE LETTER TE
13D7 CHEROKEE LETTER DI
13D8 CHEROKEE LETTER TI
13D9 CHEROKEE LETTER DO
13DA CHEROKEE LETTER DU
13DB CHEROKEE LETTER DV
13DC CHEROKEE LETTER DLA
13DD CHEROKEE LETTER TLA
13DE CHEROKEE LETTER TLE
13DF CHEROKEE LETTER TLI
13E0 CHEROKEE LETTER TLO
13E1 CHEROKEE LETTER TLU
13E2 CHEROKEE LETTER TLV
13E3 CHEROKEE LETTER TSA
13E4 CHEROKEE LETTER TSE
13E5 CHEROKEE LETTER TSI
13E6 CHEROKEE LETTER TSO
13E7 CHEROKEE LETTER TSU
13E8 CHEROKEE LETTER TSV
13E9 CHEROKEE LETTER WA
13EA CHEROKEE LETTER WE
13EB CHEROKEE LETTER WI
13EC CHEROKEE LETTER WO
13ED CHEROKEE LETTER WU
13EE CHEROKEE LETTER WV
13EF CHEROKEE LETTER YA
13F0 CHEROKEE LETTER YE
13F1 CHEROKEE LETTER YI
13F2 CHEROKEE LETTER YO
13F3 CHEROKEE LETTER YU
13F4 CHEROKEE LETTER YV
13F5 CHEROKEE LETTER MV
13F8 CHEROKEE SMALL LETTER YE
13F9 CHEROKEE SMALL LETTER YI
13FA CHEROKEE SMALL LETTER YO
13FB CHEROKEE SMALL LETTER YU
13FC CHEROKEE SMALL LETTER YV
13FD CHEROKEE SMALL LETTER MV
1400 CANADIAN SYLLABICS HYPHEN
1401 CANADIAN SYLLABICS E
1402 CANADIAN SYLLABICS AAI
1403 CANADIAN SYLLABICS I
1404 CANADIAN SYLLABICS II
1405 CANADIAN SYLLABICS O
1406 CANADIAN SYLLABICS OO
1407 CANADIAN SYLLABICS Y-CREE OO
1408 CANADIAN SYLLABICS CARRIER EE
1409 CANADIAN SYLLABICS CARRIER I
140A CANADIAN SYLLABICS A
140B CANADIAN SYLLABICS AA
140C CANADIAN SYLLABICS WE
140D CANADIAN SYLLABICS WEST-CREE WE
140E CANADIAN SYLLABICS WI
140F CANADIAN SYLLABICS WEST-CREE WI
1410 CANADIAN SYLLABICS WII
1411 CANADIAN SYLLABICS WEST-CREE WII
1412 CANADIAN SYLLABICS WO
1413 CANADIAN SYLLABICS WEST-CREE WO
1414 CANADIAN SYLLABICS WOO
1415 CANADIAN SYLLABICS WEST-CREE WOO
1416 CANADIAN SYLLABICS NASKAPI WOO
1417 CANADIAN SYLLABICS WA
1418 CANADIAN SYLLABICS WEST-CREE WA
1419 CANADIAN SYLLABICS WAA
141A CANADIAN SYLLABICS WEST-CREE WAA
141B CANADIAN SYLLABICS NASKAPI WAA
141C CANADIAN SYLLABICS AI
141D CANADIAN SYLLABICS Y-CREE W
141E CANADIAN SYLLABICS GLOTTAL STOP
141F CANADIAN SYLLABICS FINAL ACUTE
1420 CANADIAN SYLLABICS FINAL GRAVE
1421 CANADIAN SYLLABICS FINAL BOTTOM HALF RING
1422 CANADIAN SYLLABICS FINAL TOP HALF RING
1423 CANADIAN SYLLABICS FINAL RIGHT HALF RING
1424 CANADIAN SYLLABICS FINAL RING
1425 CANADIAN SYLLABICS FINAL DOUBLE ACUTE
1426 CANADIAN SYLLABICS FINAL DOUBLE SHORT VERTICAL STROKES
1427 CANADIAN SYLLABICS FINAL MIDDLE DOT
1428 CANADIAN SYLLABICS FINAL SHORT HORIZONTAL STROKE
1429 CANADIAN SYLLABICS FINAL PLUS
142A CANADIAN SYLLABICS FINAL DOWN TACK
142B CANADIAN SYLLABICS EN
142C CANADIAN SYLLABICS IN
142D CANADIAN SYLLABICS ON
142E CANADIAN SYLLABICS AN
142F CANADIAN SYLLABICS PE
1430 CANADIAN SYLLABICS PAAI
1431 CANADIAN SYLLABICS PI
1432 CANADIAN SYLLABICS PII
1433 CANADIAN SYLLABICS PO
1434 CANADIAN SYLLABICS POO
1435 CANADIAN SYLLABICS Y-CREE POO
1436 CANADIAN SYLLABICS CARRIER HEE
1437 CANADIAN SYLLABICS CARRIER HI
1438 CANADIAN SYLLABICS PA
1439 CANADIAN SYLLABICS PAA
143A CANADIAN SYLLABICS PWE
143B CANADIAN SYLLABICS WEST-CREE PWE
143C CANADIAN SYLLABICS PWI
143D CANADIAN SYLLABICS WEST-CREE PWI
143E CANADIAN SYLLABICS PWII
143F CANADIAN SYLLABICS WEST-CREE PWII
1440 CANADIAN SYLLABICS PWO
1441 CANADIAN SYLLABICS WEST-CREE PWO
1442 CANADIAN SYLLABICS PWOO
1443 CANADIAN SYLLABICS WEST-CREE PWOO
1444 CANADIAN SYLLABICS PWA
1445 CANADIAN SYLLABICS WEST-CREE PWA
1446 CANADIAN SYLLABICS PWAA
1447 CANADIAN SYLLABICS WEST-CREE PWAA
1448 CANADIAN SYLLABICS Y-CREE PWAA
1449 CANADIAN SYLLABICS P
144A CANADIAN SYLLABICS WEST-CREE P
144B CANADIAN SYLLABICS CARRIER H
144C CANADIAN SYLLABICS TE
144D CANADIAN SYLLABICS TAAI
144E CANADIAN SYLLABICS TI
144F CANADIAN SYLLABICS TII
1450 CANADIAN SYLLABICS TO
1451 CANADIAN SYLLABICS TOO
1452 CANADIAN SYLLABICS Y-CREE TOO
1453 CANADIAN SYLLABICS CARRIER DEE
1454 CANADIAN SYLLABICS CARRIER DI
1455 CANADIAN SYLLABICS TA
1456 CANADIAN SYLLABICS TAA
1457 CANADIAN SYLLABICS TWE
1458 CANADIAN SYLLABICS WEST-CREE TWE
1459 CANADIAN SYLLABICS TWI
145A CANADIAN SYLLABICS WEST-CREE TWI
145B CANADIAN SYLLABICS TWII
145C CANADIAN SYLLABICS WEST-CREE TWII
145D CANADIAN SYLLABICS TWO
145E CANADIAN SYLLABICS WEST-CREE TWO
145F CANADIAN SYLLABICS TWOO
1460 CANADIAN SYLLABICS WEST-CREE TWOO
1461 CANADIAN SYLLABICS TWA
1462 CANADIAN SYLLABICS WEST-CREE TWA
1463 CANADIAN SYLLABICS TWAA
1464 CANADIAN SYLLABICS WEST-CREE TWAA
1465 CANADIAN SYLLABICS NASKAPI TWAA
1466 CANADIAN SYLLABICS T
1467 CANADIAN SYLLABICS TTE
1468 CANADIAN SYLLABICS TTI
1469 CANADIAN SYLLABICS TTO
146A CANADIAN SYLLABICS TTA
146B CANADIAN SYLLABICS KE
146C CANADIAN SYLLABICS KAAI
146D CANADIAN SYLLABICS KI
146E CANADIAN SYLLABICS KII
146F CANADIAN SYLLABICS KO
1470 CANADIAN SYLLABICS KOO
1471 CANADIAN SYLLABICS Y-CREE KOO
1472 CANADIAN SYLLABICS KA
1473 CANADIAN SYLLABICS KAA
1474 CANADIAN SYLLABICS KWE
1475 CANADIAN SYLLABICS WEST-CREE KWE
1476 CANADIAN SYLLABICS KWI
1477 CANADIAN SYLLABICS WEST-CREE KWI
1478 CANADIAN SYLLABICS KWII
1479 CANADIAN SYLLABICS WEST-CREE KWII
147A CANADIAN SYLLABICS KWO
147B CANADIAN SYLLABICS WEST-CREE KWO
147C CANADIAN SYLLABICS KWOO
147D CANADIAN SYLLABICS WEST-CREE KWOO
147E CANADIAN SYLLABICS KWA
147F CANADIAN SYLLABICS WEST-CREE KWA
1480 CANADIAN SYLLABICS KWAA
1481 CANADIAN SYLLABICS WEST-CREE KWAA
1482 CANADIAN SYLLABICS NASKAPI KWAA
1483 CANADIAN SYLLABICS K
1484 CANADIAN SYLLABICS KW
1485 CANADIAN SYLLABICS SOUTH-SLAVEY KEH
1486 CANADIAN SYLLABICS SOUTH-SLAVEY KIH
1487 CANADIAN SYLLABICS SOUTH-SLAVEY KOH
1488 CANADIAN SYLLABICS SOUTH-SLAVEY KAH
1489 CANADIAN SYLLABICS CE
148A CANADIAN SYLLABICS CAAI
148B CANADIAN SYLLABICS CI
148C CANADIAN SYLLABICS CII
148D CANADIAN SYLLABICS CO
148E CANADIAN SYLLABICS COO
148F CANADIAN SYLLABICS Y-CREE COO
1490 CANADIAN SYLLABICS CA
1491 CANADIAN SYLLABICS CAA
1492 CANADIAN SYLLABICS CWE
1493 CANADIAN SYLLABICS WEST-CREE CWE
1494 CANADIAN SYLLABICS CWI
1495 CANADIAN SYLLABICS WEST-CREE CWI
1496 CANADIAN SYLLABICS CWII
1497 CANADIAN SYLLABICS WEST-CREE CWII
1498 CANADIAN SYLLABICS CWO
1499 CANADIAN SYLLABICS WEST-CREE CWO
149A CANADIAN SYLLABICS CWOO
149B CANADIAN SYLLABICS WEST-CREE CWOO
149C CANADIAN SYLLABICS CWA
149D CANADIAN SYLLABICS WEST-CREE CWA
149E CANADIAN SYLLABICS CWAA
149F CANADIAN SYLLABICS WEST-CREE CWAA
14A0 CANADIAN SYLLABICS NASKAPI CWAA
14A1 CANADIAN SYLLABICS C
14A2 CANADIAN SYLLABICS SAYISI TH
14A3 CANADIAN SYLLABICS ME
14A4 CANADIAN SYLLABICS MAAI
14A5 CANADIAN SYLLABICS MI
14A6 CANADIAN SYLLABICS MII
14A7 CANADIAN SYLLABICS MO
14A8 CANADIAN SYLLABICS MOO
14A9 CANADIAN SYLLABICS Y-CREE MOO
14AA CANADIAN SYLLABICS MA
14AB CANADIAN SYLLABICS MAA
14AC CANADIAN SYLLABICS MWE
14AD CANADIAN SYLLABICS WEST-CREE MWE
14AE CANADIAN SYLLABICS MWI
14AF CANADIAN SYLLABICS WEST-CREE MWI
14B0 CANADIAN SYLLABICS MWII
14B1 CANADIAN SYLLABICS WEST-CREE MWII
14B2 CANADIAN SYLLABICS MWO
14B3 CANADIAN SYLLABICS WEST-CREE MWO
14B4 CANADIAN SYLLABICS MWOO
14B5 CANADIAN SYLLABICS WEST-CREE MWOO
14B6 CANADIAN SYLLABICS MWA
14B7 CANADIAN SYLLABICS WEST-CREE MWA
14B8 CANADIAN SYLLABICS MWAA
14B9 CANADIAN SYLLABICS WEST-CREE MWAA
14BA CANADIAN SYLLABICS NASKAPI MWAA
14BB CANADIAN SYLLABICS M
14BC CANADIAN SYLLABICS WEST-CREE M
14BD CANADIAN SYLLABICS MH
14BE CANADIAN SYLLABICS ATHAPASCAN M
14BF CANADIAN SYLLABICS SAYISI M
14C0 CANADIAN SYLLABICS NE
14C1 CANADIAN SYLLABICS NAAI
14C2 CANADIAN SYLLABICS NI
14C3 CANADIAN SYLLABICS NII
14C4 CANADIAN SYLLABICS NO
14C5 CANADIAN SYLLABICS NOO
14C6 CANADIAN SYLLABICS Y-CREE NOO
14C7 CANADIAN SYLLABICS NA
14C8 CANADIAN SYLLABICS NAA
14C9 CANADIAN SYLLABICS NWE
14CA CANADIAN SYLLABICS WEST-CREE NWE
14CB CANADIAN SYLLABICS NWA
14CC CANADIAN SYLLABICS WEST-CREE NWA
14CD CANADIAN SYLLABICS NWAA
14CE CANADIAN SYLLABICS WEST-CREE NWAA
14CF CANADIAN SYLLABICS NASKAPI NWAA
14D0 CANADIAN SYLLABICS N
14D1 CANADIAN SYLLABICS CARRIER NG
14D2 CANADIAN SYLLABICS NH
14D3 CANADIAN SYLLABICS LE
14D4 CANADIAN SYLLABICS LAAI
14D5 CANADIAN SYLLABICS LI
14D6 CANADIAN SYLLABICS LII
14D7 CANADIAN SYLLABICS LO
14D8 CANADIAN SYLLABICS LOO
14D9 CANADIAN SYLLABICS Y-CREE LOO
14DA CANADIAN SYLLABICS LA
14DB CANADIAN SYLLABICS LAA
14DC CANADIAN SYLLABICS LWE
14DD CANADIAN SYLLABICS WEST-CREE LWE
14DE CANADIAN SYLLABICS LWI
14DF CANADIAN SYLLABICS WEST-CREE LWI
14E0 CANADIAN SYLLABICS LWII
14E1 CANADIAN SYLLABICS WEST-CREE LWII
14E2 CANADIAN SYLLABICS LWO
14E3 CANADIAN SYLLABICS WEST-CREE LWO
14E4 CANADIAN SYLLABICS LWOO
14E5 CANADIAN SYLLABICS WEST-CREE LWOO
14E6 CANADIAN SYLLABICS LWA
14E7 CANADIAN SYLLABICS WEST-CREE LWA
14E8 CANADIAN SYLLABICS LWAA
14E9 CANADIAN SYLLABICS WEST-CREE LWAA
14EA CANADIAN SYLLABICS L
14EB CANADIAN SYLLABICS WEST-CREE L
14EC CANADIAN SYLLABICS MEDIAL L
14ED CANADIAN SYLLABICS SE
14EE CANADIAN SYLLABICS SAAI
14EF CANADIAN SYLLABICS SI
14F0 CANADIAN SYLLABICS SII
14F1 CANADIAN SYLLABICS SO
14F2 CANADIAN SYLLABICS SOO
14F3 CANADIAN SYLLABICS Y-CREE SOO
14F4 CANADIAN SYLLABICS SA
14F5 CANADIAN SYLLABICS SAA
14F6 CANADIAN SYLLABICS SWE
14F7 CANADIAN SYLLABICS WEST-CREE SWE
14F8 CANADIAN SYLLABICS SWI
14F9 CANADIAN SYLLABICS WEST-CREE SWI
14FA CANADIAN SYLLABICS SWII
14FB CANADIAN SYLLABICS WEST-CREE SWII
14FC CANADIAN SYLLABICS SWO
14FD CANADIAN SYLLABICS WEST-CREE SWO
14FE CANADIAN SYLLABICS SWOO
14FF CANADIAN SYLLABICS WEST-CREE SWOO
1500 CANADIAN SYLLABICS SWA
1501 CANADIAN SYLLABICS WEST-CREE SWA
1502 CANADIAN SYLLABICS SWAA
1503 CANADIAN SYLLABICS WEST-CREE SWAA
1504 CANADIAN SYLLABICS NASKAPI SWAA
1505 CANADIAN SYLLABICS S
1506 CANADIAN SYLLABICS ATHAPASCAN S
1507 CANADIAN SYLLABICS SW
1508 CANADIAN SYLLABICS BLACKFOOT S
1509 CANADIAN SYLLABICS MOOSE-CREE SK
150A CANADIAN SYLLABICS NASKAPI SKW
150B CANADIAN SYLLABICS NASKAPI S-W
150C CANADIAN SYLLABICS NASKAPI SPWA
150D CANADIAN SYLLABICS NASKAPI STWA
150E CANADIAN SYLLABICS NASKAPI SKWA
150F CANADIAN SYLLABICS NASKAPI SCWA
1510 CANADIAN SYLLABICS SHE
1511 CANADIAN SYLLABICS SHI
1512 CANADIAN SYLLABICS SHII
1513 CANADIAN SYLLABICS SHO
1514 CANADIAN SYLLABICS SHOO
1515 CANADIAN SYLLABICS SHA
1516 CANADIAN SYLLABICS SHAA
1517 CANADIAN SYLLABICS SHWE
1518 CANADIAN SYLLABICS WEST-CREE SHWE
1519 CANADIAN SYLLABICS SHWI
151A CANADIAN SYLLABICS WEST-CREE SHWI
151B CANADIAN SYLLABICS SHWII
151C CANADIAN SYLLABICS WEST-CREE SHWII
151D CANADIAN SYLLABICS SHWO
151E CANADIAN SYLLABICS WEST-CREE SHWO
151F CANADIAN SYLLABICS SHWOO
1520 CANADIAN SYLLABICS WEST-CREE SHWOO
1521 CANADIAN SYLLABICS SHWA
1522 CANADIAN SYLLABICS WEST-CREE SHWA
1523 CANADIAN SYLLABICS SHWAA
1524 CANADIAN SYLLABICS WEST-CREE SHWAA
1525 CANADIAN SYLLABICS SH
1526 CANADIAN SYLLABICS YE
1527 CANADIAN SYLLABICS YAAI
1528 CANADIAN SYLLABICS YI
1529 CANADIAN SYLLABICS YII
152A CANADIAN SYLLABICS YO
152B CANADIAN SYLLABICS YOO
152C CANADIAN SYLLABICS Y-CREE YOO
152D CANADIAN SYLLABICS YA
152E CANADIAN SYLLABICS YAA
152F CANADIAN SYLLABICS YWE
1530 CANADIAN SYLLABICS WEST-CREE YWE
1531 CANADIAN SYLLABICS YWI
1532 CANADIAN SYLLABICS WEST-CREE YWI
1533 CANADIAN SYLLABICS YWII
1534 CANADIAN SYLLABICS WEST-CREE YWII
1535 CANADIAN SYLLABICS YWO
1536 CANADIAN SYLLABICS WEST-CREE YWO
1537 CANADIAN SYLLABICS YWOO
1538 CANADIAN SYLLABICS WEST-CREE YWOO
1539 CANADIAN SYLLABICS YWA
153A CANADIAN SYLLABICS WEST-CREE YWA
153B CANADIAN SYLLABICS YWAA
153C CANADIAN SYLLABICS WEST-CREE YWAA
153D CANADIAN SYLLABICS NASKAPI YWAA
153E CANADIAN SYLLABICS Y
153F CANADIAN SYLLABICS BIBLE-CREE Y
1540 CANADIAN SYLLABICS WEST-CREE Y
1541 CANADIAN SYLLABICS SAYISI YI
1542 CANADIAN SYLLABICS RE
1543 CANADIAN SYLLABICS R-CREE RE
1544 CANADIAN SYLLABICS WEST-CREE LE
1545 CANADIAN SYLLABICS RAAI
1546 CANADIAN SYLLABICS RI
1547 CANADIAN SYLLABICS RII
1548 CANADIAN SYLLABICS RO
1549 CANADIAN SYLLABICS ROO
154A CANADIAN SYLLABICS WEST-CREE LO
154B CANADIAN SYLLABICS RA
154C CANADIAN SYLLABICS RAA
154D CANADIAN SYLLABICS WEST-CREE LA
154E CANADIAN SYLLABICS RWAA
154F CANADIAN SYLLABICS WEST-CREE RWAA
1550 CANADIAN SYLLABICS R
1551 CANADIAN SYLLABICS WEST-CREE R
1552 CANADIAN SYLLABICS MEDIAL R
1553 CANADIAN SYLLABICS FE
1554 CANADIAN SYLLABICS FAAI
1555 CANADIAN SYLLABICS FI
1556 CANADIAN SYLLABICS FII
1557 CANADIAN SYLLABICS FO
1558 CANADIAN SYLLABICS FOO
1559 CANADIAN SYLLABICS FA
155A CANADIAN SYLLABICS FAA
155B CANADIAN SYLLABICS FWAA
155C CANADIAN SYLLABICS WEST-CREE FWAA
155D CANADIAN SYLLABICS F
155E CANADIAN SYLLABICS THE
155F CANADIAN SYLLABICS N-CREE THE
1560 CANADIAN SYLLABICS THI
1561 CANADIAN SYLLABICS N-CREE THI
1562 CANADIAN SYLLABICS THII
1563 CANADIAN SYLLABICS N-CREE THII
1564 CANADIAN SYLLABICS THO
1565 CANADIAN SYLLABICS THOO
1566 CANADIAN SYLLABICS THA
1567 CANADIAN SYLLABICS THAA
1568 CANADIAN SYLLABICS THWAA
1569 CANADIAN SYLLABICS WEST-CREE THWAA
156A CANADIAN SYLLABICS TH
156B CANADIAN SYLLABICS TTHE
156C CANADIAN SYLLABICS TTHI
156D CANADIAN SYLLABICS TTHO
156E CANADIAN SYLLABICS TTHA
156F CANADIAN SYLLABICS TTH
1570 CANADIAN SYLLABICS TYE
1571 CANADIAN SYLLABICS TYI
1572 CANADIAN SYLLABICS TYO
1573 CANADIAN SYLLABICS TYA
1574 CANADIAN SYLLABICS NUNAVIK HE
1575 CANADIAN SYLLABICS NUNAVIK HI
1576 CANADIAN SYLLABICS NUNAVIK HII
1577 CANADIAN SYLLABICS NUNAVIK HO
1578 CANADIAN SYLLABICS NUNAVIK HOO
1579 CANADIAN SYLLABICS NUNAVIK HA
157A CANADIAN SYLLABICS NUNAVIK HAA
157B CANADIAN SYLLABICS NUNAVIK H
157C CANADIAN SYLLABICS NUNAVUT H
157D CANADIAN SYLLABICS HK
157E CANADIAN SYLLABICS QAAI
157F CANADIAN SYLLABICS QI
1580 CANADIAN SYLLABICS QII
1581 CANADIAN SYLLABICS QO
1582 CANADIAN SYLLABICS QOO
1583 CANADIAN SYLLABICS QA
1584 CANADIAN SYLLABICS QAA
1585 CANADIAN SYLLABICS Q
1586 CANADIAN SYLLABICS TLHE
1587 CANADIAN SYLLABICS TLHI
1588 CANADIAN SYLLABICS TLHO
1589 CANADIAN SYLLABICS TLHA
158A CANADIAN SYLLABICS WEST-CREE RE
158B CANADIAN SYLLABICS WEST-CREE RI
158C CANADIAN SYLLABICS WEST-CREE RO
158D CANADIAN SYLLABICS WEST-CREE RA
158E CANADIAN SYLLABICS NGAAI
158F CANADIAN SYLLABICS NGI
1590 CANADIAN SYLLABICS NGII
1591 CANADIAN SYLLABICS NGO
1592 CANADIAN SYLLABICS NGOO
1593 CANADIAN SYLLABICS NGA
1594 CANADIAN SYLLABICS NGAA
1595 CANADIAN SYLLABICS NG
1596 CANADIAN SYLLABICS NNG
1597 CANADIAN SYLLABICS SAYISI SHE
1598 CANADIAN SYLLABICS SAYISI SHI
1599 CANADIAN SYLLABICS SAYISI SHO
159A CANADIAN SYLLABICS SAYISI SHA
159B CANADIAN SYLLABICS WOODS-CREE THE
159C CANADIAN SYLLABICS WOODS-CREE THI
159D CANADIAN SYLLABICS WOODS-CREE THO
159E CANADIAN SYLLABICS WOODS-CREE THA
159F CANADIAN SYLLABICS WOODS-CREE TH
15A0 CANADIAN SYLLABICS LHI
15A1 CANADIAN SYLLABICS LHII
15A2 CANADIAN SYLLABICS LHO
15A3 CANADIAN SYLLABICS LHOO
15A4 CANADIAN SYLLABICS LHA
15A5 CANADIAN SYLLABICS LHAA
15A6 CANADIAN SYLLABICS LH
15A7 CANADIAN SYLLABICS TH-CREE THE
15A8 CANADIAN SYLLABICS TH-CREE THI
15A9 CANADIAN SYLLABICS TH-CREE THII
15AA CANADIAN SYLLABICS TH-CREE THO
15AB CANADIAN SYLLABICS TH-CREE THOO
15AC CANADIAN SYLLABICS TH-CREE THA
15AD CANADIAN SYLLABICS TH-CREE THAA
15AE CANADIAN SYLLABICS TH-CREE TH
15AF CANADIAN SYLLABICS AIVILIK B
15B0 CANADIAN SYLLABICS BLACKFOOT E
15B1 CANADIAN SYLLABICS BLACKFOOT I
15B2 CANADIAN SYLLABICS BLACKFOOT O
15B3 CANADIAN SYLLABICS BLACKFOOT A
15B4 CANADIAN SYLLABICS BLACKFOOT WE
15B5 CANADIAN SYLLABICS BLACKFOOT WI
15B6 CANADIAN SYLLABICS BLACKFOOT WO
15B7 CANADIAN SYLLABICS BLACKFOOT WA
15B8 CANADIAN SYLLABICS BLACKFOOT NE
15B9 CANADIAN SYLLABICS BLACKFOOT NI
15BA CANADIAN SYLLABICS BLACKFOOT NO
15BB CANADIAN SYLLABICS BLACKFOOT NA
15BC CANADIAN SYLLABICS BLACKFOOT KE
15BD CANADIAN SYLLABICS BLACKFOOT KI
15BE CANADIAN SYLLABICS BLACKFOOT KO
15BF CANADIAN SYLLABICS BLACKFOOT KA
15C0 CANADIAN SYLLABICS SAYISI HE
15C1 CANADIAN SYLLABICS SAYISI HI
15C2 CANADIAN SYLLABICS SAYISI HO
15C3 CANADIAN SYLLABICS SAYISI HA
15C4 CANADIAN SYLLABICS CARRIER GHU
15C5 CANADIAN SYLLABICS CARRIER GHO
15C6 CANADIAN SYLLABICS CARRIER GHE
15C7 CANADIAN SYLLABICS CARRIER GHEE
15C8 CANADIAN SYLLABICS CARRIER GHI
15C9 CANADIAN SYLLABICS CARRIER GHA
15CA CANADIAN SYLLABICS CARRIER RU
15CB CANADIAN SYLLABICS CARRIER RO
15CC CANADIAN SYLLABICS CARRIER RE
15CD CANADIAN SYLLABICS CARRIER REE
15CE CANADIAN SYLLABICS CARRIER RI
15CF CANADIAN SYLLABICS CARRIER RA
15D0 CANADIAN SYLLABICS CARRIER WU
15D1 CANADIAN SYLLABICS CARRIER WO
15D2 CANADIAN SYLLABICS CARRIER WE
15D3 CANADIAN SYLLABICS CARRIER WEE
15D4 CANADIAN SYLLABICS CARRIER WI
15D5 CANADIAN SYLLABICS CARRIER WA
15D6 CANADIAN SYLLABICS CARRIER HWU
15D7 CANADIAN SYLLABICS CARRIER HWO
15D8 CANADIAN SYLLABICS CARRIER HWE
15D9 CANADIAN SYLLABICS CARRIER HWEE
15DA CANADIAN SYLLABICS CARRIER HWI
15DB CANADIAN SYLLABICS CARRIER HWA
15DC CANADIAN SYLLABICS CARRIER THU
15DD CANADIAN SYLLABICS CARRIER THO
15DE CANADIAN SYLLABICS CARRIER THE
15DF CANADIAN SYLLABICS CARRIER THEE
15E0 CANADIAN SYLLABICS CARRIER THI
15E1 CANADIAN SYLLABICS CARRIER THA
15E2 CANADIAN SYLLABICS CARRIER TTU
15E3 CANADIAN SYLLABICS CARRIER TTO
15E4 CANADIAN SYLLABICS CARRIER TTE
15E5 CANADIAN SYLLABICS CARRIER TTEE
15E6 CANADIAN SYLLABICS CARRIER TTI
15E7 CANADIAN SYLLABICS CARRIER TTA
15E8 CANADIAN SYLLABICS CARRIER PU
15E9 CANADIAN SYLLABICS CARRIER PO
15EA CANADIAN SYLLABICS CARRIER PE
15EB CANADIAN SYLLABICS CARRIER PEE
15EC CANADIAN SYLLABICS CARRIER PI
15ED CANADIAN SYLLABICS CARRIER PA
15EE CANADIAN SYLLABICS CARRIER P
15EF CANADIAN SYLLABICS CARRIER GU
15F0 CANADIAN SYLLABICS CARRIER GO
15F1 CANADIAN SYLLABICS CARRIER GE
15F2 CANADIAN SYLLABICS CARRIER GEE
15F3 CANADIAN SYLLABICS CARRIER GI
15F4 CANADIAN SYLLABICS CARRIER GA
15F5 CANADIAN SYLLABICS CARRIER KHU
15F6 CANADIAN SYLLABICS CARRIER KHO
15F7 CANADIAN SYLLABICS CARRIER KHE
15F8 CANADIAN SYLLABICS CARRIER KHEE
15F9 CANADIAN SYLLABICS CARRIER KHI
15FA CANADIAN SYLLABICS CARRIER KHA
15FB CANADIAN SYLLABICS CARRIER KKU
15FC CANADIAN SYLLABICS CARRIER KKO
15FD CANADIAN SYLLABICS CARRIER KKE
15FE CANADIAN SYLLABICS CARRIER KKEE
15FF CANADIAN SYLLABICS CARRIER KKI
1600 CANADIAN SYLLABICS CARRIER KKA
1601 CANADIAN SYLLABICS CARRIER KK
1602 CANADIAN SYLLABICS CARRIER NU
1603 CANADIAN SYLLABICS CARRIER NO
1604 CANADIAN SYLLABICS CARRIER NE
1605 CANADIAN SYLLABICS CARRIER NEE
1606 CANADIAN SYLLABICS CARRIER NI
1607 CANADIAN SYLLABICS CARRIER NA
1608 CANADIAN SYLLABICS CARRIER MU
1609 CANADIAN SYLLABICS CARRIER MO
160A CANADIAN SYLLABICS CARRIER ME
160B CANADIAN SYLLABICS CARRIER MEE
160C CANADIAN SYLLABICS CARRIER MI
160D CANADIAN SYLLABICS CARRIER MA
160E CANADIAN SYLLABICS CARRIER YU
160F CANADIAN SYLLABICS CARRIER YO
1610 CANADIAN SYLLABICS CARRIER YE
1611 CANADIAN SYLLABICS CARRIER YEE
1612 CANADIAN SYLLABICS CARRIER YI
1613 CANADIAN SYLLABICS CARRIER YA
1614 CANADIAN SYLLABICS CARRIER JU
1615 CANADIAN SYLLABICS SAYISI JU
1616 CANADIAN SYLLABICS CARRIER JO
1617 CANADIAN SYLLABICS CARRIER JE
1618 CANADIAN SYLLABICS CARRIER JEE
1619 CANADIAN SYLLABICS CARRIER JI
161A CANADIAN SYLLABICS SAYISI JI
161B CANADIAN SYLLABICS CARRIER JA
161C CANADIAN SYLLABICS CARRIER JJU
161D CANADIAN SYLLABICS CARRIER JJO
161E CANADIAN SYLLABICS CARRIER JJE
161F CANADIAN SYLLABICS CARRIER JJEE
1620 CANADIAN SYLLABICS CARRIER JJI
1621 CANADIAN SYLLABICS CARRIER JJA
1622 CANADIAN SYLLABICS CARRIER LU
1623 CANADIAN SYLLABICS CARRIER LO
1624 CANADIAN SYLLABICS CARRIER LE
1625 CANADIAN SYLLABICS CARRIER LEE
1626 CANADIAN SYLLABICS CARRIER LI
1627 CANADIAN SYLLABICS CARRIER LA
1628 CANADIAN SYLLABICS CARRIER DLU
1629 CANADIAN SYLLABICS CARRIER DLO
162A CANADIAN SYLLABICS CARRIER DLE
162B CANADIAN SYLLABICS CARRIER DLEE
162C CANADIAN SYLLABICS CARRIER DLI
162D CANADIAN SYLLABICS CARRIER DLA
162E CANADIAN SYLLABICS CARRIER LHU
162F CANADIAN SYLLABICS CARRIER LHO
1630 CANADIAN SYLLABICS CARRIER LHE
1631 CANADIAN SYLLABICS CARRIER LHEE
1632 CANADIAN SYLLABICS CARRIER LHI
1633 CANADIAN SYLLABICS CARRIER LHA
1634 CANADIAN SYLLABICS CARRIER TLHU
1635 CANADIAN SYLLABICS CARRIER TLHO
1636 CANADIAN SYLLABICS CARRIER TLHE
1637 CANADIAN SYLLABICS CARRIER TLHEE
1638 CANADIAN SYLLABICS CARRIER TLHI
1639 CANADIAN SYLLABICS CARRIER TLHA
163A CANADIAN SYLLABICS CARRIER TLU
163B CANADIAN SYLLABICS CARRIER TLO
163C CANADIAN SYLLABICS CARRIER TLE
163D CANADIAN SYLLABICS CARRIER TLEE
163E CANADIAN SYLLABICS CARRIER TLI
163F CANADIAN SYLLABICS CARRIER TLA
1640 CANADIAN SYLLABICS CARRIER ZU
1641 CANADIAN SYLLABICS CARRIER ZO
1642 CANADIAN SYLLABICS CARRIER ZE
1643 CANADIAN SYLLABICS CARRIER ZEE
1644 CANADIAN SYLLABICS CARRIER ZI
1645 CANADIAN SYLLABICS CARRIER ZA
1646 CANADIAN SYLLABICS CARRIER Z
1647 CANADIAN SYLLABICS CARRIER INITIAL Z
1648 CANADIAN SYLLABICS CARRIER DZU
1649 CANADIAN SYLLABICS CARRIER DZO
164A CANADIAN SYLLABICS CARRIER DZE
164B CANADIAN SYLLABICS CARRIER DZEE
164C CANADIAN SYLLABICS CARRIER DZI
164D CANADIAN SYLLABICS CARRIER DZA
164E CANADIAN SYLLABICS CARRIER SU
164F CANADIAN SYLLABICS CARRIER SO
1650 CANADIAN SYLLABICS CARRIER SE
1651 CANADIAN SYLLABICS CARRIER SEE
1652 CANADIAN SYLLABICS CARRIER SI
1653 CANADIAN SYLLABICS CARRIER SA
1654 CANADIAN SYLLABICS CARRIER SHU
1655 CANADIAN SYLLABICS CARRIER SHO
1656 CANADIAN SYLLABICS CARRIER SHE
1657 CANADIAN SYLLABICS CARRIER SHEE
1658 CANADIAN SYLLABICS CARRIER SHI
1659 CANADIAN SYLLABICS CARRIER SHA
165A CANADIAN SYLLABICS CARRIER SH
165B CANADIAN SYLLABICS CARRIER TSU
165C CANADIAN SYLLABICS CARRIER TSO
165D CANADIAN SYLLABICS CARRIER TSE
165E CANADIAN SYLLABICS CARRIER TSEE
165F CANADIAN SYLLABICS CARRIER TSI
1660 CANADIAN SYLLABICS CARRIER TSA
1661 CANADIAN SYLLABICS CARRIER CHU
1662 CANADIAN SYLLABICS CARRIER CHO
1663 CANADIAN SYLLABICS CARRIER CHE
1664 CANADIAN SYLLABICS CARRIER CHEE
1665 CANADIAN SYLLABICS CARRIER CHI
1666 CANADIAN SYLLABICS CARRIER CHA
1667 CANADIAN SYLLABICS CARRIER TTSU
1668 CANADIAN SYLLABICS CARRIER TTSO
1669 CANADIAN SYLLABICS CARRIER TTSE
166A CANADIAN SYLLABICS CARRIER TTSEE
166B CANADIAN SYLLABICS CARRIER TTSI
166C CANADIAN SYLLABICS CARRIER TTSA
166D CANADIAN SYLLABICS CHI SIGN
166E CANADIAN SYLLABICS FULL STOP
166F CANADIAN SYLLABICS QAI
1670 CANADIAN SYLLABICS NGAI
1671 CANADIAN SYLLABICS NNGI
1672 CANADIAN SYLLABICS NNGII
1673 CANADIAN SYLLABICS NNGO
1674 CANADIAN SYLLABICS NNGOO
1675 CANADIAN SYLLABICS NNGA
1676 CANADIAN SYLLABICS NNGAA
1677 CANADIAN SYLLABICS WOODS-CREE THWEE
1678 CANADIAN SYLLABICS WOODS-CREE THWI
1679 CANADIAN SYLLABICS WOODS-CREE THWII
167A CANADIAN SYLLABICS WOODS-CREE THWO
167B CANADIAN SYLLABICS WOODS-CREE THWOO
167C CANADIAN SYLLABICS WOODS-CREE THWA
167D CANADIAN SYLLABICS WOODS-CREE THWAA
167E CANADIAN SYLLABICS WOODS-CREE FINAL TH
167F CANADIAN SYLLABICS BLACKFOOT W
1680 OGHAM SPACE MARK
1681 OGHAM LETTER BEITH
1682 OGHAM LETTER LUIS
1683 OGHAM LETTER FEARN
1684 OGHAM LETTER SAIL
1685 OGHAM LETTER NION
1686 OGHAM LETTER UATH
1687 OGHAM LETTER DAIR
1688 OGHAM LETTER TINNE
1689 OGHAM LETTER COLL
168A OGHAM LETTER CEIRT
168B OGHAM LETTER MUIN
168C OGHAM LETTER GORT
168D OGHAM LETTER NGEADAL
168E OGHAM LETTER STRAIF
168F OGHAM LETTER RUIS
1690 OGHAM LETTER AILM
1691 OGHAM LETTER ONN
1692 OGHAM LETTER UR
1693 OGHAM LETTER EADHADH
1694 OGHAM LETTER IODHADH
1695 OGHAM LETTER EABHADH
1696 OGHAM LETTER OR
1697 OGHAM LETTER UILLEANN
1698 OGHAM LETTER IFIN
1699 OGHAM LETTER EAMHANCHOLL
169A OGHAM LETTER PEITH
169B OGHAM FEATHER MARK
169C OGHAM REVERSED FEATHER MARK
16A0 RUNIC LETTER FEHU FEOH FE F
16A1 RUNIC LETTER V
16A2 RUNIC LETTER URUZ UR U
16A3 RUNIC LETTER YR
16A4 RUNIC LETTER Y
16A5 RUNIC LETTER W
16A6 RUNIC LETTER THURISAZ THURS THORN
16A7 RUNIC LETTER ETH
16A8 RUNIC LETTER ANSUZ A
16A9 RUNIC LETTER OS O
16AA RUNIC LETTER AC A
16AB RUNIC LETTER AESC
16AC RUNIC LETTER LONG-BRANCH-OSS O
16AD RUNIC LETTER SHORT-TWIG-OSS O
16AE RUNIC LETTER O
16AF RUNIC LETTER OE
16B0 RUNIC LETTER ON
16B1 RUNIC LETTER RAIDO RAD REID R
16B2 RUNIC LETTER KAUNA
16B3 RUNIC LETTER CEN
16B4 RUNIC LETTER KAUN K
16B5 RUNIC LETTER G
16B6 RUNIC LETTER ENG
16B7 RUNIC LETTER GEBO GYFU G
16B8 RUNIC LETTER GAR
16B9 RUNIC LETTER WUNJO WYNN W
16BA RUNIC LETTER HAGLAZ H
16BB RUNIC LETTER HAEGL H
16BC RUNIC LETTER LONG-BRANCH-HAGALL H
16BD RUNIC LETTER SHORT-TWIG-HAGALL H
16BE RUNIC LETTER NAUDIZ NYD NAUD N
16BF RUNIC LETTER SHORT-TWIG-NAUD N
16C0 RUNIC LETTER DOTTED-N
16C1 RUNIC LETTER ISAZ IS ISS I
16C2 RUNIC LETTER E
16C3 RUNIC LETTER JERAN J
16C4 RUNIC LETTER GER
16C5 RUNIC LETTER LONG-BRANCH-AR AE
16C6 RUNIC LETTER SHORT-TWIG-AR A
16C7 RUNIC LETTER IWAZ EOH
16C8 RUNIC LETTER PERTHO PEORTH P
16C9 RUNIC LETTER ALGIZ EOLHX
16CA RUNIC LETTER SOWILO S
16CB RUNIC LETTER SIGEL LONG-BRANCH-SOL S
16CC RUNIC LETTER SHORT-TWIG-SOL S
16CD RUNIC LETTER C
16CE RUNIC LETTER Z
16CF RUNIC LETTER TIWAZ TIR TYR T
16D0 RUNIC LETTER SHORT-TWIG-TYR T
16D1 RUNIC LETTER D
16D2 RUNIC LETTER BERKANAN BEORC BJARKAN B
16D3 RUNIC LETTER SHORT-TWIG-BJARKAN B
16D4 RUNIC LETTER DOTTED-P
16D5 RUNIC LETTER OPEN-P
16D6 RUNIC LETTER EHWAZ EH E
16D7 RUNIC LETTER MANNAZ MAN M
16D8 RUNIC LETTER LONG-BRANCH-MADR M
16D9 RUNIC LETTER SHORT-TWIG-MADR M
16DA RUNIC LETTER LAUKAZ LAGU LOGR L
16DB RUNIC LETTER DOTTED-L
16DC RUNIC LETTER INGWAZ
16DD RUNIC LETTER ING
16DE RUNIC LETTER DAGAZ DAEG D
16DF RUNIC LETTER OTHALAN ETHEL O
16E0 RUNIC LETTER EAR
16E1 RUNIC LETTER IOR
16E2 RUNIC LETTER CWEORTH
16E3 RUNIC LETTER CALC
16E4 RUNIC LETTER CEALC
16E5 RUNIC LETTER STAN
16E6 RUNIC LETTER LONG-BRANCH-YR
16E7 RUNIC LETTER SHORT-TWIG-YR
16E8 RUNIC LETTER ICELANDIC-YR
16E9 RUNIC LETTER Q
16EA RUNIC LETTER X
16EB RUNIC SINGLE PUNCTUATION
16EC RUNIC MULTIPLE PUNCTUATION
16ED RUNIC CROSS PUNCTUATION
16EE RUNIC ARLAUG SYMBOL
16EF RUNIC TVIMADUR SYMBOL
16F0 RUNIC BELGTHOR SYMBOL
16F1 RUNIC LETTER K
16F2 RUNIC LETTER SH
16F3 RUNIC LETTER OO
16F4 RUNIC LETTER FRANKS CASKET OS
16F5 RUNIC LETTER FRANKS CASKET IS
16F6 RUNIC LETTER FRANKS CASKET EH
16F7 RUNIC LETTER FRANKS CASKET AC
16F8 RUNIC LETTER FRANKS CASKET AESC
1700 TAGALOG LETTER A
1701 TAGALOG LETTER I
1702 TAGALOG LETTER U
1703 TAGALOG LETTER KA
1704 TAGALOG LETTER GA
1705 TAGALOG LETTER NGA
1706 TAGALOG LETTER TA
1707 TAGALOG LETTER DA
1708 TAGALOG LETTER NA
1709 TAGALOG LETTER PA
170A TAGALOG LETTER BA
170B TAGALOG LETTER MA
170C TAGALOG LETTER YA
170D TAGALOG LETTER RA
170E TAGALOG LETTER LA
170F TAGALOG LETTER WA
1710 TAGALOG LETTER SA
1711 TAGALOG LETTER HA
1712 TAGALOG VOWEL SIGN I
1713 TAGALOG VOWEL SIGN U
1714 TAGALOG SIGN VIRAMA
1715 TAGALOG SIGN PAMUDPOD
171F TAGALOG LETTER ARCHAIC RA
1720 HANUNOO LETTER A
1721 HANUNOO LETTER I
1722 HANUNOO LETTER U
1723 HANUNOO LETTER KA
1724 HANUNOO LETTER GA
1725 HANUNOO LETTER NGA
1726 HANUNOO LETTER TA
1727 HANUNOO LETTER DA
1728 HANUNOO LETTER NA
1729 HANUNOO LETTER PA
172A HANUNOO LETTER BA
172B HANUNOO LETTER MA
172C HANUNOO LETTER YA
172D HANUNOO LETTER RA
172E HANUNOO LETTER LA
172F HANUNOO LETTER WA
1730 HANUNOO LETTER SA
1731 HANUNOO LETTER HA
1732 HANUNOO VOWEL SIGN I
1733 HANUNOO VOWEL SIGN U
1734 HANUNOO SIGN PAMUDPOD
1735 PHILIPPINE SINGLE PUNCTUATION
1736 PHILIPPINE DOUBLE PUNCTUATION
1740 BUHID LETTER A
1741 BUHID LETTER I
1742 BUHID LETTER U
1743 BUHID LETTER KA
1744 BUHID LETTER GA
1745 BUHID LETTER NGA
1746 BUHID LETTER TA
1747 BUHID LETTER DA
1748 BUHID LETTER NA
1749 BUHID LETTER PA
174A BUHID LETTER BA
174B BUHID LETTER MA
174C BUHID LETTER YA
174D BUHID LETTER RA
174E BUHID LETTER LA
174F BUHID LETTER WA
1750 BUHID LETTER SA
1751 BUHID LETTER HA
1752 BUHID VOWEL SIGN I
1753 BUHID VOWEL SIGN U
1760 TAGBANWA LETTER A
1761 TAGBANWA LETTER I
1762 TAGBANWA LETTER U
1763 TAGBANWA LETTER KA
1764 TAGBANWA LETTER GA
1765 TAGBANWA LETTER NGA
1766 TAGBANWA LETTER TA
1767 TAGBANWA LETTER DA
1768 TAGBANWA LETTER NA
1769 TAGBANWA LETTER PA
176A TAGBANWA LETTER BA
176B TAGBANWA LETTER MA
176C TAGBANWA LETTER YA
176E TAGBANWA LETTER LA
176F TAGBANWA LETTER WA
1770 TAGBANWA LETTER SA
1772 TAGBANWA VOWEL SIGN I
1773 TAGBANWA VOWEL SIGN U
1780 KHMER LETTER KA
1781 KHMER LETTER KHA
1782 KHMER LETTER KO
1783 KHMER LETTER KHO
1784 KHMER LETTER NGO
1785 KHMER LETTER CA
1786 KHMER LETTER CHA
1787 KHMER LETTER CO
1788 KHMER LETTER CHO
1789 KHMER LETTER NYO
178A KHMER LETTER DA
178B KHMER LETTER TTHA
178C KHMER LETTER DO
178D KHMER LETTER TTHO
178E KHMER LETTER NNO
178F KHMER LETTER TA
1790 KHMER LETTER THA
1791 KHMER LETTER TO
1792 KHMER LETTER THO
1793 KHMER LETTER NO
1794 KHMER LETTER BA
1795 KHMER LETTER PHA
1796 KHMER LETTER PO
1797 KHMER LETTER PHO
1798 KHMER LETTER MO
1799 KHMER LETTER YO
179A KHMER LETTER RO
179B KHMER LETTER LO
179C KHMER LETTER VO
179D KHMER LETTER SHA
179E KHMER LETTER SSO
179F KHMER LETTER SA
17A0 KHMER LETTER HA
17A1 KHMER LETTER LA
17A2 KHMER LETTER QA
17A3 KHMER INDEPENDENT VOWEL QAQ
17A4 KHMER INDEPENDENT VOWEL QAA
17A5 KHMER INDEPENDENT VOWEL QI
17A6 KHMER INDEPENDENT VOWEL QII
17A7 KHMER INDEPENDENT VOWEL QU
17A8 KHMER INDEPENDENT VOWEL QUK
17A9 KHMER INDEPENDENT VOWEL QUU
17AA KHMER INDEPENDENT VOWEL QUUV
17AB KHMER INDEPENDENT VOWEL RY
17AC KHMER INDEPENDENT VOWEL RYY
17AD KHMER INDEPENDENT VOWEL LY
17AE KHMER INDEPENDENT VOWEL LYY
17AF KHMER INDEPENDENT VOWEL QE
17B0 KHMER INDEPENDENT VOWEL QAI
17B1 KHMER INDEPENDENT VOWEL QOO TYPE ONE
17B2 KHMER INDEPENDENT VOWEL QOO TYPE TWO
17B3 KHMER INDEPENDENT VOWEL QAU
17B4 KHMER VOWEL INHERENT AQ
17B5 KHMER VOWEL INHERENT AA
17B6 KHMER VOWEL SIGN AA
17B7 KHMER VOWEL SIGN I
17B8 KHMER VOWEL SIGN II
17B9 KHMER VOWEL SIGN Y
17BA KHMER VOWEL SIGN YY
17BB KHMER VOWEL SIGN U
17BC KHMER VOWEL SIGN UU
17BD KHMER VOWEL SIGN UA
17BE KHMER VOWEL SIGN OE
17BF KHMER VOWEL SIGN YA
17C0 KHMER VOWEL SIGN IE
17C1 KHMER VOWEL SIGN E
17C2 KHMER VOWEL SIGN AE
17C3 KHMER VOWEL SIGN AI
17C4 KHMER VOWEL SIGN OO
17C5 KHMER VOWEL SIGN AU
17C6 KHMER SIGN NIKAHIT
17C7 KHMER SIGN REAHMUK
17C8 KHMER SIGN YUUKALEAPINTU
17C9 KHMER SIGN MUUSIKATOAN
17CA KHMER SIGN TRIISAP
17CB KHMER SIGN BANTOC
17CC KHMER SIGN ROBAT
17CD KHMER SIGN TOANDAKHIAT
17CE KHMER SIGN KAKABAT
17CF KHMER SIGN AHSDA
17D0 KHMER SIGN SAMYOK SANNYA
17D1 KHMER SIGN VIRIAM
17D2 KHMER SIGN COENG
17D3 KHMER SIGN BATHAMASAT
17D4 KHMER SIGN KHAN
17D5 KHMER SIGN BARIYOOSAN
17D6 KHMER SIGN CAMNUC PII KUUH
17D7 KHMER SIGN LEK TOO
17D8 KHMER SIGN BEYYAL
17D9 KHMER SIGN PHNAEK MUAN
17DA KHMER SIGN KOOMUUT
17DB KHMER CURRENCY SYMBOL RIEL
17DC KHMER SIGN AVAKRAHASANYA
17DD KHMER SIGN ATTHACAN
17E0 KHMER DIGIT ZERO
17E1 KHMER DIGIT ONE
17E2 KHMER DIGIT TWO
17E3 KHMER DIGIT THREE
17E4 KHMER DIGIT FOUR
17E5 KHMER DIGIT FIVE
17E6 KHMER DIGIT SIX
17E7 KHMER DIGIT SEVEN
17E8 KHMER DIGIT EIGHT
17E9 KHMER DIGIT NINE
17F0 KHMER SYMBOL LEK ATTAK SON
17F1 KHMER SYMBOL LEK ATTAK MUOY
17F2 KHMER SYMBOL LEK ATTAK PII
17F3 KHMER SYMBOL LEK ATTAK BEI
17F4 KHMER SYMBOL LEK ATTAK BUON
17F5 KHMER SYMBOL LEK ATTAK PRAM
17F6 KHMER SYMBOL LEK ATTAK PRAM-MUOY
17F7 KHMER SYMBOL LEK ATTAK PRAM-PII
17F8 KHMER SYMBOL LEK ATTAK PRAM-BEI
17F9 KHMER SYMBOL LEK ATTAK PRAM-BUON
1800 MONGOLIAN BIRGA
1801 MONGOLIAN ELLIPSIS
1802 MONGOLIAN COMMA
1803 MONGOLIAN FULL STOP
1804 MONGOLIAN COLON
1805 MONGOLIAN FOUR DOTS
1806 MONGOLIAN TODO SOFT HYPHEN
1807 MONGOLIAN SIBE SYLLABLE BOUNDARY MARKER
1808 MONGOLIAN MANCHU COMMA
1809 MONGOLIAN MANCHU FULL STOP
180A MONGOLIAN NIRUGU
180B MONGOLIAN FREE VARIATION SELECTOR ONE, FVS1
180C MONGOLIAN FREE VARIATION SELECTOR TWO, FVS2
180D MONGOLIAN FREE VARIATION SELECTOR THREE, FVS3
180E MONGOLIAN VOWEL SEPARATOR, MVS
180F MONGOLIAN FREE VARIATION SELECTOR FOUR, FVS4
1810 MONGOLIAN DIGIT ZERO
1811 MONGOLIAN DIGIT ONE
1812 MONGOLIAN DIGIT TWO
1813 MONGOLIAN DIGIT THREE
1814 MONGOLIAN DIGIT FOUR
1815 MONGOLIAN DIGIT FIVE
1816 MONGOLIAN DIGIT SIX
1817 MONGOLIAN DIGIT SEVEN
1818 MONGOLIAN DIGIT EIGHT
1819 MONGOLIAN DIGIT NINE
1820 MONGOLIAN LETTER A
1821 MONGOLIAN LETTER E
1822 MONGOLIAN LETTER I
1823 MONGOLIAN LETTER O
1824 MONGOLIAN LETTER U
1825 MONGOLIAN LETTER OE
1826 MONGOLIAN LETTER UE
1827 MONGOLIAN LETTER EE
1828 MONGOLIAN LETTER NA
1829 MONGOLIAN LETTER ANG
182A MONGOLIAN LETTER BA
182B MONGOLIAN LETTER PA
182C MONGOLIAN LETTER QA
182D MONGOLIAN LETTER GA
182E MONGOLIAN LETTER MA
182F MONGOLIAN LETTER LA
1830 MONGOLIAN LETTER SA
1831 MONGOLIAN LETTER SHA
1832 MONGOLIAN LETTER TA
1833 MONGOLIAN LETTER DA
1834 MONGOLIAN LETTER CHA
1835 MONGOLIAN LETTER JA
1836 MONGOLIAN LETTER YA
1837 MONGOLIAN LETTER RA
1838 MONGOLIAN LETTER WA
1839 MONGOLIAN LETTER FA
183A MONGOLIAN LETTER KA
183B MONGOLIAN LETTER KHA
183C MONGOLIAN LETTER TSA
183D MONGOLIAN LETTER ZA
183E MONGOLIAN LETTER HAA
183F MONGOLIAN LETTER ZRA
1840 MONGOLIAN LETTER LHA
1841 MONGOLIAN LETTER ZHI
1842 MONGOLIAN LETTER CHI
1843 MONGOLIAN LETTER TODO LONG VOWEL SIGN
1844 MONGOLIAN LETTER TODO E
1845 MONGOLIAN LETTER TODO I
1846 MONGOLIAN LETTER TODO O
1847 MONGOLIAN LETTER TODO U
1848 MONGOLIAN LETTER TODO OE
1849 MONGOLIAN LETTER TODO UE
184A MONGOLIAN LETTER TODO ANG
184B MONGOLIAN LETTER TODO BA
184C MONGOLIAN LETTER TODO PA
184D MONGOLIAN LETTER TODO QA
184E MONGOLIAN LETTER TODO GA
184F MONGOLIAN LETTER TODO MA
1850 MONGOLIAN LETTER TODO TA
1851 MONGOLIAN LETTER TODO DA
1852 MONGOLIAN LETTER TODO CHA
1853 MONGOLIAN LETTER TODO JA
1854 MONGOLIAN LETTER TODO TSA
1855 MONGOLIAN LETTER TODO YA
1856 MONGOLIAN LETTER TODO WA
1857 MONGOLIAN LETTER TODO KA
1858 MONGOLIAN LETTER TODO GAA
1859 MONGOLIAN LETTER TODO HAA
185A MONGOLIAN LETTER TODO JIA
185B MONGOLIAN LETTER TODO NIA
185C MONGOLIAN LETTER TODO DZA
185D MONGOLIAN LETTER SIBE E
185E MONGOLIAN LETTER SIBE I
185F MONGOLIAN LETTER SIBE IY
1860 MONGOLIAN LETTER SIBE UE
1861 MONGOLIAN LETTER SIBE U
1862 MONGOLIAN LETTER SIBE ANG
1863 MONGOLIAN LETTER SIBE KA
1864 MONGOLIAN LETTER SIBE GA
1865 MONGOLIAN LETTER SIBE HA
1866 MONGOLIAN LETTER SIBE PA
1867 MONGOLIAN LETTER SIBE SHA
1868 MONGOLIAN LETTER SIBE TA
1869 MONGOLIAN LETTER SIBE DA
186A MONGOLIAN LETTER SIBE JA
186B MONGOLIAN LETTER SIBE FA
186C MONGOLIAN LETTER SIBE GAA
186D MONGOLIAN LETTER SIBE HAA
186E MONGOLIAN LETTER SIBE TSA
186F MONGOLIAN LETTER SIBE ZA
1870 MONGOLIAN LETTER SIBE RAA
1871 MONGOLIAN LETTER SIBE CHA
1872 MONGOLIAN LETTER SIBE ZHA
1873 MONGOLIAN LETTER MANCHU I
1874 MONGOLIAN LETTER MANCHU KA
1875 MONGOLIAN LETTER MANCHU RA
1876 MONGOLIAN LETTER MANCHU FA
1877 MONGOLIAN LETTER MANCHU ZHA
1878 MONGOLIAN LETTER CHA WITH TWO DOTS
1880 MONGOLIAN LETTER ALI GALI ANUSVARA ONE
1881 MONGOLIAN LETTER ALI GALI VISARGA ONE
1882 MONGOLIAN LETTER ALI GALI DAMARU
1883 MONGOLIAN LETTER ALI GALI UBADAMA
1884 MONGOLIAN LETTER ALI GALI INVERTED UBADAMA
1885 MONGOLIAN LETTER ALI GALI BALUDA
1886 MONGOLIAN LETTER ALI GALI THREE BALUDA
1887 MONGOLIAN LETTER ALI GALI A
1888 MONGOLIAN LETTER ALI GALI I
1889 MONGOLIAN LETTER ALI GALI KA
188A MONGOLIAN LETTER ALI GALI NGA
188B MONGOLIAN LETTER ALI GALI CA
188C MONGOLIAN LETTER ALI GALI TTA
188D MONGOLIAN LETTER ALI GALI TTHA
188E MONGOLIAN LETTER ALI GALI DDA
188F MONGOLIAN LETTER ALI GALI NNA
1890 MONGOLIAN LETTER ALI GALI TA
1891 MONGOLIAN LETTER ALI GALI DA
1892 MONGOLIAN LETTER ALI GALI PA
1893 MONGOLIAN LETTER ALI GALI PHA
1894 MONGOLIAN LETTER ALI GALI SSA
1895 MONGOLIAN LETTER ALI GALI ZHA
1896 MONGOLIAN LETTER ALI GALI ZA
1897 MONGOLIAN LETTER ALI GALI AH
1898 MONGOLIAN LETTER TODO ALI GALI TA
1899 MONGOLIAN LETTER TODO ALI GALI ZHA
189A MONGOLIAN LETTER MANCHU ALI GALI GHA
189B MONGOLIAN LETTER MANCHU ALI GALI NGA
189C MONGOLIAN LETTER MANCHU ALI GALI CA
189D MONGOLIAN LETTER MANCHU ALI GALI JHA
189E MONGOLIAN LETTER MANCHU ALI GALI TTA
189F MONGOLIAN LETTER MANCHU ALI GALI DDHA
18A0 MONGOLIAN LETTER MANCHU ALI GALI TA
18A1 MONGOLIAN LETTER MANCHU ALI GALI DHA
18A2 MONGOLIAN LETTER MANCHU ALI GALI SSA
18A3 MONGOLIAN LETTER MANCHU ALI GALI CYA
18A4 MONGOLIAN LETTER MANCHU ALI GALI ZHA
18A5 MONGOLIAN LETTER MANCHU ALI GALI ZA
18A6 MONGOLIAN LETTER ALI GALI HALF U
18A7 MONGOLIAN LETTER ALI GALI HALF YA
18A8 MONGOLIAN LETTER MANCHU ALI GALI BHA
18A9 MONGOLIAN LETTER ALI GALI DAGALGA
18AA MONGOLIAN LETTER MANCHU ALI GALI LHA
18B0 CANADIAN SYLLABICS OY
18B1 CANADIAN SYLLABICS AY
18B2 CANADIAN SYLLABICS AAY
18B3 CANADIAN SYLLABICS WAY
18B4 CANADIAN SYLLABICS POY
18B5 CANADIAN SYLLABICS PAY
18B6 CANADIAN SYLLABICS PWOY
18B7 CANADIAN SYLLABICS TAY
18B8 CANADIAN SYLLABICS KAY
18B9 CANADIAN SYLLABICS KWAY
18BA CANADIAN SYLLABICS MAY
18BB CANADIAN SYLLABICS NOY
18BC CANADIAN SYLLABICS NAY
18BD CANADIAN SYLLABICS LAY
18BE CANADIAN SYLLABICS SOY
18BF CANADIAN SYLLABICS SAY
18C0 CANADIAN SYLLABICS SHOY
18C1 CANADIAN SYLLABICS SHAY
18C2 CANADIAN SYLLABICS SHWOY
18C3 CANADIAN SYLLABICS YOY
18C4 CANADIAN SYLLABICS YAY
18C5 CANADIAN SYLLABICS RAY
18C6 CANADIAN SYLLABICS NWI
18C7 CANADIAN SYLLABICS OJIBWAY NWI
18C8 CANADIAN SYLLABICS NWII
18C9 CANADIAN SYLLABICS OJIBWAY NWII
18CA CANADIAN SYLLABICS NWO
18CB CANADIAN SYLLABICS OJIBWAY NWO
18CC CANADIAN SYLLABICS NWOO
18CD CANADIAN SYLLABICS OJIBWAY NWOO
18CE CANADIAN SYLLABICS RWEE
18CF CANADIAN SYLLABICS RWI
18D0 CANADIAN SYLLABICS RWII
18D1 CANADIAN SYLLABICS RWO
18D2 CANADIAN SYLLABICS RWOO
18D3 CANADIAN SYLLABICS RWA
18D4 CANADIAN SYLLABICS OJIBWAY P
18D5 CANADIAN SYLLABICS OJIBWAY T
18D6 CANADIAN SYLLABICS OJIBWAY K
18D7 CANADIAN SYLLABICS OJIBWAY C
18D8 CANADIAN SYLLABICS OJIBWAY M
18D9 CANADIAN SYLLABICS OJIBWAY N
18DA CANADIAN SYLLABICS OJIBWAY S
18DB CANADIAN SYLLABICS OJIBWAY SH
18DC CANADIAN SYLLABICS EASTERN W
18DD CANADIAN SYLLABICS WESTERN W
18DE CANADIAN SYLLABICS FINAL SMALL RING
18DF CANADIAN SYLLABICS FINAL RAISED DOT
18E0 CANADIAN SYLLABICS R-CREE RWE
18E1 CANADIAN SYLLABICS WEST-CREE LOO
18E2 CANADIAN SYLLABICS WEST-CREE LAA
18E3 CANADIAN SYLLABICS THWE
18E4 CANADIAN SYLLABICS THWA
18E5 CANADIAN SYLLABICS TTHWE
18E6 CANADIAN SYLLABICS TTHOO
18E7 CANADIAN SYLLABICS TTHAA
18E8 CANADIAN SYLLABICS TLHWE
18E9 CANADIAN SYLLABICS TLHOO
18EA CANADIAN SYLLABICS SAYISI SHWE
18EB CANADIAN SYLLABICS SAYISI SHOO
18EC CANADIAN SYLLABICS SAYISI HOO
18ED CANADIAN SYLLABICS CARRIER GWU
18EE CANADIAN SYLLABICS CARRIER DENE GEE
18EF CANADIAN SYLLABICS CARRIER GAA
18F0 CANADIAN SYLLABICS CARRIER GWA
18F1 CANADIAN SYLLABICS SAYISI JUU
18F2 CANADIAN SYLLABICS CARRIER JWA
18F3 CANADIAN SYLLABICS BEAVER DENE L
18F4 CANADIAN SYLLABICS BEAVER DENE R
18F5 CANADIAN SYLLABICS CARRIER DENTAL S
1900 LIMBU VOWEL-CARRIER LETTER
1901 LIMBU LETTER KA
1902 LIMBU LETTER KHA
1903 LIMBU LETTER GA
1904 LIMBU LETTER GHA
1905 LIMBU LETTER NGA
1906 LIMBU LETTER CA
1907 LIMBU LETTER CHA
1908 LIMBU LETTER JA
1909 LIMBU LETTER JHA
190A LIMBU LETTER YAN
190B LIMBU LETTER TA
190C LIMBU LETTER THA
190D LIMBU LETTER DA
190E LIMBU LETTER DHA
190F LIMBU LETTER NA
1910 LIMBU LETTER PA
1911 LIMBU LETTER PHA
1912 LIMBU LETTER BA
1913 LIMBU LETTER BHA
1914 LIMBU LETTER MA
1915 LIMBU LETTER YA
1916 LIMBU LETTER RA
1917 LIMBU LETTER LA
1918 LIMBU LETTER WA
1919 LIMBU LETTER SHA
191A LIMBU LETTER SSA
191B LIMBU LETTER SA
191C LIMBU LETTER HA
191D LIMBU LETTER GYAN
191E LIMBU LETTER TRA
1920 LIMBU VOWEL SIGN A
1921 LIMBU VOWEL SIGN I
1922 LIMBU VOWEL SIGN U
1923 LIMBU VOWEL SIGN EE
1924 LIMBU VOWEL SIGN AI
1925 LIMBU VOWEL SIGN OO
1926 LIMBU VOWEL SIGN AU
1927 LIMBU VOWEL SIGN E
1928 LIMBU VOWEL SIGN O
1929 LIMBU SUBJOINED LETTER YA
192A LIMBU SUBJOINED LETTER RA
192B LIMBU SUBJOINED LETTER WA
1930 LIMBU SMALL LETTER KA
1931 LIMBU SMALL LETTER NGA
1932 LIMBU SMALL LETTER ANUSVARA
1933 LIMBU SMALL LETTER TA
1934 LIMBU SMALL LETTER NA
1935 LIMBU SMALL LETTER PA
1936 LIMBU SMALL LETTER MA
1937 LIMBU SMALL LETTER RA
1938 LIMBU SMALL LETTER LA
1939 LIMBU SIGN MUKPHRENG
193A LIMBU SIGN KEMPHRENG
193B LIMBU SIGN SA-I
1940 LIMBU SIGN LOO
1944 LIMBU EXCLAMATION MARK
1945 LIMBU QUESTION MARK
1946 LIMBU DIGIT ZERO
1947 LIMBU DIGIT ONE
1948 LIMBU DIGIT TWO
1949 LIMBU DIGIT THREE
194A LIMBU DIGIT FOUR
194B LIMBU DIGIT FIVE
194C LIMBU DIGIT SIX
194D LIMBU DIGIT SEVEN
194E LIMBU DIGIT EIGHT
194F LIMBU DIGIT NINE
1950 TAI LE LETTER KA
1951 TAI LE LETTER XA
1952 TAI LE LETTER NGA
1953 TAI LE LETTER TSA
1954 TAI LE LETTER SA
1955 TAI LE LETTER YA
1956 TAI LE LETTER TA
1957 TAI LE LETTER THA
1958 TAI LE LETTER LA
1959 TAI LE LETTER PA
195A TAI LE LETTER PHA
195B TAI LE LETTER MA
195C TAI LE LETTER FA
195D TAI LE LETTER VA
195E TAI LE LETTER HA
195F TAI LE LETTER QA
1960 TAI LE LETTER KHA
1961 TAI LE LETTER TSHA
1962 TAI LE LETTER NA
1963 TAI LE LETTER A
1964 TAI LE LETTER I
1965 TAI LE LETTER EE
1966 TAI LE LETTER EH
1967 TAI LE LETTER U
1968 TAI LE LETTER OO
1969 TAI LE LETTER O
196A TAI LE LETTER UE
196B TAI LE LETTER E
196C TAI LE LETTER AUE
196D TAI LE LETTER AI
1970 TAI LE LETTER TONE-2
1971 TAI LE LETTER TONE-3
1972 TAI LE LETTER TONE-4
1973 TAI LE LETTER TONE-5
1974 TAI LE LETTER TONE-6
1980 NEW TAI LUE LETTER HIGH QA
1981 NEW TAI LUE LETTER LOW QA
1982 NEW TAI LUE LETTER HIGH KA
1983 NEW TAI LUE LETTER HIGH XA
1984 NEW TAI LUE LETTER HIGH NGA
1985 NEW TAI LUE LETTER LOW KA
1986 NEW TAI LUE LETTER LOW XA
1987 NEW TAI LUE LETTER LOW NGA
1988 NEW TAI LUE LETTER HIGH TSA
1989 NEW TAI LUE LETTER HIGH SA
198A NEW TAI LUE LETTER HIGH YA
198B NEW TAI LUE LETTER LOW TSA
198C NEW TAI LUE LETTER LOW SA
198D NEW TAI LUE LETTER LOW YA
198E NEW TAI LUE LETTER HIGH TA
198F NEW TAI LUE LETTER HIGH THA
1990 NEW TAI LUE LETTER HIGH NA
1991 NEW TAI LUE LETTER LOW TA
1992 NEW TAI LUE LETTER LOW THA
1993 NEW TAI LUE LETTER LOW NA
1994 NEW TAI LUE LETTER HIGH PA
1995 NEW TAI LUE LETTER HIGH PHA
1996 NEW TAI LUE LETTER HIGH MA
1997 NEW TAI LUE LETTER LOW PA
1998 NEW TAI LUE LETTER LOW PHA
1999 NEW TAI LUE LETTER LOW MA
199A NEW TAI LUE LETTER HIGH FA
199B NEW TAI LUE LETTER HIGH VA
199C NEW TAI LUE LETTER HIGH LA
199D NEW TAI LUE LETTER LOW FA
199E NEW TAI LUE LETTER LOW VA
199F NEW TAI LUE LETTER LOW LA
19A0 NEW TAI LUE LETTER HIGH HA
19A1 NEW TAI LUE LETTER HIGH DA
19A2 NEW TAI LUE LETTER HIGH BA
19A3 NEW TAI LUE LETTER LOW HA
19A4 NEW TAI LUE LETTER LOW DA
19A5 NEW TAI LUE LETTER LOW BA
19A6 NEW TAI LUE LETTER HIGH KVA
19A7 NEW TAI LUE LETTER HIGH XVA
19A8 NEW TAI LUE LETTER LOW KVA
19A9 NEW TAI LUE LETTER LOW XVA
19AA NEW TAI LUE LETTER HIGH SUA
19AB NEW TAI LUE LETTER LOW SUA
19B0 NEW TAI LUE VOWEL SIGN VOWEL SHORTENER
19B1 NEW TAI LUE VOWEL SIGN AA
19B2 NEW TAI LUE VOWEL SIGN II
19B3 NEW TAI LUE VOWEL SIGN U
19B4 NEW TAI LUE VOWEL SIGN UU
19B5 NEW TAI LUE VOWEL SIGN E
19B6 NEW TAI LUE VOWEL SIGN AE
19B7 NEW TAI LUE VOWEL SIGN O
19B8 NEW TAI LUE VOWEL SIGN OA
19B9 NEW TAI LUE VOWEL SIGN UE
19BA NEW TAI LUE VOWEL SIGN AY
19BB NEW TAI LUE VOWEL SIGN AAY
19BC NEW TAI LUE VOWEL SIGN UY
19BD NEW TAI LUE VOWEL SIGN OY
19BE NEW TAI LUE VOWEL SIGN OAY
19BF NEW TAI LUE VOWEL SIGN UEY
19C0 NEW TAI LUE VOWEL SIGN IY
19C1 NEW TAI LUE LETTER FINAL V
19C2 NEW TAI LUE LETTER FINAL NG
19C3 NEW TAI LUE LETTER FINAL N
19C4 NEW TAI LUE LETTER FINAL M
19C5 NEW TAI LUE LETTER FINAL K
19C6 NEW TAI LUE LETTER FINAL D
19C7 NEW TAI LUE LETTER FINAL B
19C8 NEW TAI LUE TONE MARK-1
19C9 NEW TAI LUE TONE MARK-2
19D0 NEW TAI LUE DIGIT ZERO
19D1 NEW TAI LUE DIGIT ONE
19D2 NEW TAI LUE DIGIT TWO
19D3 NEW TAI LUE DIGIT THREE
19D4 NEW TAI LUE DIGIT FOUR
19D5 NEW TAI LUE DIGIT FIVE
19D6 NEW TAI LUE DIGIT SIX
19D7 NEW TAI LUE DIGIT SEVEN
19D8 NEW TAI LUE DIGIT EIGHT
19D9 NEW TAI LUE DIGIT NINE
19DA NEW TAI LUE THAM DIGIT ONE
19DE NEW TAI LUE SIGN LAE
19DF NEW TAI LUE SIGN LAEV
19E0 KHMER SYMBOL PATHAMASAT
19E1 KHMER SYMBOL MUOY KOET
19E2 KHMER SYMBOL PII KOET
19E3 KHMER SYMBOL BEI KOET
19E4 KHMER SYMBOL BUON KOET
19E5 KHMER SYMBOL PRAM KOET
19E6 KHMER SYMBOL PRAM-MUOY KOET
19E7 KHMER SYMBOL PRAM-PII KOET
19E8 KHMER SYMBOL PRAM-BEI KOET
19E9 KHMER SYMBOL PRAM-BUON KOET
19EA KHMER SYMBOL DAP KOET
19EB KHMER SYMBOL DAP-MUOY KOET
19EC KHMER SYMBOL DAP-PII KOET
19ED KHMER SYMBOL DAP-BEI KOET
19EE KHMER SYMBOL DAP-BUON KOET
19EF KHMER SYMBOL DAP-PRAM KOET
19F0 KHMER SYMBOL TUTEYASAT
19F1 KHMER SYMBOL MUOY ROC
19F2 KHMER SYMBOL PII ROC
19F3 KHMER SYMBOL BEI ROC
19F4 KHMER SYMBOL BUON ROC
19F5 KHMER SYMBOL PRAM ROC
19F6 KHMER SYMBOL PRAM-MUOY ROC
19F7 KHMER SYMBOL PRAM-PII ROC
19F8 KHMER SYMBOL PRAM-BEI ROC
19F9 KHMER SYMBOL PRAM-BUON ROC
19FA KHMER SYMBOL DAP ROC
19FB KHMER SYMBOL DAP-MUOY ROC
19FC KHMER SYMBOL DAP-PII ROC
19FD KHMER SYMBOL DAP-BEI ROC
19FE KHMER SYMBOL DAP-BUON ROC
19FF KHMER SYMBOL DAP-PRAM ROC
1A00 BUGINESE LETTER KA
1A01 BUGINESE LETTER GA
1A02 BUGINESE LETTER NGA
1A03 BUGINESE LETTER NGKA
1A04 BUGINESE LETTER PA
1A05 BUGINESE LETTER BA
1A06 BUGINESE LETTER MA
1A07 BUGINESE LETTER MPA
1A08 BUGINESE LETTER TA
1A09 BUGINESE LETTER DA
1A0A BUGINESE LETTER NA
1A0B BUGINESE LETTER NRA
1A0C BUGINESE LETTER CA
1A0D BUGINESE LETTER JA
1A0E BUGINESE LETTER NYA
1A0F BUGINESE LETTER NYCA
1A10 BUGINESE LETTER YA
1A11 BUGINESE LETTER RA
1A12 BUGINESE LETTER LA
1A13 BUGINESE LETTER VA
1A14 BUGINESE LETTER SA
1A15 BUGINESE LETTER A
1A16 BUGINESE LETTER HA
1A17 BUGINESE VOWEL SIGN I
1A18 BUGINESE VOWEL SIGN U
1A19 BUGINESE VOWEL SIGN E
1A1A BUGINESE VOWEL SIGN O
1A1B BUGINESE VOWEL SIGN AE
1A1E BUGINESE PALLAWA
1A1F BUGINESE END OF SECTION
1A20 TAI THAM LETTER HIGH KA
1A21 TAI THAM LETTER HIGH KHA
1A22 TAI THAM LETTER HIGH KXA
1A23 TAI THAM LETTER LOW KA
1A24 TAI THAM LETTER LOW KXA
1A25 TAI THAM LETTER LOW KHA
1A26 TAI THAM LETTER NGA
1A27 TAI THAM LETTER HIGH CA
1A28 TAI THAM LETTER HIGH CHA
1A29 TAI THAM LETTER LOW CA
1A2A TAI THAM LETTER LOW SA
1A2B TAI THAM LETTER LOW CHA
1A2C TAI THAM LETTER NYA
1A2D TAI THAM LETTER RATA
1A2E TAI THAM LETTER HIGH RATHA
1A2F TAI THAM LETTER DA
1A30 TAI THAM LETTER LOW RATHA
1A31 TAI THAM LETTER RANA
1A32 TAI THAM LETTER HIGH TA
1A33 TAI THAM LETTER HIGH THA
1A34 TAI THAM LETTER LOW TA
1A35 TAI THAM LETTER LOW THA
1A36 TAI THAM LETTER NA
1A37 TAI THAM LETTER BA
1A38 TAI THAM LETTER HIGH PA
1A39 TAI THAM LETTER HIGH PHA
1A3A TAI THAM LETTER HIGH FA
1A3B TAI THAM LETTER LOW PA
1A3C TAI THAM LETTER LOW FA
1A3D TAI THAM LETTER LOW PHA
1A3E TAI THAM LETTER MA
1A3F TAI THAM LETTER LOW YA
1A40 TAI THAM LETTER HIGH YA
1A41 TAI THAM LETTER RA
1A42 TAI THAM LETTER RUE
1A43 TAI THAM LETTER LA
1A44 TAI THAM LETTER LUE
1A45 TAI THAM LETTER WA
1A46 TAI THAM LETTER HIGH SHA
1A47 TAI THAM LETTER HIGH SSA
1A48 TAI THAM LETTER HIGH SA
1A49 TAI THAM LETTER HIGH HA
1A4A TAI THAM LETTER LLA
1A4B TAI THAM LETTER A
1A4C TAI THAM LETTER LOW HA
1A4D TAI THAM LETTER I
1A4E TAI THAM LETTER II
1A4F TAI THAM LETTER U
1A50 TAI THAM LETTER UU
1A51 TAI THAM LETTER EE
1A52 TAI THAM LETTER OO
1A53 TAI THAM LETTER LAE
1A54 TAI THAM LETTER GREAT SA
1A55 TAI THAM CONSONANT SIGN MEDIAL RA
1A56 TAI THAM CONSONANT SIGN MEDIAL LA
1A57 TAI THAM CONSONANT SIGN LA TANG LAI
1A58 TAI THAM SIGN MAI KANG LAI
1A59 TAI THAM CONSONANT SIGN FINAL NGA
1A5A TAI THAM CONSONANT SIGN LOW PA
1A5B TAI THAM CONSONANT SIGN HIGH RATHA OR LOW PA
1A5C TAI THAM CONSONANT SIGN MA
1A5D TAI THAM CONSONANT SIGN BA
1A5E TAI THAM CONSONANT SIGN SA
1A60 TAI THAM SIGN SAKOT
1A61 TAI THAM VOWEL SIGN A
1A62 TAI THAM VOWEL SIGN MAI SAT
1A63 TAI THAM VOWEL SIGN AA
1A64 TAI THAM VOWEL SIGN TALL AA
1A65 TAI THAM VOWEL SIGN I
1A66 TAI THAM VOWEL SIGN II
1A67 TAI THAM VOWEL SIGN UE
1A68 TAI THAM VOWEL SIGN UUE
1A69 TAI THAM VOWEL SIGN U
1A6A TAI THAM VOWEL SIGN UU
1A6B TAI THAM VOWEL SIGN O
1A6C TAI THAM VOWEL SIGN OA BELOW
1A6D TAI THAM VOWEL SIGN OY
1A6E TAI THAM VOWEL SIGN E
1A6F TAI THAM VOWEL SIGN AE
1A70 TAI THAM VOWEL SIGN OO
1A71 TAI THAM VOWEL SIGN AI
1A72 TAI THAM VOWEL SIGN THAM AI
1A73 TAI THAM VOWEL SIGN OA ABOVE
1A74 TAI THAM SIGN MAI KANG
1A75 TAI THAM SIGN TONE-1
1A76 TAI THAM SIGN TONE-2
1A77 TAI THAM SIGN KHUEN TONE-3
1A78 TAI THAM SIGN KHUEN TONE-4
1A79 TAI THAM SIGN KHUEN TONE-5
1A7A TAI THAM SIGN RA HAAM
1A7B TAI THAM SIGN MAI SAM
1A7C TAI THAM SIGN KHUEN-LUE KARAN
1A7F TAI THAM COMBINING CRYPTOGRAMMIC DOT
1A80 TAI THAM HORA DIGIT ZERO
1A81 TAI THAM HORA DIGIT ONE
1A82 TAI THAM HORA DIGIT TWO
1A83 TAI THAM HORA DIGIT THREE
1A84 TAI THAM HORA DIGIT FOUR
1A85 TAI THAM HORA DIGIT FIVE
1A86 TAI THAM HORA DIGIT SIX
1A87 TAI THAM HORA DIGIT SEVEN
1A88 TAI THAM HORA DIGIT EIGHT
1A89 TAI THAM HORA DIGIT NINE
1A90 TAI THAM THAM DIGIT ZERO
1A91 TAI THAM THAM DIGIT ONE
1A92 TAI THAM THAM DIGIT TWO
1A93 TAI THAM THAM DIGIT THREE
1A94 TAI THAM THAM DIGIT FOUR
1A95 TAI THAM THAM DIGIT FIVE
1A96 TAI THAM THAM DIGIT SIX
1A97 TAI THAM THAM DIGIT SEVEN
1A98 TAI THAM THAM DIGIT EIGHT
1A99 TAI THAM THAM DIGIT NINE
1AA0 TAI THAM SIGN WIANG
1AA1 TAI THAM SIGN WIANGWAAK
1AA2 TAI THAM SIGN SAWAN
1AA3 TAI THAM SIGN KEOW
1AA4 TAI THAM SIGN HOY
1AA5 TAI THAM SIGN DOKMAI
1AA6 TAI THAM SIGN REVERSED ROTATED RANA
1AA7 TAI THAM SIGN MAI YAMOK
1AA8 TAI THAM SIGN KAAN
1AA9 TAI THAM SIGN KAANKUU
1AAA TAI THAM SIGN SATKAAN
1AAB TAI THAM SIGN SATKAANKUU
1AAC TAI THAM SIGN HANG
1AAD TAI THAM SIGN CAANG
1AB0 COMBINING DOUBLED CIRCUMFLEX ACCENT
1AB1 COMBINING DIAERESIS-RING
1AB2 COMBINING INFINITY
1AB3 COMBINING DOWNWARDS ARROW
1AB4 COMBINING TRIPLE DOT
1AB5 COMBINING X-X BELOW
1AB6 COMBINING WIGGLY LINE BELOW
1AB7 COMBINING OPEN MARK BELOW
1AB8 COMBINING DOUBLE OPEN MARK BELOW
1AB9 COMBINING LIGHT CENTRALIZATION STROKE BELOW
1ABA COMBINING STRONG CENTRALIZATION STROKE BELOW
1ABB COMBINING PARENTHESES ABOVE
1ABC COMBINING DOUBLE PARENTHESES ABOVE
1ABD COMBINING PARENTHESES BELOW
1ABE COMBINING PARENTHESES OVERLAY
1ABF COMBINING LATIN SMALL LETTER W BELOW
1AC0 COMBINING LATIN SMALL LETTER TURNED W BELOW
1AC1 COMBINING LEFT PARENTHESIS ABOVE LEFT
1AC2 COMBINING RIGHT PARENTHESIS ABOVE RIGHT
1AC3 COMBINING LEFT PARENTHESIS BELOW LEFT
1AC4 COMBINING RIGHT PARENTHESIS BELOW RIGHT
1AC5 COMBINING SQUARE BRACKETS ABOVE
1AC6 COMBINING NUMBER SIGN ABOVE
1AC7 COMBINING INVERTED DOUBLE ARCH ABOVE
1AC8 COMBINING PLUS SIGN ABOVE
1AC9 COMBINING DOUBLE PLUS SIGN ABOVE
1ACA COMBINING DOUBLE PLUS SIGN BELOW
1ACB COMBINING TRIPLE ACUTE ACCENT
1ACC COMBINING LATIN SMALL LETTER INSULAR G
1ACD COMBINING LATIN SMALL LETTER INSULAR R
1ACE COMBINING LATIN SMALL LETTER INSULAR T
1B00 BALINESE SIGN ULU RICEM
1B01 BALINESE SIGN ULU CANDRA
1B02 BALINESE SIGN CECEK
1B03 BALINESE SIGN SURANG
1B04 BALINESE SIGN BISAH
1B05 BALINESE LETTER AKARA
1B06 BALINESE LETTER AKARA TEDUNG
1B07 BALINESE LETTER IKARA
1B08 BALINESE LETTER IKARA TEDUNG
1B09 BALINESE LETTER UKARA
1B0A BALINESE LETTER UKARA TEDUNG
1B0B BALINESE LETTER RA REPA
1B0C BALINESE LETTER RA REPA TEDUNG
1B0D BALINESE LETTER LA LENGA
1B0E BALINESE LETTER LA LENGA TEDUNG
1B0F BALINESE LETTER EKARA
1B10 BALINESE LETTER AIKARA
1B11 BALINESE LETTER OKARA
1B12 BALINESE LETTER OKARA TEDUNG
1B13 BALINESE LETTER KA
1B14 BALINESE LETTER KA MAHAPRANA
1B15 BALINESE LETTER GA
1B16 BALINESE LETTER GA GORA
1B17 BALINESE LETTER NGA
1B18 BALINESE LETTER CA
1B19 BALINESE LETTER CA LACA
1B1A BALINESE LETTER JA
1B1B BALINESE LETTER JA JERA
1B1C BALINESE LETTER NYA
1B1D BALINESE LETTER TA LATIK
1B1E BALINESE LETTER TA MURDA MAHAPRANA
1B1F BALINESE LETTER DA MURDA ALPAPRANA
1B20 BALINESE LETTER DA MURDA MAHAPRANA
1B21 BALINESE LETTER NA RAMBAT
1B22 BALINESE LETTER TA
1B23 BALINESE LETTER TA TAWA
1B24 BALINESE LETTER DA
1B25 BALINESE LETTER DA MADU
1B26 BALINESE LETTER NA
1B27 BALINESE LETTER PA
1B28 BALINESE LETTER PA KAPAL
1B29 BALINESE LETTER BA
1B2A BALINESE LETTER BA KEMBANG
1B2B BALINESE LETTER MA
1B2C BALINESE LETTER YA
1B2D BALINESE LETTER RA
1B2E BALINESE LETTER LA
1B2F BALINESE LETTER WA
1B30 BALINESE LETTER SA SAGA
1B31 BALINESE LETTER SA SAPA
1B32 BALINESE LETTER SA
1B33 BALINESE LETTER HA
1B34 BALINESE SIGN REREKAN
1B35 BALINESE VOWEL SIGN TEDUNG
1B36 BALINESE VOWEL SIGN ULU
1B37 BALINESE VOWEL SIGN ULU SARI
1B38 BALINESE VOWEL SIGN SUKU
1B39 BALINESE VOWEL SIGN SUKU ILUT
1B3A BALINESE VOWEL SIGN RA REPA
1B3B BALINESE VOWEL SIGN RA REPA TEDUNG
1B3C BALINESE VOWEL SIGN LA LENGA
1B3D BALINESE VOWEL SIGN LA LENGA TEDUNG
1B3E BALINESE VOWEL SIGN TALING
1B3F BALINESE VOWEL SIGN TALING REPA
1B40 BALINESE VOWEL SIGN TALING TEDUNG
1B41 BALINESE VOWEL SIGN TALING REPA TEDUNG
1B42 BALINESE VOWEL SIGN PEPET
1B43 BALINESE VOWEL SIGN PEPET TEDUNG
1B44 BALINESE ADEG ADEG
1B45 BALINESE LETTER KAF SASAK
1B46 BALINESE LETTER KHOT SASAK
1B47 BALINESE LETTER TZIR SASAK
1B48 BALINESE LETTER EF SASAK
1B49 BALINESE LETTER VE SASAK
1B4A BALINESE LETTER ZAL SASAK
1B4B BALINESE LETTER ASYURA SASAK
1B4C BALINESE LETTER ARCHAIC JNYA
1B50 BALINESE DIGIT ZERO
1B51 BALINESE DIGIT ONE
1B52 BALINESE DIGIT TWO
1B53 BALINESE DIGIT THREE
1B54 BALINESE DIGIT FOUR
1B55 BALINESE DIGIT FIVE
1B56 BALINESE DIGIT SIX
1B57 BALINESE DIGIT SEVEN
1B58 BALINESE DIGIT EIGHT
1B59 BALINESE DIGIT NINE
1B5A BALINESE PANTI
1B5B BALINESE PAMADA
1B5C BALINESE WINDU
1B5D BALINESE CARIK PAMUNGKAH
1B5E BALINESE CARIK SIKI
1B5F BALINESE CARIK PAREREN
1B60 BALINESE PAMENENG
1B61 BALINESE MUSICAL SYMBOL DONG
1B62 BALINESE MUSICAL SYMBOL DENG
1B63 BALINESE MUSICAL SYMBOL DUNG
1B64 BALINESE MUSICAL SYMBOL DANG
1B65 BALINESE MUSICAL SYMBOL DANG SURANG
1B66 BALINESE MUSICAL SYMBOL DING
1B67 BALINESE MUSICAL SYMBOL DAENG
1B68 BALINESE MUSICAL SYMBOL DEUNG
1B69 BALINESE MUSICAL SYMBOL DAING
1B6A BALINESE MUSICAL SYMBOL DANG GEDE
1B6B BALINESE MUSICAL SYMBOL COMBINING TEGEH
1B6C BALINESE MUSICAL SYMBOL COMBINING ENDEP
1B6D BALINESE MUSICAL SYMBOL COMBINING KEMPUL
1B6E BALINESE MUSICAL SYMBOL COMBINING KEMPLI
1B6F BALINESE MUSICAL SYMBOL COMBINING JEGOGAN
1B70 BALINESE MUSICAL SYMBOL COMBINING KEMPUL WITH JEGOGAN
1B71 BALINESE MUSICAL SYMBOL COMBINING KEMPLI WITH JEGOGAN
1B72 BALINESE MUSICAL SYMBOL COMBINING BENDE
1B73 BALINESE MUSICAL SYMBOL COMBINING GONG
1B74 BALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DUG
1B75 BALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DAG
1B76 BALINESE MUSICAL SYMBOL RIGHT-HAND CLOSED TUK
1B77 BALINESE MUSICAL SYMBOL RIGHT-HAND CLOSED TAK
1B78 BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PANG
1B79 BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PUNG
1B7A BALINESE MUSICAL SYMBOL LEFT-HAND CLOSED PLAK
1B7B BALINESE MUSICAL SYMBOL LEFT-HAND CLOSED PLUK
1B7C BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PING
1B7D BALINESE PANTI LANTANG
1B7E BALINESE PAMADA LANTANG
1B80 SUNDANESE SIGN PANYECEK
1B81 SUNDANESE SIGN PANGLAYAR
1B82 SUNDANESE SIGN PANGWISAD
1B83 SUNDANESE LETTER A
1B84 SUNDANESE LETTER I
1B85 SUNDANESE LETTER U
1B86 SUNDANESE LETTER AE
1B87 SUNDANESE LETTER O
1B88 SUNDANESE LETTER E
1B89 SUNDANESE LETTER EU
1B8A SUNDANESE LETTER KA
1B8B SUNDANESE LETTER QA
1B8C SUNDANESE LETTER GA
1B8D SUNDANESE LETTER NGA
1B8E SUNDANESE LETTER CA
1B8F SUNDANESE LETTER JA
1B90 SUNDANESE LETTER ZA
1B91 SUNDANESE LETTER NYA
1B92 SUNDANESE LETTER TA
1B93 SUNDANESE LETTER DA
1B94 SUNDANESE LETTER NA
1B95 SUNDANESE LETTER PA
1B96 SUNDANESE LETTER FA
1B97 SUNDANESE LETTER VA
1B98 SUNDANESE LETTER BA
1B99 SUNDANESE LETTER MA
1B9A SUNDANESE LETTER YA
1B9B SUNDANESE LETTER RA
1B9C SUNDANESE LETTER LA
1B9D SUNDANESE LETTER WA
1B9E SUNDANESE LETTER SA
1B9F SUNDANESE LETTER XA
1BA0 SUNDANESE LETTER HA
1BA1 SUNDANESE CONSONANT SIGN PAMINGKAL
1BA2 SUNDANESE CONSONANT SIGN PANYAKRA
1BA3 SUNDANESE CONSONANT SIGN PANYIKU
1BA4 SUNDANESE VOWEL SIGN PANGHULU
1BA5 SUNDANESE VOWEL SIGN PANYUKU
1BA6 SUNDANESE VOWEL SIGN PANAELAENG
1BA7 SUNDANESE VOWEL SIGN PANOLONG
1BA8 SUNDANESE VOWEL SIGN PAMEPET
1BA9 SUNDANESE VOWEL SIGN PANEULEUNG
1BAA SUNDANESE SIGN PAMAAEH
1BAB SUNDANESE SIGN VIRAMA
1BAC SUNDANESE CONSONANT SIGN PASANGAN MA
1BAD SUNDANESE CONSONANT SIGN PASANGAN WA
1BAE SUNDANESE LETTER KHA
1BAF SUNDANESE LETTER SYA
1BB0 SUNDANESE DIGIT ZERO
1BB1 SUNDANESE DIGIT ONE
1BB2 SUNDANESE DIGIT TWO
1BB3 SUNDANESE DIGIT THREE
1BB4 SUNDANESE DIGIT FOUR
1BB5 SUNDANESE DIGIT FIVE
1BB6 SUNDANESE DIGIT SIX
1BB7 SUNDANESE DIGIT SEVEN
1BB8 SUNDANESE DIGIT EIGHT
1BB9 SUNDANESE DIGIT NINE
1BBA SUNDANESE AVAGRAHA
1BBB SUNDANESE LETTER REU
1BBC SUNDANESE LETTER LEU
1BBD SUNDANESE LETTER BHA
1BBE SUNDANESE LETTER FINAL K
1BBF SUNDANESE LETTER FINAL M
1BC0 BATAK LETTER A
1BC1 BATAK LETTER SIMALUNGUN A
1BC2 BATAK LETTER HA
1BC3 BATAK LETTER SIMALUNGUN HA
1BC4 BATAK LETTER MANDAILING HA
1BC5 BATAK LETTER BA
1BC6 BATAK LETTER KARO BA
1BC7 BATAK LETTER PA
1BC8 BATAK LETTER SIMALUNGUN PA
1BC9 BATAK LETTER NA
1BCA BATAK LETTER MANDAILING NA
1BCB BATAK LETTER WA
1BCC BATAK LETTER SIMALUNGUN WA
1BCD BATAK LETTER PAKPAK WA
1BCE BATAK LETTER GA
1BCF BATAK LETTER SIMALUNGUN GA
1BD0 BATAK LETTER JA
1BD1 BATAK LETTER DA
1BD2 BATAK LETTER RA
1BD3 BATAK LETTER SIMALUNGUN RA
1BD4 BATAK LETTER MA
1BD5 BATAK LETTER SIMALUNGUN MA
1BD6 BATAK LETTER SOUTHERN TA
1BD7 BATAK LETTER NORTHERN TA
1BD8 BATAK LETTER SA
1BD9 BATAK LETTER SIMALUNGUN SA
1BDA BATAK LETTER MANDAILING SA
1BDB BATAK LETTER YA
1BDC BATAK LETTER SIMALUNGUN YA
1BDD BATAK LETTER NGA
1BDE BATAK LETTER LA
1BDF BATAK LETTER SIMALUNGUN LA
1BE0 BATAK LETTER NYA
1BE1 BATAK LETTER CA
1BE2 BATAK LETTER NDA
1BE3 BATAK LETTER MBA
1BE4 BATAK LETTER I
1BE5 BATAK LETTER U
1BE6 BATAK SIGN TOMPI
1BE7 BATAK VOWEL SIGN E
1BE8 BATAK VOWEL SIGN PAKPAK E
1BE9 BATAK VOWEL SIGN EE
1BEA BATAK VOWEL SIGN I
1BEB BATAK VOWEL SIGN KARO I
1BEC BATAK VOWEL SIGN O
1BED BATAK VOWEL SIGN KARO O
1BEE BATAK VOWEL SIGN U
1BEF BATAK VOWEL SIGN U FOR SIMALUNGUN SA
1BF0 BATAK CONSONANT SIGN NG
1BF1 BATAK CONSONANT SIGN H
1BF2 BATAK PANGOLAT
1BF3 BATAK PANONGONAN
1BFC BATAK SYMBOL BINDU NA METEK
1BFD BATAK SYMBOL BINDU PINARBORAS
1BFE BATAK SYMBOL BINDU JUDUL
1BFF BATAK SYMBOL BINDU PANGOLAT
1C00 LEPCHA LETTER KA
1C01 LEPCHA LETTER KLA
1C02 LEPCHA LETTER KHA
1C03 LEPCHA LETTER GA
1C04 LEPCHA LETTER GLA
1C05 LEPCHA LETTER NGA
1C06 LEPCHA LETTER CA
1C07 LEPCHA LETTER CHA
1C08 LEPCHA LETTER JA
1C09 LEPCHA LETTER NYA
1C0A LEPCHA LETTER TA
1C0B LEPCHA LETTER THA
1C0C LEPCHA LETTER DA
1C0D LEPCHA LETTER NA
1C0E LEPCHA LETTER PA
1C0F LEPCHA LETTER PLA
1C10 LEPCHA LETTER PHA
1C11 LEPCHA LETTER FA
1C12 LEPCHA LETTER FLA
1C13 LEPCHA LETTER BA
1C14 LEPCHA LETTER BLA
1C15 LEPCHA LETTER MA
1C16 LEPCHA LETTER MLA
1C17 LEPCHA LETTER TSA
1C18 LEPCHA LETTER TSHA
1C19 LEPCHA LETTER DZA
1C1A LEPCHA LETTER YA
1C1B LEPCHA LETTER RA
1C1C LEPCHA LETTER LA
1C1D LEPCHA LETTER HA
1C1E LEPCHA LETTER HLA
1C1F LEPCHA LETTER VA
1C20 LEPCHA LETTER SA
1C21 LEPCHA LETTER SHA
1C22 LEPCHA LETTER WA
1C23 LEPCHA LETTER A
1C24 LEPCHA SUBJOINED LETTER YA
1C25 LEPCHA SUBJOINED LETTER RA
1C26 LEPCHA VOWEL SIGN AA
1C27 LEPCHA VOWEL SIGN I
1C28 LEPCHA VOWEL SIGN O
1C29 LEPCHA VOWEL SIGN OO
1C2A LEPCHA VOWEL SIGN U
1C2B LEPCHA VOWEL SIGN UU
1C2C LEPCHA VOWEL SIGN E
1C2D LEPCHA CONSONANT SIGN K
1C2E LEPCHA CONSONANT SIGN M
1C2F LEPCHA CONSONANT SIGN L
1C30 LEPCHA CONSONANT SIGN N
1C31 LEPCHA CONSONANT SIGN P
1C32 LEPCHA CONSONANT SIGN R
1C33 LEPCHA CONSONANT SIGN T
1C34 LEPCHA CONSONANT SIGN NYIN-DO
1C35 LEPCHA CONSONANT SIGN KANG
1C36 LEPCHA SIGN RAN
1C37 LEPCHA SIGN NUKTA
1C3B LEPCHA PUNCTUATION TA-ROL
1C3C LEPCHA PUNCTUATION NYET THYOOM TA-ROL
1C3D LEPCHA PUNCTUATION CER-WA
1C3E LEPCHA PUNCTUATION TSHOOK CER-WA
1C3F LEPCHA PUNCTUATION TSHOOK
1C40 LEPCHA DIGIT ZERO
1C41 LEPCHA DIGIT ONE
1C42 LEPCHA DIGIT TWO
1C43 LEPCHA DIGIT THREE
1C44 LEPCHA DIGIT FOUR
1C45 LEPCHA DIGIT FIVE
1C46 LEPCHA DIGIT SIX
1C47 LEPCHA DIGIT SEVEN
1C48 LEPCHA DIGIT EIGHT
1C49 LEPCHA DIGIT NINE
1C4D LEPCHA LETTER TTA
1C4E LEPCHA LETTER TTHA
1C4F LEPCHA LETTER DDA
1C50 OL CHIKI DIGIT ZERO
1C51 OL CHIKI DIGIT ONE
1C52 OL CHIKI DIGIT TWO
1C53 OL CHIKI DIGIT THREE
1C54 OL CHIKI DIGIT FOUR
1C55 OL CHIKI DIGIT FIVE
1C56 OL CHIKI DIGIT SIX
1C57 OL CHIKI DIGIT SEVEN
1C58 OL CHIKI DIGIT EIGHT
1C59 OL CHIKI DIGIT NINE
1C5A OL CHIKI LETTER LA
1C5B OL CHIKI LETTER AT
1C5C OL CHIKI LETTER AG
1C5D OL CHIKI LETTER ANG
1C5E OL CHIKI LETTER AL
1C5F OL CHIKI LETTER LAA
1C60 OL CHIKI LETTER AAK
1C61 OL CHIKI LETTER AAJ
1C62 OL CHIKI LETTER AAM
1C63 OL CHIKI LETTER AAW
1C64 OL CHIKI LETTER LI
1C65 OL CHIKI LETTER IS
1C66 OL CHIKI LETTER IH
1C67 OL CHIKI LETTER INY
1C68 OL CHIKI LETTER IR
1C69 OL CHIKI LETTER LU
1C6A OL CHIKI LETTER UC
1C6B OL CHIKI LETTER UD
1C6C OL CHIKI LETTER UNN
1C6D OL CHIKI LETTER UY
1C6E OL CHIKI LETTER LE
1C6F OL CHIKI LETTER EP
1C70 OL CHIKI LETTER EDD
1C71 OL CHIKI LETTER EN
1C72 OL CHIKI LETTER ERR
1C73 OL CHIKI LETTER LO
1C74 OL CHIKI LETTER OTT
1C75 OL CHIKI LETTER OB
1C76 OL CHIKI LETTER OV
1C77 OL CHIKI LETTER OH
1C78 OL CHIKI MU TTUDDAG
1C79 OL CHIKI GAAHLAA TTUDDAAG
1C7A OL CHIKI MU-GAAHLAA TTUDDAAG
1C7B OL CHIKI RELAA
1C7C OL CHIKI PHAARKAA
1C7D OL CHIKI AHAD
1C7E OL CHIKI PUNCTUATION MUCAAD
1C7F OL CHIKI PUNCTUATION DOUBLE MUCAAD
1C80 CYRILLIC SMALL LETTER ROUNDED VE
1C81 CYRILLIC SMALL LETTER LONG-LEGGED DE
1C82 CYRILLIC SMALL LETTER NARROW O
1C83 CYRILLIC SMALL LETTER WIDE ES
1C84 CYRILLIC SMALL LETTER TALL TE
1C85 CYRILLIC SMALL LETTER THREE-LEGGED TE
1C86 CYRILLIC SMALL LETTER TALL HARD SIGN
1C87 CYRILLIC SMALL LETTER TALL YAT
1C88 CYRILLIC SMALL LETTER UNBLENDED UK
1C90 GEORGIAN MTAVRULI CAPITAL LETTER AN
1C91 GEORGIAN MTAVRULI CAPITAL LETTER BAN
1C92 GEORGIAN MTAVRULI CAPITAL LETTER GAN
1C93 GEORGIAN MTAVRULI CAPITAL LETTER DON
1C94 GEORGIAN MTAVRULI CAPITAL LETTER EN
1C95 GEORGIAN MTAVRULI CAPITAL LETTER VIN
1C96 GEORGIAN MTAVRULI CAPITAL LETTER ZEN
1C97 GEORGIAN MTAVRULI CAPITAL LETTER TAN
1C98 GEORGIAN MTAVRULI CAPITAL LETTER IN
1C99 GEORGIAN MTAVRULI CAPITAL LETTER KAN
1C9A GEORGIAN MTAVRULI CAPITAL LETTER LAS
1C9B GEORGIAN MTAVRULI CAPITAL LETTER MAN
1C9C GEORGIAN MTAVRULI CAPITAL LETTER NAR
1C9D GEORGIAN MTAVRULI CAPITAL LETTER ON
1C9E GEORGIAN MTAVRULI CAPITAL LETTER PAR
1C9F GEORGIAN MTAVRULI CAPITAL LETTER ZHAR
1CA0 GEORGIAN MTAVRULI CAPITAL LETTER RAE
1CA1 GEORGIAN MTAVRULI CAPITAL LETTER SAN
1CA2 GEORGIAN MTAVRULI CAPITAL LETTER TAR
1CA3 GEORGIAN MTAVRULI CAPITAL LETTER UN
1CA4 GEORGIAN MTAVRULI CAPITAL LETTER PHAR
1CA5 GEORGIAN MTAVRULI CAPITAL LETTER KHAR
1CA6 GEORGIAN MTAVRULI CAPITAL LETTER GHAN
1CA7 GEORGIAN MTAVRULI CAPITAL LETTER QAR
1CA8 GEORGIAN MTAVRULI CAPITAL LETTER SHIN
1CA9 GEORGIAN MTAVRULI CAPITAL LETTER CHIN
1CAA GEORGIAN MTAVRULI CAPITAL LETTER CAN
1CAB GEORGIAN MTAVRULI CAPITAL LETTER JIL
1CAC GEORGIAN MTAVRULI CAPITAL LETTER CIL
1CAD GEORGIAN MTAVRULI CAPITAL LETTER CHAR
1CAE GEORGIAN MTAVRULI CAPITAL LETTER XAN
1CAF GEORGIAN MTAVRULI CAPITAL LETTER JHAN
1CB0 GEORGIAN MTAVRULI CAPITAL LETTER HAE
1CB1 GEORGIAN MTAVRULI CAPITAL LETTER HE
1CB2 GEORGIAN MTAVRULI CAPITAL LETTER HIE
1CB3 GEORGIAN MTAVRULI CAPITAL LETTER WE
1CB4 GEORGIAN MTAVRULI CAPITAL LETTER HAR
1CB5 GEORGIAN MTAVRULI CAPITAL LETTER HOE
1CB6 GEORGIAN MTAVRULI CAPITAL LETTER FI
1CB7 GEORGIAN MTAVRULI CAPITAL LETTER YN
1CB8 GEORGIAN MTAVRULI CAPITAL LETTER ELIFI
1CB9 GEORGIAN MTAVRULI CAPITAL LETTER TURNED GAN
1CBA GEORGIAN MTAVRULI CAPITAL LETTER AIN
1CBD GEORGIAN MTAVRULI CAPITAL LETTER AEN
1CBE GEORGIAN MTAVRULI CAPITAL LETTER HARD SIGN
1CBF GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
1CC0 SUNDANESE PUNCTUATION BINDU SURYA
1CC1 SUNDANESE PUNCTUATION BINDU PANGLONG
1CC2 SUNDANESE PUNCTUATION BINDU PURNAMA
1CC3 SUNDANESE PUNCTUATION BINDU CAKRA
1CC4 SUNDANESE PUNCTUATION BINDU LEU SATANGA
1CC5 SUNDANESE PUNCTUATION BINDU KA SATANGA
1CC6 SUNDANESE PUNCTUATION BINDU DA SATANGA
1CC7 SUNDANESE PUNCTUATION BINDU BA SATANGA
1CD0 VEDIC TONE KARSHANA
1CD1 VEDIC TONE SHARA
1CD2 VEDIC TONE PRENKHA
1CD3 VEDIC SIGN NIHSHVASA
1CD4 VEDIC SIGN YAJURVEDIC MIDLINE SVARITA
1CD5 VEDIC TONE YAJURVEDIC AGGRAVATED INDEPENDENT SVARITA
1CD6 VEDIC TONE YAJURVEDIC INDEPENDENT SVARITA
1CD7 VEDIC TONE YAJURVEDIC KATHAKA INDEPENDENT SVARITA
1CD8 VEDIC TONE CANDRA BELOW
1CD9 VEDIC TONE YAJURVEDIC KATHAKA INDEPENDENT SVARITA SCHROEDER
1CDA VEDIC TONE DOUBLE SVARITA
1CDB VEDIC TONE TRIPLE SVARITA
1CDC VEDIC TONE KATHAKA ANUDATTA
1CDD VEDIC TONE DOT BELOW
1CDE VEDIC TONE TWO DOTS BELOW
1CDF VEDIC TONE THREE DOTS BELOW
1CE0 VEDIC TONE RIGVEDIC KASHMIRI INDEPENDENT SVARITA
1CE1 VEDIC TONE ATHARVAVEDIC INDEPENDENT SVARITA
1CE2 VEDIC SIGN VISARGA SVARITA
1CE3 VEDIC SIGN VISARGA UDATTA
1CE4 VEDIC SIGN REVERSED VISARGA UDATTA
1CE5 VEDIC SIGN VISARGA ANUDATTA
1CE6 VEDIC SIGN REVERSED VISARGA ANUDATTA
1CE7 VEDIC SIGN VISARGA UDATTA WITH TAIL
1CE8 VEDIC SIGN VISARGA ANUDATTA WITH TAIL
1CE9 VEDIC SIGN ANUSVARA ANTARGOMUKHA
1CEA VEDIC SIGN ANUSVARA BAHIRGOMUKHA
1CEB VEDIC SIGN ANUSVARA VAMAGOMUKHA
1CEC VEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAIL
1CED VEDIC SIGN TIRYAK
1CEE VEDIC SIGN HEXIFORM LONG ANUSVARA
1CEF VEDIC SIGN LONG ANUSVARA
1CF0 VEDIC SIGN RTHANG LONG ANUSVARA
1CF1 VEDIC SIGN ANUSVARA UBHAYATO MUKHA
1CF2 VEDIC SIGN ARDHAVISARGA
1CF3 VEDIC SIGN ROTATED ARDHAVISARGA
1CF4 VEDIC TONE CANDRA ABOVE
1CF5 VEDIC SIGN JIHVAMULIYA
1CF6 VEDIC SIGN UPADHMANIYA
1CF7 VEDIC SIGN ATIKRAMA
1CF8 VEDIC TONE RING ABOVE
1CF9 VEDIC TONE DOUBLE RING ABOVE
1CFA VEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHA
1D00 LATIN LETTER SMALL CAPITAL A
1D01 LATIN LETTER SMALL CAPITAL AE
1D02 LATIN SMALL LETTER TURNED AE
1D03 LATIN LETTER SMALL CAPITAL BARRED B
1D04 LATIN LETTER SMALL CAPITAL C
1D05 LATIN LETTER SMALL CAPITAL D
1D06 LATIN LETTER SMALL CAPITAL ETH
1D07 LATIN LETTER SMALL CAPITAL E
1D08 LATIN SMALL LETTER TURNED OPEN E
1D09 LATIN SMALL LETTER TURNED I
1D0A LATIN LETTER SMALL CAPITAL J
1D0B LATIN LETTER SMALL CAPITAL K
1D0C LATIN LETTER SMALL CAPITAL L WITH STROKE
1D0D LATIN LETTER SMALL CAPITAL M
1D0E LATIN LETTER SMALL CAPITAL REVERSED N
1D0F LATIN LETTER SMALL CAPITAL O
1D10 LATIN LETTER SMALL CAPITAL OPEN O
1D11 LATIN SMALL LETTER SIDEWAYS O
1D12 LATIN SMALL LETTER SIDEWAYS OPEN O
1D13 LATIN SMALL LETTER SIDEWAYS O WITH STROKE
1D14 LATIN SMALL LETTER TURNED OE
1D15 LATIN LETTER SMALL CAPITAL OU
1D16 LATIN SMALL LETTER TOP HALF O
1D17 LATIN SMALL LETTER BOTTOM HALF O
1D18 LATIN LETTER SMALL CAPITAL P
1D19 LATIN LETTER SMALL CAPITAL REVERSED R
1D1A LATIN LETTER SMALL CAPITAL TURNED R
1D1B LATIN LETTER SMALL CAPITAL T
1D1C LATIN LETTER SMALL CAPITAL U
1D1D LATIN SMALL LETTER SIDEWAYS U
1D1E LATIN SMALL LETTER SIDEWAYS DIAERESIZED U
1D1F LATIN SMALL LETTER SIDEWAYS TURNED M
1D20 LATIN LETTER SMALL CAPITAL V
1D21 LATIN LETTER SMALL CAPITAL W
1D22 LATIN LETTER SMALL CAPITAL Z
1D23 LATIN LETTER SMALL CAPITAL EZH
1D24 LATIN LETTER VOICED LARYNGEAL SPIRANT
1D25 LATIN LETTER AIN
1D26 GREEK LETTER SMALL CAPITAL GAMMA
1D27 GREEK LETTER SMALL CAPITAL LAMDA
1D28 GREEK LETTER SMALL CAPITAL PI
1D29 GREEK LETTER SMALL CAPITAL RHO
1D2A GREEK LETTER SMALL CAPITAL PSI
1D2B CYRILLIC LETTER SMALL CAPITAL EL
1D2C MODIFIER LETTER CAPITAL A
1D2D MODIFIER LETTER CAPITAL AE
1D2E MODIFIER LETTER CAPITAL B
1D2F MODIFIER LETTER CAPITAL BARRED B
1D30 MODIFIER LETTER CAPITAL D
1D31 MODIFIER LETTER CAPITAL E
1D32 MODIFIER LETTER CAPITAL REVERSED E
1D33 MODIFIER LETTER CAPITAL G
1D34 MODIFIER LETTER CAPITAL H
1D35 MODIFIER LETTER CAPITAL I
1D36 MODIFIER LETTER CAPITAL J
1D37 MODIFIER LETTER CAPITAL K
1D38 MODIFIER LETTER CAPITAL L
1D39 MODIFIER LETTER CAPITAL M
1D3A MODIFIER LETTER CAPITAL N
1D3B MODIFIER LETTER CAPITAL REVERSED N
1D3C MODIFIER LETTER CAPITAL O
1D3D MODIFIER LETTER CAPITAL OU
1D3E MODIFIER LETTER CAPITAL P
1D3F MODIFIER LETTER CAPITAL R
1D40 MODIFIER LETTER CAPITAL T
1D41 MODIFIER LETTER CAPITAL U
1D42 MODIFIER LETTER CAPITAL W
1D43 MODIFIER LETTER SMALL A
1D44 MODIFIER LETTER SMALL TURNED A
1D45 MODIFIER LETTER SMALL ALPHA
1D46 MODIFIER LETTER SMALL TURNED AE
1D47 MODIFIER LETTER SMALL B
1D48 MODIFIER LETTER SMALL D
1D49 MODIFIER LETTER SMALL E
1D4A MODIFIER LETTER SMALL SCHWA
1D4B MODIFIER LETTER SMALL OPEN E
1D4C MODIFIER LETTER SMALL TURNED OPEN E
1D4D MODIFIER LETTER SMALL G
1D4E MODIFIER LETTER SMALL TURNED I
1D4F MODIFIER LETTER SMALL K
1D50 MODIFIER LETTER SMALL M
1D51 MODIFIER LETTER SMALL ENG
1D52 MODIFIER LETTER SMALL O
1D53 MODIFIER LETTER SMALL OPEN O
1D54 MODIFIER LETTER SMALL TOP HALF O
1D55 MODIFIER LETTER SMALL BOTTOM HALF O
1D56 MODIFIER LETTER SMALL P
1D57 MODIFIER LETTER SMALL T
1D58 MODIFIER LETTER SMALL U
1D59 MODIFIER LETTER SMALL SIDEWAYS U
1D5A MODIFIER LETTER SMALL TURNED M
1D5B MODIFIER LETTER SMALL V
1D5C MODIFIER LETTER SMALL AIN
1D5D MODIFIER LETTER SMALL BETA
1D5E MODIFIER LETTER SMALL GREEK GAMMA
1D5F MODIFIER LETTER SMALL DELTA
1D60 MODIFIER LETTER SMALL GREEK PHI
1D61 MODIFIER LETTER SMALL CHI
1D62 LATIN SUBSCRIPT SMALL LETTER I
1D63 LATIN SUBSCRIPT SMALL LETTER R
1D64 LATIN SUBSCRIPT SMALL LETTER U
1D65 LATIN SUBSCRIPT SMALL LETTER V
1D66 GREEK SUBSCRIPT SMALL LETTER BETA
1D67 GREEK SUBSCRIPT SMALL LETTER GAMMA
1D68 GREEK SUBSCRIPT SMALL LETTER RHO
1D69 GREEK SUBSCRIPT SMALL LETTER PHI
1D6A GREEK SUBSCRIPT SMALL LETTER CHI
1D6B LATIN SMALL LETTER UE
1D6C LATIN SMALL LETTER B WITH MIDDLE TILDE
1D6D LATIN SMALL LETTER D WITH MIDDLE TILDE
1D6E LATIN SMALL LETTER F WITH MIDDLE TILDE
1D6F LATIN SMALL LETTER M WITH MIDDLE TILDE
1D70 LATIN SMALL LETTER N WITH MIDDLE TILDE
1D71 LATIN SMALL LETTER P WITH MIDDLE TILDE
1D72 LATIN SMALL LETTER R WITH MIDDLE TILDE
1D73 LATIN SMALL LETTER R WITH FISHHOOK AND MIDDLE TILDE
1D74 LATIN SMALL LETTER S WITH MIDDLE TILDE
1D75 LATIN SMALL LETTER T WITH MIDDLE TILDE
1D76 LATIN SMALL LETTER Z WITH MIDDLE TILDE
1D77 LATIN SMALL LETTER TURNED G
1D78 MODIFIER LETTER CYRILLIC EN
1D79 LATIN SMALL LETTER INSULAR G
1D7A LATIN SMALL LETTER TH WITH STRIKETHROUGH
1D7B LATIN SMALL CAPITAL LETTER I WITH STROKE
1D7C LATIN SMALL LETTER IOTA WITH STROKE
1D7D LATIN SMALL LETTER P WITH STROKE
1D7E LATIN SMALL CAPITAL LETTER U WITH STROKE
1D7F LATIN SMALL LETTER UPSILON WITH STROKE
1D80 LATIN SMALL LETTER B WITH PALATAL HOOK
1D81 LATIN SMALL LETTER D WITH PALATAL HOOK
1D82 LATIN SMALL LETTER F WITH PALATAL HOOK
1D83 LATIN SMALL LETTER G WITH PALATAL HOOK
1D84 LATIN SMALL LETTER K WITH PALATAL HOOK
1D85 LATIN SMALL LETTER L WITH PALATAL HOOK
1D86 LATIN SMALL LETTER M WITH PALATAL HOOK
1D87 LATIN SMALL LETTER N WITH PALATAL HOOK
1D88 LATIN SMALL LETTER P WITH PALATAL HOOK
1D89 LATIN SMALL LETTER R WITH PALATAL HOOK
1D8A LATIN SMALL LETTER S WITH PALATAL HOOK
1D8B LATIN SMALL LETTER ESH WITH PALATAL HOOK
1D8C LATIN SMALL LETTER V WITH PALATAL HOOK
1D8D LATIN SMALL LETTER X WITH PALATAL HOOK
1D8E LATIN SMALL LETTER Z WITH PALATAL HOOK
1D8F LATIN SMALL LETTER A WITH RETROFLEX HOOK
1D90 LATIN SMALL LETTER ALPHA WITH RETROFLEX HOOK
1D91 LATIN SMALL LETTER D WITH HOOK AND TAIL
1D92 LATIN SMALL LETTER E WITH RETROFLEX HOOK
1D93 LATIN SMALL LETTER OPEN E WITH RETROFLEX HOOK
1D94 LATIN SMALL LETTER REVERSED OPEN E WITH RETROFLEX HOOK
1D95 LATIN SMALL LETTER SCHWA WITH RETROFLEX HOOK
1D96 LATIN SMALL LETTER I WITH RETROFLEX HOOK
1D97 LATIN SMALL LETTER OPEN O WITH RETROFLEX HOOK
1D98 LATIN SMALL LETTER ESH WITH RETROFLEX HOOK
1D99 LATIN SMALL LETTER U WITH RETROFLEX HOOK
1D9A LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
1D9B MODIFIER LETTER SMALL TURNED ALPHA
1D9C MODIFIER LETTER SMALL C
1D9D MODIFIER LETTER SMALL C WITH CURL
1D9E MODIFIER LETTER SMALL ETH
1D9F MODIFIER LETTER SMALL REVERSED OPEN E
1DA0 MODIFIER LETTER SMALL F
1DA1 MODIFIER LETTER SMALL DOTLESS J WITH STROKE
1DA2 MODIFIER LETTER SMALL SCRIPT G
1DA3 MODIFIER LETTER SMALL TURNED H
1DA4 MODIFIER LETTER SMALL I WITH STROKE
1DA5 MODIFIER LETTER SMALL IOTA
1DA6 MODIFIER LETTER SMALL CAPITAL I
1DA7 MODIFIER LETTER SMALL CAPITAL I WITH STROKE
1DA8 MODIFIER LETTER SMALL J WITH CROSSED-TAIL
1DA9 MODIFIER LETTER SMALL L WITH RETROFLEX HOOK
1DAA MODIFIER LETTER SMALL L WITH PALATAL HOOK
1DAB MODIFIER LETTER SMALL CAPITAL L
1DAC MODIFIER LETTER SMALL M WITH HOOK
1DAD MODIFIER LETTER SMALL TURNED M WITH LONG LEG
1DAE MODIFIER LETTER SMALL N WITH LEFT HOOK
1DAF MODIFIER LETTER SMALL N WITH RETROFLEX HOOK
1DB0 MODIFIER LETTER SMALL CAPITAL N
1DB1 MODIFIER LETTER SMALL BARRED O
1DB2 MODIFIER LETTER SMALL PHI
1DB3 MODIFIER LETTER SMALL S WITH HOOK
1DB4 MODIFIER LETTER SMALL ESH
1DB5 MODIFIER LETTER SMALL T WITH PALATAL HOOK
1DB6 MODIFIER LETTER SMALL U BAR
1DB7 MODIFIER LETTER SMALL UPSILON
1DB8 MODIFIER LETTER SMALL CAPITAL U
1DB9 MODIFIER LETTER SMALL V WITH HOOK
1DBA MODIFIER LETTER SMALL TURNED V
1DBB MODIFIER LETTER SMALL Z
1DBC MODIFIER LETTER SMALL Z WITH RETROFLEX HOOK
1DBD MODIFIER LETTER SMALL Z WITH CURL
1DBE MODIFIER LETTER SMALL EZH
1DBF MODIFIER LETTER SMALL THETA
1DC0 COMBINING DOTTED GRAVE ACCENT
1DC1 COMBINING DOTTED ACUTE ACCENT
1DC2 COMBINING SNAKE BELOW
1DC3 COMBINING SUSPENSION MARK
1DC4 COMBINING MACRON-ACUTE
1DC5 COMBINING GRAVE-MACRON
1DC6 COMBINING MACRON-GRAVE
1DC7 COMBINING ACUTE-MACRON
1DC8 COMBINING GRAVE-ACUTE-GRAVE
1DC9 COMBINING ACUTE-GRAVE-ACUTE
1DCA COMBINING LATIN SMALL LETTER R BELOW
1DCB COMBINING BREVE-MACRON
1DCC COMBINING MACRON-BREVE
1DCD COMBINING DOUBLE CIRCUMFLEX ABOVE
1DCE COMBINING OGONEK ABOVE
1DCF COMBINING ZIGZAG BELOW
1DD0 COMBINING IS BELOW
1DD1 COMBINING UR ABOVE
1DD2 COMBINING US ABOVE
1DD3 COMBINING LATIN SMALL LETTER FLATTENED OPEN A ABOVE
1DD4 COMBINING LATIN SMALL LETTER AE
1DD5 COMBINING LATIN SMALL LETTER AO
1DD6 COMBINING LATIN SMALL LETTER AV
1DD7 COMBINING LATIN SMALL LETTER C CEDILLA
1DD8 COMBINING LATIN SMALL LETTER INSULAR D
1DD9 COMBINING LATIN SMALL LETTER ETH
1DDA COMBINING LATIN SMALL LETTER G
1DDB COMBINING LATIN LETTER SMALL CAPITAL G
1DDC COMBINING LATIN SMALL LETTER K
1DDD COMBINING LATIN SMALL LETTER L
1DDE COMBINING LATIN LETTER SMALL CAPITAL L
1DDF COMBINING LATIN LETTER SMALL CAPITAL M
1DE0 COMBINING LATIN SMALL LETTER N
1DE1 COMBINING LATIN LETTER SMALL CAPITAL N
1DE2 COMBINING LATIN LETTER SMALL CAPITAL R
1DE3 COMBINING LATIN SMALL LETTER R ROTUNDA
1DE4 COMBINING LATIN SMALL LETTER S
1DE5 COMBINING LATIN SMALL LETTER LONG S
1DE6 COMBINING LATIN SMALL LETTER Z
1DE7 COMBINING LATIN SMALL LETTER ALPHA
1DE8 COMBINING LATIN SMALL LETTER B
1DE9 COMBINING LATIN SMALL LETTER BETA
1DEA COMBINING LATIN SMALL LETTER SCHWA
1DEB COMBINING LATIN SMALL LETTER F
1DEC COMBINING LATIN SMALL LETTER L WITH DOUBLE MIDDLE TILDE
1DED COMBINING LATIN SMALL LETTER O WITH LIGHT CENTRALIZATION STROKE
1DEE COMBINING LATIN SMALL LETTER P
1DEF COMBINING LATIN SMALL LETTER ESH
1DF0 COMBINING LATIN SMALL LETTER U WITH LIGHT CENTRALIZATION STROKE
1DF1 COMBINING LATIN SMALL LETTER W
1DF2 COMBINING LATIN SMALL LETTER A WITH DIAERESIS
1DF3 COMBINING LATIN SMALL LETTER O WITH DIAERESIS
1DF4 COMBINING LATIN SMALL LETTER U WITH DIAERESIS
1DF5 COMBINING UP TACK ABOVE
1DF6 COMBINING KAVYKA ABOVE RIGHT
1DF7 COMBINING KAVYKA ABOVE LEFT
1DF8 COMBINING DOT ABOVE LEFT
1DF9 COMBINING WIDE INVERTED BRIDGE BELOW
1DFA COMBINING DOT BELOW LEFT
1DFB COMBINING DELETION MARK
1DFC COMBINING DOUBLE INVERTED BREVE BELOW
1DFD COMBINING ALMOST EQUAL TO BELOW
1DFE COMBINING LEFT ARROWHEAD ABOVE
1DFF COMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOW
1E00 LATIN CAPITAL LETTER A WITH RING BELOW
1E01 LATIN SMALL LETTER A WITH RING BELOW
1E02 LATIN CAPITAL LETTER B WITH DOT ABOVE
1E03 LATIN SMALL LETTER B WITH DOT ABOVE
1E04 LATIN CAPITAL LETTER B WITH DOT BELOW
1E05 LATIN SMALL LETTER B WITH DOT BELOW
1E06 LATIN CAPITAL LETTER B WITH LINE BELOW
1E07 LATIN SMALL LETTER B WITH LINE BELOW
1E08 LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
1E09 LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
1E0A LATIN CAPITAL LETTER D WITH DOT ABOVE
1E0B LATIN SMALL LETTER D WITH DOT ABOVE
1E0C LATIN CAPITAL LETTER D WITH DOT BELOW
1E0D LATIN SMALL LETTER D WITH DOT BELOW
1E0E LATIN CAPITAL LETTER D WITH LINE BELOW
1E0F LATIN SMALL LETTER D WITH LINE BELOW
1E10 LATIN CAPITAL LETTER D WITH CEDILLA
1E11 LATIN SMALL LETTER D WITH CEDILLA
1E12 LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
1E13 LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
1E14 LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
1E15 LATIN SMALL LETTER E WITH MACRON AND GRAVE
1E16 LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
1E17 LATIN SMALL LETTER E WITH MACRON AND ACUTE
1E18 LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
1E19 LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
1E1A LATIN CAPITAL LETTER E WITH TILDE BELOW
1E1B LATIN SMALL LETTER E WITH TILDE BELOW
1E1C LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
1E1D LATIN SMALL LETTER E WITH CEDILLA AND BREVE
1E1E LATIN CAPITAL LETTER F WITH DOT ABOVE
1E1F LATIN SMALL LETTER F WITH DOT ABOVE
1E20 LATIN CAPITAL LETTER G WITH MACRON
1E21 LATIN SMALL LETTER G WITH MACRON
1E22 LATIN CAPITAL LETTER H WITH DOT ABOVE
1E23 LATIN SMALL LETTER H WITH DOT ABOVE
1E24 LATIN CAPITAL LETTER H WITH DOT BELOW
1E25 LATIN SMALL LETTER H WITH DOT BELOW
1E26 LATIN CAPITAL LETTER H WITH DIAERESIS
1E27 LATIN SMALL LETTER H WITH DIAERESIS
1E28 LATIN CAPITAL LETTER H WITH CEDILLA
1E29 LATIN SMALL LETTER H WITH CEDILLA
1E2A LATIN CAPITAL LETTER H WITH BREVE BELOW
1E2B LATIN SMALL LETTER H WITH BREVE BELOW
1E2C LATIN CAPITAL LETTER I WITH TILDE BELOW
1E2D LATIN SMALL LETTER I WITH TILDE BELOW
1E2E LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
1E2F LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
1E30 LATIN CAPITAL LETTER K WITH ACUTE
1E31 LATIN SMALL LETTER K WITH ACUTE
1E32 LATIN CAPITAL LETTER K WITH DOT BELOW
1E33 LATIN SMALL LETTER K WITH DOT BELOW
1E34 LATIN CAPITAL LETTER K WITH LINE BELOW
1E35 LATIN SMALL LETTER K WITH LINE BELOW
1E36 LATIN CAPITAL LETTER L WITH DOT BELOW
1E37 LATIN SMALL LETTER L WITH DOT BELOW
1E38 LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
1E39 LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
1E3A LATIN CAPITAL LETTER L WITH LINE BELOW
1E3B LATIN SMALL LETTER L WITH LINE BELOW
1E3C LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
1E3D LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
1E3E LATIN CAPITAL LETTER M WITH ACUTE
1E3F LATIN SMALL LETTER M WITH ACUTE
1E40 LATIN CAPITAL LETTER M WITH DOT ABOVE
1E41 LATIN SMALL LETTER M WITH DOT ABOVE
1E42 LATIN CAPITAL LETTER M WITH DOT BELOW
1E43 LATIN SMALL LETTER M WITH DOT BELOW
1E44 LATIN CAPITAL LETTER N WITH DOT ABOVE
1E45 LATIN SMALL LETTER N WITH DOT ABOVE
1E46 LATIN CAPITAL LETTER N WITH DOT BELOW
1E47 LATIN SMALL LETTER N WITH DOT BELOW
1E48 LATIN CAPITAL LETTER N WITH LINE BELOW
1E49 LATIN SMALL LETTER N WITH LINE BELOW
1E4A LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
1E4B LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
1E4C LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
1E4D LATIN SMALL LETTER O WITH TILDE AND ACUTE
1E4E LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
1E4F LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
1E50 LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
1E51 LATIN SMALL LETTER O WITH MACRON AND GRAVE
1E52 LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
1E53 LATIN SMALL LETTER O WITH MACRON AND ACUTE
1E54 LATIN CAPITAL LETTER P WITH ACUTE
1E55 LATIN SMALL LETTER P WITH ACUTE
1E56 LATIN CAPITAL LETTER P WITH DOT ABOVE
1E57 LATIN SMALL LETTER P WITH DOT ABOVE
1E58 LATIN CAPITAL LETTER R WITH DOT ABOVE
1E59 LATIN SMALL LETTER R WITH DOT ABOVE
1E5A LATIN CAPITAL LETTER R WITH DOT BELOW
1E5B LATIN SMALL LETTER R WITH DOT BELOW
1E5C LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
1E5D LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
1E5E LATIN CAPITAL LETTER R WITH LINE BELOW
1E5F LATIN SMALL LETTER R WITH LINE BELOW
1E60 LATIN CAPITAL LETTER S WITH DOT ABOVE
1E61 LATIN SMALL LETTER S WITH DOT ABOVE
1E62 LATIN CAPITAL LETTER S WITH DOT BELOW
1E63 LATIN SMALL LETTER S WITH DOT BELOW
1E64 LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
1E65 LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
1E66 LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
1E67 LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
1E68 LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
1E69 LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
1E6A LATIN CAPITAL LETTER T WITH DOT ABOVE
1E6B LATIN SMALL LETTER T WITH DOT ABOVE
1E6C LATIN CAPITAL LETTER T WITH DOT BELOW
1E6D LATIN SMALL LETTER T WITH DOT BELOW
1E6E LATIN CAPITAL LETTER T WITH LINE BELOW
1E6F LATIN SMALL LETTER T WITH LINE BELOW
1E70 LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
1E71 LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
1E72 LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
1E73 LATIN SMALL LETTER U WITH DIAERESIS BELOW
1E74 LATIN CAPITAL LETTER U WITH TILDE BELOW
1E75 LATIN SMALL LETTER U WITH TILDE BELOW
1E76 LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
1E77 LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
1E78 LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
1E79 LATIN SMALL LETTER U WITH TILDE AND ACUTE
1E7A LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
1E7B LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
1E7C LATIN CAPITAL LETTER V WITH TILDE
1E7D LATIN SMALL LETTER V WITH TILDE
1E7E LATIN CAPITAL LETTER V WITH DOT BELOW
1E7F LATIN SMALL LETTER V WITH DOT BELOW
1E80 LATIN CAPITAL LETTER W WITH GRAVE
1E81 LATIN SMALL LETTER W WITH GRAVE
1E82 LATIN CAPITAL LETTER W WITH ACUTE
1E83 LATIN SMALL LETTER W WITH ACUTE
1E84 LATIN CAPITAL LETTER W WITH DIAERESIS
1E85 LATIN SMALL LETTER W WITH DIAERESIS
1E86 LATIN CAPITAL LETTER W WITH DOT ABOVE
1E87 LATIN SMALL LETTER W WITH DOT ABOVE
1E88 LATIN CAPITAL LETTER W WITH DOT BELOW
1E89 LATIN SMALL LETTER W WITH DOT BELOW
1E8A LATIN CAPITAL LETTER X WITH DOT ABOVE
1E8B LATIN SMALL LETTER X WITH DOT ABOVE
1E8C LATIN CAPITAL LETTER X WITH DIAERESIS
1E8D LATIN SMALL LETTER X WITH DIAERESIS
1E8E LATIN CAPITAL LETTER Y WITH DOT ABOVE
1E8F LATIN SMALL LETTER Y WITH DOT ABOVE
1E90 LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
1E91 LATIN SMALL LETTER Z WITH CIRCUMFLEX
1E92 LATIN CAPITAL LETTER Z WITH DOT BELOW
1E93 LATIN SMALL LETTER Z WITH DOT BELOW
1E94 LATIN CAPITAL LETTER Z WITH LINE BELOW
1E95 LATIN SMALL LETTER Z WITH LINE BELOW
1E96 LATIN SMALL LETTER H WITH LINE BELOW
1E97 LATIN SMALL LETTER T WITH DIAERESIS
1E98 LATIN SMALL LETTER W WITH RING ABOVE
1E99 LATIN SMALL LETTER Y WITH RING ABOVE
1E9A LATIN SMALL LETTER A WITH RIGHT HALF RING
1E9B LATIN SMALL LETTER LONG S WITH DOT ABOVE
1E9C LATIN SMALL LETTER LONG S WITH DIAGONAL STROKE
1E9D LATIN SMALL LETTER LONG S WITH HIGH STROKE
1E9E LATIN CAPITAL LETTER SHARP S
1E9F LATIN SMALL LETTER DELTA
1EA0 LATIN CAPITAL LETTER A WITH DOT BELOW
1EA1 LATIN SMALL LETTER A WITH DOT BELOW
1EA2 LATIN CAPITAL LETTER A WITH HOOK ABOVE
1EA3 LATIN SMALL LETTER A WITH HOOK ABOVE
1EA4 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
1EA5 LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
1EA6 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
1EA7 LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
1EA8 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
1EA9 LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
1EAA LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
1EAB LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
1EAC LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
1EAD LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
1EAE LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
1EAF LATIN SMALL LETTER A WITH BREVE AND ACUTE
1EB0 LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
1EB1 LATIN SMALL LETTER A WITH BREVE AND GRAVE
1EB2 LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
1EB3 LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
1EB4 LATIN CAPITAL LETTER A WITH BREVE AND TILDE
1EB5 LATIN SMALL LETTER A WITH BREVE AND TILDE
1EB6 LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
1EB7 LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
1EB8 LATIN CAPITAL LETTER E WITH DOT BELOW
1EB9 LATIN SMALL LETTER E WITH DOT BELOW
1EBA LATIN CAPITAL LETTER E WITH HOOK ABOVE
1EBB LATIN SMALL LETTER E WITH HOOK ABOVE
1EBC LATIN CAPITAL LETTER E WITH TILDE
1EBD LATIN SMALL LETTER E WITH TILDE
1EBE LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
1EBF LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
1EC0 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
1EC1 LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
1EC2 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
1EC3 LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
1EC4 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
1EC5 LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
1EC6 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
1EC7 LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
1EC8 LATIN CAPITAL LETTER I WITH HOOK ABOVE
1EC9 LATIN SMALL LETTER I WITH HOOK ABOVE
1ECA LATIN CAPITAL LETTER I WITH DOT BELOW
1ECB LATIN SMALL LETTER I WITH DOT BELOW
1ECC LATIN CAPITAL LETTER O WITH DOT BELOW
1ECD LATIN SMALL LETTER O WITH DOT BELOW
1ECE LATIN CAPITAL LETTER O WITH HOOK ABOVE
1ECF LATIN SMALL LETTER O WITH HOOK ABOVE
1ED0 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
1ED1 LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
1ED2 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
1ED3 LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
1ED4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
1ED5 LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
1ED6 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
1ED7 LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
1ED8 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
1ED9 LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
1EDA LATIN CAPITAL LETTER O WITH HORN AND ACUTE
1EDB LATIN SMALL LETTER O WITH HORN AND ACUTE
1EDC LATIN CAPITAL LETTER O WITH HORN AND GRAVE
1EDD LATIN SMALL LETTER O WITH HORN AND GRAVE
1EDE LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
1EDF LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
1EE0 LATIN CAPITAL LETTER O WITH HORN AND TILDE
1EE1 LATIN SMALL LETTER O WITH HORN AND TILDE
1EE2 LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
1EE3 LATIN SMALL LETTER O WITH HORN AND DOT BELOW
1EE4 LATIN CAPITAL LETTER U WITH DOT BELOW
1EE5 LATIN SMALL LETTER U WITH DOT BELOW
1EE6 LATIN CAPITAL LETTER U WITH HOOK ABOVE
1EE7 LATIN SMALL LETTER U WITH HOOK ABOVE
1EE8 LATIN CAPITAL LETTER U WITH HORN AND ACUTE
1EE9 LATIN SMALL LETTER U WITH HORN AND ACUTE
1EEA LATIN CAPITAL LETTER U WITH HORN AND GRAVE
1EEB LATIN SMALL LETTER U WITH HORN AND GRAVE
1EEC LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
1EED LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
1EEE LATIN CAPITAL LETTER U WITH HORN AND TILDE
1EEF LATIN SMALL LETTER U WITH HORN AND TILDE
1EF0 LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
1EF1 LATIN SMALL LETTER U WITH HORN AND DOT BELOW
1EF2 LATIN CAPITAL LETTER Y WITH GRAVE
1EF3 LATIN SMALL LETTER Y WITH GRAVE
1EF4 LATIN CAPITAL LETTER Y WITH DOT BELOW
1EF5 LATIN SMALL LETTER Y WITH DOT BELOW
1EF6 LATIN CAPITAL LETTER Y WITH HOOK ABOVE
1EF7 LATIN SMALL LETTER Y WITH HOOK ABOVE
1EF8 LATIN CAPITAL LETTER Y WITH TILDE
1EF9 LATIN SMALL LETTER Y WITH TILDE
1EFA LATIN CAPITAL LETTER MIDDLE-WELSH LL
1EFB LATIN SMALL LETTER MIDDLE-WELSH LL
1EFC LATIN CAPITAL LETTER MIDDLE-WELSH V
1EFD LATIN SMALL LETTER MIDDLE-WELSH V
1EFE LATIN CAPITAL LETTER Y WITH LOOP
1EFF LATIN SMALL LETTER Y WITH LOOP
1F00 GREEK SMALL LETTER ALPHA WITH PSILI
1F01 GREEK SMALL LETTER ALPHA WITH DASIA
1F02 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA
1F03 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA
1F04 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA
1F05 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA
1F06 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI
1F07 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
1F08 GREEK CAPITAL LETTER ALPHA WITH PSILI
1F09 GREEK CAPITAL LETTER ALPHA WITH DASIA
1F0A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
1F0B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
1F0C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
1F0D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
1F0E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
1F0F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
1F10 GREEK SMALL LETTER EPSILON WITH PSILI
1F11 GREEK SMALL LETTER EPSILON WITH DASIA
1F12 GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA
1F13 GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA
1F14 GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA
1F15 GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
1F18 GREEK CAPITAL LETTER EPSILON WITH PSILI
1F19 GREEK CAPITAL LETTER EPSILON WITH DASIA
1F1A GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
1F1B GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
1F1C GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
1F1D GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
1F20 GREEK SMALL LETTER ETA WITH PSILI
1F21 GREEK SMALL LETTER ETA WITH DASIA
1F22 GREEK SMALL LETTER ETA WITH PSILI AND VARIA
1F23 GREEK SMALL LETTER ETA WITH DASIA AND VARIA
1F24 GREEK SMALL LETTER ETA WITH PSILI AND OXIA
1F25 GREEK SMALL LETTER ETA WITH DASIA AND OXIA
1F26 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI
1F27 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
1F28 GREEK CAPITAL LETTER ETA WITH PSILI
1F29 GREEK CAPITAL LETTER ETA WITH DASIA
1F2A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
1F2B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
1F2C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
1F2D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
1F2E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
1F2F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
1F30 GREEK SMALL LETTER IOTA WITH PSILI
1F31 GREEK SMALL LETTER IOTA WITH DASIA
1F32 GREEK SMALL LETTER IOTA WITH PSILI AND VARIA
1F33 GREEK SMALL LETTER IOTA WITH DASIA AND VARIA
1F34 GREEK SMALL LETTER IOTA WITH PSILI AND OXIA
1F35 GREEK SMALL LETTER IOTA WITH DASIA AND OXIA
1F36 GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI
1F37 GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
1F38 GREEK CAPITAL LETTER IOTA WITH PSILI
1F39 GREEK CAPITAL LETTER IOTA WITH DASIA
1F3A GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
1F3B GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
1F3C GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
1F3D GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
1F3E GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
1F3F GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
1F40 GREEK SMALL LETTER OMICRON WITH PSILI
1F41 GREEK SMALL LETTER OMICRON WITH DASIA
1F42 GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA
1F43 GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA
1F44 GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA
1F45 GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
1F48 GREEK CAPITAL LETTER OMICRON WITH PSILI
1F49 GREEK CAPITAL LETTER OMICRON WITH DASIA
1F4A GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
1F4B GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
1F4C GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
1F4D GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
1F50 GREEK SMALL LETTER UPSILON WITH PSILI
1F51 GREEK SMALL LETTER UPSILON WITH DASIA
1F52 GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA
1F53 GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA
1F54 GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA
1F55 GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA
1F56 GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI
1F57 GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
1F59 GREEK CAPITAL LETTER UPSILON WITH DASIA
1F5B GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
1F5D GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
1F5F GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
1F60 GREEK SMALL LETTER OMEGA WITH PSILI
1F61 GREEK SMALL LETTER OMEGA WITH DASIA
1F62 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA
1F63 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA
1F64 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA
1F65 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA
1F66 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI
1F67 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
1F68 GREEK CAPITAL LETTER OMEGA WITH PSILI
1F69 GREEK CAPITAL LETTER OMEGA WITH DASIA
1F6A GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
1F6B GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
1F6C GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
1F6D GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
1F6E GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
1F6F GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
1F70 GREEK SMALL LETTER ALPHA WITH VARIA
1F71 GREEK SMALL LETTER ALPHA WITH OXIA
1F72 GREEK SMALL LETTER EPSILON WITH VARIA
1F73 GREEK SMALL LETTER EPSILON WITH OXIA
1F74 GREEK SMALL LETTER ETA WITH VARIA
1F75 GREEK SMALL LETTER ETA WITH OXIA
1F76 GREEK SMALL LETTER IOTA WITH VARIA
1F77 GREEK SMALL LETTER IOTA WITH OXIA
1F78 GREEK SMALL LETTER OMICRON WITH VARIA
1F79 GREEK SMALL LETTER OMICRON WITH OXIA
1F7A GREEK SMALL LETTER UPSILON WITH VARIA
1F7B GREEK SMALL LETTER UPSILON WITH OXIA
1F7C GREEK SMALL LETTER OMEGA WITH VARIA
1F7D GREEK SMALL LETTER OMEGA WITH OXIA
1F80 GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
1F81 GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
1F82 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
1F83 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
1F84 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
1F85 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
1F86 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
1F87 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
1F88 GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
1F89 GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
1F8A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
1F8B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
1F8C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
1F8D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
1F8E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
1F8F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
1F90 GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
1F91 GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
1F92 GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
1F93 GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
1F94 GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
1F95 GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
1F96 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
1F97 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
1F98 GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
1F99 GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
1F9A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
1F9B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
1F9C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
1F9D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
1F9E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
1F9F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
1FA0 GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
1FA1 GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
1FA2 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
1FA3 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
1FA4 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
1FA5 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
1FA6 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
1FA7 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
1FA8 GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
1FA9 GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
1FAA GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
1FAB GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
1FAC GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
1FAD GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
1FAE GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
1FAF GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
1FB0 GREEK SMALL LETTER ALPHA WITH VRACHY
1FB1 GREEK SMALL LETTER ALPHA WITH MACRON
1FB2 GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
1FB3 GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
1FB4 GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
1FB6 GREEK SMALL LETTER ALPHA WITH PERISPOMENI
1FB7 GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
1FB8 GREEK CAPITAL LETTER ALPHA WITH VRACHY
1FB9 GREEK CAPITAL LETTER ALPHA WITH MACRON
1FBA GREEK CAPITAL LETTER ALPHA WITH VARIA
1FBB GREEK CAPITAL LETTER ALPHA WITH OXIA
1FBC GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
1FBD GREEK KORONIS
1FBE GREEK PROSGEGRAMMENI
1FBF GREEK PSILI
1FC0 GREEK PERISPOMENI
1FC1 GREEK DIALYTIKA AND PERISPOMENI
1FC2 GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
1FC3 GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
1FC4 GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
1FC6 GREEK SMALL LETTER ETA WITH PERISPOMENI
1FC7 GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
1FC8 GREEK CAPITAL LETTER EPSILON WITH VARIA
1FC9 GREEK CAPITAL LETTER EPSILON WITH OXIA
1FCA GREEK CAPITAL LETTER ETA WITH VARIA
1FCB GREEK CAPITAL LETTER ETA WITH OXIA
1FCC GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
1FCD GREEK PSILI AND VARIA
1FCE GREEK PSILI AND OXIA
1FCF GREEK PSILI AND PERISPOMENI
1FD0 GREEK SMALL LETTER IOTA WITH VRACHY
1FD1 GREEK SMALL LETTER IOTA WITH MACRON
1FD2 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
1FD3 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
1FD6 GREEK SMALL LETTER IOTA WITH PERISPOMENI
1FD7 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
1FD8 GREEK CAPITAL LETTER IOTA WITH VRACHY
1FD9 GREEK CAPITAL LETTER IOTA WITH MACRON
1FDA GREEK CAPITAL LETTER IOTA WITH VARIA
1FDB GREEK CAPITAL LETTER IOTA WITH OXIA
1FDD GREEK DASIA AND VARIA
1FDE GREEK DASIA AND OXIA
1FDF GREEK DASIA AND PERISPOMENI
1FE0 GREEK SMALL LETTER UPSILON WITH VRACHY
1FE1 GREEK SMALL LETTER UPSILON WITH MACRON
1FE2 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
1FE3 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
1FE4 GREEK SMALL LETTER RHO WITH PSILI
1FE5 GREEK SMALL LETTER RHO WITH DASIA
1FE6 GREEK SMALL LETTER UPSILON WITH PERISPOMENI
1FE7 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
1FE8 GREEK CAPITAL LETTER UPSILON WITH VRACHY
1FE9 GREEK CAPITAL LETTER UPSILON WITH MACRON
1FEA GREEK CAPITAL LETTER UPSILON WITH VARIA
1FEB GREEK CAPITAL LETTER UPSILON WITH OXIA
1FEC GREEK CAPITAL LETTER RHO WITH DASIA
1FED GREEK DIALYTIKA AND VARIA
1FEE GREEK DIALYTIKA AND OXIA
1FEF GREEK VARIA
1FF2 GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
1FF3 GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
1FF4 GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
1FF6 GREEK SMALL LETTER OMEGA WITH PERISPOMENI
1FF7 GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
1FF8 GREEK CAPITAL LETTER OMICRON WITH VARIA
1FF9 GREEK CAPITAL LETTER OMICRON WITH OXIA
1FFA GREEK CAPITAL LETTER OMEGA WITH VARIA
1FFB GREEK CAPITAL LETTER OMEGA WITH OXIA
1FFC GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
1FFD GREEK OXIA
1FFE GREEK DASIA
2000 EN QUAD
2001 EM QUAD
2002 EN SPACE
2003 EM SPACE
2004 THREE-PER-EM SPACE
2005 FOUR-PER-EM SPACE
2006 SIX-PER-EM SPACE
2007 FIGURE SPACE
2008 PUNCTUATION SPACE
2009 THIN SPACE
200A HAIR SPACE
200B ZERO WIDTH SPACE, ZWSP
200C ZERO WIDTH NON-JOINER, ZWNJ
200D ZERO WIDTH JOINER, ZWJ
200E LEFT-TO-RIGHT MARK, LRM
200F RIGHT-TO-LEFT MARK, RLM
2010 HYPHEN
2011 NON-BREAKING HYPHEN
2012 FIGURE DASH
2013 EN DASH
2014 EM DASH
2015 HORIZONTAL BAR
2016 DOUBLE VERTICAL LINE
2017 DOUBLE LOW LINE
2018 LEFT SINGLE QUOTATION MARK
2019 RIGHT SINGLE QUOTATION MARK
201A SINGLE LOW-9 QUOTATION MARK
201B SINGLE HIGH-REVERSED-9 QUOTATION MARK
201C LEFT DOUBLE QUOTATION MARK
201D RIGHT DOUBLE QUOTATION MARK
201E DOUBLE LOW-9 QUOTATION MARK
201F DOUBLE HIGH-REVERSED-9 QUOTATION MARK
2020 DAGGER
2021 DOUBLE DAGGER
2022 BULLET
2023 TRIANGULAR BULLET
2024 ONE DOT LEADER
2025 TWO DOT LEADER
2026 HORIZONTAL ELLIPSIS
2027 HYPHENATION POINT
2028 LINE SEPARATOR
2029 PARAGRAPH SEPARATOR
202A LEFT-TO-RIGHT EMBEDDING, LRE
202B RIGHT-TO-LEFT EMBEDDING, RLE
202C POP DIRECTIONAL FORMATTING, PDF
202D LEFT-TO-RIGHT OVERRIDE, LRO
202E RIGHT-TO-LEFT OVERRIDE, RLO
202F NARROW NO-BREAK SPACE, NNBSP
2030 PER MILLE SIGN
2031 PER TEN THOUSAND SIGN
2032 PRIME
2033 DOUBLE PRIME
2034 TRIPLE PRIME
2035 REVERSED PRIME
2036 REVERSED DOUBLE PRIME
2037 REVERSED TRIPLE PRIME
2038 CARET
2039 SINGLE LEFT-POINTING ANGLE QUOTATION MARK
203A SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
203B REFERENCE MARK
203C DOUBLE EXCLAMATION MARK
203D INTERROBANG
203E OVERLINE
203F UNDERTIE
2040 CHARACTER TIE
2041 CARET INSERTION POINT
2042 ASTERISM
2043 HYPHEN BULLET
2044 FRACTION SLASH
2045 LEFT SQUARE BRACKET WITH QUILL
2046 RIGHT SQUARE BRACKET WITH QUILL
2047 DOUBLE QUESTION MARK
2048 QUESTION EXCLAMATION MARK
2049 EXCLAMATION QUESTION MARK
204A TIRONIAN SIGN ET
204B REVERSED PILCROW SIGN
204C BLACK LEFTWARDS BULLET
204D BLACK RIGHTWARDS BULLET
204E LOW ASTERISK
204F REVERSED SEMICOLON
2050 CLOSE UP
2051 TWO ASTERISKS ALIGNED VERTICALLY
2052 COMMERCIAL MINUS SIGN
2053 SWUNG DASH
2054 INVERTED UNDERTIE
2055 FLOWER PUNCTUATION MARK
2056 THREE DOT PUNCTUATION
2057 QUADRUPLE PRIME
2058 FOUR DOT PUNCTUATION
2059 FIVE DOT PUNCTUATION
205A TWO DOT PUNCTUATION
205B FOUR DOT MARK
205C DOTTED CROSS
205D TRICOLON
205E VERTICAL FOUR DOTS
205F MEDIUM MATHEMATICAL SPACE, MMSP
2060 WORD JOINER, WJ
2061 FUNCTION APPLICATION
2062 INVISIBLE TIMES
2063 INVISIBLE SEPARATOR
2064 INVISIBLE PLUS
2066 LEFT-TO-RIGHT ISOLATE, LRI
2067 RIGHT-TO-LEFT ISOLATE, RLI
2068 FIRST STRONG ISOLATE, FSI
2069 POP DIRECTIONAL ISOLATE, PDI
206A INHIBIT SYMMETRIC SWAPPING
206B ACTIVATE SYMMETRIC SWAPPING
206C INHIBIT ARABIC FORM SHAPING
206D ACTIVATE ARABIC FORM SHAPING
206E NATIONAL DIGIT SHAPES
206F NOMINAL DIGIT SHAPES
2070 SUPERSCRIPT ZERO
2071 SUPERSCRIPT LATIN SMALL LETTER I
2074 SUPERSCRIPT FOUR
2075 SUPERSCRIPT FIVE
2076 SUPERSCRIPT SIX
2077 SUPERSCRIPT SEVEN
2078 SUPERSCRIPT EIGHT
2079 SUPERSCRIPT NINE
207A SUPERSCRIPT PLUS SIGN
207B SUPERSCRIPT MINUS
207C SUPERSCRIPT EQUALS SIGN
207D SUPERSCRIPT LEFT PARENTHESIS
207E SUPERSCRIPT RIGHT PARENTHESIS
207F SUPERSCRIPT LATIN SMALL LETTER N
2080 SUBSCRIPT ZERO
2081 SUBSCRIPT ONE
2082 SUBSCRIPT TWO
2083 SUBSCRIPT THREE
2084 SUBSCRIPT FOUR
2085 SUBSCRIPT FIVE
2086 SUBSCRIPT SIX
2087 SUBSCRIPT SEVEN
2088 SUBSCRIPT EIGHT
2089 SUBSCRIPT NINE
208A SUBSCRIPT PLUS SIGN
208B SUBSCRIPT MINUS
208C SUBSCRIPT EQUALS SIGN
208D SUBSCRIPT LEFT PARENTHESIS
208E SUBSCRIPT RIGHT PARENTHESIS
2090 LATIN SUBSCRIPT SMALL LETTER A
2091 LATIN SUBSCRIPT SMALL LETTER E
2092 LATIN SUBSCRIPT SMALL LETTER O
2093 LATIN SUBSCRIPT SMALL LETTER X
2094 LATIN SUBSCRIPT SMALL LETTER SCHWA
2095 LATIN SUBSCRIPT SMALL LETTER H
2096 LATIN SUBSCRIPT SMALL LETTER K
2097 LATIN SUBSCRIPT SMALL LETTER L
2098 LATIN SUBSCRIPT SMALL LETTER M
2099 LATIN SUBSCRIPT SMALL LETTER N
209A LATIN SUBSCRIPT SMALL LETTER P
209B LATIN SUBSCRIPT SMALL LETTER S
209C LATIN SUBSCRIPT SMALL LETTER T
20A0 EURO-CURRENCY SIGN
20A1 COLON SIGN
20A2 CRUZEIRO SIGN
20A3 FRENCH FRANC SIGN
20A4 LIRA SIGN
20A5 MILL SIGN
20A6 NAIRA SIGN
20A7 PESETA SIGN
20A8 RUPEE SIGN
20A9 WON SIGN
20AA NEW SHEQEL SIGN
20AB DONG SIGN
20AC EURO SIGN
20AD KIP SIGN
20AE TUGRIK SIGN
20AF DRACHMA SIGN
20B0 GERMAN PENNY SIGN
20B1 PESO SIGN
20B2 GUARANI SIGN
20B3 AUSTRAL SIGN
20B4 HRYVNIA SIGN
20B5 CEDI SIGN
20B6 LIVRE TOURNOIS SIGN
20B7 SPESMILO SIGN
20B8 TENGE SIGN
20B9 INDIAN RUPEE SIGN
20BA TURKISH LIRA SIGN
20BB NORDIC MARK SIGN
20BC MANAT SIGN
20BD RUBLE SIGN
20BE LARI SIGN
20BF BITCOIN SIGN
20C0 SOM SIGN
20D0 COMBINING LEFT HARPOON ABOVE
20D1 COMBINING RIGHT HARPOON ABOVE
20D2 COMBINING LONG VERTICAL LINE OVERLAY
20D3 COMBINING SHORT VERTICAL LINE OVERLAY
20D4 COMBINING ANTICLOCKWISE ARROW ABOVE
20D5 COMBINING CLOCKWISE ARROW ABOVE
20D6 COMBINING LEFT ARROW ABOVE
20D7 COMBINING RIGHT ARROW ABOVE
20D8 COMBINING RING OVERLAY
20D9 COMBINING CLOCKWISE RING OVERLAY
20DA COMBINING ANTICLOCKWISE RING OVERLAY
20DB COMBINING THREE DOTS ABOVE
20DC COMBINING FOUR DOTS ABOVE
20DD COMBINING ENCLOSING CIRCLE
20DE COMBINING ENCLOSING SQUARE
20DF COMBINING ENCLOSING DIAMOND
20E0 COMBINING ENCLOSING CIRCLE BACKSLASH
20E1 COMBINING LEFT RIGHT ARROW ABOVE
20E2 COMBINING ENCLOSING SCREEN
20E3 COMBINING ENCLOSING KEYCAP
20E4 COMBINING ENCLOSING UPWARD POINTING TRIANGLE
20E5 COMBINING REVERSE SOLIDUS OVERLAY
20E6 COMBINING DOUBLE VERTICAL STROKE OVERLAY
20E7 COMBINING ANNUITY SYMBOL
20E8 COMBINING TRIPLE UNDERDOT
20E9 COMBINING WIDE BRIDGE ABOVE
20EA COMBINING LEFTWARDS ARROW OVERLAY
20EB COMBINING LONG DOUBLE SOLIDUS OVERLAY
20EC COMBINING RIGHTWARDS HARPOON WITH BARB DOWNWARDS
20ED COMBINING LEFTWARDS HARPOON WITH BARB DOWNWARDS
20EE COMBINING LEFT ARROW BELOW
20EF COMBINING RIGHT ARROW BELOW
20F0 COMBINING ASTERISK ABOVE
2100 ACCOUNT OF
2101 ADDRESSED TO THE SUBJECT
2102 DOUBLE-STRUCK CAPITAL C
2103 DEGREE CELSIUS
2104 CENTRE LINE SYMBOL
2105 CARE OF
2106 CADA UNA
2107 EULER CONSTANT
2108 SCRUPLE
2109 DEGREE FAHRENHEIT
210A SCRIPT SMALL G
210B SCRIPT CAPITAL H
210C BLACK-LETTER CAPITAL H
210D DOUBLE-STRUCK CAPITAL H
210E PLANCK CONSTANT
210F PLANCK CONSTANT OVER TWO PI
2110 SCRIPT CAPITAL I
2111 BLACK-LETTER CAPITAL I
2112 SCRIPT CAPITAL L
2113 SCRIPT SMALL L
2114 L B BAR SYMBOL
2115 DOUBLE-STRUCK CAPITAL N
2116 NUMERO SIGN
2117 SOUND RECORDING COPYRIGHT
2118 SCRIPT CAPITAL P
2119 DOUBLE-STRUCK CAPITAL P
211A DOUBLE-STRUCK CAPITAL Q
211B SCRIPT CAPITAL R
211C BLACK-LETTER CAPITAL R
211D DOUBLE-STRUCK CAPITAL R
211E PRESCRIPTION TAKE
211F RESPONSE
2120 SERVICE MARK
2121 TELEPHONE SIGN
2122 TRADE MARK SIGN
2123 VERSICLE
2124 DOUBLE-STRUCK CAPITAL Z
2125 OUNCE SIGN
2126 OHM SIGN
2127 INVERTED OHM SIGN
2128 BLACK-LETTER CAPITAL Z
2129 TURNED GREEK SMALL LETTER IOTA
212A KELVIN SIGN
212B ANGSTROM SIGN
212C SCRIPT CAPITAL B
212D BLACK-LETTER CAPITAL C
212E ESTIMATED SYMBOL
212F SCRIPT SMALL E
2130 SCRIPT CAPITAL E
2131 SCRIPT CAPITAL F
2132 TURNED CAPITAL F
2133 SCRIPT CAPITAL M
2134 SCRIPT SMALL O
2135 ALEF SYMBOL
2136 BET SYMBOL
2137 GIMEL SYMBOL
2138 DALET SYMBOL
2139 INFORMATION SOURCE
213A ROTATED CAPITAL Q
213B FACSIMILE SIGN
213C DOUBLE-STRUCK SMALL PI
213D DOUBLE-STRUCK SMALL GAMMA
213E DOUBLE-STRUCK CAPITAL GAMMA
213F DOUBLE-STRUCK CAPITAL PI
2140 DOUBLE-STRUCK N-ARY SUMMATION
2141 TURNED SANS-SERIF CAPITAL G
2142 TURNED SANS-SERIF CAPITAL L
2143 REVERSED SANS-SERIF CAPITAL L
2144 TURNED SANS-SERIF CAPITAL Y
2145 DOUBLE-STRUCK ITALIC CAPITAL D
2146 DOUBLE-STRUCK ITALIC SMALL D
2147 DOUBLE-STRUCK ITALIC SMALL E
2148 DOUBLE-STRUCK ITALIC SMALL I
2149 DOUBLE-STRUCK ITALIC SMALL J
214A PROPERTY LINE
214B TURNED AMPERSAND
214C PER SIGN
214D AKTIESELSKAB
214E TURNED SMALL F
214F SYMBOL FOR SAMARITAN SOURCE
2150 VULGAR FRACTION ONE SEVENTH
2151 VULGAR FRACTION ONE NINTH
2152 VULGAR FRACTION ONE TENTH
2153 VULGAR FRACTION ONE THIRD
2154 VULGAR FRACTION TWO THIRDS
2155 VULGAR FRACTION ONE FIFTH
2156 VULGAR FRACTION TWO FIFTHS
2157 VULGAR FRACTION THREE FIFTHS
2158 VULGAR FRACTION FOUR FIFTHS
2159 VULGAR FRACTION ONE SIXTH
215A VULGAR FRACTION FIVE SIXTHS
215B VULGAR FRACTION ONE EIGHTH
215C VULGAR FRACTION THREE EIGHTHS
215D VULGAR FRACTION FIVE EIGHTHS
215E VULGAR FRACTION SEVEN EIGHTHS
215F FRACTION NUMERATOR ONE
2160 ROMAN NUMERAL ONE
2161 ROMAN NUMERAL TWO
2162 ROMAN NUMERAL THREE
2163 ROMAN NUMERAL FOUR
2164 ROMAN NUMERAL FIVE
2165 ROMAN NUMERAL SIX
2166 ROMAN NUMERAL SEVEN
2167 ROMAN NUMERAL EIGHT
2168 ROMAN NUMERAL NINE
2169 ROMAN NUMERAL TEN
216A ROMAN NUMERAL ELEVEN
216B ROMAN NUMERAL TWELVE
216C ROMAN NUMERAL FIFTY
216D ROMAN NUMERAL ONE HUNDRED
216E ROMAN NUMERAL FIVE HUNDRED
216F ROMAN NUMERAL ONE THOUSAND
2170 SMALL ROMAN NUMERAL ONE
2171 SMALL ROMAN NUMERAL TWO
2172 SMALL ROMAN NUMERAL THREE
2173 SMALL ROMAN NUMERAL FOUR
2174 SMALL ROMAN NUMERAL FIVE
2175 SMALL ROMAN NUMERAL SIX
2176 SMALL ROMAN NUMERAL SEVEN
2177 SMALL ROMAN NUMERAL EIGHT
2178 SMALL ROMAN NUMERAL NINE
2179 SMALL ROMAN NUMERAL TEN
217A SMALL ROMAN NUMERAL ELEVEN
217B SMALL ROMAN NUMERAL TWELVE
217C SMALL ROMAN NUMERAL FIFTY
217D SMALL ROMAN NUMERAL ONE HUNDRED
217E SMALL ROMAN NUMERAL FIVE HUNDRED
217F SMALL ROMAN NUMERAL ONE THOUSAND
2180 ROMAN NUMERAL ONE THOUSAND C D
2181 ROMAN NUMERAL FIVE THOUSAND
2182 ROMAN NUMERAL TEN THOUSAND
2183 ROMAN NUMERAL REVERSED ONE HUNDRED
2184 LATIN SMALL LETTER REVERSED C
2185 ROMAN NUMERAL SIX LATE FORM
2186 ROMAN NUMERAL FIFTY EARLY FORM
2187 ROMAN NUMERAL FIFTY THOUSAND
2188 ROMAN NUMERAL ONE HUNDRED THOUSAND
2189 VULGAR FRACTION ZERO THIRDS
218A TURNED DIGIT TWO
218B TURNED DIGIT THREE
2190 LEFTWARDS ARROW
2191 UPWARDS ARROW
2192 RIGHTWARDS ARROW
2193 DOWNWARDS ARROW
2194 LEFT RIGHT ARROW
2195 UP DOWN ARROW
2196 NORTH WEST ARROW
2197 NORTH EAST ARROW
2198 SOUTH EAST ARROW
2199 SOUTH WEST ARROW
219A LEFTWARDS ARROW WITH STROKE
219B RIGHTWARDS ARROW WITH STROKE
219C LEFTWARDS WAVE ARROW
219D RIGHTWARDS WAVE ARROW
219E LEFTWARDS TWO HEADED ARROW
219F UPWARDS TWO HEADED ARROW
21A0 RIGHTWARDS TWO HEADED ARROW
21A1 DOWNWARDS TWO HEADED ARROW
21A2 LEFTWARDS ARROW WITH TAIL
21A3 RIGHTWARDS ARROW WITH TAIL
21A4 LEFTWARDS ARROW FROM BAR
21A5 UPWARDS ARROW FROM BAR
21A6 RIGHTWARDS ARROW FROM BAR
21A7 DOWNWARDS ARROW FROM BAR
21A8 UP DOWN ARROW WITH BASE
21A9 LEFTWARDS ARROW WITH HOOK
21AA RIGHTWARDS ARROW WITH HOOK
21AB LEFTWARDS ARROW WITH LOOP
21AC RIGHTWARDS ARROW WITH LOOP
21AD LEFT RIGHT WAVE ARROW
21AE LEFT RIGHT ARROW WITH STROKE
21AF DOWNWARDS ZIGZAG ARROW
21B0 UPWARDS ARROW WITH TIP LEFTWARDS
21B1 UPWARDS ARROW WITH TIP RIGHTWARDS
21B2 DOWNWARDS ARROW WITH TIP LEFTWARDS
21B3 DOWNWARDS ARROW WITH TIP RIGHTWARDS
21B4 RIGHTWARDS ARROW WITH CORNER DOWNWARDS
21B5 DOWNWARDS ARROW WITH CORNER LEFTWARDS
21B6 ANTICLOCKWISE TOP SEMICIRCLE ARROW
21B7 CLOCKWISE TOP SEMICIRCLE ARROW
21B8 NORTH WEST ARROW TO LONG BAR
21B9 LEFTWARDS ARROW TO BAR OVER RIGHTWARDS ARROW TO BAR
21BA ANTICLOCKWISE OPEN CIRCLE ARROW
21BB CLOCKWISE OPEN CIRCLE ARROW
21BC LEFTWARDS HARPOON WITH BARB UPWARDS
21BD LEFTWARDS HARPOON WITH BARB DOWNWARDS
21BE UPWARDS HARPOON WITH BARB RIGHTWARDS
21BF UPWARDS HARPOON WITH BARB LEFTWARDS
21C0 RIGHTWARDS HARPOON WITH BARB UPWARDS
21C1 RIGHTWARDS HARPOON WITH BARB DOWNWARDS
21C2 DOWNWARDS HARPOON WITH BARB RIGHTWARDS
21C3 DOWNWARDS HARPOON WITH BARB LEFTWARDS
21C4 RIGHTWARDS ARROW OVER LEFTWARDS ARROW
21C5 UPWARDS ARROW LEFTWARDS OF DOWNWARDS ARROW
21C6 LEFTWARDS ARROW OVER RIGHTWARDS ARROW
21C7 LEFTWARDS PAIRED ARROWS
21C8 UPWARDS PAIRED ARROWS
21C9 RIGHTWARDS PAIRED ARROWS
21CA DOWNWARDS PAIRED ARROWS
21CB LEFTWARDS HARPOON OVER RIGHTWARDS HARPOON
21CC RIGHTWARDS HARPOON OVER LEFTWARDS HARPOON
21CD LEFTWARDS DOUBLE ARROW WITH STROKE
21CE LEFT RIGHT DOUBLE ARROW WITH STROKE
21CF RIGHTWARDS DOUBLE ARROW WITH STROKE
21D0 LEFTWARDS DOUBLE ARROW
21D1 UPWARDS DOUBLE ARROW
21D2 RIGHTWARDS DOUBLE ARROW
21D3 DOWNWARDS DOUBLE ARROW
21D4 LEFT RIGHT DOUBLE ARROW
21D5 UP DOWN DOUBLE ARROW
21D6 NORTH WEST DOUBLE ARROW
21D7 NORTH EAST DOUBLE ARROW
21D8 SOUTH EAST DOUBLE ARROW
21D9 SOUTH WEST DOUBLE ARROW
21DA LEFTWARDS TRIPLE ARROW
21DB RIGHTWARDS TRIPLE ARROW
21DC LEFTWARDS SQUIGGLE ARROW
21DD RIGHTWARDS SQUIGGLE ARROW
21DE UPWARDS ARROW WITH DOUBLE STROKE
21DF DOWNWARDS ARROW WITH DOUBLE STROKE
21E0 LEFTWARDS DASHED ARROW
21E1 UPWARDS DASHED ARROW
21E2 RIGHTWARDS DASHED ARROW
21E3 DOWNWARDS DASHED ARROW
21E4 LEFTWARDS ARROW TO BAR
21E5 RIGHTWARDS ARROW TO BAR
21E6 LEFTWARDS WHITE ARROW
21E7 UPWARDS WHITE ARROW
21E8 RIGHTWARDS WHITE ARROW
21E9 DOWNWARDS WHITE ARROW
21EA UPWARDS WHITE ARROW FROM BAR
21EB UPWARDS WHITE ARROW ON PEDESTAL
21EC UPWARDS WHITE ARROW ON PEDESTAL WITH HORIZONTAL BAR
21ED UPWARDS WHITE ARROW ON PEDESTAL WITH VERTICAL BAR
21EE UPWARDS WHITE DOUBLE ARROW
21EF UPWARDS WHITE DOUBLE ARROW ON PEDESTAL
21F0 RIGHTWARDS WHITE ARROW FROM WALL
21F1 NORTH WEST ARROW TO CORNER
21F2 SOUTH EAST ARROW TO CORNER
21F3 UP DOWN WHITE ARROW
21F4 RIGHT ARROW WITH SMALL CIRCLE
21F5 DOWNWARDS ARROW LEFTWARDS OF UPWARDS ARROW
21F6 THREE RIGHTWARDS ARROWS
21F7 LEFTWARDS ARROW WITH VERTICAL STROKE
21F8 RIGHTWARDS ARROW WITH VERTICAL STROKE
21F9 LEFT RIGHT ARROW WITH VERTICAL STROKE
21FA LEFTWARDS ARROW WITH DOUBLE VERTICAL STROKE
21FB RIGHTWARDS ARROW WITH DOUBLE VERTICAL STROKE
21FC LEFT RIGHT ARROW WITH DOUBLE VERTICAL STROKE
21FD LEFTWARDS OPEN-HEADED ARROW
21FE RIGHTWARDS OPEN-HEADED ARROW
21FF LEFT RIGHT OPEN-HEADED ARROW
2200 FOR ALL
2201 COMPLEMENT
2202 PARTIAL DIFFERENTIAL
2203 THERE EXISTS
2204 THERE DOES NOT EXIST
2205 EMPTY SET
2206 INCREMENT
2207 NABLA
2208 ELEMENT OF
2209 NOT AN ELEMENT OF
220A SMALL ELEMENT OF
220B CONTAINS AS MEMBER
220C DOES NOT CONTAIN AS MEMBER
220D SMALL CONTAINS AS MEMBER
220E END OF PROOF
220F N-ARY PRODUCT
2210 N-ARY COPRODUCT
2211 N-ARY SUMMATION
2212 MINUS SIGN
2213 MINUS-OR-PLUS SIGN
2214 DOT PLUS
2215 DIVISION SLASH
2216 SET MINUS
2217 ASTERISK OPERATOR
2218 RING OPERATOR
2219 BULLET OPERATOR
221A SQUARE ROOT
221B CUBE ROOT
221C FOURTH ROOT
221D PROPORTIONAL TO
221E INFINITY
221F RIGHT ANGLE
2220 ANGLE
2221 MEASURED ANGLE
2222 SPHERICAL ANGLE
2223 DIVIDES
2224 DOES NOT DIVIDE
2225 PARALLEL TO
2226 NOT PARALLEL TO
2227 LOGICAL AND
2228 LOGICAL OR
2229 INTERSECTION
222A UNION
222B INTEGRAL
222C DOUBLE INTEGRAL
222D TRIPLE INTEGRAL
222E CONTOUR INTEGRAL
222F SURFACE INTEGRAL
2230 VOLUME INTEGRAL
2231 CLOCKWISE INTEGRAL
2232 CLOCKWISE CONTOUR INTEGRAL
2233 ANTICLOCKWISE CONTOUR INTEGRAL
2234 THEREFORE
2235 BECAUSE
2236 RATIO
2237 PROPORTION
2238 DOT MINUS
2239 EXCESS
223A GEOMETRIC PROPORTION
223B HOMOTHETIC
223C TILDE OPERATOR
223D REVERSED TILDE
223E INVERTED LAZY S
223F SINE WAVE
2240 WREATH PRODUCT
2241 NOT TILDE
2242 MINUS TILDE
2243 ASYMPTOTICALLY EQUAL TO
2244 NOT ASYMPTOTICALLY EQUAL TO
2245 APPROXIMATELY EQUAL TO
2246 APPROXIMATELY BUT NOT ACTUALLY EQUAL TO
2247 NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO
2248 ALMOST EQUAL TO
2249 NOT ALMOST EQUAL TO
224A ALMOST EQUAL OR EQUAL TO
224B TRIPLE TILDE
224C ALL EQUAL TO
224D EQUIVALENT TO
224E GEOMETRICALLY EQUIVALENT TO
224F DIFFERENCE BETWEEN
2250 APPROACHES THE LIMIT
2251 GEOMETRICALLY EQUAL TO
2252 APPROXIMATELY EQUAL TO OR THE IMAGE OF
2253 IMAGE OF OR APPROXIMATELY EQUAL TO
2254 COLON EQUALS
2255 EQUALS COLON
2256 RING IN EQUAL TO
2257 RING EQUAL TO
2258 CORRESPONDS TO
2259 ESTIMATES
225A EQUIANGULAR TO
225B STAR EQUALS
225C DELTA EQUAL TO
225D EQUAL TO BY DEFINITION
225E MEASURED BY
225F QUESTIONED EQUAL TO
2260 NOT EQUAL TO
2261 IDENTICAL TO
2262 NOT IDENTICAL TO
2263 STRICTLY EQUIVALENT TO
2264 LESS-THAN OR EQUAL TO
2265 GREATER-THAN OR EQUAL TO
2266 LESS-THAN OVER EQUAL TO
2267 GREATER-THAN OVER EQUAL TO
2268 LESS-THAN BUT NOT EQUAL TO
2269 GREATER-THAN BUT NOT EQUAL TO
226A MUCH LESS-THAN
226B MUCH GREATER-THAN
226C BETWEEN
226D NOT EQUIVALENT TO
226E NOT LESS-THAN
226F NOT GREATER-THAN
2270 NEITHER LESS-THAN NOR EQUAL TO
2271 NEITHER GREATER-THAN NOR EQUAL TO
2272 LESS-THAN OR EQUIVALENT TO
2273 GREATER-THAN OR EQUIVALENT TO
2274 NEITHER LESS-THAN NOR EQUIVALENT TO
2275 NEITHER GREATER-THAN NOR EQUIVALENT TO
2276 LESS-THAN OR GREATER-THAN
2277 GREATER-THAN OR LESS-THAN
2278 NEITHER LESS-THAN NOR GREATER-THAN
2279 NEITHER GREATER-THAN NOR LESS-THAN
227A PRECEDES
227B SUCCEEDS
227C PRECEDES OR EQUAL TO
227D SUCCEEDS OR EQUAL TO
227E PRECEDES OR EQUIVALENT TO
227F SUCCEEDS OR EQUIVALENT TO
2280 DOES NOT PRECEDE
2281 DOES NOT SUCCEED
2282 SUBSET OF
2283 SUPERSET OF
2284 NOT A SUBSET OF
2285 NOT A SUPERSET OF
2286 SUBSET OF OR EQUAL TO
2287 SUPERSET OF OR EQUAL TO
2288 NEITHER A SUBSET OF NOR EQUAL TO
2289 NEITHER A SUPERSET OF NOR EQUAL TO
228A SUBSET OF WITH NOT EQUAL TO
228B SUPERSET OF WITH NOT EQUAL TO
228C MULTISET
228D MULTISET MULTIPLICATION
228E MULTISET UNION
228F SQUARE IMAGE OF
2290 SQUARE ORIGINAL OF
2291 SQUARE IMAGE OF OR EQUAL TO
2292 SQUARE ORIGINAL OF OR EQUAL TO
2293 SQUARE CAP
2294 SQUARE CUP
2295 CIRCLED PLUS
2296 CIRCLED MINUS
2297 CIRCLED TIMES
2298 CIRCLED DIVISION SLASH
2299 CIRCLED DOT OPERATOR
229A CIRCLED RING OPERATOR
229B CIRCLED ASTERISK OPERATOR
229C CIRCLED EQUALS
229D CIRCLED DASH
229E SQUARED PLUS
229F SQUARED MINUS
22A0 SQUARED TIMES
22A1 SQUARED DOT OPERATOR
22A2 RIGHT TACK
22A3 LEFT TACK
22A4 DOWN TACK
22A5 UP TACK
22A6 ASSERTION
22A7 MODELS
22A8 TRUE
22A9 FORCES
22AA TRIPLE VERTICAL BAR RIGHT TURNSTILE
22AB DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
22AC DOES NOT PROVE
22AD NOT TRUE
22AE DOES NOT FORCE
22AF NEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
22B0 PRECEDES UNDER RELATION
22B1 SUCCEEDS UNDER RELATION
22B2 NORMAL SUBGROUP OF
22B3 CONTAINS AS NORMAL SUBGROUP
22B4 NORMAL SUBGROUP OF OR EQUAL TO
22B5 CONTAINS AS NORMAL SUBGROUP OR EQUAL TO
22B6 ORIGINAL OF
22B7 IMAGE OF
22B8 MULTIMAP
22B9 HERMITIAN CONJUGATE MATRIX
22BA INTERCALATE
22BB XOR
22BC NAND
22BD NOR
22BE RIGHT ANGLE WITH ARC
22BF RIGHT TRIANGLE
22C0 N-ARY LOGICAL AND
22C1 N-ARY LOGICAL OR
22C2 N-ARY INTERSECTION
22C3 N-ARY UNION
22C4 DIAMOND OPERATOR
22C5 DOT OPERATOR
22C6 STAR OPERATOR
22C7 DIVISION TIMES
22C8 BOWTIE
22C9 LEFT NORMAL FACTOR SEMIDIRECT PRODUCT
22CA RIGHT NORMAL FACTOR SEMIDIRECT PRODUCT
22CB LEFT SEMIDIRECT PRODUCT
22CC RIGHT SEMIDIRECT PRODUCT
22CD REVERSED TILDE EQUALS
22CE CURLY LOGICAL OR
22CF CURLY LOGICAL AND
22D0 DOUBLE SUBSET
22D1 DOUBLE SUPERSET
22D2 DOUBLE INTERSECTION
22D3 DOUBLE UNION
22D4 PITCHFORK
22D5 EQUAL AND PARALLEL TO
22D6 LESS-THAN WITH DOT
22D7 GREATER-THAN WITH DOT
22D8 VERY MUCH LESS-THAN
22D9 VERY MUCH GREATER-THAN
22DA LESS-THAN EQUAL TO OR GREATER-THAN
22DB GREATER-THAN EQUAL TO OR LESS-THAN
22DC EQUAL TO OR LESS-THAN
22DD EQUAL TO OR GREATER-THAN
22DE EQUAL TO OR PRECEDES
22DF EQUAL TO OR SUCCEEDS
22E0 DOES NOT PRECEDE OR EQUAL
22E1 DOES NOT SUCCEED OR EQUAL
22E2 NOT SQUARE IMAGE OF OR EQUAL TO
22E3 NOT SQUARE ORIGINAL OF OR EQUAL TO
22E4 SQUARE IMAGE OF OR NOT EQUAL TO
22E5 SQUARE ORIGINAL OF OR NOT EQUAL TO
22E6 LESS-THAN BUT NOT EQUIVALENT TO
22E7 GREATER-THAN BUT NOT EQUIVALENT TO
22E8 PRECEDES BUT NOT EQUIVALENT TO
22E9 SUCCEEDS BUT NOT EQUIVALENT TO
22EA NOT NORMAL SUBGROUP OF
22EB DOES NOT CONTAIN AS NORMAL SUBGROUP
22EC NOT NORMAL SUBGROUP OF OR EQUAL TO
22ED DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL
22EE VERTICAL ELLIPSIS
22EF MIDLINE HORIZONTAL ELLIPSIS
22F0 UP RIGHT DIAGONAL ELLIPSIS
22F1 DOWN RIGHT DIAGONAL ELLIPSIS
22F2 ELEMENT OF WITH LONG HORIZONTAL STROKE
22F3 ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
22F4 SMALL ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
22F5 ELEMENT OF WITH DOT ABOVE
22F6 ELEMENT OF WITH OVERBAR
22F7 SMALL ELEMENT OF WITH OVERBAR
22F8 ELEMENT OF WITH UNDERBAR
22F9 ELEMENT OF WITH TWO HORIZONTAL STROKES
22FA CONTAINS WITH LONG HORIZONTAL STROKE
22FB CONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
22FC SMALL CONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
22FD CONTAINS WITH OVERBAR
22FE SMALL CONTAINS WITH OVERBAR
22FF Z NOTATION BAG MEMBERSHIP
2300 DIAMETER SIGN
2301 ELECTRIC ARROW
2302 HOUSE
2303 UP ARROWHEAD
2304 DOWN ARROWHEAD
2305 PROJECTIVE
2306 PERSPECTIVE
2307 WAVY LINE
2308 LEFT CEILING
2309 RIGHT CEILING
230A LEFT FLOOR
230B RIGHT FLOOR
230C BOTTOM RIGHT CROP
230D BOTTOM LEFT CROP
230E TOP RIGHT CROP
230F TOP LEFT CROP
2310 REVERSED NOT SIGN
2311 SQUARE LOZENGE
2312 ARC
2313 SEGMENT
2314 SECTOR
2315 TELEPHONE RECORDER
2316 POSITION INDICATOR
2317 VIEWDATA SQUARE
2318 PLACE OF INTEREST SIGN
2319 TURNED NOT SIGN
231A WATCH
231B HOURGLASS
231C TOP LEFT CORNER
231D TOP RIGHT CORNER
231E BOTTOM LEFT CORNER
231F BOTTOM RIGHT CORNER
2320 TOP HALF INTEGRAL
2321 BOTTOM HALF INTEGRAL
2322 FROWN
2323 SMILE
2324 UP ARROWHEAD BETWEEN TWO HORIZONTAL BARS
2325 OPTION KEY
2326 ERASE TO THE RIGHT
2327 X IN A RECTANGLE BOX
2328 KEYBOARD
2329 LEFT-POINTING ANGLE BRACKET
232A RIGHT-POINTING ANGLE BRACKET
232B ERASE TO THE LEFT
232C BENZENE RING
232D CYLINDRICITY
232E ALL AROUND-PROFILE
232F SYMMETRY
2330 TOTAL RUNOUT
2331 DIMENSION ORIGIN
2332 CONICAL TAPER
2333 SLOPE
2334 COUNTERBORE
2335 COUNTERSINK
2336 APL FUNCTIONAL SYMBOL I-BEAM
2337 APL FUNCTIONAL SYMBOL SQUISH QUAD
2338 APL FUNCTIONAL SYMBOL QUAD EQUAL
2339 APL FUNCTIONAL SYMBOL QUAD DIVIDE
233A APL FUNCTIONAL SYMBOL QUAD DIAMOND
233B APL FUNCTIONAL SYMBOL QUAD JOT
233C APL FUNCTIONAL SYMBOL QUAD CIRCLE
233D APL FUNCTIONAL SYMBOL CIRCLE STILE
233E APL FUNCTIONAL SYMBOL CIRCLE JOT
233F APL FUNCTIONAL SYMBOL SLASH BAR
2340 APL FUNCTIONAL SYMBOL BACKSLASH BAR
2341 APL FUNCTIONAL SYMBOL QUAD SLASH
2342 APL FUNCTIONAL SYMBOL QUAD BACKSLASH
2343 APL FUNCTIONAL SYMBOL QUAD LESS-THAN
2344 APL FUNCTIONAL SYMBOL QUAD GREATER-THAN
2345 APL FUNCTIONAL SYMBOL LEFTWARDS VANE
2346 APL FUNCTIONAL SYMBOL RIGHTWARDS VANE
2347 APL FUNCTIONAL SYMBOL QUAD LEFTWARDS ARROW
2348 APL FUNCTIONAL SYMBOL QUAD RIGHTWARDS ARROW
2349 APL FUNCTIONAL SYMBOL CIRCLE BACKSLASH
234A APL FUNCTIONAL SYMBOL DOWN TACK UNDERBAR
234B APL FUNCTIONAL SYMBOL DELTA STILE
234C APL FUNCTIONAL SYMBOL QUAD DOWN CARET
234D APL FUNCTIONAL SYMBOL QUAD DELTA
234E APL FUNCTIONAL SYMBOL DOWN TACK JOT
234F APL FUNCTIONAL SYMBOL UPWARDS VANE
2350 APL FUNCTIONAL SYMBOL QUAD UPWARDS ARROW
2351 APL FUNCTIONAL SYMBOL UP TACK OVERBAR
2352 APL FUNCTIONAL SYMBOL DEL STILE
2353 APL FUNCTIONAL SYMBOL QUAD UP CARET
2354 APL FUNCTIONAL SYMBOL QUAD DEL
2355 APL FUNCTIONAL SYMBOL UP TACK JOT
2356 APL FUNCTIONAL SYMBOL DOWNWARDS VANE
2357 APL FUNCTIONAL SYMBOL QUAD DOWNWARDS ARROW
2358 APL FUNCTIONAL SYMBOL QUOTE UNDERBAR
2359 APL FUNCTIONAL SYMBOL DELTA UNDERBAR
235A APL FUNCTIONAL SYMBOL DIAMOND UNDERBAR
235B APL FUNCTIONAL SYMBOL JOT UNDERBAR
235C APL FUNCTIONAL SYMBOL CIRCLE UNDERBAR
235D APL FUNCTIONAL SYMBOL UP SHOE JOT
235E APL FUNCTIONAL SYMBOL QUOTE QUAD
235F APL FUNCTIONAL SYMBOL CIRCLE STAR
2360 APL FUNCTIONAL SYMBOL QUAD COLON
2361 APL FUNCTIONAL SYMBOL UP TACK DIAERESIS
2362 APL FUNCTIONAL SYMBOL DEL DIAERESIS
2363 APL FUNCTIONAL SYMBOL STAR DIAERESIS
2364 APL FUNCTIONAL SYMBOL JOT DIAERESIS
2365 APL FUNCTIONAL SYMBOL CIRCLE DIAERESIS
2366 APL FUNCTIONAL SYMBOL DOWN SHOE STILE
2367 APL FUNCTIONAL SYMBOL LEFT SHOE STILE
2368 APL FUNCTIONAL SYMBOL TILDE DIAERESIS
2369 APL FUNCTIONAL SYMBOL GREATER-THAN DIAERESIS
236A APL FUNCTIONAL SYMBOL COMMA BAR
236B APL FUNCTIONAL SYMBOL DEL TILDE
236C APL FUNCTIONAL SYMBOL ZILDE
236D APL FUNCTIONAL SYMBOL STILE TILDE
236E APL FUNCTIONAL SYMBOL SEMICOLON UNDERBAR
236F APL FUNCTIONAL SYMBOL QUAD NOT EQUAL
2370 APL FUNCTIONAL SYMBOL QUAD QUESTION
2371 APL FUNCTIONAL SYMBOL DOWN CARET TILDE
2372 APL FUNCTIONAL SYMBOL UP CARET TILDE
2373 APL FUNCTIONAL SYMBOL IOTA
2374 APL FUNCTIONAL SYMBOL RHO
2375 APL FUNCTIONAL SYMBOL OMEGA
2376 APL FUNCTIONAL SYMBOL ALPHA UNDERBAR
2377 APL FUNCTIONAL SYMBOL EPSILON UNDERBAR
2378 APL FUNCTIONAL SYMBOL IOTA UNDERBAR
2379 APL FUNCTIONAL SYMBOL OMEGA UNDERBAR
237A APL FUNCTIONAL SYMBOL ALPHA
237B NOT CHECK MARK
237C RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW
237D SHOULDERED OPEN BOX
237E BELL SYMBOL
237F VERTICAL LINE WITH MIDDLE DOT
2380 INSERTION SYMBOL
2381 CONTINUOUS UNDERLINE SYMBOL
2382 DISCONTINUOUS UNDERLINE SYMBOL
2383 EMPHASIS SYMBOL
2384 COMPOSITION SYMBOL
2385 WHITE SQUARE WITH CENTRE VERTICAL LINE
2386 ENTER SYMBOL
2387 ALTERNATIVE KEY SYMBOL
2388 HELM SYMBOL
2389 CIRCLED HORIZONTAL BAR WITH NOTCH
238A CIRCLED TRIANGLE DOWN
238B BROKEN CIRCLE WITH NORTHWEST ARROW
238C UNDO SYMBOL
238D MONOSTABLE SYMBOL
238E HYSTERESIS SYMBOL
238F OPEN-CIRCUIT-OUTPUT H-TYPE SYMBOL
2390 OPEN-CIRCUIT-OUTPUT L-TYPE SYMBOL
2391 PASSIVE-PULL-DOWN-OUTPUT SYMBOL
2392 PASSIVE-PULL-UP-OUTPUT SYMBOL
2393 DIRECT CURRENT SYMBOL FORM TWO
2394 SOFTWARE-FUNCTION SYMBOL
2395 APL FUNCTIONAL SYMBOL QUAD
2396 DECIMAL SEPARATOR KEY SYMBOL
2397 PREVIOUS PAGE
2398 NEXT PAGE
2399 PRINT SCREEN SYMBOL
239A CLEAR SCREEN SYMBOL
239B LEFT PARENTHESIS UPPER HOOK
239C LEFT PARENTHESIS EXTENSION
239D LEFT PARENTHESIS LOWER HOOK
239E RIGHT PARENTHESIS UPPER HOOK
239F RIGHT PARENTHESIS EXTENSION
23A0 RIGHT PARENTHESIS LOWER HOOK
23A1 LEFT SQUARE BRACKET UPPER CORNER
23A2 LEFT SQUARE BRACKET EXTENSION
23A3 LEFT SQUARE BRACKET LOWER CORNER
23A4 RIGHT SQUARE BRACKET UPPER CORNER
23A5 RIGHT SQUARE BRACKET EXTENSION
23A6 RIGHT SQUARE BRACKET LOWER CORNER
23A7 LEFT CURLY BRACKET UPPER HOOK
23A8 LEFT CURLY BRACKET MIDDLE PIECE
23A9 LEFT CURLY BRACKET LOWER HOOK
23AA CURLY BRACKET EXTENSION
23AB RIGHT CURLY BRACKET UPPER HOOK
23AC RIGHT CURLY BRACKET MIDDLE PIECE
23AD RIGHT CURLY BRACKET LOWER HOOK
23AE INTEGRAL EXTENSION
23AF HORIZONTAL LINE EXTENSION
23B0 UPPER LEFT OR LOWER RIGHT CURLY BRACKET SECTION
23B1 UPPER RIGHT OR LOWER LEFT CURLY BRACKET SECTION
23B2 SUMMATION TOP
23B3 SUMMATION BOTTOM
23B4 TOP SQUARE BRACKET
23B5 BOTTOM SQUARE BRACKET
23B6 BOTTOM SQUARE BRACKET OVER TOP SQUARE BRACKET
23B7 RADICAL SYMBOL BOTTOM
23B8 LEFT VERTICAL BOX LINE
23B9 RIGHT VERTICAL BOX LINE
23BA HORIZONTAL SCAN LINE-1
23BB HORIZONTAL SCAN LINE-3
23BC HORIZONTAL SCAN LINE-7
23BD HORIZONTAL SCAN LINE-9
23BE DENTISTRY SYMBOL LIGHT VERTICAL AND TOP RIGHT
23BF DENTISTRY SYMBOL LIGHT VERTICAL AND BOTTOM RIGHT
23C0 DENTISTRY SYMBOL LIGHT VERTICAL WITH CIRCLE
23C1 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH CIRCLE
23C2 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH CIRCLE
23C3 DENTISTRY SYMBOL LIGHT VERTICAL WITH TRIANGLE
23C4 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH TRIANGLE
23C5 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH TRIANGLE
23C6 DENTISTRY SYMBOL LIGHT VERTICAL AND WAVE
23C7 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH WAVE
23C8 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH WAVE
23C9 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL
23CA DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL
23CB DENTISTRY SYMBOL LIGHT VERTICAL AND TOP LEFT
23CC DENTISTRY SYMBOL LIGHT VERTICAL AND BOTTOM LEFT
23CD SQUARE FOOT
23CE RETURN SYMBOL
23CF EJECT SYMBOL
23D0 VERTICAL LINE EXTENSION
23D1 METRICAL BREVE
23D2 METRICAL LONG OVER SHORT
23D3 METRICAL SHORT OVER LONG
23D4 METRICAL LONG OVER TWO SHORTS
23D5 METRICAL TWO SHORTS OVER LONG
23D6 METRICAL TWO SHORTS JOINED
23D7 METRICAL TRISEME
23D8 METRICAL TETRASEME
23D9 METRICAL PENTASEME
23DA EARTH GROUND
23DB FUSE
23DC TOP PARENTHESIS
23DD BOTTOM PARENTHESIS
23DE TOP CURLY BRACKET
23DF BOTTOM CURLY BRACKET
23E0 TOP TORTOISE SHELL BRACKET
23E1 BOTTOM TORTOISE SHELL BRACKET
23E2 WHITE TRAPEZIUM
23E3 BENZENE RING WITH CIRCLE
23E4 STRAIGHTNESS
23E5 FLATNESS
23E6 AC CURRENT
23E7 ELECTRICAL INTERSECTION
23E8 DECIMAL EXPONENT SYMBOL
23E9 BLACK RIGHT-POINTING DOUBLE TRIANGLE
23EA BLACK LEFT-POINTING DOUBLE TRIANGLE
23EB BLACK UP-POINTING DOUBLE TRIANGLE
23EC BLACK DOWN-POINTING DOUBLE TRIANGLE
23ED BLACK RIGHT-POINTING DOUBLE TRIANGLE WITH VERTICAL BAR
23EE BLACK LEFT-POINTING DOUBLE TRIANGLE WITH VERTICAL BAR
23EF BLACK RIGHT-POINTING TRIANGLE WITH DOUBLE VERTICAL BAR
23F0 ALARM CLOCK
23F1 STOPWATCH
23F2 TIMER CLOCK
23F3 HOURGLASS WITH FLOWING SAND
23F4 BLACK MEDIUM LEFT-POINTING TRIANGLE
23F5 BLACK MEDIUM RIGHT-POINTING TRIANGLE
23F6 BLACK MEDIUM UP-POINTING TRIANGLE
23F7 BLACK MEDIUM DOWN-POINTING TRIANGLE
23F8 DOUBLE VERTICAL BAR
23F9 BLACK SQUARE FOR STOP
23FA BLACK CIRCLE FOR RECORD
23FB POWER SYMBOL
23FC POWER ON-OFF SYMBOL
23FD POWER ON SYMBOL
23FE POWER SLEEP SYMBOL
23FF OBSERVER EYE SYMBOL
2400 SYMBOL FOR NULL
2401 SYMBOL FOR START OF HEADING
2402 SYMBOL FOR START OF TEXT
2403 SYMBOL FOR END OF TEXT
2404 SYMBOL FOR END OF TRANSMISSION
2405 SYMBOL FOR ENQUIRY
2406 SYMBOL FOR ACKNOWLEDGE
2407 SYMBOL FOR BELL
2408 SYMBOL FOR BACKSPACE
2409 SYMBOL FOR HORIZONTAL TABULATION
240A SYMBOL FOR LINE FEED
240B SYMBOL FOR VERTICAL TABULATION
240C SYMBOL FOR FORM FEED
240D SYMBOL FOR CARRIAGE RETURN
240E SYMBOL FOR SHIFT OUT
240F SYMBOL FOR SHIFT IN
2410 SYMBOL FOR DATA LINK ESCAPE
2411 SYMBOL FOR DEVICE CONTROL ONE
2412 SYMBOL FOR DEVICE CONTROL TWO
2413 SYMBOL FOR DEVICE CONTROL THREE
2414 SYMBOL FOR DEVICE CONTROL FOUR
2415 SYMBOL FOR NEGATIVE ACKNOWLEDGE
2416 SYMBOL FOR SYNCHRONOUS IDLE
2417 SYMBOL FOR END OF TRANSMISSION BLOCK
2418 SYMBOL FOR CANCEL
2419 SYMBOL FOR END OF MEDIUM
241A SYMBOL FOR SUBSTITUTE
241B SYMBOL FOR ESCAPE
241C SYMBOL FOR FILE SEPARATOR
241D SYMBOL FOR GROUP SEPARATOR
241E SYMBOL FOR RECORD SEPARATOR
241F SYMBOL FOR UNIT SEPARATOR
2420 SYMBOL FOR SPACE
2421 SYMBOL FOR DELETE
2422 BLANK SYMBOL
2423 OPEN BOX
2424 SYMBOL FOR NEWLINE
2425 SYMBOL FOR DELETE FORM TWO
2426 SYMBOL FOR SUBSTITUTE FORM TWO
2440 OCR HOOK
2441 OCR CHAIR
2442 OCR FORK
2443 OCR INVERTED FORK
2444 OCR BELT BUCKLE
2445 OCR BOW TIE
2446 OCR BRANCH BANK IDENTIFICATION
2447 OCR AMOUNT OF CHECK
2448 OCR DASH
2449 OCR CUSTOMER ACCOUNT NUMBER
244A OCR DOUBLE BACKSLASH
2460 CIRCLED DIGIT ONE
2461 CIRCLED DIGIT TWO
2462 CIRCLED DIGIT THREE
2463 CIRCLED DIGIT FOUR
2464 CIRCLED DIGIT FIVE
2465 CIRCLED DIGIT SIX
2466 CIRCLED DIGIT SEVEN
2467 CIRCLED DIGIT EIGHT
2468 CIRCLED DIGIT NINE
2469 CIRCLED NUMBER TEN
246A CIRCLED NUMBER ELEVEN
246B CIRCLED NUMBER TWELVE
246C CIRCLED NUMBER THIRTEEN
246D CIRCLED NUMBER FOURTEEN
246E CIRCLED NUMBER FIFTEEN
246F CIRCLED NUMBER SIXTEEN
2470 CIRCLED NUMBER SEVENTEEN
2471 CIRCLED NUMBER EIGHTEEN
2472 CIRCLED NUMBER NINETEEN
2473 CIRCLED NUMBER TWENTY
2474 PARENTHESIZED DIGIT ONE
2475 PARENTHESIZED DIGIT TWO
2476 PARENTHESIZED DIGIT THREE
2477 PARENTHESIZED DIGIT FOUR
2478 PARENTHESIZED DIGIT FIVE
2479 PARENTHESIZED DIGIT SIX
247A PARENTHESIZED DIGIT SEVEN
247B PARENTHESIZED DIGIT EIGHT
247C PARENTHESIZED DIGIT NINE
247D PARENTHESIZED NUMBER TEN
247E PARENTHESIZED NUMBER ELEVEN
247F PARENTHESIZED NUMBER TWELVE
2480 PARENTHESIZED NUMBER THIRTEEN
2481 PARENTHESIZED NUMBER FOURTEEN
2482 PARENTHESIZED NUMBER FIFTEEN
2483 PARENTHESIZED NUMBER SIXTEEN
2484 PARENTHESIZED NUMBER SEVENTEEN
2485 PARENTHESIZED NUMBER EIGHTEEN
2486 PARENTHESIZED NUMBER NINETEEN
2487 PARENTHESIZED NUMBER TWENTY
2488 DIGIT ONE FULL STOP
2489 DIGIT TWO FULL STOP
248A DIGIT THREE FULL STOP
248B DIGIT FOUR FULL STOP
248C DIGIT FIVE FULL STOP
248D DIGIT SIX FULL STOP
248E DIGIT SEVEN FULL STOP
248F DIGIT EIGHT FULL STOP
2490 DIGIT NINE FULL STOP
2491 NUMBER TEN FULL STOP
2492 NUMBER ELEVEN FULL STOP
2493 NUMBER TWELVE FULL STOP
2494 NUMBER THIRTEEN FULL STOP
2495 NUMBER FOURTEEN FULL STOP
2496 NUMBER FIFTEEN FULL STOP
2497 NUMBER SIXTEEN FULL STOP
2498 NUMBER SEVENTEEN FULL STOP
2499 NUMBER EIGHTEEN FULL STOP
249A NUMBER NINETEEN FULL STOP
249B NUMBER TWENTY FULL STOP
249C PARENTHESIZED LATIN SMALL LETTER A
249D PARENTHESIZED LATIN SMALL LETTER B
249E PARENTHESIZED LATIN SMALL LETTER C
249F PARENTHESIZED LATIN SMALL LETTER D
24A0 PARENTHESIZED LATIN SMALL LETTER E
24A1 PARENTHESIZED LATIN SMALL LETTER F
24A2 PARENTHESIZED LATIN SMALL LETTER G
24A3 PARENTHESIZED LATIN SMALL LETTER H
24A4 PARENTHESIZED LATIN SMALL LETTER I
24A5 PARENTHESIZED LATIN SMALL LETTER J
24A6 PARENTHESIZED LATIN SMALL LETTER K
24A7 PARENTHESIZED LATIN SMALL LETTER L
24A8 PARENTHESIZED LATIN SMALL LETTER M
24A9 PARENTHESIZED LATIN SMALL LETTER N
24AA PARENTHESIZED LATIN SMALL LETTER O
24AB PARENTHESIZED LATIN SMALL LETTER P
24AC PARENTHESIZED LATIN SMALL LETTER Q
24AD PARENTHESIZED LATIN SMALL LETTER R
24AE PARENTHESIZED LATIN SMALL LETTER S
24AF PARENTHESIZED LATIN SMALL LETTER T
24B0 PARENTHESIZED LATIN SMALL LETTER U
24B1 PARENTHESIZED LATIN SMALL LETTER V
24B2 PARENTHESIZED LATIN SMALL LETTER W
24B3 PARENTHESIZED LATIN SMALL LETTER X
24B4 PARENTHESIZED LATIN SMALL LETTER Y
24B5 PARENTHESIZED LATIN SMALL LETTER Z
24B6 CIRCLED LATIN CAPITAL LETTER A
24B7 CIRCLED LATIN CAPITAL LETTER B
24B8 CIRCLED LATIN CAPITAL LETTER C
24B9 CIRCLED LATIN CAPITAL LETTER D
24BA CIRCLED LATIN CAPITAL LETTER E
24BB CIRCLED LATIN CAPITAL LETTER F
24BC CIRCLED LATIN CAPITAL LETTER G
24BD CIRCLED LATIN CAPITAL LETTER H
24BE CIRCLED LATIN CAPITAL LETTER I
24BF CIRCLED LATIN CAPITAL LETTER J
24C0 CIRCLED LATIN CAPITAL LETTER K
24C1 CIRCLED LATIN CAPITAL LETTER L
24C2 CIRCLED LATIN CAPITAL LETTER M
24C3 CIRCLED LATIN CAPITAL LETTER N
24C4 CIRCLED LATIN CAPITAL LETTER O
24C5 CIRCLED LATIN CAPITAL LETTER P
24C6 CIRCLED LATIN CAPITAL LETTER Q
24C7 CIRCLED LATIN CAPITAL LETTER R
24C8 CIRCLED LATIN CAPITAL LETTER S
24C9 CIRCLED LATIN CAPITAL LETTER T
24CA CIRCLED LATIN CAPITAL LETTER U
24CB CIRCLED LATIN CAPITAL LETTER V
24CC CIRCLED LATIN CAPITAL LETTER W
24CD CIRCLED LATIN CAPITAL LETTER X
24CE CIRCLED LATIN CAPITAL LETTER Y
24CF CIRCLED LATIN CAPITAL LETTER Z
24D0 CIRCLED LATIN SMALL LETTER A
24D1 CIRCLED LATIN SMALL LETTER B
24D2 CIRCLED LATIN SMALL LETTER C
24D3 CIRCLED LATIN SMALL LETTER D
24D4 CIRCLED LATIN SMALL LETTER E
24D5 CIRCLED LATIN SMALL LETTER F
24D6 CIRCLED LATIN SMALL LETTER G
24D7 CIRCLED LATIN SMALL LETTER H
24D8 CIRCLED LATIN SMALL LETTER I
24D9 CIRCLED LATIN SMALL LETTER J
24DA CIRCLED LATIN SMALL LETTER K
24DB CIRCLED LATIN SMALL LETTER L
24DC CIRCLED LATIN SMALL LETTER M
24DD CIRCLED LATIN SMALL LETTER N
24DE CIRCLED LATIN SMALL LETTER O
24DF CIRCLED LATIN SMALL LETTER P
24E0 CIRCLED LATIN SMALL LETTER Q
24E1 CIRCLED LATIN SMALL LETTER R
24E2 CIRCLED LATIN SMALL LETTER S
24E3 CIRCLED LATIN SMALL LETTER T
24E4 CIRCLED LATIN SMALL LETTER U
24E5 CIRCLED LATIN SMALL LETTER V
24E6 CIRCLED LATIN SMALL LETTER W
24E7 CIRCLED LATIN SMALL LETTER X
24E8 CIRCLED LATIN SMALL LETTER Y
24E9 CIRCLED LATIN SMALL LETTER Z
24EA CIRCLED DIGIT ZERO
24EB NEGATIVE CIRCLED NUMBER ELEVEN
24EC NEGATIVE CIRCLED NUMBER TWELVE
24ED NEGATIVE CIRCLED NUMBER THIRTEEN
24EE NEGATIVE CIRCLED NUMBER FOURTEEN
24EF NEGATIVE CIRCLED NUMBER FIFTEEN
24F0 NEGATIVE CIRCLED NUMBER SIXTEEN
24F1 NEGATIVE CIRCLED NUMBER SEVENTEEN
24F2 NEGATIVE CIRCLED NUMBER EIGHTEEN
24F3 NEGATIVE CIRCLED NUMBER NINETEEN
24F4 NEGATIVE CIRCLED NUMBER TWENTY
24F5 DOUBLE CIRCLED DIGIT ONE
24F6 DOUBLE CIRCLED DIGIT TWO
24F7 DOUBLE CIRCLED DIGIT THREE
24F8 DOUBLE CIRCLED DIGIT FOUR
24F9 DOUBLE CIRCLED DIGIT FIVE
24FA DOUBLE CIRCLED DIGIT SIX
24FB DOUBLE CIRCLED DIGIT SEVEN
24FC DOUBLE CIRCLED DIGIT EIGHT
24FD DOUBLE CIRCLED DIGIT NINE
24FE DOUBLE CIRCLED NUMBER TEN
24FF NEGATIVE CIRCLED DIGIT ZERO
2500 BOX DRAWINGS LIGHT HORIZONTAL
2501 BOX DRAWINGS HEAVY HORIZONTAL
2502 BOX DRAWINGS LIGHT VERTICAL
2503 BOX DRAWINGS HEAVY VERTICAL
2504 BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL
2505 BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL
2506 BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL
2507 BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL
2508 BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL
2509 BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL
250A BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL
250B BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL
250C BOX DRAWINGS LIGHT DOWN AND RIGHT
250D BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY
250E BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT
250F BOX DRAWINGS HEAVY DOWN AND RIGHT
2510 BOX DRAWINGS LIGHT DOWN AND LEFT
2511 BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY
2512 BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT
2513 BOX DRAWINGS HEAVY DOWN AND LEFT
2514 BOX DRAWINGS LIGHT UP AND RIGHT
2515 BOX DRAWINGS UP LIGHT AND RIGHT HEAVY
2516 BOX DRAWINGS UP HEAVY AND RIGHT LIGHT
2517 BOX DRAWINGS HEAVY UP AND RIGHT
2518 BOX DRAWINGS LIGHT UP AND LEFT
2519 BOX DRAWINGS UP LIGHT AND LEFT HEAVY
251A BOX DRAWINGS UP HEAVY AND LEFT LIGHT
251B BOX DRAWINGS HEAVY UP AND LEFT
251C BOX DRAWINGS LIGHT VERTICAL AND RIGHT
251D BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
251E BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT
251F BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT
2520 BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
2521 BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY
2522 BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY
2523 BOX DRAWINGS HEAVY VERTICAL AND RIGHT
2524 BOX DRAWINGS LIGHT VERTICAL AND LEFT
2525 BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
2526 BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT
2527 BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT
2528 BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
2529 BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY
252A BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY
252B BOX DRAWINGS HEAVY VERTICAL AND LEFT
252C BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
252D BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT
252E BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT
252F BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
2530 BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
2531 BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY
2532 BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY
2533 BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
2534 BOX DRAWINGS LIGHT UP AND HORIZONTAL
2535 BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT
2536 BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT
2537 BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
2538 BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
2539 BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY
253A BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY
253B BOX DRAWINGS HEAVY UP AND HORIZONTAL
253C BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
253D BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT
253E BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT
253F BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
2540 BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT
2541 BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT
2542 BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
2543 BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT
2544 BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT
2545 BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT
2546 BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT
2547 BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY
2548 BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY
2549 BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY
254A BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY
254B BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
254C BOX DRAWINGS LIGHT DOUBLE DASH HORIZONTAL
254D BOX DRAWINGS HEAVY DOUBLE DASH HORIZONTAL
254E BOX DRAWINGS LIGHT DOUBLE DASH VERTICAL
254F BOX DRAWINGS HEAVY DOUBLE DASH VERTICAL
2550 BOX DRAWINGS DOUBLE HORIZONTAL
2551 BOX DRAWINGS DOUBLE VERTICAL
2552 BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE
2553 BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE
2554 BOX DRAWINGS DOUBLE DOWN AND RIGHT
2555 BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE
2556 BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE
2557 BOX DRAWINGS DOUBLE DOWN AND LEFT
2558 BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE
2559 BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE
255A BOX DRAWINGS DOUBLE UP AND RIGHT
255B BOX DRAWINGS UP SINGLE AND LEFT DOUBLE
255C BOX DRAWINGS UP DOUBLE AND LEFT SINGLE
255D BOX DRAWINGS DOUBLE UP AND LEFT
255E BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE
255F BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE
2560 BOX DRAWINGS DOUBLE VERTICAL AND RIGHT
2561 BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE
2562 BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE
2563 BOX DRAWINGS DOUBLE VERTICAL AND LEFT
2564 BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE
2565 BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE
2566 BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL
2567 BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE
2568 BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE
2569 BOX DRAWINGS DOUBLE UP AND HORIZONTAL
256A BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE
256B BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE
256C BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL
256D BOX DRAWINGS LIGHT ARC DOWN AND RIGHT
256E BOX DRAWINGS LIGHT ARC DOWN AND LEFT
256F BOX DRAWINGS LIGHT ARC UP AND LEFT
2570 BOX DRAWINGS LIGHT ARC UP AND RIGHT
2571 BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT
2572 BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT
2573 BOX DRAWINGS LIGHT DIAGONAL CROSS
2574 BOX DRAWINGS LIGHT LEFT
2575 BOX DRAWINGS LIGHT UP
2576 BOX DRAWINGS LIGHT RIGHT
2577 BOX DRAWINGS LIGHT DOWN
2578 BOX DRAWINGS HEAVY LEFT
2579 BOX DRAWINGS HEAVY UP
257A BOX DRAWINGS HEAVY RIGHT
257B BOX DRAWINGS HEAVY DOWN
257C BOX DRAWINGS LIGHT LEFT AND HEAVY RIGHT
257D BOX DRAWINGS LIGHT UP AND HEAVY DOWN
257E BOX DRAWINGS HEAVY LEFT AND LIGHT RIGHT
257F BOX DRAWINGS HEAVY UP AND LIGHT DOWN
2580 UPPER HALF BLOCK
2581 LOWER ONE EIGHTH BLOCK
2582 LOWER ONE QUARTER BLOCK
2583 LOWER THREE EIGHTHS BLOCK
2584 LOWER HALF BLOCK
2585 LOWER FIVE EIGHTHS BLOCK
2586 LOWER THREE QUARTERS BLOCK
2587 LOWER SEVEN EIGHTHS BLOCK
2588 FULL BLOCK
2589 LEFT SEVEN EIGHTHS BLOCK
258A LEFT THREE QUARTERS BLOCK
258B LEFT FIVE EIGHTHS BLOCK
258C LEFT HALF BLOCK
258D LEFT THREE EIGHTHS BLOCK
258E LEFT ONE QUARTER BLOCK
258F LEFT ONE EIGHTH BLOCK
2590 RIGHT HALF BLOCK
2591 LIGHT SHADE
2592 MEDIUM SHADE
2593 DARK SHADE
2594 UPPER ONE EIGHTH BLOCK
2595 RIGHT ONE EIGHTH BLOCK
2596 QUADRANT LOWER LEFT
2597 QUADRANT LOWER RIGHT
2598 QUADRANT UPPER LEFT
2599 QUADRANT UPPER LEFT AND LOWER LEFT AND LOWER RIGHT
259A QUADRANT UPPER LEFT AND LOWER RIGHT
259B QUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER LEFT
259C QUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER RIGHT
259D QUADRANT UPPER RIGHT
259E QUADRANT UPPER RIGHT AND LOWER LEFT
259F QUADRANT UPPER RIGHT AND LOWER LEFT AND LOWER RIGHT
25A0 BLACK SQUARE
25A1 WHITE SQUARE
25A2 WHITE SQUARE WITH ROUNDED CORNERS
25A3 WHITE SQUARE CONTAINING BLACK SMALL SQUARE
25A4 SQUARE WITH HORIZONTAL FILL
25A5 SQUARE WITH VERTICAL FILL
25A6 SQUARE WITH ORTHOGONAL CROSSHATCH FILL
25A7 SQUARE WITH UPPER LEFT TO LOWER RIGHT FILL
25A8 SQUARE WITH UPPER RIGHT TO LOWER LEFT FILL
25A9 SQUARE WITH DIAGONAL CROSSHATCH FILL
25AA BLACK SMALL SQUARE
25AB WHITE SMALL SQUARE
25AC BLACK RECTANGLE
25AD WHITE RECTANGLE
25AE BLACK VERTICAL RECTANGLE
25AF WHITE VERTICAL RECTANGLE
25B0 BLACK PARALLELOGRAM
25B1 WHITE PARALLELOGRAM
25B2 BLACK UP-POINTING TRIANGLE
25B3 WHITE UP-POINTING TRIANGLE
25B4 BLACK UP-POINTING SMALL TRIANGLE
25B5 WHITE UP-POINTING SMALL TRIANGLE
25B6 BLACK RIGHT-POINTING TRIANGLE
25B7 WHITE RIGHT-POINTING TRIANGLE
25B8 BLACK RIGHT-POINTING SMALL TRIANGLE
25B9 WHITE RIGHT-POINTING SMALL TRIANGLE
25BA BLACK RIGHT-POINTING POINTER
25BB WHITE RIGHT-POINTING POINTER
25BC BLACK DOWN-POINTING TRIANGLE
25BD WHITE DOWN-POINTING TRIANGLE
25BE BLACK DOWN-POINTING SMALL TRIANGLE
25BF WHITE DOWN-POINTING SMALL TRIANGLE
25C0 BLACK LEFT-POINTING TRIANGLE
25C1 WHITE LEFT-POINTING TRIANGLE
25C2 BLACK LEFT-POINTING SMALL TRIANGLE
25C3 WHITE LEFT-POINTING SMALL TRIANGLE
25C4 BLACK LEFT-POINTING POINTER
25C5 WHITE LEFT-POINTING POINTER
25C6 BLACK DIAMOND
25C7 WHITE DIAMOND
25C8 WHITE DIAMOND CONTAINING BLACK SMALL DIAMOND
25C9 FISHEYE
25CA LOZENGE
25CB WHITE CIRCLE
25CC DOTTED CIRCLE
25CD CIRCLE WITH VERTICAL FILL
25CE BULLSEYE
25CF BLACK CIRCLE
25D0 CIRCLE WITH LEFT HALF BLACK
25D1 CIRCLE WITH RIGHT HALF BLACK
25D2 CIRCLE WITH LOWER HALF BLACK
25D3 CIRCLE WITH UPPER HALF BLACK
25D4 CIRCLE WITH UPPER RIGHT QUADRANT BLACK
25D5 CIRCLE WITH ALL BUT UPPER LEFT QUADRANT BLACK
25D6 LEFT HALF BLACK CIRCLE
25D7 RIGHT HALF BLACK CIRCLE
25D8 INVERSE BULLET
25D9 INVERSE WHITE CIRCLE
25DA UPPER HALF INVERSE WHITE CIRCLE
25DB LOWER HALF INVERSE WHITE CIRCLE
25DC UPPER LEFT QUADRANT CIRCULAR ARC
25DD UPPER RIGHT QUADRANT CIRCULAR ARC
25DE LOWER RIGHT QUADRANT CIRCULAR ARC
25DF LOWER LEFT QUADRANT CIRCULAR ARC
25E0 UPPER HALF CIRCLE
25E1 LOWER HALF CIRCLE
25E2 BLACK LOWER RIGHT TRIANGLE
25E3 BLACK LOWER LEFT TRIANGLE
25E4 BLACK UPPER LEFT TRIANGLE
25E5 BLACK UPPER RIGHT TRIANGLE
25E6 WHITE BULLET
25E7 SQUARE WITH LEFT HALF BLACK
25E8 SQUARE WITH RIGHT HALF BLACK
25E9 SQUARE WITH UPPER LEFT DIAGONAL HALF BLACK
25EA SQUARE WITH LOWER RIGHT DIAGONAL HALF BLACK
25EB WHITE SQUARE WITH VERTICAL BISECTING LINE
25EC WHITE UP-POINTING TRIANGLE WITH DOT
25ED UP-POINTING TRIANGLE WITH LEFT HALF BLACK
25EE UP-POINTING TRIANGLE WITH RIGHT HALF BLACK
25EF LARGE CIRCLE
25F0 WHITE SQUARE WITH UPPER LEFT QUADRANT
25F1 WHITE SQUARE WITH LOWER LEFT QUADRANT
25F2 WHITE SQUARE WITH LOWER RIGHT QUADRANT
25F3 WHITE SQUARE WITH UPPER RIGHT QUADRANT
25F4 WHITE CIRCLE WITH UPPER LEFT QUADRANT
25F5 WHITE CIRCLE WITH LOWER LEFT QUADRANT
25F6 WHITE CIRCLE WITH LOWER RIGHT QUADRANT
25F7 WHITE CIRCLE WITH UPPER RIGHT QUADRANT
25F8 UPPER LEFT TRIANGLE
25F9 UPPER RIGHT TRIANGLE
25FA LOWER LEFT TRIANGLE
25FB WHITE MEDIUM SQUARE
25FC BLACK MEDIUM SQUARE
25FD WHITE MEDIUM SMALL SQUARE
25FE BLACK MEDIUM SMALL SQUARE
25FF LOWER RIGHT TRIANGLE
2600 BLACK SUN WITH RAYS
2601 CLOUD
2602 UMBRELLA
2603 SNOWMAN
2604 COMET
2605 BLACK STAR
2606 WHITE STAR
2607 LIGHTNING
2608 THUNDERSTORM
2609 SUN
260A ASCENDING NODE
260B DESCENDING NODE
260C CONJUNCTION
260D OPPOSITION
260E BLACK TELEPHONE
260F WHITE TELEPHONE
2610 BALLOT BOX
2611 BALLOT BOX WITH CHECK
2612 BALLOT BOX WITH X
2613 SALTIRE
2614 UMBRELLA WITH RAIN DROPS
2615 HOT BEVERAGE
2616 WHITE SHOGI PIECE
2617 BLACK SHOGI PIECE
2618 SHAMROCK
2619 REVERSED ROTATED FLORAL HEART BULLET
261A BLACK LEFT POINTING INDEX
261B BLACK RIGHT POINTING INDEX
261C WHITE LEFT POINTING INDEX
261D WHITE UP POINTING INDEX
261E WHITE RIGHT POINTING INDEX
261F WHITE DOWN POINTING INDEX
2620 SKULL AND CROSSBONES
2621 CAUTION SIGN
2622 RADIOACTIVE SIGN
2623 BIOHAZARD SIGN
2624 CADUCEUS
2625 ANKH
2626 ORTHODOX CROSS
2627 CHI RHO
2628 CROSS OF LORRAINE
2629 CROSS OF JERUSALEM
262A STAR AND CRESCENT
262B FARSI SYMBOL
262C ADI SHAKTI
262D HAMMER AND SICKLE
262E PEACE SYMBOL
262F YIN YANG
2630 TRIGRAM FOR HEAVEN
2631 TRIGRAM FOR LAKE
2632 TRIGRAM FOR FIRE
2633 TRIGRAM FOR THUNDER
2634 TRIGRAM FOR WIND
2635 TRIGRAM FOR WATER
2636 TRIGRAM FOR MOUNTAIN
2637 TRIGRAM FOR EARTH
2638 WHEEL OF DHARMA
2639 WHITE FROWNING FACE
263A WHITE SMILING FACE
263B BLACK SMILING FACE
263C WHITE SUN WITH RAYS
263D FIRST QUARTER MOON
263E LAST QUARTER MOON
263F MERCURY
2640 FEMALE SIGN
2641 EARTH
2642 MALE SIGN
2643 JUPITER
2644 SATURN
2645 URANUS
2646 NEPTUNE
2647 PLUTO
2648 ARIES
2649 TAURUS
264A GEMINI
264B CANCER
264C LEO
264D VIRGO
264E LIBRA
264F SCORPIUS
2650 SAGITTARIUS
2651 CAPRICORN
2652 AQUARIUS
2653 PISCES
2654 WHITE CHESS KING
2655 WHITE CHESS QUEEN
2656 WHITE CHESS ROOK
2657 WHITE CHESS BISHOP
2658 WHITE CHESS KNIGHT
2659 WHITE CHESS PAWN
265A BLACK CHESS KING
265B BLACK CHESS QUEEN
265C BLACK CHESS ROOK
265D BLACK CHESS BISHOP
265E BLACK CHESS KNIGHT
265F BLACK CHESS PAWN
2660 BLACK SPADE SUIT
2661 WHITE HEART SUIT
2662 WHITE DIAMOND SUIT
2663 BLACK CLUB SUIT
2664 WHITE SPADE SUIT
2665 BLACK HEART SUIT
2666 BLACK DIAMOND SUIT
2667 WHITE CLUB SUIT
2668 HOT SPRINGS
2669 QUARTER NOTE
266A EIGHTH NOTE
266B BEAMED EIGHTH NOTES
266C BEAMED SIXTEENTH NOTES
266D MUSIC FLAT SIGN
266E MUSIC NATURAL SIGN
266F MUSIC SHARP SIGN
2670 WEST SYRIAC CROSS
2671 EAST SYRIAC CROSS
2672 UNIVERSAL RECYCLING SYMBOL
2673 RECYCLING SYMBOL FOR TYPE-1 PLASTICS
2674 RECYCLING SYMBOL FOR TYPE-2 PLASTICS
2675 RECYCLING SYMBOL FOR TYPE-3 PLASTICS
2676 RECYCLING SYMBOL FOR TYPE-4 PLASTICS
2677 RECYCLING SYMBOL FOR TYPE-5 PLASTICS
2678 RECYCLING SYMBOL FOR TYPE-6 PLASTICS
2679 RECYCLING SYMBOL FOR TYPE-7 PLASTICS
267A RECYCLING SYMBOL FOR GENERIC MATERIALS
267B BLACK UNIVERSAL RECYCLING SYMBOL
267C RECYCLED PAPER SYMBOL
267D PARTIALLY-RECYCLED PAPER SYMBOL
267E PERMANENT PAPER SIGN
267F WHEELCHAIR SYMBOL
2680 DIE FACE-1
2681 DIE FACE-2
2682 DIE FACE-3
2683 DIE FACE-4
2684 DIE FACE-5
2685 DIE FACE-6
2686 WHITE CIRCLE WITH DOT RIGHT
2687 WHITE CIRCLE WITH TWO DOTS
2688 BLACK CIRCLE WITH WHITE DOT RIGHT
2689 BLACK CIRCLE WITH TWO WHITE DOTS
268A MONOGRAM FOR YANG
268B MONOGRAM FOR YIN
268C DIGRAM FOR GREATER YANG
268D DIGRAM FOR LESSER YIN
268E DIGRAM FOR LESSER YANG
268F DIGRAM FOR GREATER YIN
2690 WHITE FLAG
2691 BLACK FLAG
2692 HAMMER AND PICK
2693 ANCHOR
2694 CROSSED SWORDS
2695 STAFF OF AESCULAPIUS
2696 SCALES
2697 ALEMBIC
2698 FLOWER
2699 GEAR
269A STAFF OF HERMES
269B ATOM SYMBOL
269C FLEUR-DE-LIS
269D OUTLINED WHITE STAR
269E THREE LINES CONVERGING RIGHT
269F THREE LINES CONVERGING LEFT
26A0 WARNING SIGN
26A1 HIGH VOLTAGE SIGN
26A2 DOUBLED FEMALE SIGN
26A3 DOUBLED MALE SIGN
26A4 INTERLOCKED FEMALE AND MALE SIGN
26A5 MALE AND FEMALE SIGN
26A6 MALE WITH STROKE SIGN
26A7 MALE WITH STROKE AND MALE AND FEMALE SIGN
26A8 VERTICAL MALE WITH STROKE SIGN
26A9 HORIZONTAL MALE WITH STROKE SIGN
26AA MEDIUM WHITE CIRCLE
26AB MEDIUM BLACK CIRCLE
26AC MEDIUM SMALL WHITE CIRCLE
26AD MARRIAGE SYMBOL
26AE DIVORCE SYMBOL
26AF UNMARRIED PARTNERSHIP SYMBOL
26B0 COFFIN
26B1 FUNERAL URN
26B2 NEUTER
26B3 CERES
26B4 PALLAS
26B5 JUNO
26B6 VESTA
26B7 CHIRON
26B8 BLACK MOON LILITH
26B9 SEXTILE
26BA SEMISEXTILE
26BB QUINCUNX
26BC SESQUIQUADRATE
26BD SOCCER BALL
26BE BASEBALL
26BF SQUARED KEY
26C0 WHITE DRAUGHTS MAN
26C1 WHITE DRAUGHTS KING
26C2 BLACK DRAUGHTS MAN
26C3 BLACK DRAUGHTS KING
26C4 SNOWMAN WITHOUT SNOW
26C5 SUN BEHIND CLOUD
26C6 RAIN
26C7 BLACK SNOWMAN
26C8 THUNDER CLOUD AND RAIN
26C9 TURNED WHITE SHOGI PIECE
26CA TURNED BLACK SHOGI PIECE
26CB WHITE DIAMOND IN SQUARE
26CC CROSSING LANES
26CD DISABLED CAR
26CE OPHIUCHUS
26CF PICK
26D0 CAR SLIDING
26D1 HELMET WITH WHITE CROSS
26D2 CIRCLED CROSSING LANES
26D3 CHAINS
26D4 NO ENTRY
26D5 ALTERNATE ONE-WAY LEFT WAY TRAFFIC
26D6 BLACK TWO-WAY LEFT WAY TRAFFIC
26D7 WHITE TWO-WAY LEFT WAY TRAFFIC
26D8 BLACK LEFT LANE MERGE
26D9 WHITE LEFT LANE MERGE
26DA DRIVE SLOW SIGN
26DB HEAVY WHITE DOWN-POINTING TRIANGLE
26DC LEFT CLOSED ENTRY
26DD SQUARED SALTIRE
26DE FALLING DIAGONAL IN WHITE CIRCLE IN BLACK SQUARE
26DF BLACK TRUCK
26E0 RESTRICTED LEFT ENTRY-1
26E1 RESTRICTED LEFT ENTRY-2
26E2 ASTRONOMICAL SYMBOL FOR URANUS
26E3 HEAVY CIRCLE WITH STROKE AND TWO DOTS ABOVE
26E4 PENTAGRAM
26E5 RIGHT-HANDED INTERLACED PENTAGRAM
26E6 LEFT-HANDED INTERLACED PENTAGRAM
26E7 INVERTED PENTAGRAM
26E8 BLACK CROSS ON SHIELD
26E9 SHINTO SHRINE
26EA CHURCH
26EB CASTLE
26EC HISTORIC SITE
26ED GEAR WITHOUT HUB
26EE GEAR WITH HANDLES
26EF MAP SYMBOL FOR LIGHTHOUSE
26F0 MOUNTAIN
26F1 UMBRELLA ON GROUND
26F2 FOUNTAIN
26F3 FLAG IN HOLE
26F4 FERRY
26F5 SAILBOAT
26F6 SQUARE FOUR CORNERS
26F7 SKIER
26F8 ICE SKATE
26F9 PERSON WITH BALL
26FA TENT
26FB JAPANESE BANK SYMBOL
26FC HEADSTONE GRAVEYARD SYMBOL
26FD FUEL PUMP
26FE CUP ON BLACK SQUARE
26FF WHITE FLAG WITH HORIZONTAL MIDDLE BLACK STRIPE
2700 BLACK SAFETY SCISSORS
2701 UPPER BLADE SCISSORS
2702 BLACK SCISSORS
2703 LOWER BLADE SCISSORS
2704 WHITE SCISSORS
2705 WHITE HEAVY CHECK MARK
2706 TELEPHONE LOCATION SIGN
2707 TAPE DRIVE
2708 AIRPLANE
2709 ENVELOPE
270A RAISED FIST
270B RAISED HAND
270C VICTORY HAND
270D WRITING HAND
270E LOWER RIGHT PENCIL
270F PENCIL
2710 UPPER RIGHT PENCIL
2711 WHITE NIB
2712 BLACK NIB
2713 CHECK MARK
2714 HEAVY CHECK MARK
2715 MULTIPLICATION X
2716 HEAVY MULTIPLICATION X
2717 BALLOT X
2718 HEAVY BALLOT X
2719 OUTLINED GREEK CROSS
271A HEAVY GREEK CROSS
271B OPEN CENTRE CROSS
271C HEAVY OPEN CENTRE CROSS
271D LATIN CROSS
271E SHADOWED WHITE LATIN CROSS
271F OUTLINED LATIN CROSS
2720 MALTESE CROSS
2721 STAR OF DAVID
2722 FOUR TEARDROP-SPOKED ASTERISK
2723 FOUR BALLOON-SPOKED ASTERISK
2724 HEAVY FOUR BALLOON-SPOKED ASTERISK
2725 FOUR CLUB-SPOKED ASTERISK
2726 BLACK FOUR POINTED STAR
2727 WHITE FOUR POINTED STAR
2728 SPARKLES
2729 STRESS OUTLINED WHITE STAR
272A CIRCLED WHITE STAR
272B OPEN CENTRE BLACK STAR
272C BLACK CENTRE WHITE STAR
272D OUTLINED BLACK STAR
272E HEAVY OUTLINED BLACK STAR
272F PINWHEEL STAR
2730 SHADOWED WHITE STAR
2731 HEAVY ASTERISK
2732 OPEN CENTRE ASTERISK
2733 EIGHT SPOKED ASTERISK
2734 EIGHT POINTED BLACK STAR
2735 EIGHT POINTED PINWHEEL STAR
2736 SIX POINTED BLACK STAR
2737 EIGHT POINTED RECTILINEAR BLACK STAR
2738 HEAVY EIGHT POINTED RECTILINEAR BLACK STAR
2739 TWELVE POINTED BLACK STAR
273A SIXTEEN POINTED ASTERISK
273B TEARDROP-SPOKED ASTERISK
273C OPEN CENTRE TEARDROP-SPOKED ASTERISK
273D HEAVY TEARDROP-SPOKED ASTERISK
273E SIX PETALLED BLACK AND WHITE FLORETTE
273F BLACK FLORETTE
2740 WHITE FLORETTE
2741 EIGHT PETALLED OUTLINED BLACK FLORETTE
2742 CIRCLED OPEN CENTRE EIGHT POINTED STAR
2743 HEAVY TEARDROP-SPOKED PINWHEEL ASTERISK
2744 SNOWFLAKE
2745 TIGHT TRIFOLIATE SNOWFLAKE
2746 HEAVY CHEVRON SNOWFLAKE
2747 SPARKLE
2748 HEAVY SPARKLE
2749 BALLOON-SPOKED ASTERISK
274A EIGHT TEARDROP-SPOKED PROPELLER ASTERISK
274B HEAVY EIGHT TEARDROP-SPOKED PROPELLER ASTERISK
274C CROSS MARK
274D SHADOWED WHITE CIRCLE
274E NEGATIVE SQUARED CROSS MARK
274F LOWER RIGHT DROP-SHADOWED WHITE SQUARE
2750 UPPER RIGHT DROP-SHADOWED WHITE SQUARE
2751 LOWER RIGHT SHADOWED WHITE SQUARE
2752 UPPER RIGHT SHADOWED WHITE SQUARE
2753 BLACK QUESTION MARK ORNAMENT
2754 WHITE QUESTION MARK ORNAMENT
2755 WHITE EXCLAMATION MARK ORNAMENT
2756 BLACK DIAMOND MINUS WHITE X
2757 HEAVY EXCLAMATION MARK SYMBOL
2758 LIGHT VERTICAL BAR
2759 MEDIUM VERTICAL BAR
275A HEAVY VERTICAL BAR
275B HEAVY SINGLE TURNED COMMA QUOTATION MARK ORNAMENT
275C HEAVY SINGLE COMMA QUOTATION MARK ORNAMENT
275D HEAVY DOUBLE TURNED COMMA QUOTATION MARK ORNAMENT
275E HEAVY DOUBLE COMMA QUOTATION MARK ORNAMENT
275F HEAVY LOW SINGLE COMMA QUOTATION MARK ORNAMENT
2760 HEAVY LOW DOUBLE COMMA QUOTATION MARK ORNAMENT
2761 CURVED STEM PARAGRAPH SIGN ORNAMENT
2762 HEAVY EXCLAMATION MARK ORNAMENT
2763 HEAVY HEART EXCLAMATION MARK ORNAMENT
2764 HEAVY BLACK HEART
2765 ROTATED HEAVY BLACK HEART BULLET
2766 FLORAL HEART
2767 ROTATED FLORAL HEART BULLET
2768 MEDIUM LEFT PARENTHESIS ORNAMENT
2769 MEDIUM RIGHT PARENTHESIS ORNAMENT
276A MEDIUM FLATTENED LEFT PARENTHESIS ORNAMENT
276B MEDIUM FLATTENED RIGHT PARENTHESIS ORNAMENT
276C MEDIUM LEFT-POINTING ANGLE BRACKET ORNAMENT
276D MEDIUM RIGHT-POINTING ANGLE BRACKET ORNAMENT
276E HEAVY LEFT-POINTING ANGLE QUOTATION MARK ORNAMENT
276F HEAVY RIGHT-POINTING ANGLE QUOTATION MARK ORNAMENT
2770 HEAVY LEFT-POINTING ANGLE BRACKET ORNAMENT
2771 HEAVY RIGHT-POINTING ANGLE BRACKET ORNAMENT
2772 LIGHT LEFT TORTOISE SHELL BRACKET ORNAMENT
2773 LIGHT RIGHT TORTOISE SHELL BRACKET ORNAMENT
2774 MEDIUM LEFT CURLY BRACKET ORNAMENT
2775 MEDIUM RIGHT CURLY BRACKET ORNAMENT
2776 DINGBAT NEGATIVE CIRCLED DIGIT ONE
2777 DINGBAT NEGATIVE CIRCLED DIGIT TWO
2778 DINGBAT NEGATIVE CIRCLED DIGIT THREE
2779 DINGBAT NEGATIVE CIRCLED DIGIT FOUR
277A DINGBAT NEGATIVE CIRCLED DIGIT FIVE
277B DINGBAT NEGATIVE CIRCLED DIGIT SIX
277C DINGBAT NEGATIVE CIRCLED DIGIT SEVEN
277D DINGBAT NEGATIVE CIRCLED DIGIT EIGHT
277E DINGBAT NEGATIVE CIRCLED DIGIT NINE
277F DINGBAT NEGATIVE CIRCLED NUMBER TEN
2780 DINGBAT CIRCLED SANS-SERIF DIGIT ONE
2781 DINGBAT CIRCLED SANS-SERIF DIGIT TWO
2782 DINGBAT CIRCLED SANS-SERIF DIGIT THREE
2783 DINGBAT CIRCLED SANS-SERIF DIGIT FOUR
2784 DINGBAT CIRCLED SANS-SERIF DIGIT FIVE
2785 DINGBAT CIRCLED SANS-SERIF DIGIT SIX
2786 DINGBAT CIRCLED SANS-SERIF DIGIT SEVEN
2787 DINGBAT CIRCLED SANS-SERIF DIGIT EIGHT
2788 DINGBAT CIRCLED SANS-SERIF DIGIT NINE
2789 DINGBAT CIRCLED SANS-SERIF NUMBER TEN
278A DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ONE
278B DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT TWO
278C DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT THREE
278D DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FOUR
278E DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FIVE
278F DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SIX
2790 DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SEVEN
2791 DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT EIGHT
2792 DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT NINE
2793 DINGBAT NEGATIVE CIRCLED SANS-SERIF NUMBER TEN
2794 HEAVY WIDE-HEADED RIGHTWARDS ARROW
2795 HEAVY PLUS SIGN
2796 HEAVY MINUS SIGN
2797 HEAVY DIVISION SIGN
2798 HEAVY SOUTH EAST ARROW
2799 HEAVY RIGHTWARDS ARROW
279A HEAVY NORTH EAST ARROW
279B DRAFTING POINT RIGHTWARDS ARROW
279C HEAVY ROUND-TIPPED RIGHTWARDS ARROW
279D TRIANGLE-HEADED RIGHTWARDS ARROW
279E HEAVY TRIANGLE-HEADED RIGHTWARDS ARROW
279F DASHED TRIANGLE-HEADED RIGHTWARDS ARROW
27A0 HEAVY DASHED TRIANGLE-HEADED RIGHTWARDS ARROW
27A1 BLACK RIGHTWARDS ARROW
27A2 THREE-D TOP-LIGHTED RIGHTWARDS ARROWHEAD
27A3 THREE-D BOTTOM-LIGHTED RIGHTWARDS ARROWHEAD
27A4 BLACK RIGHTWARDS ARROWHEAD
27A5 HEAVY BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROW
27A6 HEAVY BLACK CURVED UPWARDS AND RIGHTWARDS ARROW
27A7 SQUAT BLACK RIGHTWARDS ARROW
27A8 HEAVY CONCAVE-POINTED BLACK RIGHTWARDS ARROW
27A9 RIGHT-SHADED WHITE RIGHTWARDS ARROW
27AA LEFT-SHADED WHITE RIGHTWARDS ARROW
27AB BACK-TILTED SHADOWED WHITE RIGHTWARDS ARROW
27AC FRONT-TILTED SHADOWED WHITE RIGHTWARDS ARROW
27AD HEAVY LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27AE HEAVY UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27AF NOTCHED LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27B0 CURLY LOOP
27B1 NOTCHED UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
27B2 CIRCLED HEAVY WHITE RIGHTWARDS ARROW
27B3 WHITE-FEATHERED RIGHTWARDS ARROW
27B4 BLACK-FEATHERED SOUTH EAST ARROW
27B5 BLACK-FEATHERED RIGHTWARDS ARROW
27B6 BLACK-FEATHERED NORTH EAST ARROW
27B7 HEAVY BLACK-FEATHERED SOUTH EAST ARROW
27B8 HEAVY BLACK-FEATHERED RIGHTWARDS ARROW
27B9 HEAVY BLACK-FEATHERED NORTH EAST ARROW
27BA TEARDROP-BARBED RIGHTWARDS ARROW
27BB HEAVY TEARDROP-SHANKED RIGHTWARDS ARROW
27BC WEDGE-TAILED RIGHTWARDS ARROW
27BD HEAVY WEDGE-TAILED RIGHTWARDS ARROW
27BE OPEN-OUTLINED RIGHTWARDS ARROW
27BF DOUBLE CURLY LOOP
27C0 THREE DIMENSIONAL ANGLE
27C1 WHITE TRIANGLE CONTAINING SMALL WHITE TRIANGLE
27C2 PERPENDICULAR
27C3 OPEN SUBSET
27C4 OPEN SUPERSET
27C5 LEFT S-SHAPED BAG DELIMITER
27C6 RIGHT S-SHAPED BAG DELIMITER
27C7 OR WITH DOT INSIDE
27C8 REVERSE SOLIDUS PRECEDING SUBSET
27C9 SUPERSET PRECEDING SOLIDUS
27CA VERTICAL BAR WITH HORIZONTAL STROKE
27CB MATHEMATICAL RISING DIAGONAL
27CC LONG DIVISION
27CD MATHEMATICAL FALLING DIAGONAL
27CE SQUARED LOGICAL AND
27CF SQUARED LOGICAL OR
27D0 WHITE DIAMOND WITH CENTRED DOT
27D1 AND WITH DOT
27D2 ELEMENT OF OPENING UPWARDS
27D3 LOWER RIGHT CORNER WITH DOT
27D4 UPPER LEFT CORNER WITH DOT
27D5 LEFT OUTER JOIN
27D6 RIGHT OUTER JOIN
27D7 FULL OUTER JOIN
27D8 LARGE UP TACK
27D9 LARGE DOWN TACK
27DA LEFT AND RIGHT DOUBLE TURNSTILE
27DB LEFT AND RIGHT TACK
27DC LEFT MULTIMAP
27DD LONG RIGHT TACK
27DE LONG LEFT TACK
27DF UP TACK WITH CIRCLE ABOVE
27E0 LOZENGE DIVIDED BY HORIZONTAL RULE
27E1 WHITE CONCAVE-SIDED DIAMOND
27E2 WHITE CONCAVE-SIDED DIAMOND WITH LEFTWARDS TICK
27E3 WHITE CONCAVE-SIDED DIAMOND WITH RIGHTWARDS TICK
27E4 WHITE SQUARE WITH LEFTWARDS TICK
27E5 WHITE SQUARE WITH RIGHTWARDS TICK
27E6 MATHEMATICAL LEFT WHITE SQUARE BRACKET
27E7 MATHEMATICAL RIGHT WHITE SQUARE BRACKET
27E8 MATHEMATICAL LEFT ANGLE BRACKET
27E9 MATHEMATICAL RIGHT ANGLE BRACKET
27EA MATHEMATICAL LEFT DOUBLE ANGLE BRACKET
27EB MATHEMATICAL RIGHT DOUBLE ANGLE BRACKET
27EC MATHEMATICAL LEFT WHITE TORTOISE SHELL BRACKET
27ED MATHEMATICAL RIGHT WHITE TORTOISE SHELL BRACKET
27EE MATHEMATICAL LEFT FLATTENED PARENTHESIS
27EF MATHEMATICAL RIGHT FLATTENED PARENTHESIS
27F0 UPWARDS QUADRUPLE ARROW
27F1 DOWNWARDS QUADRUPLE ARROW
27F2 ANTICLOCKWISE GAPPED CIRCLE ARROW
27F3 CLOCKWISE GAPPED CIRCLE ARROW
27F4 RIGHT ARROW WITH CIRCLED PLUS
27F5 LONG LEFTWARDS ARROW
27F6 LONG RIGHTWARDS ARROW
27F7 LONG LEFT RIGHT ARROW
27F8 LONG LEFTWARDS DOUBLE ARROW
27F9 LONG RIGHTWARDS DOUBLE ARROW
27FA LONG LEFT RIGHT DOUBLE ARROW
27FB LONG LEFTWARDS ARROW FROM BAR
27FC LONG RIGHTWARDS ARROW FROM BAR
27FD LONG LEFTWARDS DOUBLE ARROW FROM BAR
27FE LONG RIGHTWARDS DOUBLE ARROW FROM BAR
27FF LONG RIGHTWARDS SQUIGGLE ARROW
2800 BRAILLE PATTERN BLANK
2801 BRAILLE PATTERN DOTS-1
2802 BRAILLE PATTERN DOTS-2
2803 BRAILLE PATTERN DOTS-12
2804 BRAILLE PATTERN DOTS-3
2805 BRAILLE PATTERN DOTS-13
2806 BRAILLE PATTERN DOTS-23
2807 BRAILLE PATTERN DOTS-123
2808 BRAILLE PATTERN DOTS-4
2809 BRAILLE PATTERN DOTS-14
280A BRAILLE PATTERN DOTS-24
280B BRAILLE PATTERN DOTS-124
280C BRAILLE PATTERN DOTS-34
280D BRAILLE PATTERN DOTS-134
280E BRAILLE PATTERN DOTS-234
280F BRAILLE PATTERN DOTS-1234
2810 BRAILLE PATTERN DOTS-5
2811 BRAILLE PATTERN DOTS-15
2812 BRAILLE PATTERN DOTS-25
2813 BRAILLE PATTERN DOTS-125
2814 BRAILLE PATTERN DOTS-35
2815 BRAILLE PATTERN DOTS-135
2816 BRAILLE PATTERN DOTS-235
2817 BRAILLE PATTERN DOTS-1235
2818 BRAILLE PATTERN DOTS-45
2819 BRAILLE PATTERN DOTS-145
281A BRAILLE PATTERN DOTS-245
281B BRAILLE PATTERN DOTS-1245
281C BRAILLE PATTERN DOTS-345
281D BRAILLE PATTERN DOTS-1345
281E BRAILLE PATTERN DOTS-2345
281F BRAILLE PATTERN DOTS-12345
2820 BRAILLE PATTERN DOTS-6
2821 BRAILLE PATTERN DOTS-16
2822 BRAILLE PATTERN DOTS-26
2823 BRAILLE PATTERN DOTS-126
2824 BRAILLE PATTERN DOTS-36
2825 BRAILLE PATTERN DOTS-136
2826 BRAILLE PATTERN DOTS-236
2827 BRAILLE PATTERN DOTS-1236
2828 BRAILLE PATTERN DOTS-46
2829 BRAILLE PATTERN DOTS-146
282A BRAILLE PATTERN DOTS-246
282B BRAILLE PATTERN DOTS-1246
282C BRAILLE PATTERN DOTS-346
282D BRAILLE PATTERN DOTS-1346
282E BRAILLE PATTERN DOTS-2346
282F BRAILLE PATTERN DOTS-12346
2830 BRAILLE PATTERN DOTS-56
2831 BRAILLE PATTERN DOTS-156
2832 BRAILLE PATTERN DOTS-256
2833 BRAILLE PATTERN DOTS-1256
2834 BRAILLE PATTERN DOTS-356
2835 BRAILLE PATTERN DOTS-1356
2836 BRAILLE PATTERN DOTS-2356
2837 BRAILLE PATTERN DOTS-12356
2838 BRAILLE PATTERN DOTS-456
2839 BRAILLE PATTERN DOTS-1456
283A BRAILLE PATTERN DOTS-2456
283B BRAILLE PATTERN DOTS-12456
283C BRAILLE PATTERN DOTS-3456
283D BRAILLE PATTERN DOTS-13456
283E BRAILLE PATTERN DOTS-23456
283F BRAILLE PATTERN DOTS-123456
2840 BRAILLE PATTERN DOTS-7
2841 BRAILLE PATTERN DOTS-17
2842 BRAILLE PATTERN DOTS-27
2843 BRAILLE PATTERN DOTS-127
2844 BRAILLE PATTERN DOTS-37
2845 BRAILLE PATTERN DOTS-137
2846 BRAILLE PATTERN DOTS-237
2847 BRAILLE PATTERN DOTS-1237
2848 BRAILLE PATTERN DOTS-47
2849 BRAILLE PATTERN DOTS-147
284A BRAILLE PATTERN DOTS-247
284B BRAILLE PATTERN DOTS-1247
284C BRAILLE PATTERN DOTS-347
284D BRAILLE PATTERN DOTS-1347
284E BRAILLE PATTERN DOTS-2347
284F BRAILLE PATTERN DOTS-12347
2850 BRAILLE PATTERN DOTS-57
2851 BRAILLE PATTERN DOTS-157
2852 BRAILLE PATTERN DOTS-257
2853 BRAILLE PATTERN DOTS-1257
2854 BRAILLE PATTERN DOTS-357
2855 BRAILLE PATTERN DOTS-1357
2856 BRAILLE PATTERN DOTS-2357
2857 BRAILLE PATTERN DOTS-12357
2858 BRAILLE PATTERN DOTS-457
2859 BRAILLE PATTERN DOTS-1457
285A BRAILLE PATTERN DOTS-2457
285B BRAILLE PATTERN DOTS-12457
285C BRAILLE PATTERN DOTS-3457
285D BRAILLE PATTERN DOTS-13457
285E BRAILLE PATTERN DOTS-23457
285F BRAILLE PATTERN DOTS-123457
2860 BRAILLE PATTERN DOTS-67
2861 BRAILLE PATTERN DOTS-167
2862 BRAILLE PATTERN DOTS-267
2863 BRAILLE PATTERN DOTS-1267
2864 BRAILLE PATTERN DOTS-367
2865 BRAILLE PATTERN DOTS-1367
2866 BRAILLE PATTERN DOTS-2367
2867 BRAILLE PATTERN DOTS-12367
2868 BRAILLE PATTERN DOTS-467
2869 BRAILLE PATTERN DOTS-1467
286A BRAILLE PATTERN DOTS-2467
286B BRAILLE PATTERN DOTS-12467
286C BRAILLE PATTERN DOTS-3467
286D BRAILLE PATTERN DOTS-13467
286E BRAILLE PATTERN DOTS-23467
286F BRAILLE PATTERN DOTS-123467
2870 BRAILLE PATTERN DOTS-567
2871 BRAILLE PATTERN DOTS-1567
2872 BRAILLE PATTERN DOTS-2567
2873 BRAILLE PATTERN DOTS-12567
2874 BRAILLE PATTERN DOTS-3567
2875 BRAILLE PATTERN DOTS-13567
2876 BRAILLE PATTERN DOTS-23567
2877 BRAILLE PATTERN DOTS-123567
2878 BRAILLE PATTERN DOTS-4567
2879 BRAILLE PATTERN DOTS-14567
287A BRAILLE PATTERN DOTS-24567
287B BRAILLE PATTERN DOTS-124567
287C BRAILLE PATTERN DOTS-34567
287D BRAILLE PATTERN DOTS-134567
287E BRAILLE PATTERN DOTS-234567
287F BRAILLE PATTERN DOTS-1234567
2880 BRAILLE PATTERN DOTS-8
2881 BRAILLE PATTERN DOTS-18
2882 BRAILLE PATTERN DOTS-28
2883 BRAILLE PATTERN DOTS-128
2884 BRAILLE PATTERN DOTS-38
2885 BRAILLE PATTERN DOTS-138
2886 BRAILLE PATTERN DOTS-238
2887 BRAILLE PATTERN DOTS-1238
2888 BRAILLE PATTERN DOTS-48
2889 BRAILLE PATTERN DOTS-148
288A BRAILLE PATTERN DOTS-248
288B BRAILLE PATTERN DOTS-1248
288C BRAILLE PATTERN DOTS-348
288D BRAILLE PATTERN DOTS-1348
288E BRAILLE PATTERN DOTS-2348
288F BRAILLE PATTERN DOTS-12348
2890 BRAILLE PATTERN DOTS-58
2891 BRAILLE PATTERN DOTS-158
2892 BRAILLE PATTERN DOTS-258
2893 BRAILLE PATTERN DOTS-1258
2894 BRAILLE PATTERN DOTS-358
2895 BRAILLE PATTERN DOTS-1358
2896 BRAILLE PATTERN DOTS-2358
2897 BRAILLE PATTERN DOTS-12358
2898 BRAILLE PATTERN DOTS-458
2899 BRAILLE PATTERN DOTS-1458
289A BRAILLE PATTERN DOTS-2458
289B BRAILLE PATTERN DOTS-12458
289C BRAILLE PATTERN DOTS-3458
289D BRAILLE PATTERN DOTS-13458
289E BRAILLE PATTERN DOTS-23458
289F BRAILLE PATTERN DOTS-123458
28A0 BRAILLE PATTERN DOTS-68
28A1 BRAILLE PATTERN DOTS-168
28A2 BRAILLE PATTERN DOTS-268
28A3 BRAILLE PATTERN DOTS-1268
28A4 BRAILLE PATTERN DOTS-368
28A5 BRAILLE PATTERN DOTS-1368
28A6 BRAILLE PATTERN DOTS-2368
28A7 BRAILLE PATTERN DOTS-12368
28A8 BRAILLE PATTERN DOTS-468
28A9 BRAILLE PATTERN DOTS-1468
28AA BRAILLE PATTERN DOTS-2468
28AB BRAILLE PATTERN DOTS-12468
28AC BRAILLE PATTERN DOTS-3468
28AD BRAILLE PATTERN DOTS-13468
28AE BRAILLE PATTERN DOTS-23468
28AF BRAILLE PATTERN DOTS-123468
28B0 BRAILLE PATTERN DOTS-568
28B1 BRAILLE PATTERN DOTS-1568
28B2 BRAILLE PATTERN DOTS-2568
28B3 BRAILLE PATTERN DOTS-12568
28B4 BRAILLE PATTERN DOTS-3568
28B5 BRAILLE PATTERN DOTS-13568
28B6 BRAILLE PATTERN DOTS-23568
28B7 BRAILLE PATTERN DOTS-123568
28B8 BRAILLE PATTERN DOTS-4568
28B9 BRAILLE PATTERN DOTS-14568
28BA BRAILLE PATTERN DOTS-24568
28BB BRAILLE PATTERN DOTS-124568
28BC BRAILLE PATTERN DOTS-34568
28BD BRAILLE PATTERN DOTS-134568
28BE BRAILLE PATTERN DOTS-234568
28BF BRAILLE PATTERN DOTS-1234568
28C0 BRAILLE PATTERN DOTS-78
28C1 BRAILLE PATTERN DOTS-178
28C2 BRAILLE PATTERN DOTS-278
28C3 BRAILLE PATTERN DOTS-1278
28C4 BRAILLE PATTERN DOTS-378
28C5 BRAILLE PATTERN DOTS-1378
28C6 BRAILLE PATTERN DOTS-2378
28C7 BRAILLE PATTERN DOTS-12378
28C8 BRAILLE PATTERN DOTS-478
28C9 BRAILLE PATTERN DOTS-1478
28CA BRAILLE PATTERN DOTS-2478
28CB BRAILLE PATTERN DOTS-12478
28CC BRAILLE PATTERN DOTS-3478
28CD BRAILLE PATTERN DOTS-13478
28CE BRAILLE PATTERN DOTS-23478
28CF BRAILLE PATTERN DOTS-123478
28D0 BRAILLE PATTERN DOTS-578
28D1 BRAILLE PATTERN DOTS-1578
28D2 BRAILLE PATTERN DOTS-2578
28D3 BRAILLE PATTERN DOTS-12578
28D4 BRAILLE PATTERN DOTS-3578
28D5 BRAILLE PATTERN DOTS-13578
28D6 BRAILLE PATTERN DOTS-23578
28D7 BRAILLE PATTERN DOTS-123578
28D8 BRAILLE PATTERN DOTS-4578
28D9 BRAILLE PATTERN DOTS-14578
28DA BRAILLE PATTERN DOTS-24578
28DB BRAILLE PATTERN DOTS-124578
28DC BRAILLE PATTERN DOTS-34578
28DD BRAILLE PATTERN DOTS-134578
28DE BRAILLE PATTERN DOTS-234578
28DF BRAILLE PATTERN DOTS-1234578
28E0 BRAILLE PATTERN DOTS-678
28E1 BRAILLE PATTERN DOTS-1678
28E2 BRAILLE PATTERN DOTS-2678
28E3 BRAILLE PATTERN DOTS-12678
28E4 BRAILLE PATTERN DOTS-3678
28E5 BRAILLE PATTERN DOTS-13678
28E6 BRAILLE PATTERN DOTS-23678
28E7 BRAILLE PATTERN DOTS-123678
28E8 BRAILLE PATTERN DOTS-4678
28E9 BRAILLE PATTERN DOTS-14678
28EA BRAILLE PATTERN DOTS-24678
28EB BRAILLE PATTERN DOTS-124678
28EC BRAILLE PATTERN DOTS-34678
28ED BRAILLE PATTERN DOTS-134678
28EE BRAILLE PATTERN DOTS-234678
28EF BRAILLE PATTERN DOTS-1234678
28F0 BRAILLE PATTERN DOTS-5678
28F1 BRAILLE PATTERN DOTS-15678
28F2 BRAILLE PATTERN DOTS-25678
28F3 BRAILLE PATTERN DOTS-125678
28F4 BRAILLE PATTERN DOTS-35678
28F5 BRAILLE PATTERN DOTS-135678
28F6 BRAILLE PATTERN DOTS-235678
28F7 BRAILLE PATTERN DOTS-1235678
28F8 BRAILLE PATTERN DOTS-45678
28F9 BRAILLE PATTERN DOTS-145678
28FA BRAILLE PATTERN DOTS-245678
28FB BRAILLE PATTERN DOTS-1245678
28FC BRAILLE PATTERN DOTS-345678
28FD BRAILLE PATTERN DOTS-1345678
28FE BRAILLE PATTERN DOTS-2345678
28FF BRAILLE PATTERN DOTS-12345678
2900 RIGHTWARDS TWO-HEADED ARROW WITH VERTICAL STROKE
2901 RIGHTWARDS TWO-HEADED ARROW WITH DOUBLE VERTICAL STROKE
2902 LEFTWARDS DOUBLE ARROW WITH VERTICAL STROKE
2903 RIGHTWARDS DOUBLE ARROW WITH VERTICAL STROKE
2904 LEFT RIGHT DOUBLE ARROW WITH VERTICAL STROKE
2905 RIGHTWARDS TWO-HEADED ARROW FROM BAR
2906 LEFTWARDS DOUBLE ARROW FROM BAR
2907 RIGHTWARDS DOUBLE ARROW FROM BAR
2908 DOWNWARDS ARROW WITH HORIZONTAL STROKE
2909 UPWARDS ARROW WITH HORIZONTAL STROKE
290A UPWARDS TRIPLE ARROW
290B DOWNWARDS TRIPLE ARROW
290C LEFTWARDS DOUBLE DASH ARROW
290D RIGHTWARDS DOUBLE DASH ARROW
290E LEFTWARDS TRIPLE DASH ARROW
290F RIGHTWARDS TRIPLE DASH ARROW
2910 RIGHTWARDS TWO-HEADED TRIPLE DASH ARROW
2911 RIGHTWARDS ARROW WITH DOTTED STEM
2912 UPWARDS ARROW TO BAR
2913 DOWNWARDS ARROW TO BAR
2914 RIGHTWARDS ARROW WITH TAIL WITH VERTICAL STROKE
2915 RIGHTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE
2916 RIGHTWARDS TWO-HEADED ARROW WITH TAIL
2917 RIGHTWARDS TWO-HEADED ARROW WITH TAIL WITH VERTICAL STROKE
2918 RIGHTWARDS TWO-HEADED ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE
2919 LEFTWARDS ARROW-TAIL
291A RIGHTWARDS ARROW-TAIL
291B LEFTWARDS DOUBLE ARROW-TAIL
291C RIGHTWARDS DOUBLE ARROW-TAIL
291D LEFTWARDS ARROW TO BLACK DIAMOND
291E RIGHTWARDS ARROW TO BLACK DIAMOND
291F LEFTWARDS ARROW FROM BAR TO BLACK DIAMOND
2920 RIGHTWARDS ARROW FROM BAR TO BLACK DIAMOND
2921 NORTH WEST AND SOUTH EAST ARROW
2922 NORTH EAST AND SOUTH WEST ARROW
2923 NORTH WEST ARROW WITH HOOK
2924 NORTH EAST ARROW WITH HOOK
2925 SOUTH EAST ARROW WITH HOOK
2926 SOUTH WEST ARROW WITH HOOK
2927 NORTH WEST ARROW AND NORTH EAST ARROW
2928 NORTH EAST ARROW AND SOUTH EAST ARROW
2929 SOUTH EAST ARROW AND SOUTH WEST ARROW
292A SOUTH WEST ARROW AND NORTH WEST ARROW
292B RISING DIAGONAL CROSSING FALLING DIAGONAL
292C FALLING DIAGONAL CROSSING RISING DIAGONAL
292D SOUTH EAST ARROW CROSSING NORTH EAST ARROW
292E NORTH EAST ARROW CROSSING SOUTH EAST ARROW
292F FALLING DIAGONAL CROSSING NORTH EAST ARROW
2930 RISING DIAGONAL CROSSING SOUTH EAST ARROW
2931 NORTH EAST ARROW CROSSING NORTH WEST ARROW
2932 NORTH WEST ARROW CROSSING NORTH EAST ARROW
2933 WAVE ARROW POINTING DIRECTLY RIGHT
2934 ARROW POINTING RIGHTWARDS THEN CURVING UPWARDS
2935 ARROW POINTING RIGHTWARDS THEN CURVING DOWNWARDS
2936 ARROW POINTING DOWNWARDS THEN CURVING LEFTWARDS
2937 ARROW POINTING DOWNWARDS THEN CURVING RIGHTWARDS
2938 RIGHT-SIDE ARC CLOCKWISE ARROW
2939 LEFT-SIDE ARC ANTICLOCKWISE ARROW
293A TOP ARC ANTICLOCKWISE ARROW
293B BOTTOM ARC ANTICLOCKWISE ARROW
293C TOP ARC CLOCKWISE ARROW WITH MINUS
293D TOP ARC ANTICLOCKWISE ARROW WITH PLUS
293E LOWER RIGHT SEMICIRCULAR CLOCKWISE ARROW
293F LOWER LEFT SEMICIRCULAR ANTICLOCKWISE ARROW
2940 ANTICLOCKWISE CLOSED CIRCLE ARROW
2941 CLOCKWISE CLOSED CIRCLE ARROW
2942 RIGHTWARDS ARROW ABOVE SHORT LEFTWARDS ARROW
2943 LEFTWARDS ARROW ABOVE SHORT RIGHTWARDS ARROW
2944 SHORT RIGHTWARDS ARROW ABOVE LEFTWARDS ARROW
2945 RIGHTWARDS ARROW WITH PLUS BELOW
2946 LEFTWARDS ARROW WITH PLUS BELOW
2947 RIGHTWARDS ARROW THROUGH X
2948 LEFT RIGHT ARROW THROUGH SMALL CIRCLE
2949 UPWARDS TWO-HEADED ARROW FROM SMALL CIRCLE
294A LEFT BARB UP RIGHT BARB DOWN HARPOON
294B LEFT BARB DOWN RIGHT BARB UP HARPOON
294C UP BARB RIGHT DOWN BARB LEFT HARPOON
294D UP BARB LEFT DOWN BARB RIGHT HARPOON
294E LEFT BARB UP RIGHT BARB UP HARPOON
294F UP BARB RIGHT DOWN BARB RIGHT HARPOON
2950 LEFT BARB DOWN RIGHT BARB DOWN HARPOON
2951 UP BARB LEFT DOWN BARB LEFT HARPOON
2952 LEFTWARDS HARPOON WITH BARB UP TO BAR
2953 RIGHTWARDS HARPOON WITH BARB UP TO BAR
2954 UPWARDS HARPOON WITH BARB RIGHT TO BAR
2955 DOWNWARDS HARPOON WITH BARB RIGHT TO BAR
2956 LEFTWARDS HARPOON WITH BARB DOWN TO BAR
2957 RIGHTWARDS HARPOON WITH BARB DOWN TO BAR
2958 UPWARDS HARPOON WITH BARB LEFT TO BAR
2959 DOWNWARDS HARPOON WITH BARB LEFT TO BAR
295A LEFTWARDS HARPOON WITH BARB UP FROM BAR
295B RIGHTWARDS HARPOON WITH BARB UP FROM BAR
295C UPWARDS HARPOON WITH BARB RIGHT FROM BAR
295D DOWNWARDS HARPOON WITH BARB RIGHT FROM BAR
295E LEFTWARDS HARPOON WITH BARB DOWN FROM BAR
295F RIGHTWARDS HARPOON WITH BARB DOWN FROM BAR
2960 UPWARDS HARPOON WITH BARB LEFT FROM BAR
2961 DOWNWARDS HARPOON WITH BARB LEFT FROM BAR
2962 LEFTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB DOWN
2963 UPWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHT
2964 RIGHTWARDS HARPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB DOWN
2965 DOWNWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHT
2966 LEFTWARDS HARPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB UP
2967 LEFTWARDS HARPOON WITH BARB DOWN ABOVE RIGHTWARDS HARPOON WITH BARB DOWN
2968 RIGHTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB UP
2969 RIGHTWARDS HARPOON WITH BARB DOWN ABOVE LEFTWARDS HARPOON WITH BARB DOWN
296A LEFTWARDS HARPOON WITH BARB UP ABOVE LONG DASH
296B LEFTWARDS HARPOON WITH BARB DOWN BELOW LONG DASH
296C RIGHTWARDS HARPOON WITH BARB UP ABOVE LONG DASH
296D RIGHTWARDS HARPOON WITH BARB DOWN BELOW LONG DASH
296E UPWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHT
296F DOWNWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHT
2970 RIGHT DOUBLE ARROW WITH ROUNDED HEAD
2971 EQUALS SIGN ABOVE RIGHTWARDS ARROW
2972 TILDE OPERATOR ABOVE RIGHTWARDS ARROW
2973 LEFTWARDS ARROW ABOVE TILDE OPERATOR
2974 RIGHTWARDS ARROW ABOVE TILDE OPERATOR
2975 RIGHTWARDS ARROW ABOVE ALMOST EQUAL TO
2976 LESS-THAN ABOVE LEFTWARDS ARROW
2977 LEFTWARDS ARROW THROUGH LESS-THAN
2978 GREATER-THAN ABOVE RIGHTWARDS ARROW
2979 SUBSET ABOVE RIGHTWARDS ARROW
297A LEFTWARDS ARROW THROUGH SUBSET
297B SUPERSET ABOVE LEFTWARDS ARROW
297C LEFT FISH TAIL
297D RIGHT FISH TAIL
297E UP FISH TAIL
297F DOWN FISH TAIL
2980 TRIPLE VERTICAL BAR DELIMITER
2981 Z NOTATION SPOT
2982 Z NOTATION TYPE COLON
2983 LEFT WHITE CURLY BRACKET
2984 RIGHT WHITE CURLY BRACKET
2985 LEFT WHITE PARENTHESIS
2986 RIGHT WHITE PARENTHESIS
2987 Z NOTATION LEFT IMAGE BRACKET
2988 Z NOTATION RIGHT IMAGE BRACKET
2989 Z NOTATION LEFT BINDING BRACKET
298A Z NOTATION RIGHT BINDING BRACKET
298B LEFT SQUARE BRACKET WITH UNDERBAR
298C RIGHT SQUARE BRACKET WITH UNDERBAR
298D LEFT SQUARE BRACKET WITH TICK IN TOP CORNER
298E RIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
298F LEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
2990 RIGHT SQUARE BRACKET WITH TICK IN TOP CORNER
2991 LEFT ANGLE BRACKET WITH DOT
2992 RIGHT ANGLE BRACKET WITH DOT
2993 LEFT ARC LESS-THAN BRACKET
2994 RIGHT ARC GREATER-THAN BRACKET
2995 DOUBLE LEFT ARC GREATER-THAN BRACKET
2996 DOUBLE RIGHT ARC LESS-THAN BRACKET
2997 LEFT BLACK TORTOISE SHELL BRACKET
2998 RIGHT BLACK TORTOISE SHELL BRACKET
2999 DOTTED FENCE
299A VERTICAL ZIGZAG LINE
299B MEASURED ANGLE OPENING LEFT
299C RIGHT ANGLE VARIANT WITH SQUARE
299D MEASURED RIGHT ANGLE WITH DOT
299E ANGLE WITH S INSIDE
299F ACUTE ANGLE
29A0 SPHERICAL ANGLE OPENING LEFT
29A1 SPHERICAL ANGLE OPENING UP
29A2 TURNED ANGLE
29A3 REVERSED ANGLE
29A4 ANGLE WITH UNDERBAR
29A5 REVERSED ANGLE WITH UNDERBAR
29A6 OBLIQUE ANGLE OPENING UP
29A7 OBLIQUE ANGLE OPENING DOWN
29A8 MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND RIGHT
29A9 MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND LEFT
29AA MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND RIGHT
29AB MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND LEFT
29AC MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND UP
29AD MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND UP
29AE MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND DOWN
29AF MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND DOWN
29B0 REVERSED EMPTY SET
29B1 EMPTY SET WITH OVERBAR
29B2 EMPTY SET WITH SMALL CIRCLE ABOVE
29B3 EMPTY SET WITH RIGHT ARROW ABOVE
29B4 EMPTY SET WITH LEFT ARROW ABOVE
29B5 CIRCLE WITH HORIZONTAL BAR
29B6 CIRCLED VERTICAL BAR
29B7 CIRCLED PARALLEL
29B8 CIRCLED REVERSE SOLIDUS
29B9 CIRCLED PERPENDICULAR
29BA CIRCLE DIVIDED BY HORIZONTAL BAR AND TOP HALF DIVIDED BY VERTICAL BAR
29BB CIRCLE WITH SUPERIMPOSED X
29BC CIRCLED ANTICLOCKWISE-ROTATED DIVISION SIGN
29BD UP ARROW THROUGH CIRCLE
29BE CIRCLED WHITE BULLET
29BF CIRCLED BULLET
29C0 CIRCLED LESS-THAN
29C1 CIRCLED GREATER-THAN
29C2 CIRCLE WITH SMALL CIRCLE TO THE RIGHT
29C3 CIRCLE WITH TWO HORIZONTAL STROKES TO THE RIGHT
29C4 SQUARED RISING DIAGONAL SLASH
29C5 SQUARED FALLING DIAGONAL SLASH
29C6 SQUARED ASTERISK
29C7 SQUARED SMALL CIRCLE
29C8 SQUARED SQUARE
29C9 TWO JOINED SQUARES
29CA TRIANGLE WITH DOT ABOVE
29CB TRIANGLE WITH UNDERBAR
29CC S IN TRIANGLE
29CD TRIANGLE WITH SERIFS AT BOTTOM
29CE RIGHT TRIANGLE ABOVE LEFT TRIANGLE
29CF LEFT TRIANGLE BESIDE VERTICAL BAR
29D0 VERTICAL BAR BESIDE RIGHT TRIANGLE
29D1 BOWTIE WITH LEFT HALF BLACK
29D2 BOWTIE WITH RIGHT HALF BLACK
29D3 BLACK BOWTIE
29D4 TIMES WITH LEFT HALF BLACK
29D5 TIMES WITH RIGHT HALF BLACK
29D6 WHITE HOURGLASS
29D7 BLACK HOURGLASS
29D8 LEFT WIGGLY FENCE
29D9 RIGHT WIGGLY FENCE
29DA LEFT DOUBLE WIGGLY FENCE
29DB RIGHT DOUBLE WIGGLY FENCE
29DC INCOMPLETE INFINITY
29DD TIE OVER INFINITY
29DE INFINITY NEGATED WITH VERTICAL BAR
29DF DOUBLE-ENDED MULTIMAP
29E0 SQUARE WITH CONTOURED OUTLINE
29E1 INCREASES AS
29E2 SHUFFLE PRODUCT
29E3 EQUALS SIGN AND SLANTED PARALLEL
29E4 EQUALS SIGN AND SLANTED PARALLEL WITH TILDE ABOVE
29E5 IDENTICAL TO AND SLANTED PARALLEL
29E6 GLEICH STARK
29E7 THERMODYNAMIC
29E8 DOWN-POINTING TRIANGLE WITH LEFT HALF BLACK
29E9 DOWN-POINTING TRIANGLE WITH RIGHT HALF BLACK
29EA BLACK DIAMOND WITH DOWN ARROW
29EB BLACK LOZENGE
29EC WHITE CIRCLE WITH DOWN ARROW
29ED BLACK CIRCLE WITH DOWN ARROW
29EE ERROR-BARRED WHITE SQUARE
29EF ERROR-BARRED BLACK SQUARE
29F0 ERROR-BARRED WHITE DIAMOND
29F1 ERROR-BARRED BLACK DIAMOND
29F2 ERROR-BARRED WHITE CIRCLE
29F3 ERROR-BARRED BLACK CIRCLE
29F4 RULE-DELAYED
29F5 REVERSE SOLIDUS OPERATOR
29F6 SOLIDUS WITH OVERBAR
29F7 REVERSE SOLIDUS WITH HORIZONTAL STROKE
29F8 BIG SOLIDUS
29F9 BIG REVERSE SOLIDUS
29FA DOUBLE PLUS
29FB TRIPLE PLUS
29FC LEFT-POINTING CURVED ANGLE BRACKET
29FD RIGHT-POINTING CURVED ANGLE BRACKET
29FE TINY
29FF MINY
2A00 N-ARY CIRCLED DOT OPERATOR
2A01 N-ARY CIRCLED PLUS OPERATOR
2A02 N-ARY CIRCLED TIMES OPERATOR
2A03 N-ARY UNION OPERATOR WITH DOT
2A04 N-ARY UNION OPERATOR WITH PLUS
2A05 N-ARY SQUARE INTERSECTION OPERATOR
2A06 N-ARY SQUARE UNION OPERATOR
2A07 TWO LOGICAL AND OPERATOR
2A08 TWO LOGICAL OR OPERATOR
2A09 N-ARY TIMES OPERATOR
2A0A MODULO TWO SUM
2A0B SUMMATION WITH INTEGRAL
2A0C QUADRUPLE INTEGRAL OPERATOR
2A0D FINITE PART INTEGRAL
2A0E INTEGRAL WITH DOUBLE STROKE
2A0F INTEGRAL AVERAGE WITH SLASH
2A10 CIRCULATION FUNCTION
2A11 ANTICLOCKWISE INTEGRATION
2A12 LINE INTEGRATION WITH RECTANGULAR PATH AROUND POLE
2A13 LINE INTEGRATION WITH SEMICIRCULAR PATH AROUND POLE
2A14 LINE INTEGRATION NOT INCLUDING THE POLE
2A15 INTEGRAL AROUND A POINT OPERATOR
2A16 QUATERNION INTEGRAL OPERATOR
2A17 INTEGRAL WITH LEFTWARDS ARROW WITH HOOK
2A18 INTEGRAL WITH TIMES SIGN
2A19 INTEGRAL WITH INTERSECTION
2A1A INTEGRAL WITH UNION
2A1B INTEGRAL WITH OVERBAR
2A1C INTEGRAL WITH UNDERBAR
2A1D JOIN
2A1E LARGE LEFT TRIANGLE OPERATOR
2A1F Z NOTATION SCHEMA COMPOSITION
2A20 Z NOTATION SCHEMA PIPING
2A21 Z NOTATION SCHEMA PROJECTION
2A22 PLUS SIGN WITH SMALL CIRCLE ABOVE
2A23 PLUS SIGN WITH CIRCUMFLEX ACCENT ABOVE
2A24 PLUS SIGN WITH TILDE ABOVE
2A25 PLUS SIGN WITH DOT BELOW
2A26 PLUS SIGN WITH TILDE BELOW
2A27 PLUS SIGN WITH SUBSCRIPT TWO
2A28 PLUS SIGN WITH BLACK TRIANGLE
2A29 MINUS SIGN WITH COMMA ABOVE
2A2A MINUS SIGN WITH DOT BELOW
2A2B MINUS SIGN WITH FALLING DOTS
2A2C MINUS SIGN WITH RISING DOTS
2A2D PLUS SIGN IN LEFT HALF CIRCLE
2A2E PLUS SIGN IN RIGHT HALF CIRCLE
2A2F VECTOR OR CROSS PRODUCT
2A30 MULTIPLICATION SIGN WITH DOT ABOVE
2A31 MULTIPLICATION SIGN WITH UNDERBAR
2A32 SEMIDIRECT PRODUCT WITH BOTTOM CLOSED
2A33 SMASH PRODUCT
2A34 MULTIPLICATION SIGN IN LEFT HALF CIRCLE
2A35 MULTIPLICATION SIGN IN RIGHT HALF CIRCLE
2A36 CIRCLED MULTIPLICATION SIGN WITH CIRCUMFLEX ACCENT
2A37 MULTIPLICATION SIGN IN DOUBLE CIRCLE
2A38 CIRCLED DIVISION SIGN
2A39 PLUS SIGN IN TRIANGLE
2A3A MINUS SIGN IN TRIANGLE
2A3B MULTIPLICATION SIGN IN TRIANGLE
2A3C INTERIOR PRODUCT
2A3D RIGHTHAND INTERIOR PRODUCT
2A3E Z NOTATION RELATIONAL COMPOSITION
2A3F AMALGAMATION OR COPRODUCT
2A40 INTERSECTION WITH DOT
2A41 UNION WITH MINUS SIGN
2A42 UNION WITH OVERBAR
2A43 INTERSECTION WITH OVERBAR
2A44 INTERSECTION WITH LOGICAL AND
2A45 UNION WITH LOGICAL OR
2A46 UNION ABOVE INTERSECTION
2A47 INTERSECTION ABOVE UNION
2A48 UNION ABOVE BAR ABOVE INTERSECTION
2A49 INTERSECTION ABOVE BAR ABOVE UNION
2A4A UNION BESIDE AND JOINED WITH UNION
2A4B INTERSECTION BESIDE AND JOINED WITH INTERSECTION
2A4C CLOSED UNION WITH SERIFS
2A4D CLOSED INTERSECTION WITH SERIFS
2A4E DOUBLE SQUARE INTERSECTION
2A4F DOUBLE SQUARE UNION
2A50 CLOSED UNION WITH SERIFS AND SMASH PRODUCT
2A51 LOGICAL AND WITH DOT ABOVE
2A52 LOGICAL OR WITH DOT ABOVE
2A53 DOUBLE LOGICAL AND
2A54 DOUBLE LOGICAL OR
2A55 TWO INTERSECTING LOGICAL AND
2A56 TWO INTERSECTING LOGICAL OR
2A57 SLOPING LARGE OR
2A58 SLOPING LARGE AND
2A59 LOGICAL OR OVERLAPPING LOGICAL AND
2A5A LOGICAL AND WITH MIDDLE STEM
2A5B LOGICAL OR WITH MIDDLE STEM
2A5C LOGICAL AND WITH HORIZONTAL DASH
2A5D LOGICAL OR WITH HORIZONTAL DASH
2A5E LOGICAL AND WITH DOUBLE OVERBAR
2A5F LOGICAL AND WITH UNDERBAR
2A60 LOGICAL AND WITH DOUBLE UNDERBAR
2A61 SMALL VEE WITH UNDERBAR
2A62 LOGICAL OR WITH DOUBLE OVERBAR
2A63 LOGICAL OR WITH DOUBLE UNDERBAR
2A64 Z NOTATION DOMAIN ANTIRESTRICTION
2A65 Z NOTATION RANGE ANTIRESTRICTION
2A66 EQUALS SIGN WITH DOT BELOW
2A67 IDENTICAL WITH DOT ABOVE
2A68 TRIPLE HORIZONTAL BAR WITH DOUBLE VERTICAL STROKE
2A69 TRIPLE HORIZONTAL BAR WITH TRIPLE VERTICAL STROKE
2A6A TILDE OPERATOR WITH DOT ABOVE
2A6B TILDE OPERATOR WITH RISING DOTS
2A6C SIMILAR MINUS SIMILAR
2A6D CONGRUENT WITH DOT ABOVE
2A6E EQUALS WITH ASTERISK
2A6F ALMOST EQUAL TO WITH CIRCUMFLEX ACCENT
2A70 APPROXIMATELY EQUAL OR EQUAL TO
2A71 EQUALS SIGN ABOVE PLUS SIGN
2A72 PLUS SIGN ABOVE EQUALS SIGN
2A73 EQUALS SIGN ABOVE TILDE OPERATOR
2A74 DOUBLE COLON EQUAL
2A75 TWO CONSECUTIVE EQUALS SIGNS
2A76 THREE CONSECUTIVE EQUALS SIGNS
2A77 EQUALS SIGN WITH TWO DOTS ABOVE AND TWO DOTS BELOW
2A78 EQUIVALENT WITH FOUR DOTS ABOVE
2A79 LESS-THAN WITH CIRCLE INSIDE
2A7A GREATER-THAN WITH CIRCLE INSIDE
2A7B LESS-THAN WITH QUESTION MARK ABOVE
2A7C GREATER-THAN WITH QUESTION MARK ABOVE
2A7D LESS-THAN OR SLANTED EQUAL TO
2A7E GREATER-THAN OR SLANTED EQUAL TO
2A7F LESS-THAN OR SLANTED EQUAL TO WITH DOT INSIDE
2A80 GREATER-THAN OR SLANTED EQUAL TO WITH DOT INSIDE
2A81 LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE
2A82 GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE
2A83 LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE RIGHT
2A84 GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE LEFT
2A85 LESS-THAN OR APPROXIMATE
2A86 GREATER-THAN OR APPROXIMATE
2A87 LESS-THAN AND SINGLE-LINE NOT EQUAL TO
2A88 GREATER-THAN AND SINGLE-LINE NOT EQUAL TO
2A89 LESS-THAN AND NOT APPROXIMATE
2A8A GREATER-THAN AND NOT APPROXIMATE
2A8B LESS-THAN ABOVE DOUBLE-LINE EQUAL ABOVE GREATER-THAN
2A8C GREATER-THAN ABOVE DOUBLE-LINE EQUAL ABOVE LESS-THAN
2A8D LESS-THAN ABOVE SIMILAR OR EQUAL
2A8E GREATER-THAN ABOVE SIMILAR OR EQUAL
2A8F LESS-THAN ABOVE SIMILAR ABOVE GREATER-THAN
2A90 GREATER-THAN ABOVE SIMILAR ABOVE LESS-THAN
2A91 LESS-THAN ABOVE GREATER-THAN ABOVE DOUBLE-LINE EQUAL
2A92 GREATER-THAN ABOVE LESS-THAN ABOVE DOUBLE-LINE EQUAL
2A93 LESS-THAN ABOVE SLANTED EQUAL ABOVE GREATER-THAN ABOVE SLANTED EQUAL
2A94 GREATER-THAN ABOVE SLANTED EQUAL ABOVE LESS-THAN ABOVE SLANTED EQUAL
2A95 SLANTED EQUAL TO OR LESS-THAN
2A96 SLANTED EQUAL TO OR GREATER-THAN
2A97 SLANTED EQUAL TO OR LESS-THAN WITH DOT INSIDE
2A98 SLANTED EQUAL TO OR GREATER-THAN WITH DOT INSIDE
2A99 DOUBLE-LINE EQUAL TO OR LESS-THAN
2A9A DOUBLE-LINE EQUAL TO OR GREATER-THAN
2A9B DOUBLE-LINE SLANTED EQUAL TO OR LESS-THAN
2A9C DOUBLE-LINE SLANTED EQUAL TO OR GREATER-THAN
2A9D SIMILAR OR LESS-THAN
2A9E SIMILAR OR GREATER-THAN
2A9F SIMILAR ABOVE LESS-THAN ABOVE EQUALS SIGN
2AA0 SIMILAR ABOVE GREATER-THAN ABOVE EQUALS SIGN
2AA1 DOUBLE NESTED LESS-THAN
2AA2 DOUBLE NESTED GREATER-THAN
2AA3 DOUBLE NESTED LESS-THAN WITH UNDERBAR
2AA4 GREATER-THAN OVERLAPPING LESS-THAN
2AA5 GREATER-THAN BESIDE LESS-THAN
2AA6 LESS-THAN CLOSED BY CURVE
2AA7 GREATER-THAN CLOSED BY CURVE
2AA8 LESS-THAN CLOSED BY CURVE ABOVE SLANTED EQUAL
2AA9 GREATER-THAN CLOSED BY CURVE ABOVE SLANTED EQUAL
2AAA SMALLER THAN
2AAB LARGER THAN
2AAC SMALLER THAN OR EQUAL TO
2AAD LARGER THAN OR EQUAL TO
2AAE EQUALS SIGN WITH BUMPY ABOVE
2AAF PRECEDES ABOVE SINGLE-LINE EQUALS SIGN
2AB0 SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN
2AB1 PRECEDES ABOVE SINGLE-LINE NOT EQUAL TO
2AB2 SUCCEEDS ABOVE SINGLE-LINE NOT EQUAL TO
2AB3 PRECEDES ABOVE EQUALS SIGN
2AB4 SUCCEEDS ABOVE EQUALS SIGN
2AB5 PRECEDES ABOVE NOT EQUAL TO
2AB6 SUCCEEDS ABOVE NOT EQUAL TO
2AB7 PRECEDES ABOVE ALMOST EQUAL TO
2AB8 SUCCEEDS ABOVE ALMOST EQUAL TO
2AB9 PRECEDES ABOVE NOT ALMOST EQUAL TO
2ABA SUCCEEDS ABOVE NOT ALMOST EQUAL TO
2ABB DOUBLE PRECEDES
2ABC DOUBLE SUCCEEDS
2ABD SUBSET WITH DOT
2ABE SUPERSET WITH DOT
2ABF SUBSET WITH PLUS SIGN BELOW
2AC0 SUPERSET WITH PLUS SIGN BELOW
2AC1 SUBSET WITH MULTIPLICATION SIGN BELOW
2AC2 SUPERSET WITH MULTIPLICATION SIGN BELOW
2AC3 SUBSET OF OR EQUAL TO WITH DOT ABOVE
2AC4 SUPERSET OF OR EQUAL TO WITH DOT ABOVE
2AC5 SUBSET OF ABOVE EQUALS SIGN
2AC6 SUPERSET OF ABOVE EQUALS SIGN
2AC7 SUBSET OF ABOVE TILDE OPERATOR
2AC8 SUPERSET OF ABOVE TILDE OPERATOR
2AC9 SUBSET OF ABOVE ALMOST EQUAL TO
2ACA SUPERSET OF ABOVE ALMOST EQUAL TO
2ACB SUBSET OF ABOVE NOT EQUAL TO
2ACC SUPERSET OF ABOVE NOT EQUAL TO
2ACD SQUARE LEFT OPEN BOX OPERATOR
2ACE SQUARE RIGHT OPEN BOX OPERATOR
2ACF CLOSED SUBSET
2AD0 CLOSED SUPERSET
2AD1 CLOSED SUBSET OR EQUAL TO
2AD2 CLOSED SUPERSET OR EQUAL TO
2AD3 SUBSET ABOVE SUPERSET
2AD4 SUPERSET ABOVE SUBSET
2AD5 SUBSET ABOVE SUBSET
2AD6 SUPERSET ABOVE SUPERSET
2AD7 SUPERSET BESIDE SUBSET
2AD8 SUPERSET BESIDE AND JOINED BY DASH WITH SUBSET
2AD9 ELEMENT OF OPENING DOWNWARDS
2ADA PITCHFORK WITH TEE TOP
2ADB TRANSVERSAL INTERSECTION
2ADC FORKING
2ADD NONFORKING
2ADE SHORT LEFT TACK
2ADF SHORT DOWN TACK
2AE0 SHORT UP TACK
2AE1 PERPENDICULAR WITH S
2AE2 VERTICAL BAR TRIPLE RIGHT TURNSTILE
2AE3 DOUBLE VERTICAL BAR LEFT TURNSTILE
2AE4 VERTICAL BAR DOUBLE LEFT TURNSTILE
2AE5 DOUBLE VERTICAL BAR DOUBLE LEFT TURNSTILE
2AE6 LONG DASH FROM LEFT MEMBER OF DOUBLE VERTICAL
2AE7 SHORT DOWN TACK WITH OVERBAR
2AE8 SHORT UP TACK WITH UNDERBAR
2AE9 SHORT UP TACK ABOVE SHORT DOWN TACK
2AEA DOUBLE DOWN TACK
2AEB DOUBLE UP TACK
2AEC DOUBLE STROKE NOT SIGN
2AED REVERSED DOUBLE STROKE NOT SIGN
2AEE DOES NOT DIVIDE WITH REVERSED NEGATION SLASH
2AEF VERTICAL LINE WITH CIRCLE ABOVE
2AF0 VERTICAL LINE WITH CIRCLE BELOW
2AF1 DOWN TACK WITH CIRCLE BELOW
2AF2 PARALLEL WITH HORIZONTAL STROKE
2AF3 PARALLEL WITH TILDE OPERATOR
2AF4 TRIPLE VERTICAL BAR BINARY RELATION
2AF5 TRIPLE VERTICAL BAR WITH HORIZONTAL STROKE
2AF6 TRIPLE COLON OPERATOR
2AF7 TRIPLE NESTED LESS-THAN
2AF8 TRIPLE NESTED GREATER-THAN
2AF9 DOUBLE-LINE SLANTED LESS-THAN OR EQUAL TO
2AFA DOUBLE-LINE SLANTED GREATER-THAN OR EQUAL TO
2AFB TRIPLE SOLIDUS BINARY RELATION
2AFC LARGE TRIPLE VERTICAL BAR OPERATOR
2AFD DOUBLE SOLIDUS OPERATOR
2AFE WHITE VERTICAL BAR
2AFF N-ARY WHITE VERTICAL BAR
2B00 NORTH EAST WHITE ARROW
2B01 NORTH WEST WHITE ARROW
2B02 SOUTH EAST WHITE ARROW
2B03 SOUTH WEST WHITE ARROW
2B04 LEFT RIGHT WHITE ARROW
2B05 LEFTWARDS BLACK ARROW
2B06 UPWARDS BLACK ARROW
2B07 DOWNWARDS BLACK ARROW
2B08 NORTH EAST BLACK ARROW
2B09 NORTH WEST BLACK ARROW
2B0A SOUTH EAST BLACK ARROW
2B0B SOUTH WEST BLACK ARROW
2B0C LEFT RIGHT BLACK ARROW
2B0D UP DOWN BLACK ARROW
2B0E RIGHTWARDS ARROW WITH TIP DOWNWARDS
2B0F RIGHTWARDS ARROW WITH TIP UPWARDS
2B10 LEFTWARDS ARROW WITH TIP DOWNWARDS
2B11 LEFTWARDS ARROW WITH TIP UPWARDS
2B12 SQUARE WITH TOP HALF BLACK
2B13 SQUARE WITH BOTTOM HALF BLACK
2B14 SQUARE WITH UPPER RIGHT DIAGONAL HALF BLACK
2B15 SQUARE WITH LOWER LEFT DIAGONAL HALF BLACK
2B16 DIAMOND WITH LEFT HALF BLACK
2B17 DIAMOND WITH RIGHT HALF BLACK
2B18 DIAMOND WITH TOP HALF BLACK
2B19 DIAMOND WITH BOTTOM HALF BLACK
2B1A DOTTED SQUARE
2B1B BLACK LARGE SQUARE
2B1C WHITE LARGE SQUARE
2B1D BLACK VERY SMALL SQUARE
2B1E WHITE VERY SMALL SQUARE
2B1F BLACK PENTAGON
2B20 WHITE PENTAGON
2B21 WHITE HEXAGON
2B22 BLACK HEXAGON
2B23 HORIZONTAL BLACK HEXAGON
2B24 BLACK LARGE CIRCLE
2B25 BLACK MEDIUM DIAMOND
2B26 WHITE MEDIUM DIAMOND
2B27 BLACK MEDIUM LOZENGE
2B28 WHITE MEDIUM LOZENGE
2B29 BLACK SMALL DIAMOND
2B2A BLACK SMALL LOZENGE
2B2B WHITE SMALL LOZENGE
2B2C BLACK HORIZONTAL ELLIPSE
2B2D WHITE HORIZONTAL ELLIPSE
2B2E BLACK VERTICAL ELLIPSE
2B2F WHITE VERTICAL ELLIPSE
2B30 LEFT ARROW WITH SMALL CIRCLE
2B31 THREE LEFTWARDS ARROWS
2B32 LEFT ARROW WITH CIRCLED PLUS
2B33 LONG LEFTWARDS SQUIGGLE ARROW
2B34 LEFTWARDS TWO-HEADED ARROW WITH VERTICAL STROKE
2B35 LEFTWARDS TWO-HEADED ARROW WITH DOUBLE VERTICAL STROKE
2B36 LEFTWARDS TWO-HEADED ARROW FROM BAR
2B37 LEFTWARDS TWO-HEADED TRIPLE DASH ARROW
2B38 LEFTWARDS ARROW WITH DOTTED STEM
2B39 LEFTWARDS ARROW WITH TAIL WITH VERTICAL STROKE
2B3A LEFTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE
2B3B LEFTWARDS TWO-HEADED ARROW WITH TAIL
2B3C LEFTWARDS TWO-HEADED ARROW WITH TAIL WITH VERTICAL STROKE
2B3D LEFTWARDS TWO-HEADED ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE
2B3E LEFTWARDS ARROW THROUGH X
2B3F WAVE ARROW POINTING DIRECTLY LEFT
2B40 EQUALS SIGN ABOVE LEFTWARDS ARROW
2B41 REVERSE TILDE OPERATOR ABOVE LEFTWARDS ARROW
2B42 LEFTWARDS ARROW ABOVE REVERSE ALMOST EQUAL TO
2B43 RIGHTWARDS ARROW THROUGH GREATER-THAN
2B44 RIGHTWARDS ARROW THROUGH SUPERSET
2B45 LEFTWARDS QUADRUPLE ARROW
2B46 RIGHTWARDS QUADRUPLE ARROW
2B47 REVERSE TILDE OPERATOR ABOVE RIGHTWARDS ARROW
2B48 RIGHTWARDS ARROW ABOVE REVERSE ALMOST EQUAL TO
2B49 TILDE OPERATOR ABOVE LEFTWARDS ARROW
2B4A LEFTWARDS ARROW ABOVE ALMOST EQUAL TO
2B4B LEFTWARDS ARROW ABOVE REVERSE TILDE OPERATOR
2B4C RIGHTWARDS ARROW ABOVE REVERSE TILDE OPERATOR
2B4D DOWNWARDS TRIANGLE-HEADED ZIGZAG ARROW
2B4E SHORT SLANTED NORTH ARROW
2B4F SHORT BACKSLANTED SOUTH ARROW
2B50 WHITE MEDIUM STAR
2B51 BLACK SMALL STAR
2B52 WHITE SMALL STAR
2B53 BLACK RIGHT-POINTING PENTAGON
2B54 WHITE RIGHT-POINTING PENTAGON
2B55 HEAVY LARGE CIRCLE
2B56 HEAVY OVAL WITH OVAL INSIDE
2B57 HEAVY CIRCLE WITH CIRCLE INSIDE
2B58 HEAVY CIRCLE
2B59 HEAVY CIRCLED SALTIRE
2B5A SLANTED NORTH ARROW WITH HOOKED HEAD
2B5B BACKSLANTED SOUTH ARROW WITH HOOKED TAIL
2B5C SLANTED NORTH ARROW WITH HORIZONTAL TAIL
2B5D BACKSLANTED SOUTH ARROW WITH HORIZONTAL TAIL
2B5E BENT ARROW POINTING DOWNWARDS THEN NORTH EAST
2B5F SHORT BENT ARROW POINTING DOWNWARDS THEN NORTH EAST
2B60 LEFTWARDS TRIANGLE-HEADED ARROW
2B61 UPWARDS TRIANGLE-HEADED ARROW
2B62 RIGHTWARDS TRIANGLE-HEADED ARROW
2B63 DOWNWARDS TRIANGLE-HEADED ARROW
2B64 LEFT RIGHT TRIANGLE-HEADED ARROW
2B65 UP DOWN TRIANGLE-HEADED ARROW
2B66 NORTH WEST TRIANGLE-HEADED ARROW
2B67 NORTH EAST TRIANGLE-HEADED ARROW
2B68 SOUTH EAST TRIANGLE-HEADED ARROW
2B69 SOUTH WEST TRIANGLE-HEADED ARROW
2B6A LEFTWARDS TRIANGLE-HEADED DASHED ARROW
2B6B UPWARDS TRIANGLE-HEADED DASHED ARROW
2B6C RIGHTWARDS TRIANGLE-HEADED DASHED ARROW
2B6D DOWNWARDS TRIANGLE-HEADED DASHED ARROW
2B6E CLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROW
2B6F ANTICLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROW
2B70 LEFTWARDS TRIANGLE-HEADED ARROW TO BAR
2B71 UPWARDS TRIANGLE-HEADED ARROW TO BAR
2B72 RIGHTWARDS TRIANGLE-HEADED ARROW TO BAR
2B73 DOWNWARDS TRIANGLE-HEADED ARROW TO BAR
2B76 NORTH WEST TRIANGLE-HEADED ARROW TO BAR
2B77 NORTH EAST TRIANGLE-HEADED ARROW TO BAR
2B78 SOUTH EAST TRIANGLE-HEADED ARROW TO BAR
2B79 SOUTH WEST TRIANGLE-HEADED ARROW TO BAR
2B7A LEFTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE
2B7B UPWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE
2B7C RIGHTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE
2B7D DOWNWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE
2B7E HORIZONTAL TAB KEY
2B7F VERTICAL TAB KEY
2B80 LEFTWARDS TRIANGLE-HEADED ARROW OVER RIGHTWARDS TRIANGLE-HEADED ARROW
2B81 UPWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF DOWNWARDS TRIANGLE-HEADED ARROW
2B82 RIGHTWARDS TRIANGLE-HEADED ARROW OVER LEFTWARDS TRIANGLE-HEADED ARROW
2B83 DOWNWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF UPWARDS TRIANGLE-HEADED ARROW
2B84 LEFTWARDS TRIANGLE-HEADED PAIRED ARROWS
2B85 UPWARDS TRIANGLE-HEADED PAIRED ARROWS
2B86 RIGHTWARDS TRIANGLE-HEADED PAIRED ARROWS
2B87 DOWNWARDS TRIANGLE-HEADED PAIRED ARROWS
2B88 LEFTWARDS BLACK CIRCLED WHITE ARROW
2B89 UPWARDS BLACK CIRCLED WHITE ARROW
2B8A RIGHTWARDS BLACK CIRCLED WHITE ARROW
2B8B DOWNWARDS BLACK CIRCLED WHITE ARROW
2B8C ANTICLOCKWISE TRIANGLE-HEADED RIGHT U-SHAPED ARROW
2B8D ANTICLOCKWISE TRIANGLE-HEADED BOTTOM U-SHAPED ARROW
2B8E ANTICLOCKWISE TRIANGLE-HEADED LEFT U-SHAPED ARROW
2B8F ANTICLOCKWISE TRIANGLE-HEADED TOP U-SHAPED ARROW
2B90 RETURN LEFT
2B91 RETURN RIGHT
2B92 NEWLINE LEFT
2B93 NEWLINE RIGHT
2B94 FOUR CORNER ARROWS CIRCLING ANTICLOCKWISE
2B95 RIGHTWARDS BLACK ARROW
2B97 SYMBOL FOR TYPE A ELECTRONICS
2B98 THREE-D TOP-LIGHTED LEFTWARDS EQUILATERAL ARROWHEAD
2B99 THREE-D RIGHT-LIGHTED UPWARDS EQUILATERAL ARROWHEAD
2B9A THREE-D TOP-LIGHTED RIGHTWARDS EQUILATERAL ARROWHEAD
2B9B THREE-D LEFT-LIGHTED DOWNWARDS EQUILATERAL ARROWHEAD
2B9C BLACK LEFTWARDS EQUILATERAL ARROWHEAD
2B9D BLACK UPWARDS EQUILATERAL ARROWHEAD
2B9E BLACK RIGHTWARDS EQUILATERAL ARROWHEAD
2B9F BLACK DOWNWARDS EQUILATERAL ARROWHEAD
2BA0 DOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDS
2BA1 DOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP RIGHTWARDS
2BA2 UPWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDS
2BA3 UPWARDS TRIANGLE-HEADED ARROW WITH LONG TIP RIGHTWARDS
2BA4 LEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDS
2BA5 RIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDS
2BA6 LEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDS
2BA7 RIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDS
2BA8 BLACK CURVED DOWNWARDS AND LEFTWARDS ARROW
2BA9 BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROW
2BAA BLACK CURVED UPWARDS AND LEFTWARDS ARROW
2BAB BLACK CURVED UPWARDS AND RIGHTWARDS ARROW
2BAC BLACK CURVED LEFTWARDS AND UPWARDS ARROW
2BAD BLACK CURVED RIGHTWARDS AND UPWARDS ARROW
2BAE BLACK CURVED LEFTWARDS AND DOWNWARDS ARROW
2BAF BLACK CURVED RIGHTWARDS AND DOWNWARDS ARROW
2BB0 RIBBON ARROW DOWN LEFT
2BB1 RIBBON ARROW DOWN RIGHT
2BB2 RIBBON ARROW UP LEFT
2BB3 RIBBON ARROW UP RIGHT
2BB4 RIBBON ARROW LEFT UP
2BB5 RIBBON ARROW RIGHT UP
2BB6 RIBBON ARROW LEFT DOWN
2BB7 RIBBON ARROW RIGHT DOWN
2BB8 UPWARDS WHITE ARROW FROM BAR WITH HORIZONTAL BAR
2BB9 UP ARROWHEAD IN A RECTANGLE BOX
2BBA OVERLAPPING WHITE SQUARES
2BBB OVERLAPPING WHITE AND BLACK SQUARES
2BBC OVERLAPPING BLACK SQUARES
2BBD BALLOT BOX WITH LIGHT X
2BBE CIRCLED X
2BBF CIRCLED BOLD X
2BC0 BLACK SQUARE CENTRED
2BC1 BLACK DIAMOND CENTRED
2BC2 TURNED BLACK PENTAGON
2BC3 HORIZONTAL BLACK OCTAGON
2BC4 BLACK OCTAGON
2BC5 BLACK MEDIUM UP-POINTING TRIANGLE CENTRED
2BC6 BLACK MEDIUM DOWN-POINTING TRIANGLE CENTRED
2BC7 BLACK MEDIUM LEFT-POINTING TRIANGLE CENTRED
2BC8 BLACK MEDIUM RIGHT-POINTING TRIANGLE CENTRED
2BC9 NEPTUNE FORM TWO
2BCA TOP HALF BLACK CIRCLE
2BCB BOTTOM HALF BLACK CIRCLE
2BCC LIGHT FOUR POINTED BLACK CUSP
2BCD ROTATED LIGHT FOUR POINTED BLACK CUSP
2BCE WHITE FOUR POINTED CUSP
2BCF ROTATED WHITE FOUR POINTED CUSP
2BD0 SQUARE POSITION INDICATOR
2BD1 UNCERTAINTY SIGN
2BD2 GROUP MARK
2BD3 PLUTO FORM TWO
2BD4 PLUTO FORM THREE
2BD5 PLUTO FORM FOUR
2BD6 PLUTO FORM FIVE
2BD7 TRANSPLUTO
2BD8 PROSERPINA
2BD9 ASTRAEA
2BDA HYGIEA
2BDB PHOLUS
2BDC NESSUS
2BDD WHITE MOON SELENA
2BDE BLACK DIAMOND ON CROSS
2BDF TRUE LIGHT MOON ARTA
2BE0 CUPIDO
2BE1 HADES
2BE2 ZEUS
2BE3 KRONOS
2BE4 APOLLON
2BE5 ADMETOS
2BE6 VULCANUS
2BE7 POSEIDON
2BE8 LEFT HALF BLACK STAR
2BE9 RIGHT HALF BLACK STAR
2BEA STAR WITH LEFT HALF BLACK
2BEB STAR WITH RIGHT HALF BLACK
2BEC LEFTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS
2BED UPWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS
2BEE RIGHTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS
2BEF DOWNWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS
2BF0 ERIS FORM ONE
2BF1 ERIS FORM TWO
2BF2 SEDNA
2BF3 RUSSIAN ASTROLOGICAL SYMBOL VIGINTILE
2BF4 RUSSIAN ASTROLOGICAL SYMBOL NOVILE
2BF5 RUSSIAN ASTROLOGICAL SYMBOL QUINTILE
2BF6 RUSSIAN ASTROLOGICAL SYMBOL BINOVILE
2BF7 RUSSIAN ASTROLOGICAL SYMBOL SENTAGON
2BF8 RUSSIAN ASTROLOGICAL SYMBOL TREDECILE
2BF9 EQUALS SIGN WITH INFINITY BELOW
2BFA UNITED SYMBOL
2BFB SEPARATED SYMBOL
2BFC DOUBLED SYMBOL
2BFD PASSED SYMBOL
2BFE REVERSED RIGHT ANGLE
2BFF HELLSCHREIBER PAUSE SYMBOL
2C00 GLAGOLITIC CAPITAL LETTER AZU
2C01 GLAGOLITIC CAPITAL LETTER BUKY
2C02 GLAGOLITIC CAPITAL LETTER VEDE
2C03 GLAGOLITIC CAPITAL LETTER GLAGOLI
2C04 GLAGOLITIC CAPITAL LETTER DOBRO
2C05 GLAGOLITIC CAPITAL LETTER YESTU
2C06 GLAGOLITIC CAPITAL LETTER ZHIVETE
2C07 GLAGOLITIC CAPITAL LETTER DZELO
2C08 GLAGOLITIC CAPITAL LETTER ZEMLJA
2C09 GLAGOLITIC CAPITAL LETTER IZHE
2C0A GLAGOLITIC CAPITAL LETTER INITIAL IZHE
2C0B GLAGOLITIC CAPITAL LETTER I
2C0C GLAGOLITIC CAPITAL LETTER DJERVI
2C0D GLAGOLITIC CAPITAL LETTER KAKO
2C0E GLAGOLITIC CAPITAL LETTER LJUDIJE
2C0F GLAGOLITIC CAPITAL LETTER MYSLITE
2C10 GLAGOLITIC CAPITAL LETTER NASHI
2C11 GLAGOLITIC CAPITAL LETTER ONU
2C12 GLAGOLITIC CAPITAL LETTER POKOJI
2C13 GLAGOLITIC CAPITAL LETTER RITSI
2C14 GLAGOLITIC CAPITAL LETTER SLOVO
2C15 GLAGOLITIC CAPITAL LETTER TVRIDO
2C16 GLAGOLITIC CAPITAL LETTER UKU
2C17 GLAGOLITIC CAPITAL LETTER FRITU
2C18 GLAGOLITIC CAPITAL LETTER HERU
2C19 GLAGOLITIC CAPITAL LETTER OTU
2C1A GLAGOLITIC CAPITAL LETTER PE
2C1B GLAGOLITIC CAPITAL LETTER SHTA
2C1C GLAGOLITIC CAPITAL LETTER TSI
2C1D GLAGOLITIC CAPITAL LETTER CHRIVI
2C1E GLAGOLITIC CAPITAL LETTER SHA
2C1F GLAGOLITIC CAPITAL LETTER YERU
2C20 GLAGOLITIC CAPITAL LETTER YERI
2C21 GLAGOLITIC CAPITAL LETTER YATI
2C22 GLAGOLITIC CAPITAL LETTER SPIDERY HA
2C23 GLAGOLITIC CAPITAL LETTER YU
2C24 GLAGOLITIC CAPITAL LETTER SMALL YUS
2C25 GLAGOLITIC CAPITAL LETTER SMALL YUS WITH TAIL
2C26 GLAGOLITIC CAPITAL LETTER YO
2C27 GLAGOLITIC CAPITAL LETTER IOTATED SMALL YUS
2C28 GLAGOLITIC CAPITAL LETTER BIG YUS
2C29 GLAGOLITIC CAPITAL LETTER IOTATED BIG YUS
2C2A GLAGOLITIC CAPITAL LETTER FITA
2C2B GLAGOLITIC CAPITAL LETTER IZHITSA
2C2C GLAGOLITIC CAPITAL LETTER SHTAPIC
2C2D GLAGOLITIC CAPITAL LETTER TROKUTASTI A
2C2E GLAGOLITIC CAPITAL LETTER LATINATE MYSLITE
2C2F GLAGOLITIC CAPITAL LETTER CAUDATE CHRIVI
2C30 GLAGOLITIC SMALL LETTER AZU
2C31 GLAGOLITIC SMALL LETTER BUKY
2C32 GLAGOLITIC SMALL LETTER VEDE
2C33 GLAGOLITIC SMALL LETTER GLAGOLI
2C34 GLAGOLITIC SMALL LETTER DOBRO
2C35 GLAGOLITIC SMALL LETTER YESTU
2C36 GLAGOLITIC SMALL LETTER ZHIVETE
2C37 GLAGOLITIC SMALL LETTER DZELO
2C38 GLAGOLITIC SMALL LETTER ZEMLJA
2C39 GLAGOLITIC SMALL LETTER IZHE
2C3A GLAGOLITIC SMALL LETTER INITIAL IZHE
2C3B GLAGOLITIC SMALL LETTER I
2C3C GLAGOLITIC SMALL LETTER DJERVI
2C3D GLAGOLITIC SMALL LETTER KAKO
2C3E GLAGOLITIC SMALL LETTER LJUDIJE
2C3F GLAGOLITIC SMALL LETTER MYSLITE
2C40 GLAGOLITIC SMALL LETTER NASHI
2C41 GLAGOLITIC SMALL LETTER ONU
2C42 GLAGOLITIC SMALL LETTER POKOJI
2C43 GLAGOLITIC SMALL LETTER RITSI
2C44 GLAGOLITIC SMALL LETTER SLOVO
2C45 GLAGOLITIC SMALL LETTER TVRIDO
2C46 GLAGOLITIC SMALL LETTER UKU
2C47 GLAGOLITIC SMALL LETTER FRITU
2C48 GLAGOLITIC SMALL LETTER HERU
2C49 GLAGOLITIC SMALL LETTER OTU
2C4A GLAGOLITIC SMALL LETTER PE
2C4B GLAGOLITIC SMALL LETTER SHTA
2C4C GLAGOLITIC SMALL LETTER TSI
2C4D GLAGOLITIC SMALL LETTER CHRIVI
2C4E GLAGOLITIC SMALL LETTER SHA
2C4F GLAGOLITIC SMALL LETTER YERU
2C50 GLAGOLITIC SMALL LETTER YERI
2C51 GLAGOLITIC SMALL LETTER YATI
2C52 GLAGOLITIC SMALL LETTER SPIDERY HA
2C53 GLAGOLITIC SMALL LETTER YU
2C54 GLAGOLITIC SMALL LETTER SMALL YUS
2C55 GLAGOLITIC SMALL LETTER SMALL YUS WITH TAIL
2C56 GLAGOLITIC SMALL LETTER YO
2C57 GLAGOLITIC SMALL LETTER IOTATED SMALL YUS
2C58 GLAGOLITIC SMALL LETTER BIG YUS
2C59 GLAGOLITIC SMALL LETTER IOTATED BIG YUS
2C5A GLAGOLITIC SMALL LETTER FITA
2C5B GLAGOLITIC SMALL LETTER IZHITSA
2C5C GLAGOLITIC SMALL LETTER SHTAPIC
2C5D GLAGOLITIC SMALL LETTER TROKUTASTI A
2C5E GLAGOLITIC SMALL LETTER LATINATE MYSLITE
2C5F GLAGOLITIC SMALL LETTER CAUDATE CHRIVI
2C60 LATIN CAPITAL LETTER L WITH DOUBLE BAR
2C61 LATIN SMALL LETTER L WITH DOUBLE BAR
2C62 LATIN CAPITAL LETTER L WITH MIDDLE TILDE
2C63 LATIN CAPITAL LETTER P WITH STROKE
2C64 LATIN CAPITAL LETTER R WITH TAIL
2C65 LATIN SMALL LETTER A WITH STROKE
2C66 LATIN SMALL LETTER T WITH DIAGONAL STROKE
2C67 LATIN CAPITAL LETTER H WITH DESCENDER
2C68 LATIN SMALL LETTER H WITH DESCENDER
2C69 LATIN CAPITAL LETTER K WITH DESCENDER
2C6A LATIN SMALL LETTER K WITH DESCENDER
2C6B LATIN CAPITAL LETTER Z WITH DESCENDER
2C6C LATIN SMALL LETTER Z WITH DESCENDER
2C6D LATIN CAPITAL LETTER ALPHA
2C6E LATIN CAPITAL LETTER M WITH HOOK
2C6F LATIN CAPITAL LETTER TURNED A
2C70 LATIN CAPITAL LETTER TURNED ALPHA
2C71 LATIN SMALL LETTER V WITH RIGHT HOOK
2C72 LATIN CAPITAL LETTER W WITH HOOK
2C73 LATIN SMALL LETTER W WITH HOOK
2C74 LATIN SMALL LETTER V WITH CURL
2C75 LATIN CAPITAL LETTER HALF H
2C76 LATIN SMALL LETTER HALF H
2C77 LATIN SMALL LETTER TAILLESS PHI
2C78 LATIN SMALL LETTER E WITH NOTCH
2C79 LATIN SMALL LETTER TURNED R WITH TAIL
2C7A LATIN SMALL LETTER O WITH LOW RING INSIDE
2C7B LATIN LETTER SMALL CAPITAL TURNED E
2C7C LATIN SUBSCRIPT SMALL LETTER J
2C7D MODIFIER LETTER CAPITAL V
2C7E LATIN CAPITAL LETTER S WITH SWASH TAIL
2C7F LATIN CAPITAL LETTER Z WITH SWASH TAIL
2C80 COPTIC CAPITAL LETTER ALFA
2C81 COPTIC SMALL LETTER ALFA
2C82 COPTIC CAPITAL LETTER VIDA
2C83 COPTIC SMALL LETTER VIDA
2C84 COPTIC CAPITAL LETTER GAMMA
2C85 COPTIC SMALL LETTER GAMMA
2C86 COPTIC CAPITAL LETTER DALDA
2C87 COPTIC SMALL LETTER DALDA
2C88 COPTIC CAPITAL LETTER EIE
2C89 COPTIC SMALL LETTER EIE
2C8A COPTIC CAPITAL LETTER SOU
2C8B COPTIC SMALL LETTER SOU
2C8C COPTIC CAPITAL LETTER ZATA
2C8D COPTIC SMALL LETTER ZATA
2C8E COPTIC CAPITAL LETTER HATE
2C8F COPTIC SMALL LETTER HATE
2C90 COPTIC CAPITAL LETTER THETHE
2C91 COPTIC SMALL LETTER THETHE
2C92 COPTIC CAPITAL LETTER IAUDA
2C93 COPTIC SMALL LETTER IAUDA
2C94 COPTIC CAPITAL LETTER KAPA
2C95 COPTIC SMALL LETTER KAPA
2C96 COPTIC CAPITAL LETTER LAULA
2C97 COPTIC SMALL LETTER LAULA
2C98 COPTIC CAPITAL LETTER MI
2C99 COPTIC SMALL LETTER MI
2C9A COPTIC CAPITAL LETTER NI
2C9B COPTIC SMALL LETTER NI
2C9C COPTIC CAPITAL LETTER KSI
2C9D COPTIC SMALL LETTER KSI
2C9E COPTIC CAPITAL LETTER O
2C9F COPTIC SMALL LETTER O
2CA0 COPTIC CAPITAL LETTER PI
2CA1 COPTIC SMALL LETTER PI
2CA2 COPTIC CAPITAL LETTER RO
2CA3 COPTIC SMALL LETTER RO
2CA4 COPTIC CAPITAL LETTER SIMA
2CA5 COPTIC SMALL LETTER SIMA
2CA6 COPTIC CAPITAL LETTER TAU
2CA7 COPTIC SMALL LETTER TAU
2CA8 COPTIC CAPITAL LETTER UA
2CA9 COPTIC SMALL LETTER UA
2CAA COPTIC CAPITAL LETTER FI
2CAB COPTIC SMALL LETTER FI
2CAC COPTIC CAPITAL LETTER KHI
2CAD COPTIC SMALL LETTER KHI
2CAE COPTIC CAPITAL LETTER PSI
2CAF COPTIC SMALL LETTER PSI
2CB0 COPTIC CAPITAL LETTER OOU
2CB1 COPTIC SMALL LETTER OOU
2CB2 COPTIC CAPITAL LETTER DIALECT-P ALEF
2CB3 COPTIC SMALL LETTER DIALECT-P ALEF
2CB4 COPTIC CAPITAL LETTER OLD COPTIC AIN
2CB5 COPTIC SMALL LETTER OLD COPTIC AIN
2CB6 COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
2CB7 COPTIC SMALL LETTER CRYPTOGRAMMIC EIE
2CB8 COPTIC CAPITAL LETTER DIALECT-P KAPA
2CB9 COPTIC SMALL LETTER DIALECT-P KAPA
2CBA COPTIC CAPITAL LETTER DIALECT-P NI
2CBB COPTIC SMALL LETTER DIALECT-P NI
2CBC COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
2CBD COPTIC SMALL LETTER CRYPTOGRAMMIC NI
2CBE COPTIC CAPITAL LETTER OLD COPTIC OOU
2CBF COPTIC SMALL LETTER OLD COPTIC OOU
2CC0 COPTIC CAPITAL LETTER SAMPI
2CC1 COPTIC SMALL LETTER SAMPI
2CC2 COPTIC CAPITAL LETTER CROSSED SHEI
2CC3 COPTIC SMALL LETTER CROSSED SHEI
2CC4 COPTIC CAPITAL LETTER OLD COPTIC SHEI
2CC5 COPTIC SMALL LETTER OLD COPTIC SHEI
2CC6 COPTIC CAPITAL LETTER OLD COPTIC ESH
2CC7 COPTIC SMALL LETTER OLD COPTIC ESH
2CC8 COPTIC CAPITAL LETTER AKHMIMIC KHEI
2CC9 COPTIC SMALL LETTER AKHMIMIC KHEI
2CCA COPTIC CAPITAL LETTER DIALECT-P HORI
2CCB COPTIC SMALL LETTER DIALECT-P HORI
2CCC COPTIC CAPITAL LETTER OLD COPTIC HORI
2CCD COPTIC SMALL LETTER OLD COPTIC HORI
2CCE COPTIC CAPITAL LETTER OLD COPTIC HA
2CCF COPTIC SMALL LETTER OLD COPTIC HA
2CD0 COPTIC CAPITAL LETTER L-SHAPED HA
2CD1 COPTIC SMALL LETTER L-SHAPED HA
2CD2 COPTIC CAPITAL LETTER OLD COPTIC HEI
2CD3 COPTIC SMALL LETTER OLD COPTIC HEI
2CD4 COPTIC CAPITAL LETTER OLD COPTIC HAT
2CD5 COPTIC SMALL LETTER OLD COPTIC HAT
2CD6 COPTIC CAPITAL LETTER OLD COPTIC GANGIA
2CD7 COPTIC SMALL LETTER OLD COPTIC GANGIA
2CD8 COPTIC CAPITAL LETTER OLD COPTIC DJA
2CD9 COPTIC SMALL LETTER OLD COPTIC DJA
2CDA COPTIC CAPITAL LETTER OLD COPTIC SHIMA
2CDB COPTIC SMALL LETTER OLD COPTIC SHIMA
2CDC COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
2CDD COPTIC SMALL LETTER OLD NUBIAN SHIMA
2CDE COPTIC CAPITAL LETTER OLD NUBIAN NGI
2CDF COPTIC SMALL LETTER OLD NUBIAN NGI
2CE0 COPTIC CAPITAL LETTER OLD NUBIAN NYI
2CE1 COPTIC SMALL LETTER OLD NUBIAN NYI
2CE2 COPTIC CAPITAL LETTER OLD NUBIAN WAU
2CE3 COPTIC SMALL LETTER OLD NUBIAN WAU
2CE4 COPTIC SYMBOL KAI
2CE5 COPTIC SYMBOL MI RO
2CE6 COPTIC SYMBOL PI RO
2CE7 COPTIC SYMBOL STAUROS
2CE8 COPTIC SYMBOL TAU RO
2CE9 COPTIC SYMBOL KHI RO
2CEA COPTIC SYMBOL SHIMA SIMA
2CEB COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
2CEC COPTIC SMALL LETTER CRYPTOGRAMMIC SHEI
2CED COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
2CEE COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
2CEF COPTIC COMBINING NI ABOVE
2CF0 COPTIC COMBINING SPIRITUS ASPER
2CF1 COPTIC COMBINING SPIRITUS LENIS
2CF2 COPTIC CAPITAL LETTER BOHAIRIC KHEI
2CF3 COPTIC SMALL LETTER BOHAIRIC KHEI
2CF9 COPTIC OLD NUBIAN FULL STOP
2CFA COPTIC OLD NUBIAN DIRECT QUESTION MARK
2CFB COPTIC OLD NUBIAN INDIRECT QUESTION MARK
2CFC COPTIC OLD NUBIAN VERSE DIVIDER
2CFD COPTIC FRACTION ONE HALF
2CFE COPTIC FULL STOP
2CFF COPTIC MORPHOLOGICAL DIVIDER
2D00 GEORGIAN SMALL LETTER AN
2D01 GEORGIAN SMALL LETTER BAN
2D02 GEORGIAN SMALL LETTER GAN
2D03 GEORGIAN SMALL LETTER DON
2D04 GEORGIAN SMALL LETTER EN
2D05 GEORGIAN SMALL LETTER VIN
2D06 GEORGIAN SMALL LETTER ZEN
2D07 GEORGIAN SMALL LETTER TAN
2D08 GEORGIAN SMALL LETTER IN
2D09 GEORGIAN SMALL LETTER KAN
2D0A GEORGIAN SMALL LETTER LAS
2D0B GEORGIAN SMALL LETTER MAN
2D0C GEORGIAN SMALL LETTER NAR
2D0D GEORGIAN SMALL LETTER ON
2D0E GEORGIAN SMALL LETTER PAR
2D0F GEORGIAN SMALL LETTER ZHAR
2D10 GEORGIAN SMALL LETTER RAE
2D11 GEORGIAN SMALL LETTER SAN
2D12 GEORGIAN SMALL LETTER TAR
2D13 GEORGIAN SMALL LETTER UN
2D14 GEORGIAN SMALL LETTER PHAR
2D15 GEORGIAN SMALL LETTER KHAR
2D16 GEORGIAN SMALL LETTER GHAN
2D17 GEORGIAN SMALL LETTER QAR
2D18 GEORGIAN SMALL LETTER SHIN
2D19 GEORGIAN SMALL LETTER CHIN
2D1A GEORGIAN SMALL LETTER CAN
2D1B GEORGIAN SMALL LETTER JIL
2D1C GEORGIAN SMALL LETTER CIL
2D1D GEORGIAN SMALL LETTER CHAR
2D1E GEORGIAN SMALL LETTER XAN
2D1F GEORGIAN SMALL LETTER JHAN
2D20 GEORGIAN SMALL LETTER HAE
2D21 GEORGIAN SMALL LETTER HE
2D22 GEORGIAN SMALL LETTER HIE
2D23 GEORGIAN SMALL LETTER WE
2D24 GEORGIAN SMALL LETTER HAR
2D25 GEORGIAN SMALL LETTER HOE
2D27 GEORGIAN SMALL LETTER YN
2D2D GEORGIAN SMALL LETTER AEN
2D30 TIFINAGH LETTER YA
2D31 TIFINAGH LETTER YAB
2D32 TIFINAGH LETTER YABH
2D33 TIFINAGH LETTER YAG
2D34 TIFINAGH LETTER YAGHH
2D35 TIFINAGH LETTER BERBER ACADEMY YAJ
2D36 TIFINAGH LETTER YAJ
2D37 TIFINAGH LETTER YAD
2D38 TIFINAGH LETTER YADH
2D39 TIFINAGH LETTER YADD
2D3A TIFINAGH LETTER YADDH
2D3B TIFINAGH LETTER YEY
2D3C TIFINAGH LETTER YAF
2D3D TIFINAGH LETTER YAK
2D3E TIFINAGH LETTER TUAREG YAK
2D3F TIFINAGH LETTER YAKHH
2D40 TIFINAGH LETTER YAH
2D41 TIFINAGH LETTER BERBER ACADEMY YAH
2D42 TIFINAGH LETTER TUAREG YAH
2D43 TIFINAGH LETTER YAHH
2D44 TIFINAGH LETTER YAA
2D45 TIFINAGH LETTER YAKH
2D46 TIFINAGH LETTER TUAREG YAKH
2D47 TIFINAGH LETTER YAQ
2D48 TIFINAGH LETTER TUAREG YAQ
2D49 TIFINAGH LETTER YI
2D4A TIFINAGH LETTER YAZH
2D4B TIFINAGH LETTER AHAGGAR YAZH
2D4C TIFINAGH LETTER TUAREG YAZH
2D4D TIFINAGH LETTER YAL
2D4E TIFINAGH LETTER YAM
2D4F TIFINAGH LETTER YAN
2D50 TIFINAGH LETTER TUAREG YAGN
2D51 TIFINAGH LETTER TUAREG YANG
2D52 TIFINAGH LETTER YAP
2D53 TIFINAGH LETTER YU
2D54 TIFINAGH LETTER YAR
2D55 TIFINAGH LETTER YARR
2D56 TIFINAGH LETTER YAGH
2D57 TIFINAGH LETTER TUAREG YAGH
2D58 TIFINAGH LETTER AYER YAGH
2D59 TIFINAGH LETTER YAS
2D5A TIFINAGH LETTER YASS
2D5B TIFINAGH LETTER YASH
2D5C TIFINAGH LETTER YAT
2D5D TIFINAGH LETTER YATH
2D5E TIFINAGH LETTER YACH
2D5F TIFINAGH LETTER YATT
2D60 TIFINAGH LETTER YAV
2D61 TIFINAGH LETTER YAW
2D62 TIFINAGH LETTER YAY
2D63 TIFINAGH LETTER YAZ
2D64 TIFINAGH LETTER TAWELLEMET YAZ
2D65 TIFINAGH LETTER YAZZ
2D66 TIFINAGH LETTER YE
2D67 TIFINAGH LETTER YO
2D6F TIFINAGH MODIFIER LETTER LABIALIZATION MARK
2D70 TIFINAGH SEPARATOR MARK
2D7F TIFINAGH CONSONANT JOINER
2D80 ETHIOPIC SYLLABLE LOA
2D81 ETHIOPIC SYLLABLE MOA
2D82 ETHIOPIC SYLLABLE ROA
2D83 ETHIOPIC SYLLABLE SOA
2D84 ETHIOPIC SYLLABLE SHOA
2D85 ETHIOPIC SYLLABLE BOA
2D86 ETHIOPIC SYLLABLE TOA
2D87 ETHIOPIC SYLLABLE COA
2D88 ETHIOPIC SYLLABLE NOA
2D89 ETHIOPIC SYLLABLE NYOA
2D8A ETHIOPIC SYLLABLE GLOTTAL OA
2D8B ETHIOPIC SYLLABLE ZOA
2D8C ETHIOPIC SYLLABLE DOA
2D8D ETHIOPIC SYLLABLE DDOA
2D8E ETHIOPIC SYLLABLE JOA
2D8F ETHIOPIC SYLLABLE THOA
2D90 ETHIOPIC SYLLABLE CHOA
2D91 ETHIOPIC SYLLABLE PHOA
2D92 ETHIOPIC SYLLABLE POA
2D93 ETHIOPIC SYLLABLE GGWA
2D94 ETHIOPIC SYLLABLE GGWI
2D95 ETHIOPIC SYLLABLE GGWEE
2D96 ETHIOPIC SYLLABLE GGWE
2DA0 ETHIOPIC SYLLABLE SSA
2DA1 ETHIOPIC SYLLABLE SSU
2DA2 ETHIOPIC SYLLABLE SSI
2DA3 ETHIOPIC SYLLABLE SSAA
2DA4 ETHIOPIC SYLLABLE SSEE
2DA5 ETHIOPIC SYLLABLE SSE
2DA6 ETHIOPIC SYLLABLE SSO
2DA8 ETHIOPIC SYLLABLE CCA
2DA9 ETHIOPIC SYLLABLE CCU
2DAA ETHIOPIC SYLLABLE CCI
2DAB ETHIOPIC SYLLABLE CCAA
2DAC ETHIOPIC SYLLABLE CCEE
2DAD ETHIOPIC SYLLABLE CCE
2DAE ETHIOPIC SYLLABLE CCO
2DB0 ETHIOPIC SYLLABLE ZZA
2DB1 ETHIOPIC SYLLABLE ZZU
2DB2 ETHIOPIC SYLLABLE ZZI
2DB3 ETHIOPIC SYLLABLE ZZAA
2DB4 ETHIOPIC SYLLABLE ZZEE
2DB5 ETHIOPIC SYLLABLE ZZE
2DB6 ETHIOPIC SYLLABLE ZZO
2DB8 ETHIOPIC SYLLABLE CCHA
2DB9 ETHIOPIC SYLLABLE CCHU
2DBA ETHIOPIC SYLLABLE CCHI
2DBB ETHIOPIC SYLLABLE CCHAA
2DBC ETHIOPIC SYLLABLE CCHEE
2DBD ETHIOPIC SYLLABLE CCHE
2DBE ETHIOPIC SYLLABLE CCHO
2DC0 ETHIOPIC SYLLABLE QYA
2DC1 ETHIOPIC SYLLABLE QYU
2DC2 ETHIOPIC SYLLABLE QYI
2DC3 ETHIOPIC SYLLABLE QYAA
2DC4 ETHIOPIC SYLLABLE QYEE
2DC5 ETHIOPIC SYLLABLE QYE
2DC6 ETHIOPIC SYLLABLE QYO
2DC8 ETHIOPIC SYLLABLE KYA
2DC9 ETHIOPIC SYLLABLE KYU
2DCA ETHIOPIC SYLLABLE KYI
2DCB ETHIOPIC SYLLABLE KYAA
2DCC ETHIOPIC SYLLABLE KYEE
2DCD ETHIOPIC SYLLABLE KYE
2DCE ETHIOPIC SYLLABLE KYO
2DD0 ETHIOPIC SYLLABLE XYA
2DD1 ETHIOPIC SYLLABLE XYU
2DD2 ETHIOPIC SYLLABLE XYI
2DD3 ETHIOPIC SYLLABLE XYAA
2DD4 ETHIOPIC SYLLABLE XYEE
2DD5 ETHIOPIC SYLLABLE XYE
2DD6 ETHIOPIC SYLLABLE XYO
2DD8 ETHIOPIC SYLLABLE GYA
2DD9 ETHIOPIC SYLLABLE GYU
2DDA ETHIOPIC SYLLABLE GYI
2DDB ETHIOPIC SYLLABLE GYAA
2DDC ETHIOPIC SYLLABLE GYEE
2DDD ETHIOPIC SYLLABLE GYE
2DDE ETHIOPIC SYLLABLE GYO
2DE0 COMBINING CYRILLIC LETTER BE
2DE1 COMBINING CYRILLIC LETTER VE
2DE2 COMBINING CYRILLIC LETTER GHE
2DE3 COMBINING CYRILLIC LETTER DE
2DE4 COMBINING CYRILLIC LETTER ZHE
2DE5 COMBINING CYRILLIC LETTER ZE
2DE6 COMBINING CYRILLIC LETTER KA
2DE7 COMBINING CYRILLIC LETTER EL
2DE8 COMBINING CYRILLIC LETTER EM
2DE9 COMBINING CYRILLIC LETTER EN
2DEA COMBINING CYRILLIC LETTER O
2DEB COMBINING CYRILLIC LETTER PE
2DEC COMBINING CYRILLIC LETTER ER
2DED COMBINING CYRILLIC LETTER ES
2DEE COMBINING CYRILLIC LETTER TE
2DEF COMBINING CYRILLIC LETTER HA
2DF0 COMBINING CYRILLIC LETTER TSE
2DF1 COMBINING CYRILLIC LETTER CHE
2DF2 COMBINING CYRILLIC LETTER SHA
2DF3 COMBINING CYRILLIC LETTER SHCHA
2DF4 COMBINING CYRILLIC LETTER FITA
2DF5 COMBINING CYRILLIC LETTER ES-TE
2DF6 COMBINING CYRILLIC LETTER A
2DF7 COMBINING CYRILLIC LETTER IE
2DF8 COMBINING CYRILLIC LETTER DJERV
2DF9 COMBINING CYRILLIC LETTER MONOGRAPH UK
2DFA COMBINING CYRILLIC LETTER YAT
2DFB COMBINING CYRILLIC LETTER YU
2DFC COMBINING CYRILLIC LETTER IOTIFIED A
2DFD COMBINING CYRILLIC LETTER LITTLE YUS
2DFE COMBINING CYRILLIC LETTER BIG YUS
2DFF COMBINING CYRILLIC LETTER IOTIFIED BIG YUS
2E00 RIGHT ANGLE SUBSTITUTION MARKER
2E01 RIGHT ANGLE DOTTED SUBSTITUTION MARKER
2E02 LEFT SUBSTITUTION BRACKET
2E03 RIGHT SUBSTITUTION BRACKET
2E04 LEFT DOTTED SUBSTITUTION BRACKET
2E05 RIGHT DOTTED SUBSTITUTION BRACKET
2E06 RAISED INTERPOLATION MARKER
2E07 RAISED DOTTED INTERPOLATION MARKER
2E08 DOTTED TRANSPOSITION MARKER
2E09 LEFT TRANSPOSITION BRACKET
2E0A RIGHT TRANSPOSITION BRACKET
2E0B RAISED SQUARE
2E0C LEFT RAISED OMISSION BRACKET
2E0D RIGHT RAISED OMISSION BRACKET
2E0E EDITORIAL CORONIS
2E0F PARAGRAPHOS
2E10 FORKED PARAGRAPHOS
2E11 REVERSED FORKED PARAGRAPHOS
2E12 HYPODIASTOLE
2E13 DOTTED OBELOS
2E14 DOWNWARDS ANCORA
2E15 UPWARDS ANCORA
2E16 DOTTED RIGHT-POINTING ANGLE
2E17 DOUBLE OBLIQUE HYPHEN
2E18 INVERTED INTERROBANG
2E19 PALM BRANCH
2E1A HYPHEN WITH DIAERESIS
2E1B TILDE WITH RING ABOVE
2E1C LEFT LOW PARAPHRASE BRACKET
2E1D RIGHT LOW PARAPHRASE BRACKET
2E1E TILDE WITH DOT ABOVE
2E1F TILDE WITH DOT BELOW
2E20 LEFT VERTICAL BAR WITH QUILL
2E21 RIGHT VERTICAL BAR WITH QUILL
2E22 TOP LEFT HALF BRACKET
2E23 TOP RIGHT HALF BRACKET
2E24 BOTTOM LEFT HALF BRACKET
2E25 BOTTOM RIGHT HALF BRACKET
2E26 LEFT SIDEWAYS U BRACKET
2E27 RIGHT SIDEWAYS U BRACKET
2E28 LEFT DOUBLE PARENTHESIS
2E29 RIGHT DOUBLE PARENTHESIS
2E2A TWO DOTS OVER ONE DOT PUNCTUATION
2E2B ONE DOT OVER TWO DOTS PUNCTUATION
2E2C SQUARED FOUR DOT PUNCTUATION
2E2D FIVE DOT MARK
2E2E REVERSED QUESTION MARK
2E2F VERTICAL TILDE
2E30 RING POINT
2E31 WORD SEPARATOR MIDDLE DOT
2E32 TURNED COMMA
2E33 RAISED DOT
2E34 RAISED COMMA
2E35 TURNED SEMICOLON
2E36 DAGGER WITH LEFT GUARD
2E37 DAGGER WITH RIGHT GUARD
2E38 TURNED DAGGER
2E39 TOP HALF SECTION SIGN
2E3A TWO-EM DASH
2E3B THREE-EM DASH
2E3C STENOGRAPHIC FULL STOP
2E3D VERTICAL SIX DOTS
2E3E WIGGLY VERTICAL LINE
2E3F CAPITULUM
2E40 DOUBLE HYPHEN
2E41 REVERSED COMMA
2E42 DOUBLE LOW-REVERSED-9 QUOTATION MARK
2E43 DASH WITH LEFT UPTURN
2E44 DOUBLE SUSPENSION MARK
2E45 INVERTED LOW KAVYKA
2E46 INVERTED LOW KAVYKA WITH KAVYKA ABOVE
2E47 LOW KAVYKA
2E48 LOW KAVYKA WITH DOT
2E49 DOUBLE STACKED COMMA
2E4A DOTTED SOLIDUS
2E4B TRIPLE DAGGER
2E4C MEDIEVAL COMMA
2E4D PARAGRAPHUS MARK
2E4E PUNCTUS ELEVATUS MARK
2E4F CORNISH VERSE DIVIDER
2E50 CROSS PATTY WITH RIGHT CROSSBAR
2E51 CROSS PATTY WITH LEFT CROSSBAR
2E52 TIRONIAN SIGN CAPITAL ET
2E53 MEDIEVAL EXCLAMATION MARK
2E54 MEDIEVAL QUESTION MARK
2E55 LEFT SQUARE BRACKET WITH STROKE
2E56 RIGHT SQUARE BRACKET WITH STROKE
2E57 LEFT SQUARE BRACKET WITH DOUBLE STROKE
2E58 RIGHT SQUARE BRACKET WITH DOUBLE STROKE
2E59 TOP HALF LEFT PARENTHESIS
2E5A TOP HALF RIGHT PARENTHESIS
2E5B BOTTOM HALF LEFT PARENTHESIS
2E5C BOTTOM HALF RIGHT PARENTHESIS
2E5D OBLIQUE HYPHEN
2E80 CJK RADICAL REPEAT
2E81 CJK RADICAL CLIFF
2E82 CJK RADICAL SECOND ONE
2E83 CJK RADICAL SECOND TWO
2E84 CJK RADICAL SECOND THREE
2E85 CJK RADICAL PERSON
2E86 CJK RADICAL BOX
2E87 CJK RADICAL TABLE
2E88 CJK RADICAL KNIFE ONE
2E89 CJK RADICAL KNIFE TWO
2E8A CJK RADICAL DIVINATION
2E8B CJK RADICAL SEAL
2E8C CJK RADICAL SMALL ONE
2E8D CJK RADICAL SMALL TWO
2E8E CJK RADICAL LAME ONE
2E8F CJK RADICAL LAME TWO
2E90 CJK RADICAL LAME THREE
2E91 CJK RADICAL LAME FOUR
2E92 CJK RADICAL SNAKE
2E93 CJK RADICAL THREAD
2E94 CJK RADICAL SNOUT ONE
2E95 CJK RADICAL SNOUT TWO
2E96 CJK RADICAL HEART ONE
2E97 CJK RADICAL HEART TWO
2E98 CJK RADICAL HAND
2E99 CJK RADICAL RAP
2E9B CJK RADICAL CHOKE
2E9C CJK RADICAL SUN
2E9D CJK RADICAL MOON
2E9E CJK RADICAL DEATH
2E9F CJK RADICAL MOTHER
2EA0 CJK RADICAL CIVILIAN
2EA1 CJK RADICAL WATER ONE
2EA2 CJK RADICAL WATER TWO
2EA3 CJK RADICAL FIRE
2EA4 CJK RADICAL PAW ONE
2EA5 CJK RADICAL PAW TWO
2EA6 CJK RADICAL SIMPLIFIED HALF TREE TRUNK
2EA7 CJK RADICAL COW
2EA8 CJK RADICAL DOG
2EA9 CJK RADICAL JADE
2EAA CJK RADICAL BOLT OF CLOTH
2EAB CJK RADICAL EYE
2EAC CJK RADICAL SPIRIT ONE
2EAD CJK RADICAL SPIRIT TWO
2EAE CJK RADICAL BAMBOO
2EAF CJK RADICAL SILK
2EB0 CJK RADICAL C-SIMPLIFIED SILK
2EB1 CJK RADICAL NET ONE
2EB2 CJK RADICAL NET TWO
2EB3 CJK RADICAL NET THREE
2EB4 CJK RADICAL NET FOUR
2EB5 CJK RADICAL MESH
2EB6 CJK RADICAL SHEEP
2EB7 CJK RADICAL RAM
2EB8 CJK RADICAL EWE
2EB9 CJK RADICAL OLD
2EBA CJK RADICAL BRUSH ONE
2EBB CJK RADICAL BRUSH TWO
2EBC CJK RADICAL MEAT
2EBD CJK RADICAL MORTAR
2EBE CJK RADICAL GRASS ONE
2EBF CJK RADICAL GRASS TWO
2EC0 CJK RADICAL GRASS THREE
2EC1 CJK RADICAL TIGER
2EC2 CJK RADICAL CLOTHES
2EC3 CJK RADICAL WEST ONE
2EC4 CJK RADICAL WEST TWO
2EC5 CJK RADICAL C-SIMPLIFIED SEE
2EC6 CJK RADICAL SIMPLIFIED HORN
2EC7 CJK RADICAL HORN
2EC8 CJK RADICAL C-SIMPLIFIED SPEECH
2EC9 CJK RADICAL C-SIMPLIFIED SHELL
2ECA CJK RADICAL FOOT
2ECB CJK RADICAL C-SIMPLIFIED CART
2ECC CJK RADICAL SIMPLIFIED WALK
2ECD CJK RADICAL WALK ONE
2ECE CJK RADICAL WALK TWO
2ECF CJK RADICAL CITY
2ED0 CJK RADICAL C-SIMPLIFIED GOLD
2ED1 CJK RADICAL LONG ONE
2ED2 CJK RADICAL LONG TWO
2ED3 CJK RADICAL C-SIMPLIFIED LONG
2ED4 CJK RADICAL C-SIMPLIFIED GATE
2ED5 CJK RADICAL MOUND ONE
2ED6 CJK RADICAL MOUND TWO
2ED7 CJK RADICAL RAIN
2ED8 CJK RADICAL BLUE
2ED9 CJK RADICAL C-SIMPLIFIED TANNED LEATHER
2EDA CJK RADICAL C-SIMPLIFIED LEAF
2EDB CJK RADICAL C-SIMPLIFIED WIND
2EDC CJK RADICAL C-SIMPLIFIED FLY
2EDD CJK RADICAL EAT ONE
2EDE CJK RADICAL EAT TWO
2EDF CJK RADICAL EAT THREE
2EE0 CJK RADICAL C-SIMPLIFIED EAT
2EE1 CJK RADICAL HEAD
2EE2 CJK RADICAL C-SIMPLIFIED HORSE
2EE3 CJK RADICAL BONE
2EE4 CJK RADICAL GHOST
2EE5 CJK RADICAL C-SIMPLIFIED FISH
2EE6 CJK RADICAL C-SIMPLIFIED BIRD
2EE7 CJK RADICAL C-SIMPLIFIED SALT
2EE8 CJK RADICAL SIMPLIFIED WHEAT
2EE9 CJK RADICAL SIMPLIFIED YELLOW
2EEA CJK RADICAL C-SIMPLIFIED FROG
2EEB CJK RADICAL J-SIMPLIFIED EVEN
2EEC CJK RADICAL C-SIMPLIFIED EVEN
2EED CJK RADICAL J-SIMPLIFIED TOOTH
2EEE CJK RADICAL C-SIMPLIFIED TOOTH
2EEF CJK RADICAL J-SIMPLIFIED DRAGON
2EF0 CJK RADICAL C-SIMPLIFIED DRAGON
2EF1 CJK RADICAL TURTLE
2EF2 CJK RADICAL J-SIMPLIFIED TURTLE
2EF3 CJK RADICAL C-SIMPLIFIED TURTLE
2F00 KANGXI RADICAL ONE
2F01 KANGXI RADICAL LINE
2F02 KANGXI RADICAL DOT
2F03 KANGXI RADICAL SLASH
2F04 KANGXI RADICAL SECOND
2F05 KANGXI RADICAL HOOK
2F06 KANGXI RADICAL TWO
2F07 KANGXI RADICAL LID
2F08 KANGXI RADICAL MAN
2F09 KANGXI RADICAL LEGS
2F0A KANGXI RADICAL ENTER
2F0B KANGXI RADICAL EIGHT
2F0C KANGXI RADICAL DOWN BOX
2F0D KANGXI RADICAL COVER
2F0E KANGXI RADICAL ICE
2F0F KANGXI RADICAL TABLE
2F10 KANGXI RADICAL OPEN BOX
2F11 KANGXI RADICAL KNIFE
2F12 KANGXI RADICAL POWER
2F13 KANGXI RADICAL WRAP
2F14 KANGXI RADICAL SPOON
2F15 KANGXI RADICAL RIGHT OPEN BOX
2F16 KANGXI RADICAL HIDING ENCLOSURE
2F17 KANGXI RADICAL TEN
2F18 KANGXI RADICAL DIVINATION
2F19 KANGXI RADICAL SEAL
2F1A KANGXI RADICAL CLIFF
2F1B KANGXI RADICAL PRIVATE
2F1C KANGXI RADICAL AGAIN
2F1D KANGXI RADICAL MOUTH
2F1E KANGXI RADICAL ENCLOSURE
2F1F KANGXI RADICAL EARTH
2F20 KANGXI RADICAL SCHOLAR
2F21 KANGXI RADICAL GO
2F22 KANGXI RADICAL GO SLOWLY
2F23 KANGXI RADICAL EVENING
2F24 KANGXI RADICAL BIG
2F25 KANGXI RADICAL WOMAN
2F26 KANGXI RADICAL CHILD
2F27 KANGXI RADICAL ROOF
2F28 KANGXI RADICAL INCH
2F29 KANGXI RADICAL SMALL
2F2A KANGXI RADICAL LAME
2F2B KANGXI RADICAL CORPSE
2F2C KANGXI RADICAL SPROUT
2F2D KANGXI RADICAL MOUNTAIN
2F2E KANGXI RADICAL RIVER
2F2F KANGXI RADICAL WORK
2F30 KANGXI RADICAL ONESELF
2F31 KANGXI RADICAL TURBAN
2F32 KANGXI RADICAL DRY
2F33 KANGXI RADICAL SHORT THREAD
2F34 KANGXI RADICAL DOTTED CLIFF
2F35 KANGXI RADICAL LONG STRIDE
2F36 KANGXI RADICAL TWO HANDS
2F37 KANGXI RADICAL SHOOT
2F38 KANGXI RADICAL BOW
2F39 KANGXI RADICAL SNOUT
2F3A KANGXI RADICAL BRISTLE
2F3B KANGXI RADICAL STEP
2F3C KANGXI RADICAL HEART
2F3D KANGXI RADICAL HALBERD
2F3E KANGXI RADICAL DOOR
2F3F KANGXI RADICAL HAND
2F40 KANGXI RADICAL BRANCH
2F41 KANGXI RADICAL RAP
2F42 KANGXI RADICAL SCRIPT
2F43 KANGXI RADICAL DIPPER
2F44 KANGXI RADICAL AXE
2F45 KANGXI RADICAL SQUARE
2F46 KANGXI RADICAL NOT
2F47 KANGXI RADICAL SUN
2F48 KANGXI RADICAL SAY
2F49 KANGXI RADICAL MOON
2F4A KANGXI RADICAL TREE
2F4B KANGXI RADICAL LACK
2F4C KANGXI RADICAL STOP
2F4D KANGXI RADICAL DEATH
2F4E KANGXI RADICAL WEAPON
2F4F KANGXI RADICAL DO NOT
2F50 KANGXI RADICAL COMPARE
2F51 KANGXI RADICAL FUR
2F52 KANGXI RADICAL CLAN
2F53 KANGXI RADICAL STEAM
2F54 KANGXI RADICAL WATER
2F55 KANGXI RADICAL FIRE
2F56 KANGXI RADICAL CLAW
2F57 KANGXI RADICAL FATHER
2F58 KANGXI RADICAL DOUBLE X
2F59 KANGXI RADICAL HALF TREE TRUNK
2F5A KANGXI RADICAL SLICE
2F5B KANGXI RADICAL FANG
2F5C KANGXI RADICAL COW
2F5D KANGXI RADICAL DOG
2F5E KANGXI RADICAL PROFOUND
2F5F KANGXI RADICAL JADE
2F60 KANGXI RADICAL MELON
2F61 KANGXI RADICAL TILE
2F62 KANGXI RADICAL SWEET
2F63 KANGXI RADICAL LIFE
2F64 KANGXI RADICAL USE
2F65 KANGXI RADICAL FIELD
2F66 KANGXI RADICAL BOLT OF CLOTH
2F67 KANGXI RADICAL SICKNESS
2F68 KANGXI RADICAL DOTTED TENT
2F69 KANGXI RADICAL WHITE
2F6A KANGXI RADICAL SKIN
2F6B KANGXI RADICAL DISH
2F6C KANGXI RADICAL EYE
2F6D KANGXI RADICAL SPEAR
2F6E KANGXI RADICAL ARROW
2F6F KANGXI RADICAL STONE
2F70 KANGXI RADICAL SPIRIT
2F71 KANGXI RADICAL TRACK
2F72 KANGXI RADICAL GRAIN
2F73 KANGXI RADICAL CAVE
2F74 KANGXI RADICAL STAND
2F75 KANGXI RADICAL BAMBOO
2F76 KANGXI RADICAL RICE
2F77 KANGXI RADICAL SILK
2F78 KANGXI RADICAL JAR
2F79 KANGXI RADICAL NET
2F7A KANGXI RADICAL SHEEP
2F7B KANGXI RADICAL FEATHER
2F7C KANGXI RADICAL OLD
2F7D KANGXI RADICAL AND
2F7E KANGXI RADICAL PLOW
2F7F KANGXI RADICAL EAR
2F80 KANGXI RADICAL BRUSH
2F81 KANGXI RADICAL MEAT
2F82 KANGXI RADICAL MINISTER
2F83 KANGXI RADICAL SELF
2F84 KANGXI RADICAL ARRIVE
2F85 KANGXI RADICAL MORTAR
2F86 KANGXI RADICAL TONGUE
2F87 KANGXI RADICAL OPPOSE
2F88 KANGXI RADICAL BOAT
2F89 KANGXI RADICAL STOPPING
2F8A KANGXI RADICAL COLOR
2F8B KANGXI RADICAL GRASS
2F8C KANGXI RADICAL TIGER
2F8D KANGXI RADICAL INSECT
2F8E KANGXI RADICAL BLOOD
2F8F KANGXI RADICAL WALK ENCLOSURE
2F90 KANGXI RADICAL CLOTHES
2F91 KANGXI RADICAL WEST
2F92 KANGXI RADICAL SEE
2F93 KANGXI RADICAL HORN
2F94 KANGXI RADICAL SPEECH
2F95 KANGXI RADICAL VALLEY
2F96 KANGXI RADICAL BEAN
2F97 KANGXI RADICAL PIG
2F98 KANGXI RADICAL BADGER
2F99 KANGXI RADICAL SHELL
2F9A KANGXI RADICAL RED
2F9B KANGXI RADICAL RUN
2F9C KANGXI RADICAL FOOT
2F9D KANGXI RADICAL BODY
2F9E KANGXI RADICAL CART
2F9F KANGXI RADICAL BITTER
2FA0 KANGXI RADICAL MORNING
2FA1 KANGXI RADICAL WALK
2FA2 KANGXI RADICAL CITY
2FA3 KANGXI RADICAL WINE
2FA4 KANGXI RADICAL DISTINGUISH
2FA5 KANGXI RADICAL VILLAGE
2FA6 KANGXI RADICAL GOLD
2FA7 KANGXI RADICAL LONG
2FA8 KANGXI RADICAL GATE
2FA9 KANGXI RADICAL MOUND
2FAA KANGXI RADICAL SLAVE
2FAB KANGXI RADICAL SHORT TAILED BIRD
2FAC KANGXI RADICAL RAIN
2FAD KANGXI RADICAL BLUE
2FAE KANGXI RADICAL WRONG
2FAF KANGXI RADICAL FACE
2FB0 KANGXI RADICAL LEATHER
2FB1 KANGXI RADICAL TANNED LEATHER
2FB2 KANGXI RADICAL LEEK
2FB3 KANGXI RADICAL SOUND
2FB4 KANGXI RADICAL LEAF
2FB5 KANGXI RADICAL WIND
2FB6 KANGXI RADICAL FLY
2FB7 KANGXI RADICAL EAT
2FB8 KANGXI RADICAL HEAD
2FB9 KANGXI RADICAL FRAGRANT
2FBA KANGXI RADICAL HORSE
2FBB KANGXI RADICAL BONE
2FBC KANGXI RADICAL TALL
2FBD KANGXI RADICAL HAIR
2FBE KANGXI RADICAL FIGHT
2FBF KANGXI RADICAL SACRIFICIAL WINE
2FC0 KANGXI RADICAL CAULDRON
2FC1 KANGXI RADICAL GHOST
2FC2 KANGXI RADICAL FISH
2FC3 KANGXI RADICAL BIRD
2FC4 KANGXI RADICAL SALT
2FC5 KANGXI RADICAL DEER
2FC6 KANGXI RADICAL WHEAT
2FC7 KANGXI RADICAL HEMP
2FC8 KANGXI RADICAL YELLOW
2FC9 KANGXI RADICAL MILLET
2FCA KANGXI RADICAL BLACK
2FCB KANGXI RADICAL EMBROIDERY
2FCC KANGXI RADICAL FROG
2FCD KANGXI RADICAL TRIPOD
2FCE KANGXI RADICAL DRUM
2FCF KANGXI RADICAL RAT
2FD0 KANGXI RADICAL NOSE
2FD1 KANGXI RADICAL EVEN
2FD2 KANGXI RADICAL TOOTH
2FD3 KANGXI RADICAL DRAGON
2FD4 KANGXI RADICAL TURTLE
2FD5 KANGXI RADICAL FLUTE
2FF0 IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO RIGHT
2FF1 IDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO BELOW
2FF2 IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO MIDDLE AND RIGHT
2FF3 IDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO MIDDLE AND BELOW
2FF4 IDEOGRAPHIC DESCRIPTION CHARACTER FULL SURROUND
2FF5 IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM ABOVE
2FF6 IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM BELOW
2FF7 IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LEFT
2FF8 IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER LEFT
2FF9 IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER RIGHT
2FFA IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LOWER LEFT
2FFB IDEOGRAPHIC DESCRIPTION CHARACTER OVERLAID
3000 IDEOGRAPHIC SPACE
3001 IDEOGRAPHIC COMMA
3002 IDEOGRAPHIC FULL STOP
3003 DITTO MARK
3004 JAPANESE INDUSTRIAL STANDARD SYMBOL
3005 IDEOGRAPHIC ITERATION MARK
3006 IDEOGRAPHIC CLOSING MARK
3007 IDEOGRAPHIC NUMBER ZERO
3008 LEFT ANGLE BRACKET
3009 RIGHT ANGLE BRACKET
300A LEFT DOUBLE ANGLE BRACKET
300B RIGHT DOUBLE ANGLE BRACKET
300C LEFT CORNER BRACKET
300D RIGHT CORNER BRACKET
300E LEFT WHITE CORNER BRACKET
300F RIGHT WHITE CORNER BRACKET
3010 LEFT BLACK LENTICULAR BRACKET
3011 RIGHT BLACK LENTICULAR BRACKET
3012 POSTAL MARK
3013 GETA MARK
3014 LEFT TORTOISE SHELL BRACKET
3015 RIGHT TORTOISE SHELL BRACKET
3016 LEFT WHITE LENTICULAR BRACKET
3017 RIGHT WHITE LENTICULAR BRACKET
3018 LEFT WHITE TORTOISE SHELL BRACKET
3019 RIGHT WHITE TORTOISE SHELL BRACKET
301A LEFT WHITE SQUARE BRACKET
301B RIGHT WHITE SQUARE BRACKET
301C WAVE DASH
301D REVERSED DOUBLE PRIME QUOTATION MARK
301E DOUBLE PRIME QUOTATION MARK
301F LOW DOUBLE PRIME QUOTATION MARK
3020 POSTAL MARK FACE
3021 HANGZHOU NUMERAL ONE
3022 HANGZHOU NUMERAL TWO
3023 HANGZHOU NUMERAL THREE
3024 HANGZHOU NUMERAL FOUR
3025 HANGZHOU NUMERAL FIVE
3026 HANGZHOU NUMERAL SIX
3027 HANGZHOU NUMERAL SEVEN
3028 HANGZHOU NUMERAL EIGHT
3029 HANGZHOU NUMERAL NINE
302A IDEOGRAPHIC LEVEL TONE MARK
302B IDEOGRAPHIC RISING TONE MARK
302C IDEOGRAPHIC DEPARTING TONE MARK
302D IDEOGRAPHIC ENTERING TONE MARK
302E HANGUL SINGLE DOT TONE MARK
302F HANGUL DOUBLE DOT TONE MARK
3030 WAVY DASH
3031 VERTICAL KANA REPEAT MARK
3032 VERTICAL KANA REPEAT WITH VOICED SOUND MARK
3033 VERTICAL KANA REPEAT MARK UPPER HALF
3034 VERTICAL KANA REPEAT WITH VOICED SOUND MARK UPPER HALF
3035 VERTICAL KANA REPEAT MARK LOWER HALF
3036 CIRCLED POSTAL MARK
3037 IDEOGRAPHIC TELEGRAPH LINE FEED SEPARATOR SYMBOL
3038 HANGZHOU NUMERAL TEN
3039 HANGZHOU NUMERAL TWENTY
303A HANGZHOU NUMERAL THIRTY
303B VERTICAL IDEOGRAPHIC ITERATION MARK
303C MASU MARK
303D PART ALTERNATION MARK
303E IDEOGRAPHIC VARIATION INDICATOR
303F IDEOGRAPHIC HALF FILL SPACE
3041 HIRAGANA LETTER SMALL A
3042 HIRAGANA LETTER A
3043 HIRAGANA LETTER SMALL I
3044 HIRAGANA LETTER I
3045 HIRAGANA LETTER SMALL U
3046 HIRAGANA LETTER U
3047 HIRAGANA LETTER SMALL E
3048 HIRAGANA LETTER E
3049 HIRAGANA LETTER SMALL O
304A HIRAGANA LETTER O
304B HIRAGANA LETTER KA
304C HIRAGANA LETTER GA
304D HIRAGANA LETTER KI
304E HIRAGANA LETTER GI
304F HIRAGANA LETTER KU
3050 HIRAGANA LETTER GU
3051 HIRAGANA LETTER KE
3052 HIRAGANA LETTER GE
3053 HIRAGANA LETTER KO
3054 HIRAGANA LETTER GO
3055 HIRAGANA LETTER SA
3056 HIRAGANA LETTER ZA
3057 HIRAGANA LETTER SI
3058 HIRAGANA LETTER ZI
3059 HIRAGANA LETTER SU
305A HIRAGANA LETTER ZU
305B HIRAGANA LETTER SE
305C HIRAGANA LETTER ZE
305D HIRAGANA LETTER SO
305E HIRAGANA LETTER ZO
305F HIRAGANA LETTER TA
3060 HIRAGANA LETTER DA
3061 HIRAGANA LETTER TI
3062 HIRAGANA LETTER DI
3063 HIRAGANA LETTER SMALL TU
3064 HIRAGANA LETTER TU
3065 HIRAGANA LETTER DU
3066 HIRAGANA LETTER TE
3067 HIRAGANA LETTER DE
3068 HIRAGANA LETTER TO
3069 HIRAGANA LETTER DO
306A HIRAGANA LETTER NA
306B HIRAGANA LETTER NI
306C HIRAGANA LETTER NU
306D HIRAGANA LETTER NE
306E HIRAGANA LETTER NO
306F HIRAGANA LETTER HA
3070 HIRAGANA LETTER BA
3071 HIRAGANA LETTER PA
3072 HIRAGANA LETTER HI
3073 HIRAGANA LETTER BI
3074 HIRAGANA LETTER PI
3075 HIRAGANA LETTER HU
3076 HIRAGANA LETTER BU
3077 HIRAGANA LETTER PU
3078 HIRAGANA LETTER HE
3079 HIRAGANA LETTER BE
307A HIRAGANA LETTER PE
307B HIRAGANA LETTER HO
307C HIRAGANA LETTER BO
307D HIRAGANA LETTER PO
307E HIRAGANA LETTER MA
307F HIRAGANA LETTER MI
3080 HIRAGANA LETTER MU
3081 HIRAGANA LETTER ME
3082 HIRAGANA LETTER MO
3083 HIRAGANA LETTER SMALL YA
3084 HIRAGANA LETTER YA
3085 HIRAGANA LETTER SMALL YU
3086 HIRAGANA LETTER YU
3087 HIRAGANA LETTER SMALL YO
3088 HIRAGANA LETTER YO
3089 HIRAGANA LETTER RA
308A HIRAGANA LETTER RI
308B HIRAGANA LETTER RU
308C HIRAGANA LETTER RE
308D HIRAGANA LETTER RO
308E HIRAGANA LETTER SMALL WA
308F HIRAGANA LETTER WA
3090 HIRAGANA LETTER WI
3091 HIRAGANA LETTER WE
3092 HIRAGANA LETTER WO
3093 HIRAGANA LETTER N
3094 HIRAGANA LETTER VU
3095 HIRAGANA LETTER SMALL KA
3096 HIRAGANA LETTER SMALL KE
3099 COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK
309A COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
309B KATAKANA-HIRAGANA VOICED SOUND MARK
309C KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
309D HIRAGANA ITERATION MARK
309E HIRAGANA VOICED ITERATION MARK
309F HIRAGANA DIGRAPH YORI
30A0 KATAKANA-HIRAGANA DOUBLE HYPHEN
30A1 KATAKANA LETTER SMALL A
30A2 KATAKANA LETTER A
30A3 KATAKANA LETTER SMALL I
30A4 KATAKANA LETTER I
30A5 KATAKANA LETTER SMALL U
30A6 KATAKANA LETTER U
30A7 KATAKANA LETTER SMALL E
30A8 KATAKANA LETTER E
30A9 KATAKANA LETTER SMALL O
30AA KATAKANA LETTER O
30AB KATAKANA LETTER KA
30AC KATAKANA LETTER GA
30AD KATAKANA LETTER KI
30AE KATAKANA LETTER GI
30AF KATAKANA LETTER KU
30B0 KATAKANA LETTER GU
30B1 KATAKANA LETTER KE
30B2 KATAKANA LETTER GE
30B3 KATAKANA LETTER KO
30B4 KATAKANA LETTER GO
30B5 KATAKANA LETTER SA
30B6 KATAKANA LETTER ZA
30B7 KATAKANA LETTER SI
30B8 KATAKANA LETTER ZI
30B9 KATAKANA LETTER SU
30BA KATAKANA LETTER ZU
30BB KATAKANA LETTER SE
30BC KATAKANA LETTER ZE
30BD KATAKANA LETTER SO
30BE KATAKANA LETTER ZO
30BF KATAKANA LETTER TA
30C0 KATAKANA LETTER DA
30C1 KATAKANA LETTER TI
30C2 KATAKANA LETTER DI
30C3 KATAKANA LETTER SMALL TU
30C4 KATAKANA LETTER TU
30C5 KATAKANA LETTER DU
30C6 KATAKANA LETTER TE
30C7 KATAKANA LETTER DE
30C8 KATAKANA LETTER TO
30C9 KATAKANA LETTER DO
30CA KATAKANA LETTER NA
30CB KATAKANA LETTER NI
30CC KATAKANA LETTER NU
30CD KATAKANA LETTER NE
30CE KATAKANA LETTER NO
30CF KATAKANA LETTER HA
30D0 KATAKANA LETTER BA
30D1 KATAKANA LETTER PA
30D2 KATAKANA LETTER HI
30D3 KATAKANA LETTER BI
30D4 KATAKANA LETTER PI
30D5 KATAKANA LETTER HU
30D6 KATAKANA LETTER BU
30D7 KATAKANA LETTER PU
30D8 KATAKANA LETTER HE
30D9 KATAKANA LETTER BE
30DA KATAKANA LETTER PE
30DB KATAKANA LETTER HO
30DC KATAKANA LETTER BO
30DD KATAKANA LETTER PO
30DE KATAKANA LETTER MA
30DF KATAKANA LETTER MI
30E0 KATAKANA LETTER MU
30E1 KATAKANA LETTER ME
30E2 KATAKANA LETTER MO
30E3 KATAKANA LETTER SMALL YA
30E4 KATAKANA LETTER YA
30E5 KATAKANA LETTER SMALL YU
30E6 KATAKANA LETTER YU
30E7 KATAKANA LETTER SMALL YO
30E8 KATAKANA LETTER YO
30E9 KATAKANA LETTER RA
30EA KATAKANA LETTER RI
30EB KATAKANA LETTER RU
30EC KATAKANA LETTER RE
30ED KATAKANA LETTER RO
30EE KATAKANA LETTER SMALL WA
30EF KATAKANA LETTER WA
30F0 KATAKANA LETTER WI
30F1 KATAKANA LETTER WE
30F2 KATAKANA LETTER WO
30F3 KATAKANA LETTER N
30F4 KATAKANA LETTER VU
30F5 KATAKANA LETTER SMALL KA
30F6 KATAKANA LETTER SMALL KE
30F7 KATAKANA LETTER VA
30F8 KATAKANA LETTER VI
30F9 KATAKANA LETTER VE
30FA KATAKANA LETTER VO
30FB KATAKANA MIDDLE DOT
30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
30FD KATAKANA ITERATION MARK
30FE KATAKANA VOICED ITERATION MARK
30FF KATAKANA DIGRAPH KOTO
3105 BOPOMOFO LETTER B
3106 BOPOMOFO LETTER P
3107 BOPOMOFO LETTER M
3108 BOPOMOFO LETTER F
3109 BOPOMOFO LETTER D
310A BOPOMOFO LETTER T
310B BOPOMOFO LETTER N
310C BOPOMOFO LETTER L
310D BOPOMOFO LETTER G
310E BOPOMOFO LETTER K
310F BOPOMOFO LETTER H
3110 BOPOMOFO LETTER J
3111 BOPOMOFO LETTER Q
3112 BOPOMOFO LETTER X
3113 BOPOMOFO LETTER ZH
3114 BOPOMOFO LETTER CH
3115 BOPOMOFO LETTER SH
3116 BOPOMOFO LETTER R
3117 BOPOMOFO LETTER Z
3118 BOPOMOFO LETTER C
3119 BOPOMOFO LETTER S
311A BOPOMOFO LETTER A
311B BOPOMOFO LETTER O
311C BOPOMOFO LETTER E
311D BOPOMOFO LETTER EH
311E BOPOMOFO LETTER AI
311F BOPOMOFO LETTER EI
3120 BOPOMOFO LETTER AU
3121 BOPOMOFO LETTER OU
3122 BOPOMOFO LETTER AN
3123 BOPOMOFO LETTER EN
3124 BOPOMOFO LETTER ANG
3125 BOPOMOFO LETTER ENG
3126 BOPOMOFO LETTER ER
3127 BOPOMOFO LETTER I
3128 BOPOMOFO LETTER U
3129 BOPOMOFO LETTER IU
312A BOPOMOFO LETTER V
312B BOPOMOFO LETTER NG
312C BOPOMOFO LETTER GN
312D BOPOMOFO LETTER IH
312E BOPOMOFO LETTER O WITH DOT ABOVE
312F BOPOMOFO LETTER NN
3131 HANGUL LETTER KIYEOK
3132 HANGUL LETTER SSANGKIYEOK
3133 HANGUL LETTER KIYEOK-SIOS
3134 HANGUL LETTER NIEUN
3135 HANGUL LETTER NIEUN-CIEUC
3136 HANGUL LETTER NIEUN-HIEUH
3137 HANGUL LETTER TIKEUT
3138 HANGUL LETTER SSANGTIKEUT
3139 HANGUL LETTER RIEUL
313A HANGUL LETTER RIEUL-KIYEOK
313B HANGUL LETTER RIEUL-MIEUM
313C HANGUL LETTER RIEUL-PIEUP
313D HANGUL LETTER RIEUL-SIOS
313E HANGUL LETTER RIEUL-THIEUTH
313F HANGUL LETTER RIEUL-PHIEUPH
3140 HANGUL LETTER RIEUL-HIEUH
3141 HANGUL LETTER MIEUM
3142 HANGUL LETTER PIEUP
3143 HANGUL LETTER SSANGPIEUP
3144 HANGUL LETTER PIEUP-SIOS
3145 HANGUL LETTER SIOS
3146 HANGUL LETTER SSANGSIOS
3147 HANGUL LETTER IEUNG
3148 HANGUL LETTER CIEUC
3149 HANGUL LETTER SSANGCIEUC
314A HANGUL LETTER CHIEUCH
314B HANGUL LETTER KHIEUKH
314C HANGUL LETTER THIEUTH
314D HANGUL LETTER PHIEUPH
314E HANGUL LETTER HIEUH
314F HANGUL LETTER A
3150 HANGUL LETTER AE
3151 HANGUL LETTER YA
3152 HANGUL LETTER YAE
3153 HANGUL LETTER EO
3154 HANGUL LETTER E
3155 HANGUL LETTER YEO
3156 HANGUL LETTER YE
3157 HANGUL LETTER O
3158 HANGUL LETTER WA
3159 HANGUL LETTER WAE
315A HANGUL LETTER OE
315B HANGUL LETTER YO
315C HANGUL LETTER U
315D HANGUL LETTER WEO
315E HANGUL LETTER WE
315F HANGUL LETTER WI
3160 HANGUL LETTER YU
3161 HANGUL LETTER EU
3162 HANGUL LETTER YI
3163 HANGUL LETTER I
3164 HANGUL FILLER
3165 HANGUL LETTER SSANGNIEUN
3166 HANGUL LETTER NIEUN-TIKEUT
3167 HANGUL LETTER NIEUN-SIOS
3168 HANGUL LETTER NIEUN-PANSIOS
3169 HANGUL LETTER RIEUL-KIYEOK-SIOS
316A HANGUL LETTER RIEUL-TIKEUT
316B HANGUL LETTER RIEUL-PIEUP-SIOS
316C HANGUL LETTER RIEUL-PANSIOS
316D HANGUL LETTER RIEUL-YEORINHIEUH
316E HANGUL LETTER MIEUM-PIEUP
316F HANGUL LETTER MIEUM-SIOS
3170 HANGUL LETTER MIEUM-PANSIOS
3171 HANGUL LETTER KAPYEOUNMIEUM
3172 HANGUL LETTER PIEUP-KIYEOK
3173 HANGUL LETTER PIEUP-TIKEUT
3174 HANGUL LETTER PIEUP-SIOS-KIYEOK
3175 HANGUL LETTER PIEUP-SIOS-TIKEUT
3176 HANGUL LETTER PIEUP-CIEUC
3177 HANGUL LETTER PIEUP-THIEUTH
3178 HANGUL LETTER KAPYEOUNPIEUP
3179 HANGUL LETTER KAPYEOUNSSANGPIEUP
317A HANGUL LETTER SIOS-KIYEOK
317B HANGUL LETTER SIOS-NIEUN
317C HANGUL LETTER SIOS-TIKEUT
317D HANGUL LETTER SIOS-PIEUP
317E HANGUL LETTER SIOS-CIEUC
317F HANGUL LETTER PANSIOS
3180 HANGUL LETTER SSANGIEUNG
3181 HANGUL LETTER YESIEUNG
3182 HANGUL LETTER YESIEUNG-SIOS
3183 HANGUL LETTER YESIEUNG-PANSIOS
3184 HANGUL LETTER KAPYEOUNPHIEUPH
3185 HANGUL LETTER SSANGHIEUH
3186 HANGUL LETTER YEORINHIEUH
3187 HANGUL LETTER YO-YA
3188 HANGUL LETTER YO-YAE
3189 HANGUL LETTER YO-I
318A HANGUL LETTER YU-YEO
318B HANGUL LETTER YU-YE
318C HANGUL LETTER YU-I
318D HANGUL LETTER ARAEA
318E HANGUL LETTER ARAEAE
3190 IDEOGRAPHIC ANNOTATION LINKING MARK
3191 IDEOGRAPHIC ANNOTATION REVERSE MARK
3192 IDEOGRAPHIC ANNOTATION ONE MARK
3193 IDEOGRAPHIC ANNOTATION TWO MARK
3194 IDEOGRAPHIC ANNOTATION THREE MARK
3195 IDEOGRAPHIC ANNOTATION FOUR MARK
3196 IDEOGRAPHIC ANNOTATION TOP MARK
3197 IDEOGRAPHIC ANNOTATION MIDDLE MARK
3198 IDEOGRAPHIC ANNOTATION BOTTOM MARK
3199 IDEOGRAPHIC ANNOTATION FIRST MARK
319A IDEOGRAPHIC ANNOTATION SECOND MARK
319B IDEOGRAPHIC ANNOTATION THIRD MARK
319C IDEOGRAPHIC ANNOTATION FOURTH MARK
319D IDEOGRAPHIC ANNOTATION HEAVEN MARK
319E IDEOGRAPHIC ANNOTATION EARTH MARK
319F IDEOGRAPHIC ANNOTATION MAN MARK
31A0 BOPOMOFO LETTER BU
31A1 BOPOMOFO LETTER ZI
31A2 BOPOMOFO LETTER JI
31A3 BOPOMOFO LETTER GU
31A4 BOPOMOFO LETTER EE
31A5 BOPOMOFO LETTER ENN
31A6 BOPOMOFO LETTER OO
31A7 BOPOMOFO LETTER ONN
31A8 BOPOMOFO LETTER IR
31A9 BOPOMOFO LETTER ANN
31AA BOPOMOFO LETTER INN
31AB BOPOMOFO LETTER UNN
31AC BOPOMOFO LETTER IM
31AD BOPOMOFO LETTER NGG
31AE BOPOMOFO LETTER AINN
31AF BOPOMOFO LETTER AUNN
31B0 BOPOMOFO LETTER AM
31B1 BOPOMOFO LETTER OM
31B2 BOPOMOFO LETTER ONG
31B3 BOPOMOFO LETTER INNN
31B4 BOPOMOFO FINAL LETTER P
31B5 BOPOMOFO FINAL LETTER T
31B6 BOPOMOFO FINAL LETTER K
31B7 BOPOMOFO FINAL LETTER H
31B8 BOPOMOFO LETTER GH
31B9 BOPOMOFO LETTER LH
31BA BOPOMOFO LETTER ZY
31BB BOPOMOFO FINAL LETTER G
31BC BOPOMOFO LETTER GW
31BD BOPOMOFO LETTER KW
31BE BOPOMOFO LETTER OE
31BF BOPOMOFO LETTER AH
31C0 CJK STROKE T
31C1 CJK STROKE WG
31C2 CJK STROKE XG
31C3 CJK STROKE BXG
31C4 CJK STROKE SW
31C5 CJK STROKE HZZ
31C6 CJK STROKE HZG
31C7 CJK STROKE HP
31C8 CJK STROKE HZWG
31C9 CJK STROKE SZWG
31CA CJK STROKE HZT
31CB CJK STROKE HZZP
31CC CJK STROKE HPWG
31CD CJK STROKE HZW
31CE CJK STROKE HZZZ
31CF CJK STROKE N
31D0 CJK STROKE H
31D1 CJK STROKE S
31D2 CJK STROKE P
31D3 CJK STROKE SP
31D4 CJK STROKE D
31D5 CJK STROKE HZ
31D6 CJK STROKE HG
31D7 CJK STROKE SZ
31D8 CJK STROKE SWZ
31D9 CJK STROKE ST
31DA CJK STROKE SG
31DB CJK STROKE PD
31DC CJK STROKE PZ
31DD CJK STROKE TN
31DE CJK STROKE SZZ
31DF CJK STROKE SWG
31E0 CJK STROKE HXWG
31E1 CJK STROKE HZZZG
31E2 CJK STROKE PG
31E3 CJK STROKE Q
31F0 KATAKANA LETTER SMALL KU
31F1 KATAKANA LETTER SMALL SI
31F2 KATAKANA LETTER SMALL SU
31F3 KATAKANA LETTER SMALL TO
31F4 KATAKANA LETTER SMALL NU
31F5 KATAKANA LETTER SMALL HA
31F6 KATAKANA LETTER SMALL HI
31F7 KATAKANA LETTER SMALL HU
31F8 KATAKANA LETTER SMALL HE
31F9 KATAKANA LETTER SMALL HO
31FA KATAKANA LETTER SMALL MU
31FB KATAKANA LETTER SMALL RA
31FC KATAKANA LETTER SMALL RI
31FD KATAKANA LETTER SMALL RU
31FE KATAKANA LETTER SMALL RE
31FF KATAKANA LETTER SMALL RO
3200 PARENTHESIZED HANGUL KIYEOK
3201 PARENTHESIZED HANGUL NIEUN
3202 PARENTHESIZED HANGUL TIKEUT
3203 PARENTHESIZED HANGUL RIEUL
3204 PARENTHESIZED HANGUL MIEUM
3205 PARENTHESIZED HANGUL PIEUP
3206 PARENTHESIZED HANGUL SIOS
3207 PARENTHESIZED HANGUL IEUNG
3208 PARENTHESIZED HANGUL CIEUC
3209 PARENTHESIZED HANGUL CHIEUCH
320A PARENTHESIZED HANGUL KHIEUKH
320B PARENTHESIZED HANGUL THIEUTH
320C PARENTHESIZED HANGUL PHIEUPH
320D PARENTHESIZED HANGUL HIEUH
320E PARENTHESIZED HANGUL KIYEOK A
320F PARENTHESIZED HANGUL NIEUN A
3210 PARENTHESIZED HANGUL TIKEUT A
3211 PARENTHESIZED HANGUL RIEUL A
3212 PARENTHESIZED HANGUL MIEUM A
3213 PARENTHESIZED HANGUL PIEUP A
3214 PARENTHESIZED HANGUL SIOS A
3215 PARENTHESIZED HANGUL IEUNG A
3216 PARENTHESIZED HANGUL CIEUC A
3217 PARENTHESIZED HANGUL CHIEUCH A
3218 PARENTHESIZED HANGUL KHIEUKH A
3219 PARENTHESIZED HANGUL THIEUTH A
321A PARENTHESIZED HANGUL PHIEUPH A
321B PARENTHESIZED HANGUL HIEUH A
321C PARENTHESIZED HANGUL CIEUC U
321D PARENTHESIZED KOREAN CHARACTER OJEON
321E PARENTHESIZED KOREAN CHARACTER O HU
3220 PARENTHESIZED IDEOGRAPH ONE
3221 PARENTHESIZED IDEOGRAPH TWO
3222 PARENTHESIZED IDEOGRAPH THREE
3223 PARENTHESIZED IDEOGRAPH FOUR
3224 PARENTHESIZED IDEOGRAPH FIVE
3225 PARENTHESIZED IDEOGRAPH SIX
3226 PARENTHESIZED IDEOGRAPH SEVEN
3227 PARENTHESIZED IDEOGRAPH EIGHT
3228 PARENTHESIZED IDEOGRAPH NINE
3229 PARENTHESIZED IDEOGRAPH TEN
322A PARENTHESIZED IDEOGRAPH MOON
322B PARENTHESIZED IDEOGRAPH FIRE
322C PARENTHESIZED IDEOGRAPH WATER
322D PARENTHESIZED IDEOGRAPH WOOD
322E PARENTHESIZED IDEOGRAPH METAL
322F PARENTHESIZED IDEOGRAPH EARTH
3230 PARENTHESIZED IDEOGRAPH SUN
3231 PARENTHESIZED IDEOGRAPH STOCK
3232 PARENTHESIZED IDEOGRAPH HAVE
3233 PARENTHESIZED IDEOGRAPH SOCIETY
3234 PARENTHESIZED IDEOGRAPH NAME
3235 PARENTHESIZED IDEOGRAPH SPECIAL
3236 PARENTHESIZED IDEOGRAPH FINANCIAL
3237 PARENTHESIZED IDEOGRAPH CONGRATULATION
3238 PARENTHESIZED IDEOGRAPH LABOR
3239 PARENTHESIZED IDEOGRAPH REPRESENT
323A PARENTHESIZED IDEOGRAPH CALL
323B PARENTHESIZED IDEOGRAPH STUDY
323C PARENTHESIZED IDEOGRAPH SUPERVISE
323D PARENTHESIZED IDEOGRAPH ENTERPRISE
323E PARENTHESIZED IDEOGRAPH RESOURCE
323F PARENTHESIZED IDEOGRAPH ALLIANCE
3240 PARENTHESIZED IDEOGRAPH FESTIVAL
3241 PARENTHESIZED IDEOGRAPH REST
3242 PARENTHESIZED IDEOGRAPH SELF
3243 PARENTHESIZED IDEOGRAPH REACH
3244 CIRCLED IDEOGRAPH QUESTION
3245 CIRCLED IDEOGRAPH KINDERGARTEN
3246 CIRCLED IDEOGRAPH SCHOOL
3247 CIRCLED IDEOGRAPH KOTO
3248 CIRCLED NUMBER TEN ON BLACK SQUARE
3249 CIRCLED NUMBER TWENTY ON BLACK SQUARE
324A CIRCLED NUMBER THIRTY ON BLACK SQUARE
324B CIRCLED NUMBER FORTY ON BLACK SQUARE
324C CIRCLED NUMBER FIFTY ON BLACK SQUARE
324D CIRCLED NUMBER SIXTY ON BLACK SQUARE
324E CIRCLED NUMBER SEVENTY ON BLACK SQUARE
324F CIRCLED NUMBER EIGHTY ON BLACK SQUARE
3250 PARTNERSHIP SIGN
3251 CIRCLED NUMBER TWENTY ONE
3252 CIRCLED NUMBER TWENTY TWO
3253 CIRCLED NUMBER TWENTY THREE
3254 CIRCLED NUMBER TWENTY FOUR
3255 CIRCLED NUMBER TWENTY FIVE
3256 CIRCLED NUMBER TWENTY SIX
3257 CIRCLED NUMBER TWENTY SEVEN
3258 CIRCLED NUMBER TWENTY EIGHT
3259 CIRCLED NUMBER TWENTY NINE
325A CIRCLED NUMBER THIRTY
325B CIRCLED NUMBER THIRTY ONE
325C CIRCLED NUMBER THIRTY TWO
325D CIRCLED NUMBER THIRTY THREE
325E CIRCLED NUMBER THIRTY FOUR
325F CIRCLED NUMBER THIRTY FIVE
3260 CIRCLED HANGUL KIYEOK
3261 CIRCLED HANGUL NIEUN
3262 CIRCLED HANGUL TIKEUT
3263 CIRCLED HANGUL RIEUL
3264 CIRCLED HANGUL MIEUM
3265 CIRCLED HANGUL PIEUP
3266 CIRCLED HANGUL SIOS
3267 CIRCLED HANGUL IEUNG
3268 CIRCLED HANGUL CIEUC
3269 CIRCLED HANGUL CHIEUCH
326A CIRCLED HANGUL KHIEUKH
326B CIRCLED HANGUL THIEUTH
326C CIRCLED HANGUL PHIEUPH
326D CIRCLED HANGUL HIEUH
326E CIRCLED HANGUL KIYEOK A
326F CIRCLED HANGUL NIEUN A
3270 CIRCLED HANGUL TIKEUT A
3271 CIRCLED HANGUL RIEUL A
3272 CIRCLED HANGUL MIEUM A
3273 CIRCLED HANGUL PIEUP A
3274 CIRCLED HANGUL SIOS A
3275 CIRCLED HANGUL IEUNG A
3276 CIRCLED HANGUL CIEUC A
3277 CIRCLED HANGUL CHIEUCH A
3278 CIRCLED HANGUL KHIEUKH A
3279 CIRCLED HANGUL THIEUTH A
327A CIRCLED HANGUL PHIEUPH A
327B CIRCLED HANGUL HIEUH A
327C CIRCLED KOREAN CHARACTER CHAMKO
327D CIRCLED KOREAN CHARACTER JUEUI
327E CIRCLED HANGUL IEUNG U
327F KOREAN STANDARD SYMBOL
3280 CIRCLED IDEOGRAPH ONE
3281 CIRCLED IDEOGRAPH TWO
3282 CIRCLED IDEOGRAPH THREE
3283 CIRCLED IDEOGRAPH FOUR
3284 CIRCLED IDEOGRAPH FIVE
3285 CIRCLED IDEOGRAPH SIX
3286 CIRCLED IDEOGRAPH SEVEN
3287 CIRCLED IDEOGRAPH EIGHT
3288 CIRCLED IDEOGRAPH NINE
3289 CIRCLED IDEOGRAPH TEN
328A CIRCLED IDEOGRAPH MOON
328B CIRCLED IDEOGRAPH FIRE
328C CIRCLED IDEOGRAPH WATER
328D CIRCLED IDEOGRAPH WOOD
328E CIRCLED IDEOGRAPH METAL
328F CIRCLED IDEOGRAPH EARTH
3290 CIRCLED IDEOGRAPH SUN
3291 CIRCLED IDEOGRAPH STOCK
3292 CIRCLED IDEOGRAPH HAVE
3293 CIRCLED IDEOGRAPH SOCIETY
3294 CIRCLED IDEOGRAPH NAME
3295 CIRCLED IDEOGRAPH SPECIAL
3296 CIRCLED IDEOGRAPH FINANCIAL
3297 CIRCLED IDEOGRAPH CONGRATULATION
3298 CIRCLED IDEOGRAPH LABOR
3299 CIRCLED IDEOGRAPH SECRET
329A CIRCLED IDEOGRAPH MALE
329B CIRCLED IDEOGRAPH FEMALE
329C CIRCLED IDEOGRAPH SUITABLE
329D CIRCLED IDEOGRAPH EXCELLENT
329E CIRCLED IDEOGRAPH PRINT
329F CIRCLED IDEOGRAPH ATTENTION
32A0 CIRCLED IDEOGRAPH ITEM
32A1 CIRCLED IDEOGRAPH REST
32A2 CIRCLED IDEOGRAPH COPY
32A3 CIRCLED IDEOGRAPH CORRECT
32A4 CIRCLED IDEOGRAPH HIGH
32A5 CIRCLED IDEOGRAPH CENTRE
32A6 CIRCLED IDEOGRAPH LOW
32A7 CIRCLED IDEOGRAPH LEFT
32A8 CIRCLED IDEOGRAPH RIGHT
32A9 CIRCLED IDEOGRAPH MEDICINE
32AA CIRCLED IDEOGRAPH RELIGION
32AB CIRCLED IDEOGRAPH STUDY
32AC CIRCLED IDEOGRAPH SUPERVISE
32AD CIRCLED IDEOGRAPH ENTERPRISE
32AE CIRCLED IDEOGRAPH RESOURCE
32AF CIRCLED IDEOGRAPH ALLIANCE
32B0 CIRCLED IDEOGRAPH NIGHT
32B1 CIRCLED NUMBER THIRTY SIX
32B2 CIRCLED NUMBER THIRTY SEVEN
32B3 CIRCLED NUMBER THIRTY EIGHT
32B4 CIRCLED NUMBER THIRTY NINE
32B5 CIRCLED NUMBER FORTY
32B6 CIRCLED NUMBER FORTY ONE
32B7 CIRCLED NUMBER FORTY TWO
32B8 CIRCLED NUMBER FORTY THREE
32B9 CIRCLED NUMBER FORTY FOUR
32BA CIRCLED NUMBER FORTY FIVE
32BB CIRCLED NUMBER FORTY SIX
32BC CIRCLED NUMBER FORTY SEVEN
32BD CIRCLED NUMBER FORTY EIGHT
32BE CIRCLED NUMBER FORTY NINE
32BF CIRCLED NUMBER FIFTY
32C0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY
32C1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARY
32C2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MARCH
32C3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR APRIL
32C4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MAY
32C5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNE
32C6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JULY
32C7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUST
32C8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBER
32C9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR OCTOBER
32CA IDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBER
32CB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBER
32CC SQUARE HG
32CD SQUARE ERG
32CE SQUARE EV
32CF LIMITED LIABILITY SIGN
32D0 CIRCLED KATAKANA A
32D1 CIRCLED KATAKANA I
32D2 CIRCLED KATAKANA U
32D3 CIRCLED KATAKANA E
32D4 CIRCLED KATAKANA O
32D5 CIRCLED KATAKANA KA
32D6 CIRCLED KATAKANA KI
32D7 CIRCLED KATAKANA KU
32D8 CIRCLED KATAKANA KE
32D9 CIRCLED KATAKANA KO
32DA CIRCLED KATAKANA SA
32DB CIRCLED KATAKANA SI
32DC CIRCLED KATAKANA SU
32DD CIRCLED KATAKANA SE
32DE CIRCLED KATAKANA SO
32DF CIRCLED KATAKANA TA
32E0 CIRCLED KATAKANA TI
32E1 CIRCLED KATAKANA TU
32E2 CIRCLED KATAKANA TE
32E3 CIRCLED KATAKANA TO
32E4 CIRCLED KATAKANA NA
32E5 CIRCLED KATAKANA NI
32E6 CIRCLED KATAKANA NU
32E7 CIRCLED KATAKANA NE
32E8 CIRCLED KATAKANA NO
32E9 CIRCLED KATAKANA HA
32EA CIRCLED KATAKANA HI
32EB CIRCLED KATAKANA HU
32EC CIRCLED KATAKANA HE
32ED CIRCLED KATAKANA HO
32EE CIRCLED KATAKANA MA
32EF CIRCLED KATAKANA MI
32F0 CIRCLED KATAKANA MU
32F1 CIRCLED KATAKANA ME
32F2 CIRCLED KATAKANA MO
32F3 CIRCLED KATAKANA YA
32F4 CIRCLED KATAKANA YU
32F5 CIRCLED KATAKANA YO
32F6 CIRCLED KATAKANA RA
32F7 CIRCLED KATAKANA RI
32F8 CIRCLED KATAKANA RU
32F9 CIRCLED KATAKANA RE
32FA CIRCLED KATAKANA RO
32FB CIRCLED KATAKANA WA
32FC CIRCLED KATAKANA WI
32FD CIRCLED KATAKANA WE
32FE CIRCLED KATAKANA WO
32FF SQUARE ERA NAME REIWA
3300 SQUARE APAATO
3301 SQUARE ARUHUA
3302 SQUARE ANPEA
3303 SQUARE AARU
3304 SQUARE ININGU
3305 SQUARE INTI
3306 SQUARE UON
3307 SQUARE ESUKUUDO
3308 SQUARE EEKAA
3309 SQUARE ONSU
330A SQUARE OOMU
330B SQUARE KAIRI
330C SQUARE KARATTO
330D SQUARE KARORII
330E SQUARE GARON
330F SQUARE GANMA
3310 SQUARE GIGA
3311 SQUARE GINII
3312 SQUARE KYURII
3313 SQUARE GIRUDAA
3314 SQUARE KIRO
3315 SQUARE KIROGURAMU
3316 SQUARE KIROMEETORU
3317 SQUARE KIROWATTO
3318 SQUARE GURAMU
3319 SQUARE GURAMUTON
331A SQUARE KURUZEIRO
331B SQUARE KUROONE
331C SQUARE KEESU
331D SQUARE KORUNA
331E SQUARE KOOPO
331F SQUARE SAIKURU
3320 SQUARE SANTIIMU
3321 SQUARE SIRINGU
3322 SQUARE SENTI
3323 SQUARE SENTO
3324 SQUARE DAASU
3325 SQUARE DESI
3326 SQUARE DORU
3327 SQUARE TON
3328 SQUARE NANO
3329 SQUARE NOTTO
332A SQUARE HAITU
332B SQUARE PAASENTO
332C SQUARE PAATU
332D SQUARE BAARERU
332E SQUARE PIASUTORU
332F SQUARE PIKURU
3330 SQUARE PIKO
3331 SQUARE BIRU
3332 SQUARE HUARADDO
3333 SQUARE HUIITO
3334 SQUARE BUSSYERU
3335 SQUARE HURAN
3336 SQUARE HEKUTAARU
3337 SQUARE PESO
3338 SQUARE PENIHI
3339 SQUARE HERUTU
333A SQUARE PENSU
333B SQUARE PEEZI
333C SQUARE BEETA
333D SQUARE POINTO
333E SQUARE BORUTO
333F SQUARE HON
3340 SQUARE PONDO
3341 SQUARE HOORU
3342 SQUARE HOON
3343 SQUARE MAIKURO
3344 SQUARE MAIRU
3345 SQUARE MAHHA
3346 SQUARE MARUKU
3347 SQUARE MANSYON
3348 SQUARE MIKURON
3349 SQUARE MIRI
334A SQUARE MIRIBAARU
334B SQUARE MEGA
334C SQUARE MEGATON
334D SQUARE MEETORU
334E SQUARE YAADO
334F SQUARE YAARU
3350 SQUARE YUAN
3351 SQUARE RITTORU
3352 SQUARE RIRA
3353 SQUARE RUPII
3354 SQUARE RUUBURU
3355 SQUARE REMU
3356 SQUARE RENTOGEN
3357 SQUARE WATTO
3358 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZERO
3359 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONE
335A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWO
335B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREE
335C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOUR
335D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVE
335E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIX
335F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVEN
3360 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHT
3361 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINE
3362 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TEN
3363 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVEN
3364 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVE
3365 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEEN
3366 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURTEEN
3367 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEEN
3368 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEEN
3369 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEEN
336A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEEN
336B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINETEEN
336C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY
336D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONE
336E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWO
336F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREE
3370 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOUR
3371 SQUARE HPA
3372 SQUARE DA
3373 SQUARE AU
3374 SQUARE BAR
3375 SQUARE OV
3376 SQUARE PC
3377 SQUARE DM
3378 SQUARE DM SQUARED
3379 SQUARE DM CUBED
337A SQUARE IU
337B SQUARE ERA NAME HEISEI
337C SQUARE ERA NAME SYOUWA
337D SQUARE ERA NAME TAISYOU
337E SQUARE ERA NAME MEIZI
337F SQUARE CORPORATION
3380 SQUARE PA AMPS
3381 SQUARE NA
3382 SQUARE MU A
3383 SQUARE MA
3384 SQUARE KA
3385 SQUARE KB
3386 SQUARE MB
3387 SQUARE GB
3388 SQUARE CAL
3389 SQUARE KCAL
338A SQUARE PF
338B SQUARE NF
338C SQUARE MU F
338D SQUARE MU G
338E SQUARE MG
338F SQUARE KG
3390 SQUARE HZ
3391 SQUARE KHZ
3392 SQUARE MHZ
3393 SQUARE GHZ
3394 SQUARE THZ
3395 SQUARE MU L
3396 SQUARE ML
3397 SQUARE DL
3398 SQUARE KL
3399 SQUARE FM
339A SQUARE NM
339B SQUARE MU M
339C SQUARE MM
339D SQUARE CM
339E SQUARE KM
339F SQUARE MM SQUARED
33A0 SQUARE CM SQUARED
33A1 SQUARE M SQUARED
33A2 SQUARE KM SQUARED
33A3 SQUARE MM CUBED
33A4 SQUARE CM CUBED
33A5 SQUARE M CUBED
33A6 SQUARE KM CUBED
33A7 SQUARE M OVER S
33A8 SQUARE M OVER S SQUARED
33A9 SQUARE PA
33AA SQUARE KPA
33AB SQUARE MPA
33AC SQUARE GPA
33AD SQUARE RAD
33AE SQUARE RAD OVER S
33AF SQUARE RAD OVER S SQUARED
33B0 SQUARE PS
33B1 SQUARE NS
33B2 SQUARE MU S
33B3 SQUARE MS
33B4 SQUARE PV
33B5 SQUARE NV
33B6 SQUARE MU V
33B7 SQUARE MV
33B8 SQUARE KV
33B9 SQUARE MV MEGA
33BA SQUARE PW
33BB SQUARE NW
33BC SQUARE MU W
33BD SQUARE MW
33BE SQUARE KW
33BF SQUARE MW MEGA
33C0 SQUARE K OHM
33C1 SQUARE M OHM
33C2 SQUARE AM
33C3 SQUARE BQ
33C4 SQUARE CC
33C5 SQUARE CD
33C6 SQUARE C OVER KG
33C7 SQUARE CO
33C8 SQUARE DB
33C9 SQUARE GY
33CA SQUARE HA
33CB SQUARE HP
33CC SQUARE IN
33CD SQUARE KK
33CE SQUARE KM CAPITAL
33CF SQUARE KT
33D0 SQUARE LM
33D1 SQUARE LN
33D2 SQUARE LOG
33D3 SQUARE LX
33D4 SQUARE MB SMALL
33D5 SQUARE MIL
33D6 SQUARE MOL
33D7 SQUARE PH
33D8 SQUARE PM
33D9 SQUARE PPM
33DA SQUARE PR
33DB SQUARE SR
33DC SQUARE SV
33DD SQUARE WB
33DE SQUARE V OVER M
33DF SQUARE A OVER M
33E0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ONE
33E1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWO
33E2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREE
33E3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOUR
33E4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVE
33E5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIX
33E6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVEN
33E7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHT
33E8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINE
33E9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TEN
33EA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVEN
33EB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVE
33EC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTEEN
33ED IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEEN
33EE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEEN
33EF IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEEN
33F0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEEN
33F1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEEN
33F2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINETEEN
33F3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY
33F4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONE
33F5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWO
33F6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREE
33F7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOUR
33F8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVE
33F9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIX
33FA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVEN
33FB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHT
33FC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINE
33FD IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY
33FE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONE
33FF SQUARE GAL
3400 CJK UNIFIED IDEOGRAPH-#3400, IRGKangXi=0078.010, Def{(same as U+4E18 丘) hillock or mound}
3401 CJK UNIFIED IDEOGRAPH-#3401, IRGKangXi=0078.030, Def{to lick; to taste, a mat, bamboo bark}
3402 CJK UNIFIED IDEOGRAPH-#3402, IRGKangXi=0078.101, Def{(J) non-standard form of U+559C 喜, to like, love, enjoy; a joyful thing}
3403 CJK UNIFIED IDEOGRAPH-#3403, IRGKangXi=0079.021
3404 CJK UNIFIED IDEOGRAPH-#3404, IRGKangXi=0079.020
3405 CJK UNIFIED IDEOGRAPH-#3405, IRGKangXi=0081.180, Def{(an ancient form of U+4E94 五) five}
3406 CJK UNIFIED IDEOGRAPH-#3406, IRGKangXi=0083.011, Def{(corrupted form) to follow, to trust to; to put confidence in; to depend on, to turn around; to turn the body, (interchangeable 隱)}
3407 CJK UNIFIED IDEOGRAPH-#3407, IRGKangXi=0084.051
3408 CJK UNIFIED IDEOGRAPH-#3408, IRGKangXi=0084.051
3409 CJK UNIFIED IDEOGRAPH-#3409, IRGKangXi=0084.051
340A CJK UNIFIED IDEOGRAPH-#340A, IRGKangXi=0084.071
340B CJK UNIFIED IDEOGRAPH-#340B, IRGKangXi=0084.071
340C CJK UNIFIED IDEOGRAPH-#340C, IRGKangXi=0084.080, Def{a tribe of savages in South China}
340D CJK UNIFIED IDEOGRAPH-#340D, IRGKangXi=0084.101
340E CJK UNIFIED IDEOGRAPH-#340E, IRGKangXi=0084.101
340F CJK UNIFIED IDEOGRAPH-#340F, IRGKangXi=0084.101
3410 CJK UNIFIED IDEOGRAPH-#3410, IRGKangXi=0084.141
3411 CJK UNIFIED IDEOGRAPH-#3411, IRGKangXi=0084.141
3412 CJK UNIFIED IDEOGRAPH-#3412, IRGKangXi=0084.141
3413 CJK UNIFIED IDEOGRAPH-#3413, IRGKangXi=0084.141
3414 CJK UNIFIED IDEOGRAPH-#3414, IRGKangXi=0084.141
3415 CJK UNIFIED IDEOGRAPH-#3415, IRGKangXi=0084.141
3416 CJK UNIFIED IDEOGRAPH-#3416, IRGKangXi=0084.160, Def{㐖毒, an old name for India}
3417 CJK UNIFIED IDEOGRAPH-#3417, IRGKangXi=0084.161
3418 CJK UNIFIED IDEOGRAPH-#3418, IRGKangXi=0084.161
3419 CJK UNIFIED IDEOGRAPH-#3419, IRGKangXi=0084.201
341A CJK UNIFIED IDEOGRAPH-#341A, IRGKangXi=0084.201
341B CJK UNIFIED IDEOGRAPH-#341B, IRGKangXi=0084.201
341C CJK UNIFIED IDEOGRAPH-#341C, IRGKangXi=0084.220, Def{(same as 仇) an enemy, enmity, hatred, to hate, a rival, a match}
341D CJK UNIFIED IDEOGRAPH-#341D, IRGKangXi=0084.241
341E CJK UNIFIED IDEOGRAPH-#341E, IRGKangXi=0084.241
341F CJK UNIFIED IDEOGRAPH-#341F, IRGKangXi=0084.241
3420 CJK UNIFIED IDEOGRAPH-#3420, IRGKangXi=0084.241
3421 CJK UNIFIED IDEOGRAPH-#3421, IRGKangXi=0084.310, Def{(same as 懦) weak; timid; imbecile}
3422 CJK UNIFIED IDEOGRAPH-#3422, IRGKangXi=0084.321
3423 CJK UNIFIED IDEOGRAPH-#3423, IRGKangXi=0084.321
3424 CJK UNIFIED IDEOGRAPH-#3424, IRGKangXi=0084.321, Def{to implore; to beseech, to seek after, to beg; to pray}
3425 CJK UNIFIED IDEOGRAPH-#3425, IRGKangXi=0085.061
3426 CJK UNIFIED IDEOGRAPH-#3426, IRGKangXi=0085.071
3427 CJK UNIFIED IDEOGRAPH-#3427, IRGKangXi=0085.191, Def{(J) non-standard form of 第 U+7B2C, sequence, number; grade, degree}
3428 CJK UNIFIED IDEOGRAPH-#3428, IRGKangXi=0085.230, Def{a kind of fish in legend (a record in old books)}
3429 CJK UNIFIED IDEOGRAPH-#3429, IRGKangXi=0087.141, Def{wine cups}
342A CJK UNIFIED IDEOGRAPH-#342A, IRGKangXi=0088.111
342B CJK UNIFIED IDEOGRAPH-#342B, IRGKangXi=0088.111, Def{(same as 凶) cruel, unfortunate, sad}
342C CJK UNIFIED IDEOGRAPH-#342C, IRGKangXi=0088.150, Def{(same as U+65D2 旒, a corrupted form of U+8352 荒) a cup with pendants, a pennant, wild, barren, uncultivated}
342D CJK UNIFIED IDEOGRAPH-#342D, IRGKangXi=0089.010, Def{(same as 廩) a granary, to supply (foodstuff), to stockpile}
342E CJK UNIFIED IDEOGRAPH-#342E, IRGKangXi=0089.201, Def{(same as 襄) to help; to assist, to achieve, to rise; to raise}
342F CJK UNIFIED IDEOGRAPH-#342F, IRGKangXi=0089.241, Def{(ancient form of 庸) to employ; to use, to manifest}
3430 CJK UNIFIED IDEOGRAPH-#3430, IRGKangXi=0092.080, Def{(ancient form of 信) to believe in; to trust, truth, sincerity, confidence, a pledge or token}
3431 CJK UNIFIED IDEOGRAPH-#3431, IRGKangXi=0092.110, Def{(same as 鬒) bushy, black hair}
3432 CJK UNIFIED IDEOGRAPH-#3432, IRGKangXi=0093.080, Def{name of an island}
3433 CJK UNIFIED IDEOGRAPH-#3433, IRGKangXi=0094.030, Def{high and level on the top}
3434 CJK UNIFIED IDEOGRAPH-#3434, IRGKangXi=0094.050, Def{(non-classical of 攀) to drag down; to seize, to pull, to hold to}
3435 CJK UNIFIED IDEOGRAPH-#3435, IRGKangXi=0094.051, Def{Confucianism; scholar}
3436 CJK UNIFIED IDEOGRAPH-#3436, IRGKangXi=0094.051
3437 CJK UNIFIED IDEOGRAPH-#3437, IRGKangXi=0094.051, Def{a kind of punishment in Han Dynasty, name of chessmen in Chinese chess game (simplified form, a variant 罵) to curse; to revile; to abuse, to scold}
3438 CJK UNIFIED IDEOGRAPH-#3438, IRGKangXi=0094.100, Def{(non-classical form of 欠) to owe money, deficient, to yawn, last name}
3439 CJK UNIFIED IDEOGRAPH-#3439, IRGKangXi=0094.140, Def{(standard form of 仡) strong; valiant, a minority ethnic group in China}
343A CJK UNIFIED IDEOGRAPH-#343A, IRGKangXi=0095.220, Def{(standard form of 眾) all; the whole of; a multitude, a crowd (three or more)}
343B CJK UNIFIED IDEOGRAPH-#343B, IRGKangXi=0096.090, Def{name of a person}
343C CJK UNIFIED IDEOGRAPH-#343C, IRGKangXi=0096.110, Def{(corrupted form of 拯) to save; to lift up}
343D CJK UNIFIED IDEOGRAPH-#343D, IRGKangXi=0096.121, Def{(simplified form of 偑) name of a place, last name}
343E CJK UNIFIED IDEOGRAPH-#343E, IRGKangXi=0096.121
343F CJK UNIFIED IDEOGRAPH-#343F, IRGKangXi=0096.121
3440 CJK UNIFIED IDEOGRAPH-#3440, IRGKangXi=0096.121
3441 CJK UNIFIED IDEOGRAPH-#3441, IRGKangXi=0096.150, Def{(same as U+20B74 𠭴) short; of short stature}
3442 CJK UNIFIED IDEOGRAPH-#3442, IRGKangXi=0096.190, Def{(same as 仿) to imitate, like; resembling, according to}
3443 CJK UNIFIED IDEOGRAPH-#3443, IRGKangXi=0097.050, Def{(same as 拗) to pull; to drag, to break off, to pluck, as a flower}
3444 CJK UNIFIED IDEOGRAPH-#3444, IRGKangXi=0098.020, Def{(ancient form of 侮) to insult, to ridicule}
3445 CJK UNIFIED IDEOGRAPH-#3445, IRGKangXi=0100.021, Def{(same as 作) to make; to do; to act, to write; to compose; to rise, work}
3446 CJK UNIFIED IDEOGRAPH-#3446, IRGKangXi=0100.021
3447 CJK UNIFIED IDEOGRAPH-#3447, IRGKangXi=0100.021, Def{(a simplified form) clever; ingenious; cute; pretty}
3448 CJK UNIFIED IDEOGRAPH-#3448, IRGKangXi=0100.021, Def{(simplified form) rude; barbarous, stupid; dull, last name}
3449 CJK UNIFIED IDEOGRAPH-#3449, IRGKangXi=0100.050, Def{cannot straighten up}
344A CJK UNIFIED IDEOGRAPH-#344A, IRGKangXi=0100.200, Def{a kind of disease}
344B CJK UNIFIED IDEOGRAPH-#344B, IRGKangXi=0100.260, Def{small, poor, submit to the dominion of; slow; late}
344C CJK UNIFIED IDEOGRAPH-#344C, IRGKangXi=0101.210, Def{(same as 尪) weak; a rickety person; emaciated}
344D CJK UNIFIED IDEOGRAPH-#344D, IRGKangXi=0102.110, Def{inferior; secondary, ugly}
344E CJK UNIFIED IDEOGRAPH-#344E, IRGKangXi=0102.190, Def{last name}
344F CJK UNIFIED IDEOGRAPH-#344F, IRGKangXi=0102.230, Def{(corrupted form of U+4F47 佇) to stand and wait; to hope for}
3450 CJK UNIFIED IDEOGRAPH-#3450, IRGKangXi=0102.291, Def{to move; to start; to shake, name of a person}
3451 CJK UNIFIED IDEOGRAPH-#3451, IRGKangXi=0102.291
3452 CJK UNIFIED IDEOGRAPH-#3452, IRGKangXi=0102.291
3453 CJK UNIFIED IDEOGRAPH-#3453, IRGKangXi=0102.291
3454 CJK UNIFIED IDEOGRAPH-#3454, IRGKangXi=0102.291, Def{(a simplified form) (corrupted form of 偦) all together, mutually, last name}
3455 CJK UNIFIED IDEOGRAPH-#3455, IRGKangXi=0102.291
3456 CJK UNIFIED IDEOGRAPH-#3456, IRGKangXi=0102.291
3457 CJK UNIFIED IDEOGRAPH-#3457, IRGKangXi=0103.050, Def{a god, pregnant}
3458 CJK UNIFIED IDEOGRAPH-#3458, IRGKangXi=0103.120, Def{(same as 价) a servant, a middle-man, good, great; (a simplified form) price; value}
3459 CJK UNIFIED IDEOGRAPH-#3459, IRGKangXi=0103.240, Def{frivolous; flippant; capricious; playful}
345A CJK UNIFIED IDEOGRAPH-#345A, IRGKangXi=0103.290, Def{(non-classical form of 那) that, there}
345B CJK UNIFIED IDEOGRAPH-#345B, IRGKangXi=0104.120, Def{to shake one's head}
345C CJK UNIFIED IDEOGRAPH-#345C, IRGKangXi=0105.010, Def{(a dialect) to engrave}
345D CJK UNIFIED IDEOGRAPH-#345D, IRGKangXi=0105.060, Def{to make a fool of; idiotic, simple, stupid}
345E CJK UNIFIED IDEOGRAPH-#345E, IRGKangXi=0105.070, Def{(same as 媵) (in old time) a maid who accompanies a bride to her new home; to escort, a concubine}
345F CJK UNIFIED IDEOGRAPH-#345F, IRGKangXi=0105.220, Def{insincere and cunning person; a pretentious person}
3460 CJK UNIFIED IDEOGRAPH-#3460, IRGKangXi=0106.041
3461 CJK UNIFIED IDEOGRAPH-#3461, IRGKangXi=0106.041
3462 CJK UNIFIED IDEOGRAPH-#3462, IRGKangXi=0106.041
3463 CJK UNIFIED IDEOGRAPH-#3463, IRGKangXi=0106.130, Def{stupid; loutish, without ability; unable; lacking power}
3464 CJK UNIFIED IDEOGRAPH-#3464, IRGKangXi=0106.150, Def{(same as 媌) good looking, a prostitute}
3465 CJK UNIFIED IDEOGRAPH-#3465, IRGKangXi=0106.160, Def{disrespectful; irreverent, to make light of; to neglect; careless; rush, to exchange, (said of one's personality) easy to get along with}
3466 CJK UNIFIED IDEOGRAPH-#3466, IRGKangXi=0107.240, Def{anger; rage; angry, (same as 戾) recalcitrant; stubbornly persisting in doing something wrong; cruel; despotic}
3467 CJK UNIFIED IDEOGRAPH-#3467, IRGKangXi=0109.171, Def{(corrupted form of 悸) perturbed, to throb, palpitation of the heart}
3468 CJK UNIFIED IDEOGRAPH-#3468, IRGKangXi=0109.171, Def{great; big; tall; vast, noble; high in rank, very; much}
3469 CJK UNIFIED IDEOGRAPH-#3469, IRGKangXi=0109.171, Def{(simplified form of U+5138 儸) smart; clever}
346A CJK UNIFIED IDEOGRAPH-#346A, IRGKangXi=0109.171, Def{(non-classical form of 儕) a generation, a class; a series; a kind}
346B CJK UNIFIED IDEOGRAPH-#346B, IRGKangXi=0109.171
346C CJK UNIFIED IDEOGRAPH-#346C, IRGKangXi=0109.171
346D CJK UNIFIED IDEOGRAPH-#346D, IRGKangXi=0109.171
346E CJK UNIFIED IDEOGRAPH-#346E, IRGKangXi=0110.100, Def{last name, girl's name}
346F CJK UNIFIED IDEOGRAPH-#346F, IRGKangXi=0111.130, Def{last name, all; together; mutually, a low rank officer to take charge of the bandits in ancient time}
3470 CJK UNIFIED IDEOGRAPH-#3470, IRGKangXi=0111.160, Def{(a corrupted form) distress; very difficulty; anxiety; very tired}
3471 CJK UNIFIED IDEOGRAPH-#3471, IRGKangXi=0112.071, Def{a kind of monkey, fingering for an ancient string instrument}
3472 CJK UNIFIED IDEOGRAPH-#3472, IRGKangXi=0112.071
3473 CJK UNIFIED IDEOGRAPH-#3473, IRGKangXi=0112.100, Def{(same as 媰) pregnant (妊娠, 妊身); cruel; pretty, cute, clever, ingenious, smart; to be hired; (used for 謅) to jest, to chaff, to bawl, mean person (as opposed to real gentleman)}
3474 CJK UNIFIED IDEOGRAPH-#3474, IRGKangXi=0112.120, Def{(standard form of 侵) to usurp, to encroach upon, to raid}
3475 CJK UNIFIED IDEOGRAPH-#3475, IRGKangXi=0113.030, Def{(same as 嫉) envy; jealousy, to hate}
3476 CJK UNIFIED IDEOGRAPH-#3476, IRGKangXi=0113.040, Def{(non-classical form of 愾) anger; passion; rage, to sigh; to groan}
3477 CJK UNIFIED IDEOGRAPH-#3477, IRGKangXi=0113.060, Def{(same as 態) manner, bearing, behaviour, policy, attitude}
3478 CJK UNIFIED IDEOGRAPH-#3478, IRGKangXi=0114.051
3479 CJK UNIFIED IDEOGRAPH-#3479, IRGKangXi=0114.051, Def{(non-classical of 會) to meet; to assemble, to co-operate, a society; a guild; an association}
347A CJK UNIFIED IDEOGRAPH-#347A, IRGKangXi=0114.051, Def{(a variant of 俊) superior, handsome, refined, eminent}
347B CJK UNIFIED IDEOGRAPH-#347B, IRGKangXi=0114.130, Def{strong, robust, vigorous}
347C CJK UNIFIED IDEOGRAPH-#347C, IRGKangXi=0114.210, Def{name of a god}
347D CJK UNIFIED IDEOGRAPH-#347D, IRGKangXi=0114.270, Def{(same as 傏) to ward off; to parry; to keep out, as wind, rain, or cold}
347E CJK UNIFIED IDEOGRAPH-#347E, IRGKangXi=0114.290, Def{(interchangeable 繇) joy, delight, gratification}
347F CJK UNIFIED IDEOGRAPH-#347F, IRGKangXi=0115.130, Def{long, of space or time, profitable, excelling}
3480 CJK UNIFIED IDEOGRAPH-#3480, IRGKangXi=0115.150, Def{do not fear to}
3481 CJK UNIFIED IDEOGRAPH-#3481, IRGKangXi=0115.220, Def{(same as 俁) of great stature; stalwart, to injure, to grieve}
3482 CJK UNIFIED IDEOGRAPH-#3482, IRGKangXi=0118.230, Def{to bestow; to grant, to act; to do, quiet; still, peace; tranquility}
3483 CJK UNIFIED IDEOGRAPH-#3483, IRGKangXi=0116.090, Def{(same as 貳) capital form of two, a second job, to harbour doubts; to hesitate, to revolt}
3484 CJK UNIFIED IDEOGRAPH-#3484, IRGKangXi=0116.240, Def{brittle and friable, fear; dread; fright; scare}
3485 CJK UNIFIED IDEOGRAPH-#3485, IRGKangXi=0117.060, Def{to unbind the collar}
3486 CJK UNIFIED IDEOGRAPH-#3486, IRGKangXi=0117.110, Def{(same as 歙) to gather the harvest; to gather together}
3487 CJK UNIFIED IDEOGRAPH-#3487, IRGKangXi=0117.130, Def{(same as 憮) to skip about; to dance for joy; to soothe, slow, disappointed}
3488 CJK UNIFIED IDEOGRAPH-#3488, IRGKangXi=0117.150, Def{dangerous; lofty; steep; high and dangerous}
3489 CJK UNIFIED IDEOGRAPH-#3489, IRGKangXi=0117.220, Def{(same as 惝) alarmed; agitated}
348A CJK UNIFIED IDEOGRAPH-#348A, IRGKangXi=0117.250, Def{loquacious}
348B CJK UNIFIED IDEOGRAPH-#348B, IRGKangXi=0117.280, Def{a servant; a menial, a woodcutter}
348C CJK UNIFIED IDEOGRAPH-#348C, IRGKangXi=0118.030, Def{(ancient form of 煢) along; desolate; orphaned}
348D CJK UNIFIED IDEOGRAPH-#348D, IRGKangXi=0118.051, Def{(interchangeable U+50AB 儽) utterly weary in body and spirits; negligent; lax}
348E CJK UNIFIED IDEOGRAPH-#348E, IRGKangXi=0118.051, Def{evil, wicked, wrong, foul}
348F CJK UNIFIED IDEOGRAPH-#348F, IRGKangXi=0118.051
3490 CJK UNIFIED IDEOGRAPH-#3490, IRGKangXi=0118.051
3491 CJK UNIFIED IDEOGRAPH-#3491, IRGKangXi=0118.120, Def{grow up, commodity prices}
3492 CJK UNIFIED IDEOGRAPH-#3492, IRGKangXi=0118.220, Def{(same as 僕) a slave; a servant, used conventionally for oneself, a charioteer}
3493 CJK UNIFIED IDEOGRAPH-#3493, IRGKangXi=0119.010, Def{to escape; to abscond; to flee, (interchangeable 達)}
3494 CJK UNIFIED IDEOGRAPH-#3494, IRGKangXi=0119.070, Def{to shake one's head, ugly, not in peace}
3495 CJK UNIFIED IDEOGRAPH-#3495, IRGKangXi=0119.140, Def{unyield}
3496 CJK UNIFIED IDEOGRAPH-#3496, IRGKangXi=0119.231, Def{(same as 藕) the root-stock of the lotus; arrowroot}
3497 CJK UNIFIED IDEOGRAPH-#3497, IRGKangXi=0119.231, Def{(a variant of 儓) a servant}
3498 CJK UNIFIED IDEOGRAPH-#3498, IRGKangXi=0119.231
3499 CJK UNIFIED IDEOGRAPH-#3499, IRGKangXi=0119.250, Def{to hand or bow the head; to droop; to lower, low; beneath}
349A CJK UNIFIED IDEOGRAPH-#349A, IRGKangXi=0119.320, Def{(same as U+7A69 穩) firm; stable; secure, dependent upon others}
349B CJK UNIFIED IDEOGRAPH-#349B, IRGKangXi=0119.360, Def{to stand alone; independent}
349C CJK UNIFIED IDEOGRAPH-#349C, IRGKangXi=0120.040, Def{respectful; attentive, carefully; cautious; to heed; to be watchful, to rely on; to trust to; to lean towards}
349D CJK UNIFIED IDEOGRAPH-#349D, IRGKangXi=0120.150, Def{to deceive; artful; false}
349E CJK UNIFIED IDEOGRAPH-#349E, IRGKangXi=0120.190, Def{valiant; brave, eminent}
349F CJK UNIFIED IDEOGRAPH-#349F, IRGKangXi=0121.070, Def{(same as 褭) with a charming; slim carriage (of a woman), pretty}
34A0 CJK UNIFIED IDEOGRAPH-#34A0, IRGKangXi=0121.140, Def{narrow; contracted, quickly; fast; hasty; soon; promptly}
34A1 CJK UNIFIED IDEOGRAPH-#34A1, IRGKangXi=0121.291, Def{name of a person}
34A2 CJK UNIFIED IDEOGRAPH-#34A2, IRGKangXi=0119.231
34A3 CJK UNIFIED IDEOGRAPH-#34A3, IRGKangXi=0121.291
34A4 CJK UNIFIED IDEOGRAPH-#34A4, IRGKangXi=0121.310, Def{to have one's heart won; to submit, admire, etc. sincerely and willingly, (interchangeable 懾) to fear; to dread; to be scared of}
34A5 CJK UNIFIED IDEOGRAPH-#34A5, IRGKangXi=0121.330, Def{an immortal}
34A6 CJK UNIFIED IDEOGRAPH-#34A6, IRGKangXi=0121.340, Def{(same as 儡) puppets, very tired; fatigued; weary and weak}
34A7 CJK UNIFIED IDEOGRAPH-#34A7, IRGKangXi=0121.370, Def{(same as 儷) a pair; a couple, luxuriant; lush; exuberant}
34A8 CJK UNIFIED IDEOGRAPH-#34A8, IRGKangXi=0122.061, Def{(ancient form of 仙) an immortal; a fairy; a genie}
34A9 CJK UNIFIED IDEOGRAPH-#34A9, IRGKangXi=0122.100, Def{(same as 梴 裸) naked, to strip; to unclothe}
34AA CJK UNIFIED IDEOGRAPH-#34AA, IRGKangXi=0122.120, Def{(ancient form of 錫) tin; pewter}
34AB CJK UNIFIED IDEOGRAPH-#34AB, IRGKangXi=0124.140, Def{(ancient form of 旡) choked and unable to breath}
34AC CJK UNIFIED IDEOGRAPH-#34AC, IRGKangXi=0125.131
34AD CJK UNIFIED IDEOGRAPH-#34AD, IRGKangXi=0125.190, Def{(same as 昆) an elder brother}
34AE CJK UNIFIED IDEOGRAPH-#34AE, IRGKangXi=0125.201
34AF CJK UNIFIED IDEOGRAPH-#34AF, IRGKangXi=0125.291
34B0 CJK UNIFIED IDEOGRAPH-#34B0, IRGKangXi=0126.030, Def{(a variant, seal type of U+5168 全) perfect, complete, absolute}
34B1 CJK UNIFIED IDEOGRAPH-#34B1, IRGKangXi=0126.041
34B2 CJK UNIFIED IDEOGRAPH-#34B2, IRGKangXi=0126.080, Def{(a variant of 財) wealth; property; valuables, bribes}
34B3 CJK UNIFIED IDEOGRAPH-#34B3, IRGKangXi=0126.100, Def{(ancient form of 兩) two, a pair; a couple; both}
34B4 CJK UNIFIED IDEOGRAPH-#34B4, IRGKangXi=0126.120, Def{to come; to go out}
34B5 CJK UNIFIED IDEOGRAPH-#34B5, IRGKangXi=0127.160, Def{(same as 貌) manner; appearance; form; face; bearing}
34B6 CJK UNIFIED IDEOGRAPH-#34B6, IRGKangXi=0127.170, Def{(ancient form of 公) public; open to all}
34B7 CJK UNIFIED IDEOGRAPH-#34B7, IRGKangXi=0127.171, Def{(simple form of U+8208 興) to prosper, to begin, to increase; to rise; to raise, flourishing}
34B8 CJK UNIFIED IDEOGRAPH-#34B8, IRGKangXi=0128.050, Def{(same as U+9042 遂) to obey; to comply with; to follow the wishes of another, (an ancient form of U+6B72 歲) a year, age, the harvest}
34B9 CJK UNIFIED IDEOGRAPH-#34B9, IRGKangXi=0128.280, Def{(non-classical of 顛) to upset; to turn over, to fall, upside down}
34BA CJK UNIFIED IDEOGRAPH-#34BA, IRGKangXi=0129.191
34BB CJK UNIFIED IDEOGRAPH-#34BB, IRGKangXi=0129.290, Def{(same as 冒) to go forward with eyes covered, to risk, to rush upon, to put forth, to issue forth}
34BC CJK UNIFIED IDEOGRAPH-#34BC, IRGKangXi=0129.320, Def{average; equivalent; corresponding, to cover something carefully and tightly without a break; (Cant.) blocked}
34BD CJK UNIFIED IDEOGRAPH-#34BD, IRGKangXi=0129.350, Def{a kind of cap for children}
34BE CJK UNIFIED IDEOGRAPH-#34BE, IRGKangXi=0130.020, Def{a kind of wrapper used to cover over the face and head}
34BF CJK UNIFIED IDEOGRAPH-#34BF, IRGKangXi=0130.100, Def{a kind of turban used in ancient time}
34C0 CJK UNIFIED IDEOGRAPH-#34C0, IRGKangXi=0130.151
34C1 CJK UNIFIED IDEOGRAPH-#34C1, IRGKangXi=0130.151, Def{a net; net-like, radical 122}
34C2 CJK UNIFIED IDEOGRAPH-#34C2, IRGKangXi=0130.231, Def{(a variant of 寇) tyrannical; cruel, to rob; to plunder, banditti; thieves; highwaymen}
34C3 CJK UNIFIED IDEOGRAPH-#34C3, IRGKangXi=0131.040, Def{(standard form of U+8A6B 詫) to sacrifice by pouring out a libation of wine}
34C4 CJK UNIFIED IDEOGRAPH-#34C4, IRGKangXi=0131.080, Def{to plough; to till, black color}
34C5 CJK UNIFIED IDEOGRAPH-#34C5, IRGKangXi=0131.160, Def{frozen; cold}
34C6 CJK UNIFIED IDEOGRAPH-#34C6, IRGKangXi=0131.291
34C7 CJK UNIFIED IDEOGRAPH-#34C7, IRGKangXi=0131.291
34C8 CJK UNIFIED IDEOGRAPH-#34C8, IRGKangXi=0132.040, Def{sound of flying or speeding}
34C9 CJK UNIFIED IDEOGRAPH-#34C9, IRGKangXi=0132.080, Def{ice; icicles, cold; frost, crystallized}
34CA CJK UNIFIED IDEOGRAPH-#34CA, IRGKangXi=0132.090, Def{cold, indifferent}
34CB CJK UNIFIED IDEOGRAPH-#34CB, IRGKangXi=0132.130, Def{to freeze; to congeal; to curdle}
34CC CJK UNIFIED IDEOGRAPH-#34CC, IRGKangXi=0132.190, Def{bitterly cold}
34CD CJK UNIFIED IDEOGRAPH-#34CD, IRGKangXi=0132.201
34CE CJK UNIFIED IDEOGRAPH-#34CE, IRGKangXi=0132.220, Def{cold, cold air, bitterly cold}
34CF CJK UNIFIED IDEOGRAPH-#34CF, IRGKangXi=0132.290, Def{bitterly cold}
34D0 CJK UNIFIED IDEOGRAPH-#34D0, IRGKangXi=0132.340, Def{to rain, cold; wintry}
34D1 CJK UNIFIED IDEOGRAPH-#34D1, IRGKangXi=0132.360, Def{cold}
34D2 CJK UNIFIED IDEOGRAPH-#34D2, IRGKangXi=0133.011
34D3 CJK UNIFIED IDEOGRAPH-#34D3, IRGKangXi=0133.070, Def{thin ice}
34D4 CJK UNIFIED IDEOGRAPH-#34D4, IRGKangXi=0133.120, Def{(same as 謝) faded and fallen -- as blossoms; withered}
34D5 CJK UNIFIED IDEOGRAPH-#34D5, IRGKangXi=0133.211
34D6 CJK UNIFIED IDEOGRAPH-#34D6, IRGKangXi=0133.280, Def{bitterly cold, cold wind; the north wind}
34D7 CJK UNIFIED IDEOGRAPH-#34D7, IRGKangXi=0133.330, Def{(non-classical of 潔) clean; pure; clear}
34D8 CJK UNIFIED IDEOGRAPH-#34D8, IRGKangXi=0134.301, Def{(same as U+738A 玊) jade with some defects, a lapidary, to polish gems; a surname}
34D9 CJK UNIFIED IDEOGRAPH-#34D9, IRGKangXi=0135.151, Def{(non-classical of 凶) cruel, unfortunate, sad}
34DA CJK UNIFIED IDEOGRAPH-#34DA, IRGKangXi=0136.150, Def{a sickle, a reaping-hook, to gather in the harvest, to pare; to scrape}
34DB CJK UNIFIED IDEOGRAPH-#34DB, IRGKangXi=0137.021, Def{(incorrect form of 功) merit; achievement; meritorious, efficacy; good results}
34DC CJK UNIFIED IDEOGRAPH-#34DC, IRGKangXi=0342.041, Def{(corrupted form of 幼) young; immature; weak; delicate}
34DD CJK UNIFIED IDEOGRAPH-#34DD, IRGKangXi=0137.120, Def{(same as 刑) a law, to punish; punishment}
34DE CJK UNIFIED IDEOGRAPH-#34DE, IRGKangXi=0137.150, Def{to engrave, (interchangeable with U+5951 契) a written contract or agreement}
34DF CJK UNIFIED IDEOGRAPH-#34DF, IRGKangXi=0137.360, Def{to peel, to pare, to trim, to split, (non-classical form) to open; to unroll; to spread out}
34E0 CJK UNIFIED IDEOGRAPH-#34E0, IRGKangXi=0138.180, Def{(same as 玷) a flaw; a defect, to cut with a sword; to chop}
34E1 CJK UNIFIED IDEOGRAPH-#34E1, IRGKangXi=0138.380, Def{(same as 弣) the middle part of a bow; the handle of a bow, try to dig and get something with a knife}
34E2 CJK UNIFIED IDEOGRAPH-#34E2, IRGKangXi=0138.450, Def{to cut the flesh from the bones; to scrape off; to pick out; to get rid of}
34E3 CJK UNIFIED IDEOGRAPH-#34E3, IRGKangXi=0139.080, Def{to sink into; to pierce, (ancient form of 割) to cut; to hack; to reap}
34E4 CJK UNIFIED IDEOGRAPH-#34E4, IRGKangXi=0139.180, Def{to strip the skin of the face; an imminent calamity, to engrave; (Cant.) to pierce}
34E5 CJK UNIFIED IDEOGRAPH-#34E5, IRGKangXi=0139.251, Def{(dial.) to kill; to slaughter}
34E6 CJK UNIFIED IDEOGRAPH-#34E6, IRGKangXi=0139.251, Def{to deliberate and plan; to arrange}
34E7 CJK UNIFIED IDEOGRAPH-#34E7, IRGKangXi=0140.200, Def{agricultural implements}
34E8 CJK UNIFIED IDEOGRAPH-#34E8, IRGKangXi=0140.210, Def{(same as 刺) to pierce; to stab; to irritate; to hurt}
34E9 CJK UNIFIED IDEOGRAPH-#34E9, IRGKangXi=0140.220, Def{horned hair}
34EA CJK UNIFIED IDEOGRAPH-#34EA, IRGKangXi=0140.290, Def{(same as 朗) bright, clear, intelligent, light, brilliant, to understand, to illustrate, to cleanse}
34EB CJK UNIFIED IDEOGRAPH-#34EB, IRGKangXi=0140.291
34EC CJK UNIFIED IDEOGRAPH-#34EC, IRGKangXi=0141.150, Def{(same as 刳) to cut open; to rip up; to cut out}
34ED CJK UNIFIED IDEOGRAPH-#34ED, IRGKangXi=0141.240, Def{(same as U+97A8 鞨) to make ready the leather for shoes}
34EE CJK UNIFIED IDEOGRAPH-#34EE, IRGKangXi=0141.330, Def{(same as 雕) to engrave; to carve; to tattoo}
34EF CJK UNIFIED IDEOGRAPH-#34EF, IRGKangXi=0141.340, Def{(same as 劙) to divide, to partition, to cut; to hack; to reap}
34F0 CJK UNIFIED IDEOGRAPH-#34F0, IRGKangXi=0141.411, Def{(a variant of 劃) to rive; to divide; to mark; to cut}
34F1 CJK UNIFIED IDEOGRAPH-#34F1, IRGKangXi=0142.050, Def{to cut; to cut out; to pick out; to scoop out, (same as 鄃) name of a county in today's Shandong province}
34F2 CJK UNIFIED IDEOGRAPH-#34F2, IRGKangXi=0142.090, Def{to pare; to cut into slices}
34F3 CJK UNIFIED IDEOGRAPH-#34F3, IRGKangXi=0142.100, Def{use knife to split something}
34F4 CJK UNIFIED IDEOGRAPH-#34F4, IRGKangXi=0142.120, Def{to cut in two, to slash}
34F5 CJK UNIFIED IDEOGRAPH-#34F5, IRGKangXi=0142.210, Def{(same as 鍔) swords; a double-edged sword}
34F6 CJK UNIFIED IDEOGRAPH-#34F6, IRGKangXi=0142.220, Def{(same as 鍥) to carve; to engrave; to cut; to oppress, a sickle}
34F7 CJK UNIFIED IDEOGRAPH-#34F7, IRGKangXi=0142.290, Def{(same as 劓) to cut off the nose; cut off}
34F8 CJK UNIFIED IDEOGRAPH-#34F8, IRGKangXi=0142.370, Def{(same as 斲) to cut to pieces; to hack; to chop or hew, to carve for ornaments}
34F9 CJK UNIFIED IDEOGRAPH-#34F9, IRGKangXi=0142.390, Def{(same as 銳) a sharp-pointed weapon, acute; zealous; valiant, wound, minor injury}
34FA CJK UNIFIED IDEOGRAPH-#34FA, IRGKangXi=0142.410, Def{castration, to pare; to scrape; to sharpen to a point}
34FB CJK UNIFIED IDEOGRAPH-#34FB, IRGKangXi=0142.470, Def{(same as 剛) tough; unyielding; inflexible; hard}
34FC CJK UNIFIED IDEOGRAPH-#34FC, IRGKangXi=0143.160, Def{an incised wound; cuts}
34FD CJK UNIFIED IDEOGRAPH-#34FD, IRGKangXi=0143.220, Def{a kind of farm tool; iron spade; spade}
34FE CJK UNIFIED IDEOGRAPH-#34FE, IRGKangXi=0143.300, Def{to wound the skin; to cut open; to open out, ripped}
34FF CJK UNIFIED IDEOGRAPH-#34FF, IRGKangXi=0143.381
3500 CJK UNIFIED IDEOGRAPH-#3500, IRGKangXi=0143.400, Def{(non-classical form) to brand criminals on the face, to rob; to plunder}
3501 CJK UNIFIED IDEOGRAPH-#3501, IRGKangXi=0143.410, Def{a kind of tool; a hook; a barb; a sickle}
3502 CJK UNIFIED IDEOGRAPH-#3502, IRGKangXi=0143.420, Def{to pare; to pare away; to scrape; to sharpen to a point; to shave; to brush away}
3503 CJK UNIFIED IDEOGRAPH-#3503, IRGKangXi=0143.430, Def{(same as 絕) to cut short; to break off; to interrupt, to drag; to pull}
3504 CJK UNIFIED IDEOGRAPH-#3504, IRGKangXi=0143.440, Def{thin, small cut}
3505 CJK UNIFIED IDEOGRAPH-#3505, IRGKangXi=0143.470, Def{to cut; to hack; to reap, minced meat mixed up with ginger and cassia}
3506 CJK UNIFIED IDEOGRAPH-#3506, IRGKangXi=0144.030, Def{to cut; to hack; to reap; to trim or clip, to stab; to pierce; to brand}
3507 CJK UNIFIED IDEOGRAPH-#3507, IRGKangXi=0144.050, Def{(same as U+93FA 鏺) a kind of farm tool; a reaping-hook; used to cut grass}
3508 CJK UNIFIED IDEOGRAPH-#3508, IRGKangXi=0144.261
3509 CJK UNIFIED IDEOGRAPH-#3509, IRGKangXi=0144.261, Def{to dig; to excavate}
350A CJK UNIFIED IDEOGRAPH-#350A, IRGKangXi=0144.360, Def{to cut; to hack; to reap, to strike; to attack, to chastise rebels}
350B CJK UNIFIED IDEOGRAPH-#350B, IRGKangXi=0145.010, Def{cut into thin slices; to mince, sharp}
350C CJK UNIFIED IDEOGRAPH-#350C, IRGKangXi=0145.030, Def{to cut off; to mince, to cut up firewood}
350D CJK UNIFIED IDEOGRAPH-#350D, IRGKangXi=0145.040, Def{a hinged shear or long knife for cutting fodder; sheet-iron; etc., to cut up; to mince}
350E CJK UNIFIED IDEOGRAPH-#350E, IRGKangXi=0145.270, Def{to cut apart; to divide}
350F CJK UNIFIED IDEOGRAPH-#350F, IRGKangXi=0145.290, Def{(same as 剆) to cut open, to strike against; to clash together}
3510 CJK UNIFIED IDEOGRAPH-#3510, IRGKangXi=0145.310, Def{(corrupted form) to cut; to mince}
3511 CJK UNIFIED IDEOGRAPH-#3511, IRGKangXi=0145.350, Def{(same as U+34FC 㓼) an incised wound; cuts}
3512 CJK UNIFIED IDEOGRAPH-#3512, IRGKangXi=0145.400, Def{to cut; to cut out; to pick out; to scoop out, to cut short; to spare}
3513 CJK UNIFIED IDEOGRAPH-#3513, IRGKangXi=0146.140, Def{to stammer}
3514 CJK UNIFIED IDEOGRAPH-#3514, IRGKangXi=0146.241
3515 CJK UNIFIED IDEOGRAPH-#3515, IRGKangXi=0146.200, Def{with purpose; with ambition; with the determination and courage to get ahead}
3516 CJK UNIFIED IDEOGRAPH-#3516, IRGKangXi=0146.241
3517 CJK UNIFIED IDEOGRAPH-#3517, IRGKangXi=0146.250, Def{strong; healthy, to clasp under the arm, to persecute; to oppress, vigorous}
3518 CJK UNIFIED IDEOGRAPH-#3518, IRGKangXi=0147.071, Def{(same as 黜) to degrade; to dismiss, to reject; to dispel}
3519 CJK UNIFIED IDEOGRAPH-#3519, IRGKangXi=0147.150, Def{great strength; great power}
351A CJK UNIFIED IDEOGRAPH-#351A, IRGKangXi=0147.220, Def{to exert oneself; to make strenuous effort}
351B CJK UNIFIED IDEOGRAPH-#351B, IRGKangXi=0147.231
351C CJK UNIFIED IDEOGRAPH-#351C, IRGKangXi=0147.380, Def{(non-classical form of 勃) sudden; suddenly; quick; to change as the countenance}
351D CJK UNIFIED IDEOGRAPH-#351D, IRGKangXi=0147.450, Def{strong resistance; lazy, reluctant}
351E CJK UNIFIED IDEOGRAPH-#351E, IRGKangXi=0148.040, Def{vigorous; with great strength}
351F CJK UNIFIED IDEOGRAPH-#351F, IRGKangXi=0147.381
3520 CJK UNIFIED IDEOGRAPH-#3520, IRGKangXi=0148.090, Def{industry, sound of making strenuous efforts}
3521 CJK UNIFIED IDEOGRAPH-#3521, IRGKangXi=0148.280, Def{greatly; deeply, with great strength; vigorous}
3522 CJK UNIFIED IDEOGRAPH-#3522, IRGKangXi=0149.200, Def{to break something, to drag; to pull}
3523 CJK UNIFIED IDEOGRAPH-#3523, IRGKangXi=0149.410, Def{in ancient times to fight against the enemy by pushing down the big rocks from up high; to throw over, to urge; to exert oneself, to carry in the bosom}
3524 CJK UNIFIED IDEOGRAPH-#3524, IRGKangXi=0149.420, Def{to exert one's strength}
3525 CJK UNIFIED IDEOGRAPH-#3525, IRGKangXi=0149.430, Def{(non-classical form of 疲) full of anger; fierce anger, tired; weary; exhausted}
3526 CJK UNIFIED IDEOGRAPH-#3526, IRGKangXi=0150.020, Def{to encourage; to stimulate; to exhort to action; to encourage by rewards}
3527 CJK UNIFIED IDEOGRAPH-#3527, IRGKangXi=0150.070, Def{to incite; to urge, to aid; to help; to assist, to approve; to acquiesce}
3528 CJK UNIFIED IDEOGRAPH-#3528, IRGKangXi=0151.350, Def{name of a person}
3529 CJK UNIFIED IDEOGRAPH-#3529, IRGKangXi=0151.380, Def{hair ornaments used in old time}
352A CJK UNIFIED IDEOGRAPH-#352A, IRGKangXi=0152.030, Def{to crawl; to crawl on hands and knees}
352B CJK UNIFIED IDEOGRAPH-#352B, IRGKangXi=0152.241
352C CJK UNIFIED IDEOGRAPH-#352C, IRGKangXi=0152.310, Def{(ancient form of 卓) to establish, to surpass, eminent; lofty; profound}
352D CJK UNIFIED IDEOGRAPH-#352D, IRGKangXi=0153.060, Def{(same as 匙) a spoon, a key}
352E CJK UNIFIED IDEOGRAPH-#352E, IRGKangXi=0998.200, Def{a kind of animal which looks like a rat}
352F CJK UNIFIED IDEOGRAPH-#352F, IRGKangXi=0153.280, Def{(same as 匴) a large basket for holding cooked rice, a kind of tray made of bamboo used in ancient time}
3530 CJK UNIFIED IDEOGRAPH-#3530, IRGKangXi=0153.321, Def{(ancient form of 恆) constant; regular; continually; persevering}
3531 CJK UNIFIED IDEOGRAPH-#3531, IRGKangXi=0154.060, Def{(same as 庾) a stack of grain, a measure of 16 dǒu}
3532 CJK UNIFIED IDEOGRAPH-#3532, IRGKangXi=0154.161
3533 CJK UNIFIED IDEOGRAPH-#3533, IRGKangXi=0154.130, Def{(ancient form of 簋) a square basket of bamboo for holding grain used at sacrifices, feasts, etc.}
3534 CJK UNIFIED IDEOGRAPH-#3534, IRGKangXi=0154.140, Def{agricultural implement; farm tools, a big sacrificial vessel; a tripod of bronze with two ears; a caldron}
3535 CJK UNIFIED IDEOGRAPH-#3535, IRGKangXi=0154.180, Def{a kind of vessel}
3536 CJK UNIFIED IDEOGRAPH-#3536, IRGKangXi=0154.340, Def{a small cup, a box; a chest, a granary, cover of a vessel}
3537 CJK UNIFIED IDEOGRAPH-#3537, IRGKangXi=0155.030, Def{(ancient form of 陋) to secrete; to hide, a kind of basket}
3538 CJK UNIFIED IDEOGRAPH-#3538, IRGKangXi=0155.140, Def{(interchangeable 榹) a tray; a kind of dumbwaiter}
3539 CJK UNIFIED IDEOGRAPH-#3539, IRGKangXi=0155.210, Def{(standard form of 仂) a surplus or excess, a tithe, a great stock of merit}
353A CJK UNIFIED IDEOGRAPH-#353A, IRGKangXi=0156.090, Def{(ancient form of 世) an age, a generation of thirty years, from generation to generation}
353B CJK UNIFIED IDEOGRAPH-#353B, IRGKangXi=0156.110, Def{(same as 丕) great; distinguished}
353C CJK UNIFIED IDEOGRAPH-#353C, IRGKangXi=0157.140, Def{(non-classical form of 榫) to fit into, a tenon, connected}
353D CJK UNIFIED IDEOGRAPH-#353D, IRGKangXi=0158.190, Def{(non-classical form of 卣) a kind of wine pot or jar used in ancient time}
353E CJK UNIFIED IDEOGRAPH-#353E, IRGKangXi=0158.460, Def{(same as U+7B44 筄) last name; KangXi radical 26; a joint, a seal, (ancient form of 節)}
353F CJK UNIFIED IDEOGRAPH-#353F, IRGKangXi=0158.480, Def{a tally, like fitting the two halves of a tally, to petition the emperor that ..., to play music; to beat time}
3540 CJK UNIFIED IDEOGRAPH-#3540, IRGKangXi=0159.211
3541 CJK UNIFIED IDEOGRAPH-#3541, IRGKangXi=0159.260, Def{(non-classical form of 卻) still; but; yet; etc., to refuse to accept, to retreat; to withdraw}
3542 CJK UNIFIED IDEOGRAPH-#3542, IRGKangXi=0160.260, Def{(ancient form of 吟) to hum; to intone, to moan, to sigh, high ridges of cliffs, pebble ground}
3543 CJK UNIFIED IDEOGRAPH-#3543, IRGKangXi=0160.280, Def{to fall over; rolling on the ground}
3544 CJK UNIFIED IDEOGRAPH-#3544, IRGKangXi=0160.360, Def{(same as 厎 砥) to apply knowledge to final causes; to extend knowledge to the utmost, equally; average, smooth}
3545 CJK UNIFIED IDEOGRAPH-#3545, IRGKangXi=0160.390, Def{big; extremely; great; vast; extensive; tall}
3546 CJK UNIFIED IDEOGRAPH-#3546, IRGKangXi=0161.020, Def{fine stone; fine jade}
3547 CJK UNIFIED IDEOGRAPH-#3547, IRGKangXi=0161.030, Def{rocks collapsing (descriptive of sound), big pieces of rocks}
3548 CJK UNIFIED IDEOGRAPH-#3548, IRGKangXi=0161.090, Def{(interchangeable 庡) to linger; to walk to and from, to conceal; to cover up; to secrete; to hide; to cover over}
3549 CJK UNIFIED IDEOGRAPH-#3549, IRGKangXi=0161.140, Def{to dig a hole; to tuck in; to put the finger into}
354A CJK UNIFIED IDEOGRAPH-#354A, IRGKangXi=0161.240, Def{to slant; intricate; describe a writer's pen, lofty rocks; protruded rocks; resolute}
354B CJK UNIFIED IDEOGRAPH-#354B, IRGKangXi=0161.380, Def{rock land}
354C CJK UNIFIED IDEOGRAPH-#354C, IRGKangXi=0161.410, Def{to raise the curtain, a kind of rock}
354D CJK UNIFIED IDEOGRAPH-#354D, IRGKangXi=0162.020, Def{(same as 堆) to heap up; to pile, a heap; a pile; a mass; a crowd}
354E CJK UNIFIED IDEOGRAPH-#354E, IRGKangXi=0162.310, Def{caves; grottos on the mountain side, collapsed in ruins; destroyed; to clash; to bump against}
354F CJK UNIFIED IDEOGRAPH-#354F, IRGKangXi=0162.441, Def{(corrupted form of 廚) a kitchen, a sideboard with cupboard and drawers}
3550 CJK UNIFIED IDEOGRAPH-#3550, IRGKangXi=0162.500, Def{inclined; slanting, to upset; to be overthrown, to collapse, to fall flat}
3551 CJK UNIFIED IDEOGRAPH-#3551, IRGKangXi=0163.041, Def{(non-classical variant of U+53A8 厨) a kitchen, a sideboard with cupboard and drawers}
3552 CJK UNIFIED IDEOGRAPH-#3552, IRGKangXi=0163.050, Def{a lofty peak; a mountain peak}
3553 CJK UNIFIED IDEOGRAPH-#3553, IRGKangXi=0163.131
3554 CJK UNIFIED IDEOGRAPH-#3554, IRGKangXi=0163.171, Def{(non-classical form of 廳) a hall; a court; a lodge; a room}
3555 CJK UNIFIED IDEOGRAPH-#3555, IRGKangXi=0164.050, Def{(ancient form of 幻) deception; sleight of hand; magic}
3556 CJK UNIFIED IDEOGRAPH-#3556, IRGKangXi=0164.260, Def{(ancient form of 素) plain; unornamented, white, ordinary, simple}
3557 CJK UNIFIED IDEOGRAPH-#3557, IRGKangXi=0164.270, Def{(same as 誘) to induce; to entice; to mislead; to lead on}
3558 CJK UNIFIED IDEOGRAPH-#3558, IRGKangXi=0164.301, Def{(a variant of 參) to take part in; to visit; to counsel, one of the 28 lunar mansions; ginseng}
3559 CJK UNIFIED IDEOGRAPH-#3559, IRGKangXi=0164.390, Def{a wily rabbit; a cunning hare}
355A CJK UNIFIED IDEOGRAPH-#355A, IRGKangXi=0165.020, Def{(ancient form of 爪) claws of birds or animals, feet, to scratch, to claw, to grasp}
355B CJK UNIFIED IDEOGRAPH-#355B, IRGKangXi=0165.120, Def{(ancient form of 友) a friend; a companion; a associate, friendly, to make friends of, brotherly regard}
355C CJK UNIFIED IDEOGRAPH-#355C, IRGKangXi=0165.300, Def{(ancient form of 史) history; chronicles; annals}
355D CJK UNIFIED IDEOGRAPH-#355D, IRGKangXi=0165.391
355E CJK UNIFIED IDEOGRAPH-#355E, IRGKangXi=0166.020, Def{(same as U+5237 刷) a brush, to brush; to clean; to scrub, to print, especially from blocks}
355F CJK UNIFIED IDEOGRAPH-#355F, IRGKangXi=0166.080, Def{(same as 喟) to sign; to lament, to stop, to cut off; to sever; to interrupt}
3560 CJK UNIFIED IDEOGRAPH-#3560, IRGKangXi=0166.391, Def{(same as 雙) a pair; a couple, both, two, double, even}
3561 CJK UNIFIED IDEOGRAPH-#3561, IRGKangXi=0167.010, Def{(same as U+58D1 壑) the bed of a torrent; a gully, a pool}
3562 CJK UNIFIED IDEOGRAPH-#3562, IRGKangXi=1208.260, Def{profound significance, just now; then; in that case}
3563 CJK UNIFIED IDEOGRAPH-#3563, IRGKangXi=0172.090, Def{a marsh at the foot of the hills, (interchangeable 湢 沇) name of a river}
3564 CJK UNIFIED IDEOGRAPH-#3564, IRGKangXi=0173.080, Def{a proud temper, military equipment; a long spear}
3565 CJK UNIFIED IDEOGRAPH-#3565, IRGKangXi=0174.091
3566 CJK UNIFIED IDEOGRAPH-#3566, IRGKangXi=0175.010, Def{loud; clamour; hubbub, big mouth, to brag}
3567 CJK UNIFIED IDEOGRAPH-#3567, IRGKangXi=0175.080, Def{used to represent sound; to hum; to groan}
3568 CJK UNIFIED IDEOGRAPH-#3568, IRGKangXi=0177.021
3569 CJK UNIFIED IDEOGRAPH-#3569, IRGKangXi=0177.110, Def{(non-classical form) sound of tapping; sound of striking, opposing voices}
356A CJK UNIFIED IDEOGRAPH-#356A, IRGKangXi=0177.120, Def{birds' chirps; gabbling, sound in high pitch; to roar; aloud}
356B CJK UNIFIED IDEOGRAPH-#356B, IRGKangXi=0177.230, Def{(same as 仿 方) to imitate, like; resembling; according to, a measurement for sound effect}
356C CJK UNIFIED IDEOGRAPH-#356C, IRGKangXi=0178.100, Def{public opinion, (same as U+8A1F 訟) to go to law, to dispute, to demand justice; (Cant.) classifier for ears of corn}
356D CJK UNIFIED IDEOGRAPH-#356D, IRGKangXi=0178.230, Def{(same as 咬) to bark as a dog, obscene expressions, an interjection; implying surprise}
356E CJK UNIFIED IDEOGRAPH-#356E, IRGKangXi=0178.250, Def{to chew; to masticate, to dwell on, Chinese medicine term}
356F CJK UNIFIED IDEOGRAPH-#356F, IRGKangXi=0179.090, Def{slow-tongued; to stammer, to shout in triumph; the noise of shouting in battle}
3570 CJK UNIFIED IDEOGRAPH-#3570, IRGKangXi=0180.050, Def{sound angrily; roaring with rage, (same as 嘛) (a dialect) what; the tone implying to remind}
3571 CJK UNIFIED IDEOGRAPH-#3571, IRGKangXi=0181.070, Def{to bark (said of a dog)}
3572 CJK UNIFIED IDEOGRAPH-#3572, IRGKangXi=0181.130, Def{to gag or bribe; to satisfy one's appetite; name of an old book; (Cant.) to guess, reckon mentally; to bud}
3573 CJK UNIFIED IDEOGRAPH-#3573, IRGKangXi=0181.190, Def{(of a crowd) noisy; to brawl; to scold}
3574 CJK UNIFIED IDEOGRAPH-#3574, IRGKangXi=0181.191
3575 CJK UNIFIED IDEOGRAPH-#3575, IRGKangXi=0181.191
3576 CJK UNIFIED IDEOGRAPH-#3576, IRGKangXi=0181.191, Def{(Cant.) an expletive, exclamation}
3577 CJK UNIFIED IDEOGRAPH-#3577, IRGKangXi=0181.270, Def{(corrupted form 咟) to call, to yell, anxious, dazed, image sound, an exclamation expressing sound (such as clap hands; to fire a gun; to strike; sound of firecracker etc.)}
3578 CJK UNIFIED IDEOGRAPH-#3578, IRGKangXi=0181.310, Def{sound of rolling a boat}
3579 CJK UNIFIED IDEOGRAPH-#3579, IRGKangXi=0183.010, Def{a buckler; a shield}
357A CJK UNIFIED IDEOGRAPH-#357A, IRGKangXi=0183.100, Def{(corrupted form of 囂) looks big outside but empty inside, to yell; voice of anger, hubbub; clamour}
357B CJK UNIFIED IDEOGRAPH-#357B, IRGKangXi=0184.090, Def{to refuse, to repel}
357C CJK UNIFIED IDEOGRAPH-#357C, IRGKangXi=0185.080, Def{to brawl; to scold, noise from a crowd; hubbub; noisy; loud noise}
357D CJK UNIFIED IDEOGRAPH-#357D, IRGKangXi=0185.211, Def{a phonetic}
357E CJK UNIFIED IDEOGRAPH-#357E, IRGKangXi=0185.211
357F CJK UNIFIED IDEOGRAPH-#357F, IRGKangXi=0185.211
3580 CJK UNIFIED IDEOGRAPH-#3580, IRGKangXi=0186.100, Def{sound}
3581 CJK UNIFIED IDEOGRAPH-#3581, IRGKangXi=0186.120, Def{(non-classical form of 吝) stingy, sparing of; closefisted, to regret, to shame; to insult}
3582 CJK UNIFIED IDEOGRAPH-#3582, IRGKangXi=0187.010, Def{(same as 詍) loquacity; to talk incessantly and tediously, to laugh, joy}
3583 CJK UNIFIED IDEOGRAPH-#3583, IRGKangXi=0187.080, Def{(interchangeable 吼) to roar of animals, angry tones, (interchangeable 詬) a sense of shame; to shame, to vomit}
3584 CJK UNIFIED IDEOGRAPH-#3584, IRGKangXi=0187.220, Def{sound of calling chickens, to curse; to swear}
3585 CJK UNIFIED IDEOGRAPH-#3585, IRGKangXi=0188.160, Def{sound of a whistle, to blow}
3586 CJK UNIFIED IDEOGRAPH-#3586, IRGKangXi=0188.180, Def{to condemn; to sentence, to hold a hearing (on a legal case); to interrogate or question a prisoner, to whisper}
3587 CJK UNIFIED IDEOGRAPH-#3587, IRGKangXi=0189.130, Def{the lips; (Cant.) to sip}
3588 CJK UNIFIED IDEOGRAPH-#3588, IRGKangXi=0189.271
3589 CJK UNIFIED IDEOGRAPH-#3589, IRGKangXi=0189.271
358A CJK UNIFIED IDEOGRAPH-#358A, IRGKangXi=0189.271, Def{fathom}
358B CJK UNIFIED IDEOGRAPH-#358B, IRGKangXi=0189.271
358C CJK UNIFIED IDEOGRAPH-#358C, IRGKangXi=0189.271
358D CJK UNIFIED IDEOGRAPH-#358D, IRGKangXi=0189.271
358E CJK UNIFIED IDEOGRAPH-#358E, IRGKangXi=0189.271
358F CJK UNIFIED IDEOGRAPH-#358F, IRGKangXi=0189.290, Def{angry, to scold with loud voice, to slander; to defame}
3590 CJK UNIFIED IDEOGRAPH-#3590, IRGKangXi=0190.060, Def{sound of calling ducks}
3591 CJK UNIFIED IDEOGRAPH-#3591, IRGKangXi=0190.140, Def{to shout at, (same as 欸) to backbite, to dislike, sound of oars; the creaking or swishing sound of oars or sweeps in rowing}
3592 CJK UNIFIED IDEOGRAPH-#3592, IRGKangXi=0190.150, Def{(non-classical form of 啼) to cry; to mourn; to howl, to twitter; to crow, to sneeze; a running at the nose}
3593 CJK UNIFIED IDEOGRAPH-#3593, IRGKangXi=0191.120, Def{(corrupted form) to shout; to scold with loud voice, din; noise; a confused noise}
3594 CJK UNIFIED IDEOGRAPH-#3594, IRGKangXi=0191.180, Def{silly, foolish; doting; idiotic}
3595 CJK UNIFIED IDEOGRAPH-#3595, IRGKangXi=0191.210, Def{to spit out; to blame, name of an organic compounds}
3596 CJK UNIFIED IDEOGRAPH-#3596, IRGKangXi=0192.180, Def{(standard form) to spit out; to blame, name of an organic compounds}
3597 CJK UNIFIED IDEOGRAPH-#3597, IRGKangXi=0193.051, Def{(corrupted form) high ridges of cliffs}
3598 CJK UNIFIED IDEOGRAPH-#3598, IRGKangXi=0193.051, Def{(same as U+5507 唇) to startle, to alarm; afraid; scared, to be surprised; to be amazed}
3599 CJK UNIFIED IDEOGRAPH-#3599, IRGKangXi=0193.051
359A CJK UNIFIED IDEOGRAPH-#359A, IRGKangXi=0193.051
359B CJK UNIFIED IDEOGRAPH-#359B, IRGKangXi=0193.051
359C CJK UNIFIED IDEOGRAPH-#359C, IRGKangXi=0193.051
359D CJK UNIFIED IDEOGRAPH-#359D, IRGKangXi=0193.051
359E CJK UNIFIED IDEOGRAPH-#359E, IRGKangXi=0193.051, Def{(simplified form of 喎) a wry mouth}
359F CJK UNIFIED IDEOGRAPH-#359F, IRGKangXi=0193.080, Def{to deliver over to personally, to communicate orally with one's own mouth}
35A0 CJK UNIFIED IDEOGRAPH-#35A0, IRGKangXi=0193.130, Def{(same as 那 哪) an auxiliary (in grammar), that, there, a final particle.}
35A1 CJK UNIFIED IDEOGRAPH-#35A1, IRGKangXi=0194.040, Def{the birds singing during the night; (Cant.) interjection to indicate the speaker is thinking}
35A2 CJK UNIFIED IDEOGRAPH-#35A2, IRGKangXi=0194.050, Def{(same as 飺) to reject food}
35A3 CJK UNIFIED IDEOGRAPH-#35A3, IRGKangXi=0194.070, Def{(same as U+5485 咅) to spit out; (Cant.) the sound of spitting}
35A4 CJK UNIFIED IDEOGRAPH-#35A4, IRGKangXi=0194.130, Def{wrath; anger}
35A5 CJK UNIFIED IDEOGRAPH-#35A5, IRGKangXi=0194.210, Def{to spit; to vomit; (Cant.) to gnaw (on bones)}
35A6 CJK UNIFIED IDEOGRAPH-#35A6, IRGKangXi=0194.260, Def{loquacity}
35A7 CJK UNIFIED IDEOGRAPH-#35A7, IRGKangXi=0195.070, Def{can not see clear, (ancient form of 吻) lips}
35A8 CJK UNIFIED IDEOGRAPH-#35A8, IRGKangXi=0195.140, Def{Indistinct nasal utterance, laugh, sound of birds}
35A9 CJK UNIFIED IDEOGRAPH-#35A9, IRGKangXi=0196.140, Def{not pure, immodest, to urge, (same as 嗾) to set a dog on}
35AA CJK UNIFIED IDEOGRAPH-#35AA, IRGKangXi=0196.160, Def{sound of surprising, to laugh loudly, to flow off, loquacious, sound; voice; tone}
35AB CJK UNIFIED IDEOGRAPH-#35AB, IRGKangXi=0196.261, Def{(sound transcription) used in names for minority ethnic group in ancient China}
35AC CJK UNIFIED IDEOGRAPH-#35AC, IRGKangXi=0196.261
35AD CJK UNIFIED IDEOGRAPH-#35AD, IRGKangXi=0196.261, Def{(same as 舔) to lick; to taste; (Cant.) even; in addition}
35AE CJK UNIFIED IDEOGRAPH-#35AE, IRGKangXi=0196.261, Def{an interjection used in poems and songs}
35AF CJK UNIFIED IDEOGRAPH-#35AF, IRGKangXi=0196.261
35B0 CJK UNIFIED IDEOGRAPH-#35B0, IRGKangXi=0196.261
35B1 CJK UNIFIED IDEOGRAPH-#35B1, IRGKangXi=0196.261
35B2 CJK UNIFIED IDEOGRAPH-#35B2, IRGKangXi=0196.261
35B3 CJK UNIFIED IDEOGRAPH-#35B3, IRGKangXi=0196.261
35B4 CJK UNIFIED IDEOGRAPH-#35B4, IRGKangXi=0196.261
35B5 CJK UNIFIED IDEOGRAPH-#35B5, IRGKangXi=0196.261
35B6 CJK UNIFIED IDEOGRAPH-#35B6, IRGKangXi=0196.280, Def{(same as 捗) to swallow; to gulp, to be choked with food, (same as 諲) respectful; venerable}
35B7 CJK UNIFIED IDEOGRAPH-#35B7, IRGKangXi=0197.030, Def{bird singing, (same as 啼) to cry; to mourn; to howl, to twitter; to crow}
35B8 CJK UNIFIED IDEOGRAPH-#35B8, IRGKangXi=0197.090, Def{(same as 噱) to laugh heartily; to roar with laughter, (in Shanghai dialect) a promotional gimmick, tone (of one's speech)}
35B9 CJK UNIFIED IDEOGRAPH-#35B9, IRGKangXi=0197.150, Def{(non-classical form of噴) to spurt; to blow out; to puff out, to snort, sound of flowing water}
35BA CJK UNIFIED IDEOGRAPH-#35BA, IRGKangXi=0197.160, Def{to blow; to breath; to puff}
35BB CJK UNIFIED IDEOGRAPH-#35BB, IRGKangXi=0197.230, Def{whisper, bad language}
35BC CJK UNIFIED IDEOGRAPH-#35BC, IRGKangXi=0198.030, Def{(same as 謀) to scheme; to plot; to devise, a device}
35BD CJK UNIFIED IDEOGRAPH-#35BD, IRGKangXi=0198.060, Def{(corrupted form of 嘖) to call out; to make an uproar}
35BE CJK UNIFIED IDEOGRAPH-#35BE, IRGKangXi=0198.120, Def{(same as 咢) to beat a drum; to startle, to argue; to debate; to dispute, (interchangeable 愕) to be surprised; to be amazed; to marvel, (interchangeable 鍔) the blade or edge of a sword, beams of a house}
35BF CJK UNIFIED IDEOGRAPH-#35BF, IRGKangXi=0198.130, Def{a sound; a voice; a tone, an interjection; to hesitate; to harbour doubts}
35C0 CJK UNIFIED IDEOGRAPH-#35C0, IRGKangXi=0198.190, Def{(same as 呦 嚘) the bleating of the deer, sound of insects}
35C1 CJK UNIFIED IDEOGRAPH-#35C1, IRGKangXi=0198.310, Def{(same as 齶) the roof of the mouth, the palate}
35C2 CJK UNIFIED IDEOGRAPH-#35C2, IRGKangXi=0198.320, Def{to keep the mouth shut; to be careful how one speak, (non-classical form of U+7701 省) to examine; to watch; (Cant.) to scour}
35C3 CJK UNIFIED IDEOGRAPH-#35C3, IRGKangXi=0198.340, Def{(same as 吻) the lip, the tone of one's speech, to kiss; a kiss}
35C4 CJK UNIFIED IDEOGRAPH-#35C4, IRGKangXi=0199.040, Def{(non-classical form of 謾) to deceive; to insult}
35C5 CJK UNIFIED IDEOGRAPH-#35C5, IRGKangXi=0199.090, Def{(same as 胡) angry, the throat, what? how? why? which?}
35C6 CJK UNIFIED IDEOGRAPH-#35C6, IRGKangXi=0199.210, Def{sound of the ringed pheasant, cackling of fowls, (same as 嗝) to gag, to vomit, (a dialect) an auxiliary verb (almost same as 的)}
35C7 CJK UNIFIED IDEOGRAPH-#35C7, IRGKangXi=0199.230, Def{to swallow; to gulp, an auxiliary; usually at the end of a sentence to beg or request for something}
35C8 CJK UNIFIED IDEOGRAPH-#35C8, IRGKangXi=0199.240, Def{(same as 曼) long; extended, large}
35C9 CJK UNIFIED IDEOGRAPH-#35C9, IRGKangXi=0200.100, Def{incisive, (the large seal type 籀) to brand criminals on the face, to rob; to plunder, an edge, a two-edged weapon; a sword}
35CA CJK UNIFIED IDEOGRAPH-#35CA, IRGKangXi=0200.210, Def{public opinion; clamour; noise, (ancient form of 雷); thunder, an organic compound (porphin); (porphyrins)}
35CB CJK UNIFIED IDEOGRAPH-#35CB, IRGKangXi=0201.030, Def{(standard form of 喉) the throat; the gullet; guttural}
35CC CJK UNIFIED IDEOGRAPH-#35CC, IRGKangXi=0198.150, Def{(non-classical form) to scold with loud voice, to talk out of control; loquacious}
35CD CJK UNIFIED IDEOGRAPH-#35CD, IRGKangXi=0201.121, Def{(non-classical form) to gnaw; to bite; to masticate, sound of biting}
35CE CJK UNIFIED IDEOGRAPH-#35CE, IRGKangXi=0201.121, Def{(Cant.) final particle indicating emphasis or surprise; phonetic}
35CF CJK UNIFIED IDEOGRAPH-#35CF, IRGKangXi=0201.121
35D0 CJK UNIFIED IDEOGRAPH-#35D0, IRGKangXi=0201.121
35D1 CJK UNIFIED IDEOGRAPH-#35D1, IRGKangXi=0201.121
35D2 CJK UNIFIED IDEOGRAPH-#35D2, IRGKangXi=0201.121, Def{(in grammar) interjection; to express sadness; sorrow and emotional excitement}
35D3 CJK UNIFIED IDEOGRAPH-#35D3, IRGKangXi=0201.170, Def{mouthful}
35D4 CJK UNIFIED IDEOGRAPH-#35D4, IRGKangXi=0201.180, Def{(a dialect) joy; happiness}
35D5 CJK UNIFIED IDEOGRAPH-#35D5, IRGKangXi=0202.040, Def{(same as 雊) the crowing of a pheasant, loud and noise}
35D6 CJK UNIFIED IDEOGRAPH-#35D6, IRGKangXi=0202.160, Def{(non-classical form of 啗) to eat; to chew; to bite. to entice; to lure, to contain; to, comprise}
35D7 CJK UNIFIED IDEOGRAPH-#35D7, IRGKangXi=0202.380, Def{to separate; to part, to split; to crack, to rend; to rip open}
35D8 CJK UNIFIED IDEOGRAPH-#35D8, IRGKangXi=0203.020, Def{to bite the hard and solid food}
35D9 CJK UNIFIED IDEOGRAPH-#35D9, IRGKangXi=0203.030, Def{to shout or bawl, talking manner of a mean person, (same as 謅) to jest; to joke; to quip}
35DA CJK UNIFIED IDEOGRAPH-#35DA, IRGKangXi=0203.200, Def{vexingly verbose or wordy; prosy; complicated; annoying}
35DB CJK UNIFIED IDEOGRAPH-#35DB, IRGKangXi=0204.030, Def{(same as 笑) to laugh; to smile, to laugh at; to ridicule}
35DC CJK UNIFIED IDEOGRAPH-#35DC, IRGKangXi=0203.190, Def{(same as 嗅) to smell, to scent}
35DD CJK UNIFIED IDEOGRAPH-#35DD, IRGKangXi=0204.111
35DE CJK UNIFIED IDEOGRAPH-#35DE, IRGKangXi=0204.111
35DF CJK UNIFIED IDEOGRAPH-#35DF, IRGKangXi=0204.111
35E0 CJK UNIFIED IDEOGRAPH-#35E0, IRGKangXi=0204.111
35E1 CJK UNIFIED IDEOGRAPH-#35E1, IRGKangXi=0204.111
35E2 CJK UNIFIED IDEOGRAPH-#35E2, IRGKangXi=0204.180, Def{loquacious, the sound of singing, to sing in a loud voice, loud}
35E3 CJK UNIFIED IDEOGRAPH-#35E3, IRGKangXi=0204.330, Def{(same as 嚏) to sneeze; a running at the nose}
35E4 CJK UNIFIED IDEOGRAPH-#35E4, IRGKangXi=0204.360, Def{ashamed, restless; fidgeting; (a dialect) bashful, grievous; mournful; sad}
35E5 CJK UNIFIED IDEOGRAPH-#35E5, IRGKangXi=0205.040, Def{sound of striking, slow expiration of the breath, a deep sign}
35E6 CJK UNIFIED IDEOGRAPH-#35E6, IRGKangXi=0205.080, Def{a loud and confused noise}
35E7 CJK UNIFIED IDEOGRAPH-#35E7, IRGKangXi=0205.120, Def{to scold with loud voice, to talk out of control; loquacious}
35E8 CJK UNIFIED IDEOGRAPH-#35E8, IRGKangXi=0205.190, Def{to laugh; to smile, to laugh at; to ridicule, sound of laughter}
35E9 CJK UNIFIED IDEOGRAPH-#35E9, IRGKangXi=0205.220, Def{sound made when cold (brrr); to endure cold}
35EA CJK UNIFIED IDEOGRAPH-#35EA, IRGKangXi=0205.240, Def{(corrupted form) mouthful}
35EB CJK UNIFIED IDEOGRAPH-#35EB, IRGKangXi=0206.020, Def{to whisper; to talk in a soft and low voice}
35EC CJK UNIFIED IDEOGRAPH-#35EC, IRGKangXi=0206.110, Def{thick lips, to talk slowly}
35ED CJK UNIFIED IDEOGRAPH-#35ED, IRGKangXi=0206.190, Def{a sound; an echo}
35EE CJK UNIFIED IDEOGRAPH-#35EE, IRGKangXi=0206.201
35EF CJK UNIFIED IDEOGRAPH-#35EF, IRGKangXi=0206.201
35F0 CJK UNIFIED IDEOGRAPH-#35F0, IRGKangXi=0206.201, Def{used in transliteration; the traditional drama of Vietnam; (Cant.) all, entire, completely}
35F1 CJK UNIFIED IDEOGRAPH-#35F1, IRGKangXi=0206.300, Def{to suck; to chew, to smear the mouth with the blood of a victim when taking an oath}
35F2 CJK UNIFIED IDEOGRAPH-#35F2, IRGKangXi=0207.060, Def{to call; to summon; (Cant.) eloquent}
35F3 CJK UNIFIED IDEOGRAPH-#35F3, IRGKangXi=0207.170, Def{forgetful; having a bad memory, disgraced; shy; sensitive to shame; ashamed; (Cant.) to lower one's head}
35F4 CJK UNIFIED IDEOGRAPH-#35F4, IRGKangXi=0207.310, Def{dogs fighting, to go to law; an indictment}
35F5 CJK UNIFIED IDEOGRAPH-#35F5, IRGKangXi=0208.150, Def{breath, air, steam, not clear; not quite sure}
35F6 CJK UNIFIED IDEOGRAPH-#35F6, IRGKangXi=0208.260, Def{to expel the breath}
35F7 CJK UNIFIED IDEOGRAPH-#35F7, IRGKangXi=0209.151, Def{(simplified form) to eat; to bite; to gnaw, a dialect, usually used at the end of a sentence}
35F8 CJK UNIFIED IDEOGRAPH-#35F8, IRGKangXi=0209.151
35F9 CJK UNIFIED IDEOGRAPH-#35F9, IRGKangXi=0209.151
35FA CJK UNIFIED IDEOGRAPH-#35FA, IRGKangXi=0209.151
35FB CJK UNIFIED IDEOGRAPH-#35FB, IRGKangXi=0209.220, Def{baby's answering, (baby talk) eat; (Cant.) a particle implying doubt; slow, troublesome}
35FC CJK UNIFIED IDEOGRAPH-#35FC, IRGKangXi=0209.360, Def{eating or talking, moving of the mouth}
35FD CJK UNIFIED IDEOGRAPH-#35FD, IRGKangXi=0209.370, Def{(standard form 響) an echo; to answer; to consent, noise; sound, to make a noise}
35FE CJK UNIFIED IDEOGRAPH-#35FE, IRGKangXi=0210.130, Def{to give vent to the feelings; (Cant.) to hiss, boo, jeer}
35FF CJK UNIFIED IDEOGRAPH-#35FF, IRGKangXi=0210.230, Def{to laugh loudly, to blame; to reprimand, entrance to a cave or to a gorge}
3600 CJK UNIFIED IDEOGRAPH-#3600, IRGKangXi=0210.250, Def{to sound, a sound, the cry of a bird or animal}
3601 CJK UNIFIED IDEOGRAPH-#3601, IRGKangXi=0211.010, Def{to call; to scream; to whistle, name of a stream}
3602 CJK UNIFIED IDEOGRAPH-#3602, IRGKangXi=0211.020, Def{ashamed; bashful}
3603 CJK UNIFIED IDEOGRAPH-#3603, IRGKangXi=0211.121
3604 CJK UNIFIED IDEOGRAPH-#3604, IRGKangXi=0211.121
3605 CJK UNIFIED IDEOGRAPH-#3605, IRGKangXi=0211.270, Def{(same as U+929C 銜 U+5563 啣) to hold in the mouth}
3606 CJK UNIFIED IDEOGRAPH-#3606, IRGKangXi=0211.280, Def{to laugh at; to deride; to jeer at, to snore}
3607 CJK UNIFIED IDEOGRAPH-#3607, IRGKangXi=0211.380
3608 CJK UNIFIED IDEOGRAPH-#3608, IRGKangXi=0212.040, Def{able to take a correct view of things; to response correctly (said of small children), sound; echo}
3609 CJK UNIFIED IDEOGRAPH-#3609, IRGKangXi=0212.210, Def{to make sound; to speak; (Cant.) to tickle, squeeze}
360A CJK UNIFIED IDEOGRAPH-#360A, IRGKangXi=0212.280, Def{the abrupt and hasty sound of the crowd}
360B CJK UNIFIED IDEOGRAPH-#360B, IRGKangXi=0212.300, Def{sound; echo, difficult; troublesome}
360C CJK UNIFIED IDEOGRAPH-#360C, IRGKangXi=0212.330, Def{(non-classical form of 噱) to laugh without stopping, loud laughter}
360D CJK UNIFIED IDEOGRAPH-#360D, IRGKangXi=0212.360, Def{sound of rat; sound of insects}
360E CJK UNIFIED IDEOGRAPH-#360E, IRGKangXi=0213.171, Def{(simplified form) the roars of a tiger}
360F CJK UNIFIED IDEOGRAPH-#360F, IRGKangXi=0213.171
3610 CJK UNIFIED IDEOGRAPH-#3610, IRGKangXi=0213.200, Def{loud; to roar}
3611 CJK UNIFIED IDEOGRAPH-#3611, IRGKangXi=0213.220, Def{to curse, to swear and oath, incantations}
3612 CJK UNIFIED IDEOGRAPH-#3612, IRGKangXi=0213.471
3613 CJK UNIFIED IDEOGRAPH-#3613, IRGKangXi=0214.060, Def{prolix speech, (same as 讕) to make a false charge; to lay a false accusation}
3614 CJK UNIFIED IDEOGRAPH-#3614, IRGKangXi=0214.151, Def{eat, to bite, to gnaw, (a dialect) usually at the end of a sentence; tone (of one's speech)}
3615 CJK UNIFIED IDEOGRAPH-#3615, IRGKangXi=0214.190, Def{(non-classical form of 喊) (standard form of 婪) to call; to halloo, covetous; avaricious}
3616 CJK UNIFIED IDEOGRAPH-#3616, IRGKangXi=0214.250, Def{to make a disturbance; to quarrel; to wrangle}
3617 CJK UNIFIED IDEOGRAPH-#3617, IRGKangXi=0215.051, Def{whisper; to whistle}
3618 CJK UNIFIED IDEOGRAPH-#3618, IRGKangXi=0215.221
3619 CJK UNIFIED IDEOGRAPH-#3619, IRGKangXi=0215.230, Def{to groan; to moan}
361A CJK UNIFIED IDEOGRAPH-#361A, IRGKangXi=0215.271, Def{angry appearance; to look furious; the roaring of a tiger}
361B CJK UNIFIED IDEOGRAPH-#361B, IRGKangXi=0216.070, Def{(non-classical form of 嵌) a deep valley, piece of music in minority group}
361C CJK UNIFIED IDEOGRAPH-#361C, IRGKangXi=0216.110, Def{(non-classical form of 雔) a kind of silkworm, pair of birds; couple; companion; to match, corresponding, proper}
361D CJK UNIFIED IDEOGRAPH-#361D, IRGKangXi=0216.190, Def{to take in; to absorb, (non-classical form of 罱) a kind of spring fishing net; a kind of small net with a handle used to catch fishes or birds}
361E CJK UNIFIED IDEOGRAPH-#361E, IRGKangXi=0146.150, Def{sound of yelling while towing a boat; (Sanskrit) to call out}
361F CJK UNIFIED IDEOGRAPH-#361F, IRGKangXi=0217.061
3620 CJK UNIFIED IDEOGRAPH-#3620, IRGKangXi=0217.290, Def{to close; to stop up; to obstruct}
3621 CJK UNIFIED IDEOGRAPH-#3621, IRGKangXi=0217.310, Def{(same as 柙) a pen for wild beasts; a cage for prisoners}
3622 CJK UNIFIED IDEOGRAPH-#3622, IRGKangXi=0218.050, Def{a round cave; a round hole}
3623 CJK UNIFIED IDEOGRAPH-#3623, IRGKangXi=0218.170, Def{compasses, (same as U+5713 圓) round; circular}
3624 CJK UNIFIED IDEOGRAPH-#3624, IRGKangXi=0220.160, Def{circle of the moving water}
3625 CJK UNIFIED IDEOGRAPH-#3625, IRGKangXi=0221.160, Def{(same as 囮) to inveigle; to decoy, a decoy-bird; a go-between; a medium}
3626 CJK UNIFIED IDEOGRAPH-#3626, IRGKangXi=0223.080, Def{wall made of soil, used in name of places}
3627 CJK UNIFIED IDEOGRAPH-#3627, IRGKangXi=0225.020, Def{(same as 序) precedence, order, orderly, the east and west walls of the parlor}
3628 CJK UNIFIED IDEOGRAPH-#3628, IRGKangXi=0225.050, Def{deep; profound, empty; hollow, quite; peaceful}
3629 CJK UNIFIED IDEOGRAPH-#3629, IRGKangXi=0225.110, Def{to mate; to pair}
362A CJK UNIFIED IDEOGRAPH-#362A, IRGKangXi=0226.080, Def{(same as 耗) to waste, to destroy, to diminish; hence it came to mean, a rat, heedless; careless; inattentive; rough and coarse}
362B CJK UNIFIED IDEOGRAPH-#362B, IRGKangXi=0226.101, Def{(a variant of 阱) to fall into a well, a pit; a pitfall; a hole; a gully, to involve; to entrap}
362C CJK UNIFIED IDEOGRAPH-#362C, IRGKangXi=0226.101, Def{(same as 坳) a hollow in the ground; a cavity, (same as 物) matter; substance, all living creatures, things in general, the affairs of this world, things or matters outside oneself, others, goods, the Tibetan classical text means Buddha}
362D CJK UNIFIED IDEOGRAPH-#362D, IRGKangXi=0226.101, Def{(same as 坳) a hollow in the ground; a cavity, undulating}
362E CJK UNIFIED IDEOGRAPH-#362E, IRGKangXi=0226.101
362F CJK UNIFIED IDEOGRAPH-#362F, IRGKangXi=0226.101
3630 CJK UNIFIED IDEOGRAPH-#3630, IRGKangXi=0226.101, Def{(interchangeable 斟) name of a old country; used in name of a place}
3631 CJK UNIFIED IDEOGRAPH-#3631, IRGKangXi=0226.230, Def{(same as 坍) to fall into ruins; to collapse}
3632 CJK UNIFIED IDEOGRAPH-#3632, IRGKangXi=0226.360, Def{a cove; a bay, a bend or nook in the hills, the outside part of a curved bank; the curve of a bow}
3633 CJK UNIFIED IDEOGRAPH-#3633, IRGKangXi=0227.230, Def{(same as 丘) a hillock or mound}
3634 CJK UNIFIED IDEOGRAPH-#3634, IRGKangXi=0227.271, Def{(same as 坐) to sit, a seat}
3635 CJK UNIFIED IDEOGRAPH-#3635, IRGKangXi=0227.271, Def{name of a place; in Fujian Province}
3636 CJK UNIFIED IDEOGRAPH-#3636, IRGKangXi=0228.060, Def{(ancient form of 桀) a hen-roost, cruel, the last ruler of the Xia Dynasty}
3637 CJK UNIFIED IDEOGRAPH-#3637, IRGKangXi=0228.230, Def{(same as 艾) Mugwort, artemisia or any plant which produces moxa punk, general name for plants like mint}
3638 CJK UNIFIED IDEOGRAPH-#3638, IRGKangXi=0228.290, Def{(same as 葬) to bury, (ancient form of 坐) to sit, a seat}
3639 CJK UNIFIED IDEOGRAPH-#3639, IRGKangXi=0227.271, Def{(non-classical form of 垐) to spread soil on the road, pave the road with soil, (ancient form of 堲) sickness; illness; disease}
363A CJK UNIFIED IDEOGRAPH-#363A, IRGKangXi=0228.291, Def{cultivated land; to till lands, a clod of earth}
363B CJK UNIFIED IDEOGRAPH-#363B, IRGKangXi=0228.291, Def{to stop up; to close against; to restrain, to dam a stream and change its direction, to conceal a mound}
363C CJK UNIFIED IDEOGRAPH-#363C, IRGKangXi=0228.291
363D CJK UNIFIED IDEOGRAPH-#363D, IRGKangXi=0228.291, Def{to cultivate, plant; to care for plants}
363E CJK UNIFIED IDEOGRAPH-#363E, IRGKangXi=0228.291
363F CJK UNIFIED IDEOGRAPH-#363F, IRGKangXi=0228.300, Def{(same as 涅) to fill up, as a hole, black mud; slime, to stop up; to block; to cork}
3640 CJK UNIFIED IDEOGRAPH-#3640, IRGKangXi=0229.020, Def{a vegetable-plot, to plough; to till lands, among dikes and fields}
3641 CJK UNIFIED IDEOGRAPH-#3641, IRGKangXi=0229.140, Def{dust; dirt; a smear, (same as 牧) name of a place in old times}
3642 CJK UNIFIED IDEOGRAPH-#3642, IRGKangXi=0230.100, Def{wall; fence, (interchangeable with 兑) a hole}
3643 CJK UNIFIED IDEOGRAPH-#3643, IRGKangXi=0230.110, Def{a cavity; hole of a mound or an altar of earth}
3644 CJK UNIFIED IDEOGRAPH-#3644, IRGKangXi=0230.121, Def{below; under; low, (same as 陛) the steps to the throne, to ascend steps}
3645 CJK UNIFIED IDEOGRAPH-#3645, IRGKangXi=0230.121, Def{(same as 堡) a walled village, an earth-work; a petty military station, subdivision of a township; a ward in a city}
3646 CJK UNIFIED IDEOGRAPH-#3646, IRGKangXi=0230.121
3647 CJK UNIFIED IDEOGRAPH-#3647, IRGKangXi=0230.210, Def{place trampled by cattle and horses}
3648 CJK UNIFIED IDEOGRAPH-#3648, IRGKangXi=0230.240, Def{name of a place in Shanxi province}
3649 CJK UNIFIED IDEOGRAPH-#3649, IRGKangXi=0230.270, Def{(same as 腆) prosperous; affluence, good; virtuous, to make strong -- as liquors}
364A CJK UNIFIED IDEOGRAPH-#364A, IRGKangXi=0231.010, Def{(same as 場) an area of level ground; an open space, a threshing floor, arena for drill, etc. a place, to pile a sand-hills}
364B CJK UNIFIED IDEOGRAPH-#364B, IRGKangXi=0232.181
364C CJK UNIFIED IDEOGRAPH-#364C, IRGKangXi=0232.181, Def{(a simplified form)}
364D CJK UNIFIED IDEOGRAPH-#364D, IRGKangXi=0232.181, Def{name of a place in today's Guangdong Province}
364E CJK UNIFIED IDEOGRAPH-#364E, IRGKangXi=0233.100, Def{breach of a tank; pond, (a dialect) to cover up; to conceal; to hide; to bury}
364F CJK UNIFIED IDEOGRAPH-#364F, IRGKangXi=0233.170, Def{caverns; a grotto, a cellar, a cave-dwelling, to make a cave -- for dwelling}
3650 CJK UNIFIED IDEOGRAPH-#3650, IRGKangXi=0233.230, Def{to move, to hang down; to let fall, (same as 垛) a target, a battlement, to add up; to pile up}
3651 CJK UNIFIED IDEOGRAPH-#3651, IRGKangXi=0234.130, Def{(same as 宇) the canopy of heaven; space, to cover; to shelter}
3652 CJK UNIFIED IDEOGRAPH-#3652, IRGKangXi=0234.200, Def{(same as 野) wild; uncultivated; a wilderness, rustic, savage}
3653 CJK UNIFIED IDEOGRAPH-#3653, IRGKangXi=0234.271, Def{earth; land; soil; ground; territory, opium}
3654 CJK UNIFIED IDEOGRAPH-#3654, IRGKangXi=0234.271, Def{(non-classical form of 韓) a fence, name of a star, a small feudal State, old name for Korea}
3655 CJK UNIFIED IDEOGRAPH-#3655, IRGKangXi=0234.271, Def{(same as 塊) a lump, a piece of, a fraction}
3656 CJK UNIFIED IDEOGRAPH-#3656, IRGKangXi=0234.271
3657 CJK UNIFIED IDEOGRAPH-#3657, IRGKangXi=0234.271, Def{(same as 塊) a lump of earth; a lump, rugged; uneven}
3658 CJK UNIFIED IDEOGRAPH-#3658, IRGKangXi=0234.271, Def{used in naming a place}
3659 CJK UNIFIED IDEOGRAPH-#3659, IRGKangXi=0234.290, Def{to smear; to daub, to erase; to blot out, mud, mire, to plaster a wall with mud}
365A CJK UNIFIED IDEOGRAPH-#365A, IRGKangXi=0235.030, Def{red colored hard and solid mud (soil; clay; earth)}
365B CJK UNIFIED IDEOGRAPH-#365B, IRGKangXi=0235.190, Def{a person's name in old times}
365C CJK UNIFIED IDEOGRAPH-#365C, IRGKangXi=0236.341, Def{(ancient and corrupted form of U+81FA 臺) a lookout, a tower, a terrace, a platform, a stage}
365D CJK UNIFIED IDEOGRAPH-#365D, IRGKangXi=0236.110, Def{a dike; a barrier; an embankment}
365E CJK UNIFIED IDEOGRAPH-#365E, IRGKangXi=0236.130, Def{a small mountain; a hill}
365F CJK UNIFIED IDEOGRAPH-#365F, IRGKangXi=0238.191, Def{(a dialect) low-lying (ground) on the side of a river or lake, (a variant 塱) name of a place in Guangdong}
3660 CJK UNIFIED IDEOGRAPH-#3660, IRGKangXi=0237.010, Def{dust; dirt; a smear, (same as 曀) the sun hidden by clouds, obscure}
3661 CJK UNIFIED IDEOGRAPH-#3661, IRGKangXi=0237.040, Def{a kind of edible mushroom}
3662 CJK UNIFIED IDEOGRAPH-#3662, IRGKangXi=0237.070, Def{(same as 墁 鏝) a trowel; a plaster; a trowel for plastering}
3663 CJK UNIFIED IDEOGRAPH-#3663, IRGKangXi=0237.190, Def{to pile a sand-hills}
3664 CJK UNIFIED IDEOGRAPH-#3664, IRGKangXi=0237.230, Def{(same as 罅) a crack in earthenware, a split; rip or break apart; thus -- a grudge}
3665 CJK UNIFIED IDEOGRAPH-#3665, IRGKangXi=0238.190, Def{muddy}
3666 CJK UNIFIED IDEOGRAPH-#3666, IRGKangXi=0238.191
3667 CJK UNIFIED IDEOGRAPH-#3667, IRGKangXi=0238.210, Def{(same as 塯) earthenware; a kind of rice container}
3668 CJK UNIFIED IDEOGRAPH-#3668, IRGKangXi=0238.240, Def{(same as 畿) the royal domains set apart in ancient times for the emperor}
3669 CJK UNIFIED IDEOGRAPH-#3669, IRGKangXi=0238.270, Def{an enclosing wall}
366A CJK UNIFIED IDEOGRAPH-#366A, IRGKangXi=0238.300, Def{dark; the sun hidden by clouds, obscure}
366B CJK UNIFIED IDEOGRAPH-#366B, IRGKangXi=0239.090, Def{a fountain or spring out from the underground}
366C CJK UNIFIED IDEOGRAPH-#366C, IRGKangXi=0239.170, Def{(ancient form of 垠) a bank; a boundary}
366D CJK UNIFIED IDEOGRAPH-#366D, IRGKangXi=0239.241
366E CJK UNIFIED IDEOGRAPH-#366E, IRGKangXi=0239.241, Def{a tower, pagoda}
366F CJK UNIFIED IDEOGRAPH-#366F, IRGKangXi=0239.241, Def{(same as 藝) skill, ability in handicraft, a craft, an art, a calling, a trade, an accomplishment}
3670 CJK UNIFIED IDEOGRAPH-#3670, IRGKangXi=0240.070, Def{name of a place in ancient times}
3671 CJK UNIFIED IDEOGRAPH-#3671, IRGKangXi=0240.090, Def{earthenware used to bake cakes in Northern China}
3672 CJK UNIFIED IDEOGRAPH-#3672, IRGKangXi=0240.190, Def{(same as 壅) to stop up; to obstruct}
3673 CJK UNIFIED IDEOGRAPH-#3673, IRGKangXi=0240.200, Def{ruggedness of the road; difficulties of the way; bad luck, unable to reach one's aim}
3674 CJK UNIFIED IDEOGRAPH-#3674, IRGKangXi=0240.230, Def{to conceal; to hide, to shade; to darken}
3675 CJK UNIFIED IDEOGRAPH-#3675, IRGKangXi=0240.290, Def{(ancient form of 臺) a lookout, a tower, a terrace, a platform; a stage}
3676 CJK UNIFIED IDEOGRAPH-#3676, IRGKangXi=0240.300, Def{(large seal type of 堂) a hall; a reception room; a meeting-place, a court of justice}
3677 CJK UNIFIED IDEOGRAPH-#3677, IRGKangXi=0240.320, Def{low-lying area, to accumulate soil; to store up soil}
3678 CJK UNIFIED IDEOGRAPH-#3678, IRGKangXi=0241.020, Def{(non-classical form 墣) a clod of earth; a lump of earth; a lump}
3679 CJK UNIFIED IDEOGRAPH-#3679, IRGKangXi=0241.150, Def{a damp; a marsh, (interchangeable 夢)}
367A CJK UNIFIED IDEOGRAPH-#367A, IRGKangXi=0241.160, Def{(same as 逵 馗) cross-road; a thorough-fare; a center from which nine roads lead out, to accumulate soil, to store up soil}
367B CJK UNIFIED IDEOGRAPH-#367B, IRGKangXi=0241.220, Def{(non-classical form of 廛) living space for one family in ancient times, ground allotted to a retainer, a shop; a market place}
367C CJK UNIFIED IDEOGRAPH-#367C, IRGKangXi=0241.281, Def{(same as 壘) a military wall, a rampart, to pile up, a pile}
367D CJK UNIFIED IDEOGRAPH-#367D, IRGKangXi=0241.281
367E CJK UNIFIED IDEOGRAPH-#367E, IRGKangXi=0242.110, Def{(same as 嶨) hills with lots of big rocks}
367F CJK UNIFIED IDEOGRAPH-#367F, IRGKangXi=0242.141
3680 CJK UNIFIED IDEOGRAPH-#3680, IRGKangXi=0242.150, Def{lofty, dangerous; critical, (same as 虧) to destroy; to injure; to damage}
3681 CJK UNIFIED IDEOGRAPH-#3681, IRGKangXi=0242.350, Def{the long tail feathers which curl up; to elevate; to raise the head; warped, (interchangeable 翹) long tail feathers, to raise, outstanding}
3682 CJK UNIFIED IDEOGRAPH-#3682, IRGKangXi=0242.360, Def{dust; dirt, a cellar, a caverns}
3683 CJK UNIFIED IDEOGRAPH-#3683, IRGKangXi=0243.160, Def{to take a hint; a group of Chinese characters the construction of which suggests the meaning, the fate; good luck or bad are in the pot; don't expose; or leak out}
3684 CJK UNIFIED IDEOGRAPH-#3684, IRGKangXi=0244.111
3685 CJK UNIFIED IDEOGRAPH-#3685, IRGKangXi=0244.240, Def{to worship of God, to praise; many; much}
3686 CJK UNIFIED IDEOGRAPH-#3686, IRGKangXi=0245.080, Def{to do something in an old way; use the old method; to go to the old road}
3687 CJK UNIFIED IDEOGRAPH-#3687, IRGKangXi=0245.100, RSKangXi=35.6, Def{draw the feet up; to refuse to advance, to summarize; to gather; to collect, name of an old country, (same as U+9350 鍐) a headstall; ornament on a bridle; (Cant.) to jump up}
3688 CJK UNIFIED IDEOGRAPH-#3688, IRGKangXi=0246.061
3689 CJK UNIFIED IDEOGRAPH-#3689, IRGKangXi=0247.020, Def{(corrupted form) to make a profit on sell and buy; the profit in business, (interchangeable 嬴) a overplus; gain; profit; abundance}
368A CJK UNIFIED IDEOGRAPH-#368A, IRGKangXi=0247.140, Def{much; many; numerous, great; big; vast}
368B CJK UNIFIED IDEOGRAPH-#368B, IRGKangXi=0247.210, Def{much; many; numerous, great; big; vast}
368C CJK UNIFIED IDEOGRAPH-#368C, IRGKangXi=0247.230, Def{(same as 夥) a band; a company; a partner; an assistant}
368D CJK UNIFIED IDEOGRAPH-#368D, IRGKangXi=0247.260, Def{much; many; numerous, to get more profit, to estimate more}
368E CJK UNIFIED IDEOGRAPH-#368E, IRGKangXi=0249.111
368F CJK UNIFIED IDEOGRAPH-#368F, IRGKangXi=0249.120, Def{(corrupted form) to scatter; to disperse, to give the reins to; to allow to run wild; unstable; light; featherbrained}
3690 CJK UNIFIED IDEOGRAPH-#3690, IRGKangXi=0250.071, Def{(same as 套) a case; a wrapper; a covering; a snare, to encase; to slip over}
3691 CJK UNIFIED IDEOGRAPH-#3691, IRGKangXi=0250.071
3692 CJK UNIFIED IDEOGRAPH-#3692, IRGKangXi=0250.120, Def{concealed the stolen goods in one's dress}
3693 CJK UNIFIED IDEOGRAPH-#3693, IRGKangXi=0250.140, Def{tie-beams of a small boat}
3694 CJK UNIFIED IDEOGRAPH-#3694, IRGKangXi=0250.240, Def{instrument of torture used in ancient times, loud, an evil doer all the times, (same as 幸) to rejoice, fortunate; prosperous}
3695 CJK UNIFIED IDEOGRAPH-#3695, IRGKangXi=0250.250, Def{great; big; vast, (same as 弼) to aid; to assist; as in the government, (interchangeable 佛) Buddha}
3696 CJK UNIFIED IDEOGRAPH-#3696, IRGKangXi=0250.280, Def{(said of truth) to come out to the open; to be known by all, (ancient form 澤) glossy; shining}
3697 CJK UNIFIED IDEOGRAPH-#3697, IRGKangXi=0250.290, Def{with a big mouth, great; big; vast}
3698 CJK UNIFIED IDEOGRAPH-#3698, IRGKangXi=0250.300, Def{(interchangeable 伴) a partner; a companion, (same as 赫) bright; luminous, glorious (same as 扶) to support; to prop up; to help}
3699 CJK UNIFIED IDEOGRAPH-#3699, IRGKangXi=0251.091
369A CJK UNIFIED IDEOGRAPH-#369A, IRGKangXi=0251.140, Def{big; great; vast, very, liberal, bark (as a dog)}
369B CJK UNIFIED IDEOGRAPH-#369B, IRGKangXi=0252.051, Def{corpulent, abundant; flourishing; plenteous}
369C CJK UNIFIED IDEOGRAPH-#369C, IRGKangXi=0252.210, Def{with great power; vigorous; very strong; with great force, (same as 域) a frontier; a boundary; region, great; big; vast}
369D CJK UNIFIED IDEOGRAPH-#369D, IRGKangXi=0253.040, Def{(same as 奎) the stride made by a man, name of a star, a kind of tree}
369E CJK UNIFIED IDEOGRAPH-#369E, IRGKangXi=0253.041, Def{terror; horror; fear}
369F CJK UNIFIED IDEOGRAPH-#369F, IRGKangXi=0253.060, Def{(non-classical form) a kind of animal; with head looks like rabbit and feet deer, small rabbit}
36A0 CJK UNIFIED IDEOGRAPH-#36A0, IRGKangXi=0253.160, Def{full grown; fat; plump, fertile; rich, big and clumsy}
36A1 CJK UNIFIED IDEOGRAPH-#36A1, IRGKangXi=0254.151, Def{standing without a straight back}
36A2 CJK UNIFIED IDEOGRAPH-#36A2, IRGKangXi=0255.010, Def{(ancient form of 奴) a slave; a servant, term of depreciation, (ancient form of 侮) to insult; to ridicule}
36A3 CJK UNIFIED IDEOGRAPH-#36A3, IRGKangXi=0255.060, Def{(ancient form of U+59E3 姣 U+5B0C 嬌) handsome; pretty, cunning, meretricious allurements, to insult; to disgrace, licentious; profligate; lewd; (Cant.) amorous, in love with}
36A4 CJK UNIFIED IDEOGRAPH-#36A4, IRGKangXi=0255.100, Def{a lady officer of the monarch's palace in old times}
36A5 CJK UNIFIED IDEOGRAPH-#36A5, IRGKangXi=0255.200, Def{to become associated with each other in good manners and politeness}
36A6 CJK UNIFIED IDEOGRAPH-#36A6, IRGKangXi=0256.030, Def{(ancient form of U+59EC 姬) a charming girl, a charming concubine; a monarch's wife}
36A7 CJK UNIFIED IDEOGRAPH-#36A7, IRGKangXi=0256.040, Def{(same as 嬐) agile; adroit fast; quick; prompt, neat; tidy; orderly, to raise the head; to look up}
36A8 CJK UNIFIED IDEOGRAPH-#36A8, IRGKangXi=0256.041, Def{used in girl's name}
36A9 CJK UNIFIED IDEOGRAPH-#36A9, IRGKangXi=0256.110, Def{with a charming sprightly carriage (said of a woman)}
36AA CJK UNIFIED IDEOGRAPH-#36AA, IRGKangXi=0256.130, Def{to make a deceptive show of power, not really strict}
36AB CJK UNIFIED IDEOGRAPH-#36AB, IRGKangXi=0256.250, Def{used in girl's name, a woman's countenance; features}
36AC CJK UNIFIED IDEOGRAPH-#36AC, IRGKangXi=0257.050, Def{a lady start to doll up}
36AD CJK UNIFIED IDEOGRAPH-#36AD, IRGKangXi=0257.101, Def{used in girl's name}
36AE CJK UNIFIED IDEOGRAPH-#36AE, IRGKangXi=0257.101, Def{(non-classical form 妉,媅) happy; pleased, to laugh}
36AF CJK UNIFIED IDEOGRAPH-#36AF, IRGKangXi=0257.101, Def{(simplified form) last name}
36B0 CJK UNIFIED IDEOGRAPH-#36B0, IRGKangXi=0257.110, Def{(same as 胚) an embryo; a foetus a month old}
36B1 CJK UNIFIED IDEOGRAPH-#36B1, IRGKangXi=0257.260, Def{used in girl's name}
36B2 CJK UNIFIED IDEOGRAPH-#36B2, IRGKangXi=0257.380, Def{small and weak, used in girl's name, a woman's feature; lady's face}
36B3 CJK UNIFIED IDEOGRAPH-#36B3, IRGKangXi=0257.410, Def{a lady teacher to teach the proper rules of female behavior in ancient times}
36B4 CJK UNIFIED IDEOGRAPH-#36B4, IRGKangXi=0258.060, Def{a pretty girl; a beauty; a belle, handsome; female beauty}
36B5 CJK UNIFIED IDEOGRAPH-#36B5, IRGKangXi=0258.080, Def{used in girl's name}
36B6 CJK UNIFIED IDEOGRAPH-#36B6, IRGKangXi=0258.100, Def{(same as 姒) wife of one's husband's elder brother; (in ancient China) the elder of twins; a Chinese family name, (same as 姬) a handsome girl; a charming girl; a concubine; a Chinese family name}
36B7 CJK UNIFIED IDEOGRAPH-#36B7, IRGKangXi=0258.120, Def{(same as 嬭) the breasts of a woman; milk; a term of respect for women; grandma, one's elder sister or sisters, used for a girl's name}
36B8 CJK UNIFIED IDEOGRAPH-#36B8, IRGKangXi=0259.010, Def{used in girl's name}
36B9 CJK UNIFIED IDEOGRAPH-#36B9, IRGKangXi=0259.090, Def{(same as 媌) pretty; charming girl, (same as 懰) exquisite; fine}
36BA CJK UNIFIED IDEOGRAPH-#36BA, IRGKangXi=0259.110, Def{(non-classical form of 孕) to be pregnant; to conceive}
36BB CJK UNIFIED IDEOGRAPH-#36BB, IRGKangXi=0259.150, Def{sodomy; to bugger}
36BC CJK UNIFIED IDEOGRAPH-#36BC, IRGKangXi=0259.151, Def{(non-classical form of 妞) a girl; a little girl}
36BD CJK UNIFIED IDEOGRAPH-#36BD, IRGKangXi=0259.151, Def{used in girl's name}
36BE CJK UNIFIED IDEOGRAPH-#36BE, IRGKangXi=0259.151, Def{to have fun; to sport; to play; to frolic}
36BF CJK UNIFIED IDEOGRAPH-#36BF, IRGKangXi=0259.151, Def{a goddess's name in legend; the sister and successor of Fu Xi 伏羲, (interchangeable 庖), last name}
36C0 CJK UNIFIED IDEOGRAPH-#36C0, IRGKangXi=0259.151
36C1 CJK UNIFIED IDEOGRAPH-#36C1, IRGKangXi=0259.151, Def{quick; urgent; anxious, loyal sincere and respectful}
36C2 CJK UNIFIED IDEOGRAPH-#36C2, IRGKangXi=0260.030, Def{weak and small}
36C3 CJK UNIFIED IDEOGRAPH-#36C3, IRGKangXi=0260.140, Def{clean; pure; clear, usually used for names}
36C4 CJK UNIFIED IDEOGRAPH-#36C4, IRGKangXi=0260.160, Def{used in girl's name}
36C5 CJK UNIFIED IDEOGRAPH-#36C5, IRGKangXi=0261.050, Def{used in girl's name}
36C6 CJK UNIFIED IDEOGRAPH-#36C6, IRGKangXi=0261.270, Def{to speculate; to conjecture; to assume; to make an intelligent guess; to fathom, (of a woman) beautiful, used in girl's name}
36C7 CJK UNIFIED IDEOGRAPH-#36C7, IRGKangXi=0261.321
36C8 CJK UNIFIED IDEOGRAPH-#36C8, IRGKangXi=0261.321
36C9 CJK UNIFIED IDEOGRAPH-#36C9, IRGKangXi=0261.321
36CA CJK UNIFIED IDEOGRAPH-#36CA, IRGKangXi=0261.321, Def{to speculate; to conjecture; to assume; to make an intelligent guess; to fathom, (of a woman) beautiful, used in girl's name}
36CB CJK UNIFIED IDEOGRAPH-#36CB, IRGKangXi=0261.321
36CC CJK UNIFIED IDEOGRAPH-#36CC, IRGKangXi=0261.321
36CD CJK UNIFIED IDEOGRAPH-#36CD, IRGKangXi=0261.340, Def{to have one's ambition fulfilled; to be successful in one's career; (same as 愜) pleasing; satisfying; to gratify or be gratified, undignified; improper}
36CE CJK UNIFIED IDEOGRAPH-#36CE, IRGKangXi=0261.350, Def{an ugly woman}
36CF CJK UNIFIED IDEOGRAPH-#36CF, IRGKangXi=0261.390, Def{used in girl's name}
36D0 CJK UNIFIED IDEOGRAPH-#36D0, IRGKangXi=0262.050, Def{(non-classical form of 嫂) wife of one's elder's brother}
36D1 CJK UNIFIED IDEOGRAPH-#36D1, IRGKangXi=0262.110, Def{three women, beautiful things}
36D2 CJK UNIFIED IDEOGRAPH-#36D2, IRGKangXi=0262.120, Def{can not speak}
36D3 CJK UNIFIED IDEOGRAPH-#36D3, IRGKangXi=0262.200, Def{used in girl's name}
36D4 CJK UNIFIED IDEOGRAPH-#36D4, IRGKangXi=0262.220, Def{(same as 妦) exquisite; fine; (said of a woman's figure) very full and voluptuous; buxom, used in girl's name}
36D5 CJK UNIFIED IDEOGRAPH-#36D5, IRGKangXi=0262.240, Def{womanly, (same as 姶 ancient form) exquisite; fine, used in girl's name, clever; bright; nimble, cute; lovable}
36D6 CJK UNIFIED IDEOGRAPH-#36D6, IRGKangXi=0262.250, Def{used in girl's name}
36D7 CJK UNIFIED IDEOGRAPH-#36D7, IRGKangXi=0262.280, Def{frivolous; flippant; disrespectful; irreverent; to insult, young, used in girl's name}
36D8 CJK UNIFIED IDEOGRAPH-#36D8, IRGKangXi=0262.300, Def{used in girl's name, a fat woman}
36D9 CJK UNIFIED IDEOGRAPH-#36D9, IRGKangXi=0262.310, Def{used in girl's name, (same as 妾) a concubine, (in old China) a polite term used by a woman to refer to herself when speaking to her husband}
36DA CJK UNIFIED IDEOGRAPH-#36DA, IRGKangXi=0262.320, Def{equal; uniform, used in girl's name}
36DB CJK UNIFIED IDEOGRAPH-#36DB, IRGKangXi=0263.020, Def{(same as U+5A20 娠) pregnant, to let develop; to nourish; to foster; to nurture, slave girls}
36DC CJK UNIFIED IDEOGRAPH-#36DC, IRGKangXi=0263.030, Def{used in girl's name}
36DD CJK UNIFIED IDEOGRAPH-#36DD, IRGKangXi=0263.230, Def{used in girl's name}
36DE CJK UNIFIED IDEOGRAPH-#36DE, IRGKangXi=0263.251, Def{used in girl's name}
36DF CJK UNIFIED IDEOGRAPH-#36DF, IRGKangXi=0263.251
36E0 CJK UNIFIED IDEOGRAPH-#36E0, IRGKangXi=0263.251
36E1 CJK UNIFIED IDEOGRAPH-#36E1, IRGKangXi=0263.251
36E2 CJK UNIFIED IDEOGRAPH-#36E2, IRGKangXi=0263.251
36E3 CJK UNIFIED IDEOGRAPH-#36E3, IRGKangXi=0263.251
36E4 CJK UNIFIED IDEOGRAPH-#36E4, IRGKangXi=0263.251
36E5 CJK UNIFIED IDEOGRAPH-#36E5, IRGKangXi=0263.280, Def{to prostrate; to make obeisance, to concede or submit willingly}
36E6 CJK UNIFIED IDEOGRAPH-#36E6, IRGKangXi=0263.410, Def{good; nice; fine; excellent}
36E7 CJK UNIFIED IDEOGRAPH-#36E7, IRGKangXi=0263.420, Def{name of a county in today's Shanxi province}
36E8 CJK UNIFIED IDEOGRAPH-#36E8, IRGKangXi=0264.090, Def{(corrupted form) chaos; anarchy; distractions; confusion}
36E9 CJK UNIFIED IDEOGRAPH-#36E9, IRGKangXi=0264.100, Def{(same as 妯) sisters-in-law (a reference among wives of brothers)}
36EA CJK UNIFIED IDEOGRAPH-#36EA, IRGKangXi=0264.320, Def{to slander; to libel, a maidservant, used in girl's name, womanly}
36EB CJK UNIFIED IDEOGRAPH-#36EB, IRGKangXi=0265.210, Def{used in girl's name}
36EC CJK UNIFIED IDEOGRAPH-#36EC, IRGKangXi=0265.240, Def{used in girl's name}
36ED CJK UNIFIED IDEOGRAPH-#36ED, IRGKangXi=0265.280, Def{used in girl's name}
36EE CJK UNIFIED IDEOGRAPH-#36EE, IRGKangXi=0266.020, Def{(same as 嫂) wife of one's elder brother}
36EF CJK UNIFIED IDEOGRAPH-#36EF, IRGKangXi=0266.040, Def{(same as 嬔) to soar; to fly; to roam (same as 娩) complaisant; agreeable, a pair; a couple; persons or things that come in pairs}
36F0 CJK UNIFIED IDEOGRAPH-#36F0, IRGKangXi=0266.050, Def{(same as 婚) to marry a wife; a bride-groom}
36F1 CJK UNIFIED IDEOGRAPH-#36F1, IRGKangXi=0266.070, Def{exquisite; fine, used in girl's name}
36F2 CJK UNIFIED IDEOGRAPH-#36F2, IRGKangXi=0266.130, Def{(same as 髮) the hair on the human head, hairbreadth; hair's-breadth}
36F3 CJK UNIFIED IDEOGRAPH-#36F3, IRGKangXi=0266.150, Def{an abnormal unborn baby}
36F4 CJK UNIFIED IDEOGRAPH-#36F4, IRGKangXi=0266.190, Def{(same as 惱) to anger to annoy; to irritate}
36F5 CJK UNIFIED IDEOGRAPH-#36F5, IRGKangXi=0266.200, Def{used in girl's name}
36F6 CJK UNIFIED IDEOGRAPH-#36F6, IRGKangXi=0266.320, Def{do not care about appearance and deportment}
36F7 CJK UNIFIED IDEOGRAPH-#36F7, IRGKangXi=0266.420, Def{used in girl's name}
36F8 CJK UNIFIED IDEOGRAPH-#36F8, IRGKangXi=0266.430, Def{(corrupted form) used in girl's name}
36F9 CJK UNIFIED IDEOGRAPH-#36F9, IRGKangXi=0267.120, Def{pretty; exquisite; fine}
36FA CJK UNIFIED IDEOGRAPH-#36FA, IRGKangXi=0267.150, Def{not clean; impure}
36FB CJK UNIFIED IDEOGRAPH-#36FB, IRGKangXi=0267.180, Def{womanly; feminine}
36FC CJK UNIFIED IDEOGRAPH-#36FC, IRGKangXi=0267.260, Def{to interrupt; to put in a word, lacking in courage; cowardly, nervous; socially timid, womanly; feminine}
36FD CJK UNIFIED IDEOGRAPH-#36FD, IRGKangXi=0267.300, Def{(ancient form of 璈) to walk slowly like a woman, to ridicule; to laugh at}
36FE CJK UNIFIED IDEOGRAPH-#36FE, IRGKangXi=0267.361, Def{(said of a woman) not clean; impure}
36FF CJK UNIFIED IDEOGRAPH-#36FF, IRGKangXi=0267.361, Def{cunning; wicked, spoiled; coddled}
3700 CJK UNIFIED IDEOGRAPH-#3700, IRGKangXi=0267.361
3701 CJK UNIFIED IDEOGRAPH-#3701, IRGKangXi=0267.361
3702 CJK UNIFIED IDEOGRAPH-#3702, IRGKangXi=0267.361, Def{used in girl's name}
3703 CJK UNIFIED IDEOGRAPH-#3703, IRGKangXi=0267.361, Def{from; by; whence, to undertake; to manage, to follow}
3704 CJK UNIFIED IDEOGRAPH-#3704, IRGKangXi=0267.361, Def{last name (refer to 千家姓)}
3705 CJK UNIFIED IDEOGRAPH-#3705, IRGKangXi=0267.380, Def{to fawn on; to flatter, to love, attractive, jealous; envy; jealousy}
3706 CJK UNIFIED IDEOGRAPH-#3706, IRGKangXi=0268.010, Def{commonly known as mother in Wu}
3707 CJK UNIFIED IDEOGRAPH-#3707, IRGKangXi=0268.020, Def{unhappy, girlish}
3708 CJK UNIFIED IDEOGRAPH-#3708, IRGKangXi=0268.040, Def{a beautiful face (of a woman); pretty}
3709 CJK UNIFIED IDEOGRAPH-#3709, IRGKangXi=0268.130, Def{womanly, used in girl's name}
370A CJK UNIFIED IDEOGRAPH-#370A, IRGKangXi=0268.240, Def{exquisite; fine}
370B CJK UNIFIED IDEOGRAPH-#370B, IRGKangXi=0268.300, Def{used in girl's name}
370C CJK UNIFIED IDEOGRAPH-#370C, IRGKangXi=0269.030, Def{a different name for breasts, to breast-feed; to feed a baby with milk, stingy; niggardly; miserly}
370D CJK UNIFIED IDEOGRAPH-#370D, IRGKangXi=0269.070, Def{used in girl's name}
370E CJK UNIFIED IDEOGRAPH-#370E, IRGKangXi=0269.080, Def{slave girls, lacking in courage; nervous, a jealous woman}
370F CJK UNIFIED IDEOGRAPH-#370F, IRGKangXi=0269.100, Def{(the large seal type 妘) last name, used in girl's name}
3710 CJK UNIFIED IDEOGRAPH-#3710, IRGKangXi=0269.110, Def{used in girl's name}
3711 CJK UNIFIED IDEOGRAPH-#3711, IRGKangXi=0269.130, Def{(same as 婆) an older woman, mother of one's husband, one's grandmother}
3712 CJK UNIFIED IDEOGRAPH-#3712, IRGKangXi=0269.180, Def{maternal grandmother, a midwife, an old woman}
3713 CJK UNIFIED IDEOGRAPH-#3713, IRGKangXi=0269.181, Def{womanly}
3714 CJK UNIFIED IDEOGRAPH-#3714, IRGKangXi=0269.181
3715 CJK UNIFIED IDEOGRAPH-#3715, IRGKangXi=0270.311, Def{(same as 連) relatives through marriage}
3716 CJK UNIFIED IDEOGRAPH-#3716, IRGKangXi=0269.300, Def{exquisite; fine}
3717 CJK UNIFIED IDEOGRAPH-#3717, IRGKangXi=0269.350, Def{covetousness; greed; cupidity; avarice, licentious; obscene; to seduce}
3718 CJK UNIFIED IDEOGRAPH-#3718, IRGKangXi=0270.020, Def{delicate; lovely; beautiful, spoiled, jealous; to envy; jealousy, used in girl's name}
3719 CJK UNIFIED IDEOGRAPH-#3719, IRGKangXi=0270.120, Def{used in girl's name}
371A CJK UNIFIED IDEOGRAPH-#371A, IRGKangXi=0270.140, Def{a local government post in ancient times}
371B CJK UNIFIED IDEOGRAPH-#371B, IRGKangXi=0270.190, Def{(non-classical form of 媆) soft; gentle, attractive (same as 嫩) soft and tender, delicate, weak}
371C CJK UNIFIED IDEOGRAPH-#371C, IRGKangXi=0270.250, Def{(ancient form form of 傲) proud; haughty; overbearing rude}
371D CJK UNIFIED IDEOGRAPH-#371D, IRGKangXi=0270.260, Def{to cherish anger, hard to understand, pretty, dignified; solemn, to make sheep eyes or passes; to converse with eyes}
371E CJK UNIFIED IDEOGRAPH-#371E, IRGKangXi=0270.290, Def{beautiful; pretty, used in girl's name}
371F CJK UNIFIED IDEOGRAPH-#371F, IRGKangXi=0271.351
3720 CJK UNIFIED IDEOGRAPH-#3720, IRGKangXi=0270.311, Def{used in girl's name}
3721 CJK UNIFIED IDEOGRAPH-#3721, IRGKangXi=0270.311, Def{used in girl's name}
3722 CJK UNIFIED IDEOGRAPH-#3722, IRGKangXi=0270.311, Def{a bad term by which one addresses a woman}
3723 CJK UNIFIED IDEOGRAPH-#3723, IRGKangXi=0270.360, Def{a last name, carriage; deportment; bearing; poise (of a woman)}
3724 CJK UNIFIED IDEOGRAPH-#3724, IRGKangXi=0270.370, Def{fooling around, weak, (said of a woman's figure) slender; thin and long, used in girl's name}
3725 CJK UNIFIED IDEOGRAPH-#3725, IRGKangXi=0271.090, Def{crafty; artful}
3726 CJK UNIFIED IDEOGRAPH-#3726, IRGKangXi=0271.230, Def{used in girl's name}
3727 CJK UNIFIED IDEOGRAPH-#3727, IRGKangXi=0271.351, Def{plump, womanly}
3728 CJK UNIFIED IDEOGRAPH-#3728, IRGKangXi=0271.351, Def{beautiful; pretty}
3729 CJK UNIFIED IDEOGRAPH-#3729, IRGKangXi=0271.370, Def{jealous; to envy; jealously}
372A CJK UNIFIED IDEOGRAPH-#372A, IRGKangXi=0272.160, Def{(same as 嫀) name of a family or a clan, name of country (in ancient times)}
372B CJK UNIFIED IDEOGRAPH-#372B, IRGKangXi=0272.251, Def{(same as 媄) pretty; beautiful}
372C CJK UNIFIED IDEOGRAPH-#372C, IRGKangXi=0272.251
372D CJK UNIFIED IDEOGRAPH-#372D, IRGKangXi=0272.251
372E CJK UNIFIED IDEOGRAPH-#372E, IRGKangXi=0272.450, Def{to go beyond normal limit; excessive; out of control, to covet; greedy, to be impolite, used in girl's name}
372F CJK UNIFIED IDEOGRAPH-#372F, IRGKangXi=0273.150, Def{a word to designate woman, a word of revile used to call a woman}
3730 CJK UNIFIED IDEOGRAPH-#3730, IRGKangXi=0273.220, Def{exquisite; fine}
3731 CJK UNIFIED IDEOGRAPH-#3731, IRGKangXi=0273.290, Def{used in girl's name, soft and tender}
3732 CJK UNIFIED IDEOGRAPH-#3732, IRGKangXi=0273.330, Def{(non-classical form of 嬴) to have surplus; full; an overplus, to open out; to produce, a family name}
3733 CJK UNIFIED IDEOGRAPH-#3733, IRGKangXi=0273.350, Def{peaceful; joyful}
3734 CJK UNIFIED IDEOGRAPH-#3734, IRGKangXi=0273.390, Def{(non-classical from of 瞢) dark and obscure, obscure; not bright, used in girl's name, good; fine}
3735 CJK UNIFIED IDEOGRAPH-#3735, IRGKangXi=0273.421, Def{(said of a woman) with a charming sprightly carriage; slim; delicate and beautiful}
3736 CJK UNIFIED IDEOGRAPH-#3736, IRGKangXi=0274.171, Def{a (married) couple}
3737 CJK UNIFIED IDEOGRAPH-#3737, IRGKangXi=0274.030, Def{milk, the breasts of a woman, to suckle, word of respect for women, (for Qi's people) mother, used in girl's name}
3738 CJK UNIFIED IDEOGRAPH-#3738, IRGKangXi=0274.060, Def{(non-classical form of 孽) sin; evil; retribution, the son of a concubine}
3739 CJK UNIFIED IDEOGRAPH-#3739, IRGKangXi=0274.150, Def{name of a folk song in ancient times, used in girl's name}
373A CJK UNIFIED IDEOGRAPH-#373A, IRGKangXi=0274.190, Def{white color; no respect; imprudent; (Cant.) interesting, good}
373B CJK UNIFIED IDEOGRAPH-#373B, IRGKangXi=0274.340, Def{(non-classical form of 孌) (interchangeable 戀) to admire; to remember (old days, etc.), exquisite; fine, to obey; obedient; to comply, (same as 奱) to bind; binding, used in girl's name}
373C CJK UNIFIED IDEOGRAPH-#373C, IRGKangXi=0274.350, Def{(same as 姪) children of one's brother -- nephews or nieces, I; me (when speaking to a family friend of father's generation)}
373D CJK UNIFIED IDEOGRAPH-#373D, IRGKangXi=0277.160, Def{(ancient form of 子) child, seed, 1st terrestrial branch}
373E CJK UNIFIED IDEOGRAPH-#373E, IRGKangXi=0278.130, Def{(same as 孩) a child; an infant, young; small}
373F CJK UNIFIED IDEOGRAPH-#373F, IRGKangXi=0278.150, Def{name of a fish in legend, spawn, or roe}
3740 CJK UNIFIED IDEOGRAPH-#3740, IRGKangXi=0279.121, Def{(same as 好) to like; to love; to be fond of}
3741 CJK UNIFIED IDEOGRAPH-#3741, IRGKangXi=0279.170, Def{(same as 嬛) lonely; solitary, exquisite; fine, to worship with reverence}
3742 CJK UNIFIED IDEOGRAPH-#3742, IRGKangXi=0279.220, Def{locust, larva of Chilo simplex; a kind of moth}
3743 CJK UNIFIED IDEOGRAPH-#3743, IRGKangXi=0279.240, Def{(same as U+5A29 娩) to bear a son; to give birth}
3744 CJK UNIFIED IDEOGRAPH-#3744, IRGKangXi=0280.100, Def{(ancient form of 純) pure, sincere; honest; faithful}
3745 CJK UNIFIED IDEOGRAPH-#3745, IRGKangXi=0280.190, Def{to give suck to, infant; baby, ignorant; stupid, prudent; cautious}
3746 CJK UNIFIED IDEOGRAPH-#3746, IRGKangXi=0280.261
3747 CJK UNIFIED IDEOGRAPH-#3747, IRGKangXi=0281.011, Def{(same as 鶉) quail}
3748 CJK UNIFIED IDEOGRAPH-#3748, IRGKangXi=0281.100, Def{(same as 孿) twin}
3749 CJK UNIFIED IDEOGRAPH-#3749, IRGKangXi=0281.111, Def{(same as 宁) space between the throne and the retiring room behind it, to store up; to save up, peaceful, used as a term of comparison}
374A CJK UNIFIED IDEOGRAPH-#374A, IRGKangXi=0281.190, Def{(same as 守) to guide; to watch, to wait, to keep}
374B CJK UNIFIED IDEOGRAPH-#374B, IRGKangXi=0281.191
374C CJK UNIFIED IDEOGRAPH-#374C, IRGKangXi=0281.230, Def{(same as 疚) prolonged illness, mental discomfort, to stay in one place for a long period}
374D CJK UNIFIED IDEOGRAPH-#374D, IRGKangXi=0282.101, Def{(simplified form of 寫) to write; to draw}
374E CJK UNIFIED IDEOGRAPH-#374E, IRGKangXi=0283.051, Def{(same as 定) to decide; to fix; to settle}
374F CJK UNIFIED IDEOGRAPH-#374F, IRGKangXi=0282.150, Def{(same as 介) alone; solitary, to live alone}
3750 CJK UNIFIED IDEOGRAPH-#3750, IRGKangXi=0282.160, Def{(ancient form of 容) face; expression, to contain; to hold; to pardon}
3751 CJK UNIFIED IDEOGRAPH-#3751, IRGKangXi=0282.210, Def{(ancient form of 旁) side, by the side of; nearby}
3752 CJK UNIFIED IDEOGRAPH-#3752, IRGKangXi=0284.070, Def{(ancient form of 居) to dwell; to occupy to inhabit, a Chinese family name}
3753 CJK UNIFIED IDEOGRAPH-#3753, IRGKangXi=0284.100, Def{corresponding; equivalent, considerable; to a great extent, appropriate, (same as 凹) a hollow, concave}
3754 CJK UNIFIED IDEOGRAPH-#3754, IRGKangXi=0284.190, Def{deep bottom; the southeast corner of a house}
3755 CJK UNIFIED IDEOGRAPH-#3755, IRGKangXi=0285.080, Def{(same as 寍 寧) peace; serenity, used as a term of comparison: -- rather; it is better, would that}
3756 CJK UNIFIED IDEOGRAPH-#3756, IRGKangXi=0285.100, Def{(corrupted form) right; fitting; proper; good, should; ought to; had better}
3757 CJK UNIFIED IDEOGRAPH-#3757, IRGKangXi=0285.130, Def{spaciously rooms, expansive; vast and boundless}
3758 CJK UNIFIED IDEOGRAPH-#3758, IRGKangXi=0286.030, Def{(non-classical form) harmony; peace, pleasant; agreeable; delightful}
3759 CJK UNIFIED IDEOGRAPH-#3759, IRGKangXi=0287.090, Def{(ancient form of 寅) the third of the twelve terrestrial branches, a fellow officer, horary sign (for period from 3 to 5 a.m.)}
375A CJK UNIFIED IDEOGRAPH-#375A, IRGKangXi=0287.101
375B CJK UNIFIED IDEOGRAPH-#375B, IRGKangXi=0287.160, Def{(ancient form of 宿) a halting place; to lodge for the night, to keep over night, to cherish, asleep and perching}
375C CJK UNIFIED IDEOGRAPH-#375C, IRGKangXi=0287.170, Def{night; dark; darkness, of death, to mourn, to lose; to deprived of}
375D CJK UNIFIED IDEOGRAPH-#375D, IRGKangXi=0288.120, Def{deep, profound, far, extremely}
375E CJK UNIFIED IDEOGRAPH-#375E, IRGKangXi=0288.130, Def{pretend to be delicate and beautiful, to hang or be hanged, to be in suspension; unfounded}
375F CJK UNIFIED IDEOGRAPH-#375F, IRGKangXi=0288.201, Def{(corrupted form) in contravention of; to offend (same as 冒) incautious, to risk, to put forth}
3760 CJK UNIFIED IDEOGRAPH-#3760, IRGKangXi=0288.201, Def{(same as 冥) dark; obscure; dim, stupidity, far and high, deep; profound, night}
3761 CJK UNIFIED IDEOGRAPH-#3761, IRGKangXi=0288.201, Def{(a variant of 最) very; exceedingly; most, superior}
3762 CJK UNIFIED IDEOGRAPH-#3762, IRGKangXi=0289.140, Def{(same as 宇) a house; a roof, look; appearance, space}
3763 CJK UNIFIED IDEOGRAPH-#3763, IRGKangXi=0289.070, Def{still; calm; silent, peaceful; harmonious; serene, quite}
3764 CJK UNIFIED IDEOGRAPH-#3764, IRGKangXi=0290.020, Def{night, the dark part of a room}
3765 CJK UNIFIED IDEOGRAPH-#3765, IRGKangXi=0290.040, Def{have a nightmare, sound sleep}
3766 CJK UNIFIED IDEOGRAPH-#3766, IRGKangXi=0290.090, Def{to accumulate; to amass; to store up, superior, handsome, refined, eminent}
3767 CJK UNIFIED IDEOGRAPH-#3767, IRGKangXi=0290.100, Def{(non-classical form of 穩) stable; steady; firm, sure; secure}
3768 CJK UNIFIED IDEOGRAPH-#3768, IRGKangXi=0290.121
3769 CJK UNIFIED IDEOGRAPH-#3769, IRGKangXi=0290.130, Def{spaciously rooms, emptiness}
376A CJK UNIFIED IDEOGRAPH-#376A, IRGKangXi=0290.140, Def{a slanting house, nightmare}
376B CJK UNIFIED IDEOGRAPH-#376B, IRGKangXi=0291.180, Def{shape of the sky}
376C CJK UNIFIED IDEOGRAPH-#376C, IRGKangXi=0291.181
376D CJK UNIFIED IDEOGRAPH-#376D, IRGKangXi=0291.200, Def{(non-classical form) to awake (from errors, illusions, etc. to come to one's sense, (interchangeable 惺) clever; wise, wavering; indecisive}
376E CJK UNIFIED IDEOGRAPH-#376E, IRGKangXi=0292.010, Def{to exile; to banish, beyond the borders}
376F CJK UNIFIED IDEOGRAPH-#376F, IRGKangXi=0292.020, Def{to sojourn; sojourn, a sojourner; a visitor}
3770 CJK UNIFIED IDEOGRAPH-#3770, IRGKangXi=0293.010, Def{unable to meet, empty room}
3771 CJK UNIFIED IDEOGRAPH-#3771, IRGKangXi=0293.190, Def{(standard form) a dream; to dream, visionary, stupid}
3772 CJK UNIFIED IDEOGRAPH-#3772, IRGKangXi=0293.290, Def{(same as 寢) to sleep; to rest, a tomb, a residence}
3773 CJK UNIFIED IDEOGRAPH-#3773, IRGKangXi=0293.321
3774 CJK UNIFIED IDEOGRAPH-#3774, IRGKangXi=0294.020, Def{(same as 刓) a round off; to trim}
3775 CJK UNIFIED IDEOGRAPH-#3775, IRGKangXi=0294.040, Def{(ancient form of U+5F97 得) to get, complacent, (same as U+7919 礙) to obstruct, to hinder, to stop progress, to injure, to offend, to concern, hindrance, restraint}
3776 CJK UNIFIED IDEOGRAPH-#3776, IRGKangXi=0294.150, Def{(ancient form of 得) to get, complacent}
3777 CJK UNIFIED IDEOGRAPH-#3777, IRGKangXi=0296.011
3778 CJK UNIFIED IDEOGRAPH-#3778, IRGKangXi=0297.160, Def{(same as 弁) a conical cap worn on ceremonious occasions in ancient times, low-ranking military officers, a Chinese family name}
3779 CJK UNIFIED IDEOGRAPH-#3779, IRGKangXi=0297.200, Def{(corrupted form) a hare, a small rabbit, a family name}
377A CJK UNIFIED IDEOGRAPH-#377A, IRGKangXi=0297.300, Def{small; little, humble, to owe, deficient; lacking}
377B CJK UNIFIED IDEOGRAPH-#377B, IRGKangXi=0297.320, Def{(same as U+50C5 僅) barely enough; scarcely; almost}
377C CJK UNIFIED IDEOGRAPH-#377C, IRGKangXi=0298.070, Def{crooked; winding, to circle; to hover around}
377D CJK UNIFIED IDEOGRAPH-#377D, IRGKangXi=0298.171, Def{short, small; little}
377E CJK UNIFIED IDEOGRAPH-#377E, IRGKangXi=0298.221, Def{can not walk normally}
377F CJK UNIFIED IDEOGRAPH-#377F, IRGKangXi=0298.210, Def{(non-classical form of 跛) lame; crippled}
3780 CJK UNIFIED IDEOGRAPH-#3780, IRGKangXi=0298.270, Def{to bump, to strike}
3781 CJK UNIFIED IDEOGRAPH-#3781, IRGKangXi=0298.280, Def{can not walk normally, swollen feet}
3782 CJK UNIFIED IDEOGRAPH-#3782, IRGKangXi=0298.300, Def{ulcers, swollen feet, a kind of disease}
3783 CJK UNIFIED IDEOGRAPH-#3783, IRGKangXi=0298.350, Def{tired; weary}
3784 CJK UNIFIED IDEOGRAPH-#3784, IRGKangXi=0298.360, Def{lame; crippled; feeble}
3785 CJK UNIFIED IDEOGRAPH-#3785, IRGKangXi=0298.380, Def{not straight, improper}
3786 CJK UNIFIED IDEOGRAPH-#3786, IRGKangXi=0298.410, Def{(same as 踦) one-legged, crippled; halt, a defect, tired, the shin}
3787 CJK UNIFIED IDEOGRAPH-#3787, IRGKangXi=0299.061, Def{ulcers, swollen feet, a kind of disease (abnormal in walking)}
3788 CJK UNIFIED IDEOGRAPH-#3788, IRGKangXi=0299.060, Def{big foot, walking}
3789 CJK UNIFIED IDEOGRAPH-#3789, IRGKangXi=0299.080, Def{big foot}
378A CJK UNIFIED IDEOGRAPH-#378A, IRGKangXi=0299.140, Def{unable to walk}
378B CJK UNIFIED IDEOGRAPH-#378B, IRGKangXi=0300.050, Def{soft and meek, weak, to arrange, to regulate}
378C CJK UNIFIED IDEOGRAPH-#378C, IRGKangXi=0300.120, Def{(same as 孕) to be pregnant; to conceive}
378D CJK UNIFIED IDEOGRAPH-#378D, IRGKangXi=0300.151
378E CJK UNIFIED IDEOGRAPH-#378E, IRGKangXi=0300.241, Def{(baby talk) body waste, (a dialect) weight measurement}
378F CJK UNIFIED IDEOGRAPH-#378F, IRGKangXi=0300.250, Def{small; to store up or pile in order, to follow; to trace, a short step}
3790 CJK UNIFIED IDEOGRAPH-#3790, IRGKangXi=0301.010, Def{(an ancient form of 居), to dwell; to remain, to be in (various states and conditions), to occupy, the course of one's life}
3791 CJK UNIFIED IDEOGRAPH-#3791, IRGKangXi=0301.130, Def{(standard form of 尾) the tail; the rear, last; final}
3792 CJK UNIFIED IDEOGRAPH-#3792, IRGKangXi=0301.180, Def{(standard form) lusty; strong; gigantic strength, heavy sleep with snoring}
3793 CJK UNIFIED IDEOGRAPH-#3793, IRGKangXi=0301.220, Def{the buttocks; the rump; the sacrum, to set sideways, the male organ; (Cant.) vulgar term for the female sex organ}
3794 CJK UNIFIED IDEOGRAPH-#3794, IRGKangXi=0301.260, Def{(corrupted form) the upright bar for fastening a door}
3795 CJK UNIFIED IDEOGRAPH-#3795, IRGKangXi=0301.300, Def{(same as 屑) chips; crumbs; bits trifles, to care; to mind}
3796 CJK UNIFIED IDEOGRAPH-#3796, IRGKangXi=0301.310, Def{this, here, if so; in this case, reluctant to go forward}
3797 CJK UNIFIED IDEOGRAPH-#3797, IRGKangXi=0302.070, Def{the male organ, obscene language; vulgar expression}
3798 CJK UNIFIED IDEOGRAPH-#3798, IRGKangXi=0302.230, Def{(non-classical form of U+8C5A 豚) a small pig; (Cant.) end, bottom, rump}
3799 CJK UNIFIED IDEOGRAPH-#3799, IRGKangXi=0302.321, Def{(same as 尿) urine; to urinate}
379A CJK UNIFIED IDEOGRAPH-#379A, IRGKangXi=0302.330, Def{to pile up layer by layer, wedge, gatepost}
379B CJK UNIFIED IDEOGRAPH-#379B, IRGKangXi=0303.110, Def{red slippers, (corrupted form of 履) shoes}
379C CJK UNIFIED IDEOGRAPH-#379C, IRGKangXi=0303.370, Def{hempen sandals, boots}
379D CJK UNIFIED IDEOGRAPH-#379D, IRGKangXi=0303.391
379E CJK UNIFIED IDEOGRAPH-#379E, IRGKangXi=0303.391, Def{semen}
379F CJK UNIFIED IDEOGRAPH-#379F, IRGKangXi=0303.450, Def{savings and (or) reserves; (same as U+5960 奠) to settle; to lay; (same as U+588A 墊) to advance (money), to cushion, to sink into; (said of manner) dignified; stately; graceful}
37A0 CJK UNIFIED IDEOGRAPH-#37A0, IRGKangXi=0303.461, Def{male organ}
37A1 CJK UNIFIED IDEOGRAPH-#37A1, IRGKangXi=0304.050, Def{(same as 展) to open, to stretch, to unfold, to expand}
37A2 CJK UNIFIED IDEOGRAPH-#37A2, IRGKangXi=0304.120, Def{(ancient form of 之) to arrive at; to go to}
37A3 CJK UNIFIED IDEOGRAPH-#37A3, IRGKangXi=0305.040, Def{(same as 芬) fragrance; aroma; sweet smell; perfume}
37A4 CJK UNIFIED IDEOGRAPH-#37A4, IRGKangXi=0307.030, Def{to help; to aid; to assist (ancient form of 岑) a relatively high, pointed hill, silent; still; quiet}
37A5 CJK UNIFIED IDEOGRAPH-#37A5, IRGKangXi=0307.050, Def{deep in the mountain}
37A6 CJK UNIFIED IDEOGRAPH-#37A6, IRGKangXi=0307.070, Def{name of a mountain}
37A7 CJK UNIFIED IDEOGRAPH-#37A7, IRGKangXi=0307.090, Def{(ancient form of 會) to meet, to gather, to be able, to understand}
37A8 CJK UNIFIED IDEOGRAPH-#37A8, IRGKangXi=0307.160, Def{(corrupted form) name of a mountain (the location is unknown)}
37A9 CJK UNIFIED IDEOGRAPH-#37A9, IRGKangXi=0307.180, Def{(same as 嵐) mountain vapor; mist}
37AA CJK UNIFIED IDEOGRAPH-#37AA, IRGKangXi=0307.200, Def{the appearance of the mountain is flat and smooth, (corrupted form of 峱) name of a mountain in Shandong province}
37AB CJK UNIFIED IDEOGRAPH-#37AB, IRGKangXi=0307.260, Def{(non-classical form of 走) to walk, to run, to leave}
37AC CJK UNIFIED IDEOGRAPH-#37AC, IRGKangXi=0307.271, Def{close or near}
37AD CJK UNIFIED IDEOGRAPH-#37AD, IRGKangXi=0307.271, Def{name of an island}
37AE CJK UNIFIED IDEOGRAPH-#37AE, IRGKangXi=0307.271, Def{variant of 出 U+51FA, to go out, send out; to stand; to produce}
37AF CJK UNIFIED IDEOGRAPH-#37AF, IRGKangXi=0307.271, Def{(same as 岊) the turning area of a mountain, mountain top; summit, a high mountain, a bare (of bald) mountain or hill}
37B0 CJK UNIFIED IDEOGRAPH-#37B0, IRGKangXi=0307.290, Def{name of a mountain}
37B1 CJK UNIFIED IDEOGRAPH-#37B1, IRGKangXi=0307.320, Def{dusk; twilight}
37B2 CJK UNIFIED IDEOGRAPH-#37B2, IRGKangXi=0308.020, Def{name of a mountain, the plain between mountains}
37B3 CJK UNIFIED IDEOGRAPH-#37B3, IRGKangXi=0308.050, Def{name of a mountain}
37B4 CJK UNIFIED IDEOGRAPH-#37B4, IRGKangXi=0308.160, Def{name of a mountain, (non-classical form of 岷) the Min River (in Sichuan), Mt. Min (in Sichuan), name of a county (in old China)}
37B5 CJK UNIFIED IDEOGRAPH-#37B5, IRGKangXi=0308.200, Def{(corrupted form) frivolous; flippant; capricious; playful; sportive, cunning}
37B6 CJK UNIFIED IDEOGRAPH-#37B6, IRGKangXi=0308.300, Def{(same as 岷) the Min River (in Sichuan), Mt. Min (in Sichuan), name of a county (in old China)}
37B7 CJK UNIFIED IDEOGRAPH-#37B7, IRGKangXi=0305.060, Def{luxuriant; lush; flourishing}
37B8 CJK UNIFIED IDEOGRAPH-#37B8, IRGKangXi=0308.331, Def{(same as 呡) (simplified form of 歲) a year; age (of a person), harvest}
37B9 CJK UNIFIED IDEOGRAPH-#37B9, IRGKangXi=0309.030, Def{(same as 岢) name of a mountain in Shanxi province}
37BA CJK UNIFIED IDEOGRAPH-#37BA, IRGKangXi=0309.060, Def{(ancient form of 族) a tribe; a clan; relatives, race (of people), a class; a family (of animals)}
37BB CJK UNIFIED IDEOGRAPH-#37BB, IRGKangXi=0309.080, Def{name of a mountain}
37BC CJK UNIFIED IDEOGRAPH-#37BC, IRGKangXi=0309.120, Def{(same as 丞) to respectfully receive, to flatter; to pay court to}
37BD CJK UNIFIED IDEOGRAPH-#37BD, IRGKangXi=0309.140, Def{mountain; hill}
37BE CJK UNIFIED IDEOGRAPH-#37BE, IRGKangXi=0309.360, Def{(same as 尼) a nun, the mountains after which Confucius was named}
37BF CJK UNIFIED IDEOGRAPH-#37BF, IRGKangXi=0310.071, Def{(same as 岐) name of a mountain in Shanxi province, name of a political district in ancient China, name of a river in Shanxi province, to diverge; to branch}
37C0 CJK UNIFIED IDEOGRAPH-#37C0, IRGKangXi=0310.071, Def{name of a mountain}
37C1 CJK UNIFIED IDEOGRAPH-#37C1, IRGKangXi=0310.071, Def{(a variant of 岸) shore; bank; beach; coast}
37C2 CJK UNIFIED IDEOGRAPH-#37C2, IRGKangXi=0310.071, Def{name of cape in Shandong Province}
37C3 CJK UNIFIED IDEOGRAPH-#37C3, IRGKangXi=0310.071, Def{(corrupted form of 司) to have charge of; to preside over, a (governmental) department}
37C4 CJK UNIFIED IDEOGRAPH-#37C4, IRGKangXi=0310.160, Def{name of a mountain}
37C5 CJK UNIFIED IDEOGRAPH-#37C5, IRGKangXi=0310.210, Def{lofty; high; steep}
37C6 CJK UNIFIED IDEOGRAPH-#37C6, IRGKangXi=0311.061, Def{name of a mountain}
37C7 CJK UNIFIED IDEOGRAPH-#37C7, IRGKangXi=0311.070, Def{a land-slide, name of mountain}
37C8 CJK UNIFIED IDEOGRAPH-#37C8, IRGKangXi=0311.220, Def{name of a mountain}
37C9 CJK UNIFIED IDEOGRAPH-#37C9, IRGKangXi=0311.250, Def{name of pavilion (garden) in ancient China, name of a mountain}
37CA CJK UNIFIED IDEOGRAPH-#37CA, IRGKangXi=0311.270, Def{name of a mountain}
37CB CJK UNIFIED IDEOGRAPH-#37CB, IRGKangXi=0311.290, Def{deep and level ground gully at the foot of a hill}
37CC CJK UNIFIED IDEOGRAPH-#37CC, IRGKangXi=0311.380, Def{a mountain in today's Sichuan Province}
37CD CJK UNIFIED IDEOGRAPH-#37CD, IRGKangXi=0312.010, Def{name of a mountain, a solitary peak}
37CE CJK UNIFIED IDEOGRAPH-#37CE, IRGKangXi=0312.080, Def{a long mountain ridge, steep; lofty mountain}
37CF CJK UNIFIED IDEOGRAPH-#37CF, IRGKangXi=0312.170, Def{steep and lofty, a big valley}
37D0 CJK UNIFIED IDEOGRAPH-#37D0, IRGKangXi=0312.220, Def{big and high; lofty; steep}
37D1 CJK UNIFIED IDEOGRAPH-#37D1, IRGKangXi=0312.260, Def{mountain; hill}
37D2 CJK UNIFIED IDEOGRAPH-#37D2, IRGKangXi=0312.301, Def{mountains adjoining, mountain chain; name of a mountain}
37D3 CJK UNIFIED IDEOGRAPH-#37D3, IRGKangXi=0311.320, Def{lofty and dangerous, rocks on the mountain side}
37D4 CJK UNIFIED IDEOGRAPH-#37D4, IRGKangXi=0312.301, Def{steep and lofty, a big valley}
37D5 CJK UNIFIED IDEOGRAPH-#37D5, IRGKangXi=0312.301
37D6 CJK UNIFIED IDEOGRAPH-#37D6, IRGKangXi=0312.301, Def{(loanword from Zhuang) the plain; level ground between rocky mountain, name of county}
37D7 CJK UNIFIED IDEOGRAPH-#37D7, IRGKangXi=0312.320, Def{name of a small principality in what is now Shanxi, the home of the founder of the Zhou dynasty}
37D8 CJK UNIFIED IDEOGRAPH-#37D8, IRGKangXi=0312.330, Def{name of a mountain}
37D9 CJK UNIFIED IDEOGRAPH-#37D9, IRGKangXi=0313.070, Def{shape of the mountain, a mountain in today's Shandong Province}
37DA CJK UNIFIED IDEOGRAPH-#37DA, IRGKangXi=0313.090, Def{(same as 岐) name of a mountain in Shanxi Province, name of a political district in ancient China, name of a river in Shanxi province, to diverge; to branch}
37DB CJK UNIFIED IDEOGRAPH-#37DB, IRGKangXi=0313.150, Def{high and steep; lofty}
37DC CJK UNIFIED IDEOGRAPH-#37DC, IRGKangXi=0314.190, Def{(non-classical and abbreviated form) shape of the mountain}
37DD CJK UNIFIED IDEOGRAPH-#37DD, IRGKangXi=0314.240, Def{(same as 培) to bank up with earth, to nourish; to strengthen; to cultivate}
37DE CJK UNIFIED IDEOGRAPH-#37DE, IRGKangXi=0315.080, Def{precipitous and lofty; very steep mountains}
37DF CJK UNIFIED IDEOGRAPH-#37DF, IRGKangXi=0315.100, Def{(same as 巷) a lane; an alley}
37E0 CJK UNIFIED IDEOGRAPH-#37E0, IRGKangXi=0315.140, Def{(same as 岡) ridge (of a hill or mountain)}
37E1 CJK UNIFIED IDEOGRAPH-#37E1, IRGKangXi=0315.151
37E2 CJK UNIFIED IDEOGRAPH-#37E2, IRGKangXi=0315.151, Def{the rocks on the hills are of different shapes; rugged and weird, character different from others; cape; spit; promontory}
37E3 CJK UNIFIED IDEOGRAPH-#37E3, IRGKangXi=0315.151
37E4 CJK UNIFIED IDEOGRAPH-#37E4, IRGKangXi=0315.151, Def{(loan-word from Zhuang) used in place-names in Guangxi for level ground, a plain between mountains}
37E5 CJK UNIFIED IDEOGRAPH-#37E5, IRGKangXi=0315.151, Def{(simplified form of 嵾) uneven; rolling; rough; rugged, name of a mountain (Wudangshan)}
37E6 CJK UNIFIED IDEOGRAPH-#37E6, IRGKangXi=0315.240, Def{name of a mountain}
37E7 CJK UNIFIED IDEOGRAPH-#37E7, IRGKangXi=0316.010, Def{a cliff; a precipice}
37E8 CJK UNIFIED IDEOGRAPH-#37E8, IRGKangXi=0316.110, Def{a fountain in the side of a hill, name of a mountain}
37E9 CJK UNIFIED IDEOGRAPH-#37E9, IRGKangXi=0316.260, Def{(same as 岷) the Min River (in Sichuan), Mt. Min (in Sichuan), name of a county (in old China)}
37EA CJK UNIFIED IDEOGRAPH-#37EA, IRGKangXi=0316.280, Def{name of a mountain, lofty and steep; high}
37EB CJK UNIFIED IDEOGRAPH-#37EB, IRGKangXi=0316.350, Def{(same as 泉) spring; fountain, money (archaic)}
37EC CJK UNIFIED IDEOGRAPH-#37EC, IRGKangXi=0317.521, Def{a mountain in today's Shandong Province southwest of Mengyin county}
37ED CJK UNIFIED IDEOGRAPH-#37ED, IRGKangXi=0316.381, Def{(same as 岷) the Min River (in Sichuan), Mt. Min (in Sichuan), name of a county (in old China)}
37EE CJK UNIFIED IDEOGRAPH-#37EE, IRGKangXi=0316.381, Def{lofty; high}
37EF CJK UNIFIED IDEOGRAPH-#37EF, IRGKangXi=0316.381
37F0 CJK UNIFIED IDEOGRAPH-#37F0, IRGKangXi=0317.050, Def{name of a mountain; lofty, steep}
37F1 CJK UNIFIED IDEOGRAPH-#37F1, IRGKangXi=0317.190, Def{shape of a mountain}
37F2 CJK UNIFIED IDEOGRAPH-#37F2, IRGKangXi=0317.210, Def{mountain peak; mountain top; summit}
37F3 CJK UNIFIED IDEOGRAPH-#37F3, IRGKangXi=0317.300, Def{name of a mountain}
37F4 CJK UNIFIED IDEOGRAPH-#37F4, IRGKangXi=0317.360, Def{lump on the hills, rugged; rough; rough, lofty; steep, the dejection one feels at heart}
37F5 CJK UNIFIED IDEOGRAPH-#37F5, IRGKangXi=0317.480, Def{(non-classical form of 岡) ridge (of a hill or mountain)}
37F6 CJK UNIFIED IDEOGRAPH-#37F6, IRGKangXi=0317.521, Def{mountain peak; mountain top; summit}
37F7 CJK UNIFIED IDEOGRAPH-#37F7, IRGKangXi=0317.521, Def{a wart; a pustule; a pimple, a round lump}
37F8 CJK UNIFIED IDEOGRAPH-#37F8, IRGKangXi=0317.521, Def{(non-classical form) name of pavilion (garden) in ancient China, name of a mountain}
37F9 CJK UNIFIED IDEOGRAPH-#37F9, IRGKangXi=0318.080, Def{the appearance of a mountain is lofty and steep and profound}
37FA CJK UNIFIED IDEOGRAPH-#37FA, IRGKangXi=0318.260, Def{mountain top; summit}
37FB CJK UNIFIED IDEOGRAPH-#37FB, IRGKangXi=0318.280, Def{(same as 塹) the moat around a city, a pit; a hole or cavity in the ground (same as 嶄) (of a mountain) high and pointed, novel; new}
37FC CJK UNIFIED IDEOGRAPH-#37FC, IRGKangXi=0318.320, Def{name of a place in Shandong Province, mountain with lots of small rocks}
37FD CJK UNIFIED IDEOGRAPH-#37FD, IRGKangXi=0318.330, Def{mountain top; summit, pointing mountain}
37FE CJK UNIFIED IDEOGRAPH-#37FE, IRGKangXi=0318.390, Def{name of a mountain in county Jian}
37FF CJK UNIFIED IDEOGRAPH-#37FF, IRGKangXi=0318.400, Def{high and steep, name of a mountain}
3800 CJK UNIFIED IDEOGRAPH-#3800, IRGKangXi=0319.070, Def{(same as 島) island; isle}
3801 CJK UNIFIED IDEOGRAPH-#3801, IRGKangXi=0319.090, Def{irregular outline of a range of hills}
3802 CJK UNIFIED IDEOGRAPH-#3802, IRGKangXi=0319.111, Def{name of a place, lofty; steep, mountain with lots of small rocks}
3803 CJK UNIFIED IDEOGRAPH-#3803, IRGKangXi=0319.111
3804 CJK UNIFIED IDEOGRAPH-#3804, IRGKangXi=0319.111, Def{name of a mountain in Yunnan Province}
3805 CJK UNIFIED IDEOGRAPH-#3805, IRGKangXi=0319.120, Def{to cover; to put a cover over something, name of a mountain}
3806 CJK UNIFIED IDEOGRAPH-#3806, IRGKangXi=0319.140, Def{name of a mountain}
3807 CJK UNIFIED IDEOGRAPH-#3807, IRGKangXi=0319.190, Def{name of a mountain ridge}
3808 CJK UNIFIED IDEOGRAPH-#3808, IRGKangXi=0319.250, Def{name of a place}
3809 CJK UNIFIED IDEOGRAPH-#3809, IRGKangXi=0319.260, Def{mountain; hill, bald hills, valley; ravine}
380A CJK UNIFIED IDEOGRAPH-#380A, IRGKangXi=0319.290, Def{(same as U+5D87 嶇) a rugged, steep mountain}
380B CJK UNIFIED IDEOGRAPH-#380B, IRGKangXi=0319.360, Def{(same as 崿) a lofty mountain peak}
380C CJK UNIFIED IDEOGRAPH-#380C, IRGKangXi=0319.380, Def{high; lofty; steep, (corrupted form of 嵌)}
380D CJK UNIFIED IDEOGRAPH-#380D, IRGKangXi=0319.400, Def{lofty; majestic, name of a mountain}
380E CJK UNIFIED IDEOGRAPH-#380E, IRGKangXi=0319.410, Def{(same as U+380D 㠍) a lofty mountain}
380F CJK UNIFIED IDEOGRAPH-#380F, IRGKangXi=0320.131, Def{name of a mountain}
3810 CJK UNIFIED IDEOGRAPH-#3810, IRGKangXi=0320.131, Def{high and pointed mountain, mountain paths (same as 嶠) lofty}
3811 CJK UNIFIED IDEOGRAPH-#3811, IRGKangXi=0320.250, Def{high mountain steep rugged and uneven}
3812 CJK UNIFIED IDEOGRAPH-#3812, IRGKangXi=0320.260, Def{(same as U+37FD 㟽) mountain top; summit, pointed mountain top}
3813 CJK UNIFIED IDEOGRAPH-#3813, IRGKangXi=0320.320, Def{name of a mountain}
3814 CJK UNIFIED IDEOGRAPH-#3814, IRGKangXi=0320.330, Def{shapes of the mountain, the strategic and dangerous gorge, (a dialect) field in between of the valleys}
3815 CJK UNIFIED IDEOGRAPH-#3815, IRGKangXi=0320.390, Def{(same as 嵬) lofty, precipitous; high and dangerous mountain}
3816 CJK UNIFIED IDEOGRAPH-#3816, IRGKangXi=0321.070, Def{(same as 嶬) precipitous; nigh and dangerous, name of a mountain}
3817 CJK UNIFIED IDEOGRAPH-#3817, IRGKangXi=0321.081, Def{(same as 奧 嶴) deep in the mountain, name of a place in Zhejiang Province; at the hill side}
3818 CJK UNIFIED IDEOGRAPH-#3818, IRGKangXi=0321.361, Def{(same as 嶼) an island}
3819 CJK UNIFIED IDEOGRAPH-#3819, IRGKangXi=0321.100, Def{a mountain paths (same as 崤) name of a mountain}
381A CJK UNIFIED IDEOGRAPH-#381A, IRGKangXi=0321.230, Def{high and lofty}
381B CJK UNIFIED IDEOGRAPH-#381B, IRGKangXi=0321.270, Def{name a county}
381C CJK UNIFIED IDEOGRAPH-#381C, IRGKangXi=0321.361, Def{(same as 嶷) young kids smart and familiar with human affairs, range of mountains in Hunan Province}
381D CJK UNIFIED IDEOGRAPH-#381D, IRGKangXi=0321.370, Def{(non-classical 巑) lofty and steep mountains, to rise high}
381E CJK UNIFIED IDEOGRAPH-#381E, IRGKangXi=0322.061
381F CJK UNIFIED IDEOGRAPH-#381F, IRGKangXi=0322.061, Def{a mountain in ancient times; Lieshan in Guizhou province}
3820 CJK UNIFIED IDEOGRAPH-#3820, IRGKangXi=0322.080, Def{name of a mountain}
3821 CJK UNIFIED IDEOGRAPH-#3821, IRGKangXi=0322.140, Def{high and lofty}
3822 CJK UNIFIED IDEOGRAPH-#3822, IRGKangXi=0322.150, Def{rugged and uneven, to shrink; to recoil; to cringe}
3823 CJK UNIFIED IDEOGRAPH-#3823, IRGKangXi=0322.181, Def{a mountain in Jiangxi Province; southeast of Jingde county}
3824 CJK UNIFIED IDEOGRAPH-#3824, IRGKangXi=0322.291
3825 CJK UNIFIED IDEOGRAPH-#3825, IRGKangXi=0322.300, Def{rugged and uneven, high and lofty, to roll stones down hill, piles of stones}
3826 CJK UNIFIED IDEOGRAPH-#3826, IRGKangXi=0322.370, Def{name of a mountain}
3827 CJK UNIFIED IDEOGRAPH-#3827, IRGKangXi=0323.050, Def{a little slanted; smooth, name of a mountain}
3828 CJK UNIFIED IDEOGRAPH-#3828, IRGKangXi=0323.161, Def{misty mountain}
3829 CJK UNIFIED IDEOGRAPH-#3829, IRGKangXi=0324.061, Def{(same as 芒 荒) vast, to reach, dwell, to neglect; to leave completely unattended to}
382A CJK UNIFIED IDEOGRAPH-#382A, IRGKangXi=0325.060, Def{ancient form of 五 'five'}
382B CJK UNIFIED IDEOGRAPH-#382B, IRGKangXi=0326.031
382C CJK UNIFIED IDEOGRAPH-#382C, IRGKangXi=0326.031
382D CJK UNIFIED IDEOGRAPH-#382D, IRGKangXi=0326.050, Def{to open, to stretch; to extend, to unfold; to dilate; to prolong}
382E CJK UNIFIED IDEOGRAPH-#382E, IRGKangXi=0326.070, Def{to expand, to swell, (non-classical form of 肛) the anus}
382F CJK UNIFIED IDEOGRAPH-#382F, IRGKangXi=0327.030, Def{according to; to use; with, for}
3830 CJK UNIFIED IDEOGRAPH-#3830, IRGKangXi=0327.061
3831 CJK UNIFIED IDEOGRAPH-#3831, IRGKangXi=0327.110, Def{to set with the legs crossed (standard form of 踞); (ancient form of 箕) a winnowing basket, a dust basket, (same as 杞) a state in ancient times}
3832 CJK UNIFIED IDEOGRAPH-#3832, IRGKangXi=0328.050, Def{ripped; split}
3833 CJK UNIFIED IDEOGRAPH-#3833, IRGKangXi=0328.081
3834 CJK UNIFIED IDEOGRAPH-#3834, IRGKangXi=0328.090, Def{a pillow case}
3835 CJK UNIFIED IDEOGRAPH-#3835, IRGKangXi=0328.120, Def{a turban; headdress; a scarf, a piece of cloth (fabric) used to cover something}
3836 CJK UNIFIED IDEOGRAPH-#3836, IRGKangXi=0329.051, Def{(a variant of U+5E06 帆) a sail, to sail}
3837 CJK UNIFIED IDEOGRAPH-#3837, IRGKangXi=0329.060, Def{a vessel for rice, mats on carriage}
3838 CJK UNIFIED IDEOGRAPH-#3838, IRGKangXi=0329.120, Def{(same as 衭) the lapel or collar of a garment or robe, (Cant., same as 褲) drawers, trousers or pants}
3839 CJK UNIFIED IDEOGRAPH-#3839, IRGKangXi=0329.150, Def{a turban; a headdress a scarf}
383A CJK UNIFIED IDEOGRAPH-#383A, IRGKangXi=0329.220, Def{a fine thread, a net with small; tiny holes}
383B CJK UNIFIED IDEOGRAPH-#383B, IRGKangXi=0329.291, Def{(same as 希) rare, to hope; to expect}
383C CJK UNIFIED IDEOGRAPH-#383C, IRGKangXi=0329.291
383D CJK UNIFIED IDEOGRAPH-#383D, IRGKangXi=0329.300, Def{a dish-cloth; a mopper; a cleaning rag}
383E CJK UNIFIED IDEOGRAPH-#383E, IRGKangXi=0329.310, Def{a dish-cloth; a mopper; a cleaning rag, headdress; a scarf; turban}
383F CJK UNIFIED IDEOGRAPH-#383F, IRGKangXi=0329.410, Def{a kind of delicate and fine fabric, a napkin; kerchief or towel, headwear}
3840 CJK UNIFIED IDEOGRAPH-#3840, IRGKangXi=0330.050, Def{ragged clothing, ragged; old and wear out}
3841 CJK UNIFIED IDEOGRAPH-#3841, IRGKangXi=0330.170, Def{a robe; a long gown, a shirt; a short gown}
3842 CJK UNIFIED IDEOGRAPH-#3842, IRGKangXi=0330.210, Def{remains; remnants}
3843 CJK UNIFIED IDEOGRAPH-#3843, IRGKangXi=0331.010, Def{an artisan to soften raw silk by boiling and to dye the dressed silk}
3844 CJK UNIFIED IDEOGRAPH-#3844, IRGKangXi=0331.020, Def{collar}
3845 CJK UNIFIED IDEOGRAPH-#3845, IRGKangXi=0331.130, Def{storage room}
3846 CJK UNIFIED IDEOGRAPH-#3846, IRGKangXi=0331.191, Def{(same as U+3843 㡃) an artisan to soften raw silk by boiling and to dye the dressed silk}
3847 CJK UNIFIED IDEOGRAPH-#3847, IRGKangXi=0331.200, Def{collar, the front of a Chinese gown, lapel of a Chinese dress worn by the literati in former days}
3848 CJK UNIFIED IDEOGRAPH-#3848, IRGKangXi=0331.240, Def{the ropes attached to the bier and held by mourners, mourning garments, a ceremonial cap for high ministers in old China}
3849 CJK UNIFIED IDEOGRAPH-#3849, IRGKangXi=0332.220, Def{a kind of fabric, a county in today's Shandong Province; southwest of Huangxian}
384A CJK UNIFIED IDEOGRAPH-#384A, IRGKangXi=0333.010, Def{(same as 帢) a kind of cap worn by an officer in old China}
384B CJK UNIFIED IDEOGRAPH-#384B, IRGKangXi=0333.080, Def{a kind of headdress for man in old China, a bag used to feed the horses}
384C CJK UNIFIED IDEOGRAPH-#384C, IRGKangXi=0333.161, Def{(same as 帽) a hat, headwear, a cap (of fountain-pen, crew, etc.)}
384D CJK UNIFIED IDEOGRAPH-#384D, IRGKangXi=0333.161
384E CJK UNIFIED IDEOGRAPH-#384E, IRGKangXi=0333.161, Def{(simplified form of U+5E53 幓) (same as U+7E3F 縿) the long bands or ribbons attached to flags and banners (same as U+8942 襂) the feather decorated carriages or dresses; flapping and dangling, headdress used by man in ancient times}
384F CJK UNIFIED IDEOGRAPH-#384F, IRGKangXi=0333.170, Def{to cut a strip of cloth; a breadth of material, (same as 繻) fine gauze, frayed edges of silk, silk torn into two pieces, one of which was given as a credential and the other retained, a loose garment or cloak; fine clothes, the left over material after cutting; ragged fabric}
3850 CJK UNIFIED IDEOGRAPH-#3850, IRGKangXi=0333.210, Def{(non-classical form) flags, pennants, streamers, etc.}
3851 CJK UNIFIED IDEOGRAPH-#3851, IRGKangXi=0334.030, Def{ancient mourning turban worn by women, to hem, turban worn by men}
3852 CJK UNIFIED IDEOGRAPH-#3852, IRGKangXi=0334.090, Def{a calico sack for grain, etc.}
3853 CJK UNIFIED IDEOGRAPH-#3853, IRGKangXi=0334.130, Def{drawers; trousers or pants}
3854 CJK UNIFIED IDEOGRAPH-#3854, IRGKangXi=0334.140, Def{oiled calico, varnish-cloth used to cover the cross-bean in front of the shafts of a carriage}
3855 CJK UNIFIED IDEOGRAPH-#3855, IRGKangXi=0334.150, Def{(same as 央) sharp; distinct, bright-colored}
3856 CJK UNIFIED IDEOGRAPH-#3856, IRGKangXi=0334.170, Def{(same as 幢) to cover; a covering; a screen, a curtain for a carriage}
3857 CJK UNIFIED IDEOGRAPH-#3857, IRGKangXi=0334.350, Def{red paper, a small piece of thin paper, ropes; cords; cables}
3858 CJK UNIFIED IDEOGRAPH-#3858, IRGKangXi=0334.360, Def{screens; cloth partition; door curtain; door flap, (same as 縑) a kind of fine silk, downy, feathery; featheriness}
3859 CJK UNIFIED IDEOGRAPH-#3859, IRGKangXi=0335.070, Def{curtain for carriage}
385A CJK UNIFIED IDEOGRAPH-#385A, IRGKangXi=0335.080, Def{armour, unlined garments}
385B CJK UNIFIED IDEOGRAPH-#385B, IRGKangXi=0335.151, Def{name for a workman who applied color in painting (in ancient times)}
385C CJK UNIFIED IDEOGRAPH-#385C, IRGKangXi=0335.160, Def{ragged clothes, left over; remainder, the excess; the surplus, to cut out (of silk fabrics) as a design}
385D CJK UNIFIED IDEOGRAPH-#385D, IRGKangXi=0335.220, Def{a napkin, kerchief, a headgear; articles for dressing the hair, a calligrapher's or painter's signature, seal, dedicatory notes, etc. on a painting, etc.}
385E CJK UNIFIED IDEOGRAPH-#385E, IRGKangXi=0336.020, Def{a sack used to feed the horses, an old family name}
385F CJK UNIFIED IDEOGRAPH-#385F, IRGKangXi=0336.030, Def{bed-mats for baby}
3860 CJK UNIFIED IDEOGRAPH-#3860, IRGKangXi=0336.170, Def{a picture; one of a pair, as of scrolls, etc.}
3861 CJK UNIFIED IDEOGRAPH-#3861, IRGKangXi=0337.031, Def{(a variant of 幮 U+5E6E, 𢅥 U+22165) a screen used to make a temporary kitchen}
3862 CJK UNIFIED IDEOGRAPH-#3862, IRGKangXi=0337.130, Def{(a variant of 幔) a curtain, a screen}
3863 CJK UNIFIED IDEOGRAPH-#3863, IRGKangXi=0337.171, Def{(same as 龍) dragon, or the emperor; imperial, a Chinese family name}
3864 CJK UNIFIED IDEOGRAPH-#3864, IRGKangXi=0337.171
3865 CJK UNIFIED IDEOGRAPH-#3865, IRGKangXi=0337.180, Def{to curl up; to roll up, to wrap up; to bind}
3866 CJK UNIFIED IDEOGRAPH-#3866, IRGKangXi=0337.210, Def{mess; out of order, chaos; confusion, (of clothes) tattered; in rags}
3867 CJK UNIFIED IDEOGRAPH-#3867, IRGKangXi=0338.170, Def{to stretch open painting, (same as 幀) a picture (one of a pair as of scrolls, etc.)}
3868 CJK UNIFIED IDEOGRAPH-#3868, IRGKangXi=0338.180, Def{a label; a tag, to record, to make a note, to wipe; to scrub; to dust; to clean}
3869 CJK UNIFIED IDEOGRAPH-#3869, IRGKangXi=0338.220, Def{a girdle; a sash or scarf; a belt; a band; a ribbon; a string; a tie}
386A CJK UNIFIED IDEOGRAPH-#386A, IRGKangXi=0338.251, Def{the varnish on the floor, to erase; to obliterate, to scribble}
386B CJK UNIFIED IDEOGRAPH-#386B, IRGKangXi=0342.170, Def{anxious, unsuccessful man}
386C CJK UNIFIED IDEOGRAPH-#386C, IRGKangXi=0342.181
386D CJK UNIFIED IDEOGRAPH-#386D, IRGKangXi=0343.020, Def{(standard, ancient form of U+7E7C 繼) to continue; to carry on, to follow; to inherit; to succeed to}
386E CJK UNIFIED IDEOGRAPH-#386E, IRGKangXi=0343.040, Def{danger; precarious, to fear, lofty; high, just; honest, to grind, to cultivate; to polish}
386F CJK UNIFIED IDEOGRAPH-#386F, IRGKangXi=0343.090, Def{(ancient form of 宅) wall of a building, a house, to keep in the house, thriving; flourishing, blazing, (ancient form of 度) legal system; laws and institutions, to think; to consider; to ponder; to contemplate}
3870 CJK UNIFIED IDEOGRAPH-#3870, IRGKangXi=0343.130, Def{(large seal type 宇) a house; a roof, appearance, space; the canopy of heaven, to cover}
3871 CJK UNIFIED IDEOGRAPH-#3871, IRGKangXi=0343.150, Def{(same as 灸) (in Chinese medicine) to cauterize by burning moxa; moxa cautery; moxibustion}
3872 CJK UNIFIED IDEOGRAPH-#3872, IRGKangXi=0343.340, Def{tiles (laid upside down) on a roof, ropes, to hold fast}
3873 CJK UNIFIED IDEOGRAPH-#3873, IRGKangXi=0344.010, Def{mountain, (corrupted form of 底) underside; bottom; basis; foundation}
3874 CJK UNIFIED IDEOGRAPH-#3874, IRGKangXi=0344.070, Def{sound}
3875 CJK UNIFIED IDEOGRAPH-#3875, IRGKangXi=0344.080, Def{roof of the house connected}
3876 CJK UNIFIED IDEOGRAPH-#3876, IRGKangXi=0344.090, Def{dwell, to live in a depraved (crude; vulgar; inferior) place}
3877 CJK UNIFIED IDEOGRAPH-#3877, IRGKangXi=0344.120, Def{a family name}
3878 CJK UNIFIED IDEOGRAPH-#3878, IRGKangXi=0344.140, Def{a house; an unfinished house, uneven; irregular; unsuitable; ill-matched, tenon}
3879 CJK UNIFIED IDEOGRAPH-#3879, IRGKangXi=0344.230, Def{to rely upon each other}
387A CJK UNIFIED IDEOGRAPH-#387A, IRGKangXi=0344.240, Def{a cottage; a small house, a small cup}
387B CJK UNIFIED IDEOGRAPH-#387B, IRGKangXi=0345.030, Def{(non-classical form of 廖) name of a small ancient State, a Chinese family name}
387C CJK UNIFIED IDEOGRAPH-#387C, IRGKangXi=0345.120, Def{a granary, a cabin, as in the ship, a corridor; a hallway}
387D CJK UNIFIED IDEOGRAPH-#387D, IRGKangXi=0345.130, Def{depraved talking}
387E CJK UNIFIED IDEOGRAPH-#387E, IRGKangXi=0345.140, Def{a threshold; a door-sill}
387F CJK UNIFIED IDEOGRAPH-#387F, IRGKangXi=0345.190, Def{(standard form of 斥) to accuse; to blame, to expel; to drive off; to reject}
3880 CJK UNIFIED IDEOGRAPH-#3880, IRGKangXi=0345.220, Def{side room}
3881 CJK UNIFIED IDEOGRAPH-#3881, IRGKangXi=0345.250, Def{(same as U+388B 㢋) vast, to open up, enlarge or expand, the blot of a door; door latch, name of a person}
3882 CJK UNIFIED IDEOGRAPH-#3882, IRGKangXi=0346.010, Def{(terrains) of highly strategic; precipitous (hill, etc. a big mound; (same as 㢈) a collapsed house, to hit, to catch something}
3883 CJK UNIFIED IDEOGRAPH-#3883, IRGKangXi=0346.080, Def{high, magnanimity}
3884 CJK UNIFIED IDEOGRAPH-#3884, IRGKangXi=0346.120, Def{a kind of utensil; implement; tool, a place of worship; a place where to honor by a service or rite; a place to offer sacrifices; a kitchen}
3885 CJK UNIFIED IDEOGRAPH-#3885, IRGKangXi=0346.230, Def{a shield; a screen, a tall building; an edifice, (same as 弄) an alley; a lane}
3886 CJK UNIFIED IDEOGRAPH-#3886, IRGKangXi=0346.270, Def{(same as 廛) living space for one family in ancient times, a store; a shop, a hundred mu (equal to 733.50 square yards) cultivated land}
3887 CJK UNIFIED IDEOGRAPH-#3887, IRGKangXi=0346.291
3888 CJK UNIFIED IDEOGRAPH-#3888, IRGKangXi=0346.320, Def{a collapsed house; (same as U+5806 堆) to heap up; to pile}
3889 CJK UNIFIED IDEOGRAPH-#3889, IRGKangXi=0346.330, Def{a room with open space, an old county near today's Shandong Province Yunchengxian}
388A CJK UNIFIED IDEOGRAPH-#388A, IRGKangXi=0347.010, Def{to set side by side, to hide; to conceal, to store}
388B CJK UNIFIED IDEOGRAPH-#388B, IRGKangXi=0347.150, Def{vast, to open up, enlarge or expand}
388C CJK UNIFIED IDEOGRAPH-#388C, IRGKangXi=0348.081
388D CJK UNIFIED IDEOGRAPH-#388D, IRGKangXi=0348.200, Def{a long verandah; a long porch}
388E CJK UNIFIED IDEOGRAPH-#388E, IRGKangXi=0348.240, Def{hide; conceal, a house not so high}
388F CJK UNIFIED IDEOGRAPH-#388F, IRGKangXi=0348.270, Def{a cesspool; a manger, a trough; a flume; a chute, (same as 庾) a stack of grain, an ancient measure of capacity, a storage for water, (ancient form 逾) to pass over; to exceed}
3890 CJK UNIFIED IDEOGRAPH-#3890, IRGKangXi=0348.361
3891 CJK UNIFIED IDEOGRAPH-#3891, IRGKangXi=0348.370, Def{(corrupted form of U+3888 㢈) a collapsed house; (same as 堆) to heap up; to pile}
3892 CJK UNIFIED IDEOGRAPH-#3892, IRGKangXi=0349.240, Def{an almost collapsing house}
3893 CJK UNIFIED IDEOGRAPH-#3893, IRGKangXi=0349.280, Def{(same as 障) to separate; to screen, a screen, a veil, a dike, to defend; to guard}
3894 CJK UNIFIED IDEOGRAPH-#3894, IRGKangXi=0349.300, Def{many; numerous; a crowd; the masses, two houses connected each other}
3895 CJK UNIFIED IDEOGRAPH-#3895, IRGKangXi=0350.090, Def{(same as 廱) harmony; peace, peaceful; mild}
3896 CJK UNIFIED IDEOGRAPH-#3896, IRGKangXi=0350.150, Def{a store house, to store}
3897 CJK UNIFIED IDEOGRAPH-#3897, IRGKangXi=0351.040, Def{a high house; a high building}
3898 CJK UNIFIED IDEOGRAPH-#3898, IRGKangXi=0351.071, Def{(non-classical form) (a variant of 廉) pure, modest, incorrupt, not avaricious--thus: reasonable in price}
3899 CJK UNIFIED IDEOGRAPH-#3899, IRGKangXi=0351.100, Def{(same as 廑) a hut; cottage, careful; eager; concern}
389A CJK UNIFIED IDEOGRAPH-#389A, IRGKangXi=0351.120, Def{a corridor; a hallway; rooms around the hall (the middle room of a Chinese house), a nunnery; a convent, a cottage; a hut, a mansion}
389B CJK UNIFIED IDEOGRAPH-#389B, IRGKangXi=0351.190, Def{small, a cabinet; a cupboard}
389C CJK UNIFIED IDEOGRAPH-#389C, IRGKangXi=0351.231, Def{vast}
389D CJK UNIFIED IDEOGRAPH-#389D, IRGKangXi=0352.040, Def{cottage; a coarse house, house with flat roof}
389E CJK UNIFIED IDEOGRAPH-#389E, IRGKangXi=0352.110, Def{rooms connected, moveable house (a yurt, a portable, tentlike dwelling used by nomadic Mongols)}
389F CJK UNIFIED IDEOGRAPH-#389F, IRGKangXi=0352.290, Def{to walk slowly; to stroll; to ramble}
38A0 CJK UNIFIED IDEOGRAPH-#38A0, IRGKangXi=0353.031, Def{(same as 迥) far; distant; separated}
38A1 CJK UNIFIED IDEOGRAPH-#38A1, IRGKangXi=0354.300, Def{(same as 獎) to exhort; to encourage, to praise; to commend, to advise; to urge, to help}
38A2 CJK UNIFIED IDEOGRAPH-#38A2, IRGKangXi=0354.351
38A3 CJK UNIFIED IDEOGRAPH-#38A3, IRGKangXi=0354.370, Def{(same as 擎) to lift; to lift up; to support}
38A4 CJK UNIFIED IDEOGRAPH-#38A4, IRGKangXi=0355.101
38A5 CJK UNIFIED IDEOGRAPH-#38A5, IRGKangXi=0355.120, Def{a big piece of log to join the thick ropes; cable; used on a boat}
38A6 CJK UNIFIED IDEOGRAPH-#38A6, IRGKangXi=0355.150, Def{(corrupted form) wooden post or pile for tethering animals, small and sharp pile}
38A7 CJK UNIFIED IDEOGRAPH-#38A7, IRGKangXi=0356.120, Def{(same as 卷) a book or painting which can be easily folded or rolled up, a division of a book}
38A8 CJK UNIFIED IDEOGRAPH-#38A8, IRGKangXi=0357.010, Def{to defend; to ward off; to refuse; to reject; to oppose, a frontier pass (in Wuxian) in ancient times, a county in ancient times}
38A9 CJK UNIFIED IDEOGRAPH-#38A9, IRGKangXi=0357.040, Def{to shoot; to spurt, (same as 的) target for archery}
38AA CJK UNIFIED IDEOGRAPH-#38AA, IRGKangXi=0357.090, Def{to draw a full bow to aim at the target, to hold; to maintain; to uphold, to conduct}
38AB CJK UNIFIED IDEOGRAPH-#38AB, IRGKangXi=0357.101
38AC CJK UNIFIED IDEOGRAPH-#38AC, IRGKangXi=0357.120, Def{(same as U+5F4B 彋) a bow stretched to the full}
38AD CJK UNIFIED IDEOGRAPH-#38AD, IRGKangXi=0357.211, Def{(non-classical form of 弢) a bow-case; a scabbard, to sheathe, to conceal}
38AE CJK UNIFIED IDEOGRAPH-#38AE, IRGKangXi=0357.240, Def{(same as 弛) to unstring a bow; to relax; to neglect}
38AF CJK UNIFIED IDEOGRAPH-#38AF, IRGKangXi=0357.290, Def{(same as 弴) bows (with carved pattern) used by the son of heaven -- the emperor, flags; streamers; banners, a flag-staff made of bamboo}
38B0 CJK UNIFIED IDEOGRAPH-#38B0, IRGKangXi=0357.330, Def{to stretch a bow, the bow decorated with colorful silk and leather, bent; curved}
38B1 CJK UNIFIED IDEOGRAPH-#38B1, IRGKangXi=0358.031
38B2 CJK UNIFIED IDEOGRAPH-#38B2, IRGKangXi=0358.140, Def{(same as 巽) the 5th of the Eight Diagrams 八卦, South-east, Mild, bland, insinuating, peaceful words}
38B3 CJK UNIFIED IDEOGRAPH-#38B3, IRGKangXi=0358.150, Def{(same as 玈) black color, a black bow; bow in general}
38B4 CJK UNIFIED IDEOGRAPH-#38B4, IRGKangXi=0358.160, Def{(ancient form of 西) the west; western, European}
38B5 CJK UNIFIED IDEOGRAPH-#38B5, IRGKangXi=0358.170, Def{powerful and strong bow, (interchangeable 弽韘) archer's thumb ring of leather}
38B6 CJK UNIFIED IDEOGRAPH-#38B6, IRGKangXi=0358.191, Def{(non-classical form of 弼) device for regulating bows; bow regulator, to correct, to assist; to aid}
38B7 CJK UNIFIED IDEOGRAPH-#38B7, IRGKangXi=0358.191
38B8 CJK UNIFIED IDEOGRAPH-#38B8, IRGKangXi=0358.210, Def{(ancient form of 弼) device for regulating bows; bow regulator, to correct, to assist; to aid, to dust, to shake; to brush}
38B9 CJK UNIFIED IDEOGRAPH-#38B9, IRGKangXi=0359.061
38BA CJK UNIFIED IDEOGRAPH-#38BA, IRGKangXi=0359.080, Def{(non-classical form of 弦) string (of a bow, a musical instrument, etc.), chord of an arc, the first or last quarter of a lunar month}
38BB CJK UNIFIED IDEOGRAPH-#38BB, IRGKangXi=0359.130, Def{to draw a bow, bent}
38BC CJK UNIFIED IDEOGRAPH-#38BC, IRGKangXi=0360.010, Def{(same as 彆) awkward, unfavourable, unsuitable, not smooth}
38BD CJK UNIFIED IDEOGRAPH-#38BD, IRGKangXi=0360.160, Def{(same as 餌) a bait, cakes, food, to eat}
38BE CJK UNIFIED IDEOGRAPH-#38BE, IRGKangXi=0360.190, Def{the curved end of a bow}
38BF CJK UNIFIED IDEOGRAPH-#38BF, IRGKangXi=0360.221
38C0 CJK UNIFIED IDEOGRAPH-#38C0, IRGKangXi=0360.260, Def{to rebound, to impeach, to play}
38C1 CJK UNIFIED IDEOGRAPH-#38C1, IRGKangXi=0360.270, Def{to equip a bow with silk and muscular fibre}
38C2 CJK UNIFIED IDEOGRAPH-#38C2, IRGKangXi=0360.280, Def{to bind, to pack up the bows and the cover of the bows}
38C3 CJK UNIFIED IDEOGRAPH-#38C3, IRGKangXi=0360.350, Def{(same as 鬻) to sell; to buy, to nourish}
38C4 CJK UNIFIED IDEOGRAPH-#38C4, IRGKangXi=0360.400, Def{powerful; strong bows, the end of the bows}
38C5 CJK UNIFIED IDEOGRAPH-#38C5, IRGKangXi=0360.430, Def{bow-string; a chord; a straight road, to draw a bow, strong}
38C6 CJK UNIFIED IDEOGRAPH-#38C6, IRGKangXi=0361.250, Def{to loose the bow-string, a kind of jade, (same as 彌) full, great, boundless, to complete}
38C7 CJK UNIFIED IDEOGRAPH-#38C7, IRGKangXi=0362.080, Def{a kind of beast with long hair, other name for pig, fox, wild cat, raccoon}
38C8 CJK UNIFIED IDEOGRAPH-#38C8, IRGKangXi=0362.240, Def{a kind of animal, a kind of rat, (interchangeable 肆) extremely; excessively; reckless; without restraint, to exhaust}
38C9 CJK UNIFIED IDEOGRAPH-#38C9, IRGKangXi=0363.040, Def{(ancient form of 工) labor, work, fine; delicate, to be skilled in}
38CA CJK UNIFIED IDEOGRAPH-#38CA, IRGKangXi=0363.081
38CB CJK UNIFIED IDEOGRAPH-#38CB, IRGKangXi=0363.090, Def{(ancient form of 丹) cinnabar, red; scarlet, a pill, a sophisticated decoction}
38CC CJK UNIFIED IDEOGRAPH-#38CC, IRGKangXi=0363.110, Def{to join two things; contiguous; things connected end to end, sharp}
38CD CJK UNIFIED IDEOGRAPH-#38CD, IRGKangXi=0363.120, Def{(ancient form of 台) a raised platform, eminent; exalted, name of a star}
38CE CJK UNIFIED IDEOGRAPH-#38CE, IRGKangXi=0363.190, Def{fine (material, etc.) and delicate (workmanship, etc.) decorative design or pattern}
38CF CJK UNIFIED IDEOGRAPH-#38CF, IRGKangXi=0364.080, Def{(ancient form of 靜) still; quiet; calm; silent; peaceful}
38D0 CJK UNIFIED IDEOGRAPH-#38D0, IRGKangXi=0364.081, Def{(ancient form of 變) to change; to transform; to switch; to alter, rebellion}
38D1 CJK UNIFIED IDEOGRAPH-#38D1, IRGKangXi=0364.160, Def{double image, (non-classical form of 容) face; expression; countenance, to contain; to hold; to accept}
38D2 CJK UNIFIED IDEOGRAPH-#38D2, IRGKangXi=0364.210, Def{hair-raising, hairiness}
38D3 CJK UNIFIED IDEOGRAPH-#38D3, IRGKangXi=0364.240, Def{literary in taste; elegant; ornamental, beautiful and gorgeous}
38D4 CJK UNIFIED IDEOGRAPH-#38D4, IRGKangXi=0365.030, Def{to walk alone, to insist on one's ways in doing things}
38D5 CJK UNIFIED IDEOGRAPH-#38D5, IRGKangXi=0365.071
38D6 CJK UNIFIED IDEOGRAPH-#38D6, IRGKangXi=0365.211
38D7 CJK UNIFIED IDEOGRAPH-#38D7, IRGKangXi=0365.211
38D8 CJK UNIFIED IDEOGRAPH-#38D8, IRGKangXi=0365.240, Def{(same as 跔) spasms, cramps in the feet and legs, in cold day the joints of the feet and legs unable to stretch}
38D9 CJK UNIFIED IDEOGRAPH-#38D9, IRGKangXi=0365.310, Def{to walk in an easy and leisurely manner, (same as 宙) time--past, present and future, infinite time; time without beginning or end; eternity}
38DA CJK UNIFIED IDEOGRAPH-#38DA, IRGKangXi=0366.010, Def{to go straight forward, (same as 衕) a lane; an alley, (same as 侗) ignorant; rude; rustic; straight-forward; uninterrupted; no obstacle}
38DB CJK UNIFIED IDEOGRAPH-#38DB, IRGKangXi=0366.040, Def{walking, (ancient form form of 會) to meet, to gather, to be able, to realize}
38DC CJK UNIFIED IDEOGRAPH-#38DC, IRGKangXi=0366.060, Def{to walk; to go}
38DD CJK UNIFIED IDEOGRAPH-#38DD, IRGKangXi=0366.080, Def{a Chinese family name}
38DE CJK UNIFIED IDEOGRAPH-#38DE, IRGKangXi=0367.030, Def{to walk; to go}
38DF CJK UNIFIED IDEOGRAPH-#38DF, IRGKangXi=0367.040, Def{to go; to walk}
38E0 CJK UNIFIED IDEOGRAPH-#38E0, IRGKangXi=0367.051, Def{a rope decorated with bronze ornament (for leading cattle), a family name}
38E1 CJK UNIFIED IDEOGRAPH-#38E1, IRGKangXi=0367.051
38E2 CJK UNIFIED IDEOGRAPH-#38E2, IRGKangXi=0367.140, Def{to rest from work; to take a rest}
38E3 CJK UNIFIED IDEOGRAPH-#38E3, IRGKangXi=0367.170, Def{to walk in an unsteady way; to swing to and from, to doubt, to suspect}
38E4 CJK UNIFIED IDEOGRAPH-#38E4, IRGKangXi=0368.010, Def{(same as 踐) to tread upon; to trample, to carry out; to perform}
38E5 CJK UNIFIED IDEOGRAPH-#38E5, IRGKangXi=0369.010, Def{(same as 偫) to store up, savings and/or reserves, complete with every; having nothing short; all; the whole}
38E6 CJK UNIFIED IDEOGRAPH-#38E6, IRGKangXi=0369.020, Def{a winding path}
38E7 CJK UNIFIED IDEOGRAPH-#38E7, IRGKangXi=0369.081, Def{(same as 胤) the succession in a family; posterity; heirs, to inherit; to follow after}
38E8 CJK UNIFIED IDEOGRAPH-#38E8, IRGKangXi=0369.081
38E9 CJK UNIFIED IDEOGRAPH-#38E9, IRGKangXi=0369.081
38EA CJK UNIFIED IDEOGRAPH-#38EA, IRGKangXi=0369.100, Def{to walk with slow steps; proceed slowly}
38EB CJK UNIFIED IDEOGRAPH-#38EB, IRGKangXi=0369.110, Def{(same as 踵) the heel, to follow, to call personally at; to go personally to, (ancient form 動) to move; to start; to shake, to excite; to rouse; to take action}
38EC CJK UNIFIED IDEOGRAPH-#38EC, IRGKangXi=0369.210, Def{to walk}
38ED CJK UNIFIED IDEOGRAPH-#38ED, IRGKangXi=0369.250, Def{number, several; a few, a kind of caldron or kettle, to walk}
38EE CJK UNIFIED IDEOGRAPH-#38EE, IRGKangXi=0370.051
38EF CJK UNIFIED IDEOGRAPH-#38EF, IRGKangXi=0370.080, Def{the dresses toss and flying about in the wind, to shake; to toss, to wave, to scull; to row, to agitate}
38F0 CJK UNIFIED IDEOGRAPH-#38F0, IRGKangXi=0370.280, Def{to shake; to rattle}
38F1 CJK UNIFIED IDEOGRAPH-#38F1, IRGKangXi=0370.310, Def{common people; people in ordinary feature or countenance, to walk, to go in a hurry}
38F2 CJK UNIFIED IDEOGRAPH-#38F2, IRGKangXi=0370.320, Def{(non-classical form of 微) small, low, weak; feeble}
38F3 CJK UNIFIED IDEOGRAPH-#38F3, IRGKangXi=0371.191
38F4 CJK UNIFIED IDEOGRAPH-#38F4, IRGKangXi=0371.191
38F5 CJK UNIFIED IDEOGRAPH-#38F5, IRGKangXi=0371.220, Def{to run away; to flee; to escape; to evade, (interchangeable 達) to reach; to arrive at, to walk away without meeting each other}
38F6 CJK UNIFIED IDEOGRAPH-#38F6, IRGKangXi=0371.240, Def{to walk, to go, to hide; to conceal, very difficult to proceed}
38F7 CJK UNIFIED IDEOGRAPH-#38F7, IRGKangXi=0372.030, Def{walking}
38F8 CJK UNIFIED IDEOGRAPH-#38F8, IRGKangXi=0372.161
38F9 CJK UNIFIED IDEOGRAPH-#38F9, IRGKangXi=0372.201
38FA CJK UNIFIED IDEOGRAPH-#38FA, IRGKangXi=0375.011, Def{(variant of U+5FC3 心) the heart, the moral nature, the mind, the affections, intention, the radical on left side of the character}
38FB CJK UNIFIED IDEOGRAPH-#38FB, IRGKangXi=0375.070, Def{to remedy by punishment; to punish; to reprove; to warn, in great distress; suffering hardships; distress; trouble; worry}
38FC CJK UNIFIED IDEOGRAPH-#38FC, IRGKangXi=0376.050, Def{(same as 忍) to endure; to bear; to forbear; to repress}
38FD CJK UNIFIED IDEOGRAPH-#38FD, IRGKangXi=0376.180, Def{(ancient form of 恕) to forgive; to pardon, (ancient form 怒) anger; wrath; indignation; rage}
38FE CJK UNIFIED IDEOGRAPH-#38FE, IRGKangXi=0377.030, Def{do not care; unmindful of}
38FF CJK UNIFIED IDEOGRAPH-#38FF, IRGKangXi=0377.060, Def{painfully sad, scared; afraid; fearful, anger, melancholy; grievous; mournful; sad, to be concerned about}
3900 CJK UNIFIED IDEOGRAPH-#3900, IRGKangXi=0377.071
3901 CJK UNIFIED IDEOGRAPH-#3901, IRGKangXi=0377.090, Def{(standard form of 忝) to disgrace; to insult, to condescend}
3902 CJK UNIFIED IDEOGRAPH-#3902, IRGKangXi=0377.160, Def{(a variant of 急) anxious; hurried; urgent; hasty}
3903 CJK UNIFIED IDEOGRAPH-#3903, IRGKangXi=0377.230, Def{jealous; to envy, (same as 妨) to hinder; to obstruct, to harm; to damage}
3904 CJK UNIFIED IDEOGRAPH-#3904, IRGKangXi=0378.010, Def{raging animosity or hatred; full of anger and spite, unhappy; displeased}
3905 CJK UNIFIED IDEOGRAPH-#3905, IRGKangXi=0378.050, Def{to love, to be fond of, o like; love, affection, a breath, news, to stop; to end, (interchangeable 忌) jealous; to envy, (the small seal 愛) love}
3906 CJK UNIFIED IDEOGRAPH-#3906, IRGKangXi=0378.070, Def{nauseated; disgusted, (in medicine) an acute case, quick-tempered, to regret; to repent, small (piece of land); narrow (minded)}
3907 CJK UNIFIED IDEOGRAPH-#3907, IRGKangXi=0378.110, Def{terrified; scared; nervous, (same as 懊) regretful; remorseful; resentful}
3908 CJK UNIFIED IDEOGRAPH-#3908, IRGKangXi=0378.180, Def{sad; sorrowful}
3909 CJK UNIFIED IDEOGRAPH-#3909, IRGKangXi=0378.250, Def{fear; dread; fright; scare, crafty; low cunning}
390A CJK UNIFIED IDEOGRAPH-#390A, IRGKangXi=0378.330, Def{(same as 恔) cheerful and exuberant; spiritually elevated}
390B CJK UNIFIED IDEOGRAPH-#390B, IRGKangXi=0379.070, Def{confusion; confused or disorderly; chaotic}
390C CJK UNIFIED IDEOGRAPH-#390C, IRGKangXi=0379.080, Def{(same as 甘) to have one's heart own; to submit, admire, etc. sincerely and willingly}
390D CJK UNIFIED IDEOGRAPH-#390D, IRGKangXi=0380.020, Def{never reveal one's feeling or emotion, (same as 巧) false, artificial, illegal, clever; artful, pretty cute}
390E CJK UNIFIED IDEOGRAPH-#390E, IRGKangXi=0380.060, Def{a model or typical example; technique of standard handwriting, knowledge, to know; to feel; to understand}
390F CJK UNIFIED IDEOGRAPH-#390F, IRGKangXi=0380.170, Def{grievous; mournful; melancholy}
3910 CJK UNIFIED IDEOGRAPH-#3910, IRGKangXi=0380.200, Def{(said of music) disharmony; discord, quiet, to yield; to concede}
3911 CJK UNIFIED IDEOGRAPH-#3911, IRGKangXi=0380.240, Def{(non-classical 憂) palpitation or fluttering of the heart, to become interested in something (usually as a result of persuasion); to move the mind; to start thinking; agitated; nervous}
3912 CJK UNIFIED IDEOGRAPH-#3912, IRGKangXi=0381.120, Def{knowledge; to know; to be aware of, a bureau, state of affairs}
3913 CJK UNIFIED IDEOGRAPH-#3913, IRGKangXi=0381.130, Def{stupid; foolish; unwise; unintelligent}
3914 CJK UNIFIED IDEOGRAPH-#3914, IRGKangXi=0381.190, Def{to concede or submit willingly}
3915 CJK UNIFIED IDEOGRAPH-#3915, IRGKangXi=0381.230, Def{melancholy; grievous; mournful; sad, anxious and fearful, extremely sad}
3916 CJK UNIFIED IDEOGRAPH-#3916, IRGKangXi=0382.200, Def{to have wisdom; intelligent}
3917 CJK UNIFIED IDEOGRAPH-#3917, IRGKangXi=0383.111
3918 CJK UNIFIED IDEOGRAPH-#3918, IRGKangXi=0383.111, Def{(simplified form of U+396E 㥮) obstinate; stubborn; opinionated; obstinacy; stubbornness; intransigent, truculent; savage, ferocious; fierce}
3919 CJK UNIFIED IDEOGRAPH-#3919, IRGKangXi=0383.111, Def{(same as 恩) grace, favour, kindness, mercy}
391A CJK UNIFIED IDEOGRAPH-#391A, IRGKangXi=0383.130, Def{joy; delight; gratification}
391B CJK UNIFIED IDEOGRAPH-#391B, IRGKangXi=0383.170, Def{to remember (old days, friends, etc.); to recall, (a pronoun, a dialect) this; these; such, here, without restriction; at will, to trust; to show good faith, weak, (ancient form 飪) to cook, you, (dialect) so; that, than, that case, that (indicating degree)}
391C CJK UNIFIED IDEOGRAPH-#391C, IRGKangXi=0383.210, Def{(corrupted form of 怴) crazy; mad, anger; angry, idiotic; silly; stupid, ill-will; enmity; animus}
391D CJK UNIFIED IDEOGRAPH-#391D, IRGKangXi=0383.320, Def{palpitation or fluttering of the heart, to become interested in something (usually as a result of persuasion)}
391E CJK UNIFIED IDEOGRAPH-#391E, IRGKangXi=0383.370, Def{disappointed; to fail in attaining one's purpose; to be disappointed in attaining the ambitions, to be surprised; to be amazed; to marvel, to think; to consider; to ponder; to contemplate, to expect or hope with eagerness, a credit, straightforward; faithful and upright, to reprimand; to discipline; to punish}
391F CJK UNIFIED IDEOGRAPH-#391F, IRGKangXi=0383.410, Def{(same as 恐) fear; dread; fright; scare, (with reference to an unpleasant fact) probably; perhaps; maybe; possible; to threaten; to blackmail}
3920 CJK UNIFIED IDEOGRAPH-#3920, IRGKangXi=0384.010, Def{grieved; distressed}
3921 CJK UNIFIED IDEOGRAPH-#3921, IRGKangXi=0384.020, Def{to surprise; to amaze, afraid; scared; fearful; terrified, grieved; distressed; grieved; distressed}
3922 CJK UNIFIED IDEOGRAPH-#3922, IRGKangXi=0384.040, Def{to move the mind; agitated; nervous; to start thinking}
3923 CJK UNIFIED IDEOGRAPH-#3923, IRGKangXi=0384.070, Def{(same as 恲) noble; disinterested; generous; energetic; liberal, to divulge; to leak out, to feel dull and listless or depressed}
3924 CJK UNIFIED IDEOGRAPH-#3924, IRGKangXi=0384.100, Def{(non-classical form) mournful; melancholy; distressed; grieved; sad, to fear; fear; dread}
3925 CJK UNIFIED IDEOGRAPH-#3925, IRGKangXi=0384.340, Def{sorrowful, anxious}
3926 CJK UNIFIED IDEOGRAPH-#3926, IRGKangXi=0385.030, Def{mournful; melancholy; distressed; grieved; sad}
3927 CJK UNIFIED IDEOGRAPH-#3927, IRGKangXi=0385.170, Def{conciliation; to come to an amicable understanding; compromise}
3928 CJK UNIFIED IDEOGRAPH-#3928, IRGKangXi=0385.180, Def{scared; terrified, fear; dread; fright}
3929 CJK UNIFIED IDEOGRAPH-#3929, IRGKangXi=0386.050, Def{(same as 恪) to respect; to venerate, grave and stern; of ornament -- splendid, imposing}
392A CJK UNIFIED IDEOGRAPH-#392A, IRGKangXi=0386.100, Def{(same as 怨) to find fault with; to repine; to murmur against; to harbour resentment, hatred}
392B CJK UNIFIED IDEOGRAPH-#392B, IRGKangXi=0386.161, Def{(non-classical form)}
392C CJK UNIFIED IDEOGRAPH-#392C, IRGKangXi=0386.161, Def{(same as 恚) full of anger and vexation, rage; hatred; to hate; to bear a grudge against; animosity}
392D CJK UNIFIED IDEOGRAPH-#392D, IRGKangXi=0386.161, Def{(simplified form of 憍) (same as 驕) untamed; intractable; disobedient}
392E CJK UNIFIED IDEOGRAPH-#392E, IRGKangXi=0386.230, Def{an error; a blunder, to cheat; to swindle; to defraud, confused; deceit; to deceive}
392F CJK UNIFIED IDEOGRAPH-#392F, IRGKangXi=0386.390, Def{clever or smart in a small way, memory or recollection}
3930 CJK UNIFIED IDEOGRAPH-#3930, IRGKangXi=0386.400, Def{(same as 怍) to be ashamed; mortified; chagrined, to change color; to blush}
3931 CJK UNIFIED IDEOGRAPH-#3931, IRGKangXi=0387.040, Def{anger; passion; rage, to wait, to cheat; to swindle}
3932 CJK UNIFIED IDEOGRAPH-#3932, IRGKangXi=0387.070, Def{to think; to contemplate; to consider, to pant because of fear}
3933 CJK UNIFIED IDEOGRAPH-#3933, IRGKangXi=0387.210, Def{to depend on}
3934 CJK UNIFIED IDEOGRAPH-#3934, IRGKangXi=0387.260, Def{to move the mind; agitated; nervous; to start thinking, to rest; to cease, intelligent, strange; uncanny, light aspirations or ambitions for ...}
3935 CJK UNIFIED IDEOGRAPH-#3935, IRGKangXi=0387.270, Def{(non-classical form of 慈) compassion, mercy, kindness}
3936 CJK UNIFIED IDEOGRAPH-#3936, IRGKangXi=0387.280, Def{confuse, senile, stupid}
3937 CJK UNIFIED IDEOGRAPH-#3937, IRGKangXi=0387.330, Def{to debauch; to dissipate; to act uninhibitively or disrespectfully}
3938 CJK UNIFIED IDEOGRAPH-#3938, IRGKangXi=0388.090, Def{uneasy; disturbed; not feeling at peace}
3939 CJK UNIFIED IDEOGRAPH-#3939, IRGKangXi=0388.110, Def{hatred; grudge; enmity; animosity, (interchangeable 救) to save; to relieve; to rescue; to deliver; to aid}
393A CJK UNIFIED IDEOGRAPH-#393A, IRGKangXi=0386.060, Def{(corrupted form of 慌) blurred; dim, obscure; confused in mind}
393B CJK UNIFIED IDEOGRAPH-#393B, IRGKangXi=0388.341, Def{stupid; clumsy; (Cant.) to dirty, defile}
393C CJK UNIFIED IDEOGRAPH-#393C, IRGKangXi=0388.341
393D CJK UNIFIED IDEOGRAPH-#393D, IRGKangXi=0388.341, Def{(simplified form of 懤) grieved, pained}
393E CJK UNIFIED IDEOGRAPH-#393E, IRGKangXi=0388.380, Def{not doing well; disappointed; very discouraged; frustrate (same as 憛) worried; anxious; apprehensive, to lose head; to lose self-possession}
393F CJK UNIFIED IDEOGRAPH-#393F, IRGKangXi=0388.390, Def{to love; to be fond of, love; affection; kindness, willingly; willing, joyous; happy, to fail to remember; to forget}
3940 CJK UNIFIED IDEOGRAPH-#3940, IRGKangXi=0388.400, Def{exclusive; special (same as 德) morality; virtues, behavior; conduct}
3941 CJK UNIFIED IDEOGRAPH-#3941, IRGKangXi=0389.010, Def{(same as U+5FB7 德) morality; virtues, behavior; conduct}
3942 CJK UNIFIED IDEOGRAPH-#3942, IRGKangXi=0389.040, Def{quick; fast, sharp, to move the mind; agitated; nervous; to start thinking, to fear; to dread; to be scared of, to get; to obtain}
3943 CJK UNIFIED IDEOGRAPH-#3943, IRGKangXi=0389.060, Def{(a variant of 悶) mournful, sorrowful, melancholy; depressed, to stupefy}
3944 CJK UNIFIED IDEOGRAPH-#3944, IRGKangXi=0389.180, Def{to pity; to commiserate; to have sympathy or compassion for, scared; afraid; fearful}
3945 CJK UNIFIED IDEOGRAPH-#3945, IRGKangXi=0389.200, Def{used in a person's name, pensive}
3946 CJK UNIFIED IDEOGRAPH-#3946, IRGKangXi=0389.230, Def{indulgent and without restraint, to abandon oneself to carnal desire; to be dissolute; to debauched, to forget, slow; to delay}
3947 CJK UNIFIED IDEOGRAPH-#3947, IRGKangXi=0390.050, Def{to be perverse; to act contrary to, jealous; to envy; jealousy, obstinate; stubborn, fierce; truculent}
3948 CJK UNIFIED IDEOGRAPH-#3948, IRGKangXi=0390.080, Def{(interchangeable 惵) afraid; fearful; terrified, peaceful; quiet; calm; safe; secure; stable}
3949 CJK UNIFIED IDEOGRAPH-#3949, IRGKangXi=0390.120, Def{a little angry}
394A CJK UNIFIED IDEOGRAPH-#394A, IRGKangXi=0390.130, Def{angry, (same as 恲) generous; unselfish; liberal}
394B CJK UNIFIED IDEOGRAPH-#394B, IRGKangXi=0390.170, Def{sound of the anguish of sorrow; the pain of grief}
394C CJK UNIFIED IDEOGRAPH-#394C, IRGKangXi=0390.190, Def{prudent; cautious}
394D CJK UNIFIED IDEOGRAPH-#394D, IRGKangXi=0390.230, Def{(same as 惎) to bear jealous hatred for; to hate; to hold a grudge against, to injure atrociously; to murder}
394E CJK UNIFIED IDEOGRAPH-#394E, IRGKangXi=0390.360, Def{(abbreviated form) to resent; to hate, to neglect; negligent, joy; delight; gratification}
394F CJK UNIFIED IDEOGRAPH-#394F, IRGKangXi=0390.370, Def{bashful; shy; ashamed}
3950 CJK UNIFIED IDEOGRAPH-#3950, IRGKangXi=0390.410, Def{(ancient form of 怨) hatred; enmity; resentment, to complain; to blame; to impute}
3951 CJK UNIFIED IDEOGRAPH-#3951, IRGKangXi=0390.460, Def{(corrupted form) (standard form of 憂) sad; mournful; grieved; to worry about; concerned over; anxious; apprehensive}
3952 CJK UNIFIED IDEOGRAPH-#3952, IRGKangXi=0391.020, Def{corrupt; wicked, to hate; to dislike, (same as 猜) to be jealous and suspicious, cunning; crafty}
3953 CJK UNIFIED IDEOGRAPH-#3953, IRGKangXi=0391.040, Def{happy; contented; pleased, fragile, nearly; almost, cunning; crafty}
3954 CJK UNIFIED IDEOGRAPH-#3954, IRGKangXi=0391.100, Def{to move the mind; agitated; nervous; to start thinking}
3955 CJK UNIFIED IDEOGRAPH-#3955, IRGKangXi=0392.160, Def{(ancient form of 憐) to pity; to commiserate; to feel tender regard for}
3956 CJK UNIFIED IDEOGRAPH-#3956, IRGKangXi=0392.181, Def{red color}
3957 CJK UNIFIED IDEOGRAPH-#3957, IRGKangXi=0392.181
3958 CJK UNIFIED IDEOGRAPH-#3958, IRGKangXi=0392.181
3959 CJK UNIFIED IDEOGRAPH-#3959, IRGKangXi=0392.181
395A CJK UNIFIED IDEOGRAPH-#395A, IRGKangXi=0392.190, Def{melancholy; grievous; mournful; sad, to fear; to dread; to be scared of}
395B CJK UNIFIED IDEOGRAPH-#395B, IRGKangXi=0392.200, Def{with expedition; urgent; hasty, anxious; worried, respectfully serious; to take careful precautions; to do something in a serious manner, to love; to like; to be fond of; to be kind to}
395C CJK UNIFIED IDEOGRAPH-#395C, IRGKangXi=0392.280, Def{unsuccessful in one's career; to have not one's ambition fulfilled, generous; unselfish; liberal, uneasy; disturbed; not feeling at peace}
395D CJK UNIFIED IDEOGRAPH-#395D, IRGKangXi=0393.160, Def{to arouse to action; to encourage, to temper and grind; to train or discipline oneself (toward a goal); to sharpen (a knife); to forge to harden, (same as 弭) to stop; to end; to eliminate}
395E CJK UNIFIED IDEOGRAPH-#395E, IRGKangXi=0393.230, Def{to think deeply, to think far ahead, profound and abstruse thinking, careful deliberations; to think and contemplate thoroughly}
395F CJK UNIFIED IDEOGRAPH-#395F, IRGKangXi=0393.240, Def{enmity; animus; ill-will}
3960 CJK UNIFIED IDEOGRAPH-#3960, IRGKangXi=0394.020, Def{(same as 諝) intelligence; brilliance, shrewdness; cunning; quick-witted and full of tricks; tricky}
3961 CJK UNIFIED IDEOGRAPH-#3961, IRGKangXi=0394.090, Def{idle; shiftless}
3962 CJK UNIFIED IDEOGRAPH-#3962, IRGKangXi=0394.110, Def{bad; evil; wicked; vice, overbearing; haughty, melancholy; depressed; dejected; cheerless; despondent, considerations; contemplations; to think carefully; to turn over and over in the mind}
3963 CJK UNIFIED IDEOGRAPH-#3963, IRGKangXi=0394.150, Def{(ancient form of 惠) to benefit; to profit; to favor, kind; benevolent; gracious; gentle}
3964 CJK UNIFIED IDEOGRAPH-#3964, IRGKangXi=0394.340, Def{(non-classical form of 矜) to pity; to feel for, to have compassion on, sympathetic}
3965 CJK UNIFIED IDEOGRAPH-#3965, IRGKangXi=0395.110, Def{absent-minded; unconscious, happy; glad; joyous}
3966 CJK UNIFIED IDEOGRAPH-#3966, IRGKangXi=0395.170, Def{(a variant of 愜) satisfied; contented; cheerful, appropriate; fitting; apposite, to have one's heart won; to summit; admire, etc. sincerely and willingly}
3967 CJK UNIFIED IDEOGRAPH-#3967, IRGKangXi=0395.230, Def{(ancient form of 順) to follow; to obedient; agreeable; favorable, convenient; smooth}
3968 CJK UNIFIED IDEOGRAPH-#3968, IRGKangXi=0395.290, Def{unhappy displeased, anger; rage; angry; furious}
3969 CJK UNIFIED IDEOGRAPH-#3969, IRGKangXi=0396.020, Def{idle; lazy; slothful; inactive}
396A CJK UNIFIED IDEOGRAPH-#396A, IRGKangXi=0396.071, Def{(simplified form of U+617A 慺) diligent; industrious; sedulous, to encourage; to make efforts}
396B CJK UNIFIED IDEOGRAPH-#396B, IRGKangXi=0396.071, Def{(a variant of 惇) honest; sincere; kind; generous}
396C CJK UNIFIED IDEOGRAPH-#396C, IRGKangXi=0396.190, Def{fear; dread; fright; scare}
396D CJK UNIFIED IDEOGRAPH-#396D, IRGKangXi=0396.200, Def{luxury; waste; wasteful; prodigal; extravagant}
396E CJK UNIFIED IDEOGRAPH-#396E, IRGKangXi=0396.240, Def{obstinate; stubborn; opinionated; obstinacy; stubbornness; intransigent, truculent; savage, ferocious; fierce}
396F CJK UNIFIED IDEOGRAPH-#396F, IRGKangXi=0396.260, Def{careful; compassionate; to take an interest in; prudent; cautious, worried and grieved}
3970 CJK UNIFIED IDEOGRAPH-#3970, IRGKangXi=0396.290, Def{mournful; melancholy; distressed; grieved; sad}
3971 CJK UNIFIED IDEOGRAPH-#3971, IRGKangXi=0396.310, Def{(same as 悱) to be desirous of speaking; inarticulate; unable to give vent to one's emotion, sorrowful}
3972 CJK UNIFIED IDEOGRAPH-#3972, IRGKangXi=0397.010, Def{(same as 謓) anger; rage; angry; furious, (ancient form 慎) cautious; careful; scrupulous; prudent}
3973 CJK UNIFIED IDEOGRAPH-#3973, IRGKangXi=0397.070, Def{to survey; to measure; mensuration}
3974 CJK UNIFIED IDEOGRAPH-#3974, IRGKangXi=0397.090, Def{do not care about something; disregard, to be ashamed}
3975 CJK UNIFIED IDEOGRAPH-#3975, IRGKangXi=0397.120, Def{(a variant of 慁) to dishonour; to disgrace; to distress, grief; shame; to disobey, to be anxious; to be apprehensive; to worry, to disturb or to agitate; to harass, confused and disorderly}
3976 CJK UNIFIED IDEOGRAPH-#3976, IRGKangXi=0397.130, Def{(same as 塞) (same as 愆) to fill up; full of; filled with; rich in contents; abundance, a fault; a mistake, to exceed; to be more than, to surpass}
3977 CJK UNIFIED IDEOGRAPH-#3977, IRGKangXi=0398.010, Def{(same as 瘱) quiet; calm; still peaceful, gracefully quiet, clear and evident; obvious, deep and far; profound and abstruse, to conceal; to hide}
3978 CJK UNIFIED IDEOGRAPH-#3978, IRGKangXi=0398.060, Def{(ancient form of 愍) to sympathize with; to pity; to grieve; to commiserate}
3979 CJK UNIFIED IDEOGRAPH-#3979, IRGKangXi=0398.210, Def{to move}
397A CJK UNIFIED IDEOGRAPH-#397A, IRGKangXi=0398.261, Def{(same as 熆) to blow a fire}
397B CJK UNIFIED IDEOGRAPH-#397B, IRGKangXi=0398.261
397C CJK UNIFIED IDEOGRAPH-#397C, IRGKangXi=0398.360, Def{to be near to, close to, to recollect; to recall; to look back upon}
397D CJK UNIFIED IDEOGRAPH-#397D, IRGKangXi=0398.410, Def{feeling; emotion, fact, love; affection, nature; reason, (same as 責) one's duty, responsibility; obligation, etc., to demand; to punish, to upbraid}
397E CJK UNIFIED IDEOGRAPH-#397E, IRGKangXi=0399.050, Def{mortified; ashamed}
397F CJK UNIFIED IDEOGRAPH-#397F, IRGKangXi=0399.090, Def{(same as 慠) proud; haughty; over bearing rude}
3980 CJK UNIFIED IDEOGRAPH-#3980, IRGKangXi=0399.250, Def{joy; delight; gratification}
3981 CJK UNIFIED IDEOGRAPH-#3981, IRGKangXi=0399.260, Def{to sob; to weep, to pay attention, to be careful; to take care; to exercise caution}
3982 CJK UNIFIED IDEOGRAPH-#3982, IRGKangXi=0399.300, Def{(ancient form of 常) constantly, frequently, usually habitually, regular, common, a rule, a principle}
3983 CJK UNIFIED IDEOGRAPH-#3983, IRGKangXi=0400.090, Def{perfect personal character, many; much; plenty; lots of}
3984 CJK UNIFIED IDEOGRAPH-#3984, IRGKangXi=0400.100, Def{hard to say or predict, difficult to speak out (for fear of embarrassing or paining others, etc.) not easy to express with words}
3985 CJK UNIFIED IDEOGRAPH-#3985, IRGKangXi=0401.010, Def{(same as 慸) uneasy; feeling uneasy or uncomfortable; not feeling at peace}
3986 CJK UNIFIED IDEOGRAPH-#3986, IRGKangXi=0401.050, Def{to exaggerate; exaggeration, arrogant; showing wild imagination; preposterously fantastic, without ambition; discourage}
3987 CJK UNIFIED IDEOGRAPH-#3987, IRGKangXi=0401.160, Def{a peaceful or easy mood; calm at mind, to move; to turn the mind, conversion}
3988 CJK UNIFIED IDEOGRAPH-#3988, IRGKangXi=0401.391
3989 CJK UNIFIED IDEOGRAPH-#3989, IRGKangXi=0401.410, Def{(ancient form of 懿) virtuous, especially women, admirable; esteemed; excellent}
398A CJK UNIFIED IDEOGRAPH-#398A, IRGKangXi=0401.420, Def{lavish; luxurious, wasteful}
398B CJK UNIFIED IDEOGRAPH-#398B, IRGKangXi=0402.060, Def{absurd; wild; nonsensical; showing wild imagination; preposterously fantastic}
398C CJK UNIFIED IDEOGRAPH-#398C, IRGKangXi=0402.090, Def{keen or sharp; clever; simple and honest, sound sleep or slumber; a deep sleep, to wake up from sleep}
398D CJK UNIFIED IDEOGRAPH-#398D, IRGKangXi=0402.100, Def{(same as 愕) to be startled, alarmed, to speak out; to speak up, frank statement, an obstacle or hindrance; to obstruct; to impede; unwilling; disagreeable}
398E CJK UNIFIED IDEOGRAPH-#398E, IRGKangXi=0402.260, Def{cantankerous; perverse, stupid; dull, ignorant, to divide clearly; without ambiguity, gracefully quiet}
398F CJK UNIFIED IDEOGRAPH-#398F, IRGKangXi=0403.100, Def{weak; feeble, joy; delight; gratification}
3990 CJK UNIFIED IDEOGRAPH-#3990, IRGKangXi=0403.230, Def{a straightforward character, cannot make up one's mind; to hesitate}
3991 CJK UNIFIED IDEOGRAPH-#3991, IRGKangXi=0403.250, Def{anger; rage; angry; furious, silly; stupid; foolish}
3992 CJK UNIFIED IDEOGRAPH-#3992, IRGKangXi=0403.340, Def{to deceived, loquacious}
3993 CJK UNIFIED IDEOGRAPH-#3993, IRGKangXi=0403.350, Def{difficult intention, difficult idea or sentiment, weakened volition; dejected, fear; scare; fright}
3994 CJK UNIFIED IDEOGRAPH-#3994, IRGKangXi=0404.090, Def{(same as 懕) sickly, in poor health, undisturbed; composed; calm; quiet, to satisfy or be satisfied to gratify or be gratified; (same as 厭) content; gratification}
3995 CJK UNIFIED IDEOGRAPH-#3995, IRGKangXi=0404.160, Def{thought; idea; sentiment, intention, expectation, meaning}
3996 CJK UNIFIED IDEOGRAPH-#3996, IRGKangXi=0404.301, Def{(same as 懣) resentful; sullen; sulky; anger (especially at injustice) indignant, annoyed; depressed; bored}
3997 CJK UNIFIED IDEOGRAPH-#3997, IRGKangXi=0405.030, Def{diligent; sedulous; industrious, with a strong will power}
3998 CJK UNIFIED IDEOGRAPH-#3998, IRGKangXi=0405.070, Def{(same as 憿) swift; rapid; quick; fast, by luck or chance}
3999 CJK UNIFIED IDEOGRAPH-#3999, IRGKangXi=0406.141
399A CJK UNIFIED IDEOGRAPH-#399A, IRGKangXi=0406.200, Def{melancholy; grievous; mournful, urgent; pressing}
399B CJK UNIFIED IDEOGRAPH-#399B, IRGKangXi=0406.250, Def{to walk in a composed (comfortable) way, respectful; reverent}
399C CJK UNIFIED IDEOGRAPH-#399C, IRGKangXi=0406.280, Def{to move the mind; agitated; nervous; to start thinking, melancholy; grievous; mournful, (same as 驚) to surprise; to amaze, afraid; scared; terrified; fearful}
399D CJK UNIFIED IDEOGRAPH-#399D, IRGKangXi=0406.440, Def{beautiful; pretty, good; excellent, to usurp; to outrage, to draw; to sketch}
399E CJK UNIFIED IDEOGRAPH-#399E, IRGKangXi=0407.030, Def{(same as 懤) grieved; pained; in deep grief; to lament deeply}
399F CJK UNIFIED IDEOGRAPH-#399F, IRGKangXi=0407.050, Def{intelligent; astute; clever}
39A0 CJK UNIFIED IDEOGRAPH-#39A0, IRGKangXi=0407.161
39A1 CJK UNIFIED IDEOGRAPH-#39A1, IRGKangXi=0407.310, Def{(non-classical form of 樂) happy; glad, joyful, cheerful; elated; content; delighted; pleased; willing, enjoyable}
39A2 CJK UNIFIED IDEOGRAPH-#39A2, IRGKangXi=0407.470, Def{pure; virtuous; devotion; dedication, to have bearings; to have manners}
39A3 CJK UNIFIED IDEOGRAPH-#39A3, IRGKangXi=0407.560, Def{to talk in one's sleep}
39A4 CJK UNIFIED IDEOGRAPH-#39A4, IRGKangXi=0408.050, Def{virtuous, especially of women, admirable, esteemed, excellent}
39A5 CJK UNIFIED IDEOGRAPH-#39A5, IRGKangXi=0408.130, Def{(same as 愋) talented; intelligent; wise wisdom; knowledge, to resent; to hate, to regret}
39A6 CJK UNIFIED IDEOGRAPH-#39A6, IRGKangXi=0408.140, Def{apprehensive; fearful; afraid, ashamed, uneasy in mind; not feeling at peace; disturbed}
39A7 CJK UNIFIED IDEOGRAPH-#39A7, IRGKangXi=0408.291, Def{(same as U+61AF 憯 U+6158 慘) sad; sorrowful; grieved; miserable; tragic}
39A8 CJK UNIFIED IDEOGRAPH-#39A8, IRGKangXi=0408.310, Def{(same as 嬾) lazy; indolent; idle; inactive; reluctant; disinclined}
39A9 CJK UNIFIED IDEOGRAPH-#39A9, IRGKangXi=0408.350, Def{melancholy; grievous; mournful; to feel sorrow or grief; to grieve; to mourn; to be sad, prudent; cautious}
39AA CJK UNIFIED IDEOGRAPH-#39AA, IRGKangXi=0408.421, Def{complaint; grudge; unjust}
39AB CJK UNIFIED IDEOGRAPH-#39AB, IRGKangXi=0409.070, Def{lax and crude; idle and remiss}
39AC CJK UNIFIED IDEOGRAPH-#39AC, IRGKangXi=0409.130, Def{ashamed; abashed; mortified; humiliated, few; little scarce; rare}
39AD CJK UNIFIED IDEOGRAPH-#39AD, IRGKangXi=0409.300, Def{(same as 憐 怜) to pity; to have compassion on; to sympathize}
39AE CJK UNIFIED IDEOGRAPH-#39AE, IRGKangXi=0411.091, Def{(abbreviated form of 錢) money; cash, a unit of weight, a Chinese family name}
39AF CJK UNIFIED IDEOGRAPH-#39AF, IRGKangXi=0412.081, Def{(same as 或) a certain; some, perhaps; probably; maybe, or}
39B0 CJK UNIFIED IDEOGRAPH-#39B0, IRGKangXi=0412.110, Def{to break off (relations); to sever, to exterminate; to annihilate; to wipe out, to pierce; to stab; to irritate; to hurt, to hold weapons, agricultural implements; far tools, sharp will; eager intention; determination}
39B1 CJK UNIFIED IDEOGRAPH-#39B1, IRGKangXi=0412.221
39B2 CJK UNIFIED IDEOGRAPH-#39B2, IRGKangXi=0412.221, Def{(non-classical form of 哉) a particle expressing surprise, admiration, or grief, an expletive}
39B3 CJK UNIFIED IDEOGRAPH-#39B3, IRGKangXi=0412.291, Def{to cultivate, plant; to care for plants}
39B4 CJK UNIFIED IDEOGRAPH-#39B4, IRGKangXi=0412.300, Def{to arrest; to catch; to seize, to brawl; a hand-to-hand fight; to struggle for; to fight for, to beat; to strike; to attack}
39B5 CJK UNIFIED IDEOGRAPH-#39B5, IRGKangXi=0412.310, Def{to kill; to put to death; to slaughter; to execute, to punish, to weed out; to exterminate, a kind of weapon}
39B6 CJK UNIFIED IDEOGRAPH-#39B6, IRGKangXi=0412.350, Def{repeatedly; frequently; to alternate, common; ordinary; normal, sharp, name of a state in ancient China, bold; brave; fierce; violent; severe; strict; stringent}
39B7 CJK UNIFIED IDEOGRAPH-#39B7, IRGKangXi=0413.020, Def{(ancient form of 勇) brave; courageous; bold; valiant; fearless; gallant; heroic}
39B8 CJK UNIFIED IDEOGRAPH-#39B8, IRGKangXi=0413.141, Def{a lance with two points, a halberd with a crescent -shaped blade; weapons used in ancient times, to stimulate; to provoke; to excite; to irritate, to point with the index finger and the middle finger; to describe angry or an awe-inspiring display of military force, etc., masculine; heroic; brave}
39B9 CJK UNIFIED IDEOGRAPH-#39B9, IRGKangXi=0413.190, Def{a spear; a lance; a javelin, a battle-axe; a halberd}
39BA CJK UNIFIED IDEOGRAPH-#39BA, IRGKangXi=0413.280, Def{a lance with two points, a halberd with a crescent-shaped blade; weapons used in ancient times, to stimulate; to provoke; to excite; to irritate, to point with the index finger and the middle finger; to describe angry or an awe-inspiring display of military force, etc., masculine; heroic; brave, (in general) sharp points and edges of weapons}
39BB CJK UNIFIED IDEOGRAPH-#39BB, IRGKangXi=0413.380, Def{(ancient form of 襲) to put on; to wear, repeated; double; to inherit; to attack or take by surprise, to accord with; to unite}
39BC CJK UNIFIED IDEOGRAPH-#39BC, IRGKangXi=0414.010, Def{a wooden piles used as a cable to make immovable of a boat}
39BD CJK UNIFIED IDEOGRAPH-#39BD, IRGKangXi=0414.210, Def{with elegant appearance, with beautiful or gorgeous color; refined, learned and accomplished}
39BE CJK UNIFIED IDEOGRAPH-#39BE, IRGKangXi=0415.050, Def{(same as U+6245 扊) the bolt of a door; door latch}
39BF CJK UNIFIED IDEOGRAPH-#39BF, IRGKangXi=0415.090, Def{a window, a small door, (ancient form 戶) a door, a household}
39C0 CJK UNIFIED IDEOGRAPH-#39C0, IRGKangXi=0415.140, Def{the bolt of a door; door latch}
39C1 CJK UNIFIED IDEOGRAPH-#39C1, IRGKangXi=0415.190, Def{to close, shut the door with a bang, to soar, (ancient form 闔) a Chinese family name}
39C2 CJK UNIFIED IDEOGRAPH-#39C2, IRGKangXi=0416.060, Def{(same as 屇) a cave a hold in the ground; a hole}
39C3 CJK UNIFIED IDEOGRAPH-#39C3, IRGKangXi=0417.040, Def{(ancient form of 收) to receive what is due; to gather together, to harvest, to collect; to put away, to close, to bind, to restrain, to bring to the end, (same as 摎) entwined -- as branches of a tree; to tangle; to involve, to collaborate; to band together}
39C4 CJK UNIFIED IDEOGRAPH-#39C4, IRGKangXi=0417.080, Def{to take; to receive; to fetch; to take hold of}
39C5 CJK UNIFIED IDEOGRAPH-#39C5, IRGKangXi=0417.131, Def{to turn upside down; to exert a great deal; to pull and drag, to implicate or involve}
39C6 CJK UNIFIED IDEOGRAPH-#39C6, IRGKangXi=0417.150, Def{to cover something with a piece of cloth, a kind of cloth used to cover something}
39C7 CJK UNIFIED IDEOGRAPH-#39C7, IRGKangXi=0417.190, Def{to hold; to grasp, to maintain; to support; to keep; to uphold, a tie or stalemate}
39C8 CJK UNIFIED IDEOGRAPH-#39C8, IRGKangXi=0418.090, Def{(same as 引) to pull; to attract, to guide, to introduce, to quote, to retire}
39C9 CJK UNIFIED IDEOGRAPH-#39C9, IRGKangXi=0418.180, Def{to rub; to grind; to polish; to wear, to take; to obtain; to select, to beat; to strike; to attack}
39CA CJK UNIFIED IDEOGRAPH-#39CA, IRGKangXi=0418.220, Def{to push; to expel, to shirk; to decline, to row, to wipe and clean, to strike; to beat, to be rampant in defiance of authority}
39CB CJK UNIFIED IDEOGRAPH-#39CB, IRGKangXi=0419.080, Def{to lengthen and extenuate -- as in making noodle; to drag out}
39CC CJK UNIFIED IDEOGRAPH-#39CC, IRGKangXi=0420.080, Def{(same as 毛 芼) to select; to choose; to pick out}
39CD CJK UNIFIED IDEOGRAPH-#39CD, IRGKangXi=0420.130, Def{(same as 仿) alike; similar, (same as 搒) to involve; to affect}
39CE CJK UNIFIED IDEOGRAPH-#39CE, IRGKangXi=0421.030, Def{forked branches, stone roller, to roll, to stop; to detain; to prohibit, to clutch; to grasp}
39CF CJK UNIFIED IDEOGRAPH-#39CF, IRGKangXi=0422.121, Def{(a simplified form of 掆) to carry on the shoulders of two or more men, to raise}
39D0 CJK UNIFIED IDEOGRAPH-#39D0, IRGKangXi=0422.121, Def{(simplified form of 㩳) to hold; to grasp, to detain, to uphold, to push, to stand upright}
39D1 CJK UNIFIED IDEOGRAPH-#39D1, IRGKangXi=0422.121, Def{(a simplified form 撝) to wave, to assist}
39D2 CJK UNIFIED IDEOGRAPH-#39D2, IRGKangXi=0423.030, Def{to beat; to strike, to throw; to pitch, to scoop out to dig out}
39D3 CJK UNIFIED IDEOGRAPH-#39D3, IRGKangXi=0423.070, Def{to lead; to guide, to entice, to introduce, to draw out; to stretch, to beat; to strike}
39D4 CJK UNIFIED IDEOGRAPH-#39D4, IRGKangXi=0424.110, Def{to disturbe; to agitate; to harass, to quarrel; to wrangle}
39D5 CJK UNIFIED IDEOGRAPH-#39D5, IRGKangXi=0424.150, Def{to feel or touch with hands; to hold, to lay the hand on, to cover}
39D6 CJK UNIFIED IDEOGRAPH-#39D6, IRGKangXi=0424.200, Def{to clutch; to grasp}
39D7 CJK UNIFIED IDEOGRAPH-#39D7, IRGKangXi=0425.040, Def{to catch, to grasp with the hand, to pinch, to kill, to attach from the sideway, to strike with the hand; (Cant.) to tear, rip}
39D8 CJK UNIFIED IDEOGRAPH-#39D8, IRGKangXi=0425.060, Def{to accumulate, to massage the cheeks}
39D9 CJK UNIFIED IDEOGRAPH-#39D9, IRGKangXi=0426.060, Def{to give a playful blow, to strike against, to wring; to wrench; to twist, to turn for the better}
39DA CJK UNIFIED IDEOGRAPH-#39DA, IRGKangXi=0427.161, Def{(a dialect character) cup the hand, (corrupted form of U+65CA 瓬) clay pottery; earthenware}
39DB CJK UNIFIED IDEOGRAPH-#39DB, IRGKangXi=0427.161, Def{(simplified form of U+6394 掔) thick; firm; substantial, to drag along; to pull, to lead}
39DC CJK UNIFIED IDEOGRAPH-#39DC, IRGKangXi=0427.161, Def{(same as U+64F8 擸) to hold, to grasp, to hold the hair; to pull at; (Cant.) to take at a glance}
39DD CJK UNIFIED IDEOGRAPH-#39DD, IRGKangXi=0427.161, Def{to take}
39DE CJK UNIFIED IDEOGRAPH-#39DE, IRGKangXi=0427.161
39DF CJK UNIFIED IDEOGRAPH-#39DF, IRGKangXi=0427.161, Def{(simplified form of 擓) to wipe; to scrub; to rub, to dust; to clean, (a dialect) to scratch with fingers lightly, (a dialect) to carry on the arm}
39E0 CJK UNIFIED IDEOGRAPH-#39E0, IRGKangXi=0427.161
39E1 CJK UNIFIED IDEOGRAPH-#39E1, IRGKangXi=0428.010, Def{to move; to shake; to rock; to excite, to decrease; to reduce; to diminish; to subtract; to deduct, to shoulder; to take upon on oneself}
39E2 CJK UNIFIED IDEOGRAPH-#39E2, IRGKangXi=0429.020, Def{to depend on; to lean to; to rely on; by means of, to follow the old or traditional (practices, customs, precedents, etc.), to receive and carry on; to continue; to succeed to}
39E3 CJK UNIFIED IDEOGRAPH-#39E3, IRGKangXi=0429.080, Def{to stop, to detain; to stay; to end; still; calm, only}
39E4 CJK UNIFIED IDEOGRAPH-#39E4, IRGKangXi=0429.120, Def{(non-classical form) to jump; to leap; to bounce; (in general) get angry and leave away, to bump; (Cant.) to poke, jab}
39E5 CJK UNIFIED IDEOGRAPH-#39E5, IRGKangXi=0429.130, Def{to nip with the fingers; to take a pinch; to take up as with tongs}
39E6 CJK UNIFIED IDEOGRAPH-#39E6, IRGKangXi=0429.190, Def{to strike; to beat; to attach, (same as 揈) to move; to shake; to wield}
39E7 CJK UNIFIED IDEOGRAPH-#39E7, IRGKangXi=0429.230, Def{(non-classical form of 梳) a comb; a coarse comb; to comb}
39E8 CJK UNIFIED IDEOGRAPH-#39E8, IRGKangXi=0429.240, Def{(ancient form of 扏) slow; gradual; tardy; leisurely, to delay; to put off; to defer}
39E9 CJK UNIFIED IDEOGRAPH-#39E9, IRGKangXi=0429.250, Def{to dispel, to move; to transfer, to issue; to set aside, to stir up, turning}
39EA CJK UNIFIED IDEOGRAPH-#39EA, IRGKangXi=0430.040, Def{to ruin; to destroy; to break down, to hang up; to suspend}
39EB CJK UNIFIED IDEOGRAPH-#39EB, IRGKangXi=0430.080, Def{to hold; to grasp, to maintain; to keep, a post, to lead on, (same as 擩) to rub, to crumble by hand, to massage; mixed-up; confused}
39EC CJK UNIFIED IDEOGRAPH-#39EC, IRGKangXi=0430.110, Def{to raise; to fold the hands on the breast, to bow, to salute, a method (to pile or to raise up) of painting; (Cant.) to push from behind}
39ED CJK UNIFIED IDEOGRAPH-#39ED, IRGKangXi=0431.020, Def{to get something with both raising hands}
39EE CJK UNIFIED IDEOGRAPH-#39EE, IRGKangXi=0431.120, Def{high; tall; lofty}
39EF CJK UNIFIED IDEOGRAPH-#39EF, IRGKangXi=0431.171, Def{a bucket made of willow, (a dialect) to lift (especially when only a single person is involved)}
39F0 CJK UNIFIED IDEOGRAPH-#39F0, IRGKangXi=0427.161, Def{(a simplified form) to strike; to tap, to shock}
39F1 CJK UNIFIED IDEOGRAPH-#39F1, IRGKangXi=0431.171, Def{(same as U+62FF 拿) to bring, to take, to apprehend, to grasp}
39F2 CJK UNIFIED IDEOGRAPH-#39F2, IRGKangXi=0431.230, Def{to shake; to rattle}
39F3 CJK UNIFIED IDEOGRAPH-#39F3, IRGKangXi=0431.270, Def{to mix; to blend, to display in neat rows; to place in order}
39F4 CJK UNIFIED IDEOGRAPH-#39F4, IRGKangXi=0431.350, Def{to rub hands; to rub between the hands; to twist between the hands}
39F5 CJK UNIFIED IDEOGRAPH-#39F5, IRGKangXi=0432.060, Def{(corrupted form of U+638A 掊) to exact, to get salt from sea-water, to break up; to injure, to hold something in both hands; to scoop up mud; (Cant.) to hit, strike, knock}
39F6 CJK UNIFIED IDEOGRAPH-#39F6, IRGKangXi=0432.110, Def{the jingling of metals, to strike, to lead; to guide}
39F7 CJK UNIFIED IDEOGRAPH-#39F7, IRGKangXi=0432.200, Def{to wipe tears, to bind}
39F8 CJK UNIFIED IDEOGRAPH-#39F8, IRGKangXi=0433.220, Def{to hit; to beat; to strike; (Cant.) to drive away, chase}
39F9 CJK UNIFIED IDEOGRAPH-#39F9, IRGKangXi=0434.250, Def{to strike with fists, to strike; to blow}
39FA CJK UNIFIED IDEOGRAPH-#39FA, IRGKangXi=0434.290, Def{a finger wrapper; a covering (used to protect the fingers from getting hurt)}
39FB CJK UNIFIED IDEOGRAPH-#39FB, IRGKangXi=0434.300, Def{a thorn, to select; to pick, to push, to find out, to cut out; to engrave, sound of striking}
39FC CJK UNIFIED IDEOGRAPH-#39FC, IRGKangXi=0434.310, Def{to distribute, to scatter; to disperse}
39FD CJK UNIFIED IDEOGRAPH-#39FD, IRGKangXi=0435.020, Def{name of a county in Han dynasty; today's Inner Mongolia Autonomous Region; southeast of Baotou county}
39FE CJK UNIFIED IDEOGRAPH-#39FE, IRGKangXi=0435.040, Def{to strike, to bale out; to clean up}
39FF CJK UNIFIED IDEOGRAPH-#39FF, IRGKangXi=0435.110, Def{to bump; to knock against; to strike accidentally; to collide}
3A00 CJK UNIFIED IDEOGRAPH-#3A00, IRGKangXi=0435.170, Def{(same as 抦) to hold in the hand}
3A01 CJK UNIFIED IDEOGRAPH-#3A01, IRGKangXi=0436.080, Def{to throw; to pitch, to send, to stay, a crutch; staff or stick}
3A02 CJK UNIFIED IDEOGRAPH-#3A02, IRGKangXi=0436.090, Def{to beat; to strike; to attack}
3A03 CJK UNIFIED IDEOGRAPH-#3A03, IRGKangXi=0436.110, Def{a row; a line, to push; to clear out, (same as 朾) to bump; to knock against; to strike accidentally; to collide; (Cant.) to poke, nudge, stab; to stretch out}
3A04 CJK UNIFIED IDEOGRAPH-#3A04, IRGKangXi=0437.150, Def{to pierce; to stab, (same as 挑) to stir; to disturb; to agitate, to place the hand on, to impeach, (a dialect) to lift something heavy from one side or from the end}
3A05 CJK UNIFIED IDEOGRAPH-#3A05, IRGKangXi=0438.090, Def{inside; within; inner, a family name, (corrupted form of 抐) to take from the top, to strip; to peel}
3A06 CJK UNIFIED IDEOGRAPH-#3A06, IRGKangXi=0438.130, Def{to kill, to slaughter, to beat; to strike; to hit; to attack; (Cant.) to pile, stack}
3A07 CJK UNIFIED IDEOGRAPH-#3A07, IRGKangXi=0440.040, Def{(same as 搫) to sweep clean; to clean away, to eliminate}
3A08 CJK UNIFIED IDEOGRAPH-#3A08, IRGKangXi=0440.181, Def{(abbreviated form of 擠) to crowd; to throng; to push; to squeeze}
3A09 CJK UNIFIED IDEOGRAPH-#3A09, IRGKangXi=0440.240, Def{(same as U+62BF 抿) to smooth, to stroke; to pucker, to contract, to wipe away}
3A0A CJK UNIFIED IDEOGRAPH-#3A0A, IRGKangXi=0441.060, Def{to discard; to reject; to abandon, to feel; to stroke, to sort out the divining stalks, to drop; to lose; to fall off, to weigh; to measure weight}
3A0B CJK UNIFIED IDEOGRAPH-#3A0B, IRGKangXi=0441.080, Def{to beat; to strike; to attach}
3A0C CJK UNIFIED IDEOGRAPH-#3A0C, IRGKangXi=0441.180, Def{to milk a cow or goat}
3A0D CJK UNIFIED IDEOGRAPH-#3A0D, IRGKangXi=0441.190, Def{to defend, on a level with; even; equal, to help; to assist}
3A0E CJK UNIFIED IDEOGRAPH-#3A0E, IRGKangXi=0442.110, Def{to dye, to dip, to soak; to rub in the hands}
3A0F CJK UNIFIED IDEOGRAPH-#3A0F, IRGKangXi=0443.120, Def{to beat; to strike; to attack}
3A10 CJK UNIFIED IDEOGRAPH-#3A10, IRGKangXi=0443.160, Def{to slap lightly on the clothes or coverlet, to beat; to strike; to attack}
3A11 CJK UNIFIED IDEOGRAPH-#3A11, IRGKangXi=0443.190, Def{to count; to enumerate, to grasp the head}
3A12 CJK UNIFIED IDEOGRAPH-#3A12, IRGKangXi=0443.200, Def{(same as 刲) to cut open and clean -- as fish, to kill; to sacrifice, to stab}
3A13 CJK UNIFIED IDEOGRAPH-#3A13, IRGKangXi=0444.080, Def{(non-classical form) to close; to shut, to block up, to pull or drag out of the water; to fish up; to salvage}
3A14 CJK UNIFIED IDEOGRAPH-#3A14, IRGKangXi=0444.210, Def{to shake; to rock; to jolt; to joggle}
3A15 CJK UNIFIED IDEOGRAPH-#3A15, IRGKangXi=0445.030, Def{to shoulder; to take on the shoulder; to take upon oneself}
3A16 CJK UNIFIED IDEOGRAPH-#3A16, IRGKangXi=0445.060, RSKangXi=64.9, Def{to stab, to rob of money; to rob of riches, to go nonstop to, to cluster together; to mass; (Cant.) to squeeze in}
3A17 CJK UNIFIED IDEOGRAPH-#3A17, IRGKangXi=0445.341, Def{(variant of 捷) alert, nimble, prompt, clever, smart}
3A18 CJK UNIFIED IDEOGRAPH-#3A18, IRGKangXi=0445.341, Def{(same as U+64E4 擤) to blow the nose with the fingers; (Cant.) to scour; to rebuke; to hit with a ball}
3A19 CJK UNIFIED IDEOGRAPH-#3A19, IRGKangXi=0445.350, Def{to clasp or hold under the arm, to support; to prop up; to aid, to exchange; to alter, to lift up or off; to raise high}
3A1A CJK UNIFIED IDEOGRAPH-#3A1A, IRGKangXi=0445.410, Def{to select; to choose; to pick out, to defend; to ward off, to refuse; to reject}
3A1B CJK UNIFIED IDEOGRAPH-#3A1B, IRGKangXi=0446.010, Def{swift; rapid quick; speedy, (same as 掞) easy; smooth; suave; comfortable}
3A1C CJK UNIFIED IDEOGRAPH-#3A1C, IRGKangXi=0446.030, Def{to help each other, to shoulder; to take upon oneself}
3A1D CJK UNIFIED IDEOGRAPH-#3A1D, IRGKangXi=0446.090, Def{the very hard stuff, to sort out of divining stalks; (Cant.) to wedge in}
3A1E CJK UNIFIED IDEOGRAPH-#3A1E, IRGKangXi=0446.110, Def{to feel about; to grope in the dark}
3A1F CJK UNIFIED IDEOGRAPH-#3A1F, IRGKangXi=0446.130, Def{to touch; to contact, to ram; to butt}
3A20 CJK UNIFIED IDEOGRAPH-#3A20, IRGKangXi=0446.150, Def{(same as 塓) to smear; to daub; to spread, to paint}
3A21 CJK UNIFIED IDEOGRAPH-#3A21, IRGKangXi=0446.230, Def{to push with hands, to hold; to grasp, (in general) something as a whole; complete; undamaged}
3A22 CJK UNIFIED IDEOGRAPH-#3A22, IRGKangXi=0447.130, Def{to rip open flesh; to cut flesh; (Cant.) to join, link, connect}
3A23 CJK UNIFIED IDEOGRAPH-#3A23, IRGKangXi=0447.220
3A24 CJK UNIFIED IDEOGRAPH-#3A24, IRGKangXi=0448.090, Def{(same as 擓) to wipe; to scrub; to rub, to dust; to clean, (a dialect) to scratch with fingers lightly, (a dialect) to carry on the arm}
3A25 CJK UNIFIED IDEOGRAPH-#3A25, IRGKangXi=0448.140, Def{to hit; to strike}
3A26 CJK UNIFIED IDEOGRAPH-#3A26, IRGKangXi=0448.200, Def{(abbreviated form) to hold; to grasp, to detain, to uphold, to push, to stand upright}
3A27 CJK UNIFIED IDEOGRAPH-#3A27, IRGKangXi=0448.210, Def{wooden stick or bamboo cane (shaped segment of a circle) used to support the covering of a carriage in ancient times}
3A28 CJK UNIFIED IDEOGRAPH-#3A28, IRGKangXi=0448.340, Def{(the large seal; a type of Chinese calligraphy) to draw out; to sprout; to rid; to whip, to build, to thresh; to hull or unhusk, to beat; to pound; (Cant.) to shake down}
3A29 CJK UNIFIED IDEOGRAPH-#3A29, IRGKangXi=0448.370, Def{(non-classical form of 捷) to win; to triumph, swift; quick; rapid}
3A2A CJK UNIFIED IDEOGRAPH-#3A2A, IRGKangXi=0449.081, Def{to shake; to be disturbed; oscillating, brightness, dazzling; glaring}
3A2B CJK UNIFIED IDEOGRAPH-#3A2B, IRGKangXi=0449.081, Def{(same as U+652C 攬) to be in full possession of; to monopolize, to grasp; to seize}
3A2C CJK UNIFIED IDEOGRAPH-#3A2C, IRGKangXi=0449.081
3A2D CJK UNIFIED IDEOGRAPH-#3A2D, IRGKangXi=0449.190, Def{unreasonable, to distribute; to make known; to circulate, book cover; book jacket, letter; correspondence}
3A2E CJK UNIFIED IDEOGRAPH-#3A2E, IRGKangXi=0449.200, Def{to lift up; to take in the hand, to control, to open; (Cant.) to seize, grasp}
3A2F CJK UNIFIED IDEOGRAPH-#3A2F, IRGKangXi=0449.230, Def{(same as 擴) to enlarge; to magnify; to expand}
3A30 CJK UNIFIED IDEOGRAPH-#3A30, IRGKangXi=0449.240, Def{to turn; to shift; to move}
3A31 CJK UNIFIED IDEOGRAPH-#3A31, IRGKangXi=0449.330, Def{to shake, to wave, to row}
3A32 CJK UNIFIED IDEOGRAPH-#3A32, IRGKangXi=0449.350, Def{to back up; to support, to take; to receive; to obtain, to select; to choose}
3A33 CJK UNIFIED IDEOGRAPH-#3A33, IRGKangXi=0450.060, Def{(same as 庋) a cupboard; a closet, to put into a cupboard or closet; to put into the proper place, to wear on the head, to support; to sustain; to bear}
3A34 CJK UNIFIED IDEOGRAPH-#3A34, IRGKangXi=0450.130, Def{to turn over; to collapse; (Cant.) to lift (a lid); to uncover}
3A35 CJK UNIFIED IDEOGRAPH-#3A35, IRGKangXi=0450.160, Def{(non-classical form of 揃) to cut away; to cut off; to cut down; to eliminate; to remove; to exterminate}
3A36 CJK UNIFIED IDEOGRAPH-#3A36, IRGKangXi=0450.190, Def{(same as 搗 擣) to thresh; to hull or unhusk, to beat; to pound; to attack}
3A37 CJK UNIFIED IDEOGRAPH-#3A37, IRGKangXi=0450.200, Def{to wipe; to cleanse, (interchangeable 艱) difficult; distress; hardship}
3A38 CJK UNIFIED IDEOGRAPH-#3A38, IRGKangXi=0451.030, Def{to hit; to strike; to beat}
3A39 CJK UNIFIED IDEOGRAPH-#3A39, IRGKangXi=0451.040, Def{to crack; to break; to rip open, to split or divide up, to hang up; to suspend, to sweep clean; to eliminate}
3A3A CJK UNIFIED IDEOGRAPH-#3A3A, IRGKangXi=0451.170, Def{to ornament; to polish; to decorate, (same as 捫) to feel; to touch with hands; to hold, to search (in one's pocket, etc.)}
3A3B CJK UNIFIED IDEOGRAPH-#3A3B, IRGKangXi=0451.220, Def{to cut; to kill; to behead, gradually; little by little; by degrees, to lift; to raise, to strike; to beat, to throw; to pitch; to deliver}
3A3C CJK UNIFIED IDEOGRAPH-#3A3C, IRGKangXi=0452.090, Def{(non-classical form of U+63A0 掠) to take by force, to throw aside}
3A3D CJK UNIFIED IDEOGRAPH-#3A3D, IRGKangXi=0453.040, Def{to cut or rip open}
3A3E CJK UNIFIED IDEOGRAPH-#3A3E, IRGKangXi=0453.191, Def{a type; a model; a mode; a style}
3A3F CJK UNIFIED IDEOGRAPH-#3A3F, IRGKangXi=0453.191, Def{(a variant of 據) to receive, as communications from a subordinate, to rely on, to lean on, evidence; proof, according to; whereas}
3A40 CJK UNIFIED IDEOGRAPH-#3A40, IRGKangXi=0453.191, Def{(a variant of 據) to receive, as communications from a subordinate, to rely on, to lean on, evidence; proof, according to; whereas}
3A41 CJK UNIFIED IDEOGRAPH-#3A41, IRGKangXi=0453.191, Def{to knock, to pick out, to ridicule, (a variant) monopoly, a footbridge. a toll levied at a bridge or a ferry}
3A42 CJK UNIFIED IDEOGRAPH-#3A42, IRGKangXi=0453.191
3A43 CJK UNIFIED IDEOGRAPH-#3A43, IRGKangXi=0453.330, Def{to take or capture (a city, etc.), to gather or to collect}
3A44 CJK UNIFIED IDEOGRAPH-#3A44, IRGKangXi=0454.050, Def{to strike; to beat, to get scattered and lost}
3A45 CJK UNIFIED IDEOGRAPH-#3A45, IRGKangXi=0454.090, Def{(ancient form of U+62BD 抽) to draw out; to pull out; to take out, to sprout; to put forth shoots, to rid; to take away; to whip; (Cant.) to shake down}
3A46 CJK UNIFIED IDEOGRAPH-#3A46, IRGKangXi=0454.120, Def{to be in full possession of; to grasp, to beat cruelly; to beat brutally; to beat severely}
3A47 CJK UNIFIED IDEOGRAPH-#3A47, IRGKangXi=0454.130, Def{the thumb, to split; to tear apart, sound of ripping or tearing, to cut open with something sharp}
3A48 CJK UNIFIED IDEOGRAPH-#3A48, IRGKangXi=0454.140, Def{to restrain; restrained; timid and awkward; to feel not at home; miserably poor}
3A49 CJK UNIFIED IDEOGRAPH-#3A49, IRGKangXi=0455.150, Def{(same as 搨) to take a rubbing of an inscription on stone, etc., (same as 拉) to pull; to drag; to hold, to beat; to strike; to attack, (same as 拹) to break; to destroy; to smash}
3A4A CJK UNIFIED IDEOGRAPH-#3A4A, IRGKangXi=0455.160, Def{to pull up the sleeves and show the arms, to fight with bare hands, to desire for more than one's rightful share; to covet; greedy}
3A4B CJK UNIFIED IDEOGRAPH-#3A4B, IRGKangXi=0456.040, Def{to beat; to strike; to attack, sound of beating or hitting, a handle; a hold; to hold; to take, to guard; to watch over}
3A4C CJK UNIFIED IDEOGRAPH-#3A4C, IRGKangXi=0456.060, Def{to strike; to fight with hands, to scratch lightly; to brush; to shake; to dust, to knock down; to floor}
3A4D CJK UNIFIED IDEOGRAPH-#3A4D, IRGKangXi=0456.110, Def{to back up; to support, to take; to receive; to fetch; to obtain; to take hold of}
3A4E CJK UNIFIED IDEOGRAPH-#3A4E, IRGKangXi=0456.200, Def{(same as 擪) to press with a finger; to tuck in; to put the finger into; to put in; to stow away}
3A4F CJK UNIFIED IDEOGRAPH-#3A4F, IRGKangXi=0457.161
3A50 CJK UNIFIED IDEOGRAPH-#3A50, IRGKangXi=0457.161
3A51 CJK UNIFIED IDEOGRAPH-#3A51, IRGKangXi=0457.161
3A52 CJK UNIFIED IDEOGRAPH-#3A52, IRGKangXi=0457.280, Def{(same as U+64D2 擒) to arrest; to capture; (Cant.) to press down firmly}
3A53 CJK UNIFIED IDEOGRAPH-#3A53, IRGKangXi=0458.030, Def{to wound; to damage; to destroy; to injure}
3A54 CJK UNIFIED IDEOGRAPH-#3A54, IRGKangXi=0458.100, Def{a tool to make (crumble and to make smooth) a bow, to beat; to strike; to attack}
3A55 CJK UNIFIED IDEOGRAPH-#3A55, IRGKangXi=0458.270, Def{(same as 拎) to haul; to take; to lift, to raise}
3A56 CJK UNIFIED IDEOGRAPH-#3A56, IRGKangXi=0458.340, Def{to back up; to support}
3A57 CJK UNIFIED IDEOGRAPH-#3A57, IRGKangXi=0459.030, Def{(non-classical form of 攜) to lead by the hand, to conduct to; to take with}
3A58 CJK UNIFIED IDEOGRAPH-#3A58, IRGKangXi=0459.100, Def{(corrupted form of 檥) (interchangeable 艤) to moor to the bank}
3A59 CJK UNIFIED IDEOGRAPH-#3A59, IRGKangXi=0459.201, Def{(same as 塞) a cork or stopper; to cork; to seal, to fill up, to block}
3A5A CJK UNIFIED IDEOGRAPH-#3A5A, IRGKangXi=0460.010, Def{to collect (tax, grains), to draw together; to contract, to become less flagrant in behavior}
3A5B CJK UNIFIED IDEOGRAPH-#3A5B, IRGKangXi=0460.050, Def{(same as 摶) to roll round with the hand, to rely on, to take or follow (a trail, etc.)}
3A5C CJK UNIFIED IDEOGRAPH-#3A5C, IRGKangXi=0460.070, Def{(same as U+652C 攬) to be in possession of; to grasp, to make selective collection or coverage of; (Cant.) actually; strong}
3A5D CJK UNIFIED IDEOGRAPH-#3A5D, IRGKangXi=0460.190, Def{to compare; to estimate}
3A5E CJK UNIFIED IDEOGRAPH-#3A5E, IRGKangXi=0460.340, Def{(corrupted form) to back up; to support, to take; to receive; to fetch; to obtain; to take hold of; (Cant.) to throw, heave, fling away}
3A5F CJK UNIFIED IDEOGRAPH-#3A5F, IRGKangXi=0461.011, Def{(a dialect) to sew some ornaments (button; decorations, etc.) on the clothes}
3A60 CJK UNIFIED IDEOGRAPH-#3A60, IRGKangXi=0461.100, Def{a fierce or bloody battle, to untie; to unbind, (same as 捊) to exchange; to trade, to guide, to quote}
3A61 CJK UNIFIED IDEOGRAPH-#3A61, IRGKangXi=0461.120, Def{shake; to toss, to wave; to sway, to row}
3A62 CJK UNIFIED IDEOGRAPH-#3A62, IRGKangXi=0461.140, Def{to beat; to strike; to attack, to cut, to diminish, to decide; to judge, to weigh; to measure, to wipe; to rub, to dust; to clean}
3A63 CJK UNIFIED IDEOGRAPH-#3A63, IRGKangXi=0461.170, Def{(same as 揮) to move; to shake; to wield; to direct, to arouse}
3A64 CJK UNIFIED IDEOGRAPH-#3A64, IRGKangXi=0461.230, Def{to open; to stretch; to extend; to display; to raise, to scatter; to spread, to stir; to get excited}
3A65 CJK UNIFIED IDEOGRAPH-#3A65, IRGKangXi=0461.250, Def{(non-classical form of 攕) delicate hand (of a woman)}
3A66 CJK UNIFIED IDEOGRAPH-#3A66, IRGKangXi=0461.320, Def{(same as 攜) to lead by the hand, to conduct to; to take with, a Chinese family name}
3A67 CJK UNIFIED IDEOGRAPH-#3A67, IRGKangXi=0462.010, Def{to strike; to beat; to knock; to stone (to throw stone and hit someone); sound of hitting something}
3A68 CJK UNIFIED IDEOGRAPH-#3A68, IRGKangXi=0462.060, Def{to crack; to break, to split or divide up, to rend; to rip open, to hand up; to suspend, to worry; to think of; anxious}
3A69 CJK UNIFIED IDEOGRAPH-#3A69, IRGKangXi=0462.070, Def{(a dialect) to come to an end; to terminate}
3A6A CJK UNIFIED IDEOGRAPH-#3A6A, IRGKangXi=0462.130, Def{(corrupted form of 擷) to collect; to take up; to pick; to gather}
3A6B CJK UNIFIED IDEOGRAPH-#3A6B, IRGKangXi=0462.161
3A6C CJK UNIFIED IDEOGRAPH-#3A6C, IRGKangXi=0462.170, Def{to beat; to strike; to attack}
3A6D CJK UNIFIED IDEOGRAPH-#3A6D, IRGKangXi=0462.250, Def{(non-classical form of 攪) to disturb or to agitate; to bother; to trouble, to stir; to churn; to mix evenly by stirring}
3A6E CJK UNIFIED IDEOGRAPH-#3A6E, IRGKangXi=0463.030, Def{to lift up or off; to raise high, to unveil}
3A6F CJK UNIFIED IDEOGRAPH-#3A6F, IRGKangXi=0463.040, Def{(same as 搫) to move; to transport, to collect; to gather; to make a clean sweep of}
3A70 CJK UNIFIED IDEOGRAPH-#3A70, IRGKangXi=0463.210, Def{to stab; to cut, (same as 摷) to attack against}
3A71 CJK UNIFIED IDEOGRAPH-#3A71, IRGKangXi=0463.240, Def{to select; to choose; to pick out, to lift; to carry on the shoulders -- of two or more men, to wipe out, to brush over lightly, to carry; to take or bring along at one's convenience, to cut; to par; to trim; to shave}
3A72 CJK UNIFIED IDEOGRAPH-#3A72, IRGKangXi=0463.260, Def{a fist, to clasp}
3A73 CJK UNIFIED IDEOGRAPH-#3A73, IRGKangXi=0463.280, Def{to hold; to grasp, to detain, to uphold, to push, to stand upright; expel, to shirk, to raise up}
3A74 CJK UNIFIED IDEOGRAPH-#3A74, IRGKangXi=0463.300, Def{to seize, as a bird of prey, to seize and carry off, to grasp}
3A75 CJK UNIFIED IDEOGRAPH-#3A75, IRGKangXi=0465.020, Def{to beat; to strike; to attack}
3A76 CJK UNIFIED IDEOGRAPH-#3A76, IRGKangXi=0465.040, Def{to hold; to grasp; to seize, to retain}
3A77 CJK UNIFIED IDEOGRAPH-#3A77, IRGKangXi=0465.051
3A78 CJK UNIFIED IDEOGRAPH-#3A78, IRGKangXi=0465.130, Def{(a variant) to pile on; to fold up, to draw together; to collect}
3A79 CJK UNIFIED IDEOGRAPH-#3A79, IRGKangXi=0465.141, Def{to pile on; to fold up}
3A7A CJK UNIFIED IDEOGRAPH-#3A7A, IRGKangXi=0467.070, Def{(a dialect) a crack on the utensil}
3A7B CJK UNIFIED IDEOGRAPH-#3A7B, IRGKangXi=0467.130, Def{rugged mountain path, very skinny, to pile on upon another; accumulation; to accumulate; (Cant.) exclamation}
3A7C CJK UNIFIED IDEOGRAPH-#3A7C, IRGKangXi=0467.150, Def{many; much, more than; over}
3A7D CJK UNIFIED IDEOGRAPH-#3A7D, IRGKangXi=0467.170, Def{parasitic branches on a tree, branches growing horizontally, (variant 勑) imperial orders}
3A7E CJK UNIFIED IDEOGRAPH-#3A7E, IRGKangXi=0467.200, Def{uneven, to hang down; to let fall}
3A7F CJK UNIFIED IDEOGRAPH-#3A7F, IRGKangXi=0468.140, Def{unstable, dull, rough; coarse}
3A80 CJK UNIFIED IDEOGRAPH-#3A80, IRGKangXi=0468.191, Def{to enter, to go ahead; to proceed; to advance, to improve, to offer}
3A81 CJK UNIFIED IDEOGRAPH-#3A81, IRGKangXi=0468.240, Def{to hold; to grasp; irregular; uneven, a precipice beneath an overhanging cliff, (same as 拑) to take by force; to control with threat of force; to coerce}
3A82 CJK UNIFIED IDEOGRAPH-#3A82, IRGKangXi=0469.110, Def{unstable, dull, rough; coarse}
3A83 CJK UNIFIED IDEOGRAPH-#3A83, IRGKangXi=0469.130, Def{to thump; to beat; to strike; to attack}
3A84 CJK UNIFIED IDEOGRAPH-#3A84, IRGKangXi=0469.160, Def{broken; to break, ruined, reason; cause; right, principle; theory, to arrange; to repair, to regulate; to operate; to govern}
3A85 CJK UNIFIED IDEOGRAPH-#3A85, IRGKangXi=0469.220, Def{to change, to alter}
3A86 CJK UNIFIED IDEOGRAPH-#3A86, IRGKangXi=0470.030, Def{hidden; mysterious; secret, to conceal}
3A87 CJK UNIFIED IDEOGRAPH-#3A87, IRGKangXi=0470.130, Def{to disseminate; to spread, the weak sound of a Chinese musical instrument with seven or five strings}
3A88 CJK UNIFIED IDEOGRAPH-#3A88, IRGKangXi=0470.140, Def{to let go, to put, to raid; to aggress, to encroach up, a bad year; a year of famine or disaster}
3A89 CJK UNIFIED IDEOGRAPH-#3A89, IRGKangXi=0470.291, Def{(ancient form 合) to combine; to unite; to gather, to close; to shut}
3A8A CJK UNIFIED IDEOGRAPH-#3A8A, IRGKangXi=0470.300, Def{infested with bandits and robbers}
3A8B CJK UNIFIED IDEOGRAPH-#3A8B, IRGKangXi=0470.310, Def{(same as 扞, 捍) to resist; to oppose; to obstruct, to defend; to guard; to ward off, (same as 㪃) to thump; to beat; to strike; to attack}
3A8C CJK UNIFIED IDEOGRAPH-#3A8C, IRGKangXi=0470.330, Def{to lead on; to advance, to strike against; to break through}
3A8D CJK UNIFIED IDEOGRAPH-#3A8D, IRGKangXi=0470.370, Def{(same as 勃) suddenly, to change, as the countenance, a kind of animal (interchangeable 悖) perverse; contrary to what is right, to rebel}
3A8E CJK UNIFIED IDEOGRAPH-#3A8E, IRGKangXi=0471.070, Def{uncertain; not yet settled, irregular, to raise one's hands, to exhaust; to use up, to complete; to finish, all; entirely; totally; completely}
3A8F CJK UNIFIED IDEOGRAPH-#3A8F, IRGKangXi=0472.070, Def{(interchangeable 捭) to ruin; to destroy; to break down, sound of beating or striking}
3A90 CJK UNIFIED IDEOGRAPH-#3A90, IRGKangXi=0472.090, Def{unstable, dull, rough; coarse}
3A91 CJK UNIFIED IDEOGRAPH-#3A91, IRGKangXi=0472.190, Def{to follow; to attach; connected}
3A92 CJK UNIFIED IDEOGRAPH-#3A92, IRGKangXi=0472.240, Def{to ruin; to destroy; to break down}
3A93 CJK UNIFIED IDEOGRAPH-#3A93, IRGKangXi=0472.310, Def{disaster; calamity, evil; wicked; mean}
3A94 CJK UNIFIED IDEOGRAPH-#3A94, IRGKangXi=0472.320, Def{(same as 散) to scatter; to disperse; to break up or separate for a time; to dismiss}
3A95 CJK UNIFIED IDEOGRAPH-#3A95, IRGKangXi=0473.060, Def{to beat; to pound; to strike; to rush on; to pat; to dash; to smash, to take by force; to wrest from}
3A96 CJK UNIFIED IDEOGRAPH-#3A96, IRGKangXi=0473.070, Def{sound of beating, to strike, sound of an object dropping into water, etc.}
3A97 CJK UNIFIED IDEOGRAPH-#3A97, IRGKangXi=0473.110, Def{(a dialect) to open (a parcel; a bundle or a package); to unroll (a scroll, etc.); (Cant.) to rest, catch one's breath}
3A98 CJK UNIFIED IDEOGRAPH-#3A98, IRGKangXi=0473.171, Def{(abbreviated form of 斂) to draw together; to accumulate, to collect; to gather}
3A99 CJK UNIFIED IDEOGRAPH-#3A99, IRGKangXi=0473.171, Def{to go to the very source; to search into carefully}
3A9A CJK UNIFIED IDEOGRAPH-#3A9A, IRGKangXi=0473.171, Def{(same as 散) to scatter; to disperse; to break up or separate for a time; to dismiss}
3A9B CJK UNIFIED IDEOGRAPH-#3A9B, IRGKangXi=0473.210, Def{to pound rocks to pieces, (same as 抌) to strike; to strike with the fist}
3A9C CJK UNIFIED IDEOGRAPH-#3A9C, IRGKangXi=0473.220, Def{(same as 揣) to measure; to estimate; to suppose; to gauge what's pon another's mind; to presume; to test; to sound out, (same as 捶) to weigh in the hand}
3A9D CJK UNIFIED IDEOGRAPH-#3A9D, IRGKangXi=0474.030, Def{to beat, (same as 柬 揀) to select; to choose}
3A9E CJK UNIFIED IDEOGRAPH-#3A9E, IRGKangXi=0474.090, Def{to be held by the hand, to resist; to oppose}
3A9F CJK UNIFIED IDEOGRAPH-#3A9F, IRGKangXi=0474.130, Def{(same as 敦) to regard as important, to esteem, honest; sincere; generous}
3AA0 CJK UNIFIED IDEOGRAPH-#3AA0, IRGKangXi=0474.190, Def{(same as U+9D6E 鵮) to peck, poverty; poor, things of the same value, to take; to fetch; to obtain, to select; to choose, (of a bird) to peck}
3AA1 CJK UNIFIED IDEOGRAPH-#3AA1, IRGKangXi=0474.240, Def{to quell; to punish by force of arms, enemy; foe, to oppose; to resist}
3AA2 CJK UNIFIED IDEOGRAPH-#3AA2, IRGKangXi=0474.350, Def{(same as 捎) to beat; to strike; to attack, to be in full possession of; to grasp, to make selective collection or coverage of}
3AA3 CJK UNIFIED IDEOGRAPH-#3AA3, IRGKangXi=0474.401, Def{a place in Han dynasty, (same as U+6572 敲) to beat; to knock}
3AA4 CJK UNIFIED IDEOGRAPH-#3AA4, IRGKangXi=0474.420, Def{to finish; concluded}
3AA5 CJK UNIFIED IDEOGRAPH-#3AA5, IRGKangXi=0474.470, Def{to take; to receive; fetch; to obtain; to take hold of, to press with fingers, use the fingers to show the distance}
3AA6 CJK UNIFIED IDEOGRAPH-#3AA6, IRGKangXi=0474.500, Def{to beat; to pound; to ram down, to attack}
3AA7 CJK UNIFIED IDEOGRAPH-#3AA7, IRGKangXi=0475.130, Def{to beat; to strike; to attack}
3AA8 CJK UNIFIED IDEOGRAPH-#3AA8, IRGKangXi=0475.220, Def{to mend; to repair; to copy; to write out}
3AA9 CJK UNIFIED IDEOGRAPH-#3AA9, IRGKangXi=0475.230, Def{to strike; to beat, sound of beating}
3AAA CJK UNIFIED IDEOGRAPH-#3AAA, IRGKangXi=0475.321, Def{to use a wooden pole to support something; to raise, to begin; to start}
3AAB CJK UNIFIED IDEOGRAPH-#3AAB, IRGKangXi=0476.050, Def{(corrupted form of 叡) wise and clever, shrewd, discreet, astute, quick of perception, the divine sagacity of sages}
3AAC CJK UNIFIED IDEOGRAPH-#3AAC, IRGKangXi=0476.080, Def{to pierce; to stab; to hurt, to give; to hand over; to confer, to teach, to build, to pound (grain) in order to remove the husk}
3AAD CJK UNIFIED IDEOGRAPH-#3AAD, IRGKangXi=0476.200, Def{to draw together; to contract to fold, to collect; to gather, to hide; to conceal}
3AAE CJK UNIFIED IDEOGRAPH-#3AAE, IRGKangXi=0476.280, Def{to beat; to attack; to strike}
3AAF CJK UNIFIED IDEOGRAPH-#3AAF, IRGKangXi=0477.011, Def{(non-classical form of U+8209 舉) to lift; to raise; to commend; to praise, entire; whole}
3AB0 CJK UNIFIED IDEOGRAPH-#3AB0, IRGKangXi=0477.060, Def{(ancient form of 齋) to abstain from meat, wine, etc., to fast; penance}
3AB1 CJK UNIFIED IDEOGRAPH-#3AB1, IRGKangXi=0477.130, Def{elegant; coloured}
3AB2 CJK UNIFIED IDEOGRAPH-#3AB2, IRGKangXi=0478.011
3AB3 CJK UNIFIED IDEOGRAPH-#3AB3, IRGKangXi=0478.011
3AB4 CJK UNIFIED IDEOGRAPH-#3AB4, IRGKangXi=0478.030, Def{to measure; to estimate}
3AB5 CJK UNIFIED IDEOGRAPH-#3AB5, IRGKangXi=0478.040, Def{half of a unit of measurement used in ancient times}
3AB6 CJK UNIFIED IDEOGRAPH-#3AB6, IRGKangXi=0478.060, Def{(same as 斛) a dry measure 10 or 5 times that of a Chinese peck; a corn measure nominally holding ten pecks but generally holding about five, used in name of a place}
3AB7 CJK UNIFIED IDEOGRAPH-#3AB7, IRGKangXi=0478.130
3AB8 CJK UNIFIED IDEOGRAPH-#3AB8, IRGKangXi=0478.171, Def{(same as 斟) to fill a cup with; to pour into a cup}
3AB9 CJK UNIFIED IDEOGRAPH-#3AB9, IRGKangXi=0478.250, Def{to rob; to loot; to plunder}
3ABA CJK UNIFIED IDEOGRAPH-#3ABA, IRGKangXi=0479.010, Def{to bale out; to decant liquids, to pour out liquor, a feast, a kind of utensil used to ladle out water}
3ABB CJK UNIFIED IDEOGRAPH-#3ABB, IRGKangXi=0479.030, Def{a kind of utensil used to ladle out wine, a container that gives a measurement of rice}
3ABC CJK UNIFIED IDEOGRAPH-#3ABC, IRGKangXi=0479.130, Def{to pull; to drag; to hinder; to snatch away}
3ABD CJK UNIFIED IDEOGRAPH-#3ABD, IRGKangXi=0479.151
3ABE CJK UNIFIED IDEOGRAPH-#3ABE, IRGKangXi=0479.160, Def{to cut the flesh from the bone; to scrape off; to pick out; to get rid of, to arrest; to catch to seize, to struggle; to fight; to conflict; to vie; to compete}
3ABF CJK UNIFIED IDEOGRAPH-#3ABF, IRGKangXi=0479.161, Def{to break; to snap; to bend; to bow down}
3AC0 CJK UNIFIED IDEOGRAPH-#3AC0, IRGKangXi=0480.091, Def{huge tripod of bronze with two ears; heavy three-legged caldron or sacrificial vessel regarded as a type of imperial power, the Empire, a kind of cooking utensil used in ancient times}
3AC1 CJK UNIFIED IDEOGRAPH-#3AC1, IRGKangXi=0480.170, Def{(non-classical form of 斷) to cut apart, to sever, to break; broken; to abstain from, to conclude}
3AC2 CJK UNIFIED IDEOGRAPH-#3AC2, IRGKangXi=0480.210, Def{to chop or cut (wood)}
3AC3 CJK UNIFIED IDEOGRAPH-#3AC3, IRGKangXi=0481.050, Def{flags flying, long bands or ribbons attached to flags, streamers, etc. (ancient form of 偃) to cease; to desist from}
3AC4 CJK UNIFIED IDEOGRAPH-#3AC4, IRGKangXi=0481.100, Def{(same as 旁) side, by the side of; nearby}
3AC5 CJK UNIFIED IDEOGRAPH-#3AC5, IRGKangXi=0482.090, Def{(same as 旗) a flag; a pennant; a banner; a streamer, a sign}
3AC6 CJK UNIFIED IDEOGRAPH-#3AC6, IRGKangXi=0482.101
3AC7 CJK UNIFIED IDEOGRAPH-#3AC7, IRGKangXi=0482.101
3AC8 CJK UNIFIED IDEOGRAPH-#3AC8, IRGKangXi=0482.101
3AC9 CJK UNIFIED IDEOGRAPH-#3AC9, IRGKangXi=0483.101
3ACA CJK UNIFIED IDEOGRAPH-#3ACA, IRGKangXi=0483.180, Def{fluttering of the flag, the flags}
3ACB CJK UNIFIED IDEOGRAPH-#3ACB, IRGKangXi=0484.081, Def{(non-classical form of 旃) a silken banner hanging from a staff bent over at the top}
3ACC CJK UNIFIED IDEOGRAPH-#3ACC, IRGKangXi=0484.081
3ACD CJK UNIFIED IDEOGRAPH-#3ACD, IRGKangXi=0484.190, Def{the scallops along the lower edge of a flag; the hanging decorations (ribbons, etc.) of a flag used in ancient times, a kind of flag}
3ACE CJK UNIFIED IDEOGRAPH-#3ACE, IRGKangXi=0484.300, Def{(same as 徽 揮) mark; sign; indication; flags; pennants, streamers, etc., to prance, to brandish, to shake; to rattle, name of a flag}
3ACF CJK UNIFIED IDEOGRAPH-#3ACF, IRGKangXi=0485.010, Def{a kind of flag, flags; streamers}
3AD0 CJK UNIFIED IDEOGRAPH-#3AD0, IRGKangXi=0489.110, Def{distant and indistinguishable}
3AD1 CJK UNIFIED IDEOGRAPH-#3AD1, IRGKangXi=0490.040, Def{(same as 旨) (non-classical form 時) purpose; will; intention; objective, good; excellent, a season; an era; time}
3AD2 CJK UNIFIED IDEOGRAPH-#3AD2, IRGKangXi=0490.090, Def{(same as 廾) hands joined, to support; to aid; to help, to lean upon}
3AD3 CJK UNIFIED IDEOGRAPH-#3AD3, IRGKangXi=0490.160, Def{(ancient form of 氣) air; vapor, breath, spirit; character, bearing; manner}
3AD4 CJK UNIFIED IDEOGRAPH-#3AD4, IRGKangXi=0490.220, Def{(same as 艮) one of the Eight Diagrams for divination}
3AD5 CJK UNIFIED IDEOGRAPH-#3AD5, IRGKangXi=0490.221, Def{(same as 光) light; brightness, glossy; glory; glorious}
3AD6 CJK UNIFIED IDEOGRAPH-#3AD6, IRGKangXi=0490.221
3AD7 CJK UNIFIED IDEOGRAPH-#3AD7, IRGKangXi=0490.221, Def{(same as 厚) thick, deep friendship, to treat kindly; generous}
3AD8 CJK UNIFIED IDEOGRAPH-#3AD8, IRGKangXi=0490.380, Def{disappeared, not supported by, to avoid, unable to see}
3AD9 CJK UNIFIED IDEOGRAPH-#3AD9, IRGKangXi=0491.190, Def{the sun}
3ADA CJK UNIFIED IDEOGRAPH-#3ADA, IRGKangXi=0502.080, RSKangXi=72.4, Def{early morning, daylight}
3ADB CJK UNIFIED IDEOGRAPH-#3ADB, IRGKangXi=0492.140, Def{light; bright; brilliant, clear, intelligent; day dawn}
3ADC CJK UNIFIED IDEOGRAPH-#3ADC, IRGKangXi=0493.180, Def{light; bright; brilliant, clear, intelligent; day dawn}
3ADD CJK UNIFIED IDEOGRAPH-#3ADD, IRGKangXi=0493.210, Def{the sun}
3ADE CJK UNIFIED IDEOGRAPH-#3ADE, IRGKangXi=0493.240, Def{of look of the sun, the look of the weather}
3ADF CJK UNIFIED IDEOGRAPH-#3ADF, IRGKangXi=0494.020, Def{day after day; daily, time, moving of the sun}
3AE0 CJK UNIFIED IDEOGRAPH-#3AE0, IRGKangXi=0494.120, Def{a kind of water}
3AE1 CJK UNIFIED IDEOGRAPH-#3AE1, IRGKangXi=0494.130, Def{(same as 冬) winter, (in lunar calendar) the period from the 10th to the 12th moon}
3AE2 CJK UNIFIED IDEOGRAPH-#3AE2, IRGKangXi=0494.181
3AE3 CJK UNIFIED IDEOGRAPH-#3AE3, IRGKangXi=0494.181
3AE4 CJK UNIFIED IDEOGRAPH-#3AE4, IRGKangXi=0494.181, Def{(a variant of 昶) a long day, bright, extended, clear}
3AE5 CJK UNIFIED IDEOGRAPH-#3AE5, IRGKangXi=0495.060, Def{(non-classical form of 冥) dark; obscure; dim}
3AE6 CJK UNIFIED IDEOGRAPH-#3AE6, IRGKangXi=0495.100, Def{of look of the sun, the look of the weather}
3AE7 CJK UNIFIED IDEOGRAPH-#3AE7, IRGKangXi=0495.030, Def{(non-classical form of 暴) violent; fierce; atrocious; cruel, sudden}
3AE8 CJK UNIFIED IDEOGRAPH-#3AE8, IRGKangXi=0495.241
3AE9 CJK UNIFIED IDEOGRAPH-#3AE9, IRGKangXi=0495.241, Def{(same as 春) the first of the four season -- spring, lustful, alive, joyful, youth}
3AEA CJK UNIFIED IDEOGRAPH-#3AEA, IRGKangXi=0495.241, Def{(non-classical form of 舂) to pound (grain) in order to remove the husk}
3AEB CJK UNIFIED IDEOGRAPH-#3AEB, IRGKangXi=0495.241, Def{(non-classical form of 顯) clear; evident; eminent; prominent; renowned; famed; to expose; to make known}
3AEC CJK UNIFIED IDEOGRAPH-#3AEC, IRGKangXi=0495.241
3AED CJK UNIFIED IDEOGRAPH-#3AED, IRGKangXi=0495.241
3AEE CJK UNIFIED IDEOGRAPH-#3AEE, IRGKangXi=0494.181
3AEF CJK UNIFIED IDEOGRAPH-#3AEF, IRGKangXi=0495.300, Def{in contravention of; to offend (a superior, elder, etc.) (same as 冒) incautious; rash; imprudent}
3AF0 CJK UNIFIED IDEOGRAPH-#3AF0, IRGKangXi=0495.320, Def{light; bright; brilliant; clean; clever, to expose to sunlight, to dry something in an opening for ventilation area or a cool place}
3AF1 CJK UNIFIED IDEOGRAPH-#3AF1, IRGKangXi=0495.410, Def{mild, damp and moist, red color}
3AF2 CJK UNIFIED IDEOGRAPH-#3AF2, IRGKangXi=0496.090, Def{dark; dull}
3AF3 CJK UNIFIED IDEOGRAPH-#3AF3, IRGKangXi=0496.140, Def{(same as 晨) morning; daybreak}
3AF4 CJK UNIFIED IDEOGRAPH-#3AF4, IRGKangXi=0496.180, Def{(corrupted form of 哮) to roar; to pant; to howl}
3AF5 CJK UNIFIED IDEOGRAPH-#3AF5, IRGKangXi=0496.250, Def{white color, (corrupted form of 腓) the calf of the legs}
3AF6 CJK UNIFIED IDEOGRAPH-#3AF6, IRGKangXi=0496.270, Def{light; bright}
3AF7 CJK UNIFIED IDEOGRAPH-#3AF7, IRGKangXi=0496.350, Def{behind; the back of; afterwards; to come after (same as 期) period; time, to expect; to hope}
3AF8 CJK UNIFIED IDEOGRAPH-#3AF8, IRGKangXi=0496.380, Def{sunset}
3AF9 CJK UNIFIED IDEOGRAPH-#3AF9, IRGKangXi=0497.080, Def{light; bright, warm, genial}
3AFA CJK UNIFIED IDEOGRAPH-#3AFA, IRGKangXi=0497.130, Def{(same as 昔) bygone; of old; formerly; ancient, at night, the end}
3AFB CJK UNIFIED IDEOGRAPH-#3AFB, IRGKangXi=0497.170, Def{sunlight}
3AFC CJK UNIFIED IDEOGRAPH-#3AFC, IRGKangXi=0497.180, Def{(same as 晢) clear, bright; glorious; to be sharp and perspicacious; able to examine the tiniest things}
3AFD CJK UNIFIED IDEOGRAPH-#3AFD, IRGKangXi=0497.210, Def{darkness; the sun is dimmed}
3AFE CJK UNIFIED IDEOGRAPH-#3AFE, IRGKangXi=0497.311
3AFF CJK UNIFIED IDEOGRAPH-#3AFF, IRGKangXi=0497.311
3B00 CJK UNIFIED IDEOGRAPH-#3B00, IRGKangXi=0497.311
3B01 CJK UNIFIED IDEOGRAPH-#3B01, IRGKangXi=0497.320, Def{the sun is shining on the water}
3B02 CJK UNIFIED IDEOGRAPH-#3B02, IRGKangXi=0498.050, Def{used in person's name}
3B03 CJK UNIFIED IDEOGRAPH-#3B03, IRGKangXi=0498.080, Def{sunrise}
3B04 CJK UNIFIED IDEOGRAPH-#3B04, IRGKangXi=0498.150, Def{warm and damp (moist; humid), a little warm}
3B05 CJK UNIFIED IDEOGRAPH-#3B05, IRGKangXi=0498.180, Def{(non-classical form of 曼) long, extended, large}
3B06 CJK UNIFIED IDEOGRAPH-#3B06, IRGKangXi=0498.210, Def{(same as 湣) posthumous title conferred upon emperors and eminent officials}
3B07 CJK UNIFIED IDEOGRAPH-#3B07, IRGKangXi=0498.220, Def{name of a state in ancient China, sharp, bright-colored; brightness; bright; radiance; light}
3B08 CJK UNIFIED IDEOGRAPH-#3B08, IRGKangXi=0498.230, Def{the sun was shining and it's warm}
3B09 CJK UNIFIED IDEOGRAPH-#3B09, IRGKangXi=0498.240, Def{(a variant of 暖) warm, genial, of weather, soft and tender, pleasant and genial}
3B0A CJK UNIFIED IDEOGRAPH-#3B0A, IRGKangXi=0498.250, Def{light; bright, clear, intelligent; clever}
3B0B CJK UNIFIED IDEOGRAPH-#3B0B, IRGKangXi=0498.311, Def{name of a star}
3B0C CJK UNIFIED IDEOGRAPH-#3B0C, IRGKangXi=0498.311, Def{sunshine; sunlight}
3B0D CJK UNIFIED IDEOGRAPH-#3B0D, IRGKangXi=0498.390, Def{fierce; cruel, sudden, a scorching heat}
3B0E CJK UNIFIED IDEOGRAPH-#3B0E, IRGKangXi=0498.540, Def{(an ancient form of 顯) motes in a sunbeam, bright, fibrous, to manifest; to display, to be illustrious, evident, to seem; to appear, cocoons; chrysalis, will not have a pleasant conversation}
3B0F CJK UNIFIED IDEOGRAPH-#3B0F, IRGKangXi=0499.010, Def{light; bright, clear, intelligent; clever}
3B10 CJK UNIFIED IDEOGRAPH-#3B10, IRGKangXi=0499.060, Def{(same as 晉) a state during the period of Spring and Autumn, (same as 鄑) name of a place in today's Shandong Province}
3B11 CJK UNIFIED IDEOGRAPH-#3B11, IRGKangXi=0499.361
3B12 CJK UNIFIED IDEOGRAPH-#3B12, IRGKangXi=0499.100, Def{the sun are dimmed; darkness}
3B13 CJK UNIFIED IDEOGRAPH-#3B13, IRGKangXi=0499.120, Def{to expose to sunlight; to sun}
3B14 CJK UNIFIED IDEOGRAPH-#3B14, IRGKangXi=0499.130, Def{to defy; to disobey; to disregard, to contradict, perverse; recalcitrant; irregular; abnormal}
3B15 CJK UNIFIED IDEOGRAPH-#3B15, IRGKangXi=0499.180, Def{(corrupted form) light; bright, clear}
3B16 CJK UNIFIED IDEOGRAPH-#3B16, IRGKangXi=0499.361
3B17 CJK UNIFIED IDEOGRAPH-#3B17, IRGKangXi=0499.380, Def{(same as 晛) sunshine; light; bright, warm; genial (weather), very hot}
3B18 CJK UNIFIED IDEOGRAPH-#3B18, IRGKangXi=0499.410, Def{arid; dry; parched, impatient; restless, fierce; cruel, sudden, a scorching heat}
3B19 CJK UNIFIED IDEOGRAPH-#3B19, IRGKangXi=0499.430, Def{sunshine; light of the sun}
3B1A CJK UNIFIED IDEOGRAPH-#3B1A, IRGKangXi=0499.460, Def{light; bright, clear}
3B1B CJK UNIFIED IDEOGRAPH-#3B1B, IRGKangXi=0499.470, Def{to dry in the sun}
3B1C CJK UNIFIED IDEOGRAPH-#3B1C, IRGKangXi=0500.010, Def{(same as 晉) to advance; to increase; to flourish}
3B1D CJK UNIFIED IDEOGRAPH-#3B1D, IRGKangXi=0500.050, Def{obscure; not bright; poor visibility, white; clear; bright; pure; plain}
3B1E CJK UNIFIED IDEOGRAPH-#3B1E, IRGKangXi=0500.251, Def{(non-classical form of 暍) sunstroke, hot; feverish}
3B1F CJK UNIFIED IDEOGRAPH-#3B1F, IRGKangXi=0504.091, Def{(a dialect), an adverb, negative expression; never before; to have not happened before}
3B20 CJK UNIFIED IDEOGRAPH-#3B20, IRGKangXi=0500.310, Def{(non-classical form of 曬) to expose to sunlight; to dry in the sun (interchangeable 煞) very; much; extremely; a fierce god; a malignant deity, to conclude}
3B21 CJK UNIFIED IDEOGRAPH-#3B21, IRGKangXi=0500.370, Def{(same as 昤) sunshine; light of the sun}
3B22 CJK UNIFIED IDEOGRAPH-#3B22, IRGKangXi=0500.420, Def{sunshine; sunlight; the light of day}
3B23 CJK UNIFIED IDEOGRAPH-#3B23, IRGKangXi=0501.010, Def{luxuriant; exuberant; lush; flourishing}
3B24 CJK UNIFIED IDEOGRAPH-#3B24, IRGKangXi=0501.030, Def{wet clothes not completely dry; to dry in the sun or use sand to dry up on the ground}
3B25 CJK UNIFIED IDEOGRAPH-#3B25, IRGKangXi=0501.090, Def{(an ancient form of 曝) to sun to air; to expose to sunlight, to expose or be exposed; exposure, a tree with scatter or dispersed leaves}
3B26 CJK UNIFIED IDEOGRAPH-#3B26, IRGKangXi=0501.100, Def{brave; courageous; bold; valiant; intrepid; fearless; heroic}
3B27 CJK UNIFIED IDEOGRAPH-#3B27, IRGKangXi=0501.131, Def{(standard form) a collar, the neck, to connect collars to the clothes}
3B28 CJK UNIFIED IDEOGRAPH-#3B28, IRGKangXi=0501.131
3B29 CJK UNIFIED IDEOGRAPH-#3B29, IRGKangXi=0501.170, Def{a tiny star, numerous stars}
3B2A CJK UNIFIED IDEOGRAPH-#3B2A, IRGKangXi=0501.210, Def{(a variant of 疊) to fold up, to repeat, to duplicate}
3B2B CJK UNIFIED IDEOGRAPH-#3B2B, IRGKangXi=0501.290, Def{clear sky, a fine day}
3B2C CJK UNIFIED IDEOGRAPH-#3B2C, IRGKangXi=0501.390, Def{a family name}
3B2D CJK UNIFIED IDEOGRAPH-#3B2D, IRGKangXi=0501.410, Def{(same as, non-classical form of 皭) pure white; clean; bright, usually used for person's name in ancient times}
3B2E CJK UNIFIED IDEOGRAPH-#3B2E, IRGKangXi=0501.450, Def{gentle; mild; temperate, tender; loving; caressing}
3B2F CJK UNIFIED IDEOGRAPH-#3B2F, IRGKangXi=0501.520, Def{sunset}
3B30 CJK UNIFIED IDEOGRAPH-#3B30, IRGKangXi=0502.030, Def{(non-classical form of 臾) a moment; an instant; a little while; a short time}
3B31 CJK UNIFIED IDEOGRAPH-#3B31, IRGKangXi=0504.090, Def{(same as 替) to replace; to substitute; to decay; to decline, (a variant 朁) if, supposing, nevertheless}
3B32 CJK UNIFIED IDEOGRAPH-#3B32, IRGKangXi=0504.110, Def{to increase; to add to; to augment, to a greater extent; more, benefit; profit; advantage}
3B33 CJK UNIFIED IDEOGRAPH-#3B33, IRGKangXi=0504.300, Def{light; bright, clear}
3B34 CJK UNIFIED IDEOGRAPH-#3B34, IRGKangXi=0505.120, Def{the moon is dimmed; darkness; the word used especially by Triad Society (三合會) a secret society during the Ching Dynasty dedicated to the overthrow of the Manchus and the restoration the Ming Dynasty}
3B35 CJK UNIFIED IDEOGRAPH-#3B35, IRGKangXi=0505.210, Def{the shinbone, or tibia, (same as U+4EA4 交) to intersect, the sun and the moon in the sky, (simplified form U+81A0 膠) glue; gum, resin; sap, anything sticky}
3B36 CJK UNIFIED IDEOGRAPH-#3B36, IRGKangXi=0506.010, Def{flesh; meat of animals, (same as 浩) great; vast, many; much}
3B37 CJK UNIFIED IDEOGRAPH-#3B37, IRGKangXi=0506.011
3B38 CJK UNIFIED IDEOGRAPH-#3B38, IRGKangXi=0506.150, Def{a kind of services or ceremonies of offering sacrifices}
3B39 CJK UNIFIED IDEOGRAPH-#3B39, IRGKangXi=0986.140, RSKangXi=130.8, Def{tendons of the heel of the foot; (Cant.) elbow, heel}
3B3A CJK UNIFIED IDEOGRAPH-#3B3A, IRGKangXi=0506.211
3B3B CJK UNIFIED IDEOGRAPH-#3B3B, IRGKangXi=0506.230, Def{in between of the flesh, a state of a minority ethnic group in southern China, the moon is dimmed}
3B3C CJK UNIFIED IDEOGRAPH-#3B3C, IRGKangXi=0506.281, Def{to have; to be present; to exist; there is}
3B3D CJK UNIFIED IDEOGRAPH-#3B3D, IRGKangXi=0506.281
3B3E CJK UNIFIED IDEOGRAPH-#3B3E, IRGKangXi=0506.281
3B3F CJK UNIFIED IDEOGRAPH-#3B3F, IRGKangXi=0506.290, Def{moonlight}
3B40 CJK UNIFIED IDEOGRAPH-#3B40, IRGKangXi=0507.011
3B41 CJK UNIFIED IDEOGRAPH-#3B41, IRGKangXi=0510.080, Def{a kind of tree, a kind of lumber used to stop a carriage}
3B42 CJK UNIFIED IDEOGRAPH-#3B42, IRGKangXi=0511.050, Def{height of the end-point; the tip of a small branch; (Cant.) slender, thin}
3B43 CJK UNIFIED IDEOGRAPH-#3B43, IRGKangXi=0511.060, Def{(ancient form of 綱) the large rope of a net, main points; outline; principle}
3B44 CJK UNIFIED IDEOGRAPH-#3B44, IRGKangXi=0511.130, Def{a king of tree}
3B45 CJK UNIFIED IDEOGRAPH-#3B45, IRGKangXi=0513.011
3B46 CJK UNIFIED IDEOGRAPH-#3B46, IRGKangXi=0513.011
3B47 CJK UNIFIED IDEOGRAPH-#3B47, IRGKangXi=0514.010, Def{a kind of tree, (corrupted form of 杬) a kind of plant; (non-classical form 欖) the olive tree}
3B48 CJK UNIFIED IDEOGRAPH-#3B48, IRGKangXi=0514.030, Def{a basin; a bowl}
3B49 CJK UNIFIED IDEOGRAPH-#3B49, IRGKangXi=0514.180, Def{farm tools; a spade or shovel}
3B4A CJK UNIFIED IDEOGRAPH-#3B4A, IRGKangXi=0516.100, Def{(non-classical of 析) to split, to rip or break apart, to divide, to separate; to interpret, to explain, to analyze}
3B4B CJK UNIFIED IDEOGRAPH-#3B4B, IRGKangXi=0516.140, Def{(same as 棒) a club; a stick, to hit with a club, good; strong; wonderful}
3B4C CJK UNIFIED IDEOGRAPH-#3B4C, IRGKangXi=0516.150, Def{(same as 桙) a basin; a bathtub, a kind of tool or utensil, (interchangeable 模) form or shape of a thing; style; pattern; appearance; look, a sample}
3B4D CJK UNIFIED IDEOGRAPH-#3B4D, IRGKangXi=0517.011, Def{a variant of 七 (U+4E03)}
3B4E CJK UNIFIED IDEOGRAPH-#3B4E, IRGKangXi=0517.011, Def{(simplified form of 棡) a tall tree; a large tree, (interchangeable 扛) to carry on the shoulders of two of more men}
3B4F CJK UNIFIED IDEOGRAPH-#3B4F, IRGKangXi=0517.011, Def{(simplified form of 椲) a kind of wood (used as a kind of material to make basin and bowl, etc.); (same as 楎) a peg for hanging things on, a clothes-horse}
3B50 CJK UNIFIED IDEOGRAPH-#3B50, IRGKangXi=0517.011
3B51 CJK UNIFIED IDEOGRAPH-#3B51, IRGKangXi=0517.150, Def{a kind of tree}
3B52 CJK UNIFIED IDEOGRAPH-#3B52, IRGKangXi=0519.150, Def{variant of U+801C 耜: a farming instrument; a spade; a shovel}
3B53 CJK UNIFIED IDEOGRAPH-#3B53, IRGKangXi=0519.170, Def{a pillar arch}
3B54 CJK UNIFIED IDEOGRAPH-#3B54, IRGKangXi=0519.180, Def{a kind of tree}
3B55 CJK UNIFIED IDEOGRAPH-#3B55, IRGKangXi=0520.160, Def{a board (on the back of a donkey) for carrying things; saddle}
3B56 CJK UNIFIED IDEOGRAPH-#3B56, IRGKangXi=0521.121
3B57 CJK UNIFIED IDEOGRAPH-#3B57, IRGKangXi=0521.121
3B58 CJK UNIFIED IDEOGRAPH-#3B58, IRGKangXi=0521.150, Def{a scabbard; a sheath; a case for sword, a tree like the acacia}
3B59 CJK UNIFIED IDEOGRAPH-#3B59, IRGKangXi=0522.140, Def{a hammer}
3B5A CJK UNIFIED IDEOGRAPH-#3B5A, IRGKangXi=0522.160, Def{a kind of tree, a tray for carrying sacrificial meats and wine}
3B5B CJK UNIFIED IDEOGRAPH-#3B5B, IRGKangXi=0522.170, Def{a kind of climbing plants; rattan; bark can be used to weave cloth, a component parts of a loom}
3B5C CJK UNIFIED IDEOGRAPH-#3B5C, IRGKangXi=0523.070, Def{a kind of tree (like locust tree; similar to the ash; acacia)}
3B5D CJK UNIFIED IDEOGRAPH-#3B5D, IRGKangXi=0523.130, Def{name of a pavilion, in Xinshi, name of a place in Jingling}
3B5E CJK UNIFIED IDEOGRAPH-#3B5E, IRGKangXi=0523.180, Def{a kind inferior wood}
3B5F CJK UNIFIED IDEOGRAPH-#3B5F, IRGKangXi=0523.210, Def{torture used in ancient times}
3B60 CJK UNIFIED IDEOGRAPH-#3B60, IRGKangXi=0523.250
3B61 CJK UNIFIED IDEOGRAPH-#3B61, IRGKangXi=0524.010, Def{name of a mountain, a component parts of a loom}
3B62 CJK UNIFIED IDEOGRAPH-#3B62, IRGKangXi=0525.030, Def{mechanics, opportunity, urgent}
3B63 CJK UNIFIED IDEOGRAPH-#3B63, IRGKangXi=0526.211
3B64 CJK UNIFIED IDEOGRAPH-#3B64, IRGKangXi=0521.121, Def{(simplified form) (same as 蔦) the convolvulus; a kind of creeping plant, partridge; francolin, in Japan, the pole out side of a building used to post the public notice}
3B65 CJK UNIFIED IDEOGRAPH-#3B65, IRGKangXi=0526.211
3B66 CJK UNIFIED IDEOGRAPH-#3B66, IRGKangXi=0526.211
3B67 CJK UNIFIED IDEOGRAPH-#3B67, IRGKangXi=0526.211
3B68 CJK UNIFIED IDEOGRAPH-#3B68, IRGKangXi=0527.010, Def{(same as U+6930 椰) the cocoa-nut palm}
3B69 CJK UNIFIED IDEOGRAPH-#3B69, IRGKangXi=0527.070, Def{a kind of tree can be used as dye-stuff, the mast of a boat}
3B6A CJK UNIFIED IDEOGRAPH-#3B6A, IRGKangXi=0527.110, Def{a shrub (plant); dense growth of trees}
3B6B CJK UNIFIED IDEOGRAPH-#3B6B, IRGKangXi=0527.190, Def{a fruit tree (plum), (same as 菆) hemp stalks, a mat}
3B6C CJK UNIFIED IDEOGRAPH-#3B6C, IRGKangXi=0527.220, Def{(corrupted form of 椓) to beat; to tap; to thrash, castration as a punishment in ancient times, a castrate man, to accuse or charge}
3B6D CJK UNIFIED IDEOGRAPH-#3B6D, IRGKangXi=0529.050, Def{a kind of tree, a harrow; a drag, a written contract or agreement; a bond; a deed, handles of spears or lances used in ancient times}
3B6E CJK UNIFIED IDEOGRAPH-#3B6E, IRGKangXi=0529.060, Def{(corrupted form of 桚) a kind of liquor container, to squeeze; to press, an instrument of torture for squeezing the fingers of prisoners or witnesses in order to extort evidence or confession}
3B6F CJK UNIFIED IDEOGRAPH-#3B6F, IRGKangXi=0529.070, Def{tree with small leaves; mushroom}
3B70 CJK UNIFIED IDEOGRAPH-#3B70, IRGKangXi=0529.130, Def{to know; to recognize, to hide; to conceal, (same as 嘴) beak (of a bird), stone probe}
3B71 CJK UNIFIED IDEOGRAPH-#3B71, IRGKangXi=0531.131, Def{(same as 匣) a case; a small box}
3B72 CJK UNIFIED IDEOGRAPH-#3B72, IRGKangXi=0531.131, Def{(same as 极) a wooden frame (on the back of a donkey) for carrying things, (simplified form of 極) to exhaust, extreme; highest; topmost}
3B73 CJK UNIFIED IDEOGRAPH-#3B73, IRGKangXi=0531.131
3B74 CJK UNIFIED IDEOGRAPH-#3B74, IRGKangXi=0531.131, Def{(simplified form) the oscines}
3B75 CJK UNIFIED IDEOGRAPH-#3B75, IRGKangXi=0531.131
3B76 CJK UNIFIED IDEOGRAPH-#3B76, IRGKangXi=0537.131
3B77 CJK UNIFIED IDEOGRAPH-#3B77, IRGKangXi=0531.131
3B78 CJK UNIFIED IDEOGRAPH-#3B78, IRGKangXi=0531.190, Def{(same as 梌) the branches to spread out in all directions, the catalpa; a kind of hard wood used for making chessboard}
3B79 CJK UNIFIED IDEOGRAPH-#3B79, IRGKangXi=0533.120, Def{name of a county in today's Shandong Province, a kind of fabric produced in Donglai}
3B7A CJK UNIFIED IDEOGRAPH-#3B7A, IRGKangXi=0533.130, Def{the cherry-apple, a kind of tree}
3B7B CJK UNIFIED IDEOGRAPH-#3B7B, IRGKangXi=0533.150, Def{(same as 棠) the crab-apple; the wild plum}
3B7C CJK UNIFIED IDEOGRAPH-#3B7C, IRGKangXi=0534.080, Def{a rectangle shaped peg or short post between the beans and the roof timbers; the square peck-shaped box half-way up a Chinese flagstaff}
3B7D CJK UNIFIED IDEOGRAPH-#3B7D, IRGKangXi=0534.250, Def{(non-classical form of 柢) root; foundation; base, eaves of a house; brim}
3B7E CJK UNIFIED IDEOGRAPH-#3B7E, IRGKangXi=0535.140, Def{to cut the timber apart; to cut a tree; (Cant.) to blunt}
3B7F CJK UNIFIED IDEOGRAPH-#3B7F, IRGKangXi=0535.190, Def{oblique angle, a post to tie cattle and horses to}
3B80 CJK UNIFIED IDEOGRAPH-#3B80, IRGKangXi=0535.240, Def{tool to clear out (as a drain) the water, (same as 函) a case; a small box}
3B81 CJK UNIFIED IDEOGRAPH-#3B81, IRGKangXi=0536.020, Def{gardenia; a plant of which the nuts produces a yellow dye}
3B82 CJK UNIFIED IDEOGRAPH-#3B82, IRGKangXi=0536.050, Def{an appliance used to lift food; something like a narrow, long table, to carry; to transport}
3B83 CJK UNIFIED IDEOGRAPH-#3B83, IRGKangXi=0536.090, Def{farm tool, (same as 桵) a kind of tree}
3B84 CJK UNIFIED IDEOGRAPH-#3B84, IRGKangXi=0536.150, Def{(non-classical form of 榜) publicly posted roll of successful examinees}
3B85 CJK UNIFIED IDEOGRAPH-#3B85, IRGKangXi=0537.060, Def{a kind of tree}
3B86 CJK UNIFIED IDEOGRAPH-#3B86, IRGKangXi=0537.131, Def{(an ancient form) (same as 𣡌) leaves sprouting from the stump of a tree; shoots from an old stump, a large (a species of oak) from the bark of which a yellow dye is produced}
3B87 CJK UNIFIED IDEOGRAPH-#3B87, IRGKangXi=0537.131, Def{(standard form of 掭) (same as 栝) a builder's frame for measuring, juniper, a poker (for stirring fire, a cylinder part on the old style of wooden doors}
3B88 CJK UNIFIED IDEOGRAPH-#3B88, IRGKangXi=0537.131, Def{(non-classical form of 奈) but; how; what, a remedy; a resource, to bear, to endure}
3B89 CJK UNIFIED IDEOGRAPH-#3B89, IRGKangXi=0537.131
3B8A CJK UNIFIED IDEOGRAPH-#3B8A, IRGKangXi=0537.131
3B8B CJK UNIFIED IDEOGRAPH-#3B8B, IRGKangXi=0537.280, Def{(non-classical form of 栯) a kind of tree, a fruit tree}
3B8C CJK UNIFIED IDEOGRAPH-#3B8C, IRGKangXi=0538.030, Def{(same as U+68C9 棉) cotton; (Cant.) a kind of fruit}
3B8D CJK UNIFIED IDEOGRAPH-#3B8D, IRGKangXi=0538.070, Def{(ancient form of U+724B 牋 U+7B8B 箋) a memorandum tablet, slip of paper, fancy note-paper, a note, a document, comments}
3B8E CJK UNIFIED IDEOGRAPH-#3B8E, IRGKangXi=0538.110, Def{(non-classical form of U+76C3 盃 U+676F 杯) cup; a tumbler; a glass}
3B8F CJK UNIFIED IDEOGRAPH-#3B8F, IRGKangXi=0538.160, Def{(same as 柰) a fruit tree; a crab-apple, for which the second from is strictly used, leaves sprouting from the stump of a tree; shoots from an old stump}
3B90 CJK UNIFIED IDEOGRAPH-#3B90, IRGKangXi=0538.180, Def{a rack or a stand with three sticks to cross each other, a chopping board}
3B91 CJK UNIFIED IDEOGRAPH-#3B91, IRGKangXi=0538.210, Def{sound of broken sticks; crack of a wooden stick, a forest; wooded land}
3B92 CJK UNIFIED IDEOGRAPH-#3B92, IRGKangXi=0538.280, Def{wood, (same as 禋) to worship with sincerity and reverence, to offer sacrifices to the Heaven}
3B93 CJK UNIFIED IDEOGRAPH-#3B93, IRGKangXi=0538.320, Def{(same as 亙 堥) an extreme limit, to fill, universal, connected, revolve}
3B94 CJK UNIFIED IDEOGRAPH-#3B94, IRGKangXi=0539.050, Def{vegetable, (non-classical, abbreviated from, same as 橦) a tree which grows in Yunnan, from the flowers of which a cloth is made}
3B95 CJK UNIFIED IDEOGRAPH-#3B95, IRGKangXi=0539.090, Def{fruit, plant, (same as 檽) a fruit tree; black date}
3B96 CJK UNIFIED IDEOGRAPH-#3B96, IRGKangXi=0539.110, Def{to lay; to put, drums, to rap, to quiver, to stir up}
3B97 CJK UNIFIED IDEOGRAPH-#3B97, IRGKangXi=0539.120, Def{(same as 矜) the handle of a spear}
3B98 CJK UNIFIED IDEOGRAPH-#3B98, IRGKangXi=0539.210, Def{a tree; winter peach}
3B99 CJK UNIFIED IDEOGRAPH-#3B99, IRGKangXi=0540.150, Def{a pitfall; a hole}
3B9A CJK UNIFIED IDEOGRAPH-#3B9A, IRGKangXi=0541.010, Def{(an ancient form of 栗) the chestnut tree, a kind of metalwork in ancient times}
3B9B CJK UNIFIED IDEOGRAPH-#3B9B, IRGKangXi=0541.080, Def{(same as 匙) a spoon, (same as 椸) (a dialect) a small table in front of the bed, a rack for clothes; a clothes-horse}
3B9C CJK UNIFIED IDEOGRAPH-#3B9C, IRGKangXi=0542.010, Def{a blocks engraved for holding things}
3B9D CJK UNIFIED IDEOGRAPH-#3B9D, IRGKangXi=0542.080, Def{angle steel, legs of a table}
3B9E CJK UNIFIED IDEOGRAPH-#3B9E, IRGKangXi=0542.191, Def{(same as U+6976 楶) a wooden block or bracket on top of a column}
3B9F CJK UNIFIED IDEOGRAPH-#3B9F, IRGKangXi=0542.191, Def{a whetstone, (a dialect) a corner; a nook; a crack; an opening; a cleft}
3BA0 CJK UNIFIED IDEOGRAPH-#3BA0, IRGKangXi=0542.191
3BA1 CJK UNIFIED IDEOGRAPH-#3BA1, IRGKangXi=0677.030, Def{(same as 爟) to light a fire; to set fire to}
3BA2 CJK UNIFIED IDEOGRAPH-#3BA2, IRGKangXi=0542.191, Def{fruit}
3BA3 CJK UNIFIED IDEOGRAPH-#3BA3, IRGKangXi=0542.191, Def{(a variant of 概) a piece of wood used to strike off grain in a measure, thus:--all, to level, to adjust, generally, for the most part, (interchangeable 慨) emotional excitement, (interchangeable 溉) to wash; to cleanse; to rinse}
3BA4 CJK UNIFIED IDEOGRAPH-#3BA4, IRGKangXi=0543.050, Def{pine; fir, (same as 松 鬆) loose; lax; slack}
3BA5 CJK UNIFIED IDEOGRAPH-#3BA5, IRGKangXi=0543.120, Def{sails, the mat-covering of small boats, (interchangeable 軬) the covering of a carriage}
3BA6 CJK UNIFIED IDEOGRAPH-#3BA6, IRGKangXi=0543.250, Def{branches growing upwards, the tip of a branch or things of similar shape, (interchangeable 索) to importune}
3BA7 CJK UNIFIED IDEOGRAPH-#3BA7, IRGKangXi=0543.260, Def{a tree, a fruit}
3BA8 CJK UNIFIED IDEOGRAPH-#3BA8, IRGKangXi=0543.320, Def{China cypress}
3BA9 CJK UNIFIED IDEOGRAPH-#3BA9, IRGKangXi=0543.340, Def{a tree}
3BAA CJK UNIFIED IDEOGRAPH-#3BAA, IRGKangXi=0544.060, Def{a small boat}
3BAB CJK UNIFIED IDEOGRAPH-#3BAB, IRGKangXi=0544.070, Def{a wooden article to be used to adjust a bow, a tiger shaped wooden article used to stop the music in ancient times}
3BAC CJK UNIFIED IDEOGRAPH-#3BAC, IRGKangXi=0544.100, Def{a fruit tree}
3BAD CJK UNIFIED IDEOGRAPH-#3BAD, IRGKangXi=0544.220, Def{(same as 椷) a casket; a box; a bowl; a cup, to allow, an envelope}
3BAE CJK UNIFIED IDEOGRAPH-#3BAE, IRGKangXi=0545.010, Def{a well sweep}
3BAF CJK UNIFIED IDEOGRAPH-#3BAF, IRGKangXi=0545.030, Def{whole (uncut, unsplit) firewood; complete, whole, undamaged, general; (same as 梡) a kind of tree}
3BB0 CJK UNIFIED IDEOGRAPH-#3BB0, IRGKangXi=0545.060, Def{the small beam supporting the rafters at the eaves}
3BB1 CJK UNIFIED IDEOGRAPH-#3BB1, IRGKangXi=0545.080, Def{(said of grass, trees, vegetation and flora), (non-classical form of 森) luxuriant; exuberant; lush; flourishing}
3BB2 CJK UNIFIED IDEOGRAPH-#3BB2, IRGKangXi=0545.100, Def{a small ring or a small piece of board attached to the muzzle of an ox; to thread through a halter}
3BB3 CJK UNIFIED IDEOGRAPH-#3BB3, IRGKangXi=0545.180, Def{a tree, (same as 栚) a piece of cross-wise board used for frame on which silkworms spin}
3BB4 CJK UNIFIED IDEOGRAPH-#3BB4, IRGKangXi=0546.050, Def{(same as 艘) a numerary adjunct for ships}
3BB5 CJK UNIFIED IDEOGRAPH-#3BB5, IRGKangXi=0546.120, Def{common sandalwood}
3BB6 CJK UNIFIED IDEOGRAPH-#3BB6, IRGKangXi=0546.150, Def{(same as 槊) a tree, a fence; a palisade; a railing}
3BB7 CJK UNIFIED IDEOGRAPH-#3BB7, IRGKangXi=0546.160, Def{a tree (as maple tree)}
3BB8 CJK UNIFIED IDEOGRAPH-#3BB8, IRGKangXi=0546.170, Def{(corrupted form of 送) to send; to deliver; to present; to give, to send off}
3BB9 CJK UNIFIED IDEOGRAPH-#3BB9, IRGKangXi=0546.190, Def{a tree}
3BBA CJK UNIFIED IDEOGRAPH-#3BBA, IRGKangXi=0546.240, Def{(ancient form of 本) stem; source; foundation; origin; basis, a book}
3BBB CJK UNIFIED IDEOGRAPH-#3BBB, IRGKangXi=0542.191, Def{(same as 搔) to scratch lightly, to irritate; to annoy}
3BBC CJK UNIFIED IDEOGRAPH-#3BBC, IRGKangXi=0546.251
3BBD CJK UNIFIED IDEOGRAPH-#3BBD, IRGKangXi=0546.251
3BBE CJK UNIFIED IDEOGRAPH-#3BBE, IRGKangXi=0551.181, Def{a place in today's Hunan Province}
3BBF CJK UNIFIED IDEOGRAPH-#3BBF, IRGKangXi=0547.040, Def{a tree}
3BC0 CJK UNIFIED IDEOGRAPH-#3BC0, IRGKangXi=0547.190, Def{bonds and fetters for falcons and dogs used in hunting}
3BC1 CJK UNIFIED IDEOGRAPH-#3BC1, IRGKangXi=0548.070, Def{household registers, printing plate, supporting boards used in building walls, (non-classical form 棓) to strike, a flail, a club, planks}
3BC2 CJK UNIFIED IDEOGRAPH-#3BC2, IRGKangXi=0548.130, Def{pillar or post of a frame on which silkworms spin}
3BC3 CJK UNIFIED IDEOGRAPH-#3BC3, IRGKangXi=0548.180, Def{the varnish tree, paint, lacquer, varnish, to paint or varnish, black, sticky}
3BC4 CJK UNIFIED IDEOGRAPH-#3BC4, IRGKangXi=0549.120, Def{(of trees) luxuriant; exuberant; lush}
3BC5 CJK UNIFIED IDEOGRAPH-#3BC5, IRGKangXi=0549.190, Def{the branches growing downwards; small trees; saplings; cuttings of trees for planting}
3BC6 CJK UNIFIED IDEOGRAPH-#3BC6, IRGKangXi=0549.230, Def{a tree, a mattress}
3BC7 CJK UNIFIED IDEOGRAPH-#3BC7, IRGKangXi=0550.040, Def{(same as 梐) a stockade; a cheval de frise, a pen; a cage}
3BC8 CJK UNIFIED IDEOGRAPH-#3BC8, IRGKangXi=0550.060, Def{(non-classical form) small trees; saplings}
3BC9 CJK UNIFIED IDEOGRAPH-#3BC9, IRGKangXi=0550.090, Def{(same as 檴) a tree; material for cups, a kind of birch found in Manchuria}
3BCA CJK UNIFIED IDEOGRAPH-#3BCA, IRGKangXi=0551.030, Def{even measuring board for pecks and bushels}
3BCB CJK UNIFIED IDEOGRAPH-#3BCB, IRGKangXi=0551.070, Def{a chest; a box, footstool; footrest, a tree, handle of an awl, ring of a knife or a sword, a round pillow that roused the sleeper when he moved}
3BCC CJK UNIFIED IDEOGRAPH-#3BCC, IRGKangXi=0551.170, Def{to operate or manage}
3BCD CJK UNIFIED IDEOGRAPH-#3BCD, IRGKangXi=0551.181, Def{(same as 槳) an oar}
3BCE CJK UNIFIED IDEOGRAPH-#3BCE, IRGKangXi=0551.181, Def{stupid; dull, foolish; idiotic}
3BCF CJK UNIFIED IDEOGRAPH-#3BCF, IRGKangXi=0551.181, Def{(same as 穀) grains; corns; cereals}
3BD0 CJK UNIFIED IDEOGRAPH-#3BD0, IRGKangXi=0555.311, Def{tubular, oval, elliptical}
3BD1 CJK UNIFIED IDEOGRAPH-#3BD1, IRGKangXi=0551.181
3BD2 CJK UNIFIED IDEOGRAPH-#3BD2, IRGKangXi=0551.181
3BD3 CJK UNIFIED IDEOGRAPH-#3BD3, IRGKangXi=0552.050, Def{(same as 榻) a couch; a bed, window; shutters}
3BD4 CJK UNIFIED IDEOGRAPH-#3BD4, IRGKangXi=0552.200, Def{to hull heavily, to pound heavily}
3BD5 CJK UNIFIED IDEOGRAPH-#3BD5, IRGKangXi=0552.230, Def{sticks or frames; a wooden device used to torture the fingers by squeezing, the branches growing downwards; small trees; saplings; cuttings of trees for planting}
3BD6 CJK UNIFIED IDEOGRAPH-#3BD6, IRGKangXi=0552.260, Def{a tree, a hammer, to beat with a hammer}
3BD7 CJK UNIFIED IDEOGRAPH-#3BD7, IRGKangXi=0553.030, Def{a kind of tree}
3BD8 CJK UNIFIED IDEOGRAPH-#3BD8, IRGKangXi=0553.120, Def{(same as 梡) a small wooden stand having four legs; it was used in sacrifice, faggots}
3BD9 CJK UNIFIED IDEOGRAPH-#3BD9, IRGKangXi=0553.140, Def{the buckthorn or jujube (tree); dates}
3BDA CJK UNIFIED IDEOGRAPH-#3BDA, IRGKangXi=0553.180, Def{a kind of tree (as plum)}
3BDB CJK UNIFIED IDEOGRAPH-#3BDB, IRGKangXi=0553.220, Def{gourd shaped dates}
3BDC CJK UNIFIED IDEOGRAPH-#3BDC, IRGKangXi=0553.230, Def{(same as 椊) decayed wood}
3BDD CJK UNIFIED IDEOGRAPH-#3BDD, IRGKangXi=0554.170, Def{paulownia}
3BDE CJK UNIFIED IDEOGRAPH-#3BDE, IRGKangXi=0558.091, Def{a kind of tree; material for making clothes}
3BDF CJK UNIFIED IDEOGRAPH-#3BDF, IRGKangXi=0554.220, Def{(ancient form of 麓) foot of a hill or mountain}
3BE0 CJK UNIFIED IDEOGRAPH-#3BE0, IRGKangXi=0554.280, Def{a cross-beam; an axle, etc.}
3BE1 CJK UNIFIED IDEOGRAPH-#3BE1, IRGKangXi=0555.150, Def{to lacquer some kind of ashes on wooden articles or furniture, after drying out, smooth them and then paint, black lacquer}
3BE2 CJK UNIFIED IDEOGRAPH-#3BE2, IRGKangXi=0555.170, Def{(same as 栚) (same as 㮳) a piece of cross-wise board used for frame on which silkworms spin}
3BE3 CJK UNIFIED IDEOGRAPH-#3BE3, IRGKangXi=0555.260, Def{(same as 棻) a kind of fragrant tree used to produce perfume by burning it}
3BE4 CJK UNIFIED IDEOGRAPH-#3BE4, IRGKangXi=0555.280, Def{(same as 棘) the jujube tree, thorny brambles, a kind of date (with a sour taste)}
3BE5 CJK UNIFIED IDEOGRAPH-#3BE5, IRGKangXi=0555.290, Def{(same as 曹) a Chinese family name, surrounded, a whole day; a revolution of the sun}
3BE6 CJK UNIFIED IDEOGRAPH-#3BE6, IRGKangXi=0555.311, Def{(same as 棋) the game of chess, draughts and other similar games}
3BE7 CJK UNIFIED IDEOGRAPH-#3BE7, IRGKangXi=0555.311
3BE8 CJK UNIFIED IDEOGRAPH-#3BE8, IRGKangXi=0555.311
3BE9 CJK UNIFIED IDEOGRAPH-#3BE9, IRGKangXi=0167.081, Def{to put the growing up silkworms in different frames according to their sizes}
3BEA CJK UNIFIED IDEOGRAPH-#3BEA, IRGKangXi=0555.330, Def{(same as 櫺) carved or patterned window-railings; sills, the wooden planks which join eaves with a house}
3BEB CJK UNIFIED IDEOGRAPH-#3BEB, IRGKangXi=0555.340, Def{a wattle or bamboo fence}
3BEC CJK UNIFIED IDEOGRAPH-#3BEC, IRGKangXi=0556.080, Def{achievements; merit}
3BED CJK UNIFIED IDEOGRAPH-#3BED, IRGKangXi=0556.130, Def{(a variant of 櫓) a lookout turret on a city wall, moveable wooden tower for archers, a scull a sweep, an oar, (in ancient warfare) a big shield; a long spear}
3BEE CJK UNIFIED IDEOGRAPH-#3BEE, IRGKangXi=0556.140, Def{a kind of tree (as willow) with big leaves and in red color}
3BEF CJK UNIFIED IDEOGRAPH-#3BEF, IRGKangXi=0556.160, Def{small cup, chest; box, a kind of container usually put on one's head to carry things}
3BF0 CJK UNIFIED IDEOGRAPH-#3BF0, IRGKangXi=0556.220, Def{(same as 植) to plant; to set up, to erect, plants}
3BF1 CJK UNIFIED IDEOGRAPH-#3BF1, IRGKangXi=0557.030, Def{an enlarged bag, to wrap up, a parcel; a bundle; a package}
3BF2 CJK UNIFIED IDEOGRAPH-#3BF2, IRGKangXi=0557.080, Def{a wooden (bamboo) framework; a wooden pole; used as an obstacle to impede the door, a frame}
3BF3 CJK UNIFIED IDEOGRAPH-#3BF3, IRGKangXi=0557.170, Def{(same as 檠) a kind of tool to adjust bow; lamp holder, a tray with base}
3BF4 CJK UNIFIED IDEOGRAPH-#3BF4, IRGKangXi=0559.261
3BF5 CJK UNIFIED IDEOGRAPH-#3BF5, IRGKangXi=0558.091
3BF6 CJK UNIFIED IDEOGRAPH-#3BF6, IRGKangXi=0558.091, Def{the coir palm tree}
3BF7 CJK UNIFIED IDEOGRAPH-#3BF7, IRGKangXi=0558.270, Def{dates, a kind of oak; Quercus dentata, (same as 樸) a shrub (plant); thicket, strong and durable, a county in ancient times}
3BF8 CJK UNIFIED IDEOGRAPH-#3BF8, IRGKangXi=0558.300, Def{a basin}
3BF9 CJK UNIFIED IDEOGRAPH-#3BF9, IRGKangXi=0558.310, Def{(corrupted form of 檦) to show; to exhibit, to manifest, a kind of supporting post}
3BFA CJK UNIFIED IDEOGRAPH-#3BFA, IRGKangXi=0559.010, Def{a cabinet; a wardrobe; a cupboard, (same as 壏) hard ground}
3BFB CJK UNIFIED IDEOGRAPH-#3BFB, IRGKangXi=0559.150, Def{to bind; to tie up; a bundle; a bondage; restrictions; restraints}
3BFC CJK UNIFIED IDEOGRAPH-#3BFC, IRGKangXi=0559.261
3BFD CJK UNIFIED IDEOGRAPH-#3BFD, IRGKangXi=0559.261, Def{the areca-nut; the betel-nut}
3BFE CJK UNIFIED IDEOGRAPH-#3BFE, IRGKangXi=0559.320, Def{a tree, (same as 槽) with both flowers and fruits; half and half}
3BFF CJK UNIFIED IDEOGRAPH-#3BFF, IRGKangXi=0560.100, Def{handle of a ladle, a plant; wistaria, or wisteria, a certain trees in Sichuan; material for candle}
3C00 CJK UNIFIED IDEOGRAPH-#3C00, IRGKangXi=0560.110, Def{a kind of tree}
3C01 CJK UNIFIED IDEOGRAPH-#3C01, IRGKangXi=0560.160, Def{a tree with slanted branches}
3C02 CJK UNIFIED IDEOGRAPH-#3C02, IRGKangXi=0560.190, Def{a kind of tree; the juice of which is used to make wine}
3C03 CJK UNIFIED IDEOGRAPH-#3C03, IRGKangXi=0560.200, Def{(same as 檰) a tree, the bark of which is used in medicine-- Eucommia ulmoides, an awning of the house}
3C04 CJK UNIFIED IDEOGRAPH-#3C04, IRGKangXi=0560.230, Def{(non-classical form) a small chestnut tree, a kind of fruit; mountain plum}
3C05 CJK UNIFIED IDEOGRAPH-#3C05, IRGKangXi=0561.020, Def{(same as 樀) eaves of a house; brim, part of a loom, the cross beams on the frame on which silkworms spin, a bookcase, to abandon or give up}
3C06 CJK UNIFIED IDEOGRAPH-#3C06, IRGKangXi=0561.040, Def{(same as 韛) an instrument to blow a fire; a bellows for forge, etc.}
3C07 CJK UNIFIED IDEOGRAPH-#3C07, IRGKangXi=0561.111, Def{(a variant of 櫼) the square peck-shaped box half-way up a Chinese flagstaff}
3C08 CJK UNIFIED IDEOGRAPH-#3C08, IRGKangXi=0560.150, Def{nice and pretty}
3C09 CJK UNIFIED IDEOGRAPH-#3C09, IRGKangXi=0561.120, Def{(same as 榛) the hazel nut or filbert tree, a thorny tree}
3C0A CJK UNIFIED IDEOGRAPH-#3C0A, IRGKangXi=0561.260, Def{a kind of tree; thin leaves}
3C0B CJK UNIFIED IDEOGRAPH-#3C0B, IRGKangXi=0561.300, Def{a kind of fruit tree; betel-nut; the areca-nut}
3C0C CJK UNIFIED IDEOGRAPH-#3C0C, IRGKangXi=0561.380, Def{(non-classical form of 榷) to monopolize, to levy taxes}
3C0D CJK UNIFIED IDEOGRAPH-#3C0D, IRGKangXi=0561.400, Def{a cage, a pen, a grating, bars, window; window frame}
3C0E CJK UNIFIED IDEOGRAPH-#3C0E, IRGKangXi=0562.010, Def{(same as 檇) a wooden pestle or rammer, ancient name of a place in Zhejiang province; southwest of Jiaxing, a kind of fruit}
3C0F CJK UNIFIED IDEOGRAPH-#3C0F, IRGKangXi=0562.020, Def{(same as 篙) a bamboo pole; a pole for punting a boat}
3C10 CJK UNIFIED IDEOGRAPH-#3C10, IRGKangXi=0562.050, Def{(same as 橢) oval; oblong; elliptical}
3C11 CJK UNIFIED IDEOGRAPH-#3C11, IRGKangXi=0562.060, Def{fruit tree}
3C12 CJK UNIFIED IDEOGRAPH-#3C12, IRGKangXi=0562.101
3C13 CJK UNIFIED IDEOGRAPH-#3C13, IRGKangXi=0562.101
3C14 CJK UNIFIED IDEOGRAPH-#3C14, IRGKangXi=0562.200, Def{a kind of edible mushroom}
3C15 CJK UNIFIED IDEOGRAPH-#3C15, IRGKangXi=0562.250, Def{a ladle (often made of dried calabash or gourd)}
3C16 CJK UNIFIED IDEOGRAPH-#3C16, IRGKangXi=0563.010, Def{(same as 欖) the olive}
3C17 CJK UNIFIED IDEOGRAPH-#3C17, IRGKangXi=0563.080, Def{(same as 槭) a kind of maple (Acer palmatum)}
3C18 CJK UNIFIED IDEOGRAPH-#3C18, IRGKangXi=0563.210, Def{a kind of tree; often used to make instruments for sacrificial ceremonies in ancient times}
3C19 CJK UNIFIED IDEOGRAPH-#3C19, IRGKangXi=0563.280, Def{weak, feeble and tender branches}
3C1A CJK UNIFIED IDEOGRAPH-#3C1A, IRGKangXi=0563.310, Def{(same as 籬) a bamboo fence; a hedge}
3C1B CJK UNIFIED IDEOGRAPH-#3C1B, IRGKangXi=0563.320, Def{name of a county}
3C1C CJK UNIFIED IDEOGRAPH-#3C1C, IRGKangXi=0564.031
3C1D CJK UNIFIED IDEOGRAPH-#3C1D, IRGKangXi=0565.090, Def{to cough; cough}
3C1E CJK UNIFIED IDEOGRAPH-#3C1E, IRGKangXi=0565.140, Def{(same as 蚩 嗤) to laugh at; to make fun of; to deride; to ridicule}
3C1F CJK UNIFIED IDEOGRAPH-#3C1F, IRGKangXi=0565.160, Def{(same as 冀) hope; wish; to hope and scheme for, to stammer; to stutter; stuttering, to give}
3C20 CJK UNIFIED IDEOGRAPH-#3C20, IRGKangXi=0566.100, Def{to covet; greedy}
3C21 CJK UNIFIED IDEOGRAPH-#3C21, IRGKangXi=0566.120, Def{rash; impatient, very angry}
3C22 CJK UNIFIED IDEOGRAPH-#3C22, IRGKangXi=0566.170, Def{cough}
3C23 CJK UNIFIED IDEOGRAPH-#3C23, IRGKangXi=0566.210, Def{to vomit; to throw up; to disgorge, hiccough; shortwindedness, supposed upward motion of the spirit (in Chinese herb medicine), to sigh with grief or regret; to lament; to deplore}
3C24 CJK UNIFIED IDEOGRAPH-#3C24, IRGKangXi=0566.240, Def{to laugh heartily; to roar with laughter, to breath}
3C25 CJK UNIFIED IDEOGRAPH-#3C25, IRGKangXi=0566.270, Def{laugh}
3C26 CJK UNIFIED IDEOGRAPH-#3C26, IRGKangXi=0566.280, Def{to breath with mouth open}
3C27 CJK UNIFIED IDEOGRAPH-#3C27, IRGKangXi=0566.300, Def{to drink, to swallow}
3C28 CJK UNIFIED IDEOGRAPH-#3C28, IRGKangXi=0567.020, Def{to take breath or rest (after strenuous exercise), hiccough; shortwindedness}
3C29 CJK UNIFIED IDEOGRAPH-#3C29, IRGKangXi=0567.080, Def{sound of laughter}
3C2A CJK UNIFIED IDEOGRAPH-#3C2A, IRGKangXi=0567.150, Def{sound, wicked; mean; vicious}
3C2B CJK UNIFIED IDEOGRAPH-#3C2B, IRGKangXi=0567.210, Def{glad; joyful; happy, avaricious; greedy, cold}
3C2C CJK UNIFIED IDEOGRAPH-#3C2C, IRGKangXi=0567.240, Def{shortwindedness; hiccough, supposed upward motion of the spirit (in Chinese herb medicine), honesty; good faith; trust}
3C2D CJK UNIFIED IDEOGRAPH-#3C2D, IRGKangXi=0567.260, Def{pleasure; joy}
3C2E CJK UNIFIED IDEOGRAPH-#3C2E, IRGKangXi=0567.330, Def{to laugh at}
3C2F CJK UNIFIED IDEOGRAPH-#3C2F, IRGKangXi=0567.370, Def{the way of speaking; to spit at and not agree about something, to spit (same as 否) negative, to deny, cruel; savage; wicked; coarse; rustic}
3C30 CJK UNIFIED IDEOGRAPH-#3C30, IRGKangXi=0568.050, Def{asthmatic breathing, supposed upward motion of the spirit (in Chinese herb medicine), to long for; to covet, to hum over; to moan; to groan}
3C31 CJK UNIFIED IDEOGRAPH-#3C31, IRGKangXi=0568.060, Def{(same as 歃) to smear the mouth with the blood of a victim when taking an oath, to taste; to try}
3C32 CJK UNIFIED IDEOGRAPH-#3C32, IRGKangXi=0568.090, Def{to blow off, to blow; to puff}
3C33 CJK UNIFIED IDEOGRAPH-#3C33, IRGKangXi=0568.110, Def{asses braying, shortwindedness; hiccough, asthmatic breathing, supposed upward motion of the spirit (in Chinese herb medicine), to exhaust; extreme}
3C34 CJK UNIFIED IDEOGRAPH-#3C34, IRGKangXi=0568.130, Def{(of statement) to grate on the ear; earnest and faithful remonstrance}
3C35 CJK UNIFIED IDEOGRAPH-#3C35, IRGKangXi=0568.170, Def{to suck, to drink, to swallow}
3C36 CJK UNIFIED IDEOGRAPH-#3C36, IRGKangXi=0569.010, Def{to wrinkle the nose with sorrowful (when sob or weep), (same as 嘔) to vomit; to throw up}
3C37 CJK UNIFIED IDEOGRAPH-#3C37, IRGKangXi=0580.280, Def{resurrection; to come to life again, sick}
3C38 CJK UNIFIED IDEOGRAPH-#3C38, IRGKangXi=0569.111, Def{(a abbreviated form of 歛 斂) to collect; to gather; to hold together, to desire}
3C39 CJK UNIFIED IDEOGRAPH-#3C39, IRGKangXi=0569.120, Def{(same as 欦) to laugh; to smile, greed; avarice, to cry, to shout, overbear; arrogant}
3C3A CJK UNIFIED IDEOGRAPH-#3C3A, IRGKangXi=0569.160, Def{to drink; to swallow, disease of the throat; disease of the larynx}
3C3B CJK UNIFIED IDEOGRAPH-#3C3B, IRGKangXi=0570.020, Def{asses braying, to be happy suddenly, cruel; malignant; coarse, rude}
3C3C CJK UNIFIED IDEOGRAPH-#3C3C, IRGKangXi=0570.060, Def{greed; avarice, to blow off}
3C3D CJK UNIFIED IDEOGRAPH-#3C3D, IRGKangXi=0570.100, Def{to exert oneself to yell; to yell loudly; to shout, overbearing; arrogant, vigorous}
3C3E CJK UNIFIED IDEOGRAPH-#3C3E, IRGKangXi=0570.110, Def{songs of the people of 楚, a kind of music instrument in ancient times}
3C3F CJK UNIFIED IDEOGRAPH-#3C3F, IRGKangXi=0570.270, Def{to sigh in lamentation; to lament, to exclaim}
3C40 CJK UNIFIED IDEOGRAPH-#3C40, IRGKangXi=0571.030, Def{(same as 欼) to bite; to gnaw, (same as 嘬) to swallow a big mouthful without mastication}
3C41 CJK UNIFIED IDEOGRAPH-#3C41, IRGKangXi=0571.040, Def{to ask; to inquire; to investigate, (same as 款) sincerity; article, item, etc., to entertain, slowly}
3C42 CJK UNIFIED IDEOGRAPH-#3C42, IRGKangXi=0571.230, Def{hungry; starving, a poor harvest of grains}
3C43 CJK UNIFIED IDEOGRAPH-#3C43, IRGKangXi=0571.260, Def{(ancient form of 飲) to drink; to swallow}
3C44 CJK UNIFIED IDEOGRAPH-#3C44, IRGKangXi=0572.120, Def{to spit, to cough, keep quiet; to keep silence; speechless; wordless}
3C45 CJK UNIFIED IDEOGRAPH-#3C45, IRGKangXi=0572.200, Def{(ancient form of 懿) virtuous; fine; good}
3C46 CJK UNIFIED IDEOGRAPH-#3C46, IRGKangXi=0572.240, Def{to smile at each other}
3C47 CJK UNIFIED IDEOGRAPH-#3C47, IRGKangXi=0572.270, Def{fear; dread; fright; scare, sad; sorrowful; mournful}
3C48 CJK UNIFIED IDEOGRAPH-#3C48, IRGKangXi=0572.290, Def{name of a person}
3C49 CJK UNIFIED IDEOGRAPH-#3C49, IRGKangXi=0572.310, Def{moving of the mouth}
3C4A CJK UNIFIED IDEOGRAPH-#3C4A, IRGKangXi=0573.030, Def{asthmatic breathing; shortwindedness; hiccough; husky voice, (same as 嗄) (of voice) hoarse}
3C4B CJK UNIFIED IDEOGRAPH-#3C4B, IRGKangXi=0573.050, Def{(same as 塙 確) high; lofty; noble, sure; certain; firm; real; true}
3C4C CJK UNIFIED IDEOGRAPH-#3C4C, IRGKangXi=0573.150, Def{to take, to select, to get angry; to lose the temper}
3C4D CJK UNIFIED IDEOGRAPH-#3C4D, IRGKangXi=0573.190, Def{to owe, deficient, confused; puzzled}
3C4E CJK UNIFIED IDEOGRAPH-#3C4E, IRGKangXi=0573.220, Def{cannot be known; agnostic}
3C4F CJK UNIFIED IDEOGRAPH-#3C4F, IRGKangXi=0574.090, Def{(ancient form of 正) the right side, proper, pure, honest and virtuous, original, exactly}
3C50 CJK UNIFIED IDEOGRAPH-#3C50, IRGKangXi=0575.031
3C51 CJK UNIFIED IDEOGRAPH-#3C51, IRGKangXi=0575.031
3C52 CJK UNIFIED IDEOGRAPH-#3C52, IRGKangXi=0576.110, Def{(ancient form of 訶) to scold or blame in a loud voice}
3C53 CJK UNIFIED IDEOGRAPH-#3C53, IRGKangXi=0576.151
3C54 CJK UNIFIED IDEOGRAPH-#3C54, IRGKangXi=0576.151, Def{(same as 些) a small quantity or number; a little; a few; some}
3C55 CJK UNIFIED IDEOGRAPH-#3C55, IRGKangXi=0576.290, Def{(large seal type 歸) the marriage of a woman, to return to; to revert to; to go back, to belong to, to restore, to send back}
3C56 CJK UNIFIED IDEOGRAPH-#3C56, IRGKangXi=0576.300, Def{to stay; to stop (at a certain stage); to desist, to detain, to prohibit; to end, to come to; to stop at, still; calm, later}
3C57 CJK UNIFIED IDEOGRAPH-#3C57, IRGKangXi=0577.080, Def{a breath, news; tidings, to stop; to end}
3C58 CJK UNIFIED IDEOGRAPH-#3C58, IRGKangXi=0578.060, Def{(same as 魘) nightmare}
3C59 CJK UNIFIED IDEOGRAPH-#3C59, IRGKangXi=0578.210, Def{(same as 朽) rotten, decayed, useless, (same as 咼 剮) a wry mouth, to cut a criminal in pieces; to hack}
3C5A CJK UNIFIED IDEOGRAPH-#3C5A, IRGKangXi=0579.210, Def{(same as 殘) to destroy; to injure; to damage; to spoil, cruel and fierce, crippled}
3C5B CJK UNIFIED IDEOGRAPH-#3C5B, IRGKangXi=0579.270, Def{(non-classical form) to destroy; to injure; to damage, to exhaust, to complete; to finish, (ancient form 布) cloth; textiles}
3C5C CJK UNIFIED IDEOGRAPH-#3C5C, IRGKangXi=0579.330, Def{pestilence}
3C5D CJK UNIFIED IDEOGRAPH-#3C5D, IRGKangXi=0579.331
3C5E CJK UNIFIED IDEOGRAPH-#3C5E, IRGKangXi=0579.510, Def{danger; precarious; perilous, lofty; high, to decay; to break; to snap}
3C5F CJK UNIFIED IDEOGRAPH-#3C5F, IRGKangXi=0579.600, Def{to rip open the flesh, to break; to snap}
3C60 CJK UNIFIED IDEOGRAPH-#3C60, IRGKangXi=0580.010, Def{withered; dry, calamity; disaster, (ancient form 辜) sin; crime; guilt}
3C61 CJK UNIFIED IDEOGRAPH-#3C61, IRGKangXi=0580.160, Def{the soul out of one's body; as good as dead}
3C62 CJK UNIFIED IDEOGRAPH-#3C62, IRGKangXi=0581.020, Def{the dead}
3C63 CJK UNIFIED IDEOGRAPH-#3C63, IRGKangXi=0581.070, Def{uneven, weak; feeble}
3C64 CJK UNIFIED IDEOGRAPH-#3C64, IRGKangXi=0581.150, Def{to exhaust; extreme; topmost, to die}
3C65 CJK UNIFIED IDEOGRAPH-#3C65, IRGKangXi=0581.320, Def{the evil spirits come out, sick and poor, stupid; rude, inattentive; absent-minded; careless}
3C66 CJK UNIFIED IDEOGRAPH-#3C66, IRGKangXi=0581.350, Def{to discard, to abandon, to die; death}
3C67 CJK UNIFIED IDEOGRAPH-#3C67, IRGKangXi=0582.020, Def{stinking smell, expression of a dead person}
3C68 CJK UNIFIED IDEOGRAPH-#3C68, IRGKangXi=0582.030, Def{(same as 殮) to prepare a body for the coffin}
3C69 CJK UNIFIED IDEOGRAPH-#3C69, IRGKangXi=0582.041, Def{an abortion; stillborn}
3C6A CJK UNIFIED IDEOGRAPH-#3C6A, IRGKangXi=0582.050, Def{(standard form of 殙) stupid and confused; muddled and delirious; unconscious; in a state of coma, to breathe one's last}
3C6B CJK UNIFIED IDEOGRAPH-#3C6B, IRGKangXi=0582.060, Def{destroyed or ruined; to rot or decay; to disintegrate or decompose}
3C6C CJK UNIFIED IDEOGRAPH-#3C6C, IRGKangXi=0582.100, Def{cannot recognize people}
3C6D CJK UNIFIED IDEOGRAPH-#3C6D, IRGKangXi=0582.120, Def{(same as 毈) an infertile egg}
3C6E CJK UNIFIED IDEOGRAPH-#3C6E, IRGKangXi=0582.211, Def{(simplified form of 殨) to open, as an ulcer or sore; bursting of an abscess; inflamed; inflammation}
3C6F CJK UNIFIED IDEOGRAPH-#3C6F, IRGKangXi=0582.220, Def{get the unborn baby goat by killing the mother goat}
3C70 CJK UNIFIED IDEOGRAPH-#3C70, IRGKangXi=0582.260, Def{to destroy; to ruin, to put off}
3C71 CJK UNIFIED IDEOGRAPH-#3C71, IRGKangXi=0582.290, Def{uneven; unconscious, to know nothing about...}
3C72 CJK UNIFIED IDEOGRAPH-#3C72, IRGKangXi=0582.310, Def{withered; faded; to die, a minor illness (of foot and hand)}
3C73 CJK UNIFIED IDEOGRAPH-#3C73, IRGKangXi=0583.010, Def{lonely; lonesome}
3C74 CJK UNIFIED IDEOGRAPH-#3C74, IRGKangXi=0583.100, Def{sick, a dead animal, skeleton}
3C75 CJK UNIFIED IDEOGRAPH-#3C75, IRGKangXi=0583.190, Def{to die, to open, as an ulcer or sore}
3C76 CJK UNIFIED IDEOGRAPH-#3C76, IRGKangXi=0583.290, Def{a swollen corpse, swell}
3C77 CJK UNIFIED IDEOGRAPH-#3C77, IRGKangXi=0583.301
3C78 CJK UNIFIED IDEOGRAPH-#3C78, IRGKangXi=0583.330, Def{to exhaust; extreme; highest; farthest, to die}
3C79 CJK UNIFIED IDEOGRAPH-#3C79, IRGKangXi=0584.050, Def{to die from disease}
3C7A CJK UNIFIED IDEOGRAPH-#3C7A, IRGKangXi=0584.070, Def{(interchangeable 臚) the skin; the belly}
3C7B CJK UNIFIED IDEOGRAPH-#3C7B, IRGKangXi=0584.140, Def{diseases of dumb creatures, diseases of animals, illness; disease; ailment}
3C7C CJK UNIFIED IDEOGRAPH-#3C7C, IRGKangXi=0584.190, Def{a kind of ornaments, people wore in ancient time in order to avoid evil spirits or influences}
3C7D CJK UNIFIED IDEOGRAPH-#3C7D, IRGKangXi=0584.200, Def{(same as U+628C 抌) to hit or to strike heavily, to dip, to administer; to control, to govern; to treat; to cure, to punish}
3C7E CJK UNIFIED IDEOGRAPH-#3C7E, IRGKangXi=0584.280, Def{a kind of ornaments, people wore in ancient time in order to avoid evil spirits or influences, sound of laughter}
3C7F CJK UNIFIED IDEOGRAPH-#3C7F, IRGKangXi=0585.010, Def{(interchangeable 殼) the husk, skin or shell of fruits; the shell of snakes, insects, etc., the shells of mollusks; a bag or case made of leather for weapons, (interchangeable 慤) prudent; cautious, (same as 嗀) to vomit; to throw up, strong; durable; solid; firm; stable}
3C80 CJK UNIFIED IDEOGRAPH-#3C80, IRGKangXi=0585.050, Def{to beat; to strike; to attack, to move with happiness; excited, (a corrupted form) to laugh at}
3C81 CJK UNIFIED IDEOGRAPH-#3C81, IRGKangXi=0586.020, Def{to beat; to strike; to attack, to control, to charge; to sue, to kill all}
3C82 CJK UNIFIED IDEOGRAPH-#3C82, IRGKangXi=0586.030, Def{to push, unceremoniously; impudently}
3C83 CJK UNIFIED IDEOGRAPH-#3C83, IRGKangXi=0586.040, Def{smooth and curved, to strike or beat heavily}
3C84 CJK UNIFIED IDEOGRAPH-#3C84, IRGKangXi=0587.031, Def{unburnt bricks or tiles}
3C85 CJK UNIFIED IDEOGRAPH-#3C85, IRGKangXi=0587.090, Def{to bind, to exhaust; to use up; to complete; to finish, to slay}
3C86 CJK UNIFIED IDEOGRAPH-#3C86, IRGKangXi=0587.240, Def{noise}
3C87 CJK UNIFIED IDEOGRAPH-#3C87, IRGKangXi=0587.260, Def{(non-classical form of 鼕) the rattle of drums}
3C88 CJK UNIFIED IDEOGRAPH-#3C88, IRGKangXi=0587.270, Def{(same as 韶) the name of the music of the legendary Emperor, harmonious, (large seal type 鞀 婸) hand-drum used by pedlars; it is sounded by twirling it backwards in the hand, so that two swinging knobs can strike the face of the drum}
3C89 CJK UNIFIED IDEOGRAPH-#3C89, IRGKangXi=0588.030, Def{egg shells}
3C8A CJK UNIFIED IDEOGRAPH-#3C8A, IRGKangXi=0588.080, Def{(corrupted form of 叡) to understand thoroughly; quick or keen of perception, wise and clever, the profoundest; the divine sagacity of sages}
3C8B CJK UNIFIED IDEOGRAPH-#3C8B, IRGKangXi=0591.050, Def{a kind of animals (like rabbit, it is blue colored and much bigger)}
3C8C CJK UNIFIED IDEOGRAPH-#3C8C, IRGKangXi=0592.030, Def{dog's hair}
3C8D CJK UNIFIED IDEOGRAPH-#3C8D, IRGKangXi=0592.080, Def{woolen textiles; fine cloth}
3C8E CJK UNIFIED IDEOGRAPH-#3C8E, IRGKangXi=0592.120, Def{(non-classical form of 氀) woolen textiles; (non-classical form of 毹) woolen blanket, a kind of minority dress; stupid and vulgar}
3C8F CJK UNIFIED IDEOGRAPH-#3C8F, IRGKangXi=0592.150, Def{flying feather, soft; light}
3C90 CJK UNIFIED IDEOGRAPH-#3C90, IRGKangXi=0592.161
3C91 CJK UNIFIED IDEOGRAPH-#3C91, IRGKangXi=0592.161
3C92 CJK UNIFIED IDEOGRAPH-#3C92, IRGKangXi=0592.250, Def{(same as U+83E2 菢) to incubate; to brood; to hatch; to sit on a nest, a woolen blanket}
3C93 CJK UNIFIED IDEOGRAPH-#3C93, IRGKangXi=0593.090, Def{(a variant of 絨) fine, soft fur or hair; down, felt, camel's hair}
3C94 CJK UNIFIED IDEOGRAPH-#3C94, IRGKangXi=0593.120, Def{woolen textiles; fine cloth}
3C95 CJK UNIFIED IDEOGRAPH-#3C95, IRGKangXi=0593.130, Def{variegated; woolen fabric of different colors}
3C96 CJK UNIFIED IDEOGRAPH-#3C96, IRGKangXi=0593.140, Def{fur; feather; soft and light}
3C97 CJK UNIFIED IDEOGRAPH-#3C97, IRGKangXi=0593.190, Def{to smooth the hair, woolen textiles; fine cloth}
3C98 CJK UNIFIED IDEOGRAPH-#3C98, IRGKangXi=0593.260, Def{(corrupted form of 氍) fine woolen cloth, a mat used by the emperor in worshipping}
3C99 CJK UNIFIED IDEOGRAPH-#3C99, IRGKangXi=0593.270, Def{(same as 毹) woolen blanket with decorative design or pattern, to waste; to destroy}
3C9A CJK UNIFIED IDEOGRAPH-#3C9A, IRGKangXi=0593.290, Def{long hair}
3C9B CJK UNIFIED IDEOGRAPH-#3C9B, IRGKangXi=0594.040, Def{hairy animals}
3C9C CJK UNIFIED IDEOGRAPH-#3C9C, IRGKangXi=0594.060, Def{(non-classical form of 毯) rugs; carpets}
3C9D CJK UNIFIED IDEOGRAPH-#3C9D, IRGKangXi=0594.130, Def{(corrupt form of 氄) fine hair of birds and animals}
3C9E CJK UNIFIED IDEOGRAPH-#3C9E, IRGKangXi=0594.140, Def{short hair, growing hair}
3C9F CJK UNIFIED IDEOGRAPH-#3C9F, IRGKangXi=0594.290, Def{hair}
3CA0 CJK UNIFIED IDEOGRAPH-#3CA0, IRGKangXi=0594.330, Def{(non-classical, abbreviated form of 氂) horse tail, long hair; thick hair, (variant of 膧) a wild yak}
3CA1 CJK UNIFIED IDEOGRAPH-#3CA1, IRGKangXi=0594.350, Def{thick coat of fur, hairy}
3CA2 CJK UNIFIED IDEOGRAPH-#3CA2, IRGKangXi=0595.050, Def{hair knots in a mess}
3CA3 CJK UNIFIED IDEOGRAPH-#3CA3, IRGKangXi=0595.140, Def{woolen blanket with decorative design or pattern}
3CA4 CJK UNIFIED IDEOGRAPH-#3CA4, IRGKangXi=0595.160, Def{long hair, a few thin hair hanging down}
3CA5 CJK UNIFIED IDEOGRAPH-#3CA5, IRGKangXi=0595.220, Def{a kind of woolen textiles with curved pattern, a decorative ornament on an official hat}
3CA6 CJK UNIFIED IDEOGRAPH-#3CA6, IRGKangXi=0595.230, Def{long hair; mane}
3CA7 CJK UNIFIED IDEOGRAPH-#3CA7, IRGKangXi=0595.250, Def{hairy}
3CA8 CJK UNIFIED IDEOGRAPH-#3CA8, IRGKangXi=0595.260, Def{a decorative fine blanket on the top of a saddle, petty; trifling}
3CA9 CJK UNIFIED IDEOGRAPH-#3CA9, IRGKangXi=0595.351
3CAA CJK UNIFIED IDEOGRAPH-#3CAA, IRGKangXi=0596.120, Def{woolen blanket with decorative design or pattern, a kind of animal}
3CAB CJK UNIFIED IDEOGRAPH-#3CAB, IRGKangXi=0596.130, Def{hair knots in a mess}
3CAC CJK UNIFIED IDEOGRAPH-#3CAC, IRGKangXi=0596.180, Def{hair decorations on a helmet used in ancient times, weak and weary; weak from exhaustion; tired looking of a bird's feather}
3CAD CJK UNIFIED IDEOGRAPH-#3CAD, IRGKangXi=0596.220, Def{(corrupted form of 毯) rug; carpet; blanket}
3CAE CJK UNIFIED IDEOGRAPH-#3CAE, IRGKangXi=0596.231
3CAF CJK UNIFIED IDEOGRAPH-#3CAF, IRGKangXi=0596.360, Def{long hairy}
3CB0 CJK UNIFIED IDEOGRAPH-#3CB0, IRGKangXi=0596.380, Def{a hairy dog}
3CB1 CJK UNIFIED IDEOGRAPH-#3CB1, IRGKangXi=0597.050, Def{(same as 鬣) long beard or whiskers, a mane; bristles, as on a hog; dorsal fins}
3CB2 CJK UNIFIED IDEOGRAPH-#3CB2, IRGKangXi=0597.161, Def{fine cotton cloth; fine woolen cloth}
3CB3 CJK UNIFIED IDEOGRAPH-#3CB3, IRGKangXi=0598.090, Def{to touch; to ram, to pull with hands}
3CB4 CJK UNIFIED IDEOGRAPH-#3CB4, IRGKangXi=0599.100, Def{mind; disposition; temper}
3CB5 CJK UNIFIED IDEOGRAPH-#3CB5, IRGKangXi=0600.010, Def{(same as 霄) the skies, night, to exhaust; to dissolve, clouds or mists}
3CB6 CJK UNIFIED IDEOGRAPH-#3CB6, IRGKangXi=0600.051, Def{(old translated form) chlorine (symbol Cl)}
3CB7 CJK UNIFIED IDEOGRAPH-#3CB7, IRGKangXi=0600.051, Def{(old translated form) nitrogen (symbol N)}
3CB8 CJK UNIFIED IDEOGRAPH-#3CB8, IRGKangXi=0603.050, Def{dried up (of waterway; the channel of a river; a pond, etc.)}
3CB9 CJK UNIFIED IDEOGRAPH-#3CB9, IRGKangXi=0603.060, Def{(non-classical form of 氿) the dry soil on the river side, spring water from the hole of mountain side, a small fountain, shore; bank; beach, name of a lake in today's Jiangsu Province Yixing county}
3CBA CJK UNIFIED IDEOGRAPH-#3CBA, IRGKangXi=0603.090, Def{to wash; to rinse; to clean; to spurt; to blow out}
3CBB CJK UNIFIED IDEOGRAPH-#3CBB, IRGKangXi=0603.110, Def{(ancient form of U+6EBA 溺) sunk; perished, drawn; infatuated}
3CBC CJK UNIFIED IDEOGRAPH-#3CBC, IRGKangXi=0604.060, Def{name of a river, (same as 汊) a branching stream; branching creeks}
3CBD CJK UNIFIED IDEOGRAPH-#3CBD, IRGKangXi=0606.010, Def{name of a river in today's southwest of Shanxi Province, the wet things attach or stick up to each other, dirty and muddy}
3CBE CJK UNIFIED IDEOGRAPH-#3CBE, IRGKangXi=0607.030, Def{(same as 雨) rain, to rain down; to pour down}
3CBF CJK UNIFIED IDEOGRAPH-#3CBF, IRGKangXi=0607.051, Def{extensive body of water; broad and deep of water; momentum of moving water deep and wide}
3CC0 CJK UNIFIED IDEOGRAPH-#3CC0, IRGKangXi=0608.020, Def{a river in ancient times}
3CC1 CJK UNIFIED IDEOGRAPH-#3CC1, IRGKangXi=0609.020, Def{water flow; water current; momentum of moving water}
3CC2 CJK UNIFIED IDEOGRAPH-#3CC2, IRGKangXi=0609.080, Def{(same as 沿) to follow a course; to go along; to coast, to hand down; to continue, to conserve; along or by, as a road or a coast}
3CC3 CJK UNIFIED IDEOGRAPH-#3CC3, IRGKangXi=0611.040, Def{(non-classical of 淬) to temper iron, or steel for making swords, etc. (also used figuratively), to dip into water; to soak; to dye}
3CC4 CJK UNIFIED IDEOGRAPH-#3CC4, IRGKangXi=0611.090, Def{(a variant of 涎) spittle, saliva}
3CC5 CJK UNIFIED IDEOGRAPH-#3CC5, IRGKangXi=0611.140, Def{(ancient form of 流) to flow, to drift, to circulate, a current, to descend, unstable; weak}
3CC6 CJK UNIFIED IDEOGRAPH-#3CC6, IRGKangXi=0612.090, Def{name of a place in ancient times}
3CC7 CJK UNIFIED IDEOGRAPH-#3CC7, IRGKangXi=0612.121, Def{name of a place in today's Jiangsu Province Yixing county, to float; to waft; to swim}
3CC8 CJK UNIFIED IDEOGRAPH-#3CC8, IRGKangXi=0612.121, Def{(standard form of 沛) a great flow of water; flowing copiously, quickly; rapidly; sudden, flourishing; luxuriant; prosperous or abundant, marsh; swamp}
3CC9 CJK UNIFIED IDEOGRAPH-#3CC9, IRGKangXi=0612.121
3CCA CJK UNIFIED IDEOGRAPH-#3CCA, IRGKangXi=0612.121, Def{(same as 攸) the flowing of the water, name of a river}
3CCB CJK UNIFIED IDEOGRAPH-#3CCB, IRGKangXi=0613.050, Def{a river in ancient times}
3CCC CJK UNIFIED IDEOGRAPH-#3CCC, IRGKangXi=0614.060, Def{marshy; swampy, damp; moist}
3CCD CJK UNIFIED IDEOGRAPH-#3CCD, IRGKangXi=0615.150, Def{a place in ancient times}
3CCE CJK UNIFIED IDEOGRAPH-#3CCE, IRGKangXi=0615.160, Def{guide or lead to make the water flowing smoothly}
3CCF CJK UNIFIED IDEOGRAPH-#3CCF, IRGKangXi=0616.060, Def{a river in ancient times; in Henan Province}
3CD0 CJK UNIFIED IDEOGRAPH-#3CD0, IRGKangXi=0617.040, Def{flowing of the water, sound of the flowing water}
3CD1 CJK UNIFIED IDEOGRAPH-#3CD1, IRGKangXi=0619.040, Def{(abbreviated form of U+6EA2 溢) to flow over; to brim over, excessive}
3CD2 CJK UNIFIED IDEOGRAPH-#3CD2, IRGKangXi=0619.121, Def{(a variant of 法) statutes, laws, regulations, a rule, legal standard, plan or methods, etc.}
3CD3 CJK UNIFIED IDEOGRAPH-#3CD3, IRGKangXi=0619.121
3CD4 CJK UNIFIED IDEOGRAPH-#3CD4, IRGKangXi=0619.121, Def{to soak or be soaked; to dip; to immerse}
3CD5 CJK UNIFIED IDEOGRAPH-#3CD5, IRGKangXi=0619.121, Def{(simplified form of 灡) water in which rice has been washed, (interchangeable 瀾) a great wave; a huge billow}
3CD6 CJK UNIFIED IDEOGRAPH-#3CD6, IRGKangXi=0619.240, Def{name of a river}
3CD7 CJK UNIFIED IDEOGRAPH-#3CD7, IRGKangXi=0619.300, Def{bank of a river, (same as 汊) a branching stream}
3CD8 CJK UNIFIED IDEOGRAPH-#3CD8, IRGKangXi=0619.320, Def{a fountain or spring flows downwards, sound of the flowing water}
3CD9 CJK UNIFIED IDEOGRAPH-#3CD9, IRGKangXi=0620.030, Def{(non-classical form of U+6D93 涓) a brook, the smallest drop of water, to expel evil influences; to eliminate; rid off, to clean up, to select, a tributary (or a river), the flowing of a spring, (interchangeable with U+6CEB 泫) to glisten; to sparkle, to weep}
3CDA CJK UNIFIED IDEOGRAPH-#3CDA, IRGKangXi=0620.080, Def{flowing of the water}
3CDB CJK UNIFIED IDEOGRAPH-#3CDB, IRGKangXi=0622.020, Def{dirty; filthy, corrupt, soap, black, to unravel silk threads}
3CDC CJK UNIFIED IDEOGRAPH-#3CDC, IRGKangXi=0622.060, Def{the flowing water}
3CDD CJK UNIFIED IDEOGRAPH-#3CDD, IRGKangXi=0623.130, Def{(corrupted form of U+6D2D 洭) a river in ancient times; a part of Huangjiang, Lianjiang and Beijiang in today's north-west of Guangdong Province}
3CDE CJK UNIFIED IDEOGRAPH-#3CDE, IRGKangXi=0623.171, Def{(same as 浺) deep and far; profound and abstruse (of the sea)}
3CDF CJK UNIFIED IDEOGRAPH-#3CDF, IRGKangXi=0623.171
3CE0 CJK UNIFIED IDEOGRAPH-#3CE0, IRGKangXi=0627.311, Def{(simplified form of 澾) slippery (as a road)}
3CE1 CJK UNIFIED IDEOGRAPH-#3CE1, IRGKangXi=0627.311, Def{(simplified form of 濄) (same as 渦) name of a river, to whirl, an eddy}
3CE2 CJK UNIFIED IDEOGRAPH-#3CE2, IRGKangXi=0623.171
3CE3 CJK UNIFIED IDEOGRAPH-#3CE3, IRGKangXi=0623.171
3CE4 CJK UNIFIED IDEOGRAPH-#3CE4, IRGKangXi=0624.080, Def{(non-classical form of 淑) good; pure; virtuous, beautiful or charming (women), clear}
3CE5 CJK UNIFIED IDEOGRAPH-#3CE5, IRGKangXi=0624.090, Def{a river in ancient times}
3CE6 CJK UNIFIED IDEOGRAPH-#3CE6, IRGKangXi=0624.170, Def{a river}
3CE7 CJK UNIFIED IDEOGRAPH-#3CE7, IRGKangXi=0625.080, Def{the water flowing from ..., the current of a stream}
3CE8 CJK UNIFIED IDEOGRAPH-#3CE8, IRGKangXi=0625.100, Def{to splash (of water)}
3CE9 CJK UNIFIED IDEOGRAPH-#3CE9, IRGKangXi=0625.110, Def{great billows; large waves; the waves swelling up}
3CEA CJK UNIFIED IDEOGRAPH-#3CEA, IRGKangXi=0626.050, Def{a river}
3CEB CJK UNIFIED IDEOGRAPH-#3CEB, IRGKangXi=0626.060, Def{(corrupted form of 沓) repeated; joined, crowded together, a family name}
3CEC CJK UNIFIED IDEOGRAPH-#3CEC, IRGKangXi=0626.160, Def{an eddy; a whirlpool}
3CED CJK UNIFIED IDEOGRAPH-#3CED, IRGKangXi=0627.210, Def{(non-classical form of U+6D8E 涎) spittle, saliva}
3CEE CJK UNIFIED IDEOGRAPH-#3CEE, IRGKangXi=0627.270, Def{(interchangeable 尿) urine, to urinate}
3CEF CJK UNIFIED IDEOGRAPH-#3CEF, IRGKangXi=0627.280, Def{goddess of an ink-slab}
3CF0 CJK UNIFIED IDEOGRAPH-#3CF0, IRGKangXi=0627.311, Def{(same as 溝) ditch; waterway; moat, groove}
3CF1 CJK UNIFIED IDEOGRAPH-#3CF1, IRGKangXi=0627.311
3CF2 CJK UNIFIED IDEOGRAPH-#3CF2, IRGKangXi=0627.311
3CF3 CJK UNIFIED IDEOGRAPH-#3CF3, IRGKangXi=0627.311
3CF4 CJK UNIFIED IDEOGRAPH-#3CF4, IRGKangXi=0628.060, Def{mud; quagmire}
3CF5 CJK UNIFIED IDEOGRAPH-#3CF5, IRGKangXi=0628.120, Def{name of a river}
3CF6 CJK UNIFIED IDEOGRAPH-#3CF6, IRGKangXi=0628.150, Def{name of a river, to give suck to, (same as 醹) vintage wine, (same as 乳) milk; breasts}
3CF7 CJK UNIFIED IDEOGRAPH-#3CF7, IRGKangXi=0628.220, Def{dark blue color, to combine; to unite, to close; to shut, the water is exhausted, to get the color away by soaking in the water}
3CF8 CJK UNIFIED IDEOGRAPH-#3CF8, IRGKangXi=0629.130, Def{name of a river}
3CF9 CJK UNIFIED IDEOGRAPH-#3CF9, IRGKangXi=0629.160, Def{flood; a great flood; massive water, name a river, name of a place in today's Shanxi Province east of Yin county, (same as 汪) deep and extensive (said of water)}
3CFA CJK UNIFIED IDEOGRAPH-#3CFA, IRGKangXi=0630.100, Def{(non-classical form of 游) to swim; to float, to roam; to travel}
3CFB CJK UNIFIED IDEOGRAPH-#3CFB, IRGKangXi=0630.150, Def{a bank of earth or an embankment to block the current of the water, to stop (or block) up the flowing water}
3CFC CJK UNIFIED IDEOGRAPH-#3CFC, IRGKangXi=0630.190, Def{(same as 滭) bubbling of fountain; copious of spring (same as 淢) swift currents (interchangeable 洫) a ditch; a moat, to overflow}
3CFD CJK UNIFIED IDEOGRAPH-#3CFD, IRGKangXi=0631.020, Def{(non-classical abbreviated form of 瀰) a watery expanse, to be covered with, to disseminate every where, currents of the water}
3CFE CJK UNIFIED IDEOGRAPH-#3CFE, IRGKangXi=0631.070, Def{(standard form of 羌) name of a river}
3CFF CJK UNIFIED IDEOGRAPH-#3CFF, IRGKangXi=0631.130, Def{(same as 渫) rolling billows, to get rid of; to scatter, muddy, to ooze}
3D00 CJK UNIFIED IDEOGRAPH-#3D00, IRGKangXi=0633.040, Def{deep}
3D01 CJK UNIFIED IDEOGRAPH-#3D01, IRGKangXi=0633.060, Def{to add to; to increase; increase; full, plentiful; abundant, saliva}
3D02 CJK UNIFIED IDEOGRAPH-#3D02, IRGKangXi=0633.180, Def{great billows; great upheavals; boisterous breakers}
3D03 CJK UNIFIED IDEOGRAPH-#3D03, IRGKangXi=0633.200, Def{(same as 淚) tears}
3D04 CJK UNIFIED IDEOGRAPH-#3D04, IRGKangXi=0633.250, Def{the water flowing in a turned, crooked or winding and revolving way; an eddy}
3D05 CJK UNIFIED IDEOGRAPH-#3D05, IRGKangXi=0633.260, Def{(ancient form of 消) to melt; to thaw, to disperse; to dissipate, to cancel; to annul}
3D06 CJK UNIFIED IDEOGRAPH-#3D06, IRGKangXi=0633.290, Def{(same as 浸) to dip; to immerse; to soak; to permeate; to percolate, gradual; gradually}
3D07 CJK UNIFIED IDEOGRAPH-#3D07, IRGKangXi=0633.320, Def{(same as 涉) to wade, to experience, to involve; to implicate}
3D08 CJK UNIFIED IDEOGRAPH-#3D08, IRGKangXi=0633.330, Def{(same as 垽) sediment; dregs; precipitate; lees}
3D09 CJK UNIFIED IDEOGRAPH-#3D09, IRGKangXi=0631.180, Def{(ancient form of 濟) various; varied; numerous, elegant and dignified}
3D0A CJK UNIFIED IDEOGRAPH-#3D0A, IRGKangXi=0633.371, Def{(same as U+6E06 淵) an abyss; a gulf, to be deep}
3D0B CJK UNIFIED IDEOGRAPH-#3D0B, IRGKangXi=0633.371, Def{(simplified form of 潚) deep and clear (said of water) (same as 溲) to urinate, to immerse; to soak; to drench, to wash rice (same as 瀟) rushing rain and wind; roar of gust}
3D0C CJK UNIFIED IDEOGRAPH-#3D0C, IRGKangXi=0633.371
3D0D CJK UNIFIED IDEOGRAPH-#3D0D, IRGKangXi=0633.371
3D0E CJK UNIFIED IDEOGRAPH-#3D0E, IRGKangXi=0633.371
3D0F CJK UNIFIED IDEOGRAPH-#3D0F, IRGKangXi=0634.040, Def{the pond water is still}
3D10 CJK UNIFIED IDEOGRAPH-#3D10, IRGKangXi=0634.110, Def{great current, flood, flowing water}
3D11 CJK UNIFIED IDEOGRAPH-#3D11, IRGKangXi=0634.130, Def{(same as 泝) to trace up to a source, to go against the stream/water}
3D12 CJK UNIFIED IDEOGRAPH-#3D12, IRGKangXi=0635.210, Def{water current; water flow}
3D13 CJK UNIFIED IDEOGRAPH-#3D13, IRGKangXi=0636.010, Def{to wash rice}
3D14 CJK UNIFIED IDEOGRAPH-#3D14, IRGKangXi=0636.160, Def{(same as 潝) the noise of flowing water, swift flowing water (same as non-classical form of 澀) rough; harsh; not smooth, a slightly bitter taste}
3D15 CJK UNIFIED IDEOGRAPH-#3D15, IRGKangXi=0636.180, Def{to boil away; to boil over, boiling sound, sound of the flowing water, etc., (interchangeable 渫) rolling billows}
3D16 CJK UNIFIED IDEOGRAPH-#3D16, IRGKangXi=0637.060, Def{(same as 濼) name of a stream in Shandong Province southwest of Jinan, (same as 泊) lakes, a hot spring; a mineral spring}
3D17 CJK UNIFIED IDEOGRAPH-#3D17, IRGKangXi=0637.090, Def{deep}
3D18 CJK UNIFIED IDEOGRAPH-#3D18, IRGKangXi=0638.040, Def{to ooze out; to well out; to spring out, to well up; flood tide; the water is rising}
3D19 CJK UNIFIED IDEOGRAPH-#3D19, IRGKangXi=0638.080, Def{water dripping; a drip-tile (same as 炸 煠 爚) to fry in fat or oil, to scald}
3D1A CJK UNIFIED IDEOGRAPH-#3D1A, IRGKangXi=0638.140, Def{the small ditch in the field}
3D1B CJK UNIFIED IDEOGRAPH-#3D1B, IRGKangXi=0638.240, Def{damp; wet; moist; humid}
3D1C CJK UNIFIED IDEOGRAPH-#3D1C, IRGKangXi=0638.360, Def{(same as 遍) everywhere; all over, a time}
3D1D CJK UNIFIED IDEOGRAPH-#3D1D, IRGKangXi=0639.090, Def{(a variant of U+9ECE 黎) many; numerous, black; dark}
3D1E CJK UNIFIED IDEOGRAPH-#3D1E, IRGKangXi=0639.111, Def{(same as 滔) fluent, to fill; to prevail}
3D1F CJK UNIFIED IDEOGRAPH-#3D1F, IRGKangXi=0639.111
3D20 CJK UNIFIED IDEOGRAPH-#3D20, IRGKangXi=0639.111
3D21 CJK UNIFIED IDEOGRAPH-#3D21, IRGKangXi=0639.111
3D22 CJK UNIFIED IDEOGRAPH-#3D22, IRGKangXi=0639.111
3D23 CJK UNIFIED IDEOGRAPH-#3D23, IRGKangXi=0639.111
3D24 CJK UNIFIED IDEOGRAPH-#3D24, IRGKangXi=0639.111
3D25 CJK UNIFIED IDEOGRAPH-#3D25, IRGKangXi=0639.210, Def{great billows; heavy seas rolling in}
3D26 CJK UNIFIED IDEOGRAPH-#3D26, IRGKangXi=0639.290, Def{(corrupted form of 涫) (interchangeable 盥) to wash hands, to wash, to boil}
3D27 CJK UNIFIED IDEOGRAPH-#3D27, IRGKangXi=0640.080, Def{of flowing water, a river}
3D28 CJK UNIFIED IDEOGRAPH-#3D28, IRGKangXi=0640.130, Def{water current; water flow}
3D29 CJK UNIFIED IDEOGRAPH-#3D29, IRGKangXi=0640.180, Def{(same as 灉) name of a stream in Shandong Province, the flowing back of flooding waters, a sluice}
3D2A CJK UNIFIED IDEOGRAPH-#3D2A, IRGKangXi=0640.200, Def{name of a river}
3D2B CJK UNIFIED IDEOGRAPH-#3D2B, IRGKangXi=0640.210, Def{great flowing; big flood, name of a county in ancient times}
3D2C CJK UNIFIED IDEOGRAPH-#3D2C, IRGKangXi=0640.290, Def{a river in ancient times; head of source from today's Henan Province Xinanxian; and flowing south to Luoshui}
3D2D CJK UNIFIED IDEOGRAPH-#3D2D, IRGKangXi=0641.090, Def{immeasurable depth or profundity; extremely abstruse; unfathomable}
3D2E CJK UNIFIED IDEOGRAPH-#3D2E, IRGKangXi=0641.150, Def{water current; water flow}
3D2F CJK UNIFIED IDEOGRAPH-#3D2F, IRGKangXi=0641.160, Def{water current; water flow, name of a stream}
3D30 CJK UNIFIED IDEOGRAPH-#3D30, IRGKangXi=0641.230, Def{a river in ancient times}
3D31 CJK UNIFIED IDEOGRAPH-#3D31, IRGKangXi=0641.270, Def{(ancient form of 深) deep, profound, far, very; extremely}
3D32 CJK UNIFIED IDEOGRAPH-#3D32, IRGKangXi=0642.060, Def{a river in ancient time; today's Baiquanhe, water's edge; water-front; a bank; a limit}
3D33 CJK UNIFIED IDEOGRAPH-#3D33, IRGKangXi=0642.080, Def{(corrupted form) to smear; to spread, to paint, to blot out, mud; mire, ignorant (said of a child) innocent}
3D34 CJK UNIFIED IDEOGRAPH-#3D34, IRGKangXi=0642.140, Def{(ancient form of 湛) (interchangeable 沈 沉) sink, deep; profound, joy; delight, happy; peaceful (interchangeable 潭) deep water; deep pool, leisurely; relaxed, dewy, full; filled, wet; damp; moist, clear (interchangeable 浸) to dip; to immerse; to soak, swellings; roaring waves and billows, flowing water, (same as 霪 淫) to rain cats and dogs for a long time, a river in ancient times in Henan province Jiyuanxian (blocked)}
3D35 CJK UNIFIED IDEOGRAPH-#3D35, IRGKangXi=0642.190, Def{water current; water flow}
3D36 CJK UNIFIED IDEOGRAPH-#3D36, IRGKangXi=0642.210, Def{to water; to pour water; to irrigate, to wet; to moisten; to soak in, to wash}
3D37 CJK UNIFIED IDEOGRAPH-#3D37, IRGKangXi=0642.250, Def{wet; moist; damp, incessantly rains, flood; great current}
3D38 CJK UNIFIED IDEOGRAPH-#3D38, IRGKangXi=0643.020, Def{swift currents; swift flow of water}
3D39 CJK UNIFIED IDEOGRAPH-#3D39, IRGKangXi=0643.241
3D3A CJK UNIFIED IDEOGRAPH-#3D3A, IRGKangXi=0643.241
3D3B CJK UNIFIED IDEOGRAPH-#3D3B, IRGKangXi=0643.241
3D3C CJK UNIFIED IDEOGRAPH-#3D3C, IRGKangXi=0644.050, Def{wet; moist; damp}
3D3D CJK UNIFIED IDEOGRAPH-#3D3D, IRGKangXi=0644.130, Def{water current, water flow}
3D3E CJK UNIFIED IDEOGRAPH-#3D3E, IRGKangXi=0645.060, Def{(same as 渤) (of water) swelling or rising; an inland sea among today's Liaoning, Hebei and Shandong Provinces}
3D3F CJK UNIFIED IDEOGRAPH-#3D3F, IRGKangXi=0645.100, Def{(same as 濎) (water etc.) boiling}
3D40 CJK UNIFIED IDEOGRAPH-#3D40, IRGKangXi=0646.020, Def{water flow; water current}
3D41 CJK UNIFIED IDEOGRAPH-#3D41, IRGKangXi=0646.050, Def{vast and boundless of flowing water, a water waste, to reach}
3D42 CJK UNIFIED IDEOGRAPH-#3D42, IRGKangXi=0646.070, Def{name of a river}
3D43 CJK UNIFIED IDEOGRAPH-#3D43, IRGKangXi=0646.140, Def{name of a stream}
3D44 CJK UNIFIED IDEOGRAPH-#3D44, IRGKangXi=0646.200, Def{(same as 漢) name of a dynasty, belonging to China, the Milky Way the Han River, (interchangeable 域) a frontier; a boundary; a region; a country, to live; to stay}
3D45 CJK UNIFIED IDEOGRAPH-#3D45, IRGKangXi=0646.230, Def{a river, in today's northeast of Sichuan Province, between Bazhong and Tongjiang}
3D46 CJK UNIFIED IDEOGRAPH-#3D46, IRGKangXi=0648.160, Def{(same as 澔 浩) radiance of gems, great currents; vast water flow}
3D47 CJK UNIFIED IDEOGRAPH-#3D47, IRGKangXi=0648.281
3D48 CJK UNIFIED IDEOGRAPH-#3D48, IRGKangXi=0648.281
3D49 CJK UNIFIED IDEOGRAPH-#3D49, IRGKangXi=0648.281
3D4A CJK UNIFIED IDEOGRAPH-#3D4A, IRGKangXi=0648.360, Def{moving image; moving shadow, flowing water}
3D4B CJK UNIFIED IDEOGRAPH-#3D4B, IRGKangXi=0648.380, Def{(variant of 汨) name of a river in Hunan Province}
3D4C CJK UNIFIED IDEOGRAPH-#3D4C, IRGKangXi=0649.020, Def{to flow; to move; to stray}
3D4D CJK UNIFIED IDEOGRAPH-#3D4D, IRGKangXi=0649.210, Def{filled with water; full of water, annoyed; vexed; depressed; bored, to make tea by boiling water and then put the lid on to cover it}
3D4E CJK UNIFIED IDEOGRAPH-#3D4E, IRGKangXi=0649.220, Def{a mountain stream or torrent, a number measuring used in ancient times; a hundred million waterways (ditches) equal to a mountain stream, a river in ancient, head source in south of Henan Province, flowing east then north to combine with Gushui (today's Jianhe)}
3D4F CJK UNIFIED IDEOGRAPH-#3D4F, IRGKangXi=0650.080, Def{(same as 漼) having the appearance of depth, to shed tears; to cry or weep, sharp, bright-colored, a river in ancient times; Guishui, (interchangeable 摧) to destroy; to injure; to damage, to store up; to accumulate}
3D50 CJK UNIFIED IDEOGRAPH-#3D50, IRGKangXi=0650.110, Def{name of a river in Hubei Province, name of a state in ancient times}
3D51 CJK UNIFIED IDEOGRAPH-#3D51, IRGKangXi=0650.140, Def{name of a river}
3D52 CJK UNIFIED IDEOGRAPH-#3D52, IRGKangXi=0650.180, Def{(same as U+6CB8 沸) boiling (water, etc.) to gush; bubbling up, (sane as U+6E43 湃) the roaring of billows, name of a person}
3D53 CJK UNIFIED IDEOGRAPH-#3D53, IRGKangXi=0650.250, Def{(same as 湜) the water is clear, transparent (of water)}
3D54 CJK UNIFIED IDEOGRAPH-#3D54, IRGKangXi=0650.310, Def{not clear; not pure}
3D55 CJK UNIFIED IDEOGRAPH-#3D55, IRGKangXi=0651.140, Def{(same as 滲) to permeate; to seep; to ooze; to leak; to soak through}
3D56 CJK UNIFIED IDEOGRAPH-#3D56, IRGKangXi=0651.310, Def{damp; moist; wet}
3D57 CJK UNIFIED IDEOGRAPH-#3D57, IRGKangXi=0651.370, Def{turbulent (said of water)}
3D58 CJK UNIFIED IDEOGRAPH-#3D58, IRGKangXi=0652.321, Def{great currents}
3D59 CJK UNIFIED IDEOGRAPH-#3D59, IRGKangXi=0652.321, Def{(corrupted form of 清) pure; clear, brief; virtuous; to place in order, simple and easily understandable}
3D5A CJK UNIFIED IDEOGRAPH-#3D5A, IRGKangXi=0652.321
3D5B CJK UNIFIED IDEOGRAPH-#3D5B, IRGKangXi=0652.321
3D5C CJK UNIFIED IDEOGRAPH-#3D5C, IRGKangXi=0652.321
3D5D CJK UNIFIED IDEOGRAPH-#3D5D, IRGKangXi=0652.370, Def{moving, as in ripples; moving to and fro (said of water) (interchangeable 裔) the hem of a robe, a border, a frontier, border bribes}
3D5E CJK UNIFIED IDEOGRAPH-#3D5E, IRGKangXi=0652.460, Def{bellied in water-gas, belch; burp}
3D5F CJK UNIFIED IDEOGRAPH-#3D5F, IRGKangXi=0653.040, Def{(non-classical form of 溦) light rain; drizzle}
3D60 CJK UNIFIED IDEOGRAPH-#3D60, IRGKangXi=0653.070, Def{to gush forth; to well out; to spring out, deep water}
3D61 CJK UNIFIED IDEOGRAPH-#3D61, IRGKangXi=0653.100, Def{(said of flow of water) swift and torrential; turbulent flow of water}
3D62 CJK UNIFIED IDEOGRAPH-#3D62, IRGKangXi=0653.120, Def{a marsh in ancient times; in today's Shandong Province; between Heze and Yun Cheng}
3D63 CJK UNIFIED IDEOGRAPH-#3D63, IRGKangXi=0653.170, Def{(same as 渴) thirsty, to thirst, urgent; anxiously (awaiting, etc.), (interchangeable 愒) to rest; to stop, to idle away (time)}
3D64 CJK UNIFIED IDEOGRAPH-#3D64, IRGKangXi=0653.230, Def{to overflow; to spread}
3D65 CJK UNIFIED IDEOGRAPH-#3D65, IRGKangXi=0653.240, Def{(a variant) water currents; flow water}
3D66 CJK UNIFIED IDEOGRAPH-#3D66, IRGKangXi=0654.030, Def{(same as 瀡) slippery}
3D67 CJK UNIFIED IDEOGRAPH-#3D67, IRGKangXi=0654.060, Def{name of a river, clear, pure; to cover; to hide}
3D68 CJK UNIFIED IDEOGRAPH-#3D68, IRGKangXi=0654.080, Def{a dry land in a river; an islet, tributary stream; an affluent stream}
3D69 CJK UNIFIED IDEOGRAPH-#3D69, IRGKangXi=0654.160, Def{many; flourishing; luxuriant growth (of vegetation)}
3D6A CJK UNIFIED IDEOGRAPH-#3D6A, IRGKangXi=0654.180, Def{(same as 涎) saliva, overflowing water, to rise, to board, a region; an area; a place}
3D6B CJK UNIFIED IDEOGRAPH-#3D6B, IRGKangXi=0654.280, Def{water gushing out; bubbling up, waves sweeping over the flowing water}
3D6C CJK UNIFIED IDEOGRAPH-#3D6C, IRGKangXi=0655.070, Def{used in person's name}
3D6D CJK UNIFIED IDEOGRAPH-#3D6D, IRGKangXi=0655.100, Def{(same as 渚) an islet, a bank; water's edge; water-front, island (in the sea), (interchangeable 劌) a pool; a pond or small lake, a stream in ancient times}
3D6E CJK UNIFIED IDEOGRAPH-#3D6E, IRGKangXi=0655.201, Def{clear (fresh) water}
3D6F CJK UNIFIED IDEOGRAPH-#3D6F, IRGKangXi=0655.260, Def{sound of water flowing; flowing water}
3D70 CJK UNIFIED IDEOGRAPH-#3D70, IRGKangXi=0655.320, Def{(same as 潊) name of a stream; Xushui (or Shuanglongjiang) in ancient times, waterside; shore}
3D71 CJK UNIFIED IDEOGRAPH-#3D71, IRGKangXi=0655.360, Def{flowing water}
3D72 CJK UNIFIED IDEOGRAPH-#3D72, IRGKangXi=0656.070, Def{(same as 潕) name of a river}
3D73 CJK UNIFIED IDEOGRAPH-#3D73, IRGKangXi=0656.160, Def{(same as non-classical form of 漻) crystal-clear (water), fluent, name of stream; in today's Hubei Province Xiaoganshi}
3D74 CJK UNIFIED IDEOGRAPH-#3D74, IRGKangXi=0656.180, Def{to sink}
3D75 CJK UNIFIED IDEOGRAPH-#3D75, IRGKangXi=0656.200, Def{swift currents of the stream, sound of water flowing; flowing water}
3D76 CJK UNIFIED IDEOGRAPH-#3D76, IRGKangXi=0657.110, Def{full; filled up}
3D77 CJK UNIFIED IDEOGRAPH-#3D77, IRGKangXi=0657.120, Def{turbulent flowing of the water, swift currents}
3D78 CJK UNIFIED IDEOGRAPH-#3D78, IRGKangXi=0657.240, Def{(same as non-classical abbreviated form of 瀹) to boil, to wash; to cleanse; to soak, to cook; to stew; to decoct, to channel, to enlighten, to clean or dredge (a waterway), water currents}
3D79 CJK UNIFIED IDEOGRAPH-#3D79, IRGKangXi=0657.300, Def{waves; breakers; billows, dense; thick}
3D7A CJK UNIFIED IDEOGRAPH-#3D7A, IRGKangXi=0657.361, Def{name of a stream in ancient times; in Danyang, to plant; to sow}
3D7B CJK UNIFIED IDEOGRAPH-#3D7B, IRGKangXi=0657.361, Def{(interchangeable 膄) to perspire from embarrassment or shame}
3D7C CJK UNIFIED IDEOGRAPH-#3D7C, IRGKangXi=0657.361
3D7D CJK UNIFIED IDEOGRAPH-#3D7D, IRGKangXi=0657.370, Def{(same as abbreviated form of 灅) the strength of a current, flowing water, name of a stream; at Yoybeiping}
3D7E CJK UNIFIED IDEOGRAPH-#3D7E, IRGKangXi=0658.020, Def{cold; chilly, (ancient form of 凈) clean; pure; to purify}
3D7F CJK UNIFIED IDEOGRAPH-#3D7F, IRGKangXi=0658.050, Def{endlessly long or vast water currents}
3D80 CJK UNIFIED IDEOGRAPH-#3D80, IRGKangXi=0658.290, Def{deep and vast of the flowing water, name of a stream}
3D81 CJK UNIFIED IDEOGRAPH-#3D81, IRGKangXi=0658.310, Def{after blocking the water flow takes different waterway, sound of the flowing water}
3D82 CJK UNIFIED IDEOGRAPH-#3D82, IRGKangXi=0659.291
3D83 CJK UNIFIED IDEOGRAPH-#3D83, IRGKangXi=0659.291
3D84 CJK UNIFIED IDEOGRAPH-#3D84, IRGKangXi=0659.370, Def{to stain, dirty; filth, water flows forward}
3D85 CJK UNIFIED IDEOGRAPH-#3D85, IRGKangXi=0659.410, Def{(same as 澩) dried up mountain creeks, a tributary of Weishui (in ancient times), sound of the roaring waves and billows}
3D86 CJK UNIFIED IDEOGRAPH-#3D86, IRGKangXi=0659.430, Def{name of a stream in Hebei Province Hengshan, name of a river (same as 涂水)}
3D87 CJK UNIFIED IDEOGRAPH-#3D87, IRGKangXi=0659.460, Def{a ferry, to across the river in a boat, a ferry boat, a raft}
3D88 CJK UNIFIED IDEOGRAPH-#3D88, IRGKangXi=0660.100, Def{(same as 濙 U+6FD9) creeks}
3D89 CJK UNIFIED IDEOGRAPH-#3D89, IRGKangXi=0659.291, Def{Mandarin duck, Aix galericulata}
3D8A CJK UNIFIED IDEOGRAPH-#3D8A, IRGKangXi=0659.291
3D8B CJK UNIFIED IDEOGRAPH-#3D8B, IRGKangXi=0659.291
3D8C CJK UNIFIED IDEOGRAPH-#3D8C, IRGKangXi=0660.330, Def{name of a stream; head source from Henan Province jiyuanxian Wangwushan}
3D8D CJK UNIFIED IDEOGRAPH-#3D8D, IRGKangXi=0661.040, Def{name of a stream}
3D8E CJK UNIFIED IDEOGRAPH-#3D8E, IRGKangXi=0661.150, Def{(same as 澴) swift and torrential (said of flow of water), name of a stream in Hubei Province}
3D8F CJK UNIFIED IDEOGRAPH-#3D8F, IRGKangXi=0661.240, Def{an ancient name of a river in Henan Province Dengfengxian}
3D90 CJK UNIFIED IDEOGRAPH-#3D90, IRGKangXi=0662.101
3D91 CJK UNIFIED IDEOGRAPH-#3D91, IRGKangXi=0662.101, Def{(same as 瀲) overflowing (water), the edge of a large body of water}
3D92 CJK UNIFIED IDEOGRAPH-#3D92, IRGKangXi=0662.130, Def{the current of a stream, brave; courageous; decision with courage; (a person) of determination, swift flowing of the water currents, to float; to waft, to overflow}
3D93 CJK UNIFIED IDEOGRAPH-#3D93, IRGKangXi=0662.190, Def{to sink; to drown; to be submerged}
3D94 CJK UNIFIED IDEOGRAPH-#3D94, IRGKangXi=0662.240, Def{(said of flow of water) swift and torrential, name of a stream}
3D95 CJK UNIFIED IDEOGRAPH-#3D95, IRGKangXi=0663.091, Def{to wash rice}
3D96 CJK UNIFIED IDEOGRAPH-#3D96, IRGKangXi=0662.290, Def{rapid; turbulent of the waves and billows}
3D97 CJK UNIFIED IDEOGRAPH-#3D97, IRGKangXi=0662.320, Def{well up; spring out of a fountain or geyser, name of a fountain in today's Henan Province}
3D98 CJK UNIFIED IDEOGRAPH-#3D98, IRGKangXi=0663.060, Def{full of water}
3D99 CJK UNIFIED IDEOGRAPH-#3D99, IRGKangXi=0663.091
3D9A CJK UNIFIED IDEOGRAPH-#3D9A, IRGKangXi=0663.271, Def{(a variant of 灞) name of a river in Shanxi}
3D9B CJK UNIFIED IDEOGRAPH-#3D9B, IRGKangXi=0663.290, Def{name of a river, moving; swinging as in ripples}
3D9C CJK UNIFIED IDEOGRAPH-#3D9C, IRGKangXi=0663.360, Def{(same as 淵) deep water; vast, profound}
3D9D CJK UNIFIED IDEOGRAPH-#3D9D, IRGKangXi=0663.421
3D9E CJK UNIFIED IDEOGRAPH-#3D9E, IRGKangXi=0663.440, Def{(same as 瀼) water currents; flowing of water, name of a river in Sichuan Province, heavily bedewed}
3D9F CJK UNIFIED IDEOGRAPH-#3D9F, IRGKangXi=0664.010, Def{an ancient name for a river in Shanxi and Hebei}
3DA0 CJK UNIFIED IDEOGRAPH-#3DA0, IRGKangXi=0664.160, Def{water currents; flowing of water}
3DA1 CJK UNIFIED IDEOGRAPH-#3DA1, IRGKangXi=0665.080, Def{(ancient form of 火) bright lights and illuminations of the fire, fire sounds}
3DA2 CJK UNIFIED IDEOGRAPH-#3DA2, IRGKangXi=0666.061
3DA3 CJK UNIFIED IDEOGRAPH-#3DA3, IRGKangXi=0666.100, Def{to burn; to heat, to burn over a wider and wider area; to glow; to shine, light; brightness}
3DA4 CJK UNIFIED IDEOGRAPH-#3DA4, IRGKangXi=0666.261, Def{(non-classical form of 炒) to fry; to roast; to cook}
3DA5 CJK UNIFIED IDEOGRAPH-#3DA5, IRGKangXi=0666.261
3DA6 CJK UNIFIED IDEOGRAPH-#3DA6, IRGKangXi=0666.300, Def{ashes; embers, remnants of}
3DA7 CJK UNIFIED IDEOGRAPH-#3DA7, IRGKangXi=0667.080, Def{warm}
3DA8 CJK UNIFIED IDEOGRAPH-#3DA8, IRGKangXi=0667.100, Def{(same as 炋) fire}
3DA9 CJK UNIFIED IDEOGRAPH-#3DA9, IRGKangXi=0668.031
3DAA CJK UNIFIED IDEOGRAPH-#3DAA, IRGKangXi=0668.031
3DAB CJK UNIFIED IDEOGRAPH-#3DAB, IRGKangXi=0668.070, Def{(corrupted form of 燎) to burn, to illuminate; a signal light; brilliant}
3DAC CJK UNIFIED IDEOGRAPH-#3DAC, IRGKangXi=0668.140, Def{bright lights and illuminations of the fire, flames}
3DAD CJK UNIFIED IDEOGRAPH-#3DAD, IRGKangXi=0668.230, Def{almost dry; not completely dried}
3DAE CJK UNIFIED IDEOGRAPH-#3DAE, IRGKangXi=0668.300, Def{(same as 燒) to burn, to roast, to boil; to heat}
3DAF CJK UNIFIED IDEOGRAPH-#3DAF, IRGKangXi=0669.050, Def{fire}
3DB0 CJK UNIFIED IDEOGRAPH-#3DB0, IRGKangXi=0669.090, Def{flame; fire, light; brightness}
3DB1 CJK UNIFIED IDEOGRAPH-#3DB1, IRGKangXi=0669.101
3DB2 CJK UNIFIED IDEOGRAPH-#3DB2, IRGKangXi=0669.101, Def{measuring unit for thermodynamics}
3DB3 CJK UNIFIED IDEOGRAPH-#3DB3, IRGKangXi=0669.260, Def{(same as 燼) ashes; ember}
3DB4 CJK UNIFIED IDEOGRAPH-#3DB4, IRGKangXi=0670.110, Def{a very great fire; flourishing flames, rich; exuberant; grand; prosperous}
3DB5 CJK UNIFIED IDEOGRAPH-#3DB5, IRGKangXi=0671.081, Def{(same as U+9901 飪) to cook food thoroughly}
3DB6 CJK UNIFIED IDEOGRAPH-#3DB6, IRGKangXi=0671.081, Def{scorched, burned}
3DB7 CJK UNIFIED IDEOGRAPH-#3DB7, IRGKangXi=0671.081
3DB8 CJK UNIFIED IDEOGRAPH-#3DB8, IRGKangXi=0671.081
3DB9 CJK UNIFIED IDEOGRAPH-#3DB9, IRGKangXi=0671.250, Def{fire; flames}
3DBA CJK UNIFIED IDEOGRAPH-#3DBA, IRGKangXi=0671.290, Def{the light or glow of fire}
3DBB CJK UNIFIED IDEOGRAPH-#3DBB, IRGKangXi=0672.050, Def{(same as 烽) a conical brick-structure in which to light a beacon; (in ancient China) a tall structure (on a city wall, etc.) where fire was made to signal enemy invasion or presence of bandits}
3DBC CJK UNIFIED IDEOGRAPH-#3DBC, IRGKangXi=0672.080, Def{very hot, to burn, to heat, to roast, dazzling; bright; shining, a very great fire; flourishing flames}
3DBD CJK UNIFIED IDEOGRAPH-#3DBD, IRGKangXi=0673.111, Def{(Cant.) 火㶽, to instant-boil thin slices of meat and vegetables in a chafing dish at the dining table}
3DBE CJK UNIFIED IDEOGRAPH-#3DBE, IRGKangXi=0672.150, Def{(non-classical abbreviated form of 熛) tough and honest; upright, flames burst into a blaze}
3DBF CJK UNIFIED IDEOGRAPH-#3DBF, IRGKangXi=0672.170, Def{the smoke went up; giving forth smoke; smoking; misty}
3DC0 CJK UNIFIED IDEOGRAPH-#3DC0, IRGKangXi=0673.111, Def{(same as 焭,煢,惸) solitary; alone; single; brotherless; friendless; helpless, dice; a kind of gambling game played in ancient times}
3DC1 CJK UNIFIED IDEOGRAPH-#3DC1, IRGKangXi=0673.111
3DC2 CJK UNIFIED IDEOGRAPH-#3DC2, IRGKangXi=0673.170, Def{bright; light; brilliant}
3DC3 CJK UNIFIED IDEOGRAPH-#3DC3, IRGKangXi=0673.190, Def{long-burning fire}
3DC4 CJK UNIFIED IDEOGRAPH-#3DC4, IRGKangXi=0673.250, Def{fire; flames}
3DC5 CJK UNIFIED IDEOGRAPH-#3DC5, IRGKangXi=0674.020, Def{(same as 炒) to cook; to stew or simmer; to fry; to roast; to broil (rice)}
3DC6 CJK UNIFIED IDEOGRAPH-#3DC6, IRGKangXi=0674.080, Def{thriving; flourishing}
3DC7 CJK UNIFIED IDEOGRAPH-#3DC7, IRGKangXi=0674.140, Def{(same as 恢) great; immense; enormous; vast; extensive}
3DC8 CJK UNIFIED IDEOGRAPH-#3DC8, IRGKangXi=0674.160, Def{embers kept for starting a new fire, or any burning object (covered by ashes) which causes a fire disaster, the farm products getting heated with piling up and closing completely}
3DC9 CJK UNIFIED IDEOGRAPH-#3DC9, IRGKangXi=0674.210, Def{(a variant of U+71A8 熨) to iron, an iron for smoothing garments (same as 尉) to still; to quiet, a military official}
3DCA CJK UNIFIED IDEOGRAPH-#3DCA, IRGKangXi=0675.191, Def{(non-classical form of 焚) to burn; to set fire to}
3DCB CJK UNIFIED IDEOGRAPH-#3DCB, IRGKangXi=0675.191, Def{embers}
3DCC CJK UNIFIED IDEOGRAPH-#3DCC, IRGKangXi=0675.191
3DCD CJK UNIFIED IDEOGRAPH-#3DCD, IRGKangXi=0675.191, Def{a chemical term; known formerly as benzene}
3DCE CJK UNIFIED IDEOGRAPH-#3DCE, IRGKangXi=0676.020, Def{(same as 暍) sunstroke, hot; feverish}
3DCF CJK UNIFIED IDEOGRAPH-#3DCF, IRGKangXi=0676.030, Def{used in person's name}
3DD0 CJK UNIFIED IDEOGRAPH-#3DD0, IRGKangXi=0677.040, Def{(same as 燬) fire; blaze, to destroy by fire; to burn down}
3DD1 CJK UNIFIED IDEOGRAPH-#3DD1, IRGKangXi=0677.110, Def{(same as 煙 捈) smoke; fumes, tobacco; opium; cigarette, mist; vapor}
3DD2 CJK UNIFIED IDEOGRAPH-#3DD2, IRGKangXi=0677.140, Def{to cook meals; to cook food; to prepare a dinner}
3DD3 CJK UNIFIED IDEOGRAPH-#3DD3, IRGKangXi=0677.230, Def{(same as 熜) a torch (formed by binding the stem of the hemp), chimney; stack}
3DD4 CJK UNIFIED IDEOGRAPH-#3DD4, IRGKangXi=0677.300, Def{(same as non-classical form of 焰) flame; blaze; glowing; brilliant}
3DD5 CJK UNIFIED IDEOGRAPH-#3DD5, IRGKangXi=0678.020, Def{hearing sound}
3DD6 CJK UNIFIED IDEOGRAPH-#3DD6, IRGKangXi=0678.090, Def{(same as 照) to shine upon; to light or illumine}
3DD7 CJK UNIFIED IDEOGRAPH-#3DD7, IRGKangXi=0678.161, Def{(a corrupted form) used in person's name}
3DD8 CJK UNIFIED IDEOGRAPH-#3DD8, IRGKangXi=0678.161, Def{coal; charcoal}
3DD9 CJK UNIFIED IDEOGRAPH-#3DD9, IRGKangXi=0678.161, Def{(same as 煎) to fry in fat or oil, to cook}
3DDA CJK UNIFIED IDEOGRAPH-#3DDA, IRGKangXi=0678.161
3DDB CJK UNIFIED IDEOGRAPH-#3DDB, IRGKangXi=0678.161, Def{to boil, cook; a pot, kettle}
3DDC CJK UNIFIED IDEOGRAPH-#3DDC, IRGKangXi=0678.161
3DDD CJK UNIFIED IDEOGRAPH-#3DDD, IRGKangXi=0678.161
3DDE CJK UNIFIED IDEOGRAPH-#3DDE, IRGKangXi=0678.161
3DDF CJK UNIFIED IDEOGRAPH-#3DDF, IRGKangXi=0678.180, Def{(same as non-classical form of 嬯) to scald the bristles off a pig or the feathers off a bird}
3DE0 CJK UNIFIED IDEOGRAPH-#3DE0, IRGKangXi=0678.200, Def{A will-o'-the-wisp; a flitting light}
3DE1 CJK UNIFIED IDEOGRAPH-#3DE1, IRGKangXi=0678.260, Def{to startle; to amaze; to astound}
3DE2 CJK UNIFIED IDEOGRAPH-#3DE2, IRGKangXi=0678.280, Def{bundle of charcoal, to dry in the sun}
3DE3 CJK UNIFIED IDEOGRAPH-#3DE3, IRGKangXi=0678.290, Def{red; the color of fire}
3DE4 CJK UNIFIED IDEOGRAPH-#3DE4, IRGKangXi=0678.320, Def{rosy clouds of dawn; rosy dawn, sounds of the fire}
3DE5 CJK UNIFIED IDEOGRAPH-#3DE5, IRGKangXi=0678.350, Def{(same as 烝) steam, to cook by steaming}
3DE6 CJK UNIFIED IDEOGRAPH-#3DE6, IRGKangXi=0680.050, Def{(non-classical form of 瞁) to open the eyes with astonishment}
3DE7 CJK UNIFIED IDEOGRAPH-#3DE7, IRGKangXi=0680.101
3DE8 CJK UNIFIED IDEOGRAPH-#3DE8, IRGKangXi=0680.101
3DE9 CJK UNIFIED IDEOGRAPH-#3DE9, IRGKangXi=0680.101, Def{(a variant of 熙) bright, splendid. Intelligent, prosperous}
3DEA CJK UNIFIED IDEOGRAPH-#3DEA, IRGKangXi=0680.101, Def{(a variant) fat; rich, a stew of fish}
3DEB CJK UNIFIED IDEOGRAPH-#3DEB, IRGKangXi=0680.170, Def{to bake or dry by fire}
3DEC CJK UNIFIED IDEOGRAPH-#3DEC, IRGKangXi=0680.180, Def{fire}
3DED CJK UNIFIED IDEOGRAPH-#3DED, IRGKangXi=0680.250, Def{(same as 烽) a conical brick-structure in which to light a beacon; (in ancient China) a tall structure (on a city wall, etc.) where fire was made to signal enemy invasion or presence of bandits}
3DEE CJK UNIFIED IDEOGRAPH-#3DEE, IRGKangXi=0680.290, Def{scorched or burned wood, to burn; to heat, to destroy in fire, to roast or bake}
3DEF CJK UNIFIED IDEOGRAPH-#3DEF, IRGKangXi=0680.400, Def{smoke and dust everywhere; air pollution}
3DF0 CJK UNIFIED IDEOGRAPH-#3DF0, IRGKangXi=0681.060, Def{flames}
3DF1 CJK UNIFIED IDEOGRAPH-#3DF1, IRGKangXi=0681.141
3DF2 CJK UNIFIED IDEOGRAPH-#3DF2, IRGKangXi=0681.141
3DF3 CJK UNIFIED IDEOGRAPH-#3DF3, IRGKangXi=0681.150, Def{color of the flame}
3DF4 CJK UNIFIED IDEOGRAPH-#3DF4, IRGKangXi=0682.020, Def{(same as 爏) fire}
3DF5 CJK UNIFIED IDEOGRAPH-#3DF5, IRGKangXi=0682.160, Def{(same as 爅) fire; flame; light}
3DF6 CJK UNIFIED IDEOGRAPH-#3DF6, IRGKangXi=0682.180, Def{to dry or warm (grains) near a fire}
3DF7 CJK UNIFIED IDEOGRAPH-#3DF7, IRGKangXi=0683.120, Def{(same as 臇) thick soup; broth, fat; rich, a stew of fish}
3DF8 CJK UNIFIED IDEOGRAPH-#3DF8, IRGKangXi=0684.030, Def{(a variant of 燁) blaze of fire, splendid, glorious}
3DF9 CJK UNIFIED IDEOGRAPH-#3DF9, IRGKangXi=0684.080, Def{intense heat, to fry in intensive fire}
3DFA CJK UNIFIED IDEOGRAPH-#3DFA, IRGKangXi=0684.161
3DFB CJK UNIFIED IDEOGRAPH-#3DFB, IRGKangXi=0684.161, Def{(same as 無) without, none, a negative (a variant 嬭) to burn food in cocking; singed; burnt}
3DFC CJK UNIFIED IDEOGRAPH-#3DFC, IRGKangXi=0684.161, Def{(standard form of 燕) a swallow, comfort; ease; to soothe (interchangeable 晏) to feast; to enjoy, name of a certain feudal States}
3DFD CJK UNIFIED IDEOGRAPH-#3DFD, IRGKangXi=0684.161
3DFE CJK UNIFIED IDEOGRAPH-#3DFE, IRGKangXi=0685.020, Def{blaze of fire; burning vigorously}
3DFF CJK UNIFIED IDEOGRAPH-#3DFF, IRGKangXi=0685.060, Def{flame; fire, bitter; acrid taste}
3E00 CJK UNIFIED IDEOGRAPH-#3E00, IRGKangXi=0685.201, Def{(non-classical form of 炱) blackened with soot}
3E01 CJK UNIFIED IDEOGRAPH-#3E01, IRGKangXi=0685.201, Def{(corrupted form of 熯) dry; freely burning; to burn; to roast (dialect) to dry or heat near a fire; consume by fire, to expose to sunlight (same as 焊) to weld; to join with solder, respectful; reverent; deferential}
3E02 CJK UNIFIED IDEOGRAPH-#3E02, IRGKangXi=0685.010
3E03 CJK UNIFIED IDEOGRAPH-#3E03, IRGKangXi=0685.201, Def{(same as 點) a dot; a spot; a speck, a point, a little; a mite, to count; to check on by one, to punctuate}
3E04 CJK UNIFIED IDEOGRAPH-#3E04, IRGKangXi=0686.090, Def{(same as 齌) a raging fire}
3E05 CJK UNIFIED IDEOGRAPH-#3E05, IRGKangXi=0686.210, Def{ashes; candle end}
3E06 CJK UNIFIED IDEOGRAPH-#3E06, IRGKangXi=0686.271
3E07 CJK UNIFIED IDEOGRAPH-#3E07, IRGKangXi=0686.271
3E08 CJK UNIFIED IDEOGRAPH-#3E08, IRGKangXi=0687.020, Def{(same as 焦) scorched or burned, worried and anxious}
3E09 CJK UNIFIED IDEOGRAPH-#3E09, IRGKangXi=0687.050, Def{(non-classical form of 燮) to adapt; to adjust; to blend; to harmonize}
3E0A CJK UNIFIED IDEOGRAPH-#3E0A, IRGKangXi=0687.070, Def{fierce of fire, cruel; malicious, fire; flame}
3E0B CJK UNIFIED IDEOGRAPH-#3E0B, IRGKangXi=0687.090, Def{(interchangeable 膰) meats used in sacrifice}
3E0C CJK UNIFIED IDEOGRAPH-#3E0C, IRGKangXi=0687.171, Def{to twinkle; bright light in glimpses; flashing}
3E0D CJK UNIFIED IDEOGRAPH-#3E0D, IRGKangXi=0687.190, Def{to burn, to roast, to boil; to heat}
3E0E CJK UNIFIED IDEOGRAPH-#3E0E, IRGKangXi=0687.300, Def{warm, fire; flame}
3E0F CJK UNIFIED IDEOGRAPH-#3E0F, IRGKangXi=0688.060, Def{(same as 爢) cooked or well-done; cooked soft, (interchangeable 糜) mashed}
3E10 CJK UNIFIED IDEOGRAPH-#3E10, IRGKangXi=0688.070, Def{(an ancient form of 然) to burn; to light a fire}
3E11 CJK UNIFIED IDEOGRAPH-#3E11, IRGKangXi=0688.160, Def{(same as 爨) a cooking-stove; to cook, a mess, a kind of theatrical composition}
3E12 CJK UNIFIED IDEOGRAPH-#3E12, IRGKangXi=0688.360, Def{to absurd pursuit or desire; to wish wildly; to desire to long for (usually more than one's rightful share), to knit; to weave}
3E13 CJK UNIFIED IDEOGRAPH-#3E13, IRGKangXi=0492.121, Def{to have disappeared, avoid, not at all, not supporting by}
3E14 CJK UNIFIED IDEOGRAPH-#3E14, IRGKangXi=0689.121
3E15 CJK UNIFIED IDEOGRAPH-#3E15, IRGKangXi=0689.221, Def{(same as 攫) to seize; to take hold of; to snatch}
3E16 CJK UNIFIED IDEOGRAPH-#3E16, IRGKangXi=0690.031, Def{hide or to keep in store the vegetable in cold winter days}
3E17 CJK UNIFIED IDEOGRAPH-#3E17, IRGKangXi=0690.050, Def{not a real father or a nature father (as distinct from foster father)}
3E18 CJK UNIFIED IDEOGRAPH-#3E18, IRGKangXi=0690.080, Def{(ancient form of 萬) ten thousand, all, very; extremely; absolutely}
3E19 CJK UNIFIED IDEOGRAPH-#3E19, IRGKangXi=0690.100, Def{(dialect) father, husband of a wet nurse (addressed in ancient times)}
3E1A CJK UNIFIED IDEOGRAPH-#3E1A, IRGKangXi=0690.140, Def{scattered or dispersed and clear, to stop, to detain, a connection, lineage}
3E1B CJK UNIFIED IDEOGRAPH-#3E1B, IRGKangXi=0691.180
3E1C CJK UNIFIED IDEOGRAPH-#3E1C, IRGKangXi=0692.080, Def{to get through the space enclosed by a constellation; to go out of}
3E1D CJK UNIFIED IDEOGRAPH-#3E1D, IRGKangXi=0693.070, Def{(same as 杴) a shovel, a trough to carry off water, a wood-board; a plank}
3E1E CJK UNIFIED IDEOGRAPH-#3E1E, IRGKangXi=0693.080, Def{blocks for printing, supporting boards used in building walls}
3E1F CJK UNIFIED IDEOGRAPH-#3E1F, IRGKangXi=0693.170, Def{to cover; to close; to shut}
3E20 CJK UNIFIED IDEOGRAPH-#3E20, IRGKangXi=0693.180, Def{thick stick; stout carrying pole, beam of a bed}
3E21 CJK UNIFIED IDEOGRAPH-#3E21, IRGKangXi=0693.230, Def{board to block the water current, to cross a river by a cross-beam}
3E22 CJK UNIFIED IDEOGRAPH-#3E22, IRGKangXi=0694.080, Def{cross-beam of a bed, window, a mould; a pattern}
3E23 CJK UNIFIED IDEOGRAPH-#3E23, IRGKangXi=0694.280, Def{(same as 業) in ancient China, big piece of board (plank) on a music rack with the carving of the shape of saw's teeth; used to hang clock (musical stone, drum, etc.), instruction}
3E24 CJK UNIFIED IDEOGRAPH-#3E24, IRGKangXi=0694.290, Def{broken leather}
3E25 CJK UNIFIED IDEOGRAPH-#3E25, IRGKangXi=0695.020, Def{household registers, printing plate, supporting boards used in building walls, (same as 欃) sandalwood, water gate; sluice; floodgate}
3E26 CJK UNIFIED IDEOGRAPH-#3E26, IRGKangXi=0695.050, Def{(non-classical form same as 互) mutually; together; each other (dialect) scratching or clawing to and fro}
3E27 CJK UNIFIED IDEOGRAPH-#3E27, IRGKangXi=0695.070, Def{(interchangeable 啃) to bite; to gnaw, a gnashing sound, (standard form 齦) gums (of the teeth), erosion; to erode}
3E28 CJK UNIFIED IDEOGRAPH-#3E28, IRGKangXi=0697.050, Def{a big bull}
3E29 CJK UNIFIED IDEOGRAPH-#3E29, IRGKangXi=0697.080, Def{to desist an ox (止牛)}
3E2A CJK UNIFIED IDEOGRAPH-#3E2A, IRGKangXi=0697.100, Def{an ox walking slowly}
3E2B CJK UNIFIED IDEOGRAPH-#3E2B, IRGKangXi=0698.070, Def{(same as 犍) castrated bull; ox, name of a political division in ancient China; today's Sichuan Province, a kind of animal (looks like an ox)}
3E2C CJK UNIFIED IDEOGRAPH-#3E2C, IRGKangXi=0698.090, Def{a two-years-old ox, ox with a long body, ox with long and big feet}
3E2D CJK UNIFIED IDEOGRAPH-#3E2D, IRGKangXi=0698.100, Def{the opposite; contrary of the cattle's horns}
3E2E CJK UNIFIED IDEOGRAPH-#3E2E, IRGKangXi=0698.120, Def{bull; bullock; ox (castrated), a four years old ox}
3E2F CJK UNIFIED IDEOGRAPH-#3E2F, IRGKangXi=0699.020, Def{a fine breed of cattle (cow, bull or ox), hornless cattle}
3E30 CJK UNIFIED IDEOGRAPH-#3E30, IRGKangXi=0699.120, Def{(same as 犐) hornless cattle}
3E31 CJK UNIFIED IDEOGRAPH-#3E31, IRGKangXi=0699.130, Def{(same as 㸰,犐) hornless cattle}
3E32 CJK UNIFIED IDEOGRAPH-#3E32, IRGKangXi=0699.160, Def{a kind of mountain cattle; big and heavy}
3E33 CJK UNIFIED IDEOGRAPH-#3E33, IRGKangXi=0699.190, Def{a kind of cattle}
3E34 CJK UNIFIED IDEOGRAPH-#3E34, IRGKangXi=0700.061
3E35 CJK UNIFIED IDEOGRAPH-#3E35, IRGKangXi=0700.110, Def{the lowing of an ox}
3E36 CJK UNIFIED IDEOGRAPH-#3E36, IRGKangXi=0700.120, Def{color of the oxtail; a flat yellow}
3E37 CJK UNIFIED IDEOGRAPH-#3E37, IRGKangXi=0700.130, Def{(same as 觢) a kind of cattle with two horns straight up}
3E38 CJK UNIFIED IDEOGRAPH-#3E38, IRGKangXi=0700.170, Def{calf, bull; bullock; ox (castrated), (a variant) domesticated animals}
3E39 CJK UNIFIED IDEOGRAPH-#3E39, IRGKangXi=0700.240, Def{cattle with white stripe on the back, variegated; parti-colored, mixed; impure}
3E3A CJK UNIFIED IDEOGRAPH-#3E3A, IRGKangXi=0700.250, Def{a cow; female of certain animals}
3E3B CJK UNIFIED IDEOGRAPH-#3E3B, IRGKangXi=0700.260, Def{a kind of cattle; a yearling calf}
3E3C CJK UNIFIED IDEOGRAPH-#3E3C, IRGKangXi=0700.280, Def{(same as 犎) the zebu; or humped-ox}
3E3D CJK UNIFIED IDEOGRAPH-#3E3D, IRGKangXi=0701.050, Def{a two-years-old ox, ox with long and big feet}
3E3E CJK UNIFIED IDEOGRAPH-#3E3E, IRGKangXi=0701.080, Def{(same as 牣) to filled up; full of; filled with; replete with (interchangeable 韌) soft but tough; elastic}
3E3F CJK UNIFIED IDEOGRAPH-#3E3F, IRGKangXi=0701.171, Def{(abbreviated form of 犢) a calf}
3E40 CJK UNIFIED IDEOGRAPH-#3E40, IRGKangXi=0701.171, Def{a cow; female of an animal}
3E41 CJK UNIFIED IDEOGRAPH-#3E41, IRGKangXi=0702.050, Def{generally called the animals (cattle, sheep etc.) with mixed color of white and black}
3E42 CJK UNIFIED IDEOGRAPH-#3E42, IRGKangXi=0702.120, Def{untamed and indocility cattle, huge; big; large}
3E43 CJK UNIFIED IDEOGRAPH-#3E43, IRGKangXi=0702.130, Def{two brindled cows to plough face to each other, to cultivate; to plant after the second time of ploughing; (Cant.) to push lightly, flick off}
3E44 CJK UNIFIED IDEOGRAPH-#3E44, IRGKangXi=0702.160, Def{a kind of cattle; a cattle of short statured}
3E45 CJK UNIFIED IDEOGRAPH-#3E45, IRGKangXi=0702.230, Def{(same as 總) all; general}
3E46 CJK UNIFIED IDEOGRAPH-#3E46, IRGKangXi=0702.260, Def{the coulter of a plough; head of a brindled cow, a kind of cattle}
3E47 CJK UNIFIED IDEOGRAPH-#3E47, IRGKangXi=0702.300, Def{(same as standard form 犍) a kind of cattle, castrated bull; ox}
3E48 CJK UNIFIED IDEOGRAPH-#3E48, IRGKangXi=0703.041, Def{(same as 膧 犛) a black ox, a yak, name of a state in old times}
3E49 CJK UNIFIED IDEOGRAPH-#3E49, IRGKangXi=0703.090, Def{a legendary animal (some kind looks like a cow) with three feet, a wild ox; the bison}
3E4A CJK UNIFIED IDEOGRAPH-#3E4A, IRGKangXi=0703.120, Def{a cattle of white color}
3E4B CJK UNIFIED IDEOGRAPH-#3E4B, IRGKangXi=0703.230, Def{a cow; ox; bull; cattle, a cow with no tail}
3E4C CJK UNIFIED IDEOGRAPH-#3E4C, IRGKangXi=0703.290, Def{cattle; domestic animals, a cow; female of an animal}
3E4D CJK UNIFIED IDEOGRAPH-#3E4D, IRGKangXi=0703.300, Def{bull; bullock; ox (castrated)}
3E4E CJK UNIFIED IDEOGRAPH-#3E4E, IRGKangXi=0703.360, Def{bull; bullock; ox (castrated), 求子牛}
3E4F CJK UNIFIED IDEOGRAPH-#3E4F, IRGKangXi=0703.370, Def{a tame, gentle and yielding cow; good, virtuous}
3E50 CJK UNIFIED IDEOGRAPH-#3E50, IRGKangXi=0704.040, Def{to tow a boat against the current in swift stream from the bank; to track a boat in the shallow area}
3E51 CJK UNIFIED IDEOGRAPH-#3E51, IRGKangXi=0704.110, Def{(same as large seal type form of 牭) four-year old cattle; two and a half year-old cattle}
3E52 CJK UNIFIED IDEOGRAPH-#3E52, IRGKangXi=0704.180, Def{bull; bullock, a calf}
3E53 CJK UNIFIED IDEOGRAPH-#3E53, IRGKangXi=0704.200, Def{a kind of animal, cattle and horses to prance; to be lively}
3E54 CJK UNIFIED IDEOGRAPH-#3E54, IRGKangXi=0704.230, Def{cattle with long back, a white cattle; a cattle having white color on the back}
3E55 CJK UNIFIED IDEOGRAPH-#3E55, IRGKangXi=0704.250, Def{inferior; mean, minor}
3E56 CJK UNIFIED IDEOGRAPH-#3E56, IRGKangXi=0704.260, Def{(same as 豢) to feed animals with grains, to feed; to rear; to raise; to support}
3E57 CJK UNIFIED IDEOGRAPH-#3E57, IRGKangXi=0704.330, Def{cattle and sheep have no children}
3E58 CJK UNIFIED IDEOGRAPH-#3E58, IRGKangXi=0704.340, Def{shinbone of a cattle, a kind of cattle}
3E59 CJK UNIFIED IDEOGRAPH-#3E59, IRGKangXi=0704.420, Def{sound of calling to a calf, calf, the lowing of an ox}
3E5A CJK UNIFIED IDEOGRAPH-#3E5A, IRGKangXi=0705.120, Def{(same as 㹙) sound of calling to a calf, calf, the lowing of an ox}
3E5B CJK UNIFIED IDEOGRAPH-#3E5B, IRGKangXi=0705.170, Def{a tame, gentle and yielding cow; good, virtuous}
3E5C CJK UNIFIED IDEOGRAPH-#3E5C, IRGKangXi=0707.080, Def{(same as 犾) two dogs barking at each other, rude spoken language; rude talk; unpolished; rustic and coarse}
3E5D CJK UNIFIED IDEOGRAPH-#3E5D, IRGKangXi=0707.140, Def{a legendary wild animal, snake; serpent}
3E5E CJK UNIFIED IDEOGRAPH-#3E5E, IRGKangXi=0707.150, Def{(same as 狺) to bark (said of a dog), (same as 齗) gums (of the teeth), (same as 啀) dogs to fight with gnashing and grinning; looks very angry}
3E5F CJK UNIFIED IDEOGRAPH-#3E5F, IRGKangXi=0707.160, Def{a kind of animal, the animals to run about wildly, wild; mad; crazy, (same as 獪) cunning; artful; crafty}
3E60 CJK UNIFIED IDEOGRAPH-#3E60, IRGKangXi=0707.220, Def{(same as 豚) a small big; a sucking pig, to shuffle along without lifting the feet}
3E61 CJK UNIFIED IDEOGRAPH-#3E61, IRGKangXi=0708.090, Def{a rush and impatient disposition, (non-classical form 狷) rash; quick-tempered; narrow-minded, honest and straightforward}
3E62 CJK UNIFIED IDEOGRAPH-#3E62, IRGKangXi=0708.100, Def{a large ape found in Western China, a precious mammal found in Africa}
3E63 CJK UNIFIED IDEOGRAPH-#3E63, IRGKangXi=0708.150, Def{a kind of dog}
3E64 CJK UNIFIED IDEOGRAPH-#3E64, IRGKangXi=0708.180, Def{(same as 怯) lacking in courage; cowardly, nervous; socially timid; fright}
3E65 CJK UNIFIED IDEOGRAPH-#3E65, IRGKangXi=0708.190, Def{a yellow dog with black head}
3E66 CJK UNIFIED IDEOGRAPH-#3E66, IRGKangXi=0708.230, Def{dog with a short tail, (same as non-classical form 貂) the sable; the marten; mink}
3E67 CJK UNIFIED IDEOGRAPH-#3E67, IRGKangXi=0709.030, Def{(non-classical abbreviated form) fox-like animal}
3E68 CJK UNIFIED IDEOGRAPH-#3E68, IRGKangXi=0709.181, Def{(same as 貁 non-classical form 狖) a kind of animal (looks like weasel), an ape with black color and long tail}
3E69 CJK UNIFIED IDEOGRAPH-#3E69, IRGKangXi=0709.181
3E6A CJK UNIFIED IDEOGRAPH-#3E6A, IRGKangXi=0709.181, Def{(same as 狦) a fierce dog, a kind of animal (looks like a wolf)}
3E6B CJK UNIFIED IDEOGRAPH-#3E6B, IRGKangXi=0710.030, Def{a kind of animal (same as 夷) generally called for minority groups in old China}
3E6C CJK UNIFIED IDEOGRAPH-#3E6C, IRGKangXi=0710.180, Def{a kind of animal (of dog tribe)}
3E6D CJK UNIFIED IDEOGRAPH-#3E6D, IRGKangXi=0710.220, Def{a kind of monkey, fox}
3E6E CJK UNIFIED IDEOGRAPH-#3E6E, IRGKangXi=0710.280, Def{a kind of animal (of mule tribe) (same as 貘) tapir}
3E6F CJK UNIFIED IDEOGRAPH-#3E6F, IRGKangXi=0711.060, Def{(same as 貅) a kind of animal like a tiger, a fabulous fierce beast -- thus -- fierce; valiant; heroic; brave}
3E70 CJK UNIFIED IDEOGRAPH-#3E70, IRGKangXi=0711.101
3E71 CJK UNIFIED IDEOGRAPH-#3E71, IRGKangXi=0711.140, Def{name of a dog in the period of Warring States (usually known as the magpie, jackdaw), generally called for dogs}
3E72 CJK UNIFIED IDEOGRAPH-#3E72, IRGKangXi=0711.150, Def{a frightened pig or dog, bark, a running pig}
3E73 CJK UNIFIED IDEOGRAPH-#3E73, IRGKangXi=0711.170, Def{a kind of ape (gibbon)}
3E74 CJK UNIFIED IDEOGRAPH-#3E74, IRGKangXi=0711.220, Def{a kind of animal, fierce dog; small and cute}
3E75 CJK UNIFIED IDEOGRAPH-#3E75, IRGKangXi=0711.270, Def{a hunting-dog, to hunt in winter, imperial tour}
3E76 CJK UNIFIED IDEOGRAPH-#3E76, IRGKangXi=0712.060, Def{a kind of wild animal (monkey tribe)}
3E77 CJK UNIFIED IDEOGRAPH-#3E77, IRGKangXi=0712.071, Def{(same as 狶 豨) swine; pig; hog; big wild pig, sound used in calling pigs, a legendary appellation of an emperor in ancient times}
3E78 CJK UNIFIED IDEOGRAPH-#3E78, IRGKangXi=0712.071, Def{(same as 猊) the lion, a wild beast or wild horse which can do 500 lǐ in a day}
3E79 CJK UNIFIED IDEOGRAPH-#3E79, IRGKangXi=0712.110, Def{a kind of dog}
3E7A CJK UNIFIED IDEOGRAPH-#3E7A, IRGKangXi=0712.120, Def{dog to take food, a biter (said of a dog)}
3E7B CJK UNIFIED IDEOGRAPH-#3E7B, IRGKangXi=0712.180, Def{(same as 猧) puppy, of dog tribe}
3E7C CJK UNIFIED IDEOGRAPH-#3E7C, IRGKangXi=0712.190, Def{a kind of animal}
3E7D CJK UNIFIED IDEOGRAPH-#3E7D, IRGKangXi=0713.020, Def{to bite (said of a dog), dog to take food}
3E7E CJK UNIFIED IDEOGRAPH-#3E7E, IRGKangXi=0713.190, Def{cunning; crafty; sly; wily; artful, vigorous; strong}
3E7F CJK UNIFIED IDEOGRAPH-#3E7F, IRGKangXi=0713.240, Def{a fierce dog, bite fiercely (said of dog), hunting in winter, a kind of beast}
3E80 CJK UNIFIED IDEOGRAPH-#3E80, IRGKangXi=0714.051, Def{(same as 猱) a monkey with yellow hair, crocodile; alligator}
3E81 CJK UNIFIED IDEOGRAPH-#3E81, IRGKangXi=0714.090, Def{a kind of beast}
3E82 CJK UNIFIED IDEOGRAPH-#3E82, IRGKangXi=0714.220, Def{big and strong sheep, a ewe or she-goat, kind of dog, a dog barking loudly}
3E83 CJK UNIFIED IDEOGRAPH-#3E83, IRGKangXi=0715.100, Def{(same as 狗) a dog; canine}
3E84 CJK UNIFIED IDEOGRAPH-#3E84, IRGKangXi=0715.120, Def{(same as 狳) (a variant of 貐) a kind of beast}
3E85 CJK UNIFIED IDEOGRAPH-#3E85, IRGKangXi=0716.050, Def{(same as 猴) the monkey}
3E86 CJK UNIFIED IDEOGRAPH-#3E86, IRGKangXi=0716.051
3E87 CJK UNIFIED IDEOGRAPH-#3E87, IRGKangXi=0716.150, Def{a warden; (in ancient China) minister of public works, to observe; to watch}
3E88 CJK UNIFIED IDEOGRAPH-#3E88, IRGKangXi=0716.290, Def{to go hunting; to go on a hunting expedition}
3E89 CJK UNIFIED IDEOGRAPH-#3E89, IRGKangXi=0716.350, Def{a fierce animal of the dog tribe; with dog's head and horse tail; with yellow and black colors}
3E8A CJK UNIFIED IDEOGRAPH-#3E8A, IRGKangXi=0717.020, Def{a beast (looks like lion)}
3E8B CJK UNIFIED IDEOGRAPH-#3E8B, IRGKangXi=0717.060, Def{pig; hog}
3E8C CJK UNIFIED IDEOGRAPH-#3E8C, IRGKangXi=0717.090, Def{endless barking, two dogs are fighting}
3E8D CJK UNIFIED IDEOGRAPH-#3E8D, IRGKangXi=0717.111, Def{a kind of otter}
3E8E CJK UNIFIED IDEOGRAPH-#3E8E, IRGKangXi=0717.230, Def{fierce beast; the zebu; or humped-ox}
3E8F CJK UNIFIED IDEOGRAPH-#3E8F, IRGKangXi=0717.260, Def{(same as 貗) the badger; a wild bear}
3E90 CJK UNIFIED IDEOGRAPH-#3E90, IRGKangXi=0717.280, Def{(same as U+7360 獠) a general term for minority groups (southwest area) in old China; to hunt at night by torches; lascivious, weird, allurements}
3E91 CJK UNIFIED IDEOGRAPH-#3E91, IRGKangXi=0717.340, Def{dogs wormed through a narrow place, to oppress cruelly; to injure heartlessly, dogs biting, dog's hair, legendary monster (looks like a dwarf), bark}
3E92 CJK UNIFIED IDEOGRAPH-#3E92, IRGKangXi=0717.360, Def{barking of a frightened dog, to confuse; disorder; disturbance, cunning; artful; crafty, to fail and be exposed}
3E93 CJK UNIFIED IDEOGRAPH-#3E93, IRGKangXi=0718.041, Def{(same as 蹟) footprints; traces (interchangeable 僄) agile; adroit; nimble, frivolous; playful; flippant}
3E94 CJK UNIFIED IDEOGRAPH-#3E94, IRGKangXi=0718.190, Def{a kind of beast, a legendary monster}
3E95 CJK UNIFIED IDEOGRAPH-#3E95, IRGKangXi=0718.230, Def{the fighting sound of dogs, agile; sprightly}
3E96 CJK UNIFIED IDEOGRAPH-#3E96, IRGKangXi=0718.260, Def{dogs barking, little dog; a puppy, name of a place in today's Henan Province Xinyexian}
3E97 CJK UNIFIED IDEOGRAPH-#3E97, IRGKangXi=0719.030, Def{to bite; to gnaw; dogs bite}
3E98 CJK UNIFIED IDEOGRAPH-#3E98, IRGKangXi=0719.060, Def{a kind of animal}
3E99 CJK UNIFIED IDEOGRAPH-#3E99, IRGKangXi=0719.131
3E9A CJK UNIFIED IDEOGRAPH-#3E9A, IRGKangXi=0719.150, Def{(same as 獺) an otter}
3E9B CJK UNIFIED IDEOGRAPH-#3E9B, IRGKangXi=0719.170, Def{name of a village in Henan Province, name of a pavilion}
3E9C CJK UNIFIED IDEOGRAPH-#3E9C, IRGKangXi=0719.210, Def{fierce dog with long shaggy hair; an old name for a part of the Miao nationality (in southwestern China)}
3E9D CJK UNIFIED IDEOGRAPH-#3E9D, IRGKangXi=0720.080, Def{sound of a tiger, a fierce dog barking endlessly, a fierce dog, sound of a dog's biting}
3E9E CJK UNIFIED IDEOGRAPH-#3E9E, IRGKangXi=0720.120, Def{to sign in lamentation, crying of a piggy, a kind of beast}
3E9F CJK UNIFIED IDEOGRAPH-#3E9F, IRGKangXi=0720.180, Def{a kind of beast; looks like a deer; white-tailed (same as 蠗) a kind of animal (of monkey tribe)}
3EA0 CJK UNIFIED IDEOGRAPH-#3EA0, IRGKangXi=0721.010, Def{a black ape with long tail, (same as 貁) a kind of animal (of weasel tribe)}
3EA1 CJK UNIFIED IDEOGRAPH-#3EA1, IRGKangXi=0721.040, Def{a kind of beast}
3EA2 CJK UNIFIED IDEOGRAPH-#3EA2, IRGKangXi=0721.101, Def{a kind of animal; body is smaller than a giraffe; two short horns on the head; back and legs with black and white stripes spaced in-between}
3EA3 CJK UNIFIED IDEOGRAPH-#3EA3, IRGKangXi=0721.130, Def{a kind of animal; pig hog}
3EA4 CJK UNIFIED IDEOGRAPH-#3EA4, IRGKangXi=0721.150, Def{a kind of beast}
3EA5 CJK UNIFIED IDEOGRAPH-#3EA5, IRGKangXi=0721.160, Def{bark, (same as 毚) a cunning hare; a wily rabbit}
3EA6 CJK UNIFIED IDEOGRAPH-#3EA6, IRGKangXi=0721.171, Def{a running animal, monkey climbing along the trees, a rutted dog}
3EA7 CJK UNIFIED IDEOGRAPH-#3EA7, IRGKangXi=0722.071, Def{a white-yellow dog, a yellow dog, a mad dog, a group of dogs bark wildly}
3EA8 CJK UNIFIED IDEOGRAPH-#3EA8, IRGKangXi=0727.050, Def{fine stone like jade}
3EA9 CJK UNIFIED IDEOGRAPH-#3EA9, IRGKangXi=0727.060, Def{jade articles, (same as 玌) a kind of jade}
3EAA CJK UNIFIED IDEOGRAPH-#3EAA, IRGKangXi=0727.090, Def{(same as 璞) an uncarved of unpolished jade or gem}
3EAB CJK UNIFIED IDEOGRAPH-#3EAB, IRGKangXi=0727.101, Def{(same as 玌) a kind of jade}
3EAC CJK UNIFIED IDEOGRAPH-#3EAC, IRGKangXi=0727.110, Def{(same as 珙) a large piece of jade-stone}
3EAD CJK UNIFIED IDEOGRAPH-#3EAD, IRGKangXi=0727.180, Def{a kind of jade}
3EAE CJK UNIFIED IDEOGRAPH-#3EAE, IRGKangXi=0727.250, Def{(same as 玗) fine stone like jade}
3EAF CJK UNIFIED IDEOGRAPH-#3EAF, IRGKangXi=0727.251
3EB0 CJK UNIFIED IDEOGRAPH-#3EB0, IRGKangXi=0728.010, Def{(same as 瑡) a kind of jade}
3EB1 CJK UNIFIED IDEOGRAPH-#3EB1, IRGKangXi=0728.020, Def{a kind of jade}
3EB2 CJK UNIFIED IDEOGRAPH-#3EB2, IRGKangXi=0728.110, Def{(ancient form of 鈕) buttons}
3EB3 CJK UNIFIED IDEOGRAPH-#3EB3, IRGKangXi=0728.211, Def{(same as 玫) black mica (a sparkling red gem)}
3EB4 CJK UNIFIED IDEOGRAPH-#3EB4, IRGKangXi=0728.211, Def{name of a spirit; a god, a kind of jade}
3EB5 CJK UNIFIED IDEOGRAPH-#3EB5, IRGKangXi=0728.211, Def{(corrupted form of 玖) black jade-stone}
3EB6 CJK UNIFIED IDEOGRAPH-#3EB6, IRGKangXi=0728.211
3EB7 CJK UNIFIED IDEOGRAPH-#3EB7, IRGKangXi=0728.250, Def{(same as 珬) a kind of jade (jade decorative article for a horse)}
3EB8 CJK UNIFIED IDEOGRAPH-#3EB8, IRGKangXi=0728.290, Def{a kind of jade}
3EB9 CJK UNIFIED IDEOGRAPH-#3EB9, IRGKangXi=0729.140, Def{a kind of jade, a conical cap (wore on ceremonious occasions in ancient times) decorated with jade}
3EBA CJK UNIFIED IDEOGRAPH-#3EBA, IRGKangXi=0730.070, Def{(ancient abbreviated form of 瑁) a very precious piece of jade worn by ancient emperors to match tables borne by the nobles}
3EBB CJK UNIFIED IDEOGRAPH-#3EBB, IRGKangXi=0728.211
3EBC CJK UNIFIED IDEOGRAPH-#3EBC, IRGKangXi=0730.081
3EBD CJK UNIFIED IDEOGRAPH-#3EBD, IRGKangXi=0730.081
3EBE CJK UNIFIED IDEOGRAPH-#3EBE, IRGKangXi=0730.081
3EBF CJK UNIFIED IDEOGRAPH-#3EBF, IRGKangXi=0730.250, Def{a jade-like stone, a kind of jade}
3EC0 CJK UNIFIED IDEOGRAPH-#3EC0, IRGKangXi=0731.020, Def{fine stone which is little less valuable than jade}
3EC1 CJK UNIFIED IDEOGRAPH-#3EC1, IRGKangXi=0731.030, Def{(non-classical form of 瓌) (variant of 瑰) a kind of jasper; stone which is little less valuable than jade, extraordinary; fabulous or admirable}
3EC2 CJK UNIFIED IDEOGRAPH-#3EC2, IRGKangXi=0731.090, Def{a kind of jade, (non-classical form of 玭) pearls}
3EC3 CJK UNIFIED IDEOGRAPH-#3EC3, IRGKangXi=0732.021, Def{crooked; to bend; to twist, involved; indirect}
3EC4 CJK UNIFIED IDEOGRAPH-#3EC4, IRGKangXi=0732.021, Def{(same as 寶) treasure, precious; valuable, respectable; honorable}
3EC5 CJK UNIFIED IDEOGRAPH-#3EC5, IRGKangXi=0732.021, Def{(simplified form of U+74AF 璯) jade decorated cap, used in person's name}
3EC6 CJK UNIFIED IDEOGRAPH-#3EC6, IRGKangXi=0732.021
3EC7 CJK UNIFIED IDEOGRAPH-#3EC7, IRGKangXi=0732.021
3EC8 CJK UNIFIED IDEOGRAPH-#3EC8, IRGKangXi=0732.021
3EC9 CJK UNIFIED IDEOGRAPH-#3EC9, IRGKangXi=0732.100, Def{fine jade}
3ECA CJK UNIFIED IDEOGRAPH-#3ECA, IRGKangXi=0732.130, Def{(same as non-classical form of 璊) a red colored jade}
3ECB CJK UNIFIED IDEOGRAPH-#3ECB, IRGKangXi=0732.140, Def{(abbreviated form) (interchangeable 瓬) a kind of jade; a kind of red colored precious stone}
3ECC CJK UNIFIED IDEOGRAPH-#3ECC, IRGKangXi=0732.210, Def{fine jade}
3ECD CJK UNIFIED IDEOGRAPH-#3ECD, IRGKangXi=0732.270, Def{(interchangeable 珸) name of a sword, fine stone which is little less valuable than jade}
3ECE CJK UNIFIED IDEOGRAPH-#3ECE, IRGKangXi=0733.070, Def{(ancient form of 瑟) a large horizontal musical instrument, usually have 25 strings which pass over bridges for tuning; anciently this instrument had 50 strings, but the number varies}
3ECF CJK UNIFIED IDEOGRAPH-#3ECF, IRGKangXi=0733.151, Def{(same as U+73B2 玲) tinkling of jade pendants}
3ED0 CJK UNIFIED IDEOGRAPH-#3ED0, IRGKangXi=0733.151
3ED1 CJK UNIFIED IDEOGRAPH-#3ED1, IRGKangXi=0733.180, Def{a kind of jade}
3ED2 CJK UNIFIED IDEOGRAPH-#3ED2, IRGKangXi=0733.190, Def{red colored jade, even; regular; equal}
3ED3 CJK UNIFIED IDEOGRAPH-#3ED3, IRGKangXi=0733.200, Def{a fine jade, the stripes (lines) on a jade}
3ED4 CJK UNIFIED IDEOGRAPH-#3ED4, IRGKangXi=0734.070, Def{a kind of jade}
3ED5 CJK UNIFIED IDEOGRAPH-#3ED5, IRGKangXi=0734.080, Def{a kind of jade}
3ED6 CJK UNIFIED IDEOGRAPH-#3ED6, IRGKangXi=0734.150, Def{a kind of jade}
3ED7 CJK UNIFIED IDEOGRAPH-#3ED7, IRGKangXi=0736.060, Def{(same as 琲) a string of pearls; 500 beads}
3ED8 CJK UNIFIED IDEOGRAPH-#3ED8, IRGKangXi=0736.081
3ED9 CJK UNIFIED IDEOGRAPH-#3ED9, IRGKangXi=0736.081
3EDA CJK UNIFIED IDEOGRAPH-#3EDA, IRGKangXi=0736.081
3EDB CJK UNIFIED IDEOGRAPH-#3EDB, IRGKangXi=0736.081
3EDC CJK UNIFIED IDEOGRAPH-#3EDC, IRGKangXi=0736.081
3EDD CJK UNIFIED IDEOGRAPH-#3EDD, IRGKangXi=0736.130, Def{a kind of jade}
3EDE CJK UNIFIED IDEOGRAPH-#3EDE, IRGKangXi=0737.040, Def{(interchangeable 彬) flourishing and radiant of the colorful jades, ornamental, refined, lines or stripes on the jade}
3EDF CJK UNIFIED IDEOGRAPH-#3EDF, IRGKangXi=0738.060, Def{a kind of jade}
3EE0 CJK UNIFIED IDEOGRAPH-#3EE0, IRGKangXi=0738.100, Def{jade hanging on the sides of the cap in ancient times; used to plug the ears, to take; to receive}
3EE1 CJK UNIFIED IDEOGRAPH-#3EE1, IRGKangXi=0738.180, Def{(interchangeable 鞢) a jade ring with a small segment cut off; used to indicate separation, a thumb-ring worn by archers, tinkling sound of clicking (said of metal), (corrupted form of 渫) to remove; to eliminate}
3EE2 CJK UNIFIED IDEOGRAPH-#3EE2, IRGKangXi=0738.181
3EE3 CJK UNIFIED IDEOGRAPH-#3EE3, IRGKangXi=0738.181
3EE4 CJK UNIFIED IDEOGRAPH-#3EE4, IRGKangXi=0738.181
3EE5 CJK UNIFIED IDEOGRAPH-#3EE5, IRGKangXi=0738.181
3EE6 CJK UNIFIED IDEOGRAPH-#3EE6, IRGKangXi=0738.181
3EE7 CJK UNIFIED IDEOGRAPH-#3EE7, IRGKangXi=0740.010, Def{a kind of jade}
3EE8 CJK UNIFIED IDEOGRAPH-#3EE8, IRGKangXi=0740.111, Def{fine stone a little less valuable than jade}
3EE9 CJK UNIFIED IDEOGRAPH-#3EE9, IRGKangXi=0740.111
3EEA CJK UNIFIED IDEOGRAPH-#3EEA, IRGKangXi=0740.111, Def{a kind of jade (same as 璿 璇) fine jade}
3EEB CJK UNIFIED IDEOGRAPH-#3EEB, IRGKangXi=0740.150, Def{(same as ancient form of 珌) a gem or ornament on a scabbard}
3EEC CJK UNIFIED IDEOGRAPH-#3EEC, IRGKangXi=0740.200, Def{a kind of jade}
3EED CJK UNIFIED IDEOGRAPH-#3EED, IRGKangXi=0740.220, Def{(same as 璱) clear; pure and fine (said of jade)}
3EEE CJK UNIFIED IDEOGRAPH-#3EEE, IRGKangXi=0741.020, Def{(same as non-classical form of 璨) bright and brilliant; lustrous and luminous}
3EEF CJK UNIFIED IDEOGRAPH-#3EEF, IRGKangXi=0741.120, Def{(same as non-classical form of 㻬) a kind of jade}
3EF0 CJK UNIFIED IDEOGRAPH-#3EF0, IRGKangXi=0741.140, Def{decorated jade or ornament on a scabbard}
3EF1 CJK UNIFIED IDEOGRAPH-#3EF1, IRGKangXi=0741.200, Def{to circle; to revolve round; to surround; to encase, (same as 璡) used in person's name, jade like stone}
3EF2 CJK UNIFIED IDEOGRAPH-#3EF2, IRGKangXi=0741.220, Def{a rite or service for beginning of Autumn (date marking the beginning of one of the 24 seasonal periods in a year, falling normally on August 8 or 9)}
3EF3 CJK UNIFIED IDEOGRAPH-#3EF3, IRGKangXi=0741.221
3EF4 CJK UNIFIED IDEOGRAPH-#3EF4, IRGKangXi=0741.221
3EF5 CJK UNIFIED IDEOGRAPH-#3EF5, IRGKangXi=0741.310, Def{a kind of jade}
3EF6 CJK UNIFIED IDEOGRAPH-#3EF6, IRGKangXi=0741.340, Def{a jade tube with white and nature colors, jade with the colour of nature (green; blue; black), sapphire}
3EF7 CJK UNIFIED IDEOGRAPH-#3EF7, IRGKangXi=0742.040, Def{name of a place (be famous for fine jade), a kind of jade}
3EF8 CJK UNIFIED IDEOGRAPH-#3EF8, IRGKangXi=0742.070, Def{jade like stone}
3EF9 CJK UNIFIED IDEOGRAPH-#3EF9, IRGKangXi=0742.150, Def{a crack; a crevice, spare time; leisure}
3EFA CJK UNIFIED IDEOGRAPH-#3EFA, IRGKangXi=0742.180, Def{a kind of jade}
3EFB CJK UNIFIED IDEOGRAPH-#3EFB, IRGKangXi=0742.251
3EFC CJK UNIFIED IDEOGRAPH-#3EFC, IRGKangXi=0742.251
3EFD CJK UNIFIED IDEOGRAPH-#3EFD, IRGKangXi=0743.050, Def{(same as U+74BF 璿 U+7487 璇) fine jade}
3EFE CJK UNIFIED IDEOGRAPH-#3EFE, IRGKangXi=0743.100, Def{jade articles, fine stone which is little less valuable than jade}
3EFF CJK UNIFIED IDEOGRAPH-#3EFF, IRGKangXi=0743.190, Def{a kind of jade}
3F00 CJK UNIFIED IDEOGRAPH-#3F00, IRGKangXi=0744.021
3F01 CJK UNIFIED IDEOGRAPH-#3F01, IRGKangXi=0744.021
3F02 CJK UNIFIED IDEOGRAPH-#3F02, IRGKangXi=0744.021, Def{(same as 璺) a crack in jade (or porcelain)}
3F03 CJK UNIFIED IDEOGRAPH-#3F03, IRGKangXi=0745.021
3F04 CJK UNIFIED IDEOGRAPH-#3F04, IRGKangXi=0745.051, Def{(same as 璣) pearls, jade, etc. which are not quite circular}
3F05 CJK UNIFIED IDEOGRAPH-#3F05, IRGKangXi=0745.051
3F06 CJK UNIFIED IDEOGRAPH-#3F06, IRGKangXi=0745.051
3F07 CJK UNIFIED IDEOGRAPH-#3F07, IRGKangXi=0745.080, Def{(same as non-classical form of 瓗) (same as 瓊) fine jade or agate; red stone, exquisite; fine, variety of jade}
3F08 CJK UNIFIED IDEOGRAPH-#3F08, IRGKangXi=0745.291
3F09 CJK UNIFIED IDEOGRAPH-#3F09, IRGKangXi=0746.040, Def{green colored melons}
3F0A CJK UNIFIED IDEOGRAPH-#3F0A, IRGKangXi=0746.060, Def{a kind of melon}
3F0B CJK UNIFIED IDEOGRAPH-#3F0B, IRGKangXi=0746.130, Def{the cucumber, dried (preserved) vegetable}
3F0C CJK UNIFIED IDEOGRAPH-#3F0C, IRGKangXi=0746.150, Def{lot of melons growing but the root is weak and feeble, wearied and worn}
3F0D CJK UNIFIED IDEOGRAPH-#3F0D, IRGKangXi=0746.170, Def{pulp of a melon, a kind of melon}
3F0E CJK UNIFIED IDEOGRAPH-#3F0E, IRGKangXi=0746.180, Def{a small melon}
3F0F CJK UNIFIED IDEOGRAPH-#3F0F, IRGKangXi=0747.020, Def{a bruised, overripe or rotten melon (傷瓜,傷熟瓜)}
3F10 CJK UNIFIED IDEOGRAPH-#3F10, IRGKangXi=0747.090, Def{a kind of white melon}
3F11 CJK UNIFIED IDEOGRAPH-#3F11, IRGKangXi=0747.110, Def{pulp of a melon}
3F12 CJK UNIFIED IDEOGRAPH-#3F12, IRGKangXi=0747.120, Def{a kind of big melon, a long melon}
3F13 CJK UNIFIED IDEOGRAPH-#3F13, IRGKangXi=0747.150, Def{melon seeds, a kind of melon}
3F14 CJK UNIFIED IDEOGRAPH-#3F14, IRGKangXi=0747.180, Def{a kind of melon}
3F15 CJK UNIFIED IDEOGRAPH-#3F15, IRGKangXi=0747.240, Def{pulp of a melon}
3F16 CJK UNIFIED IDEOGRAPH-#3F16, IRGKangXi=0747.340, Def{gourd used as a ladle or dipper}
3F17 CJK UNIFIED IDEOGRAPH-#3F17, IRGKangXi=0748.010, Def{a brick}
3F18 CJK UNIFIED IDEOGRAPH-#3F18, IRGKangXi=0748.030, Def{of mountain; high and lofty}
3F19 CJK UNIFIED IDEOGRAPH-#3F19, IRGKangXi=0748.060, Def{(same as ancient form of 甃) brickwork of a well, to repair a well, to lay bricks}
3F1A CJK UNIFIED IDEOGRAPH-#3F1A, IRGKangXi=0748.130, Def{earthware; pottery (a basin; a pot; a bowl. a cistern; a crock), (interchangeable 缸) a big earthen jar}
3F1B CJK UNIFIED IDEOGRAPH-#3F1B, IRGKangXi=0748.170, Def{(same as 鈃) long necked wine vessels (bottle; pot; jar; and cups etc.)}
3F1C CJK UNIFIED IDEOGRAPH-#3F1C, IRGKangXi=0748.250, Def{(same as 盎) a basin; pot; bowl or dish, sleek; well-favoured, abundant, a musical instrument}
3F1D CJK UNIFIED IDEOGRAPH-#3F1D, IRGKangXi=0748.270, Def{(same as 盌) (a variant of 碗) a bowl; a basin; a cup; a dish}
3F1E CJK UNIFIED IDEOGRAPH-#3F1E, IRGKangXi=0748.310, Def{jar with a small mouth, an earthen jar, a jar for the ashes of the dead}
3F1F CJK UNIFIED IDEOGRAPH-#3F1F, IRGKangXi=0748.320, Def{a tiled house, brick wall of a well}
3F20 CJK UNIFIED IDEOGRAPH-#3F20, IRGKangXi=0748.330, Def{a crock with narrow opening, an earthen jar}
3F21 CJK UNIFIED IDEOGRAPH-#3F21, IRGKangXi=0748.340, Def{a small jar with a small mouth, a bottle; pitcher; jug etc.}
3F22 CJK UNIFIED IDEOGRAPH-#3F22, IRGKangXi=0749.010, Def{bricks (same as U+74F5 瓵) an earthen jar, a jar for the ashes of the dead}
3F23 CJK UNIFIED IDEOGRAPH-#3F23, IRGKangXi=0749.030, Def{bricks of the well; brick wall of a well, to build a well, to construct with bricks}
3F24 CJK UNIFIED IDEOGRAPH-#3F24, IRGKangXi=0749.050, Def{earthenware (a basin; a pot; a bowl; a crock with a narrow opening)}
3F25 CJK UNIFIED IDEOGRAPH-#3F25, IRGKangXi=0749.090, Def{bottle, earthenware}
3F26 CJK UNIFIED IDEOGRAPH-#3F26, IRGKangXi=0749.131, Def{earthenware; pottery (bottle; pitcher; jug; vase; crock with a narrow opening)}
3F27 CJK UNIFIED IDEOGRAPH-#3F27, IRGKangXi=0749.170, Def{tiles in cylinder shape; used to build a palace; a temple or a shrine}
3F28 CJK UNIFIED IDEOGRAPH-#3F28, IRGKangXi=0749.190, Def{a water-jar with ears for carrying it}
3F29 CJK UNIFIED IDEOGRAPH-#3F29, IRGKangXi=0749.210, Def{concave channels of tiling, a long-necked jar, utensil; instrument; implement}
3F2A CJK UNIFIED IDEOGRAPH-#3F2A, IRGKangXi=0749.290, Def{fragments of a tile haven't t been kiln-dried completely, tiles overlapping}
3F2B CJK UNIFIED IDEOGRAPH-#3F2B, IRGKangXi=0749.310, Def{earthenware or pottery with big opening, wine utensil (containers) with big opening}
3F2C CJK UNIFIED IDEOGRAPH-#3F2C, IRGKangXi=0749.320, Def{earthenware (a basin; a pot; a bowl; a crock etc.)}
3F2D CJK UNIFIED IDEOGRAPH-#3F2D, IRGKangXi=0749.330, Def{(interchangeable 墊) to steady by putting something underneath, to shore up; to prop}
3F2E CJK UNIFIED IDEOGRAPH-#3F2E, IRGKangXi=0749.380, Def{to rub out the filth with broken tiles, brickwork of a well, to repair a well, to lay bricks}
3F2F CJK UNIFIED IDEOGRAPH-#3F2F, IRGKangXi=0749.420, Def{earthenware (jar with a small mouth)}
3F30 CJK UNIFIED IDEOGRAPH-#3F30, IRGKangXi=0749.440, Def{an earthenware jar with a small mouth and two or four ears; used to store water for fire prevention in ancient times}
3F31 CJK UNIFIED IDEOGRAPH-#3F31, IRGKangXi=0750.050, Def{to manage or regulate the leather material, soft, hunting clothes (leather trousers)}
3F32 CJK UNIFIED IDEOGRAPH-#3F32, IRGKangXi=0750.080, Def{sound of broken tiles, thin tiles}
3F33 CJK UNIFIED IDEOGRAPH-#3F33, IRGKangXi=0750.100, Def{earthenware (a basin; a pot; a bowl; a crock etc.)}
3F34 CJK UNIFIED IDEOGRAPH-#3F34, IRGKangXi=0750.110, Def{an earthen vessel; a basin; a pot; a jar}
3F35 CJK UNIFIED IDEOGRAPH-#3F35, IRGKangXi=0750.170, Def{a small basin; a bowl. a cup; a pot; a jar}
3F36 CJK UNIFIED IDEOGRAPH-#3F36, IRGKangXi=0750.180, Def{earthenware (a basin; a pot; a bowl; a crock etc.)}
3F37 CJK UNIFIED IDEOGRAPH-#3F37, IRGKangXi=0750.210, Def{a basin, (same as 甎) a brick; a square tile}
3F38 CJK UNIFIED IDEOGRAPH-#3F38, IRGKangXi=0751.010, Def{earthenware (a basin; a pot; a bowl; a crock etc.)}
3F39 CJK UNIFIED IDEOGRAPH-#3F39, IRGKangXi=0751.050, Def{earthenware; pottery, a kind container; vessel}
3F3A CJK UNIFIED IDEOGRAPH-#3F3A, IRGKangXi=0751.110, Def{porcelain; porcelain ware; chinaware; china, a small bottle with ears}
3F3B CJK UNIFIED IDEOGRAPH-#3F3B, IRGKangXi=0751.160, Def{earthenware (a jar; a jar with a small mouth and two or four ears; a pot; a pitcher)}
3F3C CJK UNIFIED IDEOGRAPH-#3F3C, IRGKangXi=0751.180, Def{(same as 瓢) a ladle (often made of dried calabash or gourd)}
3F3D CJK UNIFIED IDEOGRAPH-#3F3D, IRGKangXi=0751.210, Def{fragments of tile incompletely kiln-dried; to scrub out filth with broken tiles}
3F3E CJK UNIFIED IDEOGRAPH-#3F3E, IRGKangXi=0752.010, Def{bricks}
3F3F CJK UNIFIED IDEOGRAPH-#3F3F, IRGKangXi=0752.040, Def{the brick wall of a well, earthenware (a jar; a jar with a small mouth and two or four ears; a pot; a pitcher)}
3F40 CJK UNIFIED IDEOGRAPH-#3F40, IRGKangXi=0752.060, Def{earthenware (a basin; a pot; a bowl; a crock etc.)}
3F41 CJK UNIFIED IDEOGRAPH-#3F41, IRGKangXi=0752.080, Def{(same as 鬲) a large earthen pot, a large iron cauldron used in old time}
3F42 CJK UNIFIED IDEOGRAPH-#3F42, IRGKangXi=0752.100, Def{a cracked container, a movement of a musical composition in Tang Dynasty}
3F43 CJK UNIFIED IDEOGRAPH-#3F43, IRGKangXi=0752.120, Def{a big piece of brick}
3F44 CJK UNIFIED IDEOGRAPH-#3F44, IRGKangXi=0752.150, Def{an earthen jar, crack of earthen container, hoarse voice}
3F45 CJK UNIFIED IDEOGRAPH-#3F45, IRGKangXi=0752.151, Def{vessel for ceremony in ancient times}
3F46 CJK UNIFIED IDEOGRAPH-#3F46, IRGKangXi=0752.200, Def{earthenware; a bowl; a cup, tiles with a circular facade at one end used along the edge of palace eaves during the Han Dynasty}
3F47 CJK UNIFIED IDEOGRAPH-#3F47, IRGKangXi=0752.240, Def{unburnt tiles, earthenware}
3F48 CJK UNIFIED IDEOGRAPH-#3F48, IRGKangXi=0752.280, Def{a large earthenware jar with a small mouth and two or four ears; a big pot; bottle; pitcher; jug; vase}
3F49 CJK UNIFIED IDEOGRAPH-#3F49, IRGKangXi=0752.290, Def{a big jar; a big basin}
3F4A CJK UNIFIED IDEOGRAPH-#3F4A, IRGKangXi=0753.080, Def{cracking of tile}
3F4B CJK UNIFIED IDEOGRAPH-#3F4B, IRGKangXi=0753.150, Def{a kind of tile, earthenware, a cylinder shaped tile used for chimney}
3F4C CJK UNIFIED IDEOGRAPH-#3F4C, IRGKangXi=0753.141, Def{the ridge of a roof}
3F4D CJK UNIFIED IDEOGRAPH-#3F4D, IRGKangXi=0754.100, Def{(same as 坩) an earthen vessel}
3F4E CJK UNIFIED IDEOGRAPH-#3F4E, IRGKangXi=0754.170, Def{the profundity of the harem, the mysterious and profound of the forbidden palace, the flourishing and exuberant of the flame}
3F4F CJK UNIFIED IDEOGRAPH-#3F4F, IRGKangXi=0754.181
3F50 CJK UNIFIED IDEOGRAPH-#3F50, IRGKangXi=0754.190, Def{(same as 馦) fragrant}
3F51 CJK UNIFIED IDEOGRAPH-#3F51, IRGKangXi=0754.230, Def{delicious; tasty (same as 醰) full flavoured; rich}
3F52 CJK UNIFIED IDEOGRAPH-#3F52, IRGKangXi=0755.160, Def{(same as 甥) the children of a sister, a son-in-law}
3F53 CJK UNIFIED IDEOGRAPH-#3F53, IRGKangXi=0755.201
3F54 CJK UNIFIED IDEOGRAPH-#3F54, IRGKangXi=0755.201
3F55 CJK UNIFIED IDEOGRAPH-#3F55, IRGKangXi=0759.050, Def{new branches growing (said of trees), hollow; empty}
3F56 CJK UNIFIED IDEOGRAPH-#3F56, IRGKangXi=0759.081, Def{(same as 男) a human; a man; a boy (non-classical form of 留) to remain; to stay, to keep, to preserve}
3F57 CJK UNIFIED IDEOGRAPH-#3F57, IRGKangXi=0759.081
3F58 CJK UNIFIED IDEOGRAPH-#3F58, IRGKangXi=0760.050, Def{borderland; the frontier, a pit; a cave, a narrow path in the field, administrative unit for rural community of 傣族 (under the feudal system), pond; a marsh, saltpond, (same as 䴚) salt marsh}
3F59 CJK UNIFIED IDEOGRAPH-#3F59, IRGKangXi=0760.320, Def{(same as 畯) official in charge of farmlands in ancient times; a bailiff or landlord, rustic; crude (ancient form of 允) to allow; to grant}
3F5A CJK UNIFIED IDEOGRAPH-#3F5A, IRGKangXi=0760.341, Def{agricultural implements; farm tools, name of a place}
3F5B CJK UNIFIED IDEOGRAPH-#3F5B, IRGKangXi=0760.360, Def{narrow trail of path in the fields, a name of an old county in today's Yunnan Province}
3F5C CJK UNIFIED IDEOGRAPH-#3F5C, IRGKangXi=0761.110
3F5D CJK UNIFIED IDEOGRAPH-#3F5D, IRGKangXi=0762.080, Def{to sink; to fall, to entrap, to crush; to capture}
3F5E CJK UNIFIED IDEOGRAPH-#3F5E, IRGKangXi=0763.031, Def{(abbreviated form of 留) to remain; to stay, to detain, to leave behind}
3F5F CJK UNIFIED IDEOGRAPH-#3F5F, IRGKangXi=0763.040, Def{to till or to plough the fields (same as 埒) an enclosure, an embankment, a dike}
3F60 CJK UNIFIED IDEOGRAPH-#3F60, IRGKangXi=0764.020, Def{connection of the ditches or waterways}
3F61 CJK UNIFIED IDEOGRAPH-#3F61, IRGKangXi=0764.100, Def{field, block up the water to irrigate the fields}
3F62 CJK UNIFIED IDEOGRAPH-#3F62, IRGKangXi=0764.110, Def{to plough the fields}
3F63 CJK UNIFIED IDEOGRAPH-#3F63, IRGKangXi=0764.140, Def{(ancient form of 域) a frontier; a boundary; a region; a country, to keep within bounds}
3F64 CJK UNIFIED IDEOGRAPH-#3F64, IRGKangXi=0764.170, Def{a small plot of land}
3F65 CJK UNIFIED IDEOGRAPH-#3F65, IRGKangXi=0765.140, Def{fertile fields; good land (during the Epoch of Spring and Autumn) name of a place in Zheng Guo (today's Henan Province LuShan Xian}
3F66 CJK UNIFIED IDEOGRAPH-#3F66, IRGKangXi=0765.190, Def{reclaimed land; flat and in even level}
3F67 CJK UNIFIED IDEOGRAPH-#3F67, IRGKangXi=0766.030, Def{name of a place}
3F68 CJK UNIFIED IDEOGRAPH-#3F68, IRGKangXi=0766.060, Def{fragmented; uncultivated and desolated fields, to eliminate; remove, to clean (interchangeable 瘥) an epidemic; a plague}
3F69 CJK UNIFIED IDEOGRAPH-#3F69, IRGKangXi=0766.110, Def{connected banks of earth or paths in the field}
3F6A CJK UNIFIED IDEOGRAPH-#3F6A, IRGKangXi=0766.260, Def{paddy field; rice field}
3F6B CJK UNIFIED IDEOGRAPH-#3F6B, IRGKangXi=0766.290, Def{seeds of the plant allied to the water-lily; used when ground into meal; as a coarse food, also as medicine (same as 壅) to block up, to bank up roots of plants, to impede (flow, etc.)}
3F6C CJK UNIFIED IDEOGRAPH-#3F6C, IRGKangXi=0766.310, Def{(same as 副) to assist, secondary (in importance), rich and dense}
3F6D CJK UNIFIED IDEOGRAPH-#3F6D, IRGKangXi=0766.350, Def{vacant lot or space, soft land; land on the river side}
3F6E CJK UNIFIED IDEOGRAPH-#3F6E, IRGKangXi=0767.041, Def{(same as 星) a point of light, stars; planets, a spark}
3F6F CJK UNIFIED IDEOGRAPH-#3F6F, IRGKangXi=0767.100
3F70 CJK UNIFIED IDEOGRAPH-#3F70, IRGKangXi=0768.010, Def{(corrupted form) carved window frame on a door, the windows with scattered or dispersed frames, a wine filterer}
3F71 CJK UNIFIED IDEOGRAPH-#3F71, IRGKangXi=0769.060, Def{griping colic, a swelling, a little painful; some not serious}
3F72 CJK UNIFIED IDEOGRAPH-#3F72, IRGKangXi=0769.070, Def{(same as U+3F71 㽱) griping colic, illness; disease}
3F73 CJK UNIFIED IDEOGRAPH-#3F73, IRGKangXi=0769.110, Def{illness; disease}
3F74 CJK UNIFIED IDEOGRAPH-#3F74, IRGKangXi=0769.160, Def{illness; disease}
3F75 CJK UNIFIED IDEOGRAPH-#3F75, IRGKangXi=0769.210, Def{(same as non-classical form of 莊) solemn, large farmhouse}
3F76 CJK UNIFIED IDEOGRAPH-#3F76, IRGKangXi=0769.221
3F77 CJK UNIFIED IDEOGRAPH-#3F77, IRGKangXi=0769.250, Def{dropsy; hydrophilic swelling}
3F78 CJK UNIFIED IDEOGRAPH-#3F78, IRGKangXi=0769.290, Def{(corrupted form 瘎) diseases of the abdomen, illness, sick for the second time; get sick again}
3F79 CJK UNIFIED IDEOGRAPH-#3F79, IRGKangXi=0770.010, Def{to faint, to feel nausea, abuse; vilification}
3F7A CJK UNIFIED IDEOGRAPH-#3F7A, IRGKangXi=0770.040, Def{to be weakened by disease; feeble; lean (a dialect) poor in quality}
3F7B CJK UNIFIED IDEOGRAPH-#3F7B, IRGKangXi=0770.150, Def{diseases; illness}
3F7C CJK UNIFIED IDEOGRAPH-#3F7C, IRGKangXi=0770.191
3F7D CJK UNIFIED IDEOGRAPH-#3F7D, IRGKangXi=0770.220, Def{(same as U+75FC 痼) chronic disease, a sore in the mouth of a child}
3F7E CJK UNIFIED IDEOGRAPH-#3F7E, IRGKangXi=0770.250, Def{a kind of woman's disease, to cut apart, to break}
3F7F CJK UNIFIED IDEOGRAPH-#3F7F, IRGKangXi=0770.260, Def{(same as 瘑) ulcer; sore; boil, illness; disease}
3F80 CJK UNIFIED IDEOGRAPH-#3F80, IRGKangXi=0771.010, Def{(interchangeable 怯) to be weakened by disease; weak; feeble, lean; emaciated, illness; disease; sick}
3F81 CJK UNIFIED IDEOGRAPH-#3F81, IRGKangXi=0771.050, Def{walking like mad; mad}
3F82 CJK UNIFIED IDEOGRAPH-#3F82, IRGKangXi=0771.060, Def{illness; disease (a dialect) a chronic disease; never recover after a long illness}
3F83 CJK UNIFIED IDEOGRAPH-#3F83, IRGKangXi=0771.110, Def{hump-backed}
3F84 CJK UNIFIED IDEOGRAPH-#3F84, IRGKangXi=0771.140, Def{(ancient form same as 讟) to libel; to slander; complaints; grumblings, blames or curses; to blame, (same as 怞) to grieve, sorrowful, sad and weary}
3F85 CJK UNIFIED IDEOGRAPH-#3F85, IRGKangXi=0771.170, Def{defects; flaws, illness; disease}
3F86 CJK UNIFIED IDEOGRAPH-#3F86, IRGKangXi=0771.200, Def{(non-classical form) (said of some kinds of reptiles or insects) ecdysis; to cast off the skin; to molt}
3F87 CJK UNIFIED IDEOGRAPH-#3F87, IRGKangXi=0771.280, Def{illness; disease; ailment}
3F88 CJK UNIFIED IDEOGRAPH-#3F88, IRGKangXi=0771.310, Def{hunchbacked; curved spine, short, dropsy}
3F89 CJK UNIFIED IDEOGRAPH-#3F89, IRGKangXi=0771.320, Def{thin; lean; slim; emaciated}
3F8A CJK UNIFIED IDEOGRAPH-#3F8A, IRGKangXi=0772.130, Def{(same as 瘦 瘠) thin; lean; slim; emaciated, a kind of skin disease, a chill, a cold, malaria, (interchangeable 瘶,瘷) cough; chilly disease; catching cold}
3F8B CJK UNIFIED IDEOGRAPH-#3F8B, IRGKangXi=0772.140, Def{dysentery; diarrhea, a sore; a boil; an ulcer}
3F8C CJK UNIFIED IDEOGRAPH-#3F8C, IRGKangXi=0772.220, Def{(same as 癬) ringworm, used for various diseases of the skin}
3F8D CJK UNIFIED IDEOGRAPH-#3F8D, IRGKangXi=0772.260, Def{diseases, sticking up or protruding of the big bottom}
3F8E CJK UNIFIED IDEOGRAPH-#3F8E, IRGKangXi=0772.290, Def{(a dialect) sickness; disease, favus (a dialect) everything is good except this particular one}
3F8F CJK UNIFIED IDEOGRAPH-#3F8F, IRGKangXi=0773.020, Def{illness; disease; sickness, idiotic; crazy; insane; silly}
3F90 CJK UNIFIED IDEOGRAPH-#3F90, IRGKangXi=0773.060, Def{ulcer, a sore; swelling and sores caused by varnish-poisoning; pestilence; epidemic, leprosy, (interchangeable 勵) to exhort to great effort; to urge}
3F91 CJK UNIFIED IDEOGRAPH-#3F91, IRGKangXi=0773.100, Def{fat; plump, a chill, a cold, malaria, a wart; a pimple, a round lump}
3F92 CJK UNIFIED IDEOGRAPH-#3F92, IRGKangXi=0773.140, Def{illness; disease, could not have access to ...}
3F93 CJK UNIFIED IDEOGRAPH-#3F93, IRGKangXi=0773.160, Def{fatigue; exhaustion; weariness, ache of the joints (of bones); gout, melancholy; depressed, anger; fury, hatred; bitterness}
3F94 CJK UNIFIED IDEOGRAPH-#3F94, IRGKangXi=0773.200, Def{long illness, ulcer; sore; boil}
3F95 CJK UNIFIED IDEOGRAPH-#3F95, IRGKangXi=0773.210, Def{a chill, a cold, malaria, shivering with cold, a shudder; a shiver; tremble}
3F96 CJK UNIFIED IDEOGRAPH-#3F96, IRGKangXi=0773.220, Def{grieved; distressed; ill with grief and exhausted}
3F97 CJK UNIFIED IDEOGRAPH-#3F97, IRGKangXi=0773.230, Def{eye disease; strabismus; squint; to look askance (a dialect) bright; light; brilliant, bright eyes}
3F98 CJK UNIFIED IDEOGRAPH-#3F98, IRGKangXi=0773.330, Def{(same as U+6897 梗) ailment; bane; distress; (Cant.) a bruise}
3F99 CJK UNIFIED IDEOGRAPH-#3F99, IRGKangXi=0773.360, Def{(same as 脪) Erysipelas, sloughing of an ulcer, painful; aching}
3F9A CJK UNIFIED IDEOGRAPH-#3F9A, IRGKangXi=0774.010, Def{itchy; ticklish}
3F9B CJK UNIFIED IDEOGRAPH-#3F9B, IRGKangXi=0774.050, Def{ugly; bad-looking}
3F9C CJK UNIFIED IDEOGRAPH-#3F9C, IRGKangXi=0774.060, Def{weak breath of a sick man}
3F9D CJK UNIFIED IDEOGRAPH-#3F9D, IRGKangXi=0774.100, Def{(corrupted form) to leak out; to ejaculate, to scatter; to disperse; to vent}
3F9E CJK UNIFIED IDEOGRAPH-#3F9E, IRGKangXi=0774.130, Def{illness; disease (interchangeable 庮) stinking smell of the decayed wood}
3F9F CJK UNIFIED IDEOGRAPH-#3F9F, IRGKangXi=0774.190, Def{(said of illness) relapse; to relapse (a dialect) relapse of typhoid fever (typhus)}
3FA0 CJK UNIFIED IDEOGRAPH-#3FA0, IRGKangXi=0774.200, Def{a fever}
3FA1 CJK UNIFIED IDEOGRAPH-#3FA1, IRGKangXi=0774.230, Def{ulcer; sore; boil}
3FA2 CJK UNIFIED IDEOGRAPH-#3FA2, IRGKangXi=0774.260, Def{a malignant disease, a chronic disease}
3FA3 CJK UNIFIED IDEOGRAPH-#3FA3, IRGKangXi=0774.300, Def{to shiver; to shudder; to tremble because of a malaria and a very bad cold}
3FA4 CJK UNIFIED IDEOGRAPH-#3FA4, IRGKangXi=0775.050, Def{disease of the throat}
3FA5 CJK UNIFIED IDEOGRAPH-#3FA5, IRGKangXi=0775.190, Def{bellyache}
3FA6 CJK UNIFIED IDEOGRAPH-#3FA6, IRGKangXi=0775.210, Def{scab over a sore, scar of an ulcer, weak; feeble}
3FA7 CJK UNIFIED IDEOGRAPH-#3FA7, IRGKangXi=0775.250, Def{favus, swelling of the glands, a lump; scrofulous swellings}
3FA8 CJK UNIFIED IDEOGRAPH-#3FA8, IRGKangXi=0775.290, Def{weak; feeble, to mourn, ulcer; cancer; carbuncle, short}
3FA9 CJK UNIFIED IDEOGRAPH-#3FA9, IRGKangXi=0776.010, Def{malignant boils}
3FAA CJK UNIFIED IDEOGRAPH-#3FAA, IRGKangXi=0776.140, Def{lean; thin; slim}
3FAB CJK UNIFIED IDEOGRAPH-#3FAB, IRGKangXi=0776.220, Def{hemiplegia -- paralysis of half of one's body}
3FAC CJK UNIFIED IDEOGRAPH-#3FAC, IRGKangXi=0776.330, Def{(same as 痎) malaria}
3FAD CJK UNIFIED IDEOGRAPH-#3FAD, IRGKangXi=0776.390, Def{to shrink; to contract; to deflate; to shorten; to reduce in length}
3FAE CJK UNIFIED IDEOGRAPH-#3FAE, IRGKangXi=0776.440, Def{(same as U+7640 癀) jaundice}
3FAF CJK UNIFIED IDEOGRAPH-#3FAF, IRGKangXi=0777.090, Def{to suffer a stroke of paralysis or apoplexy, disease of the private part (of the human body)}
3FB0 CJK UNIFIED IDEOGRAPH-#3FB0, IRGKangXi=0777.100, Def{choke, lump in throat; food stuck in the throat}
3FB1 CJK UNIFIED IDEOGRAPH-#3FB1, IRGKangXi=0777.211, Def{(abbreviated form 癈) incurable disease}
3FB2 CJK UNIFIED IDEOGRAPH-#3FB2, IRGKangXi=0780.091
3FB3 CJK UNIFIED IDEOGRAPH-#3FB3, IRGKangXi=0777.211
3FB4 CJK UNIFIED IDEOGRAPH-#3FB4, IRGKangXi=0777.211, Def{the erythema of acne rosacea}
3FB5 CJK UNIFIED IDEOGRAPH-#3FB5, IRGKangXi=0777.211, Def{a dark colored birth-mark}
3FB6 CJK UNIFIED IDEOGRAPH-#3FB6, IRGKangXi=0777.220, Def{disease of the knee}
3FB7 CJK UNIFIED IDEOGRAPH-#3FB7, IRGKangXi=0777.250, Def{to ache}
3FB8 CJK UNIFIED IDEOGRAPH-#3FB8, IRGKangXi=0777.280, Def{a scabby, itching disease}
3FB9 CJK UNIFIED IDEOGRAPH-#3FB9, IRGKangXi=0777.330, Def{lean; thin; slim, illness; disease, an epidemic; a pestilence}
3FBA CJK UNIFIED IDEOGRAPH-#3FBA, IRGKangXi=0777.370, Def{eye disease, with noxious air on, disease of an animal, decayed sore}
3FBB CJK UNIFIED IDEOGRAPH-#3FBB, IRGKangXi=0778.190, Def{not any progressive, swelling, a mark of scar on the skin}
3FBC CJK UNIFIED IDEOGRAPH-#3FBC, IRGKangXi=0778.210, Def{to suffer a stroke of paralysis or apoplexy}
3FBD CJK UNIFIED IDEOGRAPH-#3FBD, IRGKangXi=0778.260, Def{(same as 膇) swelling feet, disease of the private part of the human body}
3FBE CJK UNIFIED IDEOGRAPH-#3FBE, IRGKangXi=0778.270, Def{disease of the throat, something stuck in the throat, to be stung, emaciated; illness of losing flesh}
3FBF CJK UNIFIED IDEOGRAPH-#3FBF, IRGKangXi=0779.010, Def{roar of asthma or expectorate from a dying person}
3FC0 CJK UNIFIED IDEOGRAPH-#3FC0, IRGKangXi=0779.041, Def{(same as U+764D 癍) unhealthy marks on the skin; blotches; pustules}
3FC1 CJK UNIFIED IDEOGRAPH-#3FC1, IRGKangXi=0779.041
3FC2 CJK UNIFIED IDEOGRAPH-#3FC2, IRGKangXi=0779.100, Def{(abbreviated form of 癥) obstruction of the bowels}
3FC3 CJK UNIFIED IDEOGRAPH-#3FC3, IRGKangXi=0779.120, Def{an ox-headed boil, dysentery; diarrhea, popular name for women's peculiar diseases}
3FC4 CJK UNIFIED IDEOGRAPH-#3FC4, IRGKangXi=0779.160, Def{the groans or moans from a person in a critical condition}
3FC5 CJK UNIFIED IDEOGRAPH-#3FC5, IRGKangXi=0779.180, Def{(same as 癬) ringworm, used for various diseases of the skin}
3FC6 CJK UNIFIED IDEOGRAPH-#3FC6, IRGKangXi=0779.300, Def{damage by the pests to the rice seedling, boil; ulcer; sore}
3FC7 CJK UNIFIED IDEOGRAPH-#3FC7, IRGKangXi=0779.310, Def{paralysis, rheumatism, having no sense of feeling; numb}
3FC8 CJK UNIFIED IDEOGRAPH-#3FC8, IRGKangXi=0780.050, Def{(non-classical form of 癰) an ulcer; an abscess; a carbuncle; sometimes use for a cancer, loss of the sense of smell}
3FC9 CJK UNIFIED IDEOGRAPH-#3FC9, IRGKangXi=0780.130, Def{disease of the private part of the human body}
3FCA CJK UNIFIED IDEOGRAPH-#3FCA, IRGKangXi=0780.210, Def{very painful; agonizing (same as 憯) sad; sorrowful; grieved}
3FCB CJK UNIFIED IDEOGRAPH-#3FCB, IRGKangXi=0781.240, Def{(same as U+7619 瘙) a kind of skin disease (like sores from scabies)}
3FCC CJK UNIFIED IDEOGRAPH-#3FCC, IRGKangXi=0781.280, Def{(interchangeable U+3FBE 㾾) lump in the throat, sting of a poisonous insect (scorpion)}
3FCD CJK UNIFIED IDEOGRAPH-#3FCD, IRGKangXi=0781.370, Def{(same as U+75A5 疥) scabies}
3FCE CJK UNIFIED IDEOGRAPH-#3FCE, IRGKangXi=0781.420, Def{sick; depress and melancholy, swelling, heat rashes; heat spots; prickly heat}
3FCF CJK UNIFIED IDEOGRAPH-#3FCF, IRGKangXi=0782.021, Def{paralysis of the body}
3FD0 CJK UNIFIED IDEOGRAPH-#3FD0, IRGKangXi=0782.021
3FD1 CJK UNIFIED IDEOGRAPH-#3FD1, IRGKangXi=0782.080, Def{illness; disease; ailment}
3FD2 CJK UNIFIED IDEOGRAPH-#3FD2, IRGKangXi=0782.100, Def{illness; disease; ailment, disease of the internal organs, disease of the abdomen, palpitation of the heart}
3FD3 CJK UNIFIED IDEOGRAPH-#3FD3, IRGKangXi=0782.120, Def{disease of the hoof}
3FD4 CJK UNIFIED IDEOGRAPH-#3FD4, IRGKangXi=0782.311, Def{a wart; a pimple; a pustule on the skin}
3FD5 CJK UNIFIED IDEOGRAPH-#3FD5, IRGKangXi=0782.320, Def{ulcer; sore; boil, a wound}
3FD6 CJK UNIFIED IDEOGRAPH-#3FD6, IRGKangXi=0782.330, Def{carbuncle (interchangeable 廬) used in naming of a place, a swelling of the abdomen from constipation}
3FD7 CJK UNIFIED IDEOGRAPH-#3FD7, IRGKangXi=0782.340, Def{(same as U+3FC9 㿉) disease of the private part of the human body}
3FD8 CJK UNIFIED IDEOGRAPH-#3FD8, IRGKangXi=0783.030, Def{to get sick; to fall ill}
3FD9 CJK UNIFIED IDEOGRAPH-#3FD9, IRGKangXi=0783.110, Def{full; filled, full of air or gas, a heart full of (enthusiasm, sorrow, etc.)}
3FDA CJK UNIFIED IDEOGRAPH-#3FDA, IRGKangXi=0783.150, Def{sickness; ailment; disease}
3FDB CJK UNIFIED IDEOGRAPH-#3FDB, IRGKangXi=0783.190, Def{carbuncle, sallow and emaciated, scrofulous swellings}
3FDC CJK UNIFIED IDEOGRAPH-#3FDC, IRGKangXi=0783.271, Def{(non-classical form of 癟) shrivelled up; empty; limp; flat; not full; sunken}
3FDD CJK UNIFIED IDEOGRAPH-#3FDD, IRGKangXi=0786.050, Def{(ancient form of 香) sweet; fragrant, delicious, incense}
3FDE CJK UNIFIED IDEOGRAPH-#3FDE, IRGKangXi=0787.020, Def{(non-classical form of 耄) an old man of eighty, aged}
3FDF CJK UNIFIED IDEOGRAPH-#3FDF, IRGKangXi=0787.090, Def{(same as U+767D 白) he bright, white moon; a pure glistening white, splendid, white}
3FE0 CJK UNIFIED IDEOGRAPH-#3FE0, IRGKangXi=0788.121
3FE1 CJK UNIFIED IDEOGRAPH-#3FE1, IRGKangXi=0788.121, Def{(corrupted form of 兜) a helmet; a head-covering, a small pocket in clothes}
3FE2 CJK UNIFIED IDEOGRAPH-#3FE2, IRGKangXi=0788.260, Def{(same as 耀) to shine; to dazzle, to show off}
3FE3 CJK UNIFIED IDEOGRAPH-#3FE3, IRGKangXi=0788.300, Def{white color}
3FE4 CJK UNIFIED IDEOGRAPH-#3FE4, IRGKangXi=0788.340, Def{white, clear}
3FE5 CJK UNIFIED IDEOGRAPH-#3FE5, IRGKangXi=0789.020, Def{shine; clear and pure white feather (same as 皬) white, white but not pure}
3FE6 CJK UNIFIED IDEOGRAPH-#3FE6, IRGKangXi=0789.130, Def{to tell; to inform; to report; to accuse}
3FE7 CJK UNIFIED IDEOGRAPH-#3FE7, IRGKangXi=0789.290, Def{(same as 疇) fields, a pronoun; who?, formerly; previously, a class; a rank}
3FE8 CJK UNIFIED IDEOGRAPH-#3FE8, IRGKangXi=0790.050, Def{(same as 皪) small stones, gravel, shingle}
3FE9 CJK UNIFIED IDEOGRAPH-#3FE9, IRGKangXi=0790.100, Def{light; bright, white color}
3FEA CJK UNIFIED IDEOGRAPH-#3FEA, IRGKangXi=0790.170, Def{skin disease}
3FEB CJK UNIFIED IDEOGRAPH-#3FEB, IRGKangXi=0790.230, Def{to split tangled hemp}
3FEC CJK UNIFIED IDEOGRAPH-#3FEC, IRGKangXi=0790.250, Def{nasal disease, (same as 疤) a scar; a birthmark (same as 巴) to hope; to wish}
3FED CJK UNIFIED IDEOGRAPH-#3FED, IRGKangXi=0791.010, Def{cracks; creases; wrinkles; rumples on the skin; surname; (Cant.) hoarse}
3FEE CJK UNIFIED IDEOGRAPH-#3FEE, IRGKangXi=0791.070, Def{green; blue; black, extravasation of blood, to look pale or pallid (of the facial complexion)}
3FEF CJK UNIFIED IDEOGRAPH-#3FEF, IRGKangXi=0791.110, Def{wide piece of leather}
3FF0 CJK UNIFIED IDEOGRAPH-#3FF0, IRGKangXi=0791.150, Def{durable and solid leather}
3FF1 CJK UNIFIED IDEOGRAPH-#3FF1, IRGKangXi=0791.170, Def{dry and decayed, to peel off the skin; to scrape; to pare}
3FF2 CJK UNIFIED IDEOGRAPH-#3FF2, IRGKangXi=0791.300, Def{(same as 欹) a fierce dog, an interjection of pleasure -- Bravo! Good! (interchangeable 騎) to sit astride on}
3FF3 CJK UNIFIED IDEOGRAPH-#3FF3, IRGKangXi=0791.410
3FF4 CJK UNIFIED IDEOGRAPH-#3FF4, IRGKangXi=0791.310, Def{wide piece of leather}
3FF5 CJK UNIFIED IDEOGRAPH-#3FF5, IRGKangXi=0791.330, Def{soft leather}
3FF6 CJK UNIFIED IDEOGRAPH-#3FF6, IRGKangXi=0791.440, Def{(same as 幫) the sides of a shoe or gutter}
3FF7 CJK UNIFIED IDEOGRAPH-#3FF7, IRGKangXi=0791.450, Def{(same as 糙) rough; coarse; inferior unpolished rice, goose flesh}
3FF8 CJK UNIFIED IDEOGRAPH-#3FF8, IRGKangXi=0792.070, Def{to cast off the skin; to molt, skin}
3FF9 CJK UNIFIED IDEOGRAPH-#3FF9, IRGKangXi=0792.150, Def{jutting on the epidermis or the cuticle (of plants); (Cant.) skin peeling off}
3FFA CJK UNIFIED IDEOGRAPH-#3FFA, IRGKangXi=0792.240, Def{jutting on the epidermis; swelling, wounded; (Cant.) courageous}
3FFB CJK UNIFIED IDEOGRAPH-#3FFB, IRGKangXi=0792.390, Def{a tray; a plate; a dish, large bowl}
3FFC CJK UNIFIED IDEOGRAPH-#3FFC, IRGKangXi=0792.440, Def{(a non-classical form) a bowl, a small shallow container; a small cup}
3FFD CJK UNIFIED IDEOGRAPH-#3FFD, IRGKangXi=0793.020, Def{a small bowl; a small basin}
3FFE CJK UNIFIED IDEOGRAPH-#3FFE, IRGKangXi=0793.160, Def{vessel; container}
3FFF CJK UNIFIED IDEOGRAPH-#3FFF, IRGKangXi=0793.200, Def{wine cups}
4000 CJK UNIFIED IDEOGRAPH-#4000, IRGKangXi=0793.230, Def{cups; small cups}
4001 CJK UNIFIED IDEOGRAPH-#4001, IRGKangXi=0794.060, Def{a small bowl; a small basin, a kind of vessel to remove (or to strain out) the water}
4002 CJK UNIFIED IDEOGRAPH-#4002, IRGKangXi=0794.140, Def{a big basin}
4003 CJK UNIFIED IDEOGRAPH-#4003, IRGKangXi=0795.020, Def{a kind of vessel; a container}
4004 CJK UNIFIED IDEOGRAPH-#4004, IRGKangXi=0795.080, Def{(same as U+995B 饛) container full of food}
4005 CJK UNIFIED IDEOGRAPH-#4005, IRGKangXi=0795.160, Def{vessel; container}
4006 CJK UNIFIED IDEOGRAPH-#4006, IRGKangXi=0795.230, Def{(standard form of 盡) to exhaust; to complete; to finish; all; totally}
4007 CJK UNIFIED IDEOGRAPH-#4007, IRGKangXi=0796.110, Def{a pot, an earthen pot; a deep cooking pot}
4008 CJK UNIFIED IDEOGRAPH-#4008, IRGKangXi=0796.130, Def{a vessel; a container, a kind of animal with red hair looks like hedgehog}
4009 CJK UNIFIED IDEOGRAPH-#4009, IRGKangXi=0797.080, Def{food containers (bowl; basin, etc.) used in ancient times}
400A CJK UNIFIED IDEOGRAPH-#400A, IRGKangXi=0798.100, Def{a kind of water containers (to keep warm in cold days); a mental hot-water bottle, to stir or scratch and make it muddy or turbid}
400B CJK UNIFIED IDEOGRAPH-#400B, IRGKangXi=0798.171, Def{(same as U+9E7D 鹽) salt}
400C CJK UNIFIED IDEOGRAPH-#400C, IRGKangXi=0798.211
400D CJK UNIFIED IDEOGRAPH-#400D, IRGKangXi=0798.240, Def{vessel; container; a box; a chest; a trunk, a small cup}
400E CJK UNIFIED IDEOGRAPH-#400E, IRGKangXi=0799.040, Def{(same as non-classical form of 眄) to look askance; to ogle}
400F CJK UNIFIED IDEOGRAPH-#400F, IRGKangXi=0799.070, Def{to move one's eyes (same as 眴) to express or indicate with eyes (interchangeable 眩) dizzy; giddy}
4010 CJK UNIFIED IDEOGRAPH-#4010, IRGKangXi=0799.110, Def{to regard; to look up to, to look, to take a glance at}
4011 CJK UNIFIED IDEOGRAPH-#4011, IRGKangXi=0799.120, Def{to gouge out an eye or eyes (a corrupted form)}
4012 CJK UNIFIED IDEOGRAPH-#4012, IRGKangXi=0799.140, Def{gloomy; dark; obscure}
4013 CJK UNIFIED IDEOGRAPH-#4013, IRGKangXi=0799.160, Def{to turn one's eyes to; big eyes}
4014 CJK UNIFIED IDEOGRAPH-#4014, IRGKangXi=0799.230, Def{to look, to confuse; to dazzle}
4015 CJK UNIFIED IDEOGRAPH-#4015, IRGKangXi=0801.010, Def{angry glances; to look angrily, eyeballs}
4016 CJK UNIFIED IDEOGRAPH-#4016, IRGKangXi=0801.030, Def{to look up to; to respect}
4017 CJK UNIFIED IDEOGRAPH-#4017, IRGKangXi=0801.070, Def{to look askance at, to dislike, a kind of eye disease (tears all the times), pretty eyes, (same as 瞲) to look at in surprise}
4018 CJK UNIFIED IDEOGRAPH-#4018, IRGKangXi=0801.100, Def{(a corrupted abbreviated form) to cover the line of vision or sight --the straight line between an object and one's eyes, to look steadily at, to look at impatiently}
4019 CJK UNIFIED IDEOGRAPH-#4019, IRGKangXi=0801.180, Def{to look at, to examine; to observe; to survey}
401A CJK UNIFIED IDEOGRAPH-#401A, IRGKangXi=0802.110, Def{to lift up the eyes and look around}
401B CJK UNIFIED IDEOGRAPH-#401B, IRGKangXi=0803.040, Def{to look far with the eyes half-closed (to narrow the eyes), to look at for a long time, daybreak; dawn, to look not straight forward, to disturb; to dizzy; dark, to look at ferocious}
401C CJK UNIFIED IDEOGRAPH-#401C, IRGKangXi=0803.050, Def{down; to see obscurely, (same as U+401B 䀛)}
401D CJK UNIFIED IDEOGRAPH-#401D, IRGKangXi=0803.151
401E CJK UNIFIED IDEOGRAPH-#401E, IRGKangXi=0803.151, Def{with heavy eyelids}
401F CJK UNIFIED IDEOGRAPH-#401F, IRGKangXi=0803.170, Def{sight blurred; unclear; dim}
4020 CJK UNIFIED IDEOGRAPH-#4020, IRGKangXi=0803.220, Def{a Chinese family name, the timid look of a bird; (of birds) to look around, nervous (same as 瞿) shocked or scared}
4021 CJK UNIFIED IDEOGRAPH-#4021, IRGKangXi=0804.170, Def{(same as 覘) to spy on; to see; to observe; to inspect, to cast the eyes down, the insight of the debauchee, to move the eyes; to look around}
4022 CJK UNIFIED IDEOGRAPH-#4022, IRGKangXi=0805.090, Def{(same as 瞬) to glance; to blink; wink,, (interchangeable 眴 瞚), to indicate one's wish or intention by expressions of the eyes}
4023 CJK UNIFIED IDEOGRAPH-#4023, IRGKangXi=0805.210, Def{to look steadily at, ashamed, to look at ferocious}
4024 CJK UNIFIED IDEOGRAPH-#4024, IRGKangXi=0806.020, Def{insight (version) of jealousy}
4025 CJK UNIFIED IDEOGRAPH-#4025, IRGKangXi=0806.081, Def{(simplified form of 䁻) beautiful eyes, to look at just for a short time}
4026 CJK UNIFIED IDEOGRAPH-#4026, IRGKangXi=0806.081, Def{(a dialect) to stare at}
4027 CJK UNIFIED IDEOGRAPH-#4027, IRGKangXi=0806.100, Def{sight blurred; obscure and dim; unclear}
4028 CJK UNIFIED IDEOGRAPH-#4028, IRGKangXi=0806.140, Def{to look, an angry look; to look at some one angrily, sight blurred; obscure and dim; unclear}
4029 CJK UNIFIED IDEOGRAPH-#4029, IRGKangXi=0806.150, Def{to look askance, to look; to see, to hope, big eyes}
402A CJK UNIFIED IDEOGRAPH-#402A, IRGKangXi=0806.200, Def{birds flying up and down, to look at; to regard; to inspect}
402B CJK UNIFIED IDEOGRAPH-#402B, IRGKangXi=0806.310, Def{narrow and dim eye sight; having one eye smaller than the other, one-eyed, wink of the eyelashes, sleepy; drowsy; dim; vague and hazy}
402C CJK UNIFIED IDEOGRAPH-#402C, IRGKangXi=0806.350, Def{to glance; to wink, having one eye smaller than the other, one eyed, gazing into distance, sight blurred; obscure and dim; unclear}
402D CJK UNIFIED IDEOGRAPH-#402D, IRGKangXi=0806.360, Def{with big eyes}
402E CJK UNIFIED IDEOGRAPH-#402E, IRGKangXi=0807.130, Def{sight blurred; obscure and dim; obscured}
402F CJK UNIFIED IDEOGRAPH-#402F, IRGKangXi=0808.070, Def{to look at; to see; to observe; to examine; to consider}
4030 CJK UNIFIED IDEOGRAPH-#4030, IRGKangXi=0808.080, Def{open eyes, big eyes, to move one's eyes}
4031 CJK UNIFIED IDEOGRAPH-#4031, IRGKangXi=0808.140, Def{to examine; to observe; to survey; to study}
4032 CJK UNIFIED IDEOGRAPH-#4032, IRGKangXi=0808.180, Def{diseases in the eyes}
4033 CJK UNIFIED IDEOGRAPH-#4033, IRGKangXi=0808.230, Def{(abbreviated form) not to look straight forward, the pupil of the eyes not in the right place}
4034 CJK UNIFIED IDEOGRAPH-#4034, IRGKangXi=0808.260, Def{to look straight forward, to look, eyesight blurred; not clear}
4035 CJK UNIFIED IDEOGRAPH-#4035, IRGKangXi=0808.280, Def{(same as 瞬) to blink; wink or twinkle, by eyes (facial expressions) to indicate one's wish or intention}
4036 CJK UNIFIED IDEOGRAPH-#4036, IRGKangXi=0808.340, Def{strabismus; squint, to look askance; to ogle, (a dialect) bright; light; brilliant, bright eyes}
4037 CJK UNIFIED IDEOGRAPH-#4037, IRGKangXi=0808.350, Def{to have quick glance; look-in; to look hastily}
4038 CJK UNIFIED IDEOGRAPH-#4038, IRGKangXi=0808.400, Def{(interchangeable 䀿) to have a casual and short glance; to catch a glimpse of, pretty eyes, insight; vision, bright eyes}
4039 CJK UNIFIED IDEOGRAPH-#4039, IRGKangXi=0809.030, Def{(same as 眨) to wink; (same as 睫) eyelashes, having one eye smaller than the other, joke; witticism; pleasantry; jest; fun; (Cant.) to peep at; to blink, wink}
403A CJK UNIFIED IDEOGRAPH-#403A, IRGKangXi=0809.060, Def{squinting eyes; prominent eyes, all blind, eyesight not clear, upper eyelid which has a double fold at the lower edge}
403B CJK UNIFIED IDEOGRAPH-#403B, IRGKangXi=0809.090, Def{to look; to regard; to inspect}
403C CJK UNIFIED IDEOGRAPH-#403C, IRGKangXi=0809.100, Def{(non-classical form of 瞋) angry; anger, complaining; grudging, wide open the eyes, depressed; melancholy, to look at, to confuse, confused vision}
403D CJK UNIFIED IDEOGRAPH-#403D, IRGKangXi=0809.110, Def{to look at each other}
403E CJK UNIFIED IDEOGRAPH-#403E, IRGKangXi=0809.180, Def{many; much, more than; over}
403F CJK UNIFIED IDEOGRAPH-#403F, IRGKangXi=0809.211, Def{(same as 䀸) to have a casual and short glance; to catch a glimpse of, pretty eyes, insight; vision, bright eyes}
4040 CJK UNIFIED IDEOGRAPH-#4040, IRGKangXi=0809.211
4041 CJK UNIFIED IDEOGRAPH-#4041, IRGKangXi=0809.240, Def{(same as 䀶) strabismus; squint, to look askance; to ogle, (a dialect) bright; light; brilliant, bright eyes}
4042 CJK UNIFIED IDEOGRAPH-#4042, IRGKangXi=0809.290, Def{big eyes}
4043 CJK UNIFIED IDEOGRAPH-#4043, IRGKangXi=0809.300, Def{to look attentively; to focus one's look at; to gaze at}
4044 CJK UNIFIED IDEOGRAPH-#4044, IRGKangXi=0809.350, Def{to close the eyes, to die without regret or in peace}
4045 CJK UNIFIED IDEOGRAPH-#4045, IRGKangXi=0810.060, Def{rage; anger, to cast an angry look; to give the black looks}
4046 CJK UNIFIED IDEOGRAPH-#4046, IRGKangXi=0810.110, Def{eyes, closed eyes}
4047 CJK UNIFIED IDEOGRAPH-#4047, IRGKangXi=0810.180, Def{to look for a short time, attentive vision}
4048 CJK UNIFIED IDEOGRAPH-#4048, IRGKangXi=0810.190, Def{to examine; to survey; to inspect; to visit, to watch or see in secret; to spy; to peep}
4049 CJK UNIFIED IDEOGRAPH-#4049, IRGKangXi=0810.200, Def{(same as 䁈) to examine; to survey; to inspect; to visit, to watch or see in secret; to spy; to peep}
404A CJK UNIFIED IDEOGRAPH-#404A, IRGKangXi=0812.020, Def{dark and deep eyed, to look down, to block, to stuff, to cork; to seal}
404B CJK UNIFIED IDEOGRAPH-#404B, IRGKangXi=0812.030, Def{with one eye closed, to close the eyes; (same as U+77B8 瞸) to look askance at; (Cant.) to close the eyes}
404C CJK UNIFIED IDEOGRAPH-#404C, IRGKangXi=0812.080, Def{to see for a short time}
404D CJK UNIFIED IDEOGRAPH-#404D, IRGKangXi=0812.150, Def{blind, hollow-eyed, to look at}
404E CJK UNIFIED IDEOGRAPH-#404E, IRGKangXi=0812.210, Def{to make a close inspection, secretion of the eye, (same as 瞠) to look straight at; to stare at}
404F CJK UNIFIED IDEOGRAPH-#404F, IRGKangXi=0812.240, Def{farsightedness (as a physical defect); hypermetropia, to look from a distance}
4050 CJK UNIFIED IDEOGRAPH-#4050, IRGKangXi=0812.260, Def{deep eyed, to look at; to inspect; to see}
4051 CJK UNIFIED IDEOGRAPH-#4051, IRGKangXi=0812.320, Def{(same as 眻) pretty eyes, the space between eyebrows, (interchangeable 揚) to raise; to praise, to display}
4052 CJK UNIFIED IDEOGRAPH-#4052, IRGKangXi=0812.380, Def{to wink, to shed tears}
4053 CJK UNIFIED IDEOGRAPH-#4053, IRGKangXi=0812.400, Def{to look at; to see, to peep; to look something stealthily, angry}
4054 CJK UNIFIED IDEOGRAPH-#4054, IRGKangXi=0812.410, Def{big eyes (same as 睅) protuberant eyes, goggle-eye}
4055 CJK UNIFIED IDEOGRAPH-#4055, IRGKangXi=0813.010, Def{(same as 盿) to look at; to see, to look down; to look from above}
4056 CJK UNIFIED IDEOGRAPH-#4056, IRGKangXi=0813.161, Def{to see; to look at; to observe}
4057 CJK UNIFIED IDEOGRAPH-#4057, IRGKangXi=0813.190, Def{light; bright; brilliant, clear, to shine upon; to light or illumine}
4058 CJK UNIFIED IDEOGRAPH-#4058, IRGKangXi=0813.200, Def{to look at; to see, pretty eyes; (same as U+404F 䁏) farsightedness (as a physical defect); hypermetropia, to look from a distance}
4059 CJK UNIFIED IDEOGRAPH-#4059, IRGKangXi=0813.220, Def{eloquent eyes; to converse with eyes; to make sheep eyes or passes, to look at; to see, to look up to; to respect}
405A CJK UNIFIED IDEOGRAPH-#405A, IRGKangXi=0814.010, Def{a kind of eye disease, (a non-classical form)}
405B CJK UNIFIED IDEOGRAPH-#405B, IRGKangXi=0814.230, Def{to look; to see; to inspect}
405C CJK UNIFIED IDEOGRAPH-#405C, IRGKangXi=0814.350, Def{big eyes, a kind of eye disease}
405D CJK UNIFIED IDEOGRAPH-#405D, IRGKangXi=0815.090, Def{deluding and causing disorder (interchangeable 熒) lights shining; sparkling; twinkling; shimmering}
405E CJK UNIFIED IDEOGRAPH-#405E, IRGKangXi=0815.110, Def{(same as ancient form of 省) a province, to examine; to watch, to reduce, to diminish}
405F CJK UNIFIED IDEOGRAPH-#405F, IRGKangXi=0815.130, Def{(interchangeable 差) wrong; mistaken; erroneous, disorderly; untidy; irregular}
4060 CJK UNIFIED IDEOGRAPH-#4060, IRGKangXi=0815.160, Def{to hang down (eyes)}
4061 CJK UNIFIED IDEOGRAPH-#4061, IRGKangXi=0815.181
4062 CJK UNIFIED IDEOGRAPH-#4062, IRGKangXi=0815.190, Def{(same as 嫙) exquisite; fine}
4063 CJK UNIFIED IDEOGRAPH-#4063, IRGKangXi=0815.240, Def{to wink}
4064 CJK UNIFIED IDEOGRAPH-#4064, IRGKangXi=0815.340, Def{moistened eyes, to stare in anger or contempt}
4065 CJK UNIFIED IDEOGRAPH-#4065, IRGKangXi=0815.350, Def{to blink; to half-close the eyes, small eyes (same as U+7724 昵) very dear; very intimate; very much in love}
4066 CJK UNIFIED IDEOGRAPH-#4066, IRGKangXi=0815.410, Def{(same as U+89B0 覷) to see; to look at, to steal a glance; to glance quickly, to spy on; to peep at}
4067 CJK UNIFIED IDEOGRAPH-#4067, IRGKangXi=0816.020, Def{to open wide (eyes)}
4068 CJK UNIFIED IDEOGRAPH-#4068, IRGKangXi=0816.080, Def{to open the eyes with astonishment, (interchangeable 矐) to lose one's eyesight; to become blind; blind}
4069 CJK UNIFIED IDEOGRAPH-#4069, IRGKangXi=0816.140, Def{(same as 魚) fish, inferior horse with blind eyes}
406A CJK UNIFIED IDEOGRAPH-#406A, IRGKangXi=0816.211, Def{to wink}
406B CJK UNIFIED IDEOGRAPH-#406B, IRGKangXi=0817.020, Def{muddy; dirty, eyesight obscured}
406C CJK UNIFIED IDEOGRAPH-#406C, IRGKangXi=0817.040, Def{to act pretentiously (by slit-eyed), eyesight obscured}
406D CJK UNIFIED IDEOGRAPH-#406D, IRGKangXi=0817.070, Def{to give an angry look, to watch; to view}
406E CJK UNIFIED IDEOGRAPH-#406E, IRGKangXi=0817.150, Def{sad; mournful; grieved, to worry about, to think with one's eyes closed, to close the eyes}
406F CJK UNIFIED IDEOGRAPH-#406F, IRGKangXi=0817.160, Def{to look at; to see; to inspect; (same as U+77B8 瞸) gazing into distance, having one eye smaller than the other, to look askance at; (Cant.) to close the eyes}
4070 CJK UNIFIED IDEOGRAPH-#4070, IRGKangXi=0817.220, Def{to look in a evil or wicked way; wicked vision}
4071 CJK UNIFIED IDEOGRAPH-#4071, IRGKangXi=0817.250, Def{(same as 眗) hollow-eyed}
4072 CJK UNIFIED IDEOGRAPH-#4072, IRGKangXi=0818.060, Def{to peep; to look at something stealthily; (Cant.) to close the eyes}
4073 CJK UNIFIED IDEOGRAPH-#4073, IRGKangXi=0818.120, Def{to lose one eye}
4074 CJK UNIFIED IDEOGRAPH-#4074, IRGKangXi=0818.250, Def{keep on looking, change the facial expression}
4075 CJK UNIFIED IDEOGRAPH-#4075, IRGKangXi=0818.280, Def{newborn baby with the eyelids closed, big eyes, to close the eyes}
4076 CJK UNIFIED IDEOGRAPH-#4076, IRGKangXi=0818.290, Def{to look at something without winking, bright}
4077 CJK UNIFIED IDEOGRAPH-#4077, IRGKangXi=0818.310, Def{bright eyes, angry look; angry eyes (ancient form of 覺) to awaken, to wake up from sleep, to feel}
4078 CJK UNIFIED IDEOGRAPH-#4078, IRGKangXi=0818.320, Def{eyesight obscured, angry glances}
4079 CJK UNIFIED IDEOGRAPH-#4079, IRGKangXi=0818.330, Def{(same as U+7764 睥) to look askance -- a expression of disdain or despise}
407A CJK UNIFIED IDEOGRAPH-#407A, IRGKangXi=0818.350, Def{to look at; to see; to inspect, sharp, bright-colored; bright eyes, (same as non-classical form of 睪) spy on, to lead on}
407B CJK UNIFIED IDEOGRAPH-#407B, IRGKangXi=0820.110, Def{pretty eyes, to have a casual and short glance}
407C CJK UNIFIED IDEOGRAPH-#407C, IRGKangXi=0820.120, Def{(literally) to look but see nothing -- absent-minded, to regard as nothing}
407D CJK UNIFIED IDEOGRAPH-#407D, IRGKangXi=0820.140, Def{eyesight obscured, abnormal vision; (Cant.) to glance at, sweep the eyes over}
407E CJK UNIFIED IDEOGRAPH-#407E, IRGKangXi=0820.200, Def{red and swelling of the eye socket, dim-sighted; poor visioned}
407F CJK UNIFIED IDEOGRAPH-#407F, IRGKangXi=0820.300, Def{sleepy; drowsy, to startle; to surprise; to amaze}
4080 CJK UNIFIED IDEOGRAPH-#4080, IRGKangXi=0820.310, Def{to wink}
4081 CJK UNIFIED IDEOGRAPH-#4081, IRGKangXi=0820.350, Def{an angry look; to look at someone angrily, hollow-eyed}
4082 CJK UNIFIED IDEOGRAPH-#4082, IRGKangXi=0821.010, Def{to gaze at in terror, a surname}
4083 CJK UNIFIED IDEOGRAPH-#4083, IRGKangXi=0821.070, Def{dark; dim; eyesight obscured, angry look; angry eyes}
4084 CJK UNIFIED IDEOGRAPH-#4084, IRGKangXi=0821.160, Def{to startle; to surprise; to amaze; to look at in surprise}
4085 CJK UNIFIED IDEOGRAPH-#4085, IRGKangXi=0821.210, Def{to have insight or vision of a simple and honest person, silly}
4086 CJK UNIFIED IDEOGRAPH-#4086, IRGKangXi=0821.270, Def{weapons; arms (a lance; a spear)}
4087 CJK UNIFIED IDEOGRAPH-#4087, IRGKangXi=0822.020, Def{to pierce; to stab; to irritate; to hurt, a thorn}
4088 CJK UNIFIED IDEOGRAPH-#4088, IRGKangXi=0822.130, Def{to pierce; to stab; to irritate; to hurt, a thorn, a lance; a spear}
4089 CJK UNIFIED IDEOGRAPH-#4089, IRGKangXi=0822.270, Def{weapons; arms (a lance; a spear)}
408A CJK UNIFIED IDEOGRAPH-#408A, IRGKangXi=0823.170, Def{weapons; arms (a lance; a spear)}
408B CJK UNIFIED IDEOGRAPH-#408B, IRGKangXi=0558.091
408C CJK UNIFIED IDEOGRAPH-#408C, IRGKangXi=0823.190, Def{a short weapon (daggers; swords; knives)}
408D CJK UNIFIED IDEOGRAPH-#408D, IRGKangXi=0823.290, Def{a kind of weapon carried by the honor guard before the Emperor in ancient times}
408E CJK UNIFIED IDEOGRAPH-#408E, IRGKangXi=0823.310, Def{a short spear; a lance with two points, a halberd, to pierce; to stab; to irritate, to catch a spear from away}
408F CJK UNIFIED IDEOGRAPH-#408F, IRGKangXi=0824.120, Def{short, dog with short tail}
4090 CJK UNIFIED IDEOGRAPH-#4090, IRGKangXi=0824.140, Def{short (ancient form of 拙) stupid; crude; poor; slow and clumsy}
4091 CJK UNIFIED IDEOGRAPH-#4091, IRGKangXi=0824.150, RSKangXi=111.5, Def{short; (Cant.) intensifier}
4092 CJK UNIFIED IDEOGRAPH-#4092, IRGKangXi=0824.240, Def{short (dialect) to curry favor; to toady; to flatter; to try hard to please}
4093 CJK UNIFIED IDEOGRAPH-#4093, IRGKangXi=0825.010, Def{(standard form of 規) regulations; laws; rules; customs or usages, a pair of compasses}
4094 CJK UNIFIED IDEOGRAPH-#4094, IRGKangXi=0825.040, Def{short}
4095 CJK UNIFIED IDEOGRAPH-#4095, IRGKangXi=0825.140, Def{short}
4096 CJK UNIFIED IDEOGRAPH-#4096, IRGKangXi=0827.031, Def{(same as 石) rocks; stones; minerals, etc.}
4097 CJK UNIFIED IDEOGRAPH-#4097, IRGKangXi=0827.100, Def{to be careful; to exercise caution; to take care; to pay attention}
4098 CJK UNIFIED IDEOGRAPH-#4098, IRGKangXi=0827.191, Def{pebble; small piece of stone}
4099 CJK UNIFIED IDEOGRAPH-#4099, IRGKangXi=1270.071
409A CJK UNIFIED IDEOGRAPH-#409A, IRGKangXi=0828.110, Def{a kind of stone, rugged rocks, difficult; hard; difficulty; hardship}
409B CJK UNIFIED IDEOGRAPH-#409B, IRGKangXi=0828.120, Def{a kind of rock}
409C CJK UNIFIED IDEOGRAPH-#409C, IRGKangXi=0828.140, Def{rocky (same as 玤) fine stone which is little less valuable than jade (interchangeable 崩) to collapse; to fall}
409D CJK UNIFIED IDEOGRAPH-#409D, IRGKangXi=0828.241, Def{(same as 矺) to rap; to tap; to beat(same as 磔) torture of dismemberment used in the ancient times, sound of throwing something to the ground}
409E CJK UNIFIED IDEOGRAPH-#409E, IRGKangXi=0521.121, Def{(same as 柘) a thorny tree about 15 feet high, the leaves are used for feeding silkworms before the mulberry leaves are ready or when they are scarce, bark contains a yellow dye (interchangeable 蔗) the sugar cane}
409F CJK UNIFIED IDEOGRAPH-#409F, IRGKangXi=0828.290, Def{rocks; stones; minerals, etc.}
40A0 CJK UNIFIED IDEOGRAPH-#40A0, IRGKangXi=0828.340, Def{sound of falling rocks, the arrowhead made of stone}
40A1 CJK UNIFIED IDEOGRAPH-#40A1, IRGKangXi=0828.380, Def{(non-classical form 砥) a whetstone, smooth, to polish}
40A2 CJK UNIFIED IDEOGRAPH-#40A2, IRGKangXi=0828.400, Def{sound of falling rocks, fallen rocks}
40A3 CJK UNIFIED IDEOGRAPH-#40A3, IRGKangXi=0828.410, Def{a kind of mineral}
40A4 CJK UNIFIED IDEOGRAPH-#40A4, IRGKangXi=0828.440, Def{white stone}
40A5 CJK UNIFIED IDEOGRAPH-#40A5, IRGKangXi=0829.030, Def{(same as 賨) a fine stone resembling jade}
40A6 CJK UNIFIED IDEOGRAPH-#40A6, IRGKangXi=0829.260, Def{tired; exhausted, rugged stone, coarse whetstone, to pile up rocks on the river-bank}
40A7 CJK UNIFIED IDEOGRAPH-#40A7, IRGKangXi=0829.320, Def{rocks clustered pile up together}
40A8 CJK UNIFIED IDEOGRAPH-#40A8, IRGKangXi=0829.331
40A9 CJK UNIFIED IDEOGRAPH-#40A9, IRGKangXi=0829.390, Def{(same as 硯) an in-slab or ink-stone (same as 硜) sound of pebbles or stones rubbing or knocking together}
40AA CJK UNIFIED IDEOGRAPH-#40AA, IRGKangXi=0829.450, Def{(same as 磽) hard barren land, a kind of stone (interchangeable 銚) a small pot with a handle}
40AB CJK UNIFIED IDEOGRAPH-#40AB, IRGKangXi=0829.460, Def{noise of stones rolling down, to ram the earth for foundations, (same as U+592F 夯) to raise with force; a heavy load; burden, to fill cracks and leakages with earth (in levee construction)}
40AC CJK UNIFIED IDEOGRAPH-#40AC, IRGKangXi=0829.550, Def{big rocks on the riverside (interchangeable 栱) an arched opening}
40AD CJK UNIFIED IDEOGRAPH-#40AD, IRGKangXi=0830.050, Def{name of a place; today's Jizhou}
40AE CJK UNIFIED IDEOGRAPH-#40AE, IRGKangXi=0830.070, Def{stone, to sharpen a knife, sharp-pointed; sharp, vigorous; energetic; keen}
40AF CJK UNIFIED IDEOGRAPH-#40AF, IRGKangXi=0830.131, Def{a fine stone resembling jade}
40B0 CJK UNIFIED IDEOGRAPH-#40B0, IRGKangXi=0830.190, Def{rocks; stones; minerals, etc.}
40B1 CJK UNIFIED IDEOGRAPH-#40B1, IRGKangXi=0830.220, Def{to wear out; rubbed out; to die out; to wear away; frayed}
40B2 CJK UNIFIED IDEOGRAPH-#40B2, IRGKangXi=0830.260, Def{(same as 砝) standard weights used in scales; steelyard weights}
40B3 CJK UNIFIED IDEOGRAPH-#40B3, IRGKangXi=0830.330, Def{gravel; macadam; (interchangeable 剉) to damage; to destroy, medicines; orpiment (common monoclinic arsenic sulfide mineral, As2S3)}
40B4 CJK UNIFIED IDEOGRAPH-#40B4, IRGKangXi=0831.120, Def{(same as U+7814 研) to rub; to grind; to powder, to go the very source; to study; to investigate; to research; to examine; to search into carefully}
40B5 CJK UNIFIED IDEOGRAPH-#40B5, IRGKangXi=0831.131, Def{(a simplified form of 碽) sound of bumping or striking, arched bridge}
40B6 CJK UNIFIED IDEOGRAPH-#40B6, IRGKangXi=0831.131, Def{the Crustacea; a sea-anemone}
40B7 CJK UNIFIED IDEOGRAPH-#40B7, IRGKangXi=0831.140, Def{broken; smashed, trivial, irregular, different; varied sizes (of rocks)}
40B8 CJK UNIFIED IDEOGRAPH-#40B8, IRGKangXi=0831.190, Def{to smash or to knock to pieces of the stone, broken}
40B9 CJK UNIFIED IDEOGRAPH-#40B9, IRGKangXi=0831.210, Def{sound of the falling pieces of rocks}
40BA CJK UNIFIED IDEOGRAPH-#40BA, IRGKangXi=0831.240, Def{rocks; stones (same as 顆) a drop; a grain; a bead}
40BB CJK UNIFIED IDEOGRAPH-#40BB, IRGKangXi=0831.270, Def{to block; to stop up, to cork; to seal; (same as U+92E5 鋥) to polish; to grind; (Cant.) to push; to burst; to stuff}
40BC CJK UNIFIED IDEOGRAPH-#40BC, IRGKangXi=0831.310, Def{(a kind of) rocks; stones}
40BD CJK UNIFIED IDEOGRAPH-#40BD, IRGKangXi=0831.340, Def{(a kind of) rocks; stones}
40BE CJK UNIFIED IDEOGRAPH-#40BE, IRGKangXi=0831.350, Def{to polish; to grind; to rub}
40BF CJK UNIFIED IDEOGRAPH-#40BF, IRGKangXi=0831.380, Def{to pound (grain) in order to remove the husk; refine; polished (rice), to tread upon; to step upon a pestle (for husking grain)}
40C0 CJK UNIFIED IDEOGRAPH-#40C0, IRGKangXi=0831.390, Def{millstone; grindstone, to rub; to grind; to polish; to wear, to sharpen (a knife), to train; to harden; to temper}
40C1 CJK UNIFIED IDEOGRAPH-#40C1, IRGKangXi=0832.060, Def{uneven or rugged terrains}
40C2 CJK UNIFIED IDEOGRAPH-#40C2, IRGKangXi=0832.120, Def{dull sound of the tolling bell, sound of rolling, rolling stone, a farm tool made of stone (a stone roller for hulling grains, etc.)}
40C3 CJK UNIFIED IDEOGRAPH-#40C3, IRGKangXi=0832.211
40C4 CJK UNIFIED IDEOGRAPH-#40C4, IRGKangXi=0832.211
40C5 CJK UNIFIED IDEOGRAPH-#40C5, IRGKangXi=0832.211, Def{(simplified form 磾) dyestuff (black colored mineral)}
40C6 CJK UNIFIED IDEOGRAPH-#40C6, IRGKangXi=0832.211
40C7 CJK UNIFIED IDEOGRAPH-#40C7, IRGKangXi=0832.211, Def{ink (usually red) for imprinting of seals}
40C8 CJK UNIFIED IDEOGRAPH-#40C8, IRGKangXi=0832.240, Def{mountain rocks(same as 瑎) a black stone resembling jade}
40C9 CJK UNIFIED IDEOGRAPH-#40C9, IRGKangXi=0832.330, Def{(same as 賨) a stone resembling jade}
40CA CJK UNIFIED IDEOGRAPH-#40CA, IRGKangXi=0832.350, Def{to rub; to grind; to polish; to wear}
40CB CJK UNIFIED IDEOGRAPH-#40CB, IRGKangXi=0833.100, Def{a fine stone resembling jade}
40CC CJK UNIFIED IDEOGRAPH-#40CC, IRGKangXi=0833.180, Def{rocky mountain, used in naming a place}
40CD CJK UNIFIED IDEOGRAPH-#40CD, IRGKangXi=0833.200, Def{(ancient form of 墜) to fall down; to sink}
40CE CJK UNIFIED IDEOGRAPH-#40CE, IRGKangXi=0833.220, Def{name of a place (usually to be used in naming a place) (interchangeable 砟) small piece of coal}
40CF CJK UNIFIED IDEOGRAPH-#40CF, IRGKangXi=0833.230, Def{a kind of rock}
40D0 CJK UNIFIED IDEOGRAPH-#40D0, IRGKangXi=0833.291
40D1 CJK UNIFIED IDEOGRAPH-#40D1, IRGKangXi=0834.030, Def{(same as standard form 斑) speckles; spots; mottles (same as 磐) a massive rock}
40D2 CJK UNIFIED IDEOGRAPH-#40D2, IRGKangXi=0834.060, Def{(of land) poor and barren; field which is not arable}
40D3 CJK UNIFIED IDEOGRAPH-#40D3, IRGKangXi=0834.080, Def{to punish; to chastise; to penalize, to kill; to execute, pedal (a footboard) of a cart, brick work of a well, strong and durable, solid; firm}
40D4 CJK UNIFIED IDEOGRAPH-#40D4, IRGKangXi=0834.090, Def{sound of falling rocks, loud noise, stone with strange and unique shapes}
40D5 CJK UNIFIED IDEOGRAPH-#40D5, IRGKangXi=0834.130, Def{stoneware; stone implement, soap-stone, sound of the bumping rocks, (in music) a chord}
40D6 CJK UNIFIED IDEOGRAPH-#40D6, IRGKangXi=0834.200, Def{(same as U+9696 塢) a bank; a low wall; an entrenchment, a structure which slants to a lower center on all sides -- as a shipyard}
40D7 CJK UNIFIED IDEOGRAPH-#40D7, IRGKangXi=0834.230, Def{rocks; stones, rocks paved bank, big rocks}
40D8 CJK UNIFIED IDEOGRAPH-#40D8, IRGKangXi=0835.100, Def{simple and crude (same as 硜) the sound of pebbles or stones knocking together, obstinate; determined; resolute}
40D9 CJK UNIFIED IDEOGRAPH-#40D9, IRGKangXi=0835.120, Def{rocks; stones; minerals, etc.}
40DA CJK UNIFIED IDEOGRAPH-#40DA, IRGKangXi=0835.160, Def{rugged land of sand and pebble; poor and barren (of land) (same as 鏃) the barb of an arrow, arrowhead; the head of a javelin}
40DB CJK UNIFIED IDEOGRAPH-#40DB, IRGKangXi=0835.280, Def{(same as 鏈) chain, cable}
40DC CJK UNIFIED IDEOGRAPH-#40DC, IRGKangXi=0835.320, Def{a fine black stone}
40DD CJK UNIFIED IDEOGRAPH-#40DD, IRGKangXi=0835.370, Def{(same as 礉) rugged rocks}
40DE CJK UNIFIED IDEOGRAPH-#40DE, IRGKangXi=0836.070, Def{a kind of stone, sound of pebbles or stones rubbing or knocking together}
40DF CJK UNIFIED IDEOGRAPH-#40DF, IRGKangXi=0836.091, Def{䃟頭窰, a place in Hong Kong}
40E0 CJK UNIFIED IDEOGRAPH-#40E0, IRGKangXi=0836.091
40E1 CJK UNIFIED IDEOGRAPH-#40E1, IRGKangXi=0836.110, Def{a small pebble; small piece of stone or rock, a stone door}
40E2 CJK UNIFIED IDEOGRAPH-#40E2, IRGKangXi=0836.190, Def{name of a mountain (same as 嶔) lofty (of a mountain)}
40E3 CJK UNIFIED IDEOGRAPH-#40E3, IRGKangXi=0836.230, Def{(same as 毀) to ruin, to destroy, to break down}
40E4 CJK UNIFIED IDEOGRAPH-#40E4, IRGKangXi=0836.250, Def{black colored grindstone or whetstone}
40E5 CJK UNIFIED IDEOGRAPH-#40E5, IRGKangXi=0836.370, Def{stones; rocks}
40E6 CJK UNIFIED IDEOGRAPH-#40E6, IRGKangXi=0836.380, Def{a flat and great rock people squat or crouch on it}
40E7 CJK UNIFIED IDEOGRAPH-#40E7, IRGKangXi=0837.030, Def{sound of falling rocks}
40E8 CJK UNIFIED IDEOGRAPH-#40E8, IRGKangXi=0837.091
40E9 CJK UNIFIED IDEOGRAPH-#40E9, IRGKangXi=0837.110, Def{(same as 縭) a mineral, used as a medicine}
40EA CJK UNIFIED IDEOGRAPH-#40EA, IRGKangXi=0837.130, Def{(same as 壇) a platform for sacrificial rites; an altar, an arena; a hall for important meetings and ceremonies in ancient China}
40EB CJK UNIFIED IDEOGRAPH-#40EB, IRGKangXi=0837.250, Def{a silicate substance from the salt-wells in Szechwan; it is used as a wash for hardening plaster, etc., also for certain skin diseases, like ringworm, sulphate of copper}
40EC CJK UNIFIED IDEOGRAPH-#40EC, IRGKangXi=0837.380, Def{rocky; plenty of stones and rocks (same as 磈) piles of rugged stones; lumpy and uneven rocks}
40ED CJK UNIFIED IDEOGRAPH-#40ED, IRGKangXi=0837.390, Def{a stone box; a stone case, to cover with a piece of stone, (interchangeable 撼) to shake; to rock}
40EE CJK UNIFIED IDEOGRAPH-#40EE, IRGKangXi=0837.410, Def{a stone build hydraulic measures (to store and to vent the water) in ancient times, rocks in the mountain stream}
40EF CJK UNIFIED IDEOGRAPH-#40EF, IRGKangXi=0837.411, Def{(abbreviated form of 礫) small stones; pebble; gravel; shingle}
40F0 CJK UNIFIED IDEOGRAPH-#40F0, IRGKangXi=0838.040, Def{brick or stone steps}
40F1 CJK UNIFIED IDEOGRAPH-#40F1, IRGKangXi=0838.110, Def{(non-classical form of 䃸) electric light; a flash of lightning, wedge, preface foreword}
40F2 CJK UNIFIED IDEOGRAPH-#40F2, IRGKangXi=0838.150, Def{(same as 磐) a great rock}
40F3 CJK UNIFIED IDEOGRAPH-#40F3, IRGKangXi=0838.170, Def{falling stone, a mineral; an ore (with the element of bronze; copper), appearance of the connected mountains; a mountain range; a chain}
40F4 CJK UNIFIED IDEOGRAPH-#40F4, IRGKangXi=0838.330, Def{a whetstone used to grind precious stone (jade; a gem)}
40F5 CJK UNIFIED IDEOGRAPH-#40F5, IRGKangXi=0838.340, Def{a kind of stone, bends; curves; turns and twists of the mountains}
40F6 CJK UNIFIED IDEOGRAPH-#40F6, IRGKangXi=0838.370, Def{rugged and uneven of the rocks, small piece of stone; pebble, a fine stone resembling jade}
40F7 CJK UNIFIED IDEOGRAPH-#40F7, IRGKangXi=0838.380, Def{a kind of stone, to research and examine}
40F8 CJK UNIFIED IDEOGRAPH-#40F8, IRGKangXi=0838.390, Def{electric light; a flash of lightning, wedge, preface foreword}
40F9 CJK UNIFIED IDEOGRAPH-#40F9, IRGKangXi=0839.020, Def{jade and stone, fine jade}
40FA CJK UNIFIED IDEOGRAPH-#40FA, IRGKangXi=0839.110, Def{(a standard form of 磨) to grind, to rub, to sharpen}
40FB CJK UNIFIED IDEOGRAPH-#40FB, IRGKangXi=0839.150, Def{(same as 壩) an embankment; a dike; a levee; a dam; a bank of earth}
40FC CJK UNIFIED IDEOGRAPH-#40FC, IRGKangXi=0839.251
40FD CJK UNIFIED IDEOGRAPH-#40FD, IRGKangXi=0840.040, Def{name of a mountain (of warship), (same as 禔) happiness; good fortune; good luck; blessing, (same as standard form 祓) to exorcise; to remove evil; to cleanse; to clean; to wash away}
40FE CJK UNIFIED IDEOGRAPH-#40FE, IRGKangXi=0840.110, Def{to offer a small pig as sacrifice when worshipping the god of life}
40FF CJK UNIFIED IDEOGRAPH-#40FF, IRGKangXi=0841.010, Def{a rite or service, message offered to the gods in worship}
4100 CJK UNIFIED IDEOGRAPH-#4100, IRGKangXi=0841.031, Def{(same as 禍) calamity; disaster; evil; misfortune}
4101 CJK UNIFIED IDEOGRAPH-#4101, IRGKangXi=0841.080, Def{to worship; to honor by a service or rite; to offer sacrifices to}
4102 CJK UNIFIED IDEOGRAPH-#4102, IRGKangXi=0841.110, Def{(ancient form) to pray and to curse (the ways of treating a patient in ancient times)}
4103 CJK UNIFIED IDEOGRAPH-#4103, IRGKangXi=0841.250, Def{(ancient standard form U+6B83 殃) misfortune; calamity; retribution, a departed spirit}
4104 CJK UNIFIED IDEOGRAPH-#4104, IRGKangXi=0842.170, Def{achievement; accomplishment, (non-classical form of 禋) to worship with sincerity and reverence, to offer sacrifices to the Heaven}
4105 CJK UNIFIED IDEOGRAPH-#4105, IRGKangXi=0842.180, Def{(same as standard form U+990B 餋) to worship; to honor by a rite or service; to offer sacrifices}
4106 CJK UNIFIED IDEOGRAPH-#4106, IRGKangXi=0843.030, Def{to worship; to honor by a rite or service; to offer sacrifices, an institution, law, to perform rites in honor of gods}
4107 CJK UNIFIED IDEOGRAPH-#4107, IRGKangXi=0843.100, Def{(corrupted form) a family name}
4108 CJK UNIFIED IDEOGRAPH-#4108, IRGKangXi=0843.130, Def{sacrifice to happiness and good luck}
4109 CJK UNIFIED IDEOGRAPH-#4109, IRGKangXi=0843.150, Def{a rite or service, (same as 娥) good; beautiful, a common name for a girl}
410A CJK UNIFIED IDEOGRAPH-#410A, IRGKangXi=0843.261
410B CJK UNIFIED IDEOGRAPH-#410B, IRGKangXi=0843.270, Def{to pray for preventing or forestalling calamities, muddy}
410C CJK UNIFIED IDEOGRAPH-#410C, IRGKangXi=0843.300, Def{to perform rites in honor of the ancestral temple; the next day's service of the above mentioned rites, (same as 醊) wine poured in a libation}
410D CJK UNIFIED IDEOGRAPH-#410D, IRGKangXi=0843.310, Def{(same as U+8721 蜡) year-end sacrifice of the Zhou Dynasty; imperial sacrifice of thanksgiving to the earth for crops, offered at the end of the year, the name varying in different dynastic periods}
410E CJK UNIFIED IDEOGRAPH-#410E, IRGKangXi=0844.010, Def{(same as 婍) pretty; beautiful (of a woman)}
410F CJK UNIFIED IDEOGRAPH-#410F, IRGKangXi=0844.020, Def{(same as 祅) bizarre, calamity due to terrestrial disturbances (as distinct from 災 -- disaster which is sent down from above) (interchangeable 妖) weird; supernatural, a ghost, bewitching; seductive (said of a woman)}
4110 CJK UNIFIED IDEOGRAPH-#4110, IRGKangXi=0844.120, Def{to worship; to honor by a service or rite; to offer sacrifices, happiness; good fortune; good luck; blessing; bliss}
4111 CJK UNIFIED IDEOGRAPH-#4111, IRGKangXi=0844.171, Def{(same as standard form 䄆) (non-classical form) to worship; to honor by a rite or service; to offer sacrifices, an institution, law, to perform rites in honor of gods}
4112 CJK UNIFIED IDEOGRAPH-#4112, IRGKangXi=0844.171, Def{(same as standard form 稔) ripening of paddy or rice; a harvest, a year}
4113 CJK UNIFIED IDEOGRAPH-#4113, IRGKangXi=0844.210, Def{to worship, a rite; a service}
4114 CJK UNIFIED IDEOGRAPH-#4114, IRGKangXi=0844.290, Def{a family name}
4115 CJK UNIFIED IDEOGRAPH-#4115, IRGKangXi=0845.150, Def{(ancient form of 社) god of the land, an association; an organization; society; community}
4116 CJK UNIFIED IDEOGRAPH-#4116, IRGKangXi=0845.161, Def{used in name of a person}
4117 CJK UNIFIED IDEOGRAPH-#4117, IRGKangXi=0845.161
4118 CJK UNIFIED IDEOGRAPH-#4118, IRGKangXi=0845.230, Def{a rite; a service; to worship}
4119 CJK UNIFIED IDEOGRAPH-#4119, IRGKangXi=0846.020, Def{happiness; good luck; good fortune; blessing; bliss}
411A CJK UNIFIED IDEOGRAPH-#411A, IRGKangXi=0846.120, Def{to worship the god (of the hog; pig), blessed; bliss; divine help, to protect; to defend}
411B CJK UNIFIED IDEOGRAPH-#411B, IRGKangXi=0846.130, Def{(standard form 膢) (in ancient times) service or rites of offering sacrifices for drink and food}
411C CJK UNIFIED IDEOGRAPH-#411C, IRGKangXi=0846.170, Def{happiness and well-being; bliss; propitious; auspicious, evil spirit; a hobgoblin produced from the weird emanations of the trees and rocks on the hills, (interchangeable 魑) a mountain demon resembling a tiger}
411D CJK UNIFIED IDEOGRAPH-#411D, IRGKangXi=0846.180, Def{show no respect to the service of worship}
411E CJK UNIFIED IDEOGRAPH-#411E, IRGKangXi=0846.261
411F CJK UNIFIED IDEOGRAPH-#411F, IRGKangXi=0846.290, Def{to worship; to offer sacrifices couple times, to thank}
4120 CJK UNIFIED IDEOGRAPH-#4120, IRGKangXi=0847.050, Def{(ancient form of 禪) to sacrifice to heaven, the imperial power, as only the emperor was allowed to offer these sacrifices, to cleanse; to exorcize, of Buddhism; Buddhist}
4121 CJK UNIFIED IDEOGRAPH-#4121, IRGKangXi=0847.110, Def{northern minority ethnic group (in ancient times), (corrupted form of 襜) the lower front of a robe, gown, etc., clean and neat (said of appearance)}
4122 CJK UNIFIED IDEOGRAPH-#4122, IRGKangXi=0847.220, Def{(same as 齋) pious; respectful; chaste; pure, to abstain from meat, wine, etc., to fast, (same as 齊) equal; uniform, name of an ancient feudal state}
4123 CJK UNIFIED IDEOGRAPH-#4123, IRGKangXi=0847.251
4124 CJK UNIFIED IDEOGRAPH-#4124, IRGKangXi=0847.270, Def{to sink; to fall; decadent, to idle about; to be negligent of worship}
4125 CJK UNIFIED IDEOGRAPH-#4125, IRGKangXi=0847.401, Def{a god; a spirit; an immortal}
4126 CJK UNIFIED IDEOGRAPH-#4126, IRGKangXi=0849.070, Def{to put forth ears and to blossom of the grains (corns and cereals)}
4127 CJK UNIFIED IDEOGRAPH-#4127, IRGKangXi=0849.080, Def{a kind of grain}
4128 CJK UNIFIED IDEOGRAPH-#4128, IRGKangXi=0849.130, Def{grains producing neither flower nor fruit}
4129 CJK UNIFIED IDEOGRAPH-#4129, IRGKangXi=0849.170, Def{(same as 䴬) bran; chaff, grains; corns; cereals}
412A CJK UNIFIED IDEOGRAPH-#412A, IRGKangXi=0849.180, Def{hanging down of the ears of the grains, something to hang or be hanged or hung}
412B CJK UNIFIED IDEOGRAPH-#412B, IRGKangXi=0850.031, Def{(same as 芑) a kind of sorghum with white sprouts, a kind of wild vegetable; bitter herbs}
412C CJK UNIFIED IDEOGRAPH-#412C, IRGKangXi=0850.030, Def{(same as 移) to change; to shift; to move; to influence, to coney; to forward}
412D CJK UNIFIED IDEOGRAPH-#412D, IRGKangXi=0850.031, Def{(same as 年) a year, one's age}
412E CJK UNIFIED IDEOGRAPH-#412E, IRGKangXi=0850.050, Def{dark rice plant, millet grain (in black color), (same as 稃) bran}
412F CJK UNIFIED IDEOGRAPH-#412F, IRGKangXi=0850.140, Def{ten handle of grains (rice plant)}
4130 CJK UNIFIED IDEOGRAPH-#4130, IRGKangXi=0850.250, Def{forty bundles of rice plant, panicled millet, (same as 芽) sprout; shoot; bud}
4131 CJK UNIFIED IDEOGRAPH-#4131, IRGKangXi=0850.260, Def{a kind of grain}
4132 CJK UNIFIED IDEOGRAPH-#4132, IRGKangXi=0851.020, Def{inside; within; inner; interior; domestic, heartlessness}
4133 CJK UNIFIED IDEOGRAPH-#4133, IRGKangXi=0851.091
4134 CJK UNIFIED IDEOGRAPH-#4134, IRGKangXi=0851.091
4135 CJK UNIFIED IDEOGRAPH-#4135, IRGKangXi=0850.031
4136 CJK UNIFIED IDEOGRAPH-#4136, IRGKangXi=0851.190, Def{rice plant growing the second time, the ear of grain (corn, millet etc.) growing upward}
4137 CJK UNIFIED IDEOGRAPH-#4137, IRGKangXi=0852.010, Def{(same as 石) a measurement (in ancient times) equal to 120 catty (Chinese pound)}
4138 CJK UNIFIED IDEOGRAPH-#4138, IRGKangXi=0852.060, Def{grains producing no fruit}
4139 CJK UNIFIED IDEOGRAPH-#4139, IRGKangXi=0852.101, Def{(same as 年) a year, age, harvest}
413A CJK UNIFIED IDEOGRAPH-#413A, IRGKangXi=0852.130, Def{(same as U+7A3A 稚) young and tender; small; delicate; immature, (same as U+7A0A 稊) darnels; tares, grass}
413B CJK UNIFIED IDEOGRAPH-#413B, IRGKangXi=0852.140, Def{a second time growing of rice plant, rice plant, (dialect) sorghum; kaoliang}
413C CJK UNIFIED IDEOGRAPH-#413C, IRGKangXi=0852.170, Def{name of a place in ancient times}
413D CJK UNIFIED IDEOGRAPH-#413D, IRGKangXi=0852.180, Def{(same as U+79C8 秈, U+7C7C 籼) common rice, as distinguished from glutinous rice; (corrupted form of U+413C 䄼) name of a place in ancient times}
413E CJK UNIFIED IDEOGRAPH-#413E, IRGKangXi=0852.210, Def{thick; height, black millet}
413F CJK UNIFIED IDEOGRAPH-#413F, IRGKangXi=0852.270, Def{a kind of rice plant}
4140 CJK UNIFIED IDEOGRAPH-#4140, IRGKangXi=0852.290, Def{the stalk (stem) of grain, neat and orderly rows of rice seedling}
4141 CJK UNIFIED IDEOGRAPH-#4141, IRGKangXi=0852.310, Def{to husk rice; to get the grains by oppressing the ears of the rice plant}
4142 CJK UNIFIED IDEOGRAPH-#4142, IRGKangXi=0852.320, Def{a kind of grain (looks like millet but much smaller)}
4143 CJK UNIFIED IDEOGRAPH-#4143, IRGKangXi=0852.340, Def{late crop, to reap or harvest; to cut grain}
4144 CJK UNIFIED IDEOGRAPH-#4144, IRGKangXi=0853.010, Def{long grains}
4145 CJK UNIFIED IDEOGRAPH-#4145, IRGKangXi=0853.030, Def{farm tool; agricultural implements, to till; to plough; to cultivate}
4146 CJK UNIFIED IDEOGRAPH-#4146, IRGKangXi=0853.060, Def{dense; crowded; closely, a piece of wood used to strike off grain in a measure thus - to level; all; to adjust; overall}
4147 CJK UNIFIED IDEOGRAPH-#4147, IRGKangXi=0853.080, Def{(ancient form of 粟) grains, millet, Indian corn}
4148 CJK UNIFIED IDEOGRAPH-#4148, IRGKangXi=0853.081
4149 CJK UNIFIED IDEOGRAPH-#4149, IRGKangXi=0853.081, Def{a kind of rice plant, shaking and waving of the rice plant}
414A CJK UNIFIED IDEOGRAPH-#414A, IRGKangXi=0853.081, Def{a spacious house, (corrupted form of 秺) bundle of rice plant, name of a place}
414B CJK UNIFIED IDEOGRAPH-#414B, IRGKangXi=0853.140, Def{a kind of grain; sweet and round rice}
414C CJK UNIFIED IDEOGRAPH-#414C, IRGKangXi=0853.170, Def{stalk (stem) of wheat or rice plant}
414D CJK UNIFIED IDEOGRAPH-#414D, IRGKangXi=0853.180, Def{straight upward of the wheat and rice plant}
414E CJK UNIFIED IDEOGRAPH-#414E, IRGKangXi=0853.190, Def{different; to distinguish}
414F CJK UNIFIED IDEOGRAPH-#414F, IRGKangXi=0854.010, Def{rice boiled to gruel, congee; porridge}
4150 CJK UNIFIED IDEOGRAPH-#4150, IRGKangXi=0854.050, Def{a bundle, to bind; bondage, a small bundle of, term of measurement; 10 bundle of rice plants}
4151 CJK UNIFIED IDEOGRAPH-#4151, IRGKangXi=0854.060, Def{four bundle of grains}
4152 CJK UNIFIED IDEOGRAPH-#4152, IRGKangXi=0854.160, Def{(interchangeable 芒) a sharp-edged grass, the awn or beard of wheat, etc.}
4153 CJK UNIFIED IDEOGRAPH-#4153, IRGKangXi=0854.190, Def{curved piece of wood, to bend; to crouch, name of a tree, name of a fruit}
4154 CJK UNIFIED IDEOGRAPH-#4154, IRGKangXi=0854.240, Def{dead (withered) crops; dried grains; (same as U+43A9 䎩) to till; to plough; to cultivate}
4155 CJK UNIFIED IDEOGRAPH-#4155, IRGKangXi=0854.280, Def{a variety of millet}
4156 CJK UNIFIED IDEOGRAPH-#4156, IRGKangXi=0854.310, Def{(same as 馣) sweet-smelling, tasty; delicious, to farm; to cultivate the land, luxuriant or exuberant of growing rice, grains;; rice plants producing no fruit, to fertilize, the grains not growing; shriveled rice plants}
4157 CJK UNIFIED IDEOGRAPH-#4157, IRGKangXi=0854.320, Def{(same as 䅑) four bundle of grains, to accumulate; to store up grains, grains; rice plant}
4158 CJK UNIFIED IDEOGRAPH-#4158, IRGKangXi=0854.370, Def{wheat}
4159 CJK UNIFIED IDEOGRAPH-#4159, IRGKangXi=0854.380, Def{grass; herb; straw; weed, bundle of straw}
415A CJK UNIFIED IDEOGRAPH-#415A, IRGKangXi=0854.390, Def{closely growing of the rice plants}
415B CJK UNIFIED IDEOGRAPH-#415B, IRGKangXi=0855.060, Def{husks of rice; rice bran or chaff}
415C CJK UNIFIED IDEOGRAPH-#415C, IRGKangXi=0855.110, Def{to store up grains, weeping rice plants, an ear of grain}
415D CJK UNIFIED IDEOGRAPH-#415D, IRGKangXi=0855.130, Def{the stalk of grain; straw}
415E CJK UNIFIED IDEOGRAPH-#415E, IRGKangXi=0855.260, Def{stalks of grain (rice plant)}
415F CJK UNIFIED IDEOGRAPH-#415F, IRGKangXi=0855.311, Def{(simplified form of 穇) varieties of millet; panicled millet, ear of grain producing no fruit, short grains}
4160 CJK UNIFIED IDEOGRAPH-#4160, IRGKangXi=0855.340, Def{(same as 蕛) a kind of grass, (interchangeable 稊) darnels, tender shoot of thatch (straw; couch grass)}
4161 CJK UNIFIED IDEOGRAPH-#4161, IRGKangXi=0855.360, Def{a kind of grass, grass growing in between of (among) the grains}
4162 CJK UNIFIED IDEOGRAPH-#4162, IRGKangXi=0855.400, Def{mature or ripen of the grains (rice; corns; cereals)}
4163 CJK UNIFIED IDEOGRAPH-#4163, IRGKangXi=0856.040, Def{a variety of panicled millet, a small coarse grain resembling sorghum}
4164 CJK UNIFIED IDEOGRAPH-#4164, IRGKangXi=0856.070, Def{to plant; to sow; to cultivate}
4165 CJK UNIFIED IDEOGRAPH-#4165, IRGKangXi=0856.140, Def{ripening crops, long grains, husks; chaff; bran (of grain)}
4166 CJK UNIFIED IDEOGRAPH-#4166, IRGKangXi=0856.150, Def{feeble; weak of the rice plant}
4167 CJK UNIFIED IDEOGRAPH-#4167, IRGKangXi=0856.200, Def{fine rice seedling; growing rice in order, (same as 蔭) shade of trees; shade}
4168 CJK UNIFIED IDEOGRAPH-#4168, IRGKangXi=0856.260, Def{(same as 香) fragrant; sweet-smelling}
4169 CJK UNIFIED IDEOGRAPH-#4169, IRGKangXi=0856.300, Def{crooked and winding (of grass, trees, vegetation, flora); crooked branches of a tree adjoin each other; (same as 歧) forked; divergent, anything that goes astray; wayward}
416A CJK UNIFIED IDEOGRAPH-#416A, IRGKangXi=0856.321, Def{(a simplified form) disease of the private part, hernia, colic}
416B CJK UNIFIED IDEOGRAPH-#416B, IRGKangXi=0856.321
416C CJK UNIFIED IDEOGRAPH-#416C, IRGKangXi=0857.010, Def{small bundle of rice plant, seed, grain}
416D CJK UNIFIED IDEOGRAPH-#416D, IRGKangXi=0857.030, Def{a variety of panicled millet, a small coarse grain resembling sorghum}
416E CJK UNIFIED IDEOGRAPH-#416E, IRGKangXi=0857.050, Def{a full head of grain; bent over by its own weight, two small pieces bract on the base of the ears of a rice plant, grass; weeds; tares}
416F CJK UNIFIED IDEOGRAPH-#416F, IRGKangXi=0857.080, Def{a variety of millet}
4170 CJK UNIFIED IDEOGRAPH-#4170, IRGKangXi=0857.100, Def{a kind of grass}
4171 CJK UNIFIED IDEOGRAPH-#4171, IRGKangXi=0857.110, Def{to eat to the full; surfeited}
4172 CJK UNIFIED IDEOGRAPH-#4172, IRGKangXi=0857.120, Def{to plant; to sow wheat}
4173 CJK UNIFIED IDEOGRAPH-#4173, IRGKangXi=0857.180, Def{stalk of the panicled millet; stalk of the rice plant}
4174 CJK UNIFIED IDEOGRAPH-#4174, IRGKangXi=0857.240, Def{grains, rice plant, an ear of grain}
4175 CJK UNIFIED IDEOGRAPH-#4175, IRGKangXi=0857.270, Def{chaff; bran; husks of grain, name of a place in ancient times}
4176 CJK UNIFIED IDEOGRAPH-#4176, IRGKangXi=0858.100, Def{to hoe, to weed; to hoe up weeds}
4177 CJK UNIFIED IDEOGRAPH-#4177, IRGKangXi=0858.150, Def{an ear of grain, (non-classical form of 稌) ancient term for glutinous rice, (same as 藷) a term for plants with tubers}
4178 CJK UNIFIED IDEOGRAPH-#4178, IRGKangXi=0858.180, Def{a grain, (same as 莘) lush growth; dense growth of the grass (trees, etc.) a marshy plant, known as asarum, a name derived from the bitterness of the root which is used in medicine}
4179 CJK UNIFIED IDEOGRAPH-#4179, IRGKangXi=0859.170, Def{(same as 耬) a kind of drill for sowing grain, to till lands}
417A CJK UNIFIED IDEOGRAPH-#417A, IRGKangXi=0859.220, Def{grain in the ear; to put forth ears, (same as 秒) the beard of grain, (same as 蔈) a measuring unit of weight used in ancient times, a plant (resembling reed) much used for making brooms}
417B CJK UNIFIED IDEOGRAPH-#417B, IRGKangXi=0859.230, Def{measurement; two bundle of grains (same as 離) bend of the ears of the variety of millet (because of the weight of the ears)}
417C CJK UNIFIED IDEOGRAPH-#417C, IRGKangXi=0859.260, Def{a kind of grains, to plant; to sow; to cultivate}
417D CJK UNIFIED IDEOGRAPH-#417D, IRGKangXi=0860.041, Def{(same as 馨) fragrance or aroma (especially that which comes from after), (same as 榖) grains and corns; cereals}
417E CJK UNIFIED IDEOGRAPH-#417E, IRGKangXi=0860.120, Def{starting to put forth ears, a kind of grain}
417F CJK UNIFIED IDEOGRAPH-#417F, IRGKangXi=0860.130, Def{(non-classical form) luxuriant or exuberant (of grain) (interchangeable 華) flowery; variegated, splendour}
4180 CJK UNIFIED IDEOGRAPH-#4180, IRGKangXi=0860.180, Def{(same as U+9EE3 霉 黴) to corrupt or be corrupted; to ruin or be ruined, dark sports; black specks, black}
4181 CJK UNIFIED IDEOGRAPH-#4181, IRGKangXi=0860.350, Def{syrup; jelly-like sugar made from grains}
4182 CJK UNIFIED IDEOGRAPH-#4182, IRGKangXi=0860.430, Def{a kind of grain, (same as 稴) trees producing no fruit}
4183 CJK UNIFIED IDEOGRAPH-#4183, IRGKangXi=0860.440, Def{a kind of rice plant; Excellent crop, to choose or to select hulled rice, millet; grains}
4184 CJK UNIFIED IDEOGRAPH-#4184, IRGKangXi=0860.450, Def{a bundle, a bundle of rice plant}
4185 CJK UNIFIED IDEOGRAPH-#4185, IRGKangXi=0860.460, Def{to accumulate or to store up rice, to purchase grain with public funds and store it against famine}
4186 CJK UNIFIED IDEOGRAPH-#4186, IRGKangXi=0861.051
4187 CJK UNIFIED IDEOGRAPH-#4187, IRGKangXi=0861.051
4188 CJK UNIFIED IDEOGRAPH-#4188, IRGKangXi=0861.270, Def{(non-classical form of U+7A3A 稚) young and tender; small; delicate; immature}
4189 CJK UNIFIED IDEOGRAPH-#4189, IRGKangXi=0861.340, Def{the swing of rice plant, a kind of paddy}
418A CJK UNIFIED IDEOGRAPH-#418A, IRGKangXi=0861.420, Def{unhusked glutinous rice, to sow seeds}
418B CJK UNIFIED IDEOGRAPH-#418B, IRGKangXi=0861.441, Def{(ancient form of U+79CB 秋) the autumn, the fall of the year, a season, a time}
418C CJK UNIFIED IDEOGRAPH-#418C, IRGKangXi=0861.441
418D CJK UNIFIED IDEOGRAPH-#418D, IRGKangXi=0861.450, Def{cut crops (not yet collected and tied up)}
418E CJK UNIFIED IDEOGRAPH-#418E, IRGKangXi=0862.020, Def{grass and grains producing no fruit}
418F CJK UNIFIED IDEOGRAPH-#418F, IRGKangXi=0862.060, Def{a kind of rice plant (not glutinous and with purple colored stalk), (same as 糞) to apply fertilizers}
4190 CJK UNIFIED IDEOGRAPH-#4190, IRGKangXi=0862.190, Def{(ancient form of 國) a country; a nation; a kingdom; a state}
4191 CJK UNIFIED IDEOGRAPH-#4191, IRGKangXi=0862.230, Def{a little protruded}
4192 CJK UNIFIED IDEOGRAPH-#4192, IRGKangXi=0863.101, Def{(corrupted form of 究) to examine; to study; to investigate exhaustively}
4193 CJK UNIFIED IDEOGRAPH-#4193, IRGKangXi=0863.120, Def{a hole; a cave; a pit, to dig the ground and build underground living quarters, to make smooth of the leather}
4194 CJK UNIFIED IDEOGRAPH-#4194, IRGKangXi=0863.130, Def{to pierce through; to penetrate or bore through; to thread, to cross; (Cant.) hole, hollow, cavity}
4195 CJK UNIFIED IDEOGRAPH-#4195, IRGKangXi=0863.160, Def{to pierce through; to penetrate through; to cross, extremely; to the extreme, a hole; an opening; an aperture, with no door or window}
4196 CJK UNIFIED IDEOGRAPH-#4196, IRGKangXi=0863.190, Def{a big house, (same as 宏) great; vast; wide; ample}
4197 CJK UNIFIED IDEOGRAPH-#4197, IRGKangXi=0864.130, Def{deep and dark; profound, (same as 窈) tranquil; placid; serene, soft and pleasing; plausible; exquisite; very pleasant}
4198 CJK UNIFIED IDEOGRAPH-#4198, IRGKangXi=0864.190, Def{acupuncture, narrow and protrude, (non-classical form of 拶) a torture device in old China consisting of several contracting wooden sticks, in between which the fingers of a suspect are placed and pressed to extort confessions}
4199 CJK UNIFIED IDEOGRAPH-#4199, IRGKangXi=0864.380, Def{(a non-classical form) (same as standard form of 7AB0 窯) a kiln; a brick furnace; a pottery, a coal shaft, a cave -- for human dwelling}
419A CJK UNIFIED IDEOGRAPH-#419A, IRGKangXi=0864.390, Def{to go through; to penetrate; passed through, (same as 洞) a cave; a hole}
419B CJK UNIFIED IDEOGRAPH-#419B, IRGKangXi=0865.020, Def{something in the cave}
419C CJK UNIFIED IDEOGRAPH-#419C, IRGKangXi=0865.040, Def{empty; hollow, unreal, high and vast, the space}
419D CJK UNIFIED IDEOGRAPH-#419D, IRGKangXi=0865.050, Def{a cave; a hole, to bore through or drill a cave as a dwelling}
419E CJK UNIFIED IDEOGRAPH-#419E, IRGKangXi=0865.060, Def{deep and dark; profound, far; vast, obscure, mysterious}
419F CJK UNIFIED IDEOGRAPH-#419F, IRGKangXi=0865.100, Def{corresponding; equivalent, considerable, appropriate, (same as 凹) indented; a hollow, concave}
41A0 CJK UNIFIED IDEOGRAPH-#41A0, IRGKangXi=0865.170, Def{(non-classical form of 宦) a government official; the government service, castrated}
41A1 CJK UNIFIED IDEOGRAPH-#41A1, IRGKangXi=0865.260, Def{a cave; a hole}
41A2 CJK UNIFIED IDEOGRAPH-#41A2, IRGKangXi=0865.280, Def{to penetrate; to pierce through, a hole; an aperture; an opening, a cave}
41A3 CJK UNIFIED IDEOGRAPH-#41A3, IRGKangXi=0866.170, Def{(same as 宸) abode of the emperor, a large mansion}
41A4 CJK UNIFIED IDEOGRAPH-#41A4, IRGKangXi=0866.220, Def{(same as 穿) to pierce through; to penetrate or bore through; to wear, to cross}
41A5 CJK UNIFIED IDEOGRAPH-#41A5, IRGKangXi=0866.230, Def{(same as 䆤) (non-classical form of 穿) to pierce through; to penetrate or bore through; to wear, to cross}
41A6 CJK UNIFIED IDEOGRAPH-#41A6, IRGKangXi=0866.340, Def{(a corrupted form) (same as 深) deep; profound, the chimney; stack (on the top of a cooking stove or furnace), to bury the coffin of a dead person}
41A7 CJK UNIFIED IDEOGRAPH-#41A7, IRGKangXi=0866.360, Def{(same as 窩) a cave; a den, living quarters; a house, to hide; to harbor}
41A8 CJK UNIFIED IDEOGRAPH-#41A8, IRGKangXi=0866.490, Def{(same as 冥) the sky; the heaven; the void, Nature; God, big; large; great, light; bright}
41A9 CJK UNIFIED IDEOGRAPH-#41A9, IRGKangXi=0866.500, Def{(same as 冥) dark; obscure; dim, far and high, deep; profound, the unseen world}
41AA CJK UNIFIED IDEOGRAPH-#41AA, IRGKangXi=0866.560, Def{color of the light; brilliance or gleaming}
41AB CJK UNIFIED IDEOGRAPH-#41AB, IRGKangXi=0866.570, Def{(same as U+7A97 窗) window}
41AC CJK UNIFIED IDEOGRAPH-#41AC, IRGKangXi=0867.260, Def{(a non-classical form) (same as standard form 霣) (interchangeable 隕) to fall down}
41AD CJK UNIFIED IDEOGRAPH-#41AD, IRGKangXi=0867.280, Def{cave; inside of a cave}
41AE CJK UNIFIED IDEOGRAPH-#41AE, IRGKangXi=0867.351, Def{(same as ancient form of 浸) to dip; to immerse; to soak, gradual; gradually, name of a place in today's Henan Province}
41AF CJK UNIFIED IDEOGRAPH-#41AF, IRGKangXi=0867.390, Def{a short shaped face, charming; beautiful; delicate}
41B0 CJK UNIFIED IDEOGRAPH-#41B0, IRGKangXi=0867.400, Def{a cave; a tunnel; a grotto, name of a mountain}
41B1 CJK UNIFIED IDEOGRAPH-#41B1, IRGKangXi=0867.450, Def{big; large; thin; light, rugged; uneven, a deep cave}
41B2 CJK UNIFIED IDEOGRAPH-#41B2, IRGKangXi=0868.110, Def{a spacious house, emptiness; (Cant.) an intensifier}
41B3 CJK UNIFIED IDEOGRAPH-#41B3, IRGKangXi=0868.120, Def{(same as non-classical form 邃) far distant; remote, the extreme; the farthest}
41B4 CJK UNIFIED IDEOGRAPH-#41B4, IRGKangXi=0868.211, Def{(abbreviated form)}
41B5 CJK UNIFIED IDEOGRAPH-#41B5, IRGKangXi=0868.240, Def{an echo, a high and deep; large; big; specious house}
41B6 CJK UNIFIED IDEOGRAPH-#41B6, IRGKangXi=0868.260, Def{noise of a mouse in the cave}
41B7 CJK UNIFIED IDEOGRAPH-#41B7, IRGKangXi=0868.280, Def{a hole; an opening; a aperture; deep; far and profound}
41B8 CJK UNIFIED IDEOGRAPH-#41B8, IRGKangXi=0868.290, Def{spacious; capacious, sound (of the house), a picture (on silk) scroll}
41B9 CJK UNIFIED IDEOGRAPH-#41B9, IRGKangXi=0868.310, Def{empty; hollow, dark; obscure}
41BA CJK UNIFIED IDEOGRAPH-#41BA, IRGKangXi=0868.370, Def{whirling of the flowing water, (interchangeable 潘) water in which rice has been washed}
41BB CJK UNIFIED IDEOGRAPH-#41BB, IRGKangXi=0868.390, Def{(corrupted form of 竅) a hole; a cavity; (the mind's pores, the crux; key points}
41BC CJK UNIFIED IDEOGRAPH-#41BC, IRGKangXi=0868.450, Def{(non-classical form of standard form 窠) a nest; hole; a den; burrow, a dwelling for people}
41BD CJK UNIFIED IDEOGRAPH-#41BD, IRGKangXi=0869.040, Def{a hole; an opening, a cave, empty; hollow}
41BE CJK UNIFIED IDEOGRAPH-#41BE, IRGKangXi=0869.080, Def{big; large; thin; light, rugged; uneven, a deep cave}
41BF CJK UNIFIED IDEOGRAPH-#41BF, IRGKangXi=0869.100, Def{(same as 寱) (standard form of 囈) to talk in sleep; somniloquy}
41C0 CJK UNIFIED IDEOGRAPH-#41C0, IRGKangXi=0869.200, Def{(non-classical form) ornaments of fine feather or fur (for decoration)}
41C1 CJK UNIFIED IDEOGRAPH-#41C1, IRGKangXi=0870.100, Def{a cave; a hole}
41C2 CJK UNIFIED IDEOGRAPH-#41C2, IRGKangXi=0870.180, Def{(ancient form) fault; sin}
41C3 CJK UNIFIED IDEOGRAPH-#41C3, IRGKangXi=0870.210, Def{(same as 竢) (ancient form of 俟) to wait for, until; when; as soon as}
41C4 CJK UNIFIED IDEOGRAPH-#41C4, IRGKangXi=0870.231
41C5 CJK UNIFIED IDEOGRAPH-#41C5, IRGKangXi=0870.240, Def{to stand still}
41C6 CJK UNIFIED IDEOGRAPH-#41C6, IRGKangXi=0870.271, Def{(abbreviated form) capacity unit in France (Decalitre) equal to ten times of one litre}
41C7 CJK UNIFIED IDEOGRAPH-#41C7, IRGKangXi=0870.320, Def{(corrupted form of 氓) the people; the populace, rascal; vagabond}
41C8 CJK UNIFIED IDEOGRAPH-#41C8, IRGKangXi=0870.350, Def{to put into a cupboard or closet; to put into the proper place; to put away, to put away for safekeeping, to collect and keep}
41C9 CJK UNIFIED IDEOGRAPH-#41C9, IRGKangXi=0871.031, Def{hectolitre, equal to one hundred times of one litre}
41CA CJK UNIFIED IDEOGRAPH-#41CA, IRGKangXi=0871.031
41CB CJK UNIFIED IDEOGRAPH-#41CB, IRGKangXi=0871.060, Def{name of a god}
41CC CJK UNIFIED IDEOGRAPH-#41CC, IRGKangXi=0871.170, Def{to stand, to erect; to create; to start}
41CD CJK UNIFIED IDEOGRAPH-#41CD, IRGKangXi=0872.050, Def{to wait, (same as 齪) tidy, prudent; careful; attentive, a person's name}
41CE CJK UNIFIED IDEOGRAPH-#41CE, IRGKangXi=0872.100, Def{to startle; to amaze; to surprise, afraid; scared; fearful, respectful; reverent}
41CF CJK UNIFIED IDEOGRAPH-#41CF, IRGKangXi=0872.160, Def{to pile one upon another, trees fruits well and the branches hanging down}
41D0 CJK UNIFIED IDEOGRAPH-#41D0, IRGKangXi=0872.190, Def{(same as 搯 蒞) to arrive, from; by; through, to manage; to undertake, to follow, thin; few, distant, idle}
41D1 CJK UNIFIED IDEOGRAPH-#41D1, IRGKangXi=0872.210, Def{a standing short person, unstable in walking}
41D2 CJK UNIFIED IDEOGRAPH-#41D2, IRGKangXi=0873.050, Def{(ancient form of 屆) numerary adjunct for periodic terms or events, to arrive, to expire}
41D3 CJK UNIFIED IDEOGRAPH-#41D3, IRGKangXi=0873.120, Def{to wait for; to wait (same as 須) to stop at, to have to, a moment; a while, (interchangeable 需) to need; demand}
41D4 CJK UNIFIED IDEOGRAPH-#41D4, IRGKangXi=0873.150, Def{atrophy, paralysis; impotent, stand}
41D5 CJK UNIFIED IDEOGRAPH-#41D5, IRGKangXi=0873.210, Def{(same as 䇓) to wait for; to wait}
41D6 CJK UNIFIED IDEOGRAPH-#41D6, IRGKangXi=0878.040, Def{rope made of bamboo strips}
41D7 CJK UNIFIED IDEOGRAPH-#41D7, IRGKangXi=0878.140, Def{bamboo with long joints}
41D8 CJK UNIFIED IDEOGRAPH-#41D8, IRGKangXi=0878.150, Def{tool to collect rope; a spinning machine, spinning wheel; the tender shoots of the bamboo with a bitter flavour}
41D9 CJK UNIFIED IDEOGRAPH-#41D9, IRGKangXi=0878.230, Def{name of a variety of bamboo}
41DA CJK UNIFIED IDEOGRAPH-#41DA, IRGKangXi=0879.110, Def{(same as U+6534 攴) to tap; to rap, (same as U+64B2 撲) to pat; to beat; to strike; to dash; to smash}
41DB CJK UNIFIED IDEOGRAPH-#41DB, IRGKangXi=0879.120, Def{name of a variety of bamboo}
41DC CJK UNIFIED IDEOGRAPH-#41DC, IRGKangXi=0879.140, Def{small bamboo, a kind of bamboo}
41DD CJK UNIFIED IDEOGRAPH-#41DD, IRGKangXi=0879.191, Def{(abbreviated form) (same as 㦰) to break off (relations); to sever, to exterminate; to annihilate; to wipe out, to pierce; to stab; to irritate; to hurt, to hold weapons, agricultural implements; far tools, sharp will; eager intention; determination}
41DE CJK UNIFIED IDEOGRAPH-#41DE, IRGKangXi=0879.220, Def{name of a variety of bamboo; sweet bamboo, large bamboo}
41DF CJK UNIFIED IDEOGRAPH-#41DF, IRGKangXi=0879.250, Def{(same as 腱) tendon}
41E0 CJK UNIFIED IDEOGRAPH-#41E0, IRGKangXi=0879.280, Def{wrench to adjust the string on stringed instrument}
41E1 CJK UNIFIED IDEOGRAPH-#41E1, IRGKangXi=0880.080, Def{(same as 杼) the shuttle of a loom}
41E2 CJK UNIFIED IDEOGRAPH-#41E2, IRGKangXi=0880.130, Def{(standard form 苦) bamboo, bitter bamboo, usually used as the material for making an umbrella's handle, (same as 罟) net}
41E3 CJK UNIFIED IDEOGRAPH-#41E3, IRGKangXi=0880.160, Def{bamboo with white bark, (same as 籋) tongs; pincers; tweezers, weary; tired; fatigued, small box}
41E4 CJK UNIFIED IDEOGRAPH-#41E4, IRGKangXi=0881.180, Def{sharp; acute, clever, vigorous}
41E5 CJK UNIFIED IDEOGRAPH-#41E5, IRGKangXi=0881.210, Def{(same as 笮) narrow, boards laid across rafters, an arrow bag, a kind of liquor container, to squeeze; to press, (same as 筰) rope made from bamboo-splints, (interchangeable 鑿) to bore or pierce through}
41E6 CJK UNIFIED IDEOGRAPH-#41E6, IRGKangXi=0881.220, Def{a kind of bamboo, colorless bamboo, bamboo shoot}
41E7 CJK UNIFIED IDEOGRAPH-#41E7, IRGKangXi=0882.020, Def{to fill an opening with bamboo, (ancient form of 箕) a winnowing basket; a sieve, a dust basket; dustpan}
41E8 CJK UNIFIED IDEOGRAPH-#41E8, IRGKangXi=0882.030, Def{a bamboo basket for food containers (such as cup; plate; dish and bowl, etc.) used in ancient times, a basket for chopsticks, bamboo basket used to filter or to strain out the wine, a sail made of a thin and long strip of bamboo}
41E9 CJK UNIFIED IDEOGRAPH-#41E9, IRGKangXi=0882.060, Def{long, a big raft, a kind of equipment made of bamboo used to catch fishes}
41EA CJK UNIFIED IDEOGRAPH-#41EA, IRGKangXi=0882.070, Def{name of a variety of bamboo}
41EB CJK UNIFIED IDEOGRAPH-#41EB, IRGKangXi=0882.100, Def{bamboo-splints; laths, to plait, (same as 篦) a comb; especially a fine-toothed one)}
41EC CJK UNIFIED IDEOGRAPH-#41EC, IRGKangXi=0882.130, Def{thin bamboo laths knitted sail, a thin and long strip of bamboo; books in ancient style, to set sail; to depart}
41ED CJK UNIFIED IDEOGRAPH-#41ED, IRGKangXi=0882.170, Def{containers made of thin and long strip of bamboo or willow branches}
41EE CJK UNIFIED IDEOGRAPH-#41EE, IRGKangXi=0882.210, Def{a bamboo mat used in ancient times, (interchangeable with U+88B5 衽) a sleeping mat}
41EF CJK UNIFIED IDEOGRAPH-#41EF, IRGKangXi=0884.020, Def{a kind of bamboo with patterns on the head}
41F0 CJK UNIFIED IDEOGRAPH-#41F0, IRGKangXi=0884.050, Def{a bamboo binded torch, a kind of bamboo with patterns on the bark}
41F1 CJK UNIFIED IDEOGRAPH-#41F1, IRGKangXi=0884.110, Def{(same as 笝) a hawser; a cable; a bamboo rope used to tie on a boat, to mend a bamboo fence}
41F2 CJK UNIFIED IDEOGRAPH-#41F2, IRGKangXi=0884.150, Def{(simplified form of 筴) tongs; tweezers; pincers, to use pressure upon; to force, a kind of grass used for divination in ancient times, (same as 策) a whip, a plan}
41F3 CJK UNIFIED IDEOGRAPH-#41F3, IRGKangXi=0884.181, Def{(abbreviated form of U+7B8B 箋) a memorandum tablet, slip of paper, fancy notepaper, a note, a document}
41F4 CJK UNIFIED IDEOGRAPH-#41F4, IRGKangXi=0884.181
41F5 CJK UNIFIED IDEOGRAPH-#41F5, IRGKangXi=0884.190, Def{one of the wind instruments; a bamboo flute with seven holes}
41F6 CJK UNIFIED IDEOGRAPH-#41F6, IRGKangXi=0884.280, Def{a small rafters made of bamboo which project form the eaves and support the tiles, a kind of bamboo}
41F7 CJK UNIFIED IDEOGRAPH-#41F7, IRGKangXi=0884.290, Def{in ancient times; to separated (to rip of; to cut open) the contract or agreement on slips of bamboo and give to both parties as evidence}
41F8 CJK UNIFIED IDEOGRAPH-#41F8, IRGKangXi=0884.330, Def{a bamboo mat, a kind of bamboo, bamboo ware, a seat cushion in a carriage}
41F9 CJK UNIFIED IDEOGRAPH-#41F9, IRGKangXi=0885.060, Def{(same as 箘) a kind of bamboo}
41FA CJK UNIFIED IDEOGRAPH-#41FA, IRGKangXi=0885.070, Def{(same as 豆) beans and peas collectively, a vessel like a platter, made of wood, bronze, or porcelain, used for holding food in sacrifices, etc.}
41FB CJK UNIFIED IDEOGRAPH-#41FB, IRGKangXi=0885.080, Def{a kind of bamboo, broom; a besom}
41FC CJK UNIFIED IDEOGRAPH-#41FC, IRGKangXi=0885.090, Def{a bamboo vessel used to catch fishes}
41FD CJK UNIFIED IDEOGRAPH-#41FD, IRGKangXi=0885.130, Def{a coarse bamboo mat}
41FE CJK UNIFIED IDEOGRAPH-#41FE, IRGKangXi=0886.010, Def{a flute with 23 tubes; 4 inches long (of bamboo, etc.); a pipe; a wind instrument, (interchangeable 言) speech; words, to say, a dialect; language}
41FF CJK UNIFIED IDEOGRAPH-#41FF, IRGKangXi=0886.231
4200 CJK UNIFIED IDEOGRAPH-#4200, IRGKangXi=0885.020, Def{bamboo ware (used to hold food or clothes)}
4201 CJK UNIFIED IDEOGRAPH-#4201, IRGKangXi=0886.260, Def{tools used on a boat}
4202 CJK UNIFIED IDEOGRAPH-#4202, IRGKangXi=0886.290, Def{a kind of bamboo}
4203 CJK UNIFIED IDEOGRAPH-#4203, IRGKangXi=0886.370, Def{a kind of bamboo, hem; margin; edge of bamboo}
4204 CJK UNIFIED IDEOGRAPH-#4204, IRGKangXi=0887.010, Def{(same as 筨) a kind of solid bamboo}
4205 CJK UNIFIED IDEOGRAPH-#4205, IRGKangXi=0887.050, Def{(same as 籆) tools to unreel silk}
4206 CJK UNIFIED IDEOGRAPH-#4206, IRGKangXi=0887.070, Def{a shallow and long shaped bamboo basket}
4207 CJK UNIFIED IDEOGRAPH-#4207, IRGKangXi=0887.100, Def{(same as U+7F69 罩) a cover, a shade, a basket used to catch fish}
4208 CJK UNIFIED IDEOGRAPH-#4208, IRGKangXi=0887.130, Def{a kind of bamboo, exuberant and flourishing of bamboo, (same as 笆) a barrier made of bamboo or willow branch; bamboo fence, a mat}
4209 CJK UNIFIED IDEOGRAPH-#4209, IRGKangXi=0887.190, Def{(same as 箑) a fan, (same as 翣) feathers adorning a coffin, bamboo ware}
420A CJK UNIFIED IDEOGRAPH-#420A, IRGKangXi=0889.050, Def{name of a variety of bamboo; (Cant.) 䈊仔, a useless fellow}
420B CJK UNIFIED IDEOGRAPH-#420B, IRGKangXi=0889.120, Def{name of a variety of bamboo, a cover made of bamboo}
420C CJK UNIFIED IDEOGRAPH-#420C, IRGKangXi=0889.150, Def{name of a variety of bamboo, submerged; fertilized by water}
420D CJK UNIFIED IDEOGRAPH-#420D, IRGKangXi=0889.200, Def{name of a variety of bamboo}
420E CJK UNIFIED IDEOGRAPH-#420E, IRGKangXi=0890.020, Def{(same as 葉) a leaf; the page of a book, a leaf of a door, a lobe of the lungs or liver}
420F CJK UNIFIED IDEOGRAPH-#420F, IRGKangXi=0890.070, Def{name of a variety of bamboo, a woven tool}
4210 CJK UNIFIED IDEOGRAPH-#4210, IRGKangXi=0890.100, Def{an arched frame made of bamboo of wood board, a covering, mats woven from bamboo and other leaves top on vehicles, (same as 簂 槶) a bamboo chest; a wicker chest, a woman's headdress}
4211 CJK UNIFIED IDEOGRAPH-#4211, IRGKangXi=0890.140, Def{tools to unreel silk}
4212 CJK UNIFIED IDEOGRAPH-#4212, IRGKangXi=0890.150, Def{fragile; tender bamboo, (interchangeable 蹐) a kind of spring fishing-net, an equipment used to pull or drag (water plant, mud, etc.) out of the water}
4213 CJK UNIFIED IDEOGRAPH-#4213, IRGKangXi=0890.220, Def{a kind of bamboo with a red skin; it is used for fine mats and other purposes, an ancient musical instrument which was used to give the signal to cease playing}
4214 CJK UNIFIED IDEOGRAPH-#4214, IRGKangXi=0890.240, Def{(same as 笳) whistle made of reed without holes for fingering}
4215 CJK UNIFIED IDEOGRAPH-#4215, IRGKangXi=0890.260, Def{the metal tongue in the reed instruments, (interchangeable 匙) key, bamboo ware, (same as 筓) a clasp; a hair-pin with flat spoon-shaped ends, (same as 椸) a clothes-horse; a rack for clothes}
4216 CJK UNIFIED IDEOGRAPH-#4216, IRGKangXi=0890.270, Def{name of a variety of bamboo}
4217 CJK UNIFIED IDEOGRAPH-#4217, IRGKangXi=0891.050, Def{name of a variety of bamboo, bamboo mats, (interchangeable 葰) a cover, a shade, a basket used to catch fishes}
4218 CJK UNIFIED IDEOGRAPH-#4218, IRGKangXi=0891.220, Def{name of a variety of bamboo}
4219 CJK UNIFIED IDEOGRAPH-#4219, IRGKangXi=0892.010, Def{bamboo to wither; to dry up and die}
421A CJK UNIFIED IDEOGRAPH-#421A, IRGKangXi=0892.020, Def{a bamboo shoot, skin (bark) of a bamboo shoot, (same as 箈 苔) moss; lichen}
421B CJK UNIFIED IDEOGRAPH-#421B, IRGKangXi=0892.090, Def{small bamboo whose joints are three feet apart, used for arrows}
421C CJK UNIFIED IDEOGRAPH-#421C, IRGKangXi=0892.110, Def{a thread used by carpenters for marking, (interchangeable 沁) to leak; to soak through}
421D CJK UNIFIED IDEOGRAPH-#421D, IRGKangXi=0892.200, Def{name of a variety of bamboo, a kind of winnowing basket}
421E CJK UNIFIED IDEOGRAPH-#421E, IRGKangXi=0892.220, Def{(same as 竺) ancient name of India, a Chinese family name, (interchangeable 篤) honest; straightforward}
421F CJK UNIFIED IDEOGRAPH-#421F, IRGKangXi=0892.230, Def{broken sound, to cut grass}
4220 CJK UNIFIED IDEOGRAPH-#4220, IRGKangXi=0893.030, Def{books and volumes, letters and correspondence, to snap; to break; broken bamboo}
4221 CJK UNIFIED IDEOGRAPH-#4221, IRGKangXi=0893.070, Def{name of a variety of bamboo, (same as 檧) a small basket or a small cage}
4222 CJK UNIFIED IDEOGRAPH-#4222, IRGKangXi=0893.080, Def{name of a variety of bamboo, a thin and long strip of bamboo form making basket}
4223 CJK UNIFIED IDEOGRAPH-#4223, IRGKangXi=0893.130, Def{name of a variety of bamboo}
4224 CJK UNIFIED IDEOGRAPH-#4224, IRGKangXi=0893.170, Def{(same as 鉗) pincers; tweezers; tongs}
4225 CJK UNIFIED IDEOGRAPH-#4225, IRGKangXi=0893.190, Def{(non-classical form of 筋) tendons; sinews; muscles, name of a variety of bamboo}
4226 CJK UNIFIED IDEOGRAPH-#4226, IRGKangXi=0893.200, Def{a curved bamboo whip, thin branches of a tree}
4227 CJK UNIFIED IDEOGRAPH-#4227, IRGKangXi=0893.260, Def{(corrupted form) skin of bamboo shoots, bark of bamboo}
4228 CJK UNIFIED IDEOGRAPH-#4228, IRGKangXi=0893.291
4229 CJK UNIFIED IDEOGRAPH-#4229, IRGKangXi=0893.291
422A CJK UNIFIED IDEOGRAPH-#422A, IRGKangXi=0893.460, Def{a small, tall and no ears basket made of bamboo used to hold grains}
422B CJK UNIFIED IDEOGRAPH-#422B, IRGKangXi=0894.080, Def{(same as 笝) a hawser; a cable; a bamboo rope used to tie on a boat}
422C CJK UNIFIED IDEOGRAPH-#422C, IRGKangXi=0894.100, Def{a small; thin strip (of bamboo) net}
422D CJK UNIFIED IDEOGRAPH-#422D, IRGKangXi=0894.160, Def{name of a variety of bamboo, bamboo with high joints}
422E CJK UNIFIED IDEOGRAPH-#422E, IRGKangXi=0894.190, Def{a woven instrument; a knitting tool, name of a variety of bamboo}
422F CJK UNIFIED IDEOGRAPH-#422F, IRGKangXi=0894.200, Def{name of a variety of bamboo, utensils; implements}
4230 CJK UNIFIED IDEOGRAPH-#4230, IRGKangXi=0894.240, Def{a brush for washing kitchen utensils, a bamboo ware for holding rice used in ancient times, a basket for washing rice, a small bucket for chopsticks, (same as 梢) the tip of a branch or things of similar shape, the end of something, the rudder, (interchangeable 筲)}
4231 CJK UNIFIED IDEOGRAPH-#4231, IRGKangXi=0894.280, Def{an open bamboo basket used to feed domestic animals (cow, hog, etc.)}
4232 CJK UNIFIED IDEOGRAPH-#4232, IRGKangXi=0894.300, Def{a bamboo basket for fishing, name of a variety of bamboo, thin and flat slips of bamboo used for weaving purpose}
4233 CJK UNIFIED IDEOGRAPH-#4233, IRGKangXi=0894.340, Def{a window-leaf; something made of strips of bamboo to block sunlight}
4234 CJK UNIFIED IDEOGRAPH-#4234, IRGKangXi=0895.010, Def{a cage; a basket; a noose}
4235 CJK UNIFIED IDEOGRAPH-#4235, IRGKangXi=0895.020, Def{luxuriant; exuberant of bamboo}
4236 CJK UNIFIED IDEOGRAPH-#4236, IRGKangXi=0895.200, Def{an arrow; a dart, asparagus fern}
4237 CJK UNIFIED IDEOGRAPH-#4237, IRGKangXi=0895.290, Def{(same as standard form 格) a bamboo fence}
4238 CJK UNIFIED IDEOGRAPH-#4238, IRGKangXi=0895.400, Def{a big chest, a big basket for holding rice}
4239 CJK UNIFIED IDEOGRAPH-#4239, IRGKangXi=0896.010, Def{(same as 籔) a bamboo ware for washing rice, a measuring unit used in ancient times; equal to 16 Chinese peck}
423A CJK UNIFIED IDEOGRAPH-#423A, IRGKangXi=0896.120, Def{bamboo, a chest; a box}
423B CJK UNIFIED IDEOGRAPH-#423B, IRGKangXi=0896.140, Def{heavy bamboo mats used for deep water fishing}
423C CJK UNIFIED IDEOGRAPH-#423C, IRGKangXi=0896.210, Def{(same as 篾) name of a variety of bamboo, a thin and long strip of bamboo for making baskets, etc.; thin (bamboo) laths, a kind of small bamboo with pulp}
423D CJK UNIFIED IDEOGRAPH-#423D, IRGKangXi=0897.020, Def{name of a variety of bamboo; with short joints and white bark; big ones used as pole for punting a boat and roots used as medicine}
423E CJK UNIFIED IDEOGRAPH-#423E, IRGKangXi=0897.030, Def{(same as 稍) move a little; shake slightly, (same as 梢) the tip of a branch or things of similar shape, the end of rudder or helm, a besom for rice}
423F CJK UNIFIED IDEOGRAPH-#423F, IRGKangXi=0897.050, Def{belt of the canvas top on vehicles}
4240 CJK UNIFIED IDEOGRAPH-#4240, IRGKangXi=0897.100, Def{rectangular chest or box woven from bamboo strips (or wicker)}
4241 CJK UNIFIED IDEOGRAPH-#4241, IRGKangXi=0897.130, Def{name of a variety of bamboo, bamboo ware}
4242 CJK UNIFIED IDEOGRAPH-#4242, IRGKangXi=0897.260, Def{a basket for earth or soil, regulations; law, model}
4243 CJK UNIFIED IDEOGRAPH-#4243, IRGKangXi=0897.330, Def{sliced open bamboo with the joints used as an oar, a board made equipment with angles; children used to practice writing and taking notes in ancient times, mats, bamboo}
4244 CJK UNIFIED IDEOGRAPH-#4244, IRGKangXi=0897.350, Def{name of a variety of bamboo}
4245 CJK UNIFIED IDEOGRAPH-#4245, IRGKangXi=0898.040, Def{small bamboo whose joints are three feet apart, used for arrows, name of a variety of bamboo}
4246 CJK UNIFIED IDEOGRAPH-#4246, IRGKangXi=0898.240, Def{name of a variety of bamboo}
4247 CJK UNIFIED IDEOGRAPH-#4247, IRGKangXi=0898.371
4248 CJK UNIFIED IDEOGRAPH-#4248, IRGKangXi=0898.400, Def{a kind of bamboo with a red skin; it is used for fine mats and other purposes}
4249 CJK UNIFIED IDEOGRAPH-#4249, IRGKangXi=0898.420, Def{(interchangeable 觚) a writing-tablet; a square piece of wood, a corner; an angle; an edge}
424A CJK UNIFIED IDEOGRAPH-#424A, IRGKangXi=0898.450, Def{bamboo ware, awning in front of a cart, covering the horse or mule in the shafts; canvas top on vehicles}
424B CJK UNIFIED IDEOGRAPH-#424B, IRGKangXi=0899.030, Def{a variety of bamboo shoots that sprout in winter}
424C CJK UNIFIED IDEOGRAPH-#424C, IRGKangXi=0899.070, Def{crude bamboo mats, a vessel for raising silk-worms}
424D CJK UNIFIED IDEOGRAPH-#424D, IRGKangXi=0899.130, Def{(non-classical form) to observe; to watch; to examine}
424E CJK UNIFIED IDEOGRAPH-#424E, IRGKangXi=0899.180, Def{a cover, a shade, a basket used to catch fish}
424F CJK UNIFIED IDEOGRAPH-#424F, IRGKangXi=0899.200, Def{name of a variety of bamboo}
4250 CJK UNIFIED IDEOGRAPH-#4250, IRGKangXi=0899.210, Def{a thin and long strip of bamboo for making baskets, etc.; thin (bamboo) laths, name of a variety of bamboo, (corrupted form of U+7C35 簬) used for arrows}
4251 CJK UNIFIED IDEOGRAPH-#4251, IRGKangXi=0900.020, Def{black skinned bamboo}
4252 CJK UNIFIED IDEOGRAPH-#4252, IRGKangXi=0900.030, Def{(same as 籓) a winnow, a container for dust, (same as 藩) a bamboo fence; a barrier, a bamboo basket with handles for carrying earth or dirt, family name}
4253 CJK UNIFIED IDEOGRAPH-#4253, IRGKangXi=0900.090, Def{mattress made of bamboo strips, bed}
4254 CJK UNIFIED IDEOGRAPH-#4254, IRGKangXi=0900.130, Def{bamboo slips, crude; coarse bamboo mats}
4255 CJK UNIFIED IDEOGRAPH-#4255, IRGKangXi=0900.170, Def{a bamboo hat, a bamboo shade or covering, name of a variety of bamboo}
4256 CJK UNIFIED IDEOGRAPH-#4256, IRGKangXi=0900.230, Def{(same as 笭) framework of a carriage, a door-screen; a railing in the front and on both sides of a carriage, a stand; frame; rack in a cabin of a ship, a bamboo cage; a bamboo basket with a narrow opening}
4257 CJK UNIFIED IDEOGRAPH-#4257, IRGKangXi=0900.260, Def{a receptacle or container, as a cup, ladle, spoon, etc.}
4258 CJK UNIFIED IDEOGRAPH-#4258, IRGKangXi=0900.270, Def{to pierce; to stab with a sharp wooden stick or a piece of bamboo slip}
4259 CJK UNIFIED IDEOGRAPH-#4259, IRGKangXi=0901.160, Def{name of a variety of bamboo; big bamboo}
425A CJK UNIFIED IDEOGRAPH-#425A, IRGKangXi=0901.250, Def{name of a variety of bamboo, a bamboo shoot, a bamboo tube}
425B CJK UNIFIED IDEOGRAPH-#425B, IRGKangXi=0901.310, Def{bamboo basket for washing rice, implements used to move the silkworm}
425C CJK UNIFIED IDEOGRAPH-#425C, IRGKangXi=0901.350, Def{young and tender bamboo}
425D CJK UNIFIED IDEOGRAPH-#425D, IRGKangXi=0901.400, Def{bamboo ware}
425E CJK UNIFIED IDEOGRAPH-#425E, IRGKangXi=0901.410, Def{a box; a chest made of bamboo; bamboo ware, name of a variety of bamboo}
425F CJK UNIFIED IDEOGRAPH-#425F, IRGKangXi=0901.450, Def{a bamboo basket used to catch fishes}
4260 CJK UNIFIED IDEOGRAPH-#4260, IRGKangXi=0902.050, Def{name of a variety of bamboo; leaves to make covering and the stems make arrows}
4261 CJK UNIFIED IDEOGRAPH-#4261, IRGKangXi=0902.160, Def{name of a variety of bamboo, a towrope used to tow a boat against the current in swift stream from the bank}
4262 CJK UNIFIED IDEOGRAPH-#4262, IRGKangXi=0902.200, Def{a sieve; a screen; a sifter; a strainer}
4263 CJK UNIFIED IDEOGRAPH-#4263, IRGKangXi=0902.220, Def{slips of bamboo provided for writing in ancient times}
4264 CJK UNIFIED IDEOGRAPH-#4264, IRGKangXi=0902.321, Def{(a simplified form of 籔) a bamboo ware for washing rice, a measuring unit used in ancient times; equal to 16 Chinese peck}
4265 CJK UNIFIED IDEOGRAPH-#4265, IRGKangXi=0902.440, Def{(same as 檧) a small basket for chopsticks}
4266 CJK UNIFIED IDEOGRAPH-#4266, IRGKangXi=0903.200, Def{name of a variety of bamboo, a farm tool used to collect rice plants}
4267 CJK UNIFIED IDEOGRAPH-#4267, IRGKangXi=0903.220, Def{sound of bamboo, name of a variety of bamboo}
4268 CJK UNIFIED IDEOGRAPH-#4268, IRGKangXi=0903.270, Def{bamboo joints, name of a variety of bamboo, small bamboo}
4269 CJK UNIFIED IDEOGRAPH-#4269, IRGKangXi=0903.320, Def{a bamboo ware; a cradle}
426A CJK UNIFIED IDEOGRAPH-#426A, IRGKangXi=0903.340, Def{to grind; by turning a mill to get the chaff, bran or husks off, a mill}
426B CJK UNIFIED IDEOGRAPH-#426B, IRGKangXi=0903.360, Def{name of a variety of bamboo; used as a musical instrument}
426C CJK UNIFIED IDEOGRAPH-#426C, IRGKangXi=0903.370, Def{coarse bamboo mats}
426D CJK UNIFIED IDEOGRAPH-#426D, IRGKangXi=0904.010, Def{name of a variety of bamboo, covering made of bamboo strips}
426E CJK UNIFIED IDEOGRAPH-#426E, IRGKangXi=0904.390, Def{to plant; to set up, to damage, to lose, weak}
426F CJK UNIFIED IDEOGRAPH-#426F, IRGKangXi=0905.010, Def{dried bamboo}
4270 CJK UNIFIED IDEOGRAPH-#4270, IRGKangXi=0905.040, Def{bamboo shoots, (same as 筊) a rope made of bamboo strips, a kind of bamboo device used in divination}
4271 CJK UNIFIED IDEOGRAPH-#4271, IRGKangXi=0905.090, Def{a cradle}
4272 CJK UNIFIED IDEOGRAPH-#4272, IRGKangXi=0905.120, Def{thin and flat slips of bamboo used for weaving purpose}
4273 CJK UNIFIED IDEOGRAPH-#4273, IRGKangXi=0905.170, Def{name of a variety of bamboo, a domicile; record of the population}
4274 CJK UNIFIED IDEOGRAPH-#4274, IRGKangXi=0905.210, Def{a bamboo ware to collect and keep something in, a kind of bamboo ware used to dripping rice}
4275 CJK UNIFIED IDEOGRAPH-#4275, IRGKangXi=1427.020, Def{(same as 饌) to feed, to provide for, to prepare food; to eat and drink; delicacies, dainties}
4276 CJK UNIFIED IDEOGRAPH-#4276, IRGKangXi=0905.340, Def{a sail (of a boat), a utensil made of bamboo used to filter or to strain out wine}
4277 CJK UNIFIED IDEOGRAPH-#4277, IRGKangXi=0906.160, Def{a feather screen, a shade, imperial garden}
4278 CJK UNIFIED IDEOGRAPH-#4278, IRGKangXi=0906.200, Def{slips of bamboo for writing}
4279 CJK UNIFIED IDEOGRAPH-#4279, IRGKangXi=0906.270, Def{(same as 䉁) name of a variety of bamboo, bamboo ware}
427A CJK UNIFIED IDEOGRAPH-#427A, IRGKangXi=0906.390, Def{to deteriorate and become red; old rice; decaying rice, red rice; (Cant.) the smell of long-stored rice}
427B CJK UNIFIED IDEOGRAPH-#427B, IRGKangXi=0907.070, Def{red rice, coarse rice}
427C CJK UNIFIED IDEOGRAPH-#427C, IRGKangXi=0907.211, Def{(non-classical form of 料) materials; ingredients}
427D CJK UNIFIED IDEOGRAPH-#427D, IRGKangXi=0907.240, Def{(same as 粄) rice cake; cake made of glutinous rice}
427E CJK UNIFIED IDEOGRAPH-#427E, IRGKangXi=0907.290, Def{bad; poor quality of rice}
427F CJK UNIFIED IDEOGRAPH-#427F, IRGKangXi=0908.020, Def{(same as 糊) paste; to paste, sticky; glutinous, to stick}
4280 CJK UNIFIED IDEOGRAPH-#4280, IRGKangXi=0908.050, Def{(same as 䉿) (same as 糊) paste; to paste, sticky; glutinous, to stick}
4281 CJK UNIFIED IDEOGRAPH-#4281, IRGKangXi=0908.200, Def{(same as U+7C78 籸) leavings; refuse (from foodstuff, petroleum, oil, etc.; siftings, congee; rice gruel (the surface part); a kind of cooked rice}
4282 CJK UNIFIED IDEOGRAPH-#4282, IRGKangXi=0908.220, Def{poor quality of rice; bad rice cakes stick to each other}
4283 CJK UNIFIED IDEOGRAPH-#4283, IRGKangXi=0908.240, Def{mixing rice with broth, a grain of rice}
4284 CJK UNIFIED IDEOGRAPH-#4284, IRGKangXi=0908.270, Def{polished rice}
4285 CJK UNIFIED IDEOGRAPH-#4285, IRGKangXi=0908.280, Def{to soak rice}
4286 CJK UNIFIED IDEOGRAPH-#4286, IRGKangXi=0908.340, Def{cooked, dry rice flour, (same as 糗) cured dried grain; parched wheat or rice}
4287 CJK UNIFIED IDEOGRAPH-#4287, IRGKangXi=0909.110, Def{(same as 餔) to feed; to eat, (interchangeable 晡) time for supper, sunset}
4288 CJK UNIFIED IDEOGRAPH-#4288, IRGKangXi=0909.140, Def{(same as 酶) distiller's grains or yeast}
4289 CJK UNIFIED IDEOGRAPH-#4289, IRGKangXi=0909.170, Def{preserved fruit, cucumbers, gourds, etc.}
428A CJK UNIFIED IDEOGRAPH-#428A, IRGKangXi=0909.220, Def{well-boiled congee or rice gruel}
428B CJK UNIFIED IDEOGRAPH-#428B, IRGKangXi=0909.311, Def{(same as 妝) to adorn oneself, to disguise, to pretend}
428C CJK UNIFIED IDEOGRAPH-#428C, IRGKangXi=0909.311
428D CJK UNIFIED IDEOGRAPH-#428D, IRGKangXi=0909.350, Def{thick congee or porridge; well-boiled congee or gruel, (same as 黎) many; numerous}
428E CJK UNIFIED IDEOGRAPH-#428E, IRGKangXi=0910.050, Def{powder; flour, to roll round with the hand, thick congee or porridge}
428F CJK UNIFIED IDEOGRAPH-#428F, IRGKangXi=0910.070, Def{(non-classical form of U+7CC2 糝) mixing rice with broth, a grain of rice}
4290 CJK UNIFIED IDEOGRAPH-#4290, IRGKangXi=0910.120, Def{(same as 餛) fluffy stuffed dumplings; stuffed dumpling with delicate flour wrapping; ravioli}
4291 CJK UNIFIED IDEOGRAPH-#4291, IRGKangXi=0910.170, Def{(same as 餉 饟) pay, provisions, etc. for military or police, to entertain with food; to feast}
4292 CJK UNIFIED IDEOGRAPH-#4292, IRGKangXi=0910.201, Def{(same as 精) refined; polished (rice); unmixed, the essence, fine and delicate, keen; sharp}
4293 CJK UNIFIED IDEOGRAPH-#4293, IRGKangXi=0910.230, Def{to paste up; to attach to; to stickup; to glue}
4294 CJK UNIFIED IDEOGRAPH-#4294, IRGKangXi=0910.260, Def{polished rice; refined rice}
4295 CJK UNIFIED IDEOGRAPH-#4295, IRGKangXi=0910.270, Def{(same as 饘) well-boiled congee or gruel, thick, rich}
4296 CJK UNIFIED IDEOGRAPH-#4296, IRGKangXi=0910.300, Def{to eat rice-gruel mixed with meat}
4297 CJK UNIFIED IDEOGRAPH-#4297, IRGKangXi=0910.320, Def{a kind of grain; yellow color; not sticky, (same as 餭) fried puffy shredded, sugar-plums; sweetmeats}
4298 CJK UNIFIED IDEOGRAPH-#4298, IRGKangXi=0911.030, Def{a grain (of rice, etc.)}
4299 CJK UNIFIED IDEOGRAPH-#4299, IRGKangXi=0911.190, Def{cooked or well-done, ripe, fine rice}
429A CJK UNIFIED IDEOGRAPH-#429A, IRGKangXi=0911.270, Def{steamed dumplings}
429B CJK UNIFIED IDEOGRAPH-#429B, IRGKangXi=0912.090, Def{shreds; splinters (of rice)}
429C CJK UNIFIED IDEOGRAPH-#429C, IRGKangXi=0912.160, Def{(same as 糰) dumplings; doughnuts}
429D CJK UNIFIED IDEOGRAPH-#429D, IRGKangXi=0912.170, Def{to exile; to banish}
429E CJK UNIFIED IDEOGRAPH-#429E, IRGKangXi=0912.180, Def{to roll round with the hand; to paste up; to attach to; to stick up; to glue, poor quality of rice}
429F CJK UNIFIED IDEOGRAPH-#429F, IRGKangXi=0912.230, Def{adorned; beautified; made up; glossy, thick and sticky thing on the surface of congee; rice gruel}
42A0 CJK UNIFIED IDEOGRAPH-#42A0, IRGKangXi=0912.290, Def{(same as 氣 餼) to give a supply of grain for rations, grain, a sacrificial victim, explained as used of the living beast}
42A1 CJK UNIFIED IDEOGRAPH-#42A1, IRGKangXi=0912.330, Def{(same as non-classical form 饅) steamed bread; bread of any kind; steamed dumplings}
42A2 CJK UNIFIED IDEOGRAPH-#42A2, IRGKangXi=0912.391, Def{(same as 漿) thick fluid; starch; to starch}
42A3 CJK UNIFIED IDEOGRAPH-#42A3, IRGKangXi=0912.450, Def{light yellow dust-like fungoid growth on wine, etc., barley, chaff or husks of wheat (non-classical form of 餭) fried puffy shredded, sugar-plums; sweetmeats}
42A4 CJK UNIFIED IDEOGRAPH-#42A4, IRGKangXi=0912.470, Def{vegetable mixed with thick soup (broth), congee; gruel}
42A5 CJK UNIFIED IDEOGRAPH-#42A5, IRGKangXi=0913.010, Def{congee; porridge; rice gruel, mashed; rotten}
42A6 CJK UNIFIED IDEOGRAPH-#42A6, IRGKangXi=0913.040, Def{food (some food as glutinous rice tamale—made by wrapping the rice in broad leaves of reeds and boiled for a few hours—usually with other ingredients, as dates, meat, oyster, beams, etc.}
42A7 CJK UNIFIED IDEOGRAPH-#42A7, IRGKangXi=0913.080, Def{(same as 屁) a fart; to break wind}
42A8 CJK UNIFIED IDEOGRAPH-#42A8, IRGKangXi=0913.130, Def{storing grains; to store up food}
42A9 CJK UNIFIED IDEOGRAPH-#42A9, IRGKangXi=0912.150, Def{rice gravy}
42AA CJK UNIFIED IDEOGRAPH-#42AA, IRGKangXi=0913.190, Def{coarse rice -- unhulled, (interchangeable 糲) coarse -- of grain}
42AB CJK UNIFIED IDEOGRAPH-#42AB, IRGKangXi=0913.200, Def{coarse rice, red rice}
42AC CJK UNIFIED IDEOGRAPH-#42AC, IRGKangXi=0913.340, Def{name of a variety of grain, hollyhock; the mallow; root and flower can be used as medicine to neutralize poison; antipyretic (febrifuge)}
42AD CJK UNIFIED IDEOGRAPH-#42AD, IRGKangXi=0913.350, Def{to overturn; to pour out, to respond, to examine carefully, to stick, thick congee}
42AE CJK UNIFIED IDEOGRAPH-#42AE, IRGKangXi=0913.380, Def{name of a variety of grain}
42AF CJK UNIFIED IDEOGRAPH-#42AF, IRGKangXi=0913.460, Def{(same as standard form 穬) grains with beard (rice plant, wheat, etc.) unripe rice plant}
42B0 CJK UNIFIED IDEOGRAPH-#42B0, IRGKangXi=0914.010, Def{foodstuff; provisions; grains for human consumption}
42B1 CJK UNIFIED IDEOGRAPH-#42B1, IRGKangXi=0914.050, Def{cakes made of rice-flour}
42B2 CJK UNIFIED IDEOGRAPH-#42B2, IRGKangXi=0914.060, Def{to hull rice roughly (to make a rush pounding of the rice); coarse rice, to grind (grains, millet, etc.)}
42B3 CJK UNIFIED IDEOGRAPH-#42B3, IRGKangXi=0914.140, Def{to break into pieces; to smash; completely smashed, chips; crumbs, refined; polished (rice); unmixed}
42B4 CJK UNIFIED IDEOGRAPH-#42B4, IRGKangXi=0914.180, Def{food made of rice-flour}
42B5 CJK UNIFIED IDEOGRAPH-#42B5, IRGKangXi=0915.050, Def{young and small, (same as 絿) rash and impatient}
42B6 CJK UNIFIED IDEOGRAPH-#42B6, IRGKangXi=0915.170, Def{(same as 紖) a rope for leading cattle}
42B7 CJK UNIFIED IDEOGRAPH-#42B7, IRGKangXi=0915.190, Def{(same as 緇) black silk; a dark, drab colour, used for Buddhists, from the dark colour of their robes (same as 純) pure, honest}
42B8 CJK UNIFIED IDEOGRAPH-#42B8, IRGKangXi=0916.090, Def{(standard form of 紆) to twist; to distort, a cord}
42B9 CJK UNIFIED IDEOGRAPH-#42B9, IRGKangXi=0916.121, Def{(a abbreviated form of 纖) small, fine, delicate}
42BA CJK UNIFIED IDEOGRAPH-#42BA, IRGKangXi=0916.130, Def{an implement to draw or to collect ropes or cords}
42BB CJK UNIFIED IDEOGRAPH-#42BB, IRGKangXi=0916.190, Def{silky, very fine thread}
42BC CJK UNIFIED IDEOGRAPH-#42BC, IRGKangXi=0917.110, Def{measuring unit; end of spoilt silk}
42BD CJK UNIFIED IDEOGRAPH-#42BD, IRGKangXi=0917.130, Def{a fine thread, linen thread; silk thread; a thread; a yarn}
42BE CJK UNIFIED IDEOGRAPH-#42BE, IRGKangXi=0917.160, Def{ropes; cords; cables}
42BF CJK UNIFIED IDEOGRAPH-#42BF, IRGKangXi=0917.170, Def{(same as 衭) the lapel or collar of a garment or robe, drawers, (dialect) trousers or pants}
42C0 CJK UNIFIED IDEOGRAPH-#42C0, IRGKangXi=0918.010, Def{shoes made of hemp or hair}
42C1 CJK UNIFIED IDEOGRAPH-#42C1, IRGKangXi=0918.030, Def{(ancient form 綆) a rope for drawing up water (from a well, stream, etc.)}
42C2 CJK UNIFIED IDEOGRAPH-#42C2, IRGKangXi=0918.130, Def{dresses for the bride, green color, yellow color, (interchangeable 絞) a greenish yellow color}
42C3 CJK UNIFIED IDEOGRAPH-#42C3, IRGKangXi=0918.140, Def{stinging; hairy silk fabrics}
42C4 CJK UNIFIED IDEOGRAPH-#42C4, IRGKangXi=0919.110, Def{(same as 網) web; net; network}
42C5 CJK UNIFIED IDEOGRAPH-#42C5, IRGKangXi=0919.171
42C6 CJK UNIFIED IDEOGRAPH-#42C6, IRGKangXi=0919.171
42C7 CJK UNIFIED IDEOGRAPH-#42C7, IRGKangXi=0919.171
42C8 CJK UNIFIED IDEOGRAPH-#42C8, IRGKangXi=0919.190, Def{to bind loose hemp, old yarn, to stuff; to fill, waste silk or cotton and silk to be laid and attached each other; to padding; cushioning}
42C9 CJK UNIFIED IDEOGRAPH-#42C9, IRGKangXi=0919.220, Def{(interchangeable 䊽) a numerary adjunct (classifier) for practically everything; a thread; a yarn, clothes for the dead, linen thread; silk thread}
42CA CJK UNIFIED IDEOGRAPH-#42CA, IRGKangXi=0919.240, Def{decorations of the emperor's carriage; ornamental items on horses}
42CB CJK UNIFIED IDEOGRAPH-#42CB, IRGKangXi=0919.310, Def{(same as 罠) a kind of spring fishing net}
42CC CJK UNIFIED IDEOGRAPH-#42CC, IRGKangXi=0920.040, Def{(ancient form 堅) strong; durable; solid; firm; stable, (same as 䋗) tight; firm, pressing}
42CD CJK UNIFIED IDEOGRAPH-#42CD, IRGKangXi=0920.110, Def{(an abbreviated form of 䋪) fine and delicate silk, plain white sackcloth for mourning}
42CE CJK UNIFIED IDEOGRAPH-#42CE, IRGKangXi=0920.120, Def{to mend (clothes, etc.), (interchangeable 綻) ripped seam; a crack; split}
42CF CJK UNIFIED IDEOGRAPH-#42CF, IRGKangXi=0920.130, Def{thick and big ropes or cords; bulky cables, spoilt silk}
42D0 CJK UNIFIED IDEOGRAPH-#42D0, IRGKangXi=0920.210, Def{fabric; textile with patterns used for hem decorations, decorations on carriage and horses, varicolored silk or fine cloth loose in texture}
42D1 CJK UNIFIED IDEOGRAPH-#42D1, IRGKangXi=0921.250, Def{to tie; to knot, to join}
42D2 CJK UNIFIED IDEOGRAPH-#42D2, IRGKangXi=0921.270, Def{(corrupted form of 紓) to relax, to free from}
42D3 CJK UNIFIED IDEOGRAPH-#42D3, IRGKangXi=0921.290, Def{(abbreviated form 縐) wrinkled, to shrink, crepe, a coarse, yellowish cloth for summer wear}
42D4 CJK UNIFIED IDEOGRAPH-#42D4, IRGKangXi=0921.291, Def{to band together; to twist; to wring and to disperse fiber then to make thin silk, cloth, cotton-yarn or rope}
42D5 CJK UNIFIED IDEOGRAPH-#42D5, IRGKangXi=0921.310, Def{(same as U+7D4D 紝) to lay the warp; to weave}
42D6 CJK UNIFIED IDEOGRAPH-#42D6, IRGKangXi=0921.380, Def{long, a gown; a long dress}
42D7 CJK UNIFIED IDEOGRAPH-#42D7, IRGKangXi=0922.030, Def{(same as 䋌) (same as 堅) strong and durable, solid and firm; tight; pressing}
42D8 CJK UNIFIED IDEOGRAPH-#42D8, IRGKangXi=0922.070, Def{(same as 縋) to hand by a rope; to let down by a rope}
42D9 CJK UNIFIED IDEOGRAPH-#42D9, IRGKangXi=0922.130, Def{reins; bridle}
42DA CJK UNIFIED IDEOGRAPH-#42DA, IRGKangXi=0923.020, Def{(same as 翼) wings, fins, to help, to protect}
42DB CJK UNIFIED IDEOGRAPH-#42DB, IRGKangXi=0923.080, Def{the clustered embroidery patterns (as tiny and fine rice)}
42DC CJK UNIFIED IDEOGRAPH-#42DC, IRGKangXi=0924.120, Def{(same as U+7DAE 綮) crucial points; critical points, an embroidered banner, sheath for a lance head}
42DD CJK UNIFIED IDEOGRAPH-#42DD, IRGKangXi=0924.161, Def{(abbreviated form of 纓) a throat-band; chin strap for holding the hat, tassel; a fringe}
42DE CJK UNIFIED IDEOGRAPH-#42DE, IRGKangXi=0924.161, Def{(abbreviated form of 網) web; net; network}
42DF CJK UNIFIED IDEOGRAPH-#42DF, IRGKangXi=0924.190, Def{the weight on a steelyard, connected stitches}
42E0 CJK UNIFIED IDEOGRAPH-#42E0, IRGKangXi=0924.320, Def{(interchangeable 補) to repair; to mend, to add to, to make up}
42E1 CJK UNIFIED IDEOGRAPH-#42E1, IRGKangXi=0925.020, Def{(same as 紓) (interchangeable 舒) to relax, to free from}
42E2 CJK UNIFIED IDEOGRAPH-#42E2, IRGKangXi=0925.120, Def{to twist ropes, a belt or string to hold the bow, to tie; to bind, the cart turning back to the left, to draw or pull (of a cart), to mourn}
42E3 CJK UNIFIED IDEOGRAPH-#42E3, IRGKangXi=0926.040, Def{decorations put on the mane or horsehair, (standard form of 繁) many, troublesome, a family name}
42E4 CJK UNIFIED IDEOGRAPH-#42E4, IRGKangXi=0926.050, Def{plain white silk, to soften and whiten raw silk by boiling}
42E5 CJK UNIFIED IDEOGRAPH-#42E5, IRGKangXi=0926.060, Def{lines; stripes; veins}
42E6 CJK UNIFIED IDEOGRAPH-#42E6, IRGKangXi=0926.110, Def{ravelled silk}
42E7 CJK UNIFIED IDEOGRAPH-#42E7, IRGKangXi=0926.250, Def{(same as 絇) ornaments for the frontal part of shoes}
42E8 CJK UNIFIED IDEOGRAPH-#42E8, IRGKangXi=0926.280, Def{to set in order the old, raw, coarse, waste silk or cotton}
42E9 CJK UNIFIED IDEOGRAPH-#42E9, IRGKangXi=0926.290, Def{pretty and fine silk fabrics}
42EA CJK UNIFIED IDEOGRAPH-#42EA, IRGKangXi=0926.330, Def{(interchangeable 䋍) fine and delicate silk, plain white sackcloth for mourning}
42EB CJK UNIFIED IDEOGRAPH-#42EB, IRGKangXi=0927.020, Def{to wind round the ropes, crooked; winding; bends; turns; curves; turns and twists, to play a sound of hasty (fast) string (of a musical instrument)}
42EC CJK UNIFIED IDEOGRAPH-#42EC, IRGKangXi=0927.050, Def{rug; carpet; blanket, woolen textiles; woolen goods; woolen stuff; woolen fabrics}
42ED CJK UNIFIED IDEOGRAPH-#42ED, IRGKangXi=0927.090, Def{to weave silk fabrics with green color for longitude and white for latitude, silk fabrics from Yuyang}
42EE CJK UNIFIED IDEOGRAPH-#42EE, IRGKangXi=0927.140, Def{(the large seal; a type of Chinese calligraphy) (same as 紟) a sash, to tie, a kind of cloth or textiles, lapel of a Chinese dress, a single coverlet}
42EF CJK UNIFIED IDEOGRAPH-#42EF, IRGKangXi=0927.170, Def{(same as U+7DAE 綮) crucial points; critical points, an embroidered banner, sheath for a lance head}
42F0 CJK UNIFIED IDEOGRAPH-#42F0, IRGKangXi=0927.190, Def{to bind or restrain; restraint; restriction, timid and awkward, to wind around; to tangle, to connect; to join, together with}
42F1 CJK UNIFIED IDEOGRAPH-#42F1, IRGKangXi=0927.220, Def{(same as 斄) a wild yak, hard and curved hair, name of a county in ancient times}
42F2 CJK UNIFIED IDEOGRAPH-#42F2, IRGKangXi=0929.291, Def{(non-classical form of U+7E69 繩) a rope; a cord, to restrain, to rectify; to correct}
42F3 CJK UNIFIED IDEOGRAPH-#42F3, IRGKangXi=0929.310, Def{(same as 褙) cloth or paper pasted together; pasteboard; to mount (paintings or calligraphic works), short clothes}
42F4 CJK UNIFIED IDEOGRAPH-#42F4, IRGKangXi=0930.010, Def{silk fabrics with mixed colors}
42F5 CJK UNIFIED IDEOGRAPH-#42F5, IRGKangXi=0930.040, Def{quick, urgent, anxious, not going to succeed, poor quality of silk fabrics}
42F6 CJK UNIFIED IDEOGRAPH-#42F6, IRGKangXi=0930.060, Def{to stumble; to trip the front feet of a beast, a kind of brocade from Shu}
42F7 CJK UNIFIED IDEOGRAPH-#42F7, IRGKangXi=0930.110, Def{a kind of thick, loosely-woven raw silk fabric}
42F8 CJK UNIFIED IDEOGRAPH-#42F8, IRGKangXi=0930.170, Def{back center suture of the clothes}
42F9 CJK UNIFIED IDEOGRAPH-#42F9, IRGKangXi=0930.180, Def{(same as 幅) breadth of material (cloth or paper, etc.)}
42FA CJK UNIFIED IDEOGRAPH-#42FA, IRGKangXi=0931.070, Def{(same as 鞦) a swing (same as U+97A7 緧) a crupper; traces}
42FB CJK UNIFIED IDEOGRAPH-#42FB, IRGKangXi=0931.090, Def{to spin and weave, equal; uniform; (same as U+7D1D 紝) to lay the warp; to weave}
42FC CJK UNIFIED IDEOGRAPH-#42FC, IRGKangXi=0931.100, Def{slow; leisurely, to delay; to slacken, silk ribbons}
42FD CJK UNIFIED IDEOGRAPH-#42FD, IRGKangXi=0931.290, Def{hempen sandals, leather shoes (for children)}
42FE CJK UNIFIED IDEOGRAPH-#42FE, IRGKangXi=0932.060, Def{cotton and silk to be laid and attached each other}
42FF CJK UNIFIED IDEOGRAPH-#42FF, IRGKangXi=0932.090, Def{a decorative knot made of colorful silk (blue, yellow, red, white and black)}
4300 CJK UNIFIED IDEOGRAPH-#4300, IRGKangXi=0932.110, Def{to arrange; to repair silk, colored patterns (stripes; lines; streaks; veins)}
4301 CJK UNIFIED IDEOGRAPH-#4301, IRGKangXi=0932.200, Def{a pleat; fold, a piece of string; ribbon used to tie clothes, a rope; a line; a cord}
4302 CJK UNIFIED IDEOGRAPH-#4302, IRGKangXi=0932.280, Def{(same as 握) in ancient times, article for preparing the body for the coffin (something slip on the hand of the dead)}
4303 CJK UNIFIED IDEOGRAPH-#4303, IRGKangXi=0932.290, Def{(same as 綃) raw silk, to seal; to close}
4304 CJK UNIFIED IDEOGRAPH-#4304, IRGKangXi=0933.061, Def{(standard form of 綆) a rope for drawing up water (form a well, stream, etc.)}
4305 CJK UNIFIED IDEOGRAPH-#4305, IRGKangXi=0933.180, Def{a thick ropes}
4306 CJK UNIFIED IDEOGRAPH-#4306, IRGKangXi=0933.200, Def{silk fabrics}
4307 CJK UNIFIED IDEOGRAPH-#4307, IRGKangXi=0933.380, Def{(non-classical form of 索) a thick rope; a cable, alone, to tighten; to squeeze, to need, to demand}
4308 CJK UNIFIED IDEOGRAPH-#4308, IRGKangXi=0934.210, Def{to get something with a lasso}
4309 CJK UNIFIED IDEOGRAPH-#4309, IRGKangXi=0935.120, Def{(non-classical form of 綱) the large of a net, main point, principle}
430A CJK UNIFIED IDEOGRAPH-#430A, IRGKangXi=0935.181, Def{(corrupted form of 繇) entourage; aides; attendants, cause; means, by way of, etc., forced labour; labor service}
430B CJK UNIFIED IDEOGRAPH-#430B, IRGKangXi=0935.181, Def{a kerchief hung at the waist, a tiny sack; a purse or pouch carried hanging over the girdle}
430C CJK UNIFIED IDEOGRAPH-#430C, IRGKangXi=0935.220, Def{lace; embroidered hem (of a garment), (same as 緁) a narrow strip of woven material (such as the hem of a straw hat)}
430D CJK UNIFIED IDEOGRAPH-#430D, IRGKangXi=0935.230, Def{fine woven pattern}
430E CJK UNIFIED IDEOGRAPH-#430E, IRGKangXi=0935.290, Def{to baste for sewing, to connect; to sew; to stitch; to mend}
430F CJK UNIFIED IDEOGRAPH-#430F, IRGKangXi=0936.010
4310 CJK UNIFIED IDEOGRAPH-#4310, IRGKangXi=0936.030, Def{ropes; cords; cables}
4311 CJK UNIFIED IDEOGRAPH-#4311, IRGKangXi=0936.040, Def{(interchangeable 堅) tight; firm; fast; secure; close}
4312 CJK UNIFIED IDEOGRAPH-#4312, IRGKangXi=0936.190, Def{pure; unalloyed, honest; simple, purely}
4313 CJK UNIFIED IDEOGRAPH-#4313, IRGKangXi=0937.020, Def{(corrupted form of U+7E41 繁) many; abundant, complex; intricate}
4314 CJK UNIFIED IDEOGRAPH-#4314, IRGKangXi=0938.080, Def{(corrupted form of 漚) to soak, foam; bubble; froth}
4315 CJK UNIFIED IDEOGRAPH-#4315, IRGKangXi=0938.140, Def{(same as 縻) to tie; to fasten; to connect}
4316 CJK UNIFIED IDEOGRAPH-#4316, IRGKangXi=0938.210, Def{to gather; to assemble; to come together; to blend; to mix, disorderly; mixed, a general term for wealthy goods and textiles (in southern minority group)}
4317 CJK UNIFIED IDEOGRAPH-#4317, IRGKangXi=0938.240, Def{the left over of drawing silk (from cocoons), silk, to arrange raw, coarse, old, waste cotton or silk}
4318 CJK UNIFIED IDEOGRAPH-#4318, IRGKangXi=0939.070, Def{to knit; to twist ropes, a belt to hang sword, poor qualitied silk wadding}
4319 CJK UNIFIED IDEOGRAPH-#4319, IRGKangXi=0939.080, Def{hank of ropes}
431A CJK UNIFIED IDEOGRAPH-#431A, IRGKangXi=0939.210, Def{wrinkles; creases; folds}
431B CJK UNIFIED IDEOGRAPH-#431B, IRGKangXi=0940.031, Def{(same as 繇 徭 陶 謠 由 猶 悠 籀) entourage; aides; attendants, compulsory labor service, to make pottery or earthenware, happy, ballad; folk song, rumor, through; via; by way of, like; similar to, still; yet, far, sad, soft; slow, to deduce (interchangeable 搖 遙 傜)}
431C CJK UNIFIED IDEOGRAPH-#431C, IRGKangXi=0940.160, Def{to sew; to stitch; to suture; to mend; to patch a garment; dresses, etc.. to twist; to tie up with ropes}
431D CJK UNIFIED IDEOGRAPH-#431D, IRGKangXi=0940.220, Def{silk, a general name of silk fabrics or goods, (interchangeable 紺) reddish dark color; violet or purple}
431E CJK UNIFIED IDEOGRAPH-#431E, IRGKangXi=0941.020, Def{a knot to hang the apparatus made of reed for raising silkworms, a rope}
431F CJK UNIFIED IDEOGRAPH-#431F, IRGKangXi=0941.030, Def{to weave silk ribbons, (interchangeable 澼) to wash; to launder}
4320 CJK UNIFIED IDEOGRAPH-#4320, IRGKangXi=0941.080, Def{hold on to one's own views, to keep the mouth shut, stingy; miserly; niggardly; parsimonious}
4321 CJK UNIFIED IDEOGRAPH-#4321, IRGKangXi=0941.130, Def{to wind around; to bind; to wrap; to tangle, to bother persistently}
4322 CJK UNIFIED IDEOGRAPH-#4322, IRGKangXi=0941.160, Def{raw, coarse, old, waste cotton or silk, wooly; fluffy}
4323 CJK UNIFIED IDEOGRAPH-#4323, IRGKangXi=0941.310, Def{to accumulate; to store up}
4324 CJK UNIFIED IDEOGRAPH-#4324, IRGKangXi=0942.010, Def{a kind of unrefined or unpolished silken textiles; silken goods; silken fabrics, to sew; to patch clothes (of sword)}
4325 CJK UNIFIED IDEOGRAPH-#4325, IRGKangXi=0942.030, Def{to sew clothes; to do needle-work}
4326 CJK UNIFIED IDEOGRAPH-#4326, IRGKangXi=0942.050, Def{green color; bluish yellow}
4327 CJK UNIFIED IDEOGRAPH-#4327, IRGKangXi=0942.110, Def{(non-classical form of 紬 綢) a kind of silk fabric; thin silk goods}
4328 CJK UNIFIED IDEOGRAPH-#4328, IRGKangXi=0942.280, Def{thin silk fabrics, sound of the rubbing of clothes}
4329 CJK UNIFIED IDEOGRAPH-#4329, IRGKangXi=0942.420, Def{fine, thin stockings or socks}
432A CJK UNIFIED IDEOGRAPH-#432A, IRGKangXi=0943.090, Def{to continue; to extend; to review}
432B CJK UNIFIED IDEOGRAPH-#432B, IRGKangXi=0943.100, Def{(non-classical form of U+7E9C 纜) a hawser, a cable, a rope}
432C CJK UNIFIED IDEOGRAPH-#432C, IRGKangXi=0943.130, Def{straight; right; just, straight-forward, uninterrupted}
432D CJK UNIFIED IDEOGRAPH-#432D, IRGKangXi=0943.190, Def{raw hemp, a kind of fabric}
432E CJK UNIFIED IDEOGRAPH-#432E, IRGKangXi=0943.330, Def{silk fabrics with the color of new leaves}
432F CJK UNIFIED IDEOGRAPH-#432F, IRGKangXi=0943.340, Def{a turban; a kerchief, children's hat, a girdle, belt of the clothes}
4330 CJK UNIFIED IDEOGRAPH-#4330, IRGKangXi=0944.010, Def{connected silk, measurement}
4331 CJK UNIFIED IDEOGRAPH-#4331, IRGKangXi=0944.030, Def{(same as 䌴) uneven; silk with knots}
4332 CJK UNIFIED IDEOGRAPH-#4332, IRGKangXi=0944.050, Def{(same as 纁) light red}
4333 CJK UNIFIED IDEOGRAPH-#4333, IRGKangXi=0944.120, Def{a kind of unrefined or unpolished silken textiles; silken goods; silken fabrics}
4334 CJK UNIFIED IDEOGRAPH-#4334, IRGKangXi=0944.150, Def{uneven; silk with knots; unpolished}
4335 CJK UNIFIED IDEOGRAPH-#4335, IRGKangXi=0944.190, Def{lapel and belt}
4336 CJK UNIFIED IDEOGRAPH-#4336, IRGKangXi=0944.251, Def{(simplified form of 䊷) (same as 緇)) black silk; a dark, drab colour, used for Buddhists, from the dark colour of their robes (same as 純) pure, honest}
4337 CJK UNIFIED IDEOGRAPH-#4337, IRGKangXi=0944.251, Def{(simplified form of 紬) a thread; a clue}
4338 CJK UNIFIED IDEOGRAPH-#4338, IRGKangXi=0944.251, Def{(simplified form of 縳) fine silk fabric of bright white colour, to tie up, (interchangeable 卷) a book or painting which can be easily folded or rolled up, a division of a book, files}
4339 CJK UNIFIED IDEOGRAPH-#4339, IRGKangXi=0944.251, Def{(simplified form of 絅) (same as 褧) a garment of one colour with no lining, a dust coat}
433A CJK UNIFIED IDEOGRAPH-#433A, IRGKangXi=0944.251, Def{(simplified form) (interchangeable 耳) flourishing; thriving; vigorous}
433B CJK UNIFIED IDEOGRAPH-#433B, IRGKangXi=0944.251, Def{(simplified form) (same as 翼) wings, fins, to help; to assist}
433C CJK UNIFIED IDEOGRAPH-#433C, IRGKangXi=0944.251, Def{a kind of fine silk fabric}
433D CJK UNIFIED IDEOGRAPH-#433D, IRGKangXi=0944.251, Def{varicolored silk; motley}
433E CJK UNIFIED IDEOGRAPH-#433E, IRGKangXi=0944.251, Def{(same as U+7D1D 紝) to lay the warp; to weave}
433F CJK UNIFIED IDEOGRAPH-#433F, IRGKangXi=0944.251, Def{(simplified form) (same as 幅) the breadth of cloth or paper; a piece (of cloth)}
4340 CJK UNIFIED IDEOGRAPH-#4340, IRGKangXi=0944.251, Def{(same as U+8964 襤) clothes without hem; ragged garments; sloppily dressed}
4341 CJK UNIFIED IDEOGRAPH-#4341, IRGKangXi=0944.251, Def{(simplified form of 繸) the hem or border of a garment, a tassel}
4342 CJK UNIFIED IDEOGRAPH-#4342, IRGKangXi=0944.290, Def{tools to draw water or liquid}
4343 CJK UNIFIED IDEOGRAPH-#4343, IRGKangXi=0945.030, Def{a vase, a pitcher, earthenware}
4344 CJK UNIFIED IDEOGRAPH-#4344, IRGKangXi=0945.070, Def{an indenture (of earthenware)}
4345 CJK UNIFIED IDEOGRAPH-#4345, IRGKangXi=0945.080, Def{containers for water or wine used in ancient times}
4346 CJK UNIFIED IDEOGRAPH-#4346, IRGKangXi=0945.090, Def{container for grains used in ancient times}
4347 CJK UNIFIED IDEOGRAPH-#4347, IRGKangXi=0945.100, RSKangXi=121.5, Def{flat-bottom crock (with narrow opening); jug, pitcher, vase, etc; (Cant.) stool}
4348 CJK UNIFIED IDEOGRAPH-#4348, IRGKangXi=0945.140, Def{(same as 缾 瓶) bottle; pitcher; jug; vase}
4349 CJK UNIFIED IDEOGRAPH-#4349, IRGKangXi=0945.150, Def{traces or marks of incomplete or damage}
434A CJK UNIFIED IDEOGRAPH-#434A, IRGKangXi=0945.180, Def{clay-made blowing wind instruments; music instruments made of clay}
434B CJK UNIFIED IDEOGRAPH-#434B, IRGKangXi=0945.330, Def{bottle; pitcher; jug; vase; an earthen jar with narrow opening}
434C CJK UNIFIED IDEOGRAPH-#434C, IRGKangXi=0945.350, Def{(same as 瓿) a jar; a pot}
434D CJK UNIFIED IDEOGRAPH-#434D, IRGKangXi=0945.420, Def{unburnt bricks; tiles; earthenware, etc.}
434E CJK UNIFIED IDEOGRAPH-#434E, IRGKangXi=0946.140, Def{a spindle, a earthenware used to pinch or knead, a small crock}
434F CJK UNIFIED IDEOGRAPH-#434F, IRGKangXi=0946.241
4350 CJK UNIFIED IDEOGRAPH-#4350, IRGKangXi=0946.340, Def{(standard form 罕) a net or snare, a flag, rare; strange; scarce; few, name of a place in ancient times}
4351 CJK UNIFIED IDEOGRAPH-#4351, IRGKangXi=0947.031, Def{(same as 䍐) (standard form 罕) a net or snare, a flag, rare; strange; scarce; few, name of a place in ancient times}
4352 CJK UNIFIED IDEOGRAPH-#4352, IRGKangXi=0947.040, Def{(corrupted form) net; web; network}
4353 CJK UNIFIED IDEOGRAPH-#4353, IRGKangXi=0947.060, Def{(a corrupted form) net for catching rabbit}
4354 CJK UNIFIED IDEOGRAPH-#4354, IRGKangXi=0947.090, Def{a full net, a thick rope; a cable}
4355 CJK UNIFIED IDEOGRAPH-#4355, IRGKangXi=0947.150, Def{net; web; network, net for catching rabbit}
4356 CJK UNIFIED IDEOGRAPH-#4356, IRGKangXi=0947.200, Def{(same as 罦) a kind of net in a trap used to catch birds and beasts; a covering for a cart}
4357 CJK UNIFIED IDEOGRAPH-#4357, IRGKangXi=0947.300, Def{(same as 罥) to hang up; to bind; to entangle, net; web; network}
4358 CJK UNIFIED IDEOGRAPH-#4358, IRGKangXi=0947.340, Def{a general term for nets}
4359 CJK UNIFIED IDEOGRAPH-#4359, IRGKangXi=0947.430, Def{nets for catching birds, net; web; network}
435A CJK UNIFIED IDEOGRAPH-#435A, IRGKangXi=0948.010, Def{vast}
435B CJK UNIFIED IDEOGRAPH-#435B, IRGKangXi=0948.070, Def{a kind of tool used to get fish}
435C CJK UNIFIED IDEOGRAPH-#435C, IRGKangXi=0948.110, Def{a basket or a cage used to keep wild and domestic fowls}
435D CJK UNIFIED IDEOGRAPH-#435D, IRGKangXi=0948.250, Def{net; web; network}
435E CJK UNIFIED IDEOGRAPH-#435E, IRGKangXi=0948.291, Def{(same as 罭) a drag-net; a fine net used to catch small fish}
435F CJK UNIFIED IDEOGRAPH-#435F, IRGKangXi=0948.350, Def{a kind of fish net with fine meshes}
4360 CJK UNIFIED IDEOGRAPH-#4360, IRGKangXi=0949.271, Def{(simplified form of 䍦) turban used in ancient times}
4361 CJK UNIFIED IDEOGRAPH-#4361, IRGKangXi=0949.220, Def{small fish net, to catch fish with small fish net}
4362 CJK UNIFIED IDEOGRAPH-#4362, IRGKangXi=0949.290, Def{netted veined window-sill}
4363 CJK UNIFIED IDEOGRAPH-#4363, IRGKangXi=0950.010, Def{a kind of fish net}
4364 CJK UNIFIED IDEOGRAPH-#4364, IRGKangXi=0950.060, Def{to strain out; to filter wine; to squeeze juice out of; to draw; to press milk}
4365 CJK UNIFIED IDEOGRAPH-#4365, IRGKangXi=0950.250, Def{smoky, covering; to spread, cloth for covering food}
4366 CJK UNIFIED IDEOGRAPH-#4366, IRGKangXi=0950.280, Def{(traditional form of 䍠) a kind of turban used in ancient times}
4367 CJK UNIFIED IDEOGRAPH-#4367, IRGKangXi=0951.200, Def{(same as 牂) a ewe or she-goat}
4368 CJK UNIFIED IDEOGRAPH-#4368, IRGKangXi=0951.210, Def{to castrate, a ram or he-goat, last name}
4369 CJK UNIFIED IDEOGRAPH-#4369, IRGKangXi=0951.320, Def{(ancient form of 養) to offer provision (esp. to one's elders); to feed one's children}
436A CJK UNIFIED IDEOGRAPH-#436A, IRGKangXi=0951.340, Def{castrated ram}
436B CJK UNIFIED IDEOGRAPH-#436B, IRGKangXi=0951.350, Def{a legendary goat-like animal with 4 ears and 9 tails, a kind of goat with no horn}
436C CJK UNIFIED IDEOGRAPH-#436C, IRGKangXi=0951.370, Def{to use; to made; to act, a kind of goat with mixed colors}
436D CJK UNIFIED IDEOGRAPH-#436D, IRGKangXi=0952.080, Def{(same as 牂) a ewe or she-goat}
436E CJK UNIFIED IDEOGRAPH-#436E, IRGKangXi=0952.100, Def{young goat (sheep) under one year old, castrated ram of hundred catties (Chinese pound)}
436F CJK UNIFIED IDEOGRAPH-#436F, IRGKangXi=0952.130, Def{(same as 觤) goat with asymmetric horns}
4370 CJK UNIFIED IDEOGRAPH-#4370, IRGKangXi=0952.150, Def{a plague or a pestilence of goat (sheep); black goat or sheep, black}
4371 CJK UNIFIED IDEOGRAPH-#4371, IRGKangXi=0953.020, Def{the wild goat or sheep}
4372 CJK UNIFIED IDEOGRAPH-#4372, IRGKangXi=0953.090, Def{a kind of sheep (goat) with curled hairs}
4373 CJK UNIFIED IDEOGRAPH-#4373, IRGKangXi=0953.110, Def{disease; illness, goat (or sheep) get sick, a throbbing goat (or sheep)}
4374 CJK UNIFIED IDEOGRAPH-#4374, IRGKangXi=0953.120, Def{to squeeze and to crowd against each other (of sheep)}
4375 CJK UNIFIED IDEOGRAPH-#4375, IRGKangXi=0953.150, Def{young goat, antelope}
4376 CJK UNIFIED IDEOGRAPH-#4376, IRGKangXi=0953.160, Def{unicorn}
4377 CJK UNIFIED IDEOGRAPH-#4377, IRGKangXi=0953.230, Def{to chase each other (of sheep or goats)}
4378 CJK UNIFIED IDEOGRAPH-#4378, IRGKangXi=0953.320, Def{a legendary goat-like animal with 4 ears and 9 tails}
4379 CJK UNIFIED IDEOGRAPH-#4379, IRGKangXi=0953.330, Def{(same as U+7FB4 羶) odor of a sheep or goat, flock of goats or sheep, name and description of snuff (a kind of tobacco)}
437A CJK UNIFIED IDEOGRAPH-#437A, IRGKangXi=0954.010, Def{a legendary goat-like beast}
437B CJK UNIFIED IDEOGRAPH-#437B, IRGKangXi=0954.070, Def{young goat of sheep under one year old}
437C CJK UNIFIED IDEOGRAPH-#437C, IRGKangXi=0954.150, Def{salted mutton, to store up the meat under the ground, a goat-like beast, odor of a sheep or goat}
437D CJK UNIFIED IDEOGRAPH-#437D, IRGKangXi=0954.360, Def{a fierce goat, a castrated ram}
437E CJK UNIFIED IDEOGRAPH-#437E, IRGKangXi=0955.160, Def{flying}
437F CJK UNIFIED IDEOGRAPH-#437F, IRGKangXi=0955.240, Def{a kind of dance accompanied by music performed in ancient times, (same as 凰) the female phoenix, a legendary bird in Chinese mythology}
4380 CJK UNIFIED IDEOGRAPH-#4380, IRGKangXi=0956.060, Def{flying, to run swiftly; to go at express speed}
4381 CJK UNIFIED IDEOGRAPH-#4381, IRGKangXi=0956.120, Def{used in person's name}
4382 CJK UNIFIED IDEOGRAPH-#4382, IRGKangXi=0956.130, Def{colorful feather, a feather decorated arrow}
4383 CJK UNIFIED IDEOGRAPH-#4383, IRGKangXi=0956.140, Def{fine down; floss; wool}
4384 CJK UNIFIED IDEOGRAPH-#4384, IRGKangXi=0956.170, Def{hairy; long tail feather of a bird}
4385 CJK UNIFIED IDEOGRAPH-#4385, IRGKangXi=0956.220, Def{flying}
4386 CJK UNIFIED IDEOGRAPH-#4386, IRGKangXi=0956.270, Def{flying}
4387 CJK UNIFIED IDEOGRAPH-#4387, IRGKangXi=0956.340, Def{quick; quickly, hasty, soft and meek feather}
4388 CJK UNIFIED IDEOGRAPH-#4388, IRGKangXi=0956.350, Def{flying}
4389 CJK UNIFIED IDEOGRAPH-#4389, IRGKangXi=0956.360, Def{to walk swiftly, (same as 矞) scared; afraid; fearful; frightened}
438A CJK UNIFIED IDEOGRAPH-#438A, IRGKangXi=0956.380, Def{flying, flying away}
438B CJK UNIFIED IDEOGRAPH-#438B, IRGKangXi=0957.100, Def{fly}
438C CJK UNIFIED IDEOGRAPH-#438C, IRGKangXi=0957.110, Def{feather, flying, neat feather}
438D CJK UNIFIED IDEOGRAPH-#438D, IRGKangXi=0957.140, Def{fine and soft hair, feather}
438E CJK UNIFIED IDEOGRAPH-#438E, IRGKangXi=0957.190, Def{to glide or to hover up in the air, to soaring}
438F CJK UNIFIED IDEOGRAPH-#438F, IRGKangXi=0957.230, Def{young birds flying}
4390 CJK UNIFIED IDEOGRAPH-#4390, IRGKangXi=0957.270, Def{hairy}
4391 CJK UNIFIED IDEOGRAPH-#4391, IRGKangXi=0957.280, Def{flying over the water surface, flying up; soaring}
4392 CJK UNIFIED IDEOGRAPH-#4392, IRGKangXi=0957.380, Def{flying quickly and to attack suddenly (said of bird)}
4393 CJK UNIFIED IDEOGRAPH-#4393, IRGKangXi=0958.041, Def{flying}
4394 CJK UNIFIED IDEOGRAPH-#4394, IRGKangXi=0958.360, Def{fine and soft hair, feather}
4395 CJK UNIFIED IDEOGRAPH-#4395, IRGKangXi=0958.400, Def{flying birds}
4396 CJK UNIFIED IDEOGRAPH-#4396, IRGKangXi=0959.220, Def{soaring up in the sky}
4397 CJK UNIFIED IDEOGRAPH-#4397, IRGKangXi=0959.300, Def{flying}
4398 CJK UNIFIED IDEOGRAPH-#4398, IRGKangXi=0959.330, Def{birds flying, sound of the flying wings}
4399 CJK UNIFIED IDEOGRAPH-#4399, IRGKangXi=0960.120, Def{flying}
439A CJK UNIFIED IDEOGRAPH-#439A, IRGKangXi=0960.170, Def{the end of bird wings, bird wings}
439B CJK UNIFIED IDEOGRAPH-#439B, IRGKangXi=0960.301
439C CJK UNIFIED IDEOGRAPH-#439C, IRGKangXi=0961.090, Def{aged; to get old, advanced age; great age; old age}
439D CJK UNIFIED IDEOGRAPH-#439D, IRGKangXi=0961.120, Def{(same as 翥) to soar}
439E CJK UNIFIED IDEOGRAPH-#439E, IRGKangXi=0961.131
439F CJK UNIFIED IDEOGRAPH-#439F, IRGKangXi=0961.230, Def{to fawn on; to flatter; to please, to love, attractive}
43A0 CJK UNIFIED IDEOGRAPH-#43A0, IRGKangXi=0962.030, Def{to twist or to crumple between hands and make something turn, (same as U+611E 懦) cowardice; timid; weak and cowardly}
43A1 CJK UNIFIED IDEOGRAPH-#43A1, IRGKangXi=0962.040, Def{to shrink; to recoil; to flinch, weak, soft and meek; gentle, (corrupted form of 耎) soft; weak, pliable}
43A2 CJK UNIFIED IDEOGRAPH-#43A2, IRGKangXi=0962.220, Def{to measure (esp. for grains) by pints and pecks}
43A3 CJK UNIFIED IDEOGRAPH-#43A3, IRGKangXi=0963.080, Def{(same as 耜) a plough; a ploughshare}
43A4 CJK UNIFIED IDEOGRAPH-#43A4, IRGKangXi=0963.170, Def{to till; to plough; to cultivate, agricultural implements; farm tools}
43A5 CJK UNIFIED IDEOGRAPH-#43A5, IRGKangXi=0963.191
43A6 CJK UNIFIED IDEOGRAPH-#43A6, IRGKangXi=0963.220, Def{to till; to plough, sharp-pointed spade or shovel}
43A7 CJK UNIFIED IDEOGRAPH-#43A7, IRGKangXi=0963.240, Def{agricultural implements; farm tools (a plough; a ploughshare), to till; to plough}
43A8 CJK UNIFIED IDEOGRAPH-#43A8, IRGKangXi=0963.280, Def{to plough and sow, to farm}
43A9 CJK UNIFIED IDEOGRAPH-#43A9, IRGKangXi=0963.300, Def{to till; to plough, (same as 菑) land under cultivation for one year, to weed grass}
43AA CJK UNIFIED IDEOGRAPH-#43AA, IRGKangXi=0963.330, Def{the stalk of grain; straw, a hammer used to pound grains and corns, etc.}
43AB CJK UNIFIED IDEOGRAPH-#43AB, IRGKangXi=0964.010, Def{to plant; to sow; to cultivate}
43AC CJK UNIFIED IDEOGRAPH-#43AC, IRGKangXi=0964.161, Def{(simplified form of 䎱) (same as 耙) a drag; a harrow; a rake}
43AD CJK UNIFIED IDEOGRAPH-#43AD, IRGKangXi=0964.241, Def{a kind of farm tool made of twisted thron strips used to flaten the land}
43AE CJK UNIFIED IDEOGRAPH-#43AE, IRGKangXi=0964.180, Def{to plant; to raise or grow (plants); to cultivate}
43AF CJK UNIFIED IDEOGRAPH-#43AF, IRGKangXi=0964.190, Def{winter ploughing, to plough a dry farmland (upland), to till; to plough}
43B0 CJK UNIFIED IDEOGRAPH-#43B0, IRGKangXi=0964.290, Def{name of a place in Sichuan Province}
43B1 CJK UNIFIED IDEOGRAPH-#43B1, IRGKangXi=0964.380, Def{(same as 耙) a drag; a harrow; a rake}
43B2 CJK UNIFIED IDEOGRAPH-#43B2, IRGKangXi=0965.071, Def{ear lobe; lobule}
43B3 CJK UNIFIED IDEOGRAPH-#43B3, IRGKangXi=0965.170, Def{lose of the ear}
43B4 CJK UNIFIED IDEOGRAPH-#43B4, IRGKangXi=0966.030, Def{a legendary spirit; immortal, (non-classical form of 聖) sage, sacred; holy}
43B5 CJK UNIFIED IDEOGRAPH-#43B5, IRGKangXi=0966.220, Def{(corrupted form) to look straight forward, ashamed, to look in a mean way}
43B6 CJK UNIFIED IDEOGRAPH-#43B6, IRGKangXi=0967.030, Def{offering blood to God}
43B7 CJK UNIFIED IDEOGRAPH-#43B7, IRGKangXi=0967.160, Def{an old man with white hair, source; head (of a stream), loud cries}
43B8 CJK UNIFIED IDEOGRAPH-#43B8, IRGKangXi=0967.180, Def{to hear; to listen; to obey; to understand}
43B9 CJK UNIFIED IDEOGRAPH-#43B9, IRGKangXi=0967.300, Def{(ancient form of 聞) to hear, to learn, to convey, to smell}
43BA CJK UNIFIED IDEOGRAPH-#43BA, IRGKangXi=0967.370, Def{to learn; to hear; (Cant.) to work; to be willing}
43BB CJK UNIFIED IDEOGRAPH-#43BB, IRGKangXi=0968.050, Def{to hear, clear, bright, ears}
43BC CJK UNIFIED IDEOGRAPH-#43BC, IRGKangXi=0968.060, Def{a kind of insect}
43BD CJK UNIFIED IDEOGRAPH-#43BD, IRGKangXi=0968.090, Def{(ancient form of 聞) to hear, to learn, to convey, to smell, to hear of}
43BE CJK UNIFIED IDEOGRAPH-#43BE, IRGKangXi=0968.221, Def{(corrupted form of 睔) big and round eyes}
43BF CJK UNIFIED IDEOGRAPH-#43BF, IRGKangXi=0968.240, Def{buzzing in the ears; tinnitus aurium}
43C0 CJK UNIFIED IDEOGRAPH-#43C0, IRGKangXi=0968.370, Def{inharmonious}
43C1 CJK UNIFIED IDEOGRAPH-#43C1, IRGKangXi=0968.420, Def{unable to hear distinctly or clearly}
43C2 CJK UNIFIED IDEOGRAPH-#43C2, IRGKangXi=0968.430, Def{clever, quick of apprehension, with a good faculty of hearing}
43C3 CJK UNIFIED IDEOGRAPH-#43C3, IRGKangXi=0968.440, Def{to hear, to listen carefully}
43C4 CJK UNIFIED IDEOGRAPH-#43C4, IRGKangXi=0969.090, Def{not to listen, disease of the ears, a dull sense of hearing (inefficacious; unresponsive)}
43C5 CJK UNIFIED IDEOGRAPH-#43C5, IRGKangXi=0969.130, Def{to hear; to listen, a keen sense of hearing}
43C6 CJK UNIFIED IDEOGRAPH-#43C6, IRGKangXi=0969.160, Def{buzzing in the ears; tinnitus aurium}
43C7 CJK UNIFIED IDEOGRAPH-#43C7, IRGKangXi=0969.210, Def{to hear; barely enough to hear clearly, hearsay; groundless talk}
43C8 CJK UNIFIED IDEOGRAPH-#43C8, IRGKangXi=0970.080, Def{(standard form of 聯) to unite; to connect; to join together (same as 攣) tangled; to bind; entwined}
43C9 CJK UNIFIED IDEOGRAPH-#43C9, IRGKangXi=0970.210, Def{to waver; to shake ears (said of cattle and horses)}
43CA CJK UNIFIED IDEOGRAPH-#43CA, IRGKangXi=0970.270, Def{(same as 聾) deaf; hard of hearing}
43CB CJK UNIFIED IDEOGRAPH-#43CB, IRGKangXi=0972.010, Def{(ancient form of 肅) respectful; solemn; serious; majestic, neat and quite}
43CC CJK UNIFIED IDEOGRAPH-#43CC, IRGKangXi=0973.100, Def{to vibrate; vibration, (same as 佾) a row or file of dancers, esp. referring to those in ancient dances at sacrifices or other rites}
43CD CJK UNIFIED IDEOGRAPH-#43CD, IRGKangXi=0973.150, Def{(same as 肙 蜎) a small worm; larvae of mosquitoes, empty, to twist; to surround}
43CE CJK UNIFIED IDEOGRAPH-#43CE, IRGKangXi=0973.160, Def{milk; breasts, the young of animals, birds}
43CF CJK UNIFIED IDEOGRAPH-#43CF, IRGKangXi=0974.130, Def{(same as 吁) signs and groans; moan, name of a county}
43D0 CJK UNIFIED IDEOGRAPH-#43D0, IRGKangXi=0974.140, Def{the buttocks, the anus}
43D1 CJK UNIFIED IDEOGRAPH-#43D1, IRGKangXi=0974.150, Def{(same as U+7099 炙) to burn; to heat; to roast; to broil}
43D2 CJK UNIFIED IDEOGRAPH-#43D2, IRGKangXi=0974.160, Def{half of the body, (non-classical form of 胖) fat; obese}
43D3 CJK UNIFIED IDEOGRAPH-#43D3, IRGKangXi=0974.190, Def{(same as 脘) the stomach; gizzard (of birds and fowls)}
43D4 CJK UNIFIED IDEOGRAPH-#43D4, IRGKangXi=0975.030, Def{to eat meat, (same as 肘) the elbow, (same as 腬) fine quality of meat, (same as 衄) nose bleeding}
43D5 CJK UNIFIED IDEOGRAPH-#43D5, IRGKangXi=0975.040, Def{(ancient form of 飪) to cook food thoroughly}
43D6 CJK UNIFIED IDEOGRAPH-#43D6, IRGKangXi=0975.110, Def{a wound; a bruise; a sore, scar, muscles of the back}
43D7 CJK UNIFIED IDEOGRAPH-#43D7, IRGKangXi=0976.120, Def{unexpectedly; suddenly aching of the midriff or waist; lumbago}
43D8 CJK UNIFIED IDEOGRAPH-#43D8, IRGKangXi=0976.130, Def{a big abdomen, a female animal with a fat belly}
43D9 CJK UNIFIED IDEOGRAPH-#43D9, IRGKangXi=0976.170, Def{(same as U+9193 醓) brine from pickled meat; condiments, juicy meat pulp; mashed meat cooked with soybean sauce, short and ugly, light of the setting moon; (Cant.) 肚䏙 stomach}
43DA CJK UNIFIED IDEOGRAPH-#43DA, IRGKangXi=0976.250, Def{small, missing; wanting; lost, the floating ribs on the sides of the trunk}
43DB CJK UNIFIED IDEOGRAPH-#43DB, IRGKangXi=0977.041, Def{(a corrupted form) a kind of insects; earthworm, strips of meat cut from the flank and dried in the wind}
43DC CJK UNIFIED IDEOGRAPH-#43DC, IRGKangXi=0977.041, Def{fat; plump; obesity; obese}
43DD CJK UNIFIED IDEOGRAPH-#43DD, IRGKangXi=0977.041, Def{(simplified form of 膞) sincere; earnest, a part of offering in sacrifice, gizzard of a fowl, chopped meat, small pig, the kneecap; patella, bone of one's limbs}
43DE CJK UNIFIED IDEOGRAPH-#43DE, IRGKangXi=0977.090, Def{the belly; the abdomen}
43DF CJK UNIFIED IDEOGRAPH-#43DF, IRGKangXi=0977.130, Def{fat meat, big; large}
43E0 CJK UNIFIED IDEOGRAPH-#43E0, IRGKangXi=0977.220, Def{(ancient form of 汁) meat soup, mixed meat}
43E1 CJK UNIFIED IDEOGRAPH-#43E1, IRGKangXi=0978.050, Def{uncooked or raw meat, an organic compound; Proteose}
43E2 CJK UNIFIED IDEOGRAPH-#43E2, IRGKangXi=0978.150, Def{meat, rough; cracked skin}
43E3 CJK UNIFIED IDEOGRAPH-#43E3, IRGKangXi=0979.040, Def{(same as non-classical form 蛆) maggot}
43E4 CJK UNIFIED IDEOGRAPH-#43E4, IRGKangXi=0979.060, Def{(same as U+9856 囟) top of the human head; the skull}
43E5 CJK UNIFIED IDEOGRAPH-#43E5, IRGKangXi=0980.080, Def{dry, dandruff-like scales of dead flesh and skin}
43E6 CJK UNIFIED IDEOGRAPH-#43E6, IRGKangXi=0980.160, Def{fat of animals or plants, (same as 胋) fat; plump, fertile, animal fat}
43E7 CJK UNIFIED IDEOGRAPH-#43E7, IRGKangXi=0980.180, Def{fatty; greasy, not thick; not tight, plump and pretty of the flesh}
43E8 CJK UNIFIED IDEOGRAPH-#43E8, IRGKangXi=0981.060, Def{to swell up greatly, to swell; a swelling; a boil}
43E9 CJK UNIFIED IDEOGRAPH-#43E9, IRGKangXi=0981.080, Def{(same as 脅) to shrug the shoulders, the ribs}
43EA CJK UNIFIED IDEOGRAPH-#43EA, IRGKangXi=0982.090, Def{tendon (of meat animals); sinews; muscles}
43EB CJK UNIFIED IDEOGRAPH-#43EB, IRGKangXi=0982.120, Def{in between of the abdomen and the spine; the spinal column}
43EC CJK UNIFIED IDEOGRAPH-#43EC, IRGKangXi=0981.110, Def{the spine; the spinal column, the ridge}
43ED CJK UNIFIED IDEOGRAPH-#43ED, IRGKangXi=0982.171, Def{the cheeks; the jaw}
43EE CJK UNIFIED IDEOGRAPH-#43EE, IRGKangXi=0982.171, Def{(abbreviated form of 脇=脅) the sides of the trunk from armpits to ribs; the flank}
43EF CJK UNIFIED IDEOGRAPH-#43EF, IRGKangXi=0983.070, Def{(same as 痣) moles; birthmarks}
43F0 CJK UNIFIED IDEOGRAPH-#43F0, IRGKangXi=0983.110, Def{a kind of insects; earthworm, durable; strong and soft; pliable yet tough}
43F1 CJK UNIFIED IDEOGRAPH-#43F1, IRGKangXi=0983.120, Def{body, fat; plump}
43F2 CJK UNIFIED IDEOGRAPH-#43F2, IRGKangXi=0983.160, Def{(same as 胅) protruded bones, swelling, a crooked nose}
43F3 CJK UNIFIED IDEOGRAPH-#43F3, IRGKangXi=0983.170, Def{(in chemistry) olein, oil and grease; fats (esp. animal), the ribs and skin}
43F4 CJK UNIFIED IDEOGRAPH-#43F4, IRGKangXi=0983.180, Def{dim; ambiguous; unclear eyesight, sharp ends of something}
43F5 CJK UNIFIED IDEOGRAPH-#43F5, IRGKangXi=0983.190, Def{plump, rich and full, plentiful, swelling, a plump trunk (body)}
43F6 CJK UNIFIED IDEOGRAPH-#43F6, IRGKangXi=0983.210, Def{the stomach; gizzard (of birds and fowls), (same as 髀) buttocks, thigh}
43F7 CJK UNIFIED IDEOGRAPH-#43F7, IRGKangXi=0984.030, Def{medicine for a sore; a boil (injure by a sword or a knife)}
43F8 CJK UNIFIED IDEOGRAPH-#43F8, IRGKangXi=0985.111, Def{fish, eyes of the fishes}
43F9 CJK UNIFIED IDEOGRAPH-#43F9, IRGKangXi=0985.111, Def{to watch stealthily}
43FA CJK UNIFIED IDEOGRAPH-#43FA, IRGKangXi=0985.111, Def{to swell; swelling}
43FB CJK UNIFIED IDEOGRAPH-#43FB, IRGKangXi=0985.111, Def{(same as 能) can; to be able to, capability; talent, energy}
43FC CJK UNIFIED IDEOGRAPH-#43FC, IRGKangXi=0985.150, Def{what is left over of the food (of birds and animals), a big belly}
43FD CJK UNIFIED IDEOGRAPH-#43FD, IRGKangXi=0985.180, Def{meat pulp; mashed meat cooked with soybean sauce, last name}
43FE CJK UNIFIED IDEOGRAPH-#43FE, IRGKangXi=0985.210, Def{(non-classical form of U+43FA 䏺) to swell; swelling}
43FF CJK UNIFIED IDEOGRAPH-#43FF, IRGKangXi=0985.260, Def{the calf of the legs}
4400 CJK UNIFIED IDEOGRAPH-#4400, IRGKangXi=0986.010, Def{to dismember body of the livestock}
4401 CJK UNIFIED IDEOGRAPH-#4401, IRGKangXi=0986.070, Def{the buttocks; the rump, fat; plump, one of the seven constellations; name of a star, the anus}
4402 CJK UNIFIED IDEOGRAPH-#4402, IRGKangXi=0986.100, Def{fat of animals or plants, fat; plump; obese}
4403 CJK UNIFIED IDEOGRAPH-#4403, IRGKangXi=0986.200, Def{fat in the abdomen or intestine, protuberances of the muscle, a fetus inside the belly}
4404 CJK UNIFIED IDEOGRAPH-#4404, IRGKangXi=0986.210, Def{hunger for meat, to cook meat}
4405 CJK UNIFIED IDEOGRAPH-#4405, IRGKangXi=0986.230, Def{the buttocks; the bottom; the rump, to groan; to moan}
4406 CJK UNIFIED IDEOGRAPH-#4406, IRGKangXi=0986.240, Def{big belly, swollen of the belly}
4407 CJK UNIFIED IDEOGRAPH-#4407, IRGKangXi=0986.260, Def{clear; bright, (same as 吻) to agree; match, (same as 唇) the lips; the labia}
4408 CJK UNIFIED IDEOGRAPH-#4408, IRGKangXi=0986.300, Def{a piece of one foot two inches dried seasoned meat, to paste up; to attach to; to stick up; to glue, (same as 殖) rotten fat; grease, lard, etc., bowels, fat}
4409 CJK UNIFIED IDEOGRAPH-#4409, IRGKangXi=0987.080, Def{fat, (interchangeable 腦) the head}
440A CJK UNIFIED IDEOGRAPH-#440A, IRGKangXi=0987.140, Def{(same as 䖵) insects, (same as 餛) cakes; biscuits; pastry, an ellipse}
440B CJK UNIFIED IDEOGRAPH-#440B, IRGKangXi=0987.210, Def{sick; illness; disease, red}
440C CJK UNIFIED IDEOGRAPH-#440C, IRGKangXi=0986.170, Def{(ancient form of 腆) luxuriant; abundant; rich; plentiful of food, brazen-faced; shameless, good; virtuous, protruding, as belly, etc., to go to excess}
440D CJK UNIFIED IDEOGRAPH-#440D, IRGKangXi=0988.040, Def{(same as 䐢) excellent food; delicacies, to wrinkle; to contract, beautiful; elegant, dried and seasoned meat}
440E CJK UNIFIED IDEOGRAPH-#440E, IRGKangXi=0988.080, Def{fat; greasy}
440F CJK UNIFIED IDEOGRAPH-#440F, IRGKangXi=0988.130, Def{fat; plump}
4410 CJK UNIFIED IDEOGRAPH-#4410, IRGKangXi=0988.150, Def{curved part of the knee, between the thigh and calf}
4411 CJK UNIFIED IDEOGRAPH-#4411, IRGKangXi=0988.190, Def{thin cut of the sliced meat}
4412 CJK UNIFIED IDEOGRAPH-#4412, IRGKangXi=0988.270, Def{not fine; coarse, to stickup; to paste up; to attach to; to glue, scar}
4413 CJK UNIFIED IDEOGRAPH-#4413, IRGKangXi=0988.280, Def{soft, (same as U+811C 脜) gentle; mild; temperate facial expression}
4414 CJK UNIFIED IDEOGRAPH-#4414, IRGKangXi=0988.290, Def{muscle of the calf (of the leg), tendon (of meat animals), stopping and rising of the pulse like a plaited ropes}
4415 CJK UNIFIED IDEOGRAPH-#4415, IRGKangXi=0989.090, Def{oil and grease; fats, the bursting of a wound}
4416 CJK UNIFIED IDEOGRAPH-#4416, IRGKangXi=0989.190, Def{(same as 㕧) to laugh heartily; to roar with laughter; to groan; to moan}
4417 CJK UNIFIED IDEOGRAPH-#4417, IRGKangXi=0989.220, Def{(same as 豬) a pig, (same as 肚) the belly; the abdomen}
4418 CJK UNIFIED IDEOGRAPH-#4418, IRGKangXi=0989.230, Def{to laugh heartily; to roar with laughter, tongue of the cow}
4419 CJK UNIFIED IDEOGRAPH-#4419, IRGKangXi=0989.240, Def{(same as 膈) the diaphragm}
441A CJK UNIFIED IDEOGRAPH-#441A, IRGKangXi=0990.120, Def{good graces; great favors, luster}
441B CJK UNIFIED IDEOGRAPH-#441B, IRGKangXi=0990.131, Def{to loosen; to widen skin}
441C CJK UNIFIED IDEOGRAPH-#441C, IRGKangXi=0990.170, Def{a swollen belly; dropsical, edematous; to swell; inflated}
441D CJK UNIFIED IDEOGRAPH-#441D, IRGKangXi=0990.180, Def{thick; heavy fat of animals or plants}
441E CJK UNIFIED IDEOGRAPH-#441E, IRGKangXi=0990.190, Def{the thin membrane between skin and flesh that protects internal organs or tissues in the human body, weak; fragile; delicate, flesh, meat}
441F CJK UNIFIED IDEOGRAPH-#441F, IRGKangXi=0990.240, Def{fat; plump, bent; curved feet}
4420 CJK UNIFIED IDEOGRAPH-#4420, IRGKangXi=0990.260, Def{between the flesh, name of state in southern China (of minority group), dim moon}
4421 CJK UNIFIED IDEOGRAPH-#4421, IRGKangXi=0990.270, Def{(same as 臍) the navel; the umbilicus, the underside of a crab}
4422 CJK UNIFIED IDEOGRAPH-#4422, IRGKangXi=0990.290, Def{to wrinkle; to contract, beautiful; elegant, dried and seasoned meat, excellent food; delicacies}
4423 CJK UNIFIED IDEOGRAPH-#4423, IRGKangXi=0990.340, Def{soup made of cut meat mixed with blood}
4424 CJK UNIFIED IDEOGRAPH-#4424, IRGKangXi=0991.010, Def{dried and seasoned meat, sound of the abdomen}
4425 CJK UNIFIED IDEOGRAPH-#4425, IRGKangXi=0991.060, Def{stinky; offensive-smelling; odorous, fat; plump}
4426 CJK UNIFIED IDEOGRAPH-#4426, IRGKangXi=0991.070, Def{sleepy}
4427 CJK UNIFIED IDEOGRAPH-#4427, IRGKangXi=0991.090, Def{(same as 犒) to reward or cheer troops with food, money, gifts, etc., (same as 臛) meat broth}
4428 CJK UNIFIED IDEOGRAPH-#4428, IRGKangXi=0991.210, Def{back of the foot, the back feet of the animal, covering; shell, to strike; heat from the top}
4429 CJK UNIFIED IDEOGRAPH-#4429, IRGKangXi=0991.260, Def{fat; plump, the fetus of livestock, delicious meat}
442A CJK UNIFIED IDEOGRAPH-#442A, IRGKangXi=0992.071, Def{(same as standard form 臇) fat; rich, a stew of fish}
442B CJK UNIFIED IDEOGRAPH-#442B, IRGKangXi=0992.120, Def{obesity, fat; plump}
442C CJK UNIFIED IDEOGRAPH-#442C, IRGKangXi=0992.130, Def{crisp, brittle, light, sound from the abdomen}
442D CJK UNIFIED IDEOGRAPH-#442D, IRGKangXi=0992.140, Def{a discharge or flux, diarrhoea and dysentery}
442E CJK UNIFIED IDEOGRAPH-#442E, IRGKangXi=0992.160, Def{skin}
442F CJK UNIFIED IDEOGRAPH-#442F, IRGKangXi=0992.210, Def{ugly appearance of a person, dry meat; preserved meat}
4430 CJK UNIFIED IDEOGRAPH-#4430, IRGKangXi=0992.220, Def{(same as U+7F9E 羞) to offer food as tribute; delicacies, savoury food}
4431 CJK UNIFIED IDEOGRAPH-#4431, IRGKangXi=0992.310, Def{to pick the meat in between of bones}
4432 CJK UNIFIED IDEOGRAPH-#4432, IRGKangXi=0992.320, Def{half-raw meat; not well cooked meat, the light or grow of fire, raw meat, offensive smell, especially of fish or blood, thin sliced meat}
4433 CJK UNIFIED IDEOGRAPH-#4433, IRGKangXi=0993.050, Def{(same as 魚) fish}
4434 CJK UNIFIED IDEOGRAPH-#4434, IRGKangXi=0993.120, Def{lumbago}
4435 CJK UNIFIED IDEOGRAPH-#4435, IRGKangXi=0994.120, Def{fat; obese, swelling}
4436 CJK UNIFIED IDEOGRAPH-#4436, IRGKangXi=0994.160, Def{to cook; to boil; to decoct, diseases of the lips; hare-lip, pieces of meat}
4437 CJK UNIFIED IDEOGRAPH-#4437, IRGKangXi=0994.200, Def{paunchy; a large belly}
4438 CJK UNIFIED IDEOGRAPH-#4438, IRGKangXi=0994.240, Def{(non-classical form of 膕) the back of the knee, crooked feet}
4439 CJK UNIFIED IDEOGRAPH-#4439, IRGKangXi=0994.270, Def{preserved and dried meat, (a dialect) fried; cooked small pieces of meat, meat broth}
443A CJK UNIFIED IDEOGRAPH-#443A, IRGKangXi=0994.310, Def{delicious; tasty food, (same as U+91B0 醰) of a fine flavour}
443B CJK UNIFIED IDEOGRAPH-#443B, IRGKangXi=0994.320, Def{big pieces of dried meat}
443C CJK UNIFIED IDEOGRAPH-#443C, IRGKangXi=0995.070, Def{thin membrane of the throat; bullet}
443D CJK UNIFIED IDEOGRAPH-#443D, IRGKangXi=0993.201
443E CJK UNIFIED IDEOGRAPH-#443E, IRGKangXi=0995.120, Def{fat; plump}
443F CJK UNIFIED IDEOGRAPH-#443F, IRGKangXi=0995.210, Def{gizzard, to store up meat, the five lowest pieces of bone of the spinal column}
4440 CJK UNIFIED IDEOGRAPH-#4440, IRGKangXi=0996.100, Def{the navel, the bowels; the entrails; the viscera, (same as 癖) chronic swelling of the spleen}
4441 CJK UNIFIED IDEOGRAPH-#4441, IRGKangXi=0996.200, Def{fatty; greasy, ointment}
4442 CJK UNIFIED IDEOGRAPH-#4442, IRGKangXi=0996.310, Def{fat; plump}
4443 CJK UNIFIED IDEOGRAPH-#4443, IRGKangXi=0997.080, Def{dim moon, dark; obscure, stupid; ignorant, to cheat; to hide the truth, to cover up; to hide, plump; fleshy}
4444 CJK UNIFIED IDEOGRAPH-#4444, IRGKangXi=0997.090, Def{rich; abundant; big and strong; vigorous, husky, (same as 瘜) (nasal) polypus}
4445 CJK UNIFIED IDEOGRAPH-#4445, IRGKangXi=0997.161, Def{(non-classical form) dizzy and sleepy; slumberous}
4446 CJK UNIFIED IDEOGRAPH-#4446, IRGKangXi=0997.220, Def{feel like vomiting}
4447 CJK UNIFIED IDEOGRAPH-#4447, IRGKangXi=0997.280, Def{medicine for a sore; a boil (injure by a sword or a knife)}
4448 CJK UNIFIED IDEOGRAPH-#4448, IRGKangXi=0997.290, Def{meat to offer for worship}
4449 CJK UNIFIED IDEOGRAPH-#4449, IRGKangXi=0998.040, Def{excrement, shit; night soil}
444A CJK UNIFIED IDEOGRAPH-#444A, IRGKangXi=0998.090, Def{fat; plump, physical disease; carnal}
444B CJK UNIFIED IDEOGRAPH-#444B, IRGKangXi=0998.160, Def{fat; obese}
444C CJK UNIFIED IDEOGRAPH-#444C, IRGKangXi=0998.170, Def{what is left over of the food (of birds and animals); (same as 爛) overripe; rotten, cooked soft; well cooked}
444D CJK UNIFIED IDEOGRAPH-#444D, IRGKangXi=0998.180, Def{(same as 咽) to swallow; the throat, (same as 癭) a reddish swelling on the neck, gnarl}
444E CJK UNIFIED IDEOGRAPH-#444E, IRGKangXi=0998.190, Def{soup of pig's intestines, meat soup; meat broth}
444F CJK UNIFIED IDEOGRAPH-#444F, IRGKangXi=0998.280, Def{ugly, a legendary animal}
4450 CJK UNIFIED IDEOGRAPH-#4450, IRGKangXi=0999.200, Def{bright; light, clear, (same as 瞇) to close the eyes; to narrow the eyes}
4451 CJK UNIFIED IDEOGRAPH-#4451, IRGKangXi=1000.020, Def{(ancient form of 僕) a servant, a modest term referring to oneself}
4452 CJK UNIFIED IDEOGRAPH-#4452, IRGKangXi=1001.191, Def{a kind of farm tool used in ancient times}
4453 CJK UNIFIED IDEOGRAPH-#4453, IRGKangXi=1002.030, Def{(same as 臺) a lookout, a tower, a terrace, a platform, a stage}
4454 CJK UNIFIED IDEOGRAPH-#4454, IRGKangXi=1003.100, Def{to pound (grain) in order to remove the husk}
4455 CJK UNIFIED IDEOGRAPH-#4455, IRGKangXi=1004.041, Def{(a non-classical abbreviated form of 鼠) a rat, a mouse, squirrels, moles, etc., KangXi radical 208}
4456 CJK UNIFIED IDEOGRAPH-#4456, IRGKangXi=1005.021
4457 CJK UNIFIED IDEOGRAPH-#4457, IRGKangXi=1005.030, Def{to pound (grain) in order to remove the husk}
4458 CJK UNIFIED IDEOGRAPH-#4458, IRGKangXi=1005.100, Def{to pound; to ram down, to polish; to grind barley or wheat}
4459 CJK UNIFIED IDEOGRAPH-#4459, IRGKangXi=1007.070, Def{put out the tongue; (same as U+3401 㐁) to lick; to taste}
445A CJK UNIFIED IDEOGRAPH-#445A, IRGKangXi=1007.071, Def{(non-classical form of 甜) sweet; agreeable; pleasant}
445B CJK UNIFIED IDEOGRAPH-#445B, IRGKangXi=1007.081, Def{(standard form of 舐) to lick}
445C CJK UNIFIED IDEOGRAPH-#445C, IRGKangXi=1007.220, Def{to eat heavily, cooked rice, a meal, to drink, to swallow, to lick, the uvula, the ligule}
445D CJK UNIFIED IDEOGRAPH-#445D, IRGKangXi=1008.180, Def{(non-classical form of 斝) a small cup with ears, used in ancient times for libations}
445E CJK UNIFIED IDEOGRAPH-#445E, IRGKangXi=1008.190, Def{(same as 舜) Shun, name of a legendary Chinese ruler, said to have ruled from 2255-2205 B.C., wise; good}
445F CJK UNIFIED IDEOGRAPH-#445F, IRGKangXi=1008.210, Def{(same as 葟) luxuriant; exuberant; flourishing (said of grass and tress; vegetation; flora)}
4460 CJK UNIFIED IDEOGRAPH-#4460, IRGKangXi=1009.050, Def{long and narrow boat, a small boat}
4461 CJK UNIFIED IDEOGRAPH-#4461, IRGKangXi=1009.060, Def{(same as 艖) small boat (short and deep)}
4462 CJK UNIFIED IDEOGRAPH-#4462, IRGKangXi=1009.070, Def{(non-classical form) (same as 舠) a knife shaped small boat}
4463 CJK UNIFIED IDEOGRAPH-#4463, IRGKangXi=1009.090, Def{a sailing boat}
4464 CJK UNIFIED IDEOGRAPH-#4464, IRGKangXi=1009.210, Def{a boat; a ship; vessel}
4465 CJK UNIFIED IDEOGRAPH-#4465, IRGKangXi=1009.220, Def{a moving boat; a sailing boat, tools used on a boat, a speedy boat}
4466 CJK UNIFIED IDEOGRAPH-#4466, IRGKangXi=1010.040, Def{a boat; a ship; vessel}
4467 CJK UNIFIED IDEOGRAPH-#4467, IRGKangXi=1010.050, Def{a short, deep and small boat}
4468 CJK UNIFIED IDEOGRAPH-#4468, IRGKangXi=1010.110, Def{(same as 舵) rudder; helm}
4469 CJK UNIFIED IDEOGRAPH-#4469, IRGKangXi=1010.221, Def{a kind of inland rivers warship used in Qing Dynasty}
446A CJK UNIFIED IDEOGRAPH-#446A, IRGKangXi=1010.280, Def{a sailing; moving boat}
446B CJK UNIFIED IDEOGRAPH-#446B, IRGKangXi=1010.290, Def{tools for boat, name of a boat}
446C CJK UNIFIED IDEOGRAPH-#446C, IRGKangXi=1010.340, Def{boat, large boat, to dispel; to remove; to move, to stir up}
446D CJK UNIFIED IDEOGRAPH-#446D, IRGKangXi=1010.351, Def{war vessel; man-of-war; warship}
446E CJK UNIFIED IDEOGRAPH-#446E, IRGKangXi=1010.351
446F CJK UNIFIED IDEOGRAPH-#446F, IRGKangXi=1011.040, Def{a boat; a ship; vessel}
4470 CJK UNIFIED IDEOGRAPH-#4470, IRGKangXi=1011.100, Def{small boat}
4471 CJK UNIFIED IDEOGRAPH-#4471, IRGKangXi=1011.140, Def{a boat; a ship; vessel}
4472 CJK UNIFIED IDEOGRAPH-#4472, IRGKangXi=1011.160, Def{(same as 棹 櫂) to row a boat, an oar, a scull, a boat}
4473 CJK UNIFIED IDEOGRAPH-#4473, IRGKangXi=1011.190, Def{a horizontal bar in front of a boat, a boat}
4474 CJK UNIFIED IDEOGRAPH-#4474, IRGKangXi=1011.200, Def{a boat; a ship; vessel}
4475 CJK UNIFIED IDEOGRAPH-#4475, IRGKangXi=1011.210, Def{(corrupted form of 艒) a small boat}
4476 CJK UNIFIED IDEOGRAPH-#4476, IRGKangXi=1011.250, Def{sprightly boat; light boat}
4477 CJK UNIFIED IDEOGRAPH-#4477, IRGKangXi=1011.271
4478 CJK UNIFIED IDEOGRAPH-#4478, IRGKangXi=1011.271, Def{a group of boats, boat; warship}
4479 CJK UNIFIED IDEOGRAPH-#4479, IRGKangXi=1011.360, Def{(same as 艘) a numerary adjunct for ships, groove of a boat}
447A CJK UNIFIED IDEOGRAPH-#447A, IRGKangXi=1011.400, Def{(same as 帆) a sail (of a boat)}
447B CJK UNIFIED IDEOGRAPH-#447B, IRGKangXi=1011.410, Def{covering; curtain, or screen of a boat, to sail a boat}
447C CJK UNIFIED IDEOGRAPH-#447C, IRGKangXi=1012.040, Def{ocean-going liner, steamer, a boat; a ship; vessel}
447D CJK UNIFIED IDEOGRAPH-#447D, IRGKangXi=1012.100, Def{a large boat}
447E CJK UNIFIED IDEOGRAPH-#447E, IRGKangXi=1012.131
447F CJK UNIFIED IDEOGRAPH-#447F, IRGKangXi=1012.140, Def{small boat}
4480 CJK UNIFIED IDEOGRAPH-#4480, IRGKangXi=1012.150, Def{short and deep boat}
4481 CJK UNIFIED IDEOGRAPH-#4481, IRGKangXi=1012.180, Def{a small boat connected to the back of a big boat; used to load the passengers, cargo or goods, warship, a kind of tools used to capsize}
4482 CJK UNIFIED IDEOGRAPH-#4482, IRGKangXi=1012.210, Def{an oar}
4483 CJK UNIFIED IDEOGRAPH-#4483, IRGKangXi=1012.230, Def{(same as U+8241 造) to build; to make; to do, to arrive at; to go to}
4484 CJK UNIFIED IDEOGRAPH-#4484, IRGKangXi=1012.280, Def{fully laden of a boat, name of a ship}
4485 CJK UNIFIED IDEOGRAPH-#4485, IRGKangXi=0817.130, Def{beautiful eyes, (dialect) a pair; two}
4486 CJK UNIFIED IDEOGRAPH-#4486, IRGKangXi=1012.400, Def{name of a ship, a sailing boat}
4487 CJK UNIFIED IDEOGRAPH-#4487, IRGKangXi=1012.441, Def{name of a ship, fully load of a boat}
4488 CJK UNIFIED IDEOGRAPH-#4488, IRGKangXi=1012.560, Def{(same as 橨) side beam of a boat}
4489 CJK UNIFIED IDEOGRAPH-#4489, IRGKangXi=1013.020, Def{a long, narrow boat}
448A CJK UNIFIED IDEOGRAPH-#448A, IRGKangXi=1014.040, Def{a patient (of jaundice; icterus) facial complexion}
448B CJK UNIFIED IDEOGRAPH-#448B, IRGKangXi=1014.050, Def{air current; atmospheric current}
448C CJK UNIFIED IDEOGRAPH-#448C, IRGKangXi=1014.130, Def{dark blue color, without color, to close the eyes}
448D CJK UNIFIED IDEOGRAPH-#448D, IRGKangXi=1014.170, Def{without color, clear and lucid}
448E CJK UNIFIED IDEOGRAPH-#448E, IRGKangXi=1014.180, Def{without color, clear and lucid}
448F CJK UNIFIED IDEOGRAPH-#448F, IRGKangXi=1014.190, Def{disturbed in mind, not in good health, a fierce or ferocious appearance}
4490 CJK UNIFIED IDEOGRAPH-#4490, IRGKangXi=1014.230, Def{disturbed in mind, not in good health, a fierce or ferocious appearance, a bad look}
4491 CJK UNIFIED IDEOGRAPH-#4491, IRGKangXi=1017.011, Def{(non-classical form of 艸) grass; straw; herbs; weeds}
4492 CJK UNIFIED IDEOGRAPH-#4492, IRGKangXi=1017.060, Def{(same as 芀) name of a plant (much used for making brooms); a reed}
4493 CJK UNIFIED IDEOGRAPH-#4493, IRGKangXi=1017.110, Def{to do violence, perverse; rebellious, calamities; tribulations; miseries; crime; sin}
4494 CJK UNIFIED IDEOGRAPH-#4494, IRGKangXi=1017.191, Def{name of a variety of grass}
4495 CJK UNIFIED IDEOGRAPH-#4495, IRGKangXi=1017.200, Def{a kind of herb medicine, luxuriant; flourishing of grass and trees}
4496 CJK UNIFIED IDEOGRAPH-#4496, IRGKangXi=1018.020, Def{a kind of herb medicine (plants of the mugwort or artemisia family)}
4497 CJK UNIFIED IDEOGRAPH-#4497, IRGKangXi=1018.130, Def{(same as 芞) fragrant herb, vanilla}
4498 CJK UNIFIED IDEOGRAPH-#4498, IRGKangXi=1019.061, Def{(same as 荾) parsley, (same as 葰) ginger, pistil}
4499 CJK UNIFIED IDEOGRAPH-#4499, IRGKangXi=1019.061
449A CJK UNIFIED IDEOGRAPH-#449A, IRGKangXi=1019.070, Def{name of a variety of grass}
449B CJK UNIFIED IDEOGRAPH-#449B, IRGKangXi=1019.190, Def{name of a variety of grass}
449C CJK UNIFIED IDEOGRAPH-#449C, IRGKangXi=1019.200, Def{name of a variety of grass, a kind of herb}
449D CJK UNIFIED IDEOGRAPH-#449D, IRGKangXi=1019.240, Def{root of the thatch, straw or couch grass; a herb medicine to quench thirst}
449E CJK UNIFIED IDEOGRAPH-#449E, IRGKangXi=1019.250, Def{name of a variety of grass, hot}
449F CJK UNIFIED IDEOGRAPH-#449F, IRGKangXi=1020.060, Def{name of a variety of grass}
44A0 CJK UNIFIED IDEOGRAPH-#44A0, IRGKangXi=1020.070, Def{(same as 丰) luxuriant; exuberant of the grass and trees, young shoots of the rape-turnip--Brassica rapa}
44A1 CJK UNIFIED IDEOGRAPH-#44A1, IRGKangXi=1020.080, Def{name of a variety of grass}
44A2 CJK UNIFIED IDEOGRAPH-#44A2, IRGKangXi=1021.120, Def{sweet flag; the calamus; its leaves are hung on the door lintels on the 5th of the 5th lunar month, to avert evil influences}
44A3 CJK UNIFIED IDEOGRAPH-#44A3, IRGKangXi=1021.160, Def{(same as 苒) (of flowers and grass) lush or delicate}
44A4 CJK UNIFIED IDEOGRAPH-#44A4, IRGKangXi=1022.030, Def{name of a variety of grass}
44A5 CJK UNIFIED IDEOGRAPH-#44A5, IRGKangXi=1022.160, Def{corresponding; equivalent, considerable; to a great extent, appropriate}
44A6 CJK UNIFIED IDEOGRAPH-#44A6, IRGKangXi=1022.220, Def{grass floats on water}
44A7 CJK UNIFIED IDEOGRAPH-#44A7, IRGKangXi=1023.010, Def{a vessel or utensil made of straw, grass or weeds, a kind of grass}
44A8 CJK UNIFIED IDEOGRAPH-#44A8, IRGKangXi=1025.030, Def{a kind of vegetable, plants of the mugwort or artemisia family}
44A9 CJK UNIFIED IDEOGRAPH-#44A9, IRGKangXi=1025.160, Def{name of a variety of grass, (same as 莫) not}
44AA CJK UNIFIED IDEOGRAPH-#44AA, IRGKangXi=1027.030, Def{name of a variety of grass}
44AB CJK UNIFIED IDEOGRAPH-#44AB, IRGKangXi=1027.100, Def{grassy}
44AC CJK UNIFIED IDEOGRAPH-#44AC, IRGKangXi=1027.170, Def{(ancient form of 莫) not, (standard form of 暮) sunset; dusk}
44AD CJK UNIFIED IDEOGRAPH-#44AD, IRGKangXi=1027.201, Def{(same as 等) rank; grade, same; equal, to wait, and so on; etc.}
44AE CJK UNIFIED IDEOGRAPH-#44AE, IRGKangXi=1027.201
44AF CJK UNIFIED IDEOGRAPH-#44AF, IRGKangXi=1027.201
44B0 CJK UNIFIED IDEOGRAPH-#44B0, IRGKangXi=1027.320, Def{to follow; to trace, to submit; to accord with, to accompany, a kind of grass}
44B1 CJK UNIFIED IDEOGRAPH-#44B1, IRGKangXi=1027.360, Def{(same as 蒸) twigs of hemp used for fuel; to rise, as steam, steam}
44B2 CJK UNIFIED IDEOGRAPH-#44B2, IRGKangXi=1027.380, Def{another name name for 黃岑 (Scutellaria baikalensis)}
44B3 CJK UNIFIED IDEOGRAPH-#44B3, IRGKangXi=1028.060, Def{(same as U+6736 朵) a flower; a cluster of flowers; a bud, lobe of the ear}
44B4 CJK UNIFIED IDEOGRAPH-#44B4, IRGKangXi=1028.110, Def{name of a variety of grass}
44B5 CJK UNIFIED IDEOGRAPH-#44B5, IRGKangXi=1028.200, Def{name of a variety of grass}
44B6 CJK UNIFIED IDEOGRAPH-#44B6, IRGKangXi=1028.250, Def{(ancient form of 天) the sky; the heaven, Nature; God, day}
44B7 CJK UNIFIED IDEOGRAPH-#44B7, IRGKangXi=1028.270, Def{a kind of plant, a kind of herb; Chinese trichosanthes (Trichosanthes kirilowii); bryonia}
44B8 CJK UNIFIED IDEOGRAPH-#44B8, IRGKangXi=1029.080, Def{grass}
44B9 CJK UNIFIED IDEOGRAPH-#44B9, IRGKangXi=1030.020, Def{grassy, fruitful}
44BA CJK UNIFIED IDEOGRAPH-#44BA, IRGKangXi=1030.100, Def{(ancient form of 堇) clay}
44BB CJK UNIFIED IDEOGRAPH-#44BB, IRGKangXi=1030.130, Def{a medical herb; acrid taste; poisonless; harmless}
44BC CJK UNIFIED IDEOGRAPH-#44BC, IRGKangXi=1030.160, Def{(same as 曲) a bamboo tray for raising silkworms}
44BD CJK UNIFIED IDEOGRAPH-#44BD, IRGKangXi=1031.150, Def{Illicium anisatum, and kind of poisonous shrub}
44BE CJK UNIFIED IDEOGRAPH-#44BE, IRGKangXi=1031.381
44BF CJK UNIFIED IDEOGRAPH-#44BF, IRGKangXi=1031.381
44C0 CJK UNIFIED IDEOGRAPH-#44C0, IRGKangXi=1031.381
44C1 CJK UNIFIED IDEOGRAPH-#44C1, IRGKangXi=1031.381
44C2 CJK UNIFIED IDEOGRAPH-#44C2, IRGKangXi=1032.020, Def{name of a variety of grass, last name}
44C3 CJK UNIFIED IDEOGRAPH-#44C3, IRGKangXi=1032.070, Def{melancholy and sad, withered (said of grass and trees)}
44C4 CJK UNIFIED IDEOGRAPH-#44C4, IRGKangXi=1032.080, Def{grassy, name of a variety of grass}
44C5 CJK UNIFIED IDEOGRAPH-#44C5, IRGKangXi=1032.090, Def{(same as U+82B9 芹) celery; a kind of water plant}
44C6 CJK UNIFIED IDEOGRAPH-#44C6, IRGKangXi=1032.120, Def{to cut the; to sever grass}
44C7 CJK UNIFIED IDEOGRAPH-#44C7, IRGKangXi=1033.170, Def{luxuriant; lush of grass and trees, (same as 赫) bright; glowing, brilliant; glorious, (same as 嚇) angry, fear; dread; fright; scare}
44C8 CJK UNIFIED IDEOGRAPH-#44C8, IRGKangXi=1033.190, Def{a kind of plant; chicken-head; Euryale ferox}
44C9 CJK UNIFIED IDEOGRAPH-#44C9, IRGKangXi=1033.200, Def{name of a variety of grass, a kind of vegetable, (same as 椰) coconut; coconut palm; coconut tree}
44CA CJK UNIFIED IDEOGRAPH-#44CA, IRGKangXi=1033.340, Def{name of a variety of grass, (as moxa) (Perilla ocimoides)}
44CB CJK UNIFIED IDEOGRAPH-#44CB, IRGKangXi=1034.010, Def{elm-seeds -- used as food}
44CC CJK UNIFIED IDEOGRAPH-#44CC, IRGKangXi=1034.110, Def{polygala (a medical herb)}
44CD CJK UNIFIED IDEOGRAPH-#44CD, IRGKangXi=1034.120, Def{cruciferous flowers; with acrid seeds; used as medicine}
44CE CJK UNIFIED IDEOGRAPH-#44CE, IRGKangXi=1034.150, Def{name of a variety of grass}
44CF CJK UNIFIED IDEOGRAPH-#44CF, IRGKangXi=1035.200, Def{flourishing; luxuriant; a lush growth of flowers}
44D0 CJK UNIFIED IDEOGRAPH-#44D0, IRGKangXi=1036.030, Def{name of a variety of grass}
44D1 CJK UNIFIED IDEOGRAPH-#44D1, IRGKangXi=1036.120, Def{(same as 荓) name of a variety of grass (same as 萍) duckweeds}
44D2 CJK UNIFIED IDEOGRAPH-#44D2, IRGKangXi=1036.280, Def{(same as 蒯) Scirpus cyperinus var, concolor, a rush, from which many things are woven}
44D3 CJK UNIFIED IDEOGRAPH-#44D3, IRGKangXi=1036.401, Def{(simplified form of 薵) name of a variety of grass, (same as 葤) use grass to wrap things, to cover; to put a cover over something}
44D4 CJK UNIFIED IDEOGRAPH-#44D4, IRGKangXi=1036.401
44D5 CJK UNIFIED IDEOGRAPH-#44D5, IRGKangXi=1042.261, Def{polygala (a medical herb)}
44D6 CJK UNIFIED IDEOGRAPH-#44D6, IRGKangXi=1036.401, Def{(simplified form of 藭) Cnidium officinale, a kind of medicinal herb}
44D7 CJK UNIFIED IDEOGRAPH-#44D7, IRGKangXi=1037.230, Def{(same as U+8471 葱 U+8525 蔥) scallions; leeks; onions}
44D8 CJK UNIFIED IDEOGRAPH-#44D8, IRGKangXi=1038.050, Def{a kind of plant, name of a variety of grass}
44D9 CJK UNIFIED IDEOGRAPH-#44D9, IRGKangXi=1038.070, Def{slanting; inclined; sloping}
44DA CJK UNIFIED IDEOGRAPH-#44DA, IRGKangXi=1038.100, Def{a kind of vegetable, a water-plant; the bulbs of which art used as food -- Sagittaria sagittifolia}
44DB CJK UNIFIED IDEOGRAPH-#44DB, IRGKangXi=1038.290, Def{molecular formula (C18H12), to brush}
44DC CJK UNIFIED IDEOGRAPH-#44DC, IRGKangXi=1038.330, Def{salted or pickled vegetables}
44DD CJK UNIFIED IDEOGRAPH-#44DD, IRGKangXi=1039.170, Def{Chinese pennisetum (Pennisetum alopecuroides)}
44DE CJK UNIFIED IDEOGRAPH-#44DE, IRGKangXi=1039.260, Def{a kind of plant used as dyestuff, material for making paper and coir raincoats, edible, Chinese pennisetum (Pennisetum alopecuroides)}
44DF CJK UNIFIED IDEOGRAPH-#44DF, IRGKangXi=1040.020, Def{name of a variety of grass, (same as 茢) a broom}
44E0 CJK UNIFIED IDEOGRAPH-#44E0, IRGKangXi=1040.330, Def{the water plantain, the lotus floating on the water}
44E1 CJK UNIFIED IDEOGRAPH-#44E1, IRGKangXi=1041.080, Def{a medical herb (知母)}
44E2 CJK UNIFIED IDEOGRAPH-#44E2, IRGKangXi=1041.090, Def{name of a variety of grass}
44E3 CJK UNIFIED IDEOGRAPH-#44E3, IRGKangXi=1041.190, Def{name of a variety of grass}
44E4 CJK UNIFIED IDEOGRAPH-#44E4, IRGKangXi=1041.360, Def{vegetation}
44E5 CJK UNIFIED IDEOGRAPH-#44E5, IRGKangXi=1042.030, Def{sound of the wriggled insects on the grass}
44E6 CJK UNIFIED IDEOGRAPH-#44E6, IRGKangXi=1042.200, Def{(same as 典) a rule; a law, a tale or story, pawn; to mortgage, to take charge of}
44E7 CJK UNIFIED IDEOGRAPH-#44E7, IRGKangXi=1042.261, Def{(same as 亟) urgent; pressing}
44E8 CJK UNIFIED IDEOGRAPH-#44E8, IRGKangXi=1042.261, Def{(simplified form of 罃) long-necked bottle; an earthenware jar with a small mouth and two or four ears}
44E9 CJK UNIFIED IDEOGRAPH-#44E9, IRGKangXi=1042.261
44EA CJK UNIFIED IDEOGRAPH-#44EA, IRGKangXi=1042.261
44EB CJK UNIFIED IDEOGRAPH-#44EB, IRGKangXi=1042.261, Def{name of a mountain, in Taiwan Province}
44EC CJK UNIFIED IDEOGRAPH-#44EC, IRGKangXi=1042.261
44ED CJK UNIFIED IDEOGRAPH-#44ED, IRGKangXi=1042.261, Def{weed or grass with narrow thick glades (as bamboo); water-pepper group}
44EE CJK UNIFIED IDEOGRAPH-#44EE, IRGKangXi=1042.300, Def{lush growth; dense growth of silky grass; fine grass}
44EF CJK UNIFIED IDEOGRAPH-#44EF, IRGKangXi=1042.310, Def{water-pepper}
44F0 CJK UNIFIED IDEOGRAPH-#44F0, IRGKangXi=1042.450, Def{absinthin, Artemisia capillaris, a kind of medicinal herb}
44F1 CJK UNIFIED IDEOGRAPH-#44F1, IRGKangXi=1043.320, Def{a kind of herb medicine; Bupleurum}
44F2 CJK UNIFIED IDEOGRAPH-#44F2, IRGKangXi=1044.210, Def{fine; silky; new growth of grass}
44F3 CJK UNIFIED IDEOGRAPH-#44F3, IRGKangXi=1044.230, Def{name of a variety of grass}
44F4 CJK UNIFIED IDEOGRAPH-#44F4, IRGKangXi=1045.020, Def{edible fungus from trees}
44F5 CJK UNIFIED IDEOGRAPH-#44F5, IRGKangXi=1045.080, Def{to blossom; to flower}
44F6 CJK UNIFIED IDEOGRAPH-#44F6, IRGKangXi=1045.150, Def{(same as 藾) a kind of plant; a kind of herb}
44F7 CJK UNIFIED IDEOGRAPH-#44F7, IRGKangXi=1047.250, Def{(same as 莕 荇) Nymphoides peltalum; a kind of water plant, name of a variety of grass}
44F8 CJK UNIFIED IDEOGRAPH-#44F8, IRGKangXi=1047.340, Def{weed}
44F9 CJK UNIFIED IDEOGRAPH-#44F9, IRGKangXi=1047.510, Def{(same as 饐) to stuff full and to gulp down, cooked food which has become mouldy, sour}
44FA CJK UNIFIED IDEOGRAPH-#44FA, IRGKangXi=1047.670, Def{name of a variety of grass}
44FB CJK UNIFIED IDEOGRAPH-#44FB, IRGKangXi=1047.671
44FC CJK UNIFIED IDEOGRAPH-#44FC, IRGKangXi=1047.690, Def{name of a variety of grass, grunting of hogs}
44FD CJK UNIFIED IDEOGRAPH-#44FD, IRGKangXi=1047.720, Def{name of a variety of grass}
44FE CJK UNIFIED IDEOGRAPH-#44FE, IRGKangXi=1047.730, Def{lush; luxuriant of grass and trees}
44FF CJK UNIFIED IDEOGRAPH-#44FF, IRGKangXi=1048.160, Def{a lotus flower}
4500 CJK UNIFIED IDEOGRAPH-#4500, IRGKangXi=1048.280, Def{(same as 荽 葰) parsley}
4501 CJK UNIFIED IDEOGRAPH-#4501, IRGKangXi=1049.090, Def{name of a variety of grass}
4502 CJK UNIFIED IDEOGRAPH-#4502, IRGKangXi=1049.170, Def{vegetable soup, a kind of vegetable}
4503 CJK UNIFIED IDEOGRAPH-#4503, IRGKangXi=1049.180, Def{cordate houttuynia (Houttuynia cordata)}
4504 CJK UNIFIED IDEOGRAPH-#4504, IRGKangXi=1049.330, Def{name of a variety of grass; rushes, dodder}
4505 CJK UNIFIED IDEOGRAPH-#4505, IRGKangXi=1049.350, Def{name of a variety of trees, name of a variety of grass, fruit of the tree}
4506 CJK UNIFIED IDEOGRAPH-#4506, IRGKangXi=1050.010, Def{a kind of bean; edible}
4507 CJK UNIFIED IDEOGRAPH-#4507, IRGKangXi=1050.210, Def{a kind of plant, the root is used for food}
4508 CJK UNIFIED IDEOGRAPH-#4508, IRGKangXi=1051.110, Def{lush growth of grass, trees and fruits}
4509 CJK UNIFIED IDEOGRAPH-#4509, IRGKangXi=1051.150, Def{scattered or dispersed of grass and trees, (non-classical form of 蒜) the garlic (belongs to meat and fish diet)}
450A CJK UNIFIED IDEOGRAPH-#450A, IRGKangXi=1052.051, Def{dense growth of grass}
450B CJK UNIFIED IDEOGRAPH-#450B, IRGKangXi=1052.051, Def{a kind of tree; very tall and its leaves like bird's wings; edible}
450C CJK UNIFIED IDEOGRAPH-#450C, IRGKangXi=1052.051, Def{name of a variety of grass, (Pueraria thunbergiana); the white birch}
450D CJK UNIFIED IDEOGRAPH-#450D, IRGKangXi=1052.051
450E CJK UNIFIED IDEOGRAPH-#450E, IRGKangXi=1052.100, Def{name of a variety of grass}
450F CJK UNIFIED IDEOGRAPH-#450F, IRGKangXi=1052.120, Def{a lush; dense growth of weeds, mixed; mingled roots of the grass, (same as 簉) deputy; a deputy; an attendant, an escorting vehicle}
4510 CJK UNIFIED IDEOGRAPH-#4510, IRGKangXi=1052.240, Def{delicate, colorful and beautiful small piece of paper}
4511 CJK UNIFIED IDEOGRAPH-#4511, IRGKangXi=1052.260, Def{a white colored cluster grown at the tip of cogongrass, a kind of plants of the mugwort or artemisia family, an ear of grain}
4512 CJK UNIFIED IDEOGRAPH-#4512, IRGKangXi=1053.080, Def{name of a variety of grass}
4513 CJK UNIFIED IDEOGRAPH-#4513, IRGKangXi=1053.120, Def{herb medicine; teasel root}
4514 CJK UNIFIED IDEOGRAPH-#4514, IRGKangXi=1053.170, Def{name of a variety of grass}
4515 CJK UNIFIED IDEOGRAPH-#4515, IRGKangXi=1053.180, Def{chicken breast}
4516 CJK UNIFIED IDEOGRAPH-#4516, IRGKangXi=1053.200, Def{(same as 菱 蔆) water caltrop}
4517 CJK UNIFIED IDEOGRAPH-#4517, IRGKangXi=1054.140, Def{a kind of vegetable}
4518 CJK UNIFIED IDEOGRAPH-#4518, IRGKangXi=1054.200, Def{fish paste; fish cooked in soy sauce}
4519 CJK UNIFIED IDEOGRAPH-#4519, IRGKangXi=1054.210, Def{name of a variety of grass}
451A CJK UNIFIED IDEOGRAPH-#451A, IRGKangXi=1054.220, Def{herb medicine; the water plantain}
451B CJK UNIFIED IDEOGRAPH-#451B, IRGKangXi=1054.260, Def{fibrous plant of which cloth, rope, bag are made}
451C CJK UNIFIED IDEOGRAPH-#451C, IRGKangXi=1054.290, Def{herb medicine; hellebore}
451D CJK UNIFIED IDEOGRAPH-#451D, IRGKangXi=1054.310, Def{name of a variety of grass}
451E CJK UNIFIED IDEOGRAPH-#451E, IRGKangXi=1055.070, Def{the beginning; to start, new growing of the grass}
451F CJK UNIFIED IDEOGRAPH-#451F, IRGKangXi=1055.120, Def{shake and wave of the grass}
4520 CJK UNIFIED IDEOGRAPH-#4520, IRGKangXi=1055.240, Def{thin root (rhizome) of the lotus, (same as 梢) the tip of a branch, (same as U+77DF 槊) a long spear; a lance}
4521 CJK UNIFIED IDEOGRAPH-#4521, IRGKangXi=1055.340, Def{name of a variety of grass; perilla ocimoides, whose seeds are a bird feed}
4522 CJK UNIFIED IDEOGRAPH-#4522, IRGKangXi=1056.110, Def{(ancient form of 華) Cathay; China, splendid; gorgeous; colorful; beautiful; luxurious, a family name}
4523 CJK UNIFIED IDEOGRAPH-#4523, IRGKangXi=1056.121, Def{a tribe in ancient times}
4524 CJK UNIFIED IDEOGRAPH-#4524, IRGKangXi=1056.121, Def{the lush and luxuriant of the grass and flowers}
4525 CJK UNIFIED IDEOGRAPH-#4525, IRGKangXi=1056.121, Def{(non-classical form of U+856D 蕭) common variety of artemisia}
4526 CJK UNIFIED IDEOGRAPH-#4526, IRGKangXi=1056.121
4527 CJK UNIFIED IDEOGRAPH-#4527, IRGKangXi=1056.121, Def{(same as 蔾) a kind of plant; the caltrop; caltrap}
4528 CJK UNIFIED IDEOGRAPH-#4528, IRGKangXi=1056.121
4529 CJK UNIFIED IDEOGRAPH-#4529, IRGKangXi=1056.121
452A CJK UNIFIED IDEOGRAPH-#452A, IRGKangXi=1054.090, Def{name of a variety of grass, scattered or dispersed of the grass and trees}
452B CJK UNIFIED IDEOGRAPH-#452B, IRGKangXi=1056.121
452C CJK UNIFIED IDEOGRAPH-#452C, IRGKangXi=1056.170, Def{hemp, marijuana (female; pistillate), herbage}
452D CJK UNIFIED IDEOGRAPH-#452D, IRGKangXi=1056.200, Def{name of a variety of grass; poisonous herb}
452E CJK UNIFIED IDEOGRAPH-#452E, IRGKangXi=1056.210, Def{name of a variety of grass; growing in the water with edible flowers}
452F CJK UNIFIED IDEOGRAPH-#452F, IRGKangXi=1056.240, Def{salted; pickled leeks or scallions}
4530 CJK UNIFIED IDEOGRAPH-#4530, IRGKangXi=1056.290, Def{(variant of 葍) name of a variety of grass; perennial; creeping plants}
4531 CJK UNIFIED IDEOGRAPH-#4531, IRGKangXi=1056.300, Def{name of a variety of grass}
4532 CJK UNIFIED IDEOGRAPH-#4532, IRGKangXi=1056.360, Def{name of a variety of grass, (same as 橙) the orange}
4533 CJK UNIFIED IDEOGRAPH-#4533, IRGKangXi=1056.380, Def{chickling}
4534 CJK UNIFIED IDEOGRAPH-#4534, IRGKangXi=1056.410, Def{name of a variety of grass, growing grass, to weed (in farming); to mow grass or cut weed}
4535 CJK UNIFIED IDEOGRAPH-#4535, IRGKangXi=1056.420, Def{(same as 葌 蕑) fragrant thoroughwort (Eupatorium fortunei)}
4536 CJK UNIFIED IDEOGRAPH-#4536, IRGKangXi=1057.040, Def{name of a variety of grass, weed (in farming); to mow grass or cut weed}
4537 CJK UNIFIED IDEOGRAPH-#4537, IRGKangXi=1057.140, Def{Cyperus rotundus, a kind of medicinal herb}
4538 CJK UNIFIED IDEOGRAPH-#4538, IRGKangXi=1057.260, Def{name of a variety of grass, fragrant herb; vanilla, fragrant; delicious; balm; spice}
4539 CJK UNIFIED IDEOGRAPH-#4539, IRGKangXi=1057.320, Def{a fungus which grows on rotten plants and is eaten when fresh}
453A CJK UNIFIED IDEOGRAPH-#453A, IRGKangXi=1057.370, Def{heading; earing; to flower; to blossom (of the smartweed group), Cyperus rotundus, a kind of medicinal herb, a peduncle or footstalk of a flower or fruit; a stem; a base, new growing leaves}
453B CJK UNIFIED IDEOGRAPH-#453B, IRGKangXi=1058.050, Def{name of a variety of grass, luxuriant; lush of the grass}
453C CJK UNIFIED IDEOGRAPH-#453C, IRGKangXi=1058.110, Def{sound of the grass; parsley}
453D CJK UNIFIED IDEOGRAPH-#453D, IRGKangXi=1058.200, Def{to cover; to put a cover over something, small, low, weak, pure; clean; clear}
453E CJK UNIFIED IDEOGRAPH-#453E, IRGKangXi=1058.230, Def{a kind of fragrant herb, a kind of vegetable (growing in the water; something like the bracken)}
453F CJK UNIFIED IDEOGRAPH-#453F, IRGKangXi=1059.050, Def{lush growth; dense growth of trees and grass, to huddle together; to crowd together}
4540 CJK UNIFIED IDEOGRAPH-#4540, IRGKangXi=1059.070, Def{name of a variety of grass, Compositae; chrysanthemum family, perennial herbage; to blossom in winter}
4541 CJK UNIFIED IDEOGRAPH-#4541, IRGKangXi=1059.120, Def{suddenly, name of a variety of grass}
4542 CJK UNIFIED IDEOGRAPH-#4542, IRGKangXi=1059.291
4543 CJK UNIFIED IDEOGRAPH-#4543, IRGKangXi=1059.291, Def{(same as 蔭) the shade of a tree, to cover up; to cover or screen}
4544 CJK UNIFIED IDEOGRAPH-#4544, IRGKangXi=1050.080, Def{meat in the form of paste; salted or pickled meat}
4545 CJK UNIFIED IDEOGRAPH-#4545, IRGKangXi=1059.420, Def{name of a variety of grass}
4546 CJK UNIFIED IDEOGRAPH-#4546, IRGKangXi=1059.460, Def{round straw quilt; bedding; cushion; mattress; mat for the chair, name of a variety of grass}
4547 CJK UNIFIED IDEOGRAPH-#4547, IRGKangXi=1059.490, Def{name of a variety of grass, a kind of vegetable}
4548 CJK UNIFIED IDEOGRAPH-#4548, IRGKangXi=1060.090, Def{name of a variety of grass, (same as U+5C5F 屧) the wooden sole of a shoe}
4549 CJK UNIFIED IDEOGRAPH-#4549, IRGKangXi=1060.130, Def{a kind of plant, Pueraria thunbergiana, a creeping, edible bean whose fibers can be made into linen-like cloth and whose roots are used in herbal medicine}
454A CJK UNIFIED IDEOGRAPH-#454A, IRGKangXi=1060.170, Def{n